/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/format/paragraphstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom'
], function (Utils, AttributeUtils, Color, Border, LineHeight, StyleCollection, DOM) {

    'use strict';

    var // definitions for paragraph attributes
        DEFINITIONS = {

            nextStyleId: {
                def: '',
                scope: 'style'
            },

            alignment: {
                def: 'left',
                format: function (element, value) {
                    element.css('text-align', value);
                },
                preview: true // 'true' calls the own 'format' method
            },

            fillColor: {
                def: Color.AUTO, // auto for paragraph fill resolves to 'transparent'
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            /**
             * Line height relative to font settings. The CSS attribute
             * 'line-height' must be set separately at every descendant text
             * span because a relative line height (e.g. 200%) would not be
             * derived from the paragraph relatively to the spans, but
             * absolutely according to the paragraph's font size. Example: The
             * paragraph has a font size of 12pt and a line-height of 200%,
             * resulting in 24pt. This value will be derived absolutely to a
             * span with a font size of 6pt, resulting in a relative line
             * height of 24pt/6pt = 400% instead of the expected 200%.
             */
            lineHeight: { def: LineHeight.SINGLE },

            listLevel: { def: -1 },

            listStyleId: { def: '' },

            listLabelHidden: { def: false },

            listStartValue: { def: -1 },

            outlineLevel: { def: 9 },

            tabStops: {
                def: [],
                merge: function (tabStops1, tabStops2) {
                    // Merge tabStops2 into array tabstop1
                    // Support to clear tab stops defined in tabStops1
                    var clearedTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value === 'clear';
                        }),
                        additionalTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value !== 'clear';
                        }),
                        newTabstops = tabStops1 ? (additionalTabstops ? tabStops1.concat(additionalTabstops) : tabStops1) : (additionalTabstops ? additionalTabstops : []);

                    // Make entries in newTabstops unique
                    newTabstops = _.unique(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });

                    // Filter out cleared tabStops
                    if (clearedTabstops.length > 0) {
                        newTabstops = _.filter(newTabstops, function (tabstop) {
                            // return only tab stops which are not part of the clearedTabstops array
                            return (_.find(clearedTabstops, function (cleared) {
                                return cleared.pos === tabstop.pos;
                            }) === undefined);
                        });
                    }

                    // Sort tabStops by position
                    return _.sortBy(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });
                }
            },

            borderLeft: { def: Border.NONE },

            borderRight: { def: Border.NONE },

            borderTop: { def: Border.NONE },

            borderBottom: { def: Border.NONE },

            borderInside: { def: Border.NONE },

            indentFirstLine: { def: 0 },

            indentLeft: { def: 0 },

            indentRight: { def: 0 },

            marginTop: { def: 0 },

            marginBottom: { def: 0 },

            contextualSpacing: { def: false },

            pageBreakBefore: {
                def: false,
                format: function (element, value) {
                    if (value && !DOM.isNodeInsideTextFrame(element)) {
                        if (element.closest('tr').length > 0) {
                            element.parents('table').last().addClass('manual-page-break');
                        }
                        element.addClass('manual-page-break');
                    }
                }
            },

            pageBreakAfter: {
                def: false,
                format: function (element, value) {
                    if (value && !DOM.isNodeInsideTextFrame(element)) {
                        if (element.closest('tr').length > 0) {
                            element.parents('table').last().addClass('manual-pb-after');
                        }
                        element.addClass('manual-pb-after');
                    }
                }
            }
        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            cell: function (paragraph) { return DOM.isCellContentNode(paragraph.parent()) ? paragraph.closest('td') : null; }
        },

        // maps fill character attribute values to fill characters
        TAB_FILL_CHARS = { dot: '.', hyphen: '-', underscore: '_' },

        NBSP = '\xa0';

    // private static functions ===============================================

    function isMergeBorders(attributes1, attributes2) {
        return (Border.isVisibleBorder(attributes1.borderLeft) ||
            Border.isVisibleBorder(attributes1.borderRight) ||
            Border.isVisibleBorder(attributes1.borderTop) ||
            Border.isVisibleBorder(attributes1.borderBottom) ||
            Border.isVisibleBorder(attributes1.borderInside)) &&
            Utils.hasEqualProperties(attributes1, attributes2,
                ['borderLeft', 'borderRight', 'borderTop', 'borderBottom', 'borderInside',
                 'listStyleId', 'listLevel', 'indentLeft', 'indentRight', 'indentFirstLine']);
    }

    /**
     * Utility check if attributes have properties pageBreakBefore or pageBreakAfter
     * set to true.
     *
     * @param {Object} attributes
     *
     * @returns {Boolean}
     *
     */
    function isSetPageBreakAttribute(attributes) {
        return attributes.pageBreakBefore === true || attributes.pageBreakAfter === true;
    }

    /**
     * Fills the passed text span with a sufficient number of the specified
     * fill character.
     *
     * @param {jQuery} spanNode
     *  The span node to be filled. The current CSS formatting of this node is
     *  used to calculate the number of fill characters needed.
     *
     * @param {String} fillChar
     *  The fill character.
     *
     * @param {Number} width
     *  The target width of the tabulator, in 1/100 of millimeters.
     */
    function insertTabFillChar(spanNode, fillChar, width) {

        var // multiplier for a better average calculation
            multiplier = 15,
            // multiplier fill characters, used to calculate average character width
            checkString = Utils.repeatString(fillChar, multiplier),
            // average character width, in 1/100 mm
            charWidth = Utils.convertLengthToHmm(spanNode.contents().remove().end().text(checkString).width(), 'px') / multiplier,
            // number of characters needed to fill the specified width
            charCount = Math.floor(width / charWidth),
            // the fill character, repeated by the calculated number
            fillString = (charCount > 0) ? Utils.repeatString(fillChar, charCount) : NBSP,
            // a shortened string, if element is too wide
            shortFillString = null;

        // insert the fill string into the element
        spanNode.contents().remove().end().text(fillString);
        if (!fillString) { DOM.ensureExistingTextNode(spanNode); }

        // shorten fill string by one character, if element is too wide (e.g. due to rounding errors)
        if ((fillString.length > 1) && (Utils.convertLengthToHmm(spanNode.width(), 'px') >= width)) {
            shortFillString = fillString.slice(0, -1);
            spanNode.contents().remove().end().text(shortFillString);
            if (!shortFillString) { DOM.ensureExistingTextNode(spanNode); }
        }

        // workaround for iOS & Safari mobile - needs a text node directly within
        // the tab div to enable an external keyboard to jump through a tab chains
        // with empty spans
        if (_.device('ios') && _.device('safari')) {
            if (!spanNode[0].nextSibling) {
                spanNode.after(document.createTextNode(NBSP));
            }
        }
    }

    // class ParagraphStyles ==================================================

    /**
     * Contains the style sheets for paragraph formatting attributes. The CSS
     * formatting will be read from and written to paragraph <p> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function ParagraphStyles(docModel) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'paragraph', {
            families: 'character changes',
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateParagraphFormatting,
            previewHandler: setParagraphPreviewFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @attention
         *  Currently, only left&right aligned tabs are supported!
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateTabStops(paragraph, mergedAttributes) {

            var // default tab stop width from document settings
                defaultTabStop = docModel.getDocumentAttribute('defaultTabStop'),
                // paragraph tab stop definitions
                paraTabStops = mergedAttributes.paragraph.tabStops,
                // effective left margin of the paragraph
                marginLeft = Utils.convertCssLengthToHmm(paragraph.css('margin-left'), 'px'),
                // end position of previous tabulator
                prevTabEndPos = null,
                // first node of paragraph
                firstNode = null,
                // top position of first node
                topFirst = 0,
                // height of first node
                heightFirst = 0,
                // leftmost position used a base to calculate tab raster
                zeroLeft = 0,
                // first line  indent
                indentFirstLine = mergedAttributes.paragraph.indentFirstLine,
                // intersect function
                intersect = function (xs1, xe1, xs2, xe2) {
                    return ((xs1 <= xs2) && (xe1 >= xs2) ||
                             (xs2 <= xs1) && (xe2 >= xs1));
                },
                // active paragraph tab stops
                activeParaTabStops = null,
                // if paraTabStops has left and right oriented tabs, mixedTabs is set by the right-tab. if it is set it overwrites behaviro of text-aligment and para-margin
                mixedTab = getMixedTab(paraTabStops);

            if (docModel.getApp().isInQuit()) {
                return;
            }

            // zoom factor in floating point notation
            var  zoomFactor = docModel.getApp().getView().getZoomFactor() / 100;

            if (paragraph.length) {
                // Calculate top position and height of first paragraph child. Our paragraph
                // have always at least one child element.
                firstNode = $(paragraph.get(0).firstChild);
                topFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.position().top, 'px') : 0;
                heightFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.height(), 'px') : 0;

                //reset all tab sizes, for a much cleaner new calculating
                //(for example changing text-alignment from justify to left)
                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {
                    $(this).css('width', 0);
                });

                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {

                    var // the current tab node of the paragraph
                        tabNode = $(this),
                        // the span within the tab node containing the fill characters
                        tabSpan = $(this.firstChild),
                        // the position of the tab node
                        pos = tabNode.position(),
                        // the left bound position of the tab node in 1/100th mm
                        leftHMM = 0,
                        // the top bound position of the tab node in 1/100th mm
                        topHMM = Utils.convertLengthToHmm(pos.top, 'px'),
                        // the top bound position of the first child node of the paragraph in 1/100th mm
                        topSpanHMM = 0,
                        // the height of the first child node of the paragraph in 1/100th mm
                        heightSpanHMM = 0,
                        // the calculated width of the tab stop
                        width = 0,
                        // the fill character of the tab stop
                        fillChar = null,
                        // the tab stop values
                        tabStop,
                        // ignore the tab position and continue with the next one
                        ignore = false,
                        // tab stop type
                        tabStopType = 'left',
                        // indent first line tab stop position
                        indentFirstLineTabStopValue = 0,
                        // first line tab stop
                        indentFirstLineTabStop = null,
                        // insert index for virtual first line indent tab stop
                        indentFirstLineTabStopIndex = 0;

                    var calcLeftTab = function () {
                        return Math.max(0, tabStop.pos - (leftHMM % tabStop.pos));
                    };
                    var calcRightTab = function () {
                        return calcRightAlignTabstop(tabStop, tabNode, leftHMM);
                    };
                    var calcMiddleTab = function () {
                        return 0.5 * (calcLeftTab() + calcRightTab());
                    };

                    if (mergedAttributes.paragraph.alignment === 'center') {
                        // for a centered paragraph tab stops are calculated where the first character is treated as leftmost (0) position
                        zeroLeft = firstNode.length ? -(marginLeft + Utils.convertLengthToHmm(firstNode.position().left, 'px')) : -marginLeft;
                    } else {
                        zeroLeft = marginLeft;
                    }

                    // calculate left bound position of the tab node in 1/100th mm, including zoom factor for browsers using transf.scale
                    leftHMM = zeroLeft + Utils.convertLengthToHmm(pos.left / zoomFactor, 'px');

                    if (prevTabEndPos) {
                        // Check to see if we are in a chain of tabs. Force to use the previous tab end position
                        // as start position. Some browsers provide imprecise positions therefore we use the correct
                        // previous position from the last calculation.

                        // Check previous node and that text is empty - this is a precondition for the chaining
                        // code.
                        if (tabNode.prev().length && tabNode.prev().text().length === 0) {
                            var checkPrevPrevNode = tabNode.prev().length ? tabNode.prev().prev() : null;
                            if (checkPrevPrevNode.length && DOM.isTabNode(checkPrevPrevNode) && prevTabEndPos.top === topHMM) {
                                leftHMM = prevTabEndPos.center || prevTabEndPos.right;
                            }
                        }
                    }

                    // Special case for first line indent. For the first line the negative indent
                    // must be used as a tab stop position using the absolute value. We need to set
                    // at least one character for the tabSpan to have a valid width otherwise we
                    // won't get a correct height which is needed for the check.
                    if (indentFirstLine < 0) {
                        indentFirstLineTabStopValue = Math.abs(indentFirstLine);
                        if (marginLeft > 0) {
                            // In case of a margin left, we need to add the value to the indentFirstLineTabStopValue!
                            indentFirstLineTabStopValue += marginLeft;
                        }
                        // Fix for 29265 and 30847: Removing empty text node in span with '.contents().remove().end()'.
                        // Otherwise there are two text nodes in span after '.text('abc')' in IE.
                        tabSpan.contents().remove().end().text(NBSP);
                        heightSpanHMM = Utils.convertLengthToHmm(tabSpan.height(), 'px');
                        topSpanHMM = Utils.convertLengthToHmm(tabSpan.position().top, 'px');
                        // checkout if the first line indent is active for this special tab stop
                        if (intersect(topFirst, topFirst + heightFirst, topSpanHMM, topSpanHMM + heightSpanHMM) &&
                            (leftHMM + 10) < indentFirstLineTabStopValue) {
                            width = Math.max(0, indentFirstLineTabStopValue - leftHMM);
                        }

                        if (width > 1) {
                            // find index for the first line indent position within the paragraph tab stop array
                            indentFirstLineTabStopIndex = Utils.findLastIndex(paraTabStops, function (tab) {
                                return (indentFirstLineTabStopValue > tab.pos);
                            }) + 1;
                            // create a copy of the paragraph tab stops and add the first indent tab stop at
                            // the correct position to the active paragraph tab stop array
                            indentFirstLineTabStop = { value: 'left', pos: indentFirstLineTabStopValue, fillChar: NBSP, processed: false };
                            activeParaTabStops = paraTabStops.slice();
                            activeParaTabStops.splice(indentFirstLineTabStopIndex, 0, indentFirstLineTabStop);
                            // reset width - the tab stop position must be calculated using all tab stop positions
                            width = 0;
                        } else {
                            // reset text within span
                            tabSpan.contents().remove();
                            tabSpan[0].appendChild(document.createTextNode(''));
                            // use only paragraph tab stops
                            activeParaTabStops = paraTabStops;
                        }
                    } else {
                        // reset text within span
                        tabSpan.contents().remove();
                        tabSpan[0].appendChild(document.createTextNode(''));
                        // use only paragraph tab stops
                        activeParaTabStops = paraTabStops;
                    }

                    // Paragraph tab stops. Only paragraph tab stop can have a fill character and
                    // define a new alignment
                    if (width <= 1) {
                        tabStop = _(activeParaTabStops).find(function (tab) {
                            return ((leftHMM + 10) < tab.pos) && !tab.processed;
                        });
                        if (tabStop) {

                            tabStopType = tabStop.value || 'left';

                            // calculate tab stop size based on tab stop properties
                            if (tabStopType === 'left') {
                                // left bound tab stop
                                width = calcLeftTab();
                            } else if (tabStopType === 'right') {
                                // right bound tab stop
                                width = calcRightTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);

                                if (!ignore && _.indexOf(activeParaTabStops, tabStop) === activeParaTabStops.length - 1) {
                                    if (!_.browser.IE) {
                                        paragraph.css('width', 'calc(100% + 5mm)');
                                    }
                                }

                            } else if (tabStopType === 'center') {
                                // right bound tab stop combinded with left bound tab stop
                                width = calcMiddleTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);
                            }

                            // insert fill character
                            if (width > 1) {
                                fillChar = _.isString(tabStop.fillChar) ? TAB_FILL_CHARS[tabStop.fillChar] : NBSP;
                                if (fillChar) {
                                    insertTabFillChar(tabSpan, fillChar, width);
                                }
                                // Set processed flag to prevent using the same tab again due to rounding errors
                                // resulting by the browser calculation.
                                tabStop.processed = true;
                            }
                        }
                    }

                    if (!ignore) {
                        // only process default tab stop if tab stop is not set to ignore
                        if (width <= 1) {
                            // tab size calculation based on default tab stop
                            width = Math.max(0, defaultTabStop - (leftHMM % defaultTabStop));
                            width = (width <= 10) ? defaultTabStop : width; // no 0 tab size allowed, check for <= 10 to prevent rounding errors
                            // reset possible fill character
                            insertTabFillChar(tabSpan, NBSP, width);
                        }
                        prevTabEndPos = { right: leftHMM + width, top: topHMM };
                    }
                    // always set a width to the tab div (even an ignored tab div needs to set the size to zero)
                    tabNode.css('width', (width / 100) + 'mm');
                });

                calculateMarginForMixedTab(docModel, paragraph, mergedAttributes, mixedTab);

                // reset processed flag for tab stops again
                paraTabStops.forEach(function (tab) {
                    tab.processed = false;
                });
            }
        }

        /**
         * right tab can lay right above from the border, this cannot be triggered by MS-WORD interface,
         * but it is a typical behavior for MS-"table of content"
         * so we calculate if right tab is outside of the margin, and if true,
         * we reduce the margin-right of the paragraph
         */
        function calculateMarginForMixedTab(docModel, paragraph, paraAttributes, mixedTab) {
            paragraph.css('margin-right', null);

            if (!mixedTab) {
                return;
            }

            var pageAttrs = docModel.getDefaultAttributes('page');
            var maxParaWidth = pageAttrs.width - pageAttrs.marginLeft - pageAttrs.marginRight;
            var indentRight = paraAttributes.paragraph.indentRight;
            var currentParaWidth = mixedTab.pos + indentRight;

            if (currentParaWidth > maxParaWidth) {
                var newIndentRight = indentRight - (currentParaWidth - maxParaWidth);
                paragraph.css('margin-right', (newIndentRight / 100) + 'mm');
            }
        }

        function getMixedTab(paraTabStops) {
            var leftTab = _.find(paraTabStops, function (tab) {
                return !tab.value || tab.value === 'left';
            });
            var rightTab = _.find(paraTabStops, function (tab) {
                return tab.value === 'center' || tab.value === 'right';
            });
            if (leftTab && rightTab) {
                return rightTab;
            }
        }

        /**
         * Calculates the width of a right aligned tab element
         * depending on the position and the remaining size of
         * the following nodes.
         *
         * @param {Object} tabStop
         *  The right aligned tab stop.
         *
         * @param {jQuery} tabNode
         *  The right aligned tab node as jQuery.
         *
         * @param {Number} tabNodePosHMM
         *  The current position of the tab node in 1/100th mm.
         *
         * @returns {Number}
         *  The width of the right aligned tab stop. May be zero if the
         *  tab stop is not active.
         */
        function calcRightAlignTabstop(tabStop, tabNode, tabNodePosHMM) {
            var nextNodes = tabNode.nextAll(),
                node,
                rightPos = tabNodePosHMM,
                width = 0, i = 0, len = nextNodes.length;

            // Loop through all following nodes and sum up their width. Exceeding
            // the tab stop position or a new tab stops the iteration.
            for (i = 0; i < len && rightPos < tabStop.pos; i++) {
                node = $(nextNodes.get(i));
                if (DOM.isTabNode(node)) {
                    break;
                }
                rightPos += Utils.convertLengthToHmm(node.width(), 'px');
            }

            if (rightPos < tabStop.pos) {
                // Active right aligned tab stop. Subtract at least 1/10mm
                // rounding error regarding position values from the browser.
                width = tabStop.pos - rightPos - 10;
                width = Math.max(0, width);
            }

            return width;
        }

        /**
         * Colorize the paragraphs margin background in the color from the paragraph when needed.
         * There are some special cases were the margin should not have a color, e.g. when
         * the paragraph has a border or auto color. As the performance for formatting paragraphs
         * should not be affected by this, there is a check if this formatting is necessary.
         *
         * Normally the margin background can't have a color via CSS. So this function
         * uses a CSS boxshadow to archive this. In case the paragraph height is smaller
         * than the margin, multiple boxshadows must be stacked.
         *
         *  @param {Object} paraObj
         *   An Object containing needed and optional data to calculate the
         *   colored paragraph margins.
         *
         *  Parameters:
         *   @param {jQuery} paragraph
         *    The paragraph node, as jQuery object.
         *
         *   @param {Object} mergedAttributes
         *    A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         *
         *   @param {jQuery|Null} [prevParagraph]
         *    The previous paragraph node, as jQuery object.
         *    Use this parameter only when it's certain that the previous paragraph exists.
         *    When not, set it to Null. Important: This is e.g. not the case at document import or undo/redo.
         *
         *   @param {jQuery|Null} [nextParagraph]
         *    The next paragraph node, as jQuery object.
         *    Use this parameter only when it's certain that the previous paragraph exists.
         *    When not, set it to Null. Important: This is e.g. not the case at document import or undo/redo.
         *
         *   @param {Object|Null} [prevAttributes]
         *    Attributes from the previous paragraph.
         *    Use this parameter only when it's certain that the previous paragraph exists.
         *    When not, set it to Null. Important: This is e.g. not the case at document import or undo/redo.
         *
         *   @param {Object|Null} [nextAttributes]
         *    Attributes from the next paragraph.
         *    Use this parameter only when it's certain that the previous paragraph exists.
         *    When not, set it to Null. Important: This is e.g. not the case at document import or undo/redo.
         *
         * @param {Boolean} applyCssDirectly
         *  Whether the created CSS boxshadow markup should be returned as a string
         *  or if the CSS markup should be applied directly.
         */
        function setMarginColor(paraObj, applyCssDirectly) {

            var // the current paragraph node MUST exists
                paragraph = paraObj.paragraph,
                // the current mergedAttributes MUST exist
                mergedAttributes = paraObj.mergedAttributes,

                //the current paragraphs attributes
                paragraphAttributes = mergedAttributes.paragraph,

                // re-use when value exists in the object, otherwise get them
                prevParagraph = paraObj.prevParagraph || paragraph.prev(),
                prevAttributes = paraObj.prevAttributes || (prevParagraph.length > 0) ? self.getElementAttributes(prevParagraph) : { paragraph: {} },
                nextParagraph = paraObj.nextParagraph || paragraph.next(),
                nextAttributes = paraObj.nextAttributes || (nextParagraph.length > 0) ? self.getElementAttributes(nextParagraph) : { paragraph: {} },

                topMargin,
                bottomMargin,
                paragraphHeight,
                prevParagraphHeight,
                nextParagraphHeight,

                // container to gather CSS markup for the current, previous and next paragraph
                boxShadowString = '',
                prevBoxShadowString = '',
                nextBoxShadowString = '';

            /**
            * Initialize values. They calculation from these values are not
            * cheap performance wise, so they should only be initialized when needed.
            */
            function initValues() {
                topMargin = paragraphAttributes.marginTop;
                bottomMargin = paragraphAttributes.marginBottom;

                paragraphHeight =  Utils.convertLengthToHmm(paragraph.height(), 'px');
                prevParagraphHeight =  Utils.convertLengthToHmm(prevParagraph.height(), 'px');
                nextParagraphHeight =  Utils.convertLengthToHmm(nextParagraph.height(), 'px');
            }

            /**
            * Check if the previous paragraph has a color.
            *
            * @returns {Boolean}
            *  If the previous paragraph has a color or not.
            */
            function hasColorUp() {
                return paragraphHasColor(prevAttributes.paragraph);
            }

            /**
            * Check if the next paragraph has a color.
            *
            * @returns {Boolean}
            *  If the next paragraph has a color or not.
            */
            function hasColorDown() {
                return paragraphHasColor(nextAttributes.paragraph);
            }

            /**
            * Check if the current paragraph has a color.
            *
            * @returns {Boolean}
            *  If the current paragraph has a coloror not.
            */
            function hasColor() {
                return paragraphHasColor(paragraphAttributes);
            }

            /**
             * Check if the given paragraph has a color.
             *
             * @param {Object} paraAttr
             *  The paragraphs attributes.
             *
             * @returns {Boolean}
             *  If the given paragraph has a color or not.
             */
            function paragraphHasColor(paraAttr) {
                if (paraAttr.fillColor && paraAttr.fillColor.type !== 'auto') {
                    return true;
                } else {
                    return false;
                }
            }

            /**
             * Check if the given paragraph has a border.
             *
             * @param {Object} paraAttr
             *  The paragraph attributes.
             *
             * @returns {Boolean}
             *  If the paragraph has a border or not.
             */
            function hasNoBorder(paraAttr) {
                // exists check && ... // TODO
                if ((paraAttr.borderLeft.style !== 'none') || (paraAttr.borderRight.style !== 'none') || (paraAttr.borderTop.style !== 'none') || (paraAttr.borderBottom.style !== 'none')) { // string compare TODO
                    return false;
                } else {
                    return true;
                }
            }
            /**
            * Add a comma between two strings when they are both not empty. The resulting string is returned.
            *
            *  @param {String} str1
            *  A string.
            *
            *  @param {String} str2
            *  A string.
            *
            * @returns {String}
            *  The concatenated string.
            */
            function concatStringsWithSeperator(str1, str2) {
                if (str1 !== '' && str2 !== '') { str1 += ',' + str2; } else { str1 += str2;  }
                return str1;
            }

            /**
            * Create the boxshadow markup. It returns a string that contains the CSS as a string.
            *
            * @param {Number} offset
            *  The offset that the boxshadow should cover.
            *
            * @param {Number} paraHeight
            *  The paragraph height on which the shadow is appended.
            *
            * @param {String} color
            *  The color for the boxshadow.
            *
            * @param {Boolean} downwards
            *  The direction from resulting the boxshadow.
            *
            * @returns {String}
            *  Returns a string with the created CSS boxshadow markup
            */
            function createShadowCss(offset, paraHeight, color, downwards) {

                var boxShadowMarkUpString = '';
                var tempBoxShadowOffset;
                var seperator;
                // Reduced the height by -15 to get a bit overlapping from stacked shadows.
                // Otherwise in some cases small white lines are visible between stacked shadows.
                var paraHeightWithOverlapping = paraHeight - 15;

                // The boxshadow has the same height as the paragraph.
                // So when the margin is greater than the paragraphs height, multiple boxshadows must be used.
                var numberOfNeededBoxShadows = (Math.ceil(Math.abs(offset) / Math.abs(paraHeightWithOverlapping)));

                // create boxshadows up or downwards. Note: At the first round ( i === 0 ) only the offset is used
                // to calculate tempBoxShadowOffset, so overlapping has no effect at the border from two paragraphs
                _.times(numberOfNeededBoxShadows, function (i) {
                    // unit needs to be mm, therefore / 100 at the end
                    tempBoxShadowOffset =  downwards ? (offset - (paraHeightWithOverlapping * i)) / 100 : (offset + (paraHeightWithOverlapping * i)) / 100;
                    // check if a seperator is needed
                    seperator = (i === 0) || (i === numberOfNeededBoxShadows) ? '' : ',';

                    boxShadowMarkUpString =  boxShadowMarkUpString + seperator + '0mm ' + tempBoxShadowOffset + 'mm 0mm ' + color;
                });

                return boxShadowMarkUpString;
            }

            /**
            * Calculates the resulting height/offset from the colored margin upwards and returns it.
            *
            * @param {Number} marginBottom
            *  The margin bottom from a paragraph.
            *
            * @param {Number} nextMarginTop
            *  Related to the paragraph, considered in first Parameter, the next paragraphs margin top.
            */
            function calcColorMarginOffsetUp(marginBottom, nextMarginTop) {

                // when the marginBottom is greater than the next margin top we must not color the current margin top
                if (marginBottom > nextMarginTop) {
                    return 0;
                } else {
                    return (Math.abs(marginBottom - nextMarginTop)) * -1;
                }
            }

            /**
            * Creating the boxshadow CSS markup upwards. That means for the current, and previous paragraph.
            * For that, the previous, previous paragraph needs to be considered too in the calculation.
            * The resulting CSS markup is adds to related variables, that gathers the CSS markup at the end.
            *
            * @param {Number} topMargin
            *  The margin top from the current paragraph.
            *
            * @param {Number} paragraphHeight
            *  The height from the paragraph on which the shadow is appended.
            *
            * @param {Number} prevParagraphHeight
            *  The height from the previous paragraph.
            */
            function setMarginColorsUpwards(topMargin, paragraphHeight, prevParagraphHeight) {
                // the previous, previous paragraph
                var prevPrevParagraph = prevParagraph.prev();
                // the previous, previous attributes
                var prevPrevAttributes = (prevPrevParagraph.length > 0) ? self.getElementAttributes(prevPrevParagraph) : { paragraph: {} };

                // in case these Attributes are undefined set them to  0
                var prevPrevMarginBottom = prevPrevAttributes.paragraph.marginBottom || 0;
                var prevTopMargin = prevAttributes.paragraph.marginTop || 0;
                var prevMarginBottom = prevAttributes.paragraph.marginBottom || 0;

                // CSS markup for the previous paragraph margin top
                var prevTopBoxShadowString = '';
                // CSS markup for previous paragraph margin down
                var prevBottomBoxShadowString = '';
                // CSS markup for current paragraph margin top
                var boxShadowStringUp = '';

                // the resulting heights for all needed boxshadows
                var colorTop  = calcColorMarginOffsetUp(prevMarginBottom, topMargin);
                var prevColorBottom = prevMarginBottom;
                var prevColorTop  = calcColorMarginOffsetUp(prevPrevMarginBottom, prevTopMargin);

                // Current paragraph:
                // check if the paragraph should have a colored marginTop and create it
                if (hasColorUp() && hasColor() && hasNoBorder(paragraphAttributes) && hasNoBorder(prevAttributes.paragraph)) {
                    boxShadowStringUp = createShadowCss(colorTop, paragraphHeight, self.getCssColor(paragraphAttributes.fillColor, 'fill'), false);
                }
                // Previous paragraph:
                // check if the previous paragraph should have a colored marginBottom and create it
                if (hasColorUp() && hasColor() && hasNoBorder(paragraphAttributes) && hasNoBorder(prevAttributes.paragraph)) {
                    prevBottomBoxShadowString = createShadowCss(prevColorBottom, prevParagraphHeight, self.getCssColor(prevAttributes.paragraph.fillColor, 'fill'), true);
                }

                // Previous paragraph:
                // check if the previous paragraph should have a colored marginTop and create it
                if (paragraphHasColor(prevPrevAttributes.paragraph) && hasColorUp() && hasNoBorder(prevAttributes.paragraph) && hasNoBorder(prevPrevAttributes.paragraph)) {
                    prevTopBoxShadowString = createShadowCss(prevColorTop, prevParagraphHeight, self.getCssColor(prevAttributes.paragraph.fillColor, 'fill'), false);
                }

                // Resulting markup for the boxshadow from the previous paragraph.
                prevBoxShadowString = concatStringsWithSeperator(prevBottomBoxShadowString, prevTopBoxShadowString);
                // Resulting markup for the boxshadow from the current paragraph.
                boxShadowString = boxShadowStringUp;

            }

            /**
            * Creating the boxshadow CSS markup downwards. That means for the current, and next paragraph.
            * For that, the next, next paragraph needs to be considered too in the calculation.
            * The resulting CSS markup is adds to related variables, that gathers the CSS markup at the end.
            *
            * @param {Number} bottomMargin
            *  The margin bottom from the current paragraph.
            *
            * @param {Number} paragraphHeight
            *  The height from the paragraph on which the shadow is appended.
            *
            * @param {Number} nextParagraphHeight
            *  The height from the next paragraph.
            */
            function setMarginColorsDownwards(bottomMargin, paragraphHeight, nextParagraphHeight) {
                // the next, next paragraph
                var nextNextParagraph = nextParagraph.next();
                // the previous, previous attributes
                var nextNextAttributes = (nextNextParagraph.length > 0) ? self.getElementAttributes(nextNextParagraph) : { paragraph: {} };

                // in case these Attributes are undefined set them to  0
                var nextBottomMargin = nextAttributes.paragraph.marginBottom || 0;
                var nextMarginTop = nextAttributes.paragraph.marginTop || 0;

                // CSS markup for next paragraph margin top
                var nextTopBoxShadowString = '';
                // CSS markup for next paragraph margin bottom
                var nextBottomBoxShadowString = '';
                // CSS markup for current paragraph margin bottom
                var boxShadowStringDown = '';

                // the resulting heights for all needed boxshadows
                var colorBottom  = bottomMargin;
                var nextColorTop = calcColorMarginOffsetUp(bottomMargin, nextMarginTop);
                var nextColorBottom  = nextBottomMargin;

                // Current paragraph:
                // check if the paragraph should have a colored marginBottom and create it
                if (hasColorDown() && hasColor() && hasNoBorder(paragraphAttributes) && hasNoBorder(nextAttributes.paragraph)) {
                    boxShadowStringDown = createShadowCss(colorBottom, paragraphHeight, self.getCssColor(paragraphAttributes.fillColor, 'fill'), true);
                }
                // Next paragraph:
                // check if the next paragraph should have a colored marginTop and create it
                if (hasColorDown() && hasColor() && hasNoBorder(paragraphAttributes) && hasNoBorder(nextAttributes.paragraph)) {
                    nextTopBoxShadowString = createShadowCss(nextColorTop, nextParagraphHeight, self.getCssColor(nextAttributes.paragraph.fillColor, 'fill'), false);
                }

                // Next paragraph:
                // check if the next paragraph should have a colored marginBottom and create it
                if (paragraphHasColor(nextNextAttributes.paragraph) && hasColorDown() && hasNoBorder(nextAttributes.paragraph) && hasNoBorder(nextNextAttributes.paragraph)) {
                    nextBottomBoxShadowString = createShadowCss(nextColorBottom, nextParagraphHeight, self.getCssColor(nextAttributes.paragraph.fillColor, 'fill'), true);
                }

                // Resulting markup for the boxshadow from the next paragraph.
                nextBoxShadowString = concatStringsWithSeperator(nextBottomBoxShadowString, nextTopBoxShadowString);
                // Resulting markup for the boxshadow from the current paragraph.
                // Note: It's concatenated with the resulting boxShadowString from 'setMarginColorsUpwards()'
                boxShadowString = concatStringsWithSeperator(boxShadowString, boxShadowStringDown);

            }

            // Performance: when the paragraph has no colored paragraph up or down, it's not needed to set colored margins
            // in this case return false to indicate that - note: by default all box shadows are removed in 'updateParagraphFormatting'
            if (!hasColorUp() && !hasColorDown()) {
                return false;
            }

            initValues();
            setMarginColorsUpwards(topMargin, paragraphHeight, prevParagraphHeight);
            setMarginColorsDownwards(bottomMargin, paragraphHeight, nextParagraphHeight);

            // when a paragraph was updated from the queue, the css must be set here, because it doesn't
            // invoke the 'CSS setter' methods in updateParagraphFormatting
            if (applyCssDirectly) {
                prevParagraph.css('box-shadow', prevBoxShadowString);
                paragraph.css('box-shadow', boxShadowString);
                nextParagraph.css('box-shadow', nextBoxShadowString);

            // when the update was called from updateParagraphFormatting, return the calculated strings for the boxshadow to color the margin
            } else {
                return { boxShadowString: boxShadowString, prevBoxShadowString: prevBoxShadowString,  nextBoxShadowString: nextBoxShadowString };
            }
        }

        /**
         * Will be called for every paragraph whose character attributes have
         * been changed.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateParagraphFormatting(paragraph, mergedAttributes) {

            var // the paragraph attributes of the passed attribute map
                paragraphAttributes = mergedAttributes.paragraph,
                // the character styles/formatter
                characterStyles = docModel.getStyleCollection('character'),

                leftMargin = 0,
                rightMargin = 0,

                leftMarginTmp = 0,
                rightMarginTmp = 0,

                prevParagraph = paragraph.prev(),
                prevAttributes = (prevParagraph.length > 0) ? self.getElementAttributes(prevParagraph) : { paragraph: {} },

                nextParagraph = paragraph.next(),
                nextAttributes = (nextParagraph.length > 0) ? self.getElementAttributes(nextParagraph) : { paragraph: {} },
                collectedAttrs =  {},
                prevCollectedAttrs =  {},
                nextCollectedAttrs = {},
                textIndent = 0,
                explicitParaAttributes = null;

            function setBorder(border, position, addSpace, para) {
                var space = addSpace + ((border && border.space) ? border.space : 0),
                    paddingPosition,
                    borderPosition;

                if (para) {
                    para.css('padding-' + position, Utils.convertHmmToCssLength(space, 'px', 1));
                    para.css('border-' + position, self.getCssBorder(border));

                } else {
                    position = position.charAt(0).toUpperCase() + position.slice(1);
                    paddingPosition = 'padding' + position;
                    borderPosition = 'border' + position;

                    collectedAttrs[paddingPosition] = Utils.convertHmmToCssLength(space, 'px', 1);
                    collectedAttrs[borderPosition] = self.getCssBorder(border);
                }

                return -space;
            }

            // Always update character formatting of all child nodes which may
            // depend on paragraph settings, e.g. automatic text color which
            // depends on the paragraph fill color. Also visit all helper nodes
            // containing text spans, e.g. numbering labels.
            Utils.iterateDescendantNodes(paragraph, function (node) {

                // visiting the span inside a hard break node
                // -> this is necessary for change tracking attributes
                if (DOM.isHardBreakNode(node)) {
                    characterStyles.updateElementFormatting(node.firstChild);
                }
                if (DOM.isComplexFieldNode(node)) {
                    characterStyles.updateElementFormatting(node);
                }

                DOM.iterateTextSpans(node, function (span) {
                    characterStyles.updateElementFormatting(span, { baseAttributes: mergedAttributes });
                });
            }, undefined, { children: true });

            // update borders
            var leftPadding = paragraphAttributes.borderLeft && paragraphAttributes.borderLeft.space ? paragraphAttributes.borderLeft.space : 0;
            leftMarginTmp = setBorder(paragraphAttributes.borderLeft, 'left', 0);
            rightMarginTmp = setBorder(paragraphAttributes.borderRight, 'right', 0);

            // Fix for BUG #36298:
            // only add the border-margins (left/right) if the paragraph isn't in a textframe
            // or if they were positive. Otherwise the left/right border will be hidden.
            if (!DOM.isNodeInsideTextFrame(paragraph) || (leftMarginTmp > 0 && rightMarginTmp > 0)) {
                leftMargin += leftMarginTmp;
                rightMargin += rightMarginTmp;
            }

            var topMargin = paragraphAttributes.marginTop;
            var bottomMargin = paragraphAttributes.marginBottom;

            // inside comments the paragraph bottom margin must be 0 (even if this was not applied via operation)
            // and the paragraph must be left aligned
            if (!docModel.getApp().isODF() && !docModel.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(paragraph)) {
                bottomMargin = 0;
                collectedAttrs.textAlign = 'left';
            }

            // top border is not set if previous paragraph uses the same border settings
            if (isMergeBorders(paragraphAttributes, prevAttributes.paragraph) && !isSetPageBreakAttribute(paragraphAttributes)) { // #36293
                setBorder({ style: 'none' }, 'top', topMargin);
                _.extend(prevCollectedAttrs, {
                    paddingBottom: self.getCssBorder(paragraphAttributes.borderInside.space + bottomMargin),
                    borderBottom: self.getCssBorder(paragraphAttributes.borderInside),
                    marginBottom: 0
                });
                topMargin = 0;
            } else {
                setBorder(paragraphAttributes.borderTop, 'top', 0);
            }

            // bottom border is replaced by inner border if next paragraph uses the same border settings
            if (isMergeBorders(paragraphAttributes, nextAttributes.paragraph) && !isSetPageBreakAttribute(nextAttributes.paragraph)) { // #36293
                setBorder(paragraphAttributes.borderInside, 'bottom');
                setBorder(paragraphAttributes.borderInside, 'top', 0, nextParagraph);  // Task 30473: Updating merged borders
                prevCollectedAttrs.paddingBottom = self.getCssBorder(bottomMargin);
                bottomMargin = 0;
            } else {
                setBorder(paragraphAttributes.borderBottom, 'bottom', 0);
            }

            //calculate list indents
            var listLabel = $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR);
            var listStyleId = paragraphAttributes.listStyleId;
            if (listStyleId.length) {
                var listLevel = paragraphAttributes.listLevel,
                    lists = docModel.getListCollection();
                if (listLevel < 0) {
                    // is a numbering level assigned to the current paragraph style?
                    listLevel = lists.findIlvl(listStyleId, mergedAttributes.styleId);
                }
                if (listLevel !== -1 && listLevel < 10) {
                    var listItemCounter = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                    var listObject = lists.formatNumber(paragraphAttributes.listStyleId, listLevel, listItemCounter, 1);

                    if (listObject.indent > 0) {
                        leftPadding += listObject.indent - listObject.firstLine;
                        leftMargin += listObject.firstLine;
                    }
                    if (listLabel.length) {
                        var listSpan = listLabel.children('span');
                        if (listObject.fontSize) {
                            listSpan.css('font-size', listObject.fontSize + 'pt');
                        }
                        if (listObject.color) {
                            listSpan.css('color', self.getCssTextColor(listObject.color, [paragraphAttributes.fillColor, listObject.fillColor]));
                        }
                    }
                } else {
                    //fix for Bug 37594 some list-helper dont disappear
                    listLabel.detach();
                }
            } else {
                //fix for Bug 37594 some list-helper dont disappear
                listLabel.detach();
            }

            // paragraph margin attributes - also applying to paragraphs in a list, if they are defined as explicit attribute
            // -> handle both cases correctly: 40792 and 41118
            if (listStyleId === '') {
                leftMargin += paragraphAttributes.indentLeft ? paragraphAttributes.indentLeft : 0;
                rightMargin += paragraphAttributes.indentRight ? paragraphAttributes.indentRight : 0;
                textIndent = paragraphAttributes.indentFirstLine ? paragraphAttributes.indentFirstLine : 0;
                collectedAttrs.textIndent = textIndent / 100 + 'mm';
            } else {
                // but explicit attributes need to be handled (40792)
                explicitParaAttributes = AttributeUtils.getExplicitAttributes(paragraph);
                explicitParaAttributes = (explicitParaAttributes && explicitParaAttributes.paragraph) || {};
                if (explicitParaAttributes.indentLeft) { leftMargin += explicitParaAttributes.indentLeft; }
                if (explicitParaAttributes.indentRight) { rightMargin += explicitParaAttributes.indentRight; }
                if (explicitParaAttributes.indentFirstLine) {
                    textIndent += explicitParaAttributes.indentRight;
                    collectedAttrs.textIndent = textIndent / 100 + 'mm';
                }
            }

            if (textIndent < 0) {
                leftPadding -= textIndent;
                leftMargin += textIndent;
            }

            _.extend(collectedAttrs, {
                paddingLeft: (leftPadding / 100) + 'mm',
                // now set left/right margins
                marginLeft: (leftMargin / 100) + 'mm',
                marginRight: (rightMargin / 100) + 'mm',
                textIndent: (textIndent / 100) + 'mm'
            });

            // Overwrite of margin left for lists in draft mode (conversion from fixed unit mm to %)
            if (Utils.SMALL_DEVICE && paragraph.data('draftRatio')) {
                collectedAttrs.marginLeft = (parseInt($(paragraph).css('margin-left'), 10) * paragraph.data('draftRatio')) + '%';
            }

            //no distance between paragraph using the same style if contextualSpacing is set
            var noDistanceToPrev = prevAttributes.paragraph.contextualSpacing && (mergedAttributes.styleId === prevAttributes.styleId),
                noDistanceToNext = paragraphAttributes.contextualSpacing && (mergedAttributes.styleId === nextAttributes.styleId);
            if (noDistanceToPrev) {
                //remove bottom margin from previous paragraph
                prevCollectedAttrs.marginBottom = 0 + 'mm';
                collectedAttrs.paddingTop = 0 + 'mm';
                topMargin = 0;
            }
            if (noDistanceToNext) {
                collectedAttrs.paddingBottom = 0 + 'mm';
                bottomMargin = 0;
            }
            _.extend(collectedAttrs, {
                marginTop: (topMargin / 100) + 'mm',
                marginBottom: (bottomMargin / 100) + 'mm'
            });

            // clear the boxShadow for colored margins as the default case for the current paragraph, to remove possible old shadows
            _.extend(collectedAttrs, {
                boxShadow: ''
            });

            // When the paragraph is updated, format colored margins when needed. The document import and undo/redo are special cases and handled elsewhere.
            if (self.isImportFinished() && !docModel.isUndoRedoRunning()) {

                // create the CSS markup for the boxshadow and re-use existing values for the calculation
                var currentShadow = setMarginColor({ paragraph: paragraph, prevParagraph: prevParagraph, prevAttributes: prevAttributes, nextParagraph: nextParagraph, nextAttributes: nextAttributes, mergedAttributes: mergedAttributes }, false);

                // when currentShadow is false, colored margins for the current, previous and next paragraph don't need to be formatted
                if (currentShadow) {
                    _.extend(collectedAttrs, {
                        boxShadow: currentShadow.boxShadowString
                    });

                    _.extend(prevCollectedAttrs, {
                        boxShadow: currentShadow.prevBoxShadowString
                    });

                    _.extend(nextCollectedAttrs, {
                        boxShadow: currentShadow.nextBoxShadowString
                    });
                }
            }

            // At import or while undo/redo it is not certain that there is a next or previous paragraph.
            // As these both are mandatory for colored margin formatting, add these paragraphs to a store
            // for later formatting, when both are accessible.
            if (!self.isImportFinished() || docModel.isUndoRedoRunning()) {
                // Important:  The next and previous paragraphs are not available at document load, therefore it's 'null' here to get a valid state. The same is true for undo/redo, the direction in which the paragraphs are processed can be up or downwards //TODO maybe check with length if existing?
                docModel.addParagraphToStore({ paragraph: paragraph, prevParagraph: null, prevAttributes: null, nextParagraph: null, nextAttributes: null, mergedAttributes: mergedAttributes });
            }

            // taking care of implicit paragraph nodes behind tables after loading the document
            // (also handling final empty paragraphs behind tables inside table cell (36537)).
            if (DOM.isIncreasableParagraph(paragraph) && (!self.isImportFinished() || !docModel.selectionInNode(paragraph))) {
                collectedAttrs.height = 0;
            }

            if (!_.browser.IE) {
                collectedAttrs.width = null;
            }

            // apply collected attributes at the end
            prevParagraph.css(prevCollectedAttrs);
            paragraph.css(collectedAttrs);
            // only set css for next paragraph when there are values for it
            if (!_.isEmpty(nextCollectedAttrs)) { nextParagraph.css(nextCollectedAttrs); }

            // update the size of all tab stops in this paragraph (but only if the paragraph contains tabs (Performance))
            if (paragraph.find(DOM.TAB_NODE_SELECTOR).length > 0) {
                updateTabStops(paragraph, mergedAttributes);
            }

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(paragraph, mergedAttributes);
        }

        /**
         * Will be called for paragraphs used as preview elements in the GUI.
         *
         * @param {jQuery} paragraph
         *  The preview paragraph node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function setParagraphPreviewFormatting(paragraph, mergedAttributes) {

            var // the character styles/formatter
                characterStyles = docModel.getStyleCollection('character');

            // format the text spans contained in the paragraph element
            paragraph.children().each(function () {
                characterStyles.updateElementFormatting(this, { baseAttributes: mergedAttributes, preview: true });
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated. If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {ParagraphStyles}
         *  A reference to this instance.
         */
        this.updateTabStops = function (paragraph) {
            updateTabStops($(paragraph), this.getElementAttributes(paragraph));
            return this;
        };
       /**
        * Make the function setMarginColor public.
        *
        * @param {Object} paraObj
        *  An Object containing needed and optional data to calculate the
        *  colored paragraph margins.
        *
        * @param {Boolean} applyCssDirectly
        *  Whether the created CSS boxshadow markup should be returned as a string
        *  or if the CSS markup should be applied directly.
        */
        this.setMarginColor = function (paraObj, applyCssDirectly) {
            setMarginColor(paraObj, applyCssDirectly);
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('paragraph', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class ParagraphStyles

    // constants --------------------------------------------------------------

    /**
     * Default solid border line for paragraphs and tables.
     *
     * @constant
     */
    ParagraphStyles.SINGLE_BORDER = { style: 'single', width: 17, space: 140, color: Color.AUTO };

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: ParagraphStyles });

});
