/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/app/controller', [
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/utils/paragraphspacing',
    'io.ox/office/editframework/app/editcontroller',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/text/format/paragraphstyles',
    'io.ox/office/textframework/format/tablestyles',
    'io.ox/office/text/format/drawingstyles'
], function (Utils, Color, HyperlinkUtils, MixedBorder, ParagraphSpacing, EditController, DrawingUtils, DrawingLabels, Config, Hyperlink, Position, ParagraphStyles, TableStyles, DrawingStyles) {

    'use strict';

    // class TextController ===================================================

    /**
     * The controller of a OX Text application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function TextController(app, docModel, docView) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditController.call(this, app, docModel, docView, { updateDelay: 200 });

        // private methods ----------------------------------------------------

        /**
         * Item getter function for a list style identifier. Will be called in
         * the context of a controller item instance.
         *
         * @param {String} listType
         *  The type of the list styles: either 'bullet' or 'numbering'.
         *
         * @param {Object} attributes
         *  The merged attribute set of the selected paragraphs.
         *
         * @returns {String|Null}
         *  If unambiguous, the list style identifier contained in the passed
         *  paragraph attributes, if the type of that list matches the list
         *  type specified in the item user data ('bullets' or 'numbering'),
         *  otherwise an empty string. If ambiguous, returns the value null.
         */
        function getListStyleId(listType, attributes) {

            var listStyleId = attributes.listStyleId,
                listLevel = attributes.listLevel,
                isListType = false,
                // the list style ID with added target (only ODT)
                shortListStyleId = listStyleId, // 44393, special behavior for list in header/footer in ODT
                // the target of the header or footer
                target = null,
                // the sub string position inside the list style ID
                index = 0;

            // special check for lists in ODT header or footer
            if (listStyleId && app.isODF() && docModel.isHeaderFooterEditState()) {
                target = docModel.getActiveTarget();
                if (target) {
                    target = '_' + target;
                    index = listStyleId.indexOf(target);
                    if (index > 0) {
                        shortListStyleId = listStyleId.substring(0, index);
                    }
                }
            }

            // ambiguous attributes
            if (_.isNull(listStyleId) || _.isNull(listLevel)) { return null; }

            if (listLevel > -1) {
                switch (listType) {
                    case 'bullet':
                        isListType = docModel.getListCollection().isBulletsList(listStyleId, listLevel);
                        break;
                    case 'numbering':
                        isListType = docModel.getListCollection().isNumberingList(listStyleId, listLevel);
                        break;
                }
            }

            // return empty string, if list type in attributes does not match list type in item user data
            return isListType ? shortListStyleId : '';
        }

        /**
         * Item setter function for a list style identifier. Will be called in
         * the context of a controller item instance.
         *
         * @param {String} listType
         *  The type of the list styles: either 'bullet' or 'numbering'.
         *
         * @param {String} listStyleId
         *  The new list style identifier. The special value
         *  Lists.DEFAULT_VALUE can be used to toggle the default list style.
         */
        function setListStyleId(listType, listStyleId) {

            // simulate toggle behavior for default list style
            if (listStyleId === docModel.getDefaultListValue()) {
                if (this.getValue() === '') {
                    docModel.createDefaultList(listType);
                } else {
                    docModel.removeListAttributes();
                }
            } else {
                // list level may be null, will fall back to level 0 then...
                docModel.createSelectedListStyle(listStyleId, self.getItemValue('paragraph/list/indent'));
            }
        }

        // initialization -----------------------------------------------------

        if (_.browser.Android) {
            var originalUpdate = self.update;

            self.update = function () {
                if (docModel.isAndroidTyping()) {
                    self.executeDelayed(self.update, 1000, 'Text: android: update');
                } else {
                    originalUpdate();
                }
            };
        }

        // register item definitions
        this.registerDefinitions({

            // view -------------------------------------------------------

            'view/zoom': {
                parent: 'app/valid',
                get: function () { return docView.getZoomFactor() / 100; },
                set: function (zoom) { docView.setZoomType(zoom * 100); }
            },

            'view/zoom/dec': {
                parent: 'app/valid',
                enable: function () { return docView.getZoomFactor() > docView.getMinZoomFactor(); },
                set: function () { docView.decreaseZoomLevel(); }
            },

            'view/zoom/inc': {
                parent: 'app/valid',
                enable: function () { return docView.getZoomFactor() < docView.getMaxZoomFactor(); },
                set: function () { docView.increaseZoomLevel(); }
            },

            'view/zoom/type': {
                parent: 'app/valid',
                get: function () { return docView.getZoomType(); },
                set: function (zoomType) { docView.setZoomType(zoomType); }
            },

            'document/editable/text': {
                parent: 'document/editable',
                enable: function () { return docModel.isTextSelected() || docModel.getSelection().isAnyTextFrameDrawingSelection(); }
            },

            'document/editable/text/notextframe': {
                parent: 'document/editable/text',
                enable: function () {
                    return !docModel.getSelection().isAdditionalTextframeSelection();
                }
            },

            'document/selectall': {
                // enabled in read-only mode
                set: function () { docModel.selectAll(); },
                // restrict keyboard shortcut to application pane
                shortcut: { keyCode: 'A', ctrlOrMeta: true, selector: '.app-pane' },
                preserveFocus: true // do not return focus to document
            },

            'document/selectDrawing': {
                set: function (options) {
                    docModel.getSelection().selectNextDrawing({ backwards: Utils.getBooleanOption(options, 'backwards', false) });
                },
                shortcut: { keyCode: 'F4', shift: true }
            },
            'document/setCursorIntoTextframe': {
                set: function () {
                    docModel.getSelection().setSelectionIntoTextframe();
                },
                shortcut: { keyCode: 'F2' }
            },

            // document contents
            'document/cut': {
                parent: 'document/editable',
                enable: function () { return docModel.hasSelectedRange(); },
                set: function () { docModel.cut(); }
            },
            'document/copy': {
                // enabled in read-only mode
                enable: function () { return docModel.hasSelectedRange(); },
                set: function () { docModel.copy(); }
            },
            'document/paste': {
                parent: 'document/editable',
                enable: function () { return docModel.hasInternalClipboard(); },
                set: function () { docModel.pasteInternalClipboard(); }
            },

            // spelling

            'document/spelling/available': {
                parent: 'document/editable',
                enable: function () { return Config.SPELLING_ENABLED; }
            },
            'document/onlinespelling': {
                parent: 'document/spelling/available',
                get: function () { return docModel.getSpellChecker().isOnlineSpelling(); },
                set: function (state) { docModel.getSpellChecker().setOnlineSpelling(state); }
            },
            'document/spelling/replace': {
                parent: 'document/editable',
                set: function (replacement) {
                    var selection = docModel.getSelection(),
                        startPosition = selection.getStartPosition(),
                        wordPos = Position.getWordSelection(selection.getEnclosingParagraph(), startPosition[startPosition.length - 1]);

                    docModel.getSpellChecker().replaceWord(startPosition, wordPos, replacement);
                }
            },
            'document/spelling/ignoreword': {
                parent: 'document/editable',
                set: function (options) {
                    return docModel.getSpellChecker().addWordToIgnoreSpellcheck(options);
                }
            },
            'document/spelling/userdictionary': {
                parent: 'document/editable',
                set: function (options) {
                    return docModel.getSpellChecker().addWordToIgnoreSpellcheck(options);
                }
            },
            // Page settings

            'document/pagesettings': {
                parent: 'document/editable/text',
                set: function () { return docView.showPageSettingsDialog(); }
            },

            // paragraphs

            'paragraph/textonly/supported': {
                enable: function () { return !docModel.useSlideMode(); }
            },

            'paragraph/group': {
                parent: 'document/editable/text',
                get: function () { return docModel.getAttributes('paragraph', { maxIterations: 10 }) || {}; }
            },
            'paragraph/stylesheet': {
                parent: 'paragraph/group',
                get: function (paragraph) { return paragraph.styleId; },
                set: function (styleId) {
                    // TODO: hacked code for function "create new paragraph stylesheet"
                    if (styleId === '\x00paragraph/createstylesheet') {
                        return docModel.showInsertNewStyleDialog();
                    }
                    docModel.setAttributes('paragraph', { styleId: styleId }, { clear: true });
                }
            },
            'paragraph/createstylesheet': {
                parent: ['paragraph/group', 'character/group'],
                /*
                enable: function (paragraphHolder, characterHolder) {
                    var character = characterHolder.character;
                    var paragraph = paragraphHolder.character;
                    for (var key in character) {
                        if (_.isNull(character[key])) {return false;}
                    }
                    if (_.isEqual(paragraph, character)) {
                        return false;
                    }
                    return true;
                },
                */
                set: function () { return docModel.showInsertNewStyleDialog(); }
            },
            'paragraph/renamestylesheet': {
                set: function (styleId) { return docModel.showRenameStyleDialog(styleId); }
            },
            'paragraph/changestylesheet': {
                parent: ['paragraph/group', 'character/group', 'paragraph/textonly/supported'],
                set: function (styleId) { return docModel.changeStylesheet(styleId); }
            },
            'paragraph/deletestylesheet': {
                parent: ['paragraph/group', 'character/group', 'paragraph/textonly/supported'],
                set: function (styleId) { return docModel.deleteStylesheet(styleId); }
            },
            'paragraph/attributes': {
                parent: 'paragraph/group',
                get: function (paragraph) { return paragraph.paragraph || {}; }
            },
            'paragraph/alignment': {
                parent: ['paragraph/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return attributes.alignment; },
                set: function (alignment) { docModel.setAttribute('paragraph', 'alignment', alignment); }
                // removed for the 7.2.0 due to clashes with browser shortcuts
                /*shortcut: [
                    { keyCode: 'L', ctrlOrMeta: true, value: 'left' },
                    { keyCode: 'R', ctrlOrMeta: true, value: 'right' },
                    { keyCode: 'E', ctrlOrMeta: true, value: 'center' },
                    { keyCode: 'J', ctrlOrMeta: true, value: 'justify' }
                ]*/
            },
            'paragraph/lineheight': {
                parent: ['paragraph/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return attributes.lineHeight; },
                set: function (lineHeight) { docModel.setAttribute('paragraph', 'lineHeight', lineHeight); }
                // removed for the 7.2.0 due to clashes with browser shortcuts
                /*shortcut: [
                    { keyCode: '1', ctrlOrMeta: true, value: LineHeight.SINGLE },
                    { keyCode: '5', ctrlOrMeta: true, value: LineHeight.ONE_HALF },
                    { keyCode: '2', ctrlOrMeta: true, value: LineHeight.DOUBLE }
                ]*/
            },
            'paragraph/spacing': {
                parent: ['paragraph/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return ParagraphSpacing.helper.computeMultiplier(attributes, docModel); },
                set: function (multiplier) {
                    var marginBottom = ParagraphSpacing.helper.computeMarginBottom(multiplier, docModel);

                    // bug 46249:  margin top needs to be considered when 'Paragraph spacing' is set to none in our GUI
                    var marginTop = ParagraphSpacing.helper.getZeroSpaceMargin();
                    docModel.setAttributes('paragraph', { paragraph: { marginTop: marginTop, marginBottom: marginBottom } });
                }
            },
            'paragraph/fillcolor': {
                parent: ['paragraph/attributes', 'insert/comment/odf/supported'],
                get: function (attributes) { return attributes.fillColor; },
                set: function (color) { docModel.setAttribute('paragraph', 'fillColor', color); }
            },
            'paragraph/borders': {
                parent: 'paragraph/attributes',
                get: function (attributes) { return MixedBorder.getBorderMode(attributes, { paragraph: true }); },
                set: function (borderMode) {
                    var borderAttrs = MixedBorder.getBorderAttributes(borderMode, this.getParentValue(), ParagraphStyles.SINGLE_BORDER, { paragraph: true });
                    docModel.setAttributes('paragraph', { paragraph: borderAttrs });
                }
            },

            // toggle default bullet list, or select bullet type
            'paragraph/list/bullet': {
                parent: ['paragraph/attributes', 'insert/comment/odf/supported'],
                get: _.partial(getListStyleId, 'bullet'),
                set: _.partial(setListStyleId, 'bullet')
            },
            // toggle default numbered list, or select numbering type
            'paragraph/list/numbered': {
                parent: ['paragraph/attributes', 'insert/comment/odf/supported'],
                get: _.partial(getListStyleId, 'numbering'),
                set: _.partial(setListStyleId, 'numbering')
            },

            // parent for controller items only enabled inside lists
            'paragraph/list/enabled': {
                parent: 'paragraph/attributes',
                enable: function () {
                    var indent = this.getValue().listLevel;
                    return _.isNumber(indent) && (indent >= 0) && (indent <= 8);
                }
            },
            // change list level
            'paragraph/list/indent': {
                parent: 'paragraph/list/enabled',
                get: function (attributes) { return attributes.listLevel; },
                set: function (indent) { docModel.setAttribute('paragraph', 'listLevel', indent); }
            },
            // increase list level by one
            'paragraph/list/incindent': {
                parent: 'paragraph/list/indent',
                enable: function () { return this.getValue() < 8; },
                set: function () {
                    var indent = this.getValue();
                    if (indent < 8) {
                        docModel.setAttribute('paragraph', 'listLevel', indent + 1);
                    }
                }
            },
            // decrease list level by one
            'paragraph/list/decindent': {
                parent: 'paragraph/list/indent',
                enable: function () { return this.getValue() > 0; },
                set: function () {
                    var indent = this.getValue();
                    if (indent > 0) {
                        docModel.setAttribute('paragraph', 'listLevel', indent - 1);
                    }
                }
            },

            // characters

            'character/group': {
                parent: 'document/editable/text',
                get: function () { return docModel.getAttributes('character', { maxIterations: 10 }) || {}; }
            },
            'character/stylesheet': {
                parent: 'character/group',
                get: function (character) { return character.styleId; },
                set: function (styleId) { docModel.setAttributes('character', { styleId: styleId }, { clear: true }); }
            },
            'character/attributes': {
                parent: 'character/group',
                get: function (character) { return character.character || {}; }
            },
            'character/fontname': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.fontName; },
                set: function (fontName) { docModel.setAttribute('character', 'fontName', fontName); }
            },
            'character/fontsize': {
                parent: ['character/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return attributes.fontSize; },
                set: function (fontSize) { docModel.setAttribute('character', 'fontSize', fontSize); }
            },
            'character/bold': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.bold; },
                set: function (state) { docModel.setAttribute('character', 'bold', state); },
                shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
            },
            'character/italic': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.italic; },
                set: function (state) { docModel.setAttribute('character', 'italic', state); },
                shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
            },
            'character/underline': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.underline; },
                set: function (state) { docModel.setAttribute('character', 'underline', state); },
                shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
            },
            'character/strike': {
                parent: 'character/attributes',
                get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                set: function (state) { docModel.setAttribute('character', 'strike', state ? 'single' : 'none'); }
            },
            'character/vertalign': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.vertAlign; },
                set: function (align) { docModel.setAttribute('character', 'vertAlign', align); }
            },
            'character/color': {
                parent: ['character/attributes', 'insert/comment/odf/supported'],
                get: function (attributes) { return attributes.color; },
                set: function (color) { docModel.setAttribute('character', 'color', color); }
            },
            'character/fillcolor': {
                parent: ['character/attributes', 'insert/comment/odf/supported'],
                get: function (attributes) { return attributes.fillColor; },
                set: function (color) { docModel.setAttribute('character', 'fillColor', color); }
            },
            'character/language': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.language; },
                set: function (language) { docModel.setAttribute('character', 'language', language); }
            },
            'character/hyperlink': {
                parent: 'character/attributes',
                enable: function () { return docModel.hasEnclosingParagraph(); },
                get: function (attributes) { return attributes.url; }
                // no direct setter (see item 'character/hyperlink/dialog')
            },
            'character/hyperlink/dialog': {
                parent: 'character/hyperlink',
                get: function () { return Hyperlink.hasPopup(docModel); },
                set: function () { return docModel.insertHyperlinkDialog(); }
                // removed for the 7.2.0 due to clashes with browser shortcuts
                //shortcut: { keyCode: 'K', ctrlOrMeta: true }
            },
            'character/hyperlink/remove': {
                parent: 'character/hyperlink',
                set: function () { return docModel.removeHyperlink(); }
            },

            'character/reset': {
                parent: 'document/editable/text',
                set: function () { docModel.resetAttributes(); },
                shortcut: { keyCode: 'SPACE', ctrl: true }
            },

            'character/hyperlink/valid': {
                parent: 'character/attributes',
                enable: function (attrs) { return attrs.url && HyperlinkUtils.hasSupportedProtocol(attrs.url) && HyperlinkUtils.isValidURL(attrs.url); }
            },

            'character/format': {
                parent: 'app/valid'
            },

            'character/insert/break': {
                parent: 'document/editable/text',
                set: function () { return docModel.insertHardBreak(); }
            },
            'character/insert/tab': {
                parent: 'document/editable/text',
                set: function () { return docModel.insertTab(); }
            },
            'character/insert/pagebreak': {
                parent: ['document/editable/text/notextframe', 'insert/comment/supported'],
                enable: function () { return !docModel.isHeaderFooterEditState(); },
                set: function () { return docModel.insertManualPageBreak(); }
            },

            // header-footer
            'document/insert/headerfooter': {
                parent: ['document/editable', 'insert/comment/supported'],
                get: function () { return docModel.getPageLayout().getHeaderFooterTypeInDoc(); },
                set: function (value) { docModel.getPageLayout().setHeaderFooterTypeInDoc(value); }
            },

            // header-footer remove all
            'document/insert/headerfooter/remove': {
                parent: ['document/editable', 'insert/comment/supported'],
                enable: function () { return docModel.getPageLayout().isHeaderFooterActivated(); },
                set: function () { docModel.getPageLayout().setHeaderFooterTypeInDoc('none'); }
            },

            // reduced odf text frame functionality
            'insert/textframe/supported': {
                enable: function () { return !docModel.isReducedOdfTextframeFunctionality(); }
            },

            // reduced odf comment functionality
            'insert/comment/odf/supported': {
                enable: function () { return !docModel.isOdfCommentFunctionality(); }
            },

            'insert/comment/supported': {
                enable: function () { return !docModel.isCommentFunctionality(); }
            },

            'insert/comment/supported/odf': {
                enable: function () { return app.isODF() || !docModel.isCommentFunctionality(); }
            },

            'insert/margin/supported': {
                enable: function () { return !docModel.isHeaderFooterEditState(); }
            },

            'insert/margin/supported/odf': {
                enable: function () { return app.isODF() || !docModel.isHeaderFooterEditState(); }
            },

            // tables

            'table/group': {
                parent: 'document/editable/text',
                get: function () { return docModel.getAttributes('table') || {}; }
            },
            'table/insert/available': {
                enable: function () { return Config.MAX_TABLE_CELLS > 0; }
            },
            'table/insert': {
                parent: ['document/editable/text', 'table/insert/available', 'insert/textframe/supported', 'insert/comment/odf/supported'],
                set: function (size) { docModel.insertTable(size); },
                preserveFocus: true // do not return focus to document
            },

            'document/editable/table': {
                parent: 'table/group',
                enable: function () { return docModel.isPositionInTable(); }
            },
            'document/show/tabletab': {
                parent: 'document/editable/table',
                enable: function () { return docModel.isCellRangeSelected() || (!docModel.isCellRangeSelected() && !docModel.hasSelectedRange()); }
            },
            'table/delete': {
                parent: 'document/editable/table',
                set: function () { return docModel.deleteTable(); }
            },
            'table/insert/row': {
                parent: 'document/editable/table',
                set: function () { docModel.insertRow(); }
            },
            'table/insert/column': {
                parent: 'document/editable/table',
                set: function () { docModel.insertColumn(); }
            },
            'table/delete/row': {
                parent: 'document/editable/table',
                set: function () { return docModel.deleteRows(); }
            },
            'table/delete/column': {
                parent: 'document/editable/table',
                set: function () { return docModel.deleteColumns(); }
            },

            'table/stylesheet': {
                parent: 'document/editable/table',
                get: function (table) { return table.styleId; },
                set: function (styleId) { docModel.setAttributes('table', { styleId: styleId }, { clear: true }); }
            },
            'table/attributes': {
                parent: 'document/editable/table',
                get: function (table) { return table.table || {}; }
            },
            'table/cellattributes': {
                parent: 'document/editable/table',
                get: function () { return docModel.getAttributes('cell').cell || {}; }
            },
            'table/cellborder': {
                parent: 'table/attributes',
                get: function (attributes) { return MixedBorder.getBorderMode(attributes); },
                set: function (borderMode) {
                    var borderAttrs = MixedBorder.getBorderAttributes(borderMode, this.getParentValue(), ParagraphStyles.SINGLE_BORDER);
                    docModel.setAttributes('table', { table: borderAttrs }, { cellSpecificTableAttribute: true });
                }
            },
            'table/borderwidth': {
                parent: 'table/cellattributes',
                get: function (attributes) { return TableStyles.getBorderStyleFromAttributes(attributes); },
                set: function (borderWidth) {
                    docModel.setAttributes('table', {
                        table: TableStyles.getAttributesFromBorderStyle(borderWidth, docModel.getAttributes('table').table)
                    }, { onlyVisibleBorders: true });
                }
            },
            'table/fillcolor': {
                parent: 'table/cellattributes',
                get: function (attributes) { return attributes.fillColor; },
                set: function (color) { docModel.setAttribute('cell', 'fillColor', color); }
            },
            'table/split': {
                parent: 'document/editable/table',
                set: function () { docModel.splitTable(); }
            },

            // drawing

            'document/editable/drawing': {
                parent: 'document/editable',
                enable: function () { return docModel.isDrawingSelected(); }
            },
            'document/editable/anydrawing': {
                parent: 'document/editable',
                enable: function () { return docModel.getSelection().isAnyDrawingSelection(); }
            },
            'document/editable/onlysupporteddrawings': {
                parent: 'document/editable',
                enable: function () { return docModel.getSelection().areOnlySupportedDrawingsSelected(); }
            },
            'drawing/shapes/support/fill': {
                parent: 'document/editable/onlysupporteddrawings',
                enable: function () {
                    var selection = docModel.getSelection();
                    return (/^(shape|group)$/).test(selection.getSelectedDrawingType()) || (selection.getSelectionType() === 'text');
                }
            },
            'drawing/shapes/support/line': {
                parent: 'document/editable/onlysupporteddrawings',
                enable: function () {
                    var selection = docModel.getSelection();
                    return (/^(shape|group|image)$/).test(selection.getSelectedDrawingType()) || (selection.getSelectionType() === 'text');
                }
            },

            'drawing/attributes/full': {
                parent: 'document/editable/anydrawing',
                get: function () { return docModel.getAttributes('drawing') || {}; }
            },

            'drawing/attributes': {
                parent: 'drawing/attributes/full',
                get: function (attributes) { return attributes.drawing || {}; }
            },
            'drawing/group/attributes': {
                parent: 'document/editable/anydrawing',
                get: function () { return docModel.getAttributes('drawing', { groupOnly: true }).drawing || {}; }
            },

            'drawing/delete': {
                parent: 'document/editable/anydrawing',
                set: function () {
                    var selection = docModel.getSelection();
                    if (docModel.isDrawingSelected()) {
                        return docModel.deleteSelected();

                    } else if (selection.isAdditionalTextframeSelection()) {
                        var start = Position.getOxoPosition(docModel.getCurrentRootNode(), selection.getSelectedTextFrameDrawing(), 0),
                            end = Position.increaseLastIndex(start);

                        selection.setTextSelection(start, end);
                        return docModel.deleteSelected();
                    }
                }
            },
            'drawing/anchorTo': {
                parent: 'drawing/group/attributes',
                enable: function () {
                    var selection = docModel.getSelection();
                    return !selection.isDrawingInTextFrameSelection();

                    // TODO: Disabled, OpenOffice moves the selected Drawing out of its TextFrame!
                    // return !selection.isDrawingInTextFrameSelection() || selection.isDrawingInODFTextFrameSelection();
                },
                get: function (attributes) { return (attributes.inline) ? 'inline' : attributes.anchorVertBase; },
                set: function (value) { docModel.anchorDrawingTo(value); }
            },
            'drawing/position': {
                parent: 'drawing/group/attributes',
                enable: function (attributes) { return (!attributes.inline); },
                get: function (attributes) { return DrawingStyles.getPositionFromAttributes(attributes); },
                set: function (position) {
                    docModel.setAttributes('drawing', {
                        drawing: DrawingStyles.getAttributesFromPosition(position, docModel.getAttributes('drawing').drawing)
                    }, {
                        ctSupportsDrawingPosition: docModel.getChangeTrack().ctSupportsDrawingPosition()
                    });
                }
            },

            // drawing z-order

            'drawing/order': {
                parent: 'drawing/position',
                set: function (value) { docModel.getDrawingLayer().updateDrawingOrder(value); }
            },

            'drawing/type/label': {
                parent: 'document/editable/drawing',
                get: function () { return DrawingLabels.getDrawingTypeLabel(docModel.getSelection().getClosestSelectedDrawingType()); }
            },

            'drawing/textframeautofit': {
                enable: function () { return docModel.getSelection().isAnyTextFrameSelection({ allDrawingTypesAllowed: true }); },
                get: function () { return docModel.isAutoResizableTextFrame(); },
                set: function (state) { docModel.handleTextFrameAutoFit(state); }
            },

            // drawing fill formatting

            'drawing/fill/attributes': {
                parent: 'drawing/attributes/full',
                get: function (attributes) { return attributes.fill || {}; }
            },

            'drawing/fill/color': {
                parent: 'drawing/fill/attributes',
                get: function (attributes) { return attributes.color; },
                set: function (color) { docModel.setDrawingFillColor(color); }
            },

            // drawing line formatting

            'drawing/line/attributes': {
                parent: 'drawing/attributes/full',
                get: function (attributes) { return attributes.line || {}; }
            },

            'drawing/border/style': {
                parent: 'drawing/line/attributes',
                get: function (attributes) { return DrawingUtils.getPresetBorder(attributes); },
                set: function (border) { docModel.setDrawingBorder(border); }
            },

            'drawing/border/color': {
                parent: 'drawing/line/attributes',
                get: function (attributes) { return (attributes.type !== 'none') ? attributes.color : Color.AUTO; },
                set: function (color) { docModel.setDrawingBorderColor(color); }
            },

            // text frames

            'textframe/insert': {
                parent: ['document/editable/text/notextframe', 'insert/comment/supported'],
                set: function () { docModel.insertTextFrame(); }
            },

            // comments

            'comment/insert': {
                parent: ['document/editable/text/notextframe', 'insert/comment/supported', 'insert/margin/supported'],
                set: function () { docModel.getCommentLayer().insertComment(); }
            },

            'comment/displayModeParent': {
                enable: function () { return !docModel.getCommentLayer().isEmpty(); }
            },

            'comment/displayMode': {
                get: function () { return docModel.getCommentLayer().getDisplayMode(); },
                set: function (mode) { docModel.getCommentLayer().switchCommentDisplayMode(mode); }
            },

            'comment/authorFilter': {
                get: function () { return docModel.getCommentLayer().getListOfUnfilteredAuthors(); },
                set: function (authorList) { docModel.getCommentLayer().setAuthorFilter(authorList); }
            },

            'comment/nextComment': {
                // enabled in read-only mode
                enable: function () { return !docModel.getCommentLayer().isEmpty(); },
                set: function () { docModel.getCommentLayer().selectNextComment({ next: true }); }
            },
            'comment/prevComment': {
                // enabled in read-only mode
                enable: function () { return !docModel.getCommentLayer().isEmpty(); },
                set: function () { docModel.getCommentLayer().selectNextComment({ next: false }); }
            },

            'comment/deleteAll': {
                parent: ['document/editable'],
                enable: function () { return !docModel.getCommentLayer().isEmpty(); },
                set: function () { docModel.getCommentLayer().deleteComment(null, { all: true }); }
            },

            // show image drive, local or url dialog
            'image/insert/dialog': {
                parent: ['document/editable/text', 'insert/textframe/supported', 'insert/comment/odf/supported'],
                set: function (dialogType) { return docModel.showInsertImageDialog(dialogType); }
            },

            // change tracking

            checkCurrentChangeTracking: {
                parent: 'document/editable',
                enable: function () { return docModel.getChangeTrack().isOnChangeTrackNode(); }
            },
            toggleChangeTracking: {
                parent: 'document/editable',
                get: function () { return docModel.getChangeTrack().isActiveChangeTrackingInDocAttributes(); },
                set: function (state) { docModel.getChangeTrack().setActiveChangeTracking(state); }
            },
            acceptChangeTracking: {
                parent: 'document/editable',
                set: function () {
                    docModel.getChangeTrack().resolveChangeTracking(null, { accepted: true, all: true });
                    docView.getChangeTrackPopup().hide();
                }
            },
            rejectChangeTracking: {
                parent: 'document/editable',
                set: function () {
                    docModel.getChangeTrack().resolveChangeTracking(null, { accepted: false, all: true });
                    docView.getChangeTrackPopup().hide();
                }
            },
            acceptSelectedChangeTracking: {
                parent: 'checkCurrentChangeTracking',
                set: function () {
                    docModel.getChangeTrack().resolveChangeTracking(null, { accepted: true, all: false });
                    docView.getChangeTrackPopup().hide();
                }
            },
            acceptMoveSelectedChangeTracking: {
                parent: 'document/editable',
                set: function () {
                    docModel.getChangeTrack().resolveChangeTracking(null, { accepted: true, all: false });
                    docView.getChangeTrackPopup().hide();
                    docModel.getChangeTrack().getNextChangeTrack({ reuseSameNode: true });
                }
            },
            rejectSelectedChangeTracking: {
                parent: 'checkCurrentChangeTracking',
                set: function () {
                    docModel.getChangeTrack().resolveChangeTracking(null, { accepted: false, all: false });
                    docView.getChangeTrackPopup().hide();
                }
            },
            rejectMoveSelectedChangeTracking: {
                parent: 'document/editable',
                set: function () {
                    docModel.getChangeTrack().resolveChangeTracking(null, { accepted: false, all: false });
                    docView.getChangeTrackPopup().hide();
                    docModel.getChangeTrack().getNextChangeTrack({ reuseSameNode: true });
                }
            },
            selectNextChangeTracking: {
                parent: 'app/valid',
                set: function () { docModel.getChangeTrack().getNextChangeTrack({ next: true }); }
            },
            selectPrevChangeTracking: {
                parent: 'app/valid',
                set: function () { docModel.getChangeTrack().getNextChangeTrack({ next: false }); }
            },
            changetrackPopupShow: {
                parent: 'app/valid',
                set: function () { docView.changeTrackPopupSelect(); }
            },

            // debug
            'debug/uselocalstorage': {
                get: function () { return app.getUserSettingsValue('debugUseLocalStorage', Config.USE_LOCAL_STORAGE); },
                set: function (state) { app.setUserSettingsValue('debugUseLocalStorage', state); }
            },
            'debug/useFastLoad': {
                get: function () { return app.getUserSettingsValue('useFastLoad', true); },
                set: function (state) { app.setUserSettingsValue('useFastLoad', state); }
            },
            'debug/recordoperations': {
                get: function () { return docModel.isRecordingOperations(); },
                set: function (state) { docModel.setRecordingOperations(state); }
            },
            'debug/replayoperations': {
                parent: 'document/editable',
                enable: function () { return docView.getDebugOperationsContent() !== null; },
                set: function () { return docModel.replayOperations(docView.getDebugOperationsContent(), docView.getDebugOperationsOSN()); }
            },
            'debug/pagebreaks/toggle': {
                parent: 'debug/enabled',
                enable: function () { return !Utils.SMALL_DEVICE && !docModel.isDraftMode(); },
                get: function () { return docModel.isPageBreakMode(); },
                set: function (state) { docModel.togglePageBreakMode(state); }
            },
            'debug/draftmode/toggle': {
                parent: 'debug/enabled',
                enable: function () { return !Utils.SMALL_DEVICE; },
                get: function () { return docModel.isDraftMode(); },
                set: function (state) { docModel.toggleDraftMode(state); }
            },
            'debug/deleteHeaderFooter': {
                parent: 'document/editable',
                enable: function () { return docModel.isHeaderFooterEditState(); },
                set: function () { docModel.getPageLayout().removeActiveHeaderFooter(); }
            },
            'debug/commentView': {
                get: function () { return docModel.getCommentLayer().getCommentDisplayView(); },
                set: function (mode) { docModel.getCommentLayer().switchCommentDisplayView(mode); }
            },

            // fields
            updateField: {
                parent: 'document/editable',
                set: function () { docModel.getFieldManager().updateHighlightedField(); }
            },
            updateAllFields: {
                parent: 'document/editable',
                set: function () { docModel.getFieldManager().updateAllFields(); }
            },
            removeField: {
                parent: 'document/editable',
                set: function () { docModel.getFieldManager().removeField(); }
            },

            'document/insertfield': {
                parent: 'document/editable/text',
                set: function (value) { docModel.getFieldManager().dispatchInsertField(value, 'default'); }
            },

            'document/formatfield': {
                parent: 'document/editable',
                get: function () { return docModel.getFieldManager().getSelectedFieldFormat(); },
                set: function (value) { docModel.getFieldManager().updateFieldFormatting(value); }
            },

            'document/toggledatepicker': {
                parent: 'document/editable',
                set: function () { docView.toggleDatePicker(); }
            },

            // android-entry for selection which cant be deleted by softkeyboard
            'document/editable/selection': {
                parent: 'document/editable',
                enable: function () {
                    var selection = docModel.getSelection();
                    return !_.isEqual(selection.getStartPosition(), selection.getEndPosition());
                }
            },
            'selection/delete': {
                parent: 'document/editable/selection',
                set: function () { return docModel.deleteSelected(); }
            }
        });

        // update GUI after changed selection
        this.listenTo(docModel, 'selection', function () { self.update(); });

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = docView = null;
        });

    } // class TextController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: TextController });

});
