/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/sortlayermenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/checkbox',
    'io.ox/office/tk/control/radiogroup',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/baseframework/view/popup/layermenu',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, CheckBox, RadioGroup, RadioList, LayerMenu, Labels, gt) {

    'use strict';

    // class SortLayerMenu ====================================================

    /**
     * A layer menu with options for sorting.
     *
     * @constructor
     *
     * @extends LayerMenu
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    function SortLayerMenu(docView) {

        var // self reference
            self = this,

            hasHeadersCheckbox = null,
            directionGroup = null,
            keyRadioList = null,
            orderRadioList = null,

            hasHeaders = false,
            currentDirection = 'vertical',
            currentOrder = null,
            currentKey = null,

            maxCount = 2000;

        // base constructor ---------------------------------------------------

        LayerMenu.call(this, docView, /*#. menu title: custom sort options */ gt.pgettext('sort', 'Custom Sort'), {
            actionKey: 'cell/sort',
            actionValue: function () {
                return {
                    direction: currentDirection,
                    order: currentOrder,
                    colrow: currentKey,
                    headers: hasHeaders
                };
            },
            rootContainerNode: docView.getApp().getWindowNode()
        });

        // private methods ----------------------------------------------------

        function fillRadioList(columns) {

            var interval = docView.getSelectedRanges().first().interval(columns),
                from = interval.first,
                to = interval.last,
                controllerDefault = null;

            // remove (old) column/row values. New selection will have new values
            keyRadioList.getMenu().clearContents();

            if (interval.size() > maxCount) { return false; }

            for (var index = from; index <= to; index += 1) {
                // set default selection to the first entry
                if (_.isNull(controllerDefault)) { controllerDefault = index; }

                keyRadioList.createOptionButton(index, { label: columns ? Labels.getColLabel(index) : Labels.getRowLabel(index) });
            }
        }

        /**
         * Initializes the columns/rows radio list, after the sorting direction
         * has been changed.
         */
        function changeDirectionHandler(event, val) {
            if (currentDirection !== val) {
                currentDirection = val;
                fillRadioList(currentDirection === 'vertical');

                keyRadioList.setValue(docView.getActiveCell().get((currentDirection === 'vertical')));
                currentKey = docView.getActiveCell().get((currentDirection === 'vertical'));
            }
        }

        function changeOrderHandler(event, val) {
            if (currentOrder !== val) {
                currentOrder = val;
                orderRadioList.getMenu().hide();
            }
        }

        function changeKeyHandler(event, val) {
            if (currentKey !== val) {
                currentKey = val;
                keyRadioList.getMenu().hide();
            }
        }

        function changeHeadersHandler(event, val) {
            if (hasHeaders !== val) {
                hasHeaders = val;
            }
        }

        /**
         * Closes the menu when the selection in the document has been changed.
         */
        function changeSelectionHandler() {
            // get out here, if the custom sort layer menu isn't visible
            if (!self.isVisible()) { return; }
            self.hide();
        }

        /**
         * Fills the radio lists before the menu will be shown.
         */
        function popupBeforeShowHandler() {
            var currentRange = docView.getSelectedRanges();

            currentDirection    = 'vertical';
            hasHeaders          = docView.getCellCollection().hasRangeHeadline(currentRange[0], currentDirection);
            currentOrder        = docView.getSortState();
            currentKey          = docView.getActiveCell().get((currentDirection === 'vertical'));

            directionGroup.setValue(currentDirection);
            orderRadioList.setValue(currentOrder);
            keyRadioList.setValue(currentKey);
            hasHeadersCheckbox.setValue(hasHeaders);

            fillRadioList(true);
        }

        /**
         * Moves the menu to a default position when it has been shown for the
         * first time.
         */
        function setDefaultPosition() {
            var pos = Utils.getNodePositionInPage(docView.getAppPaneNode());
            self.setNodePosition(pos.left + (pos.width - self.getNode().outerWidth()) / 2, pos.top + 40);
        }

        // initialization -----------------------------------------------------

        // add base menu items
        this.addGroup(null, hasHeadersCheckbox = new CheckBox({ label: /*#. sorting: selected range has headers which shouldn't be sorted */ gt.pgettext('sort', 'Selection has headers'), boxed: true }))
            .addSectionLabel(/*#. sorting: the sort direction headline for 'top to bottom' or 'left to right' */ gt.pgettext('sort', 'Direction'))
            .addGroup(null, directionGroup = new RadioGroup({ label: gt.pgettext('sort', 'Direction'), radioIcons: true })
                .createOptionButton('vertical', { label: /*#. sort direction 'Top to bottom' */ gt.pgettext('sort', 'Top to bottom') })
                .createOptionButton('horizontal', { label: /*#. sort direction 'Left to right' */ gt.pgettext('sort', 'Left to right') }),
            { inline: true })
            .addSectionLabel(/*#. head line: 'sort by' options (list of column names, e.g. 'Column C') */ gt.pgettext('sort', 'Sort by'))
            .addGroup(null, keyRadioList = new RadioList({ style: 'width: 150px;', updateCaptionMode: 'label' }), { inline: true })
            .addGroup(null, orderRadioList = new RadioList({ style: 'width:150px;text-align:right;' })
                .createOptionButton('ascending', { label: /*#. sort ascending */ gt.pgettext('sort', 'Ascending') })
                .createOptionButton('descending', { label: /*#. sort descending */ gt.pgettext('sort', 'Descending') }),
            { inline: true });

        // refill the column/row selector on changing the direction
        this.listenTo(directionGroup, 'group:change', changeDirectionHandler);
        this.listenTo(orderRadioList, 'group:change', changeOrderHandler);
        this.listenTo(keyRadioList, 'group:change', changeKeyHandler);
        this.listenTo(hasHeadersCheckbox, 'group:change', changeHeadersHandler);

        // hide the pop-up menu
        this.listenTo(docView, 'change:selection', changeSelectionHandler);

        // add event handlers for the menu
        this.on({ 'popup:beforeshow': popupBeforeShowHandler });

        // set default position on first display of the menu
        this.one('popup:show', setDefaultPosition);

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = null;
        });

    } // class SortLayerMenu

    // exports ================================================================

    // derive this class from class LayerMenu
    return LayerMenu.extend({ constructor: SortLayerMenu });

});
