/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/signaturetooltip', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/popup/tooltip'
], function (Utils, ToolTip) {

    'use strict';

    // class SignatureToolTip =================================================

    /**
     * A tooltip node attached to the text area shown in cell edit mode, that
     * shows help for the parameters of the sheet function currently edited.
     *
     * @constructor
     *
     * @extends ToolTip
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     *
     * @param {jQuery} textAreaNode
     *  The <textarea> element used as anchor for the tooltip node.
     */
    function SignatureToolTip(docView, textAreaNode) {

        // localized formula resource data
        var resource = docView.getDocModel().getFormulaResource();

        // current list separator, with a trailing space character
        var SEP = Utils.escapeHTML(resource.getSeparator(true) + ' ');

        // base constructor ---------------------------------------------------

        ToolTip.call(this, { anchor: textAreaNode, classes: 'signature-tooltip', maxSize: 600 });

        // private methods ----------------------------------------------------

        /**
         * Returns a descriptor object for the specified function parameter.
         *
         * @param {FunctionHelpDescriptor} functionHelp
         *  The help descriptor of a function.
         *
         * @param {Number} paramIndex
         *  The zero-based index of a parameter of the specified function.
         *
         * @returns {Object|Null}
         *  A help descriptor if the specified function parameter is available,
         *  with the following properties:
         *  - {String} name
         *      The name of the function parameter.
         *  - {String} description
         *      A short description text for the function parameter.
         *  - {Boolean} optional
         *      Whether the parameter is optional.
         *  - {Number} cycleIndex
         *      The zero-based index of the repetition cycle, if the parameter
         *      can be used repeatedly; otherwise -1.
         *  - {Number} relIndex
         *      The relative index of the parameter in the current repetition
         *      cycle, if the parameter can be used repeatedly; otherwise -1.
         */
        function getParameterHelp(functionHelp, paramIndex) {

            // whether the parameter is part of a repetition cycle
            var repeated = (functionHelp.repeatStart >= 0) && (functionHelp.repeatStart <= paramIndex);
            // index of the repetition cycle (used as suffix for parameter names)
            var cycleIndex = repeated ? Math.floor((paramIndex - functionHelp.repeatStart) / functionHelp.cycleLength) : -1;
            // relative parameter index in current repetition cycle
            var relIndex = repeated ? ((paramIndex - functionHelp.repeatStart) % functionHelp.cycleLength) : -1;
            // the help descriptor of the current parameter
            var paramHelp = functionHelp.params[repeated ? (functionHelp.repeatStart + relIndex) : paramIndex];
            // whether the passed parameter index is valid
            var valid = _.isObject(paramHelp) && (!repeated || (cycleIndex < functionHelp.cycleCount));

            // build and return the parameter descriptor
            return valid ? {
                name: paramHelp.name,
                description: paramHelp.description,
                // parameters in the second and following repetition cycles are always optional
                optional: paramHelp.optional || (cycleIndex > 0),
                cycleIndex: cycleIndex,
                relIndex: relIndex
            } : null;
        }

        /**
         * Creates the HTML mark-up for a single function parameter.
         */
        function createParameterMarkup(functionHelp, paramIndex, activeIndex) {

            // complete parameter info descriptor
            var paramHelp = getParameterHelp(functionHelp, paramIndex);
            // the HTML mark-up for the parameter
            var markup = '';

            if (paramHelp) {
                // add a parameter separator between parameters
                if (paramIndex > 0) { markup += SEP; }
                // start a <span> element for the current parameter, highlight active parameter
                markup += '<span';
                if (paramIndex === activeIndex) { markup += ' class="param-active"'; }
                markup += '>';
                // enclose optional parameters and first parameter in (optional) repetition cycle with brackets
                if (paramHelp.optional && (paramHelp.relIndex <= 0)) { markup += '['; }
                // add the parameter name and repetition index
                markup += Utils.escapeHTML(paramHelp.name);
                if (paramHelp.cycleIndex >= 0) { markup += paramHelp.cycleIndex + 1; }
                // enclose optional parameters and last parameter in (optional) repetition cycle with brackets
                if (paramHelp.optional && ((paramHelp.relIndex < 0) || (paramHelp.relIndex + 1 === functionHelp.cycleLength))) { markup += ']'; }
                markup += '</span>';
            }
            return markup;
        }

        /**
         * Creates the HTML mark-up for a complete parameter repetition cycle.
         */
        function createCycleMarkup(functionHelp, cycleIndex, activeIndex) {

            // the HTML mark-up for the parameter cycle
            var markup = '';

            if (cycleIndex < functionHelp.cycleCount) {
                for (var firstIndex = functionHelp.repeatStart + cycleIndex * functionHelp.cycleLength, index = 0; index < functionHelp.cycleLength; index += 1) {
                    markup += createParameterMarkup(functionHelp, firstIndex + index, activeIndex);
                }
            }
            return markup;
        }

        // public methods -----------------------------------------------------

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         *
         * @param {String} funcKey
         *  The unique resource key of a sheet function.
         *
         * @param {Number} paramIndex
         *  The zero-based index of a parameter of the specified function that
         *  will be shown highlighted, and whose help text will be added to the
         *  tooltip. If set to an invalid number, no parameter will be
         *  highlighted.
         *
         * @returns {SignatureToolTip}
         *  A reference to this instance.
         */
        this.updateSignature = function (funcKey, paramIndex) {

            // the help descriptor of the function, containing the description and parameter names
            // (do nothing if no help is available for the function)
            var functionHelp = resource.getFunctionHelp(funcKey);
            if (!functionHelp) { return this.hide(); }

            // info descriptor of the active parameter
            var activeParamHelp = getParameterHelp(functionHelp, paramIndex);

            // create the leading <span> element with the function name
            var markup = '<div class="func-signature"><span class="function-name';
            if (paramIndex < 0) { markup += ' param-active'; }
            markup += '">' + Utils.escapeHTML(functionHelp.name) + '</span>(';

            // add mark-up for all function parameters
            _.times(functionHelp.params.length, function (index) {
                markup += createParameterMarkup(functionHelp, index, paramIndex);
            });

            // special handling for repeated parameters
            if (functionHelp.repeatStart >= 0) {

                // repetition cycle #0 is part of functionHelp.params, add mark-up for
                // repetition cycle #1, e.g.: "SUM(value1" becomes "SUM(value1,[value2]"
                markup += createCycleMarkup(functionHelp, 1, paramIndex);

                // add more repetition cycles around the active repeated parameter
                if (activeParamHelp && (activeParamHelp.cycleIndex >= 0)) {

                    // cycles #0 and #1 have been generated already; if active cycle is
                    // cycle #4, then cycle #2 is omitted, and an ellipsis will be added
                    if (activeParamHelp.cycleIndex >= 4) {
                        markup += SEP + '&hellip;';
                    }
                    // if active cycle is cycle #3 or higher, add preceding cycle
                    if (activeParamHelp.cycleIndex >= 3) {
                        markup += createCycleMarkup(functionHelp, activeParamHelp.cycleIndex - 1, paramIndex);
                    }
                    // if active cycle is cycle #2 or higher, generate it
                    if (activeParamHelp.cycleIndex >= 2) {
                        markup += createCycleMarkup(functionHelp, activeParamHelp.cycleIndex, paramIndex);
                    }
                    // if active cycle is cycle #1 or higher, add following cycle
                    if (activeParamHelp.cycleIndex >= 1) {
                        markup += createCycleMarkup(functionHelp, activeParamHelp.cycleIndex + 1, paramIndex);
                    }
                }

                // add trailing ellipsis, unless the active or the following cycle is the last one
                if (!activeParamHelp || (activeParamHelp.cycleIndex < 0) || (activeParamHelp.cycleIndex + 2 < functionHelp.cycleCount)) {
                    markup += SEP + '&hellip;';
                }
            }

            // close the function signature
            markup += ')</div>';

            // add detailed description of current parameter
            if (activeParamHelp && activeParamHelp.description) {
                markup += '<div class="param-help">' + Utils.escapeHTML(activeParamHelp.description) + '</div>';
            }

            // and finally, fill and show the tooltip node
            return this.setContentMarkup(markup).show();
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = resource = textAreaNode = null;
        });

    } // class SignatureToolTip

    // exports ================================================================

    // derive this class from class ToolTip
    return ToolTip.extend({ constructor: SignatureToolTip });

});
