/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/cellcontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/control/compoundbutton',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Button, CompoundButton, ContextMenu, DrawingFrame, HyperlinkUtils, Labels, gt) {

    'use strict';

    // class CellContextMenu ================================================

    /**
     * A context menu for header panes. Provides menu actions to manipulate
     * entire columns/rows in the active sheet.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {GridPane} gridPane
     *  The grid pane that contains this instance.
     */
    function CellContextMenu(gridPane) {

        var // self reference
            self = this,

            // the document view
            docView = gridPane.getDocView(),

            isSingleCell = false,
            isMergedCell = false,
            containsMerge = false;

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, gridPane.getNode(), {
            // do not show context menu during cell edit mode
            enableKey: 'sheet/operation/cell',
            delay: 200
        });

        // private methods ----------------------------------------------------

        function initMenu() {

            var selRanges = docView.getSelectedRanges(),
                multiCols = (selRanges.colIntervals().size() > 1),
                multiRows = (selRanges.rowIntervals().size() > 1);

            self.destroyAllGroups();

            // the URL of a hyperlink covering the active cell (check for supported protocol)
            var cellUrl = docView.getCellURL();
            if (!HyperlinkUtils.hasSupportedProtocol(cellUrl)) { cellUrl = null; }

            // the URL returned by a formula cell (HYPERLINK function) if available
            var effectiveUrl = docView.getEffectiveURL();
            if (!HyperlinkUtils.hasSupportedProtocol(effectiveUrl)) { effectiveUrl = null; }

            // add the edit and delete hyperlink buttons for 'real' hyperlinks only (not for URLs from HYPERLINK function)
            if (cellUrl) {
                self.addGroup('hyperlink/edit/dialog', new Button({ label: Labels.EDIT_HYPERLINK_LABEL }))
                    .addGroup('hyperlink/delete', new Button({ label: Labels.REMOVE_HYPERLINK_LABEL }));
            }

            // show the open hyperlink button for any available hyperlink
            if (effectiveUrl) {
                self.addGroup(null, new Button({ label: Labels.OPEN_HYPERLINK_LABEL, href: effectiveUrl }))
                    .addSeparator();
                docView.getApp().enableTooltipOnShow(self, effectiveUrl, { anchorBorders: 'top bottom' });
            }

            // sub-menu for all row operations
            var rowMenu = new CompoundButton(docView, { attributes: { 'data-custom': 'row_ctm' }, autoCloseParent: false, autoHideGroups: true, label: (multiRows ? gt('Rows') : gt('Row')), anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                .addGroup('row/insert', new Button((multiRows ? Labels.INSERT_ROWS_OPTIONS : Labels.INSERT_ROW_OPTIONS)))
                .addGroup('row/delete', new Button((multiRows ? Labels.DELETE_ROWS_OPTIONS : Labels.DELETE_ROW_OPTIONS)));

            // sub-menu for all column operations
            var columnMenu = new CompoundButton(docView, { attributes: { 'data-custom': 'column_ctm' }, autoCloseParent: false, autoHideGroups: true, label: (multiCols ? gt('Columns') : gt('Column')), anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                .addGroup('column/insert', new Button((multiCols ? Labels.INSERT_COLUMNS_OPTIONS : Labels.INSERT_COLUMN_OPTIONS)))
                .addGroup('column/delete', new Button((multiCols ? Labels.DELETE_COLUMNS_OPTIONS : Labels.DELETE_COLUMN_OPTIONS)));

            // sub-menu for all insert operations (except columns/rows)
            var insertMenu = new CompoundButton(docView, { attributes: { 'data-custom': 'insert_ctm' }, autoCloseParent: false, autoHideGroups: true, label: Labels.INSERT_HEADER_LABEL, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                .addGroup('cell/autoformula',       new Button(Labels.INSERT_SUM_OPTIONS))
                .addGroup('function/insert/dialog', new Button(Labels.INSERT_FUNCTION_OPTIONS))
                .addGroup('hyperlink/edit/dialog',  new Button(Labels.INSERT_HYPERLINK_OPTIONS))
                .addGroup('name/insert/dialog',     new Button({ label: gt('Named range') }))
                .addGroup('image/insert/dialog',    new Button(Labels.INSERT_IMAGE_OPTIONS));

            if (docView.getApp().isOOXML()) {
                insertMenu.addGroup('chart/insert', new Button(Utils.extendOptions(Labels.INSERT_CHART_OPTIONS, { value: 'column standard' })));
            }

            self.addGroup(null, rowMenu,    { visibleKey: 'sheet/operation/unlocked' })
                .addGroup(null, columnMenu, { visibleKey: 'sheet/operation/unlocked' })
                .addGroup(null, insertMenu)
                .addSeparator()
                .addGroup('table/filter',        new Button(Labels.FILTER_OPTIONS))
                .addGroup('cell/sort/menu',      new Button({ label: gt('Sort') }))
                .addGroup('cell/clear/values',   new Button(Labels.CLEAN_OPTIONS));

            if (docView.getApp().isOOXML()) {
                self.addGroup('view/split/frozen',   new Button({ label: gt('Freeze sheet'), value: 'toggle' }));
            }

            if (!isSingleCell && !containsMerge && !isMergedCell) {
                self.addGroup('cell/merge', new Button({ label: gt('Merge cells'), value: 'toggle' }));
            } else if (containsMerge || isMergedCell) {
                self.addGroup('cell/merge', new Button({ label: gt('Unmerge cells'), value: 'toggle' }));
            }

            self.updateAllGroups();
        }

        /**
         * Selects the column/row that has been clicked with the right mouse
         * button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            var // the source event that caused to open the context menu
                sourceEvent = event.sourceEvent,
                // the address of the clicked cell
                address = gridPane.getCellDataByOffset(sourceEvent.pageX, sourceEvent.pageY).address,

                mergeCollection = docView.getMergeCollection();

            // get out, if user clicked on a drawing frame
            if ($(sourceEvent.target).closest(DrawingFrame.NODE_SELECTOR).length > 0) {
                event.preventDefault();
                return;
            }

            isSingleCell = docView.isSingleCellSelection();
            isMergedCell = mergeCollection.isMergedCell(address);
            containsMerge = docView.hasMergedRangeSelected();
        }

        // public methods ----
        this.getXY = function (event) {
            return {
                pageX: event.sourceEvent.pageX,
                pageY: event.sourceEvent.pageY
            };
        };

        // initialization -----------------------------------------------------

        // lazy initialization of the menu items according to the orientation of the header pane
        this.on('popup:beforeshow', initMenu);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = gridPane = docView = null;
        });

    } // class CellContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: CellContextMenu });

});
