/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/sortmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/popup/sortlayermenu',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, SheetUtils, SortLayerMenu, gt) {

    'use strict';

    // mix-in class SortMixin =================================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * sorting selected ranges.
     *
     * @constructor
     */
    function SortMixin() {

        var // self reference (the spreadsheet view)
            self = this,

            // custom sort menu
            sortLayerMenu = new SortLayerMenu(this),

            // first selected range
            range = null,

            // range after expanding
            newRange = null,

            // should the selected range be checked
            doCheck = true,

            // prevent resetting the sort options (in case of auto expand "change:selection")
            preventReset = false,

            // state of sorting
            state = null;

        // private methods ----------------------------------------------------

        /**
          * Resets all sort-options by sort-type
          *
          * @param {String} sorttype
          *  The type on which the reset should be executed
          */
        function resetSortOptions() {
            if (preventReset === true) { preventReset = false; return this; }

            state = null;

            self.getApp().getController().update();
        }

        /**
         * checks whether the selection is sortable or not
         *
         * @returns {Boolean}
         */
        function isSortable() {

            // get out here, sheet is protected
            if (!self.ensureUnlockedSelectionSync({ lockTables: 'header' })) {
                return false;
            }

            // get current selection
            var ranges = self.getSelectedRanges();

            // the selected range(s)
            range = ranges.first();

            // get out here, if more than one range is selected
            if (ranges.length > 1) {
                self.yellMessage('sort:multiselect');
                return false;
            }

            // get out here, if selection is blank (nothing to sort)
            if (self.getCellCollection().areRangesBlank(range)) {
                self.yellMessage('sort:blank');
                return false;
            }

            if ((range.cols() > SheetUtils.MAX_SORT_LINES_COUNT) || (range.rows() > SheetUtils.MAX_SORT_LINES_COUNT)) {
                self.yellMessage('sort:overflow');
                return false;
            }
            return true;
        }

        /**
         * Check if the selection is expandable or actually must expand.
         */
        function expandSelection(options) {

            // exit when the expand-possibility should not be check anymore
            if (!doCheck) { return $.when(); }

            function doExpand() {

                preventReset = true;

                // yell, if the new range has too many columns/rows included (returns false)
                if (newRange.cols() > SheetUtils.MAX_CHANGE_COLS_COUNT) {
                    self.yellMessage('cols:overflow');
                    return false;
                }
                if (newRange.rows() > SheetUtils.MAX_CHANGE_ROWS_COUNT) {
                    self.yellMessage('rows:overflow');
                    return false;
                }

                // select the new range
                self.selectRange(newRange, { active: self.getActiveCell() });

                // the selected range
                range = self.getSelectedRanges().first();

                return $.when();
            }

            newRange = self.getCellCollection().findContentRange(range);

            // the selection is only within one row/col. Ask the user whether expand selection or not.
            if ((range.singleCol() !== range.singleRow()) && newRange.differs(range)) {

                // ask the user to expand the current selection
                var promise = self.showQueryDialog(gt('Sorting'), gt('Expand the selection?'), { cancel: true });

                return promise.done(function (answer) {

                    // expand the selection automatically to current used rectangle
                    if (answer) { doExpand(); }

                    if (Utils.getBooleanOption(options, 'custom', false)) {
                        sortLayerMenu.show();
                        self.executeDelayed(function () {
                            sortLayerMenu.grabFocus({ bottom: true });
                        }, 100);
                    }

                    // stop checking the selected range. The user has made a decision already for this selection.
                    doCheck = false;
                });
            }

            // only one cell is selected. Expand selection automatically (don't ask the user).
            if (range.single() && newRange.differs(range)) {
                // expand the selection automatically to current used rectangle
                return doExpand();
            }

            return $.when();
        }

        // methods ------------------------------------------------------------

        /**
         * Prepares and executes the sorting
         *
         * @param {String} sorttype
         *  The type on which currently be sorted ('standard', 'custom')
         *
         * @param {String} order
         *  The order-method ('ascending', 'descending', ...)
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the selected cell range has
         *  been sorted.
         */
        this.sortCellRange = function (options) {
            // get out here, if there is something unexpected
            if (!isSortable()) { return $.when(); }

            range = self.getSelectedRanges().first();

            var cellCollection  = self.getCellCollection(),
                pushToggleState = false,
                orderBy         = Utils.getStringOption(options, 'order', 'ascending');

            // detect order-direction in case of 'toggle'
            if (orderBy === 'toggle') {
                // get old order-direction, or use 'asc' by default
                orderBy = state || 'ascending';
                // prepare updating the toggle button state
                pushToggleState = true;
            }

            // open the dialog to ask the user whether the range should expand or not
            var dialog = expandSelection();

            return dialog.then(function () {
                return self.getSheetModel().createAndApplyOperations(function (generator) {
                    return cellCollection.generateSortOperations(generator, range, self.getActiveCell(), orderBy, options);
                }, { storeSelection: true });
            })
            .done(function () {
                // set new/current sort state to toggle button
                state = pushToggleState ? (orderBy === 'ascending') ? 'descending' : 'ascending' : null;
                if (sortLayerMenu.isVisible()) { sortLayerMenu.hide(); }
            });
        };

        /**
         * Checks the sort-environment. Decides, whether the selection should expand, or in doubt, asks the user.
         */
        this.showSortMenu = function () {
            if (isSortable()) {
                expandSelection({ custom: true })
                .then(function () {
                    resetSortOptions();
                    sortLayerMenu.show();
                    self.executeDelayed(function () {
                        sortLayerMenu.grabFocus({ bottom: true });
                    }, 100);
                });
            }
        };

        /**
          * Provides the current sort-state for (e.g.) the toggle-button
          */
        this.getSortState = function () {
            return (state === null) ? 'ascending' : state;
        };

        // initialization -----------------------------------------------------

        // reset the state, if the selection changes (sort settings won't be saved)
        this.on('change:selection', function () {
            resetSortOptions();
            // reset also the expand decision
            doCheck = true;
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sortLayerMenu.destroy();
            self = sortLayerMenu = range = newRange = state = null;
        });

    } // class SortMixin

    // exports ================================================================

    return SortMixin;

});
