/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/chartcreator', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, SheetUtils, Operations, TokenArray, gt) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        Range3D = SheetUtils.Range3D,
        Range3DArray = SheetUtils.Range3DArray,

        // default width of new chart objects
        CHART_DEFAULT_WIDTH = Utils.convertHmmToLength(15000, 'px', 1),

        // default height of new chart objects
        CHART_DEFAULT_HEIGHT = Utils.convertHmmToLength(10000, 'px', 1);

    var DEFAULT_LINECOLOR = { transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }], type: 'scheme', value: 'text1' };
    var STANDARD_SHAPE = { type: 'solid', color: DEFAULT_LINECOLOR, width: 36 };
    var NONE_SHAPE = { type: 'none', color: null, width: null };
    var HEAD_CHAR = { fontSize: 16, bold: false };

    var INSERT_DEFAULTS = (function () {

        var AXIS_ENABLED =  { axis: { label: true },  line: NONE_SHAPE };
        var AXIS_DISABLED = { axis: { label: false }, line: NONE_SHAPE };
        var GRID_ENABLED =  { line: STANDARD_SHAPE };
        var GRID_DISABLED = { line: NONE_SHAPE };

        return {
            normal: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED
            },
            xy: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_ENABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED
            },
            pie: {
                xAxis: AXIS_DISABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_DISABLED,
                yGrid: GRID_DISABLED
            }
        };
    }());

    function getFormula(docModel, sheet, from, to) {
        var range = new Range(from, to);
        var tokenArray = new TokenArray(docModel, 'chart');
        tokenArray.appendRange(range, { sheet: sheet, abs: true });
        return tokenArray.getFormula('op');
    }

    function addDataSeries(docModel, generator, options) {

        var sourceSheet = options.sourceSheet;
        var position = options.position;
        var series = options.series;
        var keyFrom = options.keyFrom;
        var keyTo = options.keyTo;
        var title = options.title;
        var valueFrom = options.valueFrom;
        var valueTo = options.valueTo;
        var bubbleFrom = options.bubbleFrom;
        var bubbleTo = options.bubbleTo;
        var bubbleArray = options.bubbleArray;
        var lineAttrs = options.lineAttrs;

        var insert = {
            values: getFormula(docModel, sourceSheet, valueFrom, valueTo)
        };
        if (title) {
            insert.title = getFormula(docModel, sourceSheet, title);
        }
        if (keyFrom) {
            insert.names = getFormula(docModel, sourceSheet, keyFrom, keyTo);
        }
        if (bubbleFrom) {
            insert.bubbles = getFormula(docModel, sourceSheet, bubbleFrom, bubbleTo);
        } else if (bubbleArray) {
            insert.bubbles = bubbleArray;
        }
        var properties = {
            series: series,
            attrs: {
                series: insert,
                line: lineAttrs || { type: 'solid' },
                fill: { type: 'solid' }
            }
        };
        generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
    }

    function generateSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, axis, chart, lineAttrs) {

        if (_.isUndefined(chart.varyColors)) {
            chart.varyColors = true;
        }

        generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: { chart: chart } });

        var start = sourceRange.start;
        var end = sourceRange.end;

        var useAxis;
        if (axis !== 0 && axis !== 1) {
            if (cells[0] && !cells[1]) {
                useAxis = 0;
            } else if (!cells[0] && cells[1]) {
                useAxis = 1;
            } else {
                var width = end[0] - start[0];
                var height = end[1] - start[1];
                if (width > height) {
                    useAxis = 1;
                } else {
                    useAxis = 0;
                }
            }
        } else {
            useAxis = axis;
        }

        var seriesCount;
        if (chart.type && chart.type.indexOf('bubble') === 0) {
            seriesCount = generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, useAxis, lineAttrs);
        } else {
            seriesCount = generateStandardSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, useAxis, lineAttrs);
        }
        if ((seriesCount === 1 && !chart.varyColors) || (chart.type === 'scatter2d' || chart.type === 'line2d')) {
            var send = {};
            if (seriesCount === 1 && !chart.varyColors) {
                send.varyColors = true;
            }
            //CURVED is a workaround, filter has not all the information it needs!
            send.curved = chart.curved;
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: { chart: send } });
        }
        return seriesCount;
    }

    /**
     * makes an address array, needed for the code, which does not know if its row or column oriented
     */
    function makeAddress(activeAxis, otherAxis, activeValue, otherValue) {
        var address = Address.A1.clone();
        address[activeAxis] = activeValue;
        address[otherAxis] = otherValue;
        return address;
    }

    /**
     * generate all series from start to end.
     * Depending if first row/column contains a String,
     * it is defined as Names (datapoint) and Titles (dataseries)
     *
     * @param {Array<Boolean>} isNumber
     *  first is a Boolean for the cell under the first cell
     *  second is a Boolean for the cell right of the first cell
     *
     * @param {Number} activeAxis
     *  [1 or 0] column or row for the direction of any series data
     */
    function generateStandardSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, activeAxis, lineAttrs) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        var otherAxis = 1 - activeAxis;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        for (var i = 0; i < max; i++) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            addDataSeries(docModel, generator, {
                sourceSheet: sourceSheet,
                position: position,
                series: i,
                keyFrom: keyFrom,
                keyTo: keyTo,
                title: title,
                valueFrom: valueFrom,
                valueTo: valueTo,
                lineAttrs: lineAttrs
            });
        }
        return max;
    }

    /**
     * generate all bubble-series from start to end.
     * Every second row/columns are values, the others are bubbles (sizes).
     * Depending if first row/column contains a String,
     * it is defined as Titles (dataseries).
     * If the count of the series is odd,
     * the first row/column is defined as Names (datapoints)
     *
     * @param {Array<Boolean>} isNumber
     *  first is a Boolean for the cell under the first cell
     *  second is a Boolean for the cell right of the first cell
     *
     * @param {Number} activeAxis
     *  [1 or 0] column or row for the direction of any series data
     */
    function generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, cells, activeAxis, lineAttrs) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        if (max === 0) {
            //exception only for 1D - Bubbles, behaves like excel
            activeAxis = (activeAxis - 1) * -1;
            max = end[activeAxis] - start[activeAxis];
        }
        var otherAxis = (activeAxis - 1) * -1;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        var size = 0;

        for (var i = 0; i < max; i += 2) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            var bubbleFrom = null;
            var bubbleTo = null;
            var bubbleArray = null;

            if (i === max - 1) {
                bubbleArray = [];
                var arraySize = 1 + (valueTo[otherAxis] - valueFrom[otherAxis]);
                for (var j = 0; j < arraySize; j++) {
                    bubbleArray.push(1);
                }
            } else {
                bubbleFrom = makeAddress(activeAxis, otherAxis, valueFrom[activeAxis] + 1, valueFrom[otherAxis]);
                bubbleTo = makeAddress(activeAxis, otherAxis, valueTo[activeAxis] + 1, valueTo[otherAxis]);
            }

            addDataSeries(docModel, generator, {
                sourceSheet: sourceSheet,
                position: position,
                series: size,
                keyFrom: keyFrom,
                keyTo: keyTo,
                title: title,
                valueFrom: valueFrom,
                valueTo: valueTo,
                bubbleFrom: bubbleFrom,
                bubbleTo: bubbleTo,
                bubbleArray: bubbleArray,
                lineAttrs: lineAttrs
            });
            size++;
        }
        return size;
    }

    function getContentForRange(docModel, docView, sourceRange, sourceSheet, axis, forceTitle, forceNames) {

        var sheetModel = docModel.getSheetModel(sourceSheet);

        var start = sourceRange.start;
        var cols = sourceRange.cols();
        var rows = sourceRange.rows();

        if (cols > 50 || rows > 50) {
            docView.yell({ type: 'warning', message: gt('It is not possible to create a chart out of more than 50 input cells.') });
            return $.Deferred().reject();
        }

        var rightEntry = sheetModel.getColCollection().getNextVisibleEntry(start[0] + 1);
        var bottEntry = sheetModel.getRowCollection().getNextVisibleEntry(start[1] + 1);

        var rightAddress = new Address(rightEntry ? rightEntry.index : start[0], start[1]);
        var bottAddress = new Address(start[0], bottEntry ? bottEntry.index : start[1]);

        var ranges = new Range3DArray(
            Range3D.createFromAddress(bottAddress, sourceSheet),
            Range3D.createFromAddress(rightAddress, sourceSheet),
            Range3D.createFromAddress(start, sourceSheet)
        );

        // first array element of the result is an array with the contents of the three cells
        var res = docModel.getRangeContents(ranges, { attributes: true, display: true });
        var bottomCell = res[0];
        var rightCell = res[1];
        var startCell = res[2];

        function cellIsLabel(cell) {
            if (!startCell) { return false; }
            if (!startCell.display) { return true; } // empty strings (blank cells) or null values (invalid number format)
            return cell.format.isAnyDateTime() || !_.isNumber(cell.value);
        }

        var bottom = false;
        var right = false;

        if (cols === 1 || rows === 1) {
            if (axis !== 0 && axis !== 1) {
                if (cols === 1) {
                    axis = 0;
                } else {
                    axis = 1;
                }
            }
            var firstString = !_.isNumber(startCell.value);
            if (axis === 1) {
                if (firstString) {
                    bottom = true;
                }
            } else {
                if (firstString) {
                    right = true;
                }
            }
        } else {
            bottom = cellIsLabel(bottomCell);
            right = cellIsLabel(rightCell);
        }

        if (!_.isUndefined(forceTitle)) {
            if (axis === 0) {
                right = forceTitle;
            } else if (axis === 1) {
                bottom = forceTitle;
            }
        }
        if (!_.isUndefined(forceNames)) {
            if (axis === 0) {
                bottom = forceNames;
            } else if (axis === 1) {
                right = forceNames;
            }
        }

        return $.when({ cells: [bottom, right], axis: axis });
    }

    // static class ChartCreator ==============================================

    var ChartCreator = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates and inserts a new chart into the active sheet.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the chart has been created and
     *  inserted successfully, or that will be rejected on any error.
     */
    ChartCreator.createChart = function (app, chart) {

        var docModel = app.getModel();
        var docView = app.getView();
        var gridPane = docView.getActiveGridPane();

        var drawingCollection = docView.getDrawingCollection();

        var range = docView.getActiveRange();

        var promise = getContentForRange(docModel, docView, range, docView.getActiveSheet());

        promise = promise.then(function (data) {

            var rect = gridPane.getVisibleRectangle();

            var clip = {
                left: Math.max(0, Math.round(rect.left + (rect.width - CHART_DEFAULT_WIDTH) / 2)),
                top: Math.max(0, Math.round(rect.top + (rect.height - CHART_DEFAULT_HEIGHT) / 2)),
                width: CHART_DEFAULT_WIDTH,
                height: CHART_DEFAULT_HEIGHT
            };

            var attrs = drawingCollection.getAttributeSetForRectangle(clip);

            return docView.insertDrawing('chart', attrs, function (generator, sheet, position) {

                if (!chart.chartStyleId) {
                    chart.chartStyleId = 2;
                }

                var seriesCount = generateSeriesOperations(docModel, range, sheet, generator, position, data.cells, data.axis, chart);

                var legendPos;
                if (seriesCount === 1) {
                    //TODO: title
                    legendPos = 'off';
                    generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: 'main', attrs: { character: HEAD_CHAR } });
                } else {
                    legendPos = 'bottom';
                }
                generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: { legend: { pos: legendPos } } });

                var defaults;
                switch (chart.type) {
                    case 'bubble2d':
                    case 'scatter2d':
                        defaults = INSERT_DEFAULTS.xy;
                        break;
                    case 'pie2d':
                    case 'donut2d':
                        defaults = INSERT_DEFAULTS.pie;
                        break;
                    default:
                        defaults = INSERT_DEFAULTS.normal;
                }

                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: 'x', attrs: defaults.xAxis });
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: 'y', attrs: defaults.yAxis });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: 'x', attrs: defaults.xGrid });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: 'y', attrs: defaults.yGrid });
            });
        });

        return promise;
    };

    /**
     *
     * @param {Range} range
     *  whole start to end area of the new series
     *
     * @param {Number} sheet
     *
     * @param {ChartModel} chartModel
     *  is used for the position and id in the document and as fallback for chart-type-checks
     *
     * @param {String} axis
     *  series direction
     *
     * @param {Object} chartAttrs
     *  is used for changing other attributes of the chart and for the type info (bubble or not)
     *
     * @param {Boolean} forceTitle
     *  is set, it overwrites the internal logic for title data
     *
     * @param {Boolean} forceNames
     *  is set, it overwrites the internal logic for names data
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the chart has been changed
     *  successfully, or that will be rejected on any error.
     */
    ChartCreator.updateSeries = function (app, sourceRange, sourceSheet, chartModel, axis, chartAttrs, forceTitle, forceNames) {

        if (!chartAttrs) {
            var mergedAttributes = chartModel.getMergedAttributeSet(true);
            //CURVED is a workaround, filter has not all the information it needs!
            chartAttrs = {
                type: mergedAttributes.chart.type,
                curved: mergedAttributes.chart.curved
            };
        }

        var docModel = app.getModel(),
            docView = app.getView();

        var promise = getContentForRange(docModel, docView, sourceRange, sourceSheet, axis, forceTitle, forceNames);

        promise = promise.then(function (data) {

            //only overwritten if source range height is 1 oder width is 1
            if (data.axis === 0 || data.axis === 1) {
                axis = data.axis;
            }
            var position = chartModel.getSheetPosition();
            var lineAttrs = null;
            if (chartModel.isMarkerOnly()) {
                lineAttrs = NONE_SHAPE;
            }

            return chartModel.getSheetModel().createAndApplyOperations(function (generator) {

                var clonedData = chartModel.getCloneData(),
                    oldCount = chartModel.getSeriesCount(),
                    seriesCount = generateSeriesOperations(docModel, sourceRange, sourceSheet, generator, position, data.cells, axis, chartAttrs, lineAttrs);

                _.times(oldCount, function () {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: seriesCount });
                });

                // UNDO ---
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: clonedData.attrs }, { undo: true });

                _.each(clonedData.series, function (obj, index) {
                    generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, { series: index, attrs: obj }, { undo: true });
                });

                _.times(seriesCount, function () {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: clonedData.series.length }, { undo: true });
                });

                // callback MUST return a promise
                return $.when();
            }, { storeSelection: true });
        });

        return promise;
    };

    ChartCreator.generateOperationsFromModelData = function (app, generator, position, chartModelData) {
        var chartAttrs = {
            chart: chartModelData.attrs.chart,
            fill: chartModelData.attrs.chart,
            line: chartModelData.attrs.line
        };
        var axes = chartModelData.axes;
        var series = chartModelData.series;
        var title = chartModelData.title;
        var legend = chartModelData.legend;

        generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: chartAttrs });

        _.each(series, function (serie, index) {
            var properties = {
                series: index,
                attrs: serie
            };
            generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
        });

        _.each(axes, function (axisData, name) {
            if (!_.isEmpty(axisData.axis)) {
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: name, attrs: axisData.axis });
            }
            if (!_.isEmpty(axisData.grid)) {
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: name, attrs: axisData.grid });
            }
            if (!_.isEmpty(axisData.title)) {
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: name, attrs: axisData.title });
            }
        });

        if (!_.isEmpty(title)) {
            generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: 'main', attrs: title });
        }

        if (!_.isEmpty(legend)) {
            generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: legend });
        }

    };

    ChartCreator.getStandardShape = function () {
        return STANDARD_SHAPE;
    };

    ChartCreator.getHeadChar = function () {
        return HEAD_CHAR;
    };

    ChartCreator.getNoneShape = function () {
        return NONE_SHAPE;
    };

    // exports ================================================================

    return ChartCreator;

});
