/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/rangearray', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/arraytemplate',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/intervalarray'
], function (Utils, IteratorUtils, ArrayTemplate, Interval, Range, IntervalArray) {

    'use strict';

    // private static functions ===============================================

    /**
     * Creates a new band interval instance.
     */
    function createBandInterval(first, last, ranges) {
        var interval = new Interval(first, last);
        if (ranges) { interval.ranges = ranges; }
        return interval;
    }

    /**
     * Divides the cell range addresses into a sorted array of column or row
     * intervals representing the boundaries of all cell ranges. See public
     * methods RangeArray.getColBands() and RangeArray.getRowBands() for more
     * details.
     */
    function createBandIntervals(ranges, columns, options) {

        // whether to add the sorted original ranges to the resulting interval bands
        var addRanges = Utils.getBooleanOption(options, 'ranges', false);
        // whether to add the merged range intervals inside the bands to the resulting interval bands
        var addIntervals = Utils.getBooleanOption(options, 'intervals', false);
        // whether to add the original ranges intermediately (needed for ranges, or for intervals)
        var createRanges = addRanges || addIntervals;
        // the resulting interval bands
        var bandIntervals = new IntervalArray();
        // the start array index in 'bandIntervals' used to search for matching intervals
        var startIndex = 0;

        // work on a clone sorted by start column/row index
        ranges = ranges.clone().sortBy(function (range) { return range.getStart(columns); });

        // build the array of row bands
        ranges.forEach(function (range) {

            // the start and end band index of the range
            var first = range.getStart(columns);
            var last = range.getEnd(columns);

            // update the row bands containing the range
            for (var bandIndex = startIndex; bandIndex < bandIntervals.length; bandIndex += 1) {

                // the current band interval
                var bandInterval = bandIntervals[bandIndex];

                // row band is above the range, ignore it in the next iterations
                if (bandInterval.last < first) {
                    startIndex = bandIndex + 1;
                    continue;
                }

                // band interval needs to be split (range starts inside the band)
                if (bandInterval.first < first) {
                    bandIntervals.splice(bandIndex, 0, createBandInterval(bandInterval.first, first - 1, createRanges ? bandInterval.ranges.clone() : null));
                    bandInterval.first = first;
                    startIndex = bandIndex + 1;
                    // next iteration of the for loop will process the current row band again
                    continue;
                }

                // band interval needs to be split (range ends inside the band)
                if (last < bandInterval.last) {
                    bandIntervals.splice(bandIndex + 1, 0, createBandInterval(last + 1, bandInterval.last, createRanges ? bandInterval.ranges.clone() : null));
                    bandInterval.last = last;
                }

                // now, current band interval contains the range completely
                if (createRanges) { bandInterval.ranges.push(range); }

                // continue with next range, if end of range found
                if (last === bandInterval.last) { return; }

                // adjust start index in case a new band interval needs to be appended
                first = bandInterval.last + 1;
            }

            // no band interval found, create and append a new band interval
            bandIntervals.push(createBandInterval(first, last, createRanges ? new RangeArray(range) : null));
        });

        // sort the ranges in each band (option 'ranges'), create the merged intervals (option 'intervals')
        bandIntervals.forEach(function (bandInterval) {
            // create the merged intervals from the original ranges
            if (addIntervals) { bandInterval.intervals = bandInterval.ranges.intervals(!columns).merge(); }
            // sort the original ranges, or remove them completely
            if (addRanges) { bandInterval.ranges.sortBy(function (range) { return range.getStart(!columns); }); } else { delete bandInterval.ranges; }
        });

        // further merge adjoining band intervals with equal merged inner intervals originating from different
        // ranges (only required, if these ranges will not be returned)
        if (!addRanges && addIntervals) {
            bandIntervals.forEachReverse(function (bandInterval, index) {
                var nextInterval = bandIntervals[index + 1];
                if (nextInterval && (bandInterval.last + 1 === nextInterval.first) && bandInterval.intervals.equals(nextInterval.intervals)) {
                    bandInterval.last = nextInterval.last;
                    bandIntervals.splice(index + 1, 1);
                }
            });
        }

        return bandIntervals;
    }

    /**
     * Converts the passed array of row bands to an array cell range addresses,
     * as returned by the public method RangeArray.merge().
     *
     * @param {IntervalArray} rowBands
     *  An array of row bands, as returned for example by the method
     *  RangeArray.getRowBands(). Each array element must contain an additional
     *  property 'intervals' containing the column intervals of the row band.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses resulting from the passed row bands.
     */
    function mergeRangesFromRowBands(rowBands) {

        // resulting array of cell range addresses
        var resultRanges = new RangeArray();
        // current ranges whose last row ends in the current row band
        var bandRanges = [];

        // appends the passed ranges to the result array
        function appendResultRanges(ranges) {
            resultRanges = resultRanges.concat(ranges);
        }

        // returns whether the passed range precedes the specified column interval,
        // or whether it starts at the interval, but ends at another column
        function isPrecedingUnextendableRange(range, interval) {
            return (range.start[0] < interval.first) || ((range.start[0] === interval.first) && (range.end[0] !== interval.last)) || ((interval.first < range.start[0]) && (range.start[0] <= interval.last));
        }

        // build ranges from the row bands and their column intervals
        rowBands.forEach(function (rowBand, bandIndex) {

            // the preceding row band
            var prevRowBand = rowBands[bandIndex - 1];
            // array index into the bandRanges array
            var bandRangeIndex = 0;

            // try to extend ranges from the previous row band with ranges from the current row band
            if (prevRowBand && (prevRowBand.last + 1 === rowBand.first)) {

                // process all intervals of the current row band, extend ranges or move them to the result array
                rowBand.intervals.forEach(function (colInterval) {

                    // move all ranges from bandRanges to resultRanges that cannot be merged with the current interval anymore
                    // (taking advantage of the fact that all ranges in bandRanges are distinct and ordered by column)
                    while ((bandRangeIndex < bandRanges.length) && isPrecedingUnextendableRange(bandRanges[bandRangeIndex], colInterval)) {
                        appendResultRanges(bandRanges.splice(bandRangeIndex, 1));
                    }

                    // extend the current band range to this row band, if column interval matches
                    var bandRange = bandRanges[bandRangeIndex];
                    if (bandRange && (bandRange.start[0] === colInterval.first) && (bandRange.end[0] === colInterval.last)) {
                        bandRange.end[1] = rowBand.last;
                    } else {
                        bandRanges.splice(bandRangeIndex, 0, Range.createFromIntervals(colInterval, rowBand));
                    }
                    bandRangeIndex += 1;
                });

                // move all remaining unextended ranges from bandRanges to resultRanges
                appendResultRanges(bandRanges.splice(bandRangeIndex));

            } else {

                // store all old band ranges in the result array, create new ranges for this row band
                appendResultRanges(bandRanges);
                bandRanges = RangeArray.createFromColIntervals(rowBand.intervals, rowBand.first, rowBand.last);
            }
        });

        // append remaining band ranges to the result
        appendResultRanges(bandRanges);
        return resultRanges;
    }

    // class RangeArray =======================================================

    /**
     * Represents an array of cell range addresses. The array elements are
     * instances of the class Range.
     *
     * @constructor
     *
     * @extends Array
     */
    var RangeArray = ArrayTemplate.create(Range);

    // static methods ---------------------------------------------------------

    /**
     * Creates an array of cell range addresses from the passed column and row
     * intervals, by combining each column interval with each row interval.
     *
     * @param {IntervalArray|Interval} colIntervals
     *  The column intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @param {IntervalArray|Interval} rowIntervals
     *  The row intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses created from the passed intervals.
     */
    RangeArray.createFromIntervals = function (colIntervals, rowIntervals) {
        var result = new RangeArray();
        IntervalArray.forEach(rowIntervals, function (rowInterval) {
            IntervalArray.forEach(colIntervals, function (colInterval) {
                result.push(Range.createFromIntervals(colInterval, rowInterval));
            });
        });
        return result;
    };

    /**
     * Creates an array of cell range addresses from the passed column
     * intervals and row indexes.
     *
     * @param {IntervalArray|Interval} colIntervals
     *  The column intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @param {Number} firstRow
     *  The index of the first row in all new cell range addresses.
     *
     * @param {Number} [lastRow]
     *  The index of the last row in all new cell range addresses. If omitted,
     *  all ranges will cover a single row only.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses created from the passed column
     *  intervals and row indexes.
     */
    RangeArray.createFromColIntervals = function (colIntervals, firstRow, lastRow) {
        return RangeArray.createFromIntervals(colIntervals, new Interval(firstRow, lastRow));
    };

    /**
     * Creates an array of cell range addresses from the passed row intervals
     * and column indexes.
     *
     * @param {IntervalArray|Interval} rowIntervals
     *  The row intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter. Note that this method is
     *  one of the rare cases where row parameters precede column parameters,
     *  in order to provide an optional column parameter.
     *
     * @param {Number} firstCol
     *  The index of the first column in all new cell range addresses.
     *
     * @param {Number} [lastCol]
     *  The index of the last row in all new cell range addresses. If omitted,
     *  all ranges will cover a single column only.
     *
     * @returns {Range}
     *  The array of cell range addresses created from the passed row intervals
     *  and column indexes.
     */
    RangeArray.createFromRowIntervals = function (rowIntervals, firstCol, lastCol) {
        return RangeArray.createFromIntervals(new Interval(firstCol, lastCol), rowIntervals);
    };

    /**
     * Creates an array of cell range addresses from the passed array of single
     * cell addresses. The cell ranges in the resulting array will be merged,
     * i.e. no cell range address overlaps with or adjoins exactly to another
     * cell range address.
     *
     * @param {AddressArray} addresses
     *  An unordered array of cell addresses to be merged to cell range
     *  addresses.
     *
     * @returns {RangeArray}
     *  A merged array of cell range addresses from the passed cell addresses.
     */
    RangeArray.mergeAddresses = function (addresses) {

        // simplifications for small arrays
        switch (addresses.length) {
            case 0:
                return new RangeArray();
            case 1:
                return new RangeArray(new Range(addresses[0]));
        }

        // the row bands containing column intervals for all existing rows
        var rowBands = new IntervalArray();

        // sort and unify the cell addresses
        addresses = addresses.unify().sort();

        // create row bands with all rows that contains any cells
        addresses.forEach(function (address) {

            var // the last row band in the array of all row bands
                lastRowBand = _.last(rowBands),
                // the last interval in the current row band
                lastInterval = null;

            // create a new row band on demand
            if (!lastRowBand || (lastRowBand.first !== address[1])) {
                rowBands.push(lastRowBand = new Interval(address[1]));
                lastRowBand.intervals = new IntervalArray();
            }

            // get the last interval, extend it or create a new interval
            lastInterval =  _.last(lastRowBand.intervals);
            if (lastInterval && (lastInterval.last + 1 === address[0])) {
                lastInterval.last += 1;
            } else {
                lastRowBand.intervals.push(new Interval(address[0]));
            }
        });

        // convert the row bands to an array of cell ranges
        return mergeRangesFromRowBands(rowBands);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for this array of cell range addresses that
     * can be used for example as key in an associative map. This method is
     * faster than the method RangeArray.toString().
     *
     * @returns {String}
     *  A unique string key for this instance.
     */
    RangeArray.prototype.key = function () {
        return this.map(function (range) { return range.key(); }).join(' ');
    };

    /**
     * Returns the number of cells covered by all cell range address.
     *
     * @returns {Number}
     *  The number of cells covered by all cell range address.
     */
    RangeArray.prototype.cells = function () {
        return this.reduce(function (count, range) { return count + range.cells(); }, 0);
    };

    /**
     * Returns whether a cell range address in this array contains the passed
     * column index.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed column
     *  index.
     */
    RangeArray.prototype.containsCol = function (col) {
        return this.some(function (range) { return range.containsCol(col); });
    };

    /**
     * Returns whether a cell range address in this array contains the passed
     * row index.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed row
     *  index.
     */
    RangeArray.prototype.containsRow = function (row) {
        return this.some(function (range) { return range.containsRow(row); });
    };

    /**
     * Returns whether a range address in this array contains the passed column
     * or row index.
     *
     * @param {Number} index
     *  The column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to check for a column index (true), or a row index (false).
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed column
     *  or row index.
     */
    RangeArray.prototype.containsIndex = function (index, columns) {
        return columns ? this.containsCol(index) : this.containsRow(index);
    };

    /**
     * Returns whether a cell range address in this array contains the passed
     * cell address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed cell
     *  address.
     */
    RangeArray.prototype.containsAddress = function (address) {
        return this.some(function (range) { return range.containsAddress(address); });
    };

    /**
     * Returns whether all cell range addresses in the passed array are
     * contained completely in any of the cell range addresses in this array.
     *
     * @param {RangeArray|Range} ranges
     *  The other cell range addresses to be checked. This method also accepts
     *  a single cell range address as parameter.
     *
     * @returns {Boolean}
     *  Whether all cell range addresses in the passed array are contained
     *  completely in any of the cell range addresses in this array.
     */
    RangeArray.prototype.contains = function (ranges) {
        return RangeArray.every(ranges, function (range2) {
            return this.some(function (range1) {
                return range1.contains(range2);
            });
        }, this);
    };

    /**
     * Returns whether any cell range address in this array overlaps with any
     * cell range address in the passed array.
     *
     * @param {RangeArray|Range} ranges
     *  The other cell range addresses to be checked. This method also accepts
     *  a single cell range address as parameter.
     *
     * @returns {Boolean}
     *  Whether any cell range address in this array overlaps with any cell
     *  range address in the passed array.
     */
    RangeArray.prototype.overlaps = function (ranges) {
        return this.some(function (range1) {
            return RangeArray.some(ranges, function (range2) {
                return range1.overlaps(range2);
            });
        });
    };

    /**
     * Returns whether any pair of cell range addresses in this array overlap
     * each other.
     *
     * @returns {Boolean}
     *  Whether any pair of cell range addresses in this array overlap each
     *  other.
     */
    RangeArray.prototype.overlapsSelf = function () {

        // simplifications for small arrays
        switch (this.length) {
            case 0:
            case 1:
                return false;
            case 2:
                return this[0].overlaps(this[1]);
        }

        // work on sorted clones for performance (check adjacent array elements)
        var ranges = this.clone().sort(function (range1, range2) { return range1.start[1] - range2.start[1]; });
        return ranges.some(function (range1, index) {
            var result = false;
            Utils.iterateArray(ranges, function (range2) {
                // a range with distinct row interval and all following ranges cannot overlap anymore
                if (range1.end[1] < range2.start[1]) { return Utils.BREAK; }
                if ((result = range1.overlaps(range2))) { return Utils.BREAK; }
            }, { begin: index + 1 });
            return result;
        });
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed column index.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed
     *  column index; or null, if no range has been found.
     */
    RangeArray.prototype.findByCol = function (col) {
        return _.find(this, function (range) { return range.containsCol(col); }) || null;
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed row index.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed row
     *  index; or null, if no range has been found.
     */
    RangeArray.prototype.findByRow = function (row) {
        return _.find(this, function (range) { return range.containsRow(row); }) || null;
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed column or row index.
     *
     * @param {Number} index
     *  The zero-based column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to search for a column index (true), or a row index (false).
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed
     *  column or row index; or null, if no range has been found.
     */
    RangeArray.prototype.findByIndex = function (index, columns) {
        return columns ? this.findByCol(index) : this.findByRow(index);
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed cell address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed
     *  cell address; or null, if no range has been found.
     */
    RangeArray.prototype.findByAddress = function (address) {
        return _.find(this, function (range) { return range.containsAddress(address); }) || null;
    };

    /**
     * Returns the column intervals of all cell range addresses in this array.
     *
     * @returns {IntervalArray}
     *  The column intervals of all cell range addresses in this array.
     */
    RangeArray.prototype.colIntervals = function () {
        return IntervalArray.invoke(this, 'colInterval');
    };

    /**
     * Returns the row intervals of all cell range addresses in this array.
     *
     * @returns {IntervalArray}
     *  The row intervals of all cell range addresses in this array.
     */
    RangeArray.prototype.rowIntervals = function () {
        return IntervalArray.invoke(this, 'rowInterval');
    };

    /**
     * Returns the column or row intervals of all cell range addresses in this
     * array.
     *
     * @param {Boolean} columns
     *  Whether to return the column intervals (true), or the row intervals
     *  (false) of the cell range address.
     *
     * @returns {IntervalArray}
     *  The column or row intervals of all cell range addresses in this array.
     */
    RangeArray.prototype.intervals = function (columns) {
        return columns ? this.colIntervals() : this.rowIntervals();
    };

    /**
     * Returns the address of the bounding cell range of all range addresses in
     * this array (the smallest range that contains all ranges in the array).
     *
     * @returns {Range|Null}
     *  The address of the bounding cell range containing all range addresses;
     *  or null, if this range array is empty.
     */
    RangeArray.prototype.boundary = function () {

        switch (this.length) {
            case 0:
                return null;
            case 1:
                // this method must not return original array elements
                return this[0].clone();
        }

        var result = this[0].clone();
        this.forEach(function (range) {
            result.start[0] = Math.min(result.start[0], range.start[0]);
            result.start[1] = Math.min(result.start[1], range.start[1]);
            result.end[0] = Math.max(result.end[0], range.end[0]);
            result.end[1] = Math.max(result.end[1], range.end[1]);
        });
        return result;
    };

    /**
     * Returns a shallow copy of this cell range address array that does not
     * contain any duplicate cell range addresses.
     *
     * @returns {RangeArray}
     *  A copy of this cell range address array without any duplicates. The
     *  array will contain the same ranges as contained in this array (no deep
     *  clones of the ranges).
     */
    RangeArray.prototype.unify = function () {

        // nothing to do for empty array, or single range; but return a cloned array
        if (this.length <= 1) { return this.clone(); }

        // a map with the unique keys of all result ranges (prevents to search for duplicates in the result)
        var map = {};

        // remove all ranges from the result that are already contained in the map
        return this.reject(function (range) {
            var key = range.key();
            if (map[key]) { return true; }
            map[key] = true;
        });
    };

    /**
     * Returns a new array with only the cell range addresses in this array
     * that are not completely covered by other cell ranges in this array.
     *
     * @returns {RangeArray}
     *  A new array with only the cell range addresses in this array that are
     *  not completely covered by other cell ranges in this array. The array
     *  will contain the same ranges as contained in this array (no deep clones
     *  of the ranges).
     */
    RangeArray.prototype.filterCovered = function () {

        // nothing to do for empty array, or single range; but return a cloned array
        if (this.length <= 1) { return this.clone(); }

        // RangeArray.unify() is fast and may help to reduce the following O(n^2) loop
        var result = this.unify();

        // remove all ranges from the result that are covered by any other range in the array
        // (duplicate ranges have been removed above, therefore they cannot remove each other)
        return result.reject(function (range1, index1) {
            // remove range1 from the result, if it is covered by another range in the array
            return result.some(function (range2, index2) {
                return (index1 !== index2) && range2.contains(range1);
            });
        });
    };

    /**
     * Divides the cell range addresses into a sorted array of column intervals
     * representing the left and right boundaries of all cell ranges.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ranges=false]
     *      If set to true, the elements in the returned array will contain the
     *      additional property 'ranges' (instane of class RangeArray) with the
     *      addresses of the original cell ranges covering the respective
     *      column band, sorted by start row index.
     *  @param {Boolean} [options.intervals=false]
     *      If set to true, the elements in the returned array will contain the
     *      additional property 'intervals' (instance of class IntervalArray)
     *      containing the merged row intervals of the ranges covered by the
     *      respective column band.
     *
     * @returns {IntervalArray}
     *  An array of column intervals representing all cell range addresses.
     *  Each array element contains the additional property 'ranges' if the
     *  option 'ranges' has been set (see above), and the additional property
     *  'intervals', if the option 'intervals' has been set (see above).
     */
    RangeArray.prototype.getColBands = function (options) {
        return createBandIntervals(this, true, options);
    };

    /**
     * Divides the cell range addresses into a sorted array of row intervals
     * representing the upper and lower boundaries of all cell ranges.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ranges=false]
     *      If set to true, the elements in the returned array will contain the
     *      additional property 'ranges' (instane of class RangeArray) with the
     *      addresses of the original cell ranges covering the respective row
     *      band, sorted by start column index.
     *  @param {Boolean} [options.intervals=false]
     *      If set to true, the elements in the returned array will contain the
     *      additional property 'intervals' (instance of class IntervalArray)
     *      containing the merged column intervals of the ranges covered by the
     *      respective row band.
     *
     * @returns {IntervalArray}
     *  An array of row intervals representing all cell range addresses. Each
     *  array element contains the additional property 'ranges' if the option
     *  'ranges' has been set (see above), and the additional property
     *  'intervals', if the option 'intervals' has been set (see above).
     */
    RangeArray.prototype.getRowBands = function (options) {
        return createBandIntervals(this, false, options);
    };

    /**
     * Merges the cell range addresses in this array so that no cell range
     * address in the result overlaps with or adjoins exactly to another cell
     * range address.
     *
     * Example: The range array [A1:D4,C3:F6] will be merged to the ranges
     * A1:D2 (upper part of first range), A3:F4 (lower part of first range, and
     * upper part of second range), and C5:F6 (lower part of second range).
     *
     * @returns {RangeArray}
     *  An array of merged cell range addresses that do not overlap, and that
     *  exactly cover the cell range addresses in this array.
     */
    RangeArray.prototype.merge = function () {

        // at most one range: return directly instead of running the complex row band code
        if (this.length <= 1) { return this.clone(true); }

        // calculate the row bands of the own ranges, add the merged column intervals
        var rowBands = this.getRowBands({ intervals: true });

        // return the merged ranges calculated from the row bands
        return mergeRangesFromRowBands(rowBands);
    };

    /**
     * Works similar to merging the range array (see method RangeArray.merge())
     * but divides the passed ranges into smaller parts, so that the resulting
     * ranges do not cover the boundaries of any of the original ranges.
     *
     * Example: The range array [A1:D4,C3:F6] will be partitioned to the ranges
     * A1:D2 (upper part of first range above the upper boundary of the second
     * range), A3:B4 (lower-left part of first range that does not cover the
     * second range), C3:D4 (common parts of both ranges), E3:F4 (upper-right
     * part of second range), and C5:F6 (lower part of second range).
     *
     * @returns {RangeArray}
     *  An array of cell range addresses that do not overlap, that exactly
     *  cover the cell range addresses in this array, and that are partitioned
     *  in a way that the resulting ranges do not cover any boundary of an
     *  original range. Each range object in the resulting array will contain
     *  an additional property 'coveredBy' that is a RangeArray containing the
     *  original ranges of this instance covering that particular result range.
     */
    RangeArray.prototype.partition = function () {

        // partition the first range according to the second range
        function addPartitionRanges(ranges, range1, range2) {

            // ranges do not overlap: nothing to partition
            if (!range1.overlaps(range2)) {
                ranges.push(range1.clone());
                return;
            }

            // shortcuts to the start/end addresses
            var start1 = range1.start;
            var start2 = range2.start;
            var end1 = range1.end;
            var end2 = range2.end;

            // row indexes of the inner rows covered by both ranges
            var row1 = Math.max(start1[1], start2[1]);
            var row2 = Math.min(end1[1], end2[1]);

            // put the upper part of the first range, if it starts above the second range
            if (start1[1] < start2[1]) {
                ranges.push(Range.create(start1[0], start1[1], end1[0], start2[1] - 1));
            }

            // put the left part of the first range, if it starts left of the second range
            if (start1[0] < start2[0]) {
                ranges.push(Range.create(start1[0], row1, start2[0] - 1, row2));
            }

            // put the common part of both ranges
            ranges.push(range1.intersect(range2));

            // put the right part of the first range, if it ends right of the second range
            if (end1[0] > end2[0]) {
                ranges.push(Range.create(end2[0] + 1, row1, end1[0], row2));
            }

            // put the lower part of the first range, if it ends below the second range
            if (end1[1] > end2[1]) {
                ranges.push(Range.create(start1[0], end2[1] + 1, end1[0], end1[1]));
            }
        }

        // process each range in this array, split the ranges that already exist in the result
        var resultRanges = new RangeArray();
        this.forEach(function (newRange) {

            // partition all collected result ranges, without adding the new range yet
            var nextRanges = new RangeArray();
            resultRanges.forEach(function (resultRange) { addPartitionRanges(nextRanges, resultRange, newRange); });

            // add the uncovered parts of the new range to the result
            resultRanges = nextRanges.append(new RangeArray(newRange).difference(nextRanges));
        });

        // add the original ranges for each result range
        resultRanges.forEach(function (resultRange) {
            resultRange.coveredBy = this.filter(function (origRange) { return origRange.contains(resultRange); });
        }, this);

        return resultRanges;
    };

    /**
     * Returns the cell range addresses covered by this array, and the passed
     * array of cell range addresses. More precisely, returns the existing
     * intersection ranges from all pairs of the cross product of the arrays of
     * cell range addresses.
     *
     * @param {RangeArray|Range} ranges
     *  The other cell range addresses to be intersected with the cell range
     *  addresses in this array. This method also accepts a single cell range
     *  address as parameter.
     *
     * @returns {RangeArray}
     *  The cell range addresses covered by this array and the passed array.
     */
    RangeArray.prototype.intersect = function (ranges) {
        var result = new RangeArray();
        this.forEach(function (range1) {
            RangeArray.forEach(ranges, function (range2) {
                var range = range1.intersect(range2);
                if (range) { result.push(range); }
            });
        });
        return result;
    };

    /**
     * Returns the difference of the cell range addresses in this array, and
     * the passed cell range addresses (all cell range addresses contained in
     * this array, that are not contained in the passed array).
     *
     * @param {RangeArray|Range} ranges
     *  The cell range addresses to be removed from this array. This method
     *  also accepts a single cell range address as parameter.
     *
     * @returns {RangeArray}
     *  All cell range addresses that are contained in this array, but not in
     *  the passed array. May be an empty array, if the passed cell range
     *  addresses completely cover all cell range addresses in this array.
     */
    RangeArray.prototype.difference = function (ranges) {

        // the resulting ranges
        var result = this;

        // reduce ranges in 'result' with each range from the passed array
        RangeArray.some(ranges, function (range2) {

            // initialize the arrays for this iteration
            var source = result;
            result = new RangeArray();

            // process each range in the source range list
            source.forEach(function (range1) {

                // check if the ranges cover each other at all
                if (!range1.overlaps(range2)) {
                    result.push(range1.clone());
                    return;
                }

                // do nothing if range2 covers range1 completely (delete range1 from the result)
                if (range2.contains(range1)) {
                    return;
                }

                // if range1 starts above range2, extract the upper part of range1
                if (range1.start[1] < range2.start[1]) {
                    result.push(Range.create(range1.start[0], range1.start[1], range1.end[0], range2.start[1] - 1));
                }

                // if range1 starts left of range2, extract the left part of range1
                if (range1.start[0] < range2.start[0]) {
                    result.push(Range.create(range1.start[0], Math.max(range1.start[1], range2.start[1]), range2.start[0] - 1, Math.min(range1.end[1], range2.end[1])));
                }

                // if range1 ends right of range2, extract the right part of range1
                if (range1.end[0] > range2.end[0]) {
                    result.push(Range.create(range2.end[0] + 1, Math.max(range1.start[1], range2.start[1]), range1.end[0], Math.min(range1.end[1], range2.end[1])));
                }

                // if range1 ends below range2, extract the lower part of range1
                if (range1.end[1] > range2.end[1]) {
                    result.push(Range.create(range1.start[0], range2.end[1] + 1, range1.end[0], range1.end[1]));
                }
            });

            // early exit the loop, if all source ranges have been deleted already
            return result.length === 0;
        });

        // ensure to return a deep clone
        return (result === this) ? this.clone(true) : result;
    };

    /**
     * Creates an iterator that visits the single addresses of the cell ranges
     * in this array.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the ranges in this array, and the addresses in all
     *      ranges, will be visited in reversed order.
     *  @param {Boolean} [options.columns=false]
     *      If set to true, the addresses in the cell ranges will be generated
     *      in vertical order.
     *  @param {Number} [options.begin]
     *      If specified, only the ranges with an index greater than or equal
     *      to this value (in reversed mode: with an index less than or equal
     *      to this value) will be visited. If omitted, all ranges from the
     *      beginning (in reverse mode: from the end) of this array will be
     *      visited.
     *  @param {Number} [options.end]
     *      If specified, only the ranges with an index less than this value
     *      (in reverse mode: with an index greater than this value) will be
     *      visited (half-open interval!). If omitted, all ranges to the end
     *      (in reverse mode: to the beginning) of this array will be visited.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Address} value
     *      The resulting address inside the cell range currently visited.
     *  - {Range} range
     *      The original cell range from this range array containing the cell
     *      address currently visited.
     *  - {Number} index
     *      The array index of the cell range in the property 'range' of this
     *      result object.
     */
    RangeArray.prototype.addressIterator = function (options) {

        // creates an address iterator from the result of the outer array iterator
        function createAddressIterator(range) {
            return range.iterator(options);
        }

        // combines the results of the outer and inner iterators
        function createIteratorResult(arrayResult, addressResult) {
            addressResult.range = arrayResult.value;
            addressResult.index = arrayResult.index;
            return addressResult;
        }

        // the nested iterator, combining an outer array iterator with an inner address iterator
        return IteratorUtils.createNestedIterator(this.iterator(options), createAddressIterator, createIteratorResult);
    };

    // exports ================================================================

    return RangeArray;

});
