/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/changedescriptor', [
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/rangearray'
], function (AddressArray, RangeArray) {

    'use strict';

    // class ChangeDescriptor =================================================

    /**
     * Contains detailed information about changed cell contents and formatting
     * that will be collected during applying 'changeCells' operations.
     *
     * @constructor
     *
     * @property {AddressArray} allCells
     *  The addresses of all cells with a changed value, formula expression, or
     *  auto-style. The addresses will be in no specific order.
     *
     * @property {AddressArray} valueCells
     *  The addresses of all cells with changed value (including cells that
     *  just became blank). The addresses will be in no specific order.
     *
     * @property {AddressArray} formulaCells
     *  The addresses of all cells with changed formula expression (including
     *  cells where the formula has been deleted). The addresses will be in no
     *  specific order.
     *
     * @property {AddressArray} styleCells
     *  The addresses of all cells with changed auto-style. The addresses will
     *  be in no specific order.
     */
    function ChangeDescriptor() {

        this.allCells = new AddressArray();
        this.valueCells = new AddressArray();
        this.formulaCells = new AddressArray();
        this.styleCells = new AddressArray();

        this._allRanges = null;
        this._valueRanges = null;
        this._formulaRanges = null;
        this._styleRanges = null;

    } // class ChangeDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this descriptor is empty, i.e. it does not contain any
     * cell addresses.
     *
     * @returns {Boolean}
     *  Whether this descriptor is empty.
     */
    ChangeDescriptor.prototype.empty = function () {
        return this.allCells.empty();
    };

    /**
     * Inserts a new cell address into the address array properties of this
     * descriptor, according to the passed change flags.
     *
     * @param {Address} address
     *  The cell address to be inserted into this descriptor.
     *
     * @param {Object} changeFlags
     *  The change flags specifying what has changed in the cell. The following
     *  properties are expected:
     *  @param {Boolean} [changeFlags.value=false]
     *      If set to true, the value of the cell has changed.
     *  @param {Boolean} [changeFlags.formula=false]
     *      If set to true, the formula expression of the cell has changed.
     *  @param {Boolean} [changeFlags.style=false]
     *      If set to true, the auto-style of the cell has changed.
     *
     * @returns {ChangeDescriptor}
     *  A reference to this instance.
     */
    ChangeDescriptor.prototype.addAddress = function (address, changeFlags) {

        var changed = false;

        if (changeFlags.value) { this.valueCells.push(address); changed = true; }
        if (changeFlags.formula) { this.formulaCells.push(address); changed = true; }
        if (changeFlags.style) { this.styleCells.push(address); changed = true; }
        if (changed) { this.allCells.push(address); }

        return this;
    };

    ChangeDescriptor.prototype.getAllRanges = function () {
        return this._allRanges || (this._allRanges = RangeArray.mergeAddresses(this.allCells));
    };

    ChangeDescriptor.prototype.getValueRanges = function () {
        return this._valueRanges || (this._valueRanges = RangeArray.mergeAddresses(this.valueCells));
    };

    ChangeDescriptor.prototype.getFormulaRanges = function () {
        return this._formulaRanges || (this._formulaRanges = RangeArray.mergeAddresses(this.formulaCells));
    };

    ChangeDescriptor.prototype.getStyleRanges = function () {
        return this._styleRanges || (this._styleRanges = RangeArray.mergeAddresses(this.styleCells));
    };

    // exports ================================================================

    return ChangeDescriptor;

});
