/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/services/commandmixin', [
    'io.ox/office/spreadsheet/utils/sheetselection'
], function (SheetSelection) {

    'use strict';

    // mix-in class CommandMixin =============================================

    /**
     *
     * @constructor
     */
    function CommandMixin() {

        var self = this;
        var docModel = null;
        var docView = null;

        // private methods ----------------------------------------------------

        /**
         * Sets the optimal column width for the active sheet in the document
         * view.
         *
         * @returns {jQuery.Promise}
         *  A promise that resolves after setting the optimal column widths.
         */
        function setOptimalColumnWidth() {
            // TODO: use model functionality to change optimal column width

            // rescue the old selection of the active sheet
            var oldSelection = docView.getSelection();
            // the cell collection of the active sheet
            var cellCollection = docView.getCellCollection();

            // select the used area of the sheet
            docView.setCellSelection(new SheetSelection(cellCollection.getUsedRange()));

            // set the optimal column width for the used area, restore the old selection afterwards
            return docView.setOptimalColumnWidth().done(function () {
                docView.setCellSelection(oldSelection);
            });
        }

        /**
         * Sets optimal column width for the specified sheet in the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that resolves after setting the optimal column widths.
         */
        function setOptimalColumnWidthForSheet(sheet) {

            var promise = null;

            if (docView.getActiveSheet() === sheet) {
                promise = setOptimalColumnWidth();
            } else {
                promise = self.waitForEvent(docView, 'update:selection:data');
                promise = promise.then(setOptimalColumnWidth);
                docView.setActiveSheet(sheet);
            }

            return promise;
        }

        /**
         * Sets optimal column width for all sheets in the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that resolves after setting the optimal column widths.
         */
        function setOptimalColumnWidthForDocument() {

            var activeSheet = docView.getActiveSheet();

            var sheetIt = docModel.createSheetIterator({ types: 'worksheet' });
            var promise = docModel.iterateSliced(sheetIt, function (sheetModel, result) {
                return setOptimalColumnWidthForSheet(result.sheet);
            });

            return promise.done(function () {
                docView.setActiveSheet(activeSheet);
            });
        }

        function init() {

            var promise = $.when();

            if (self.getLaunchOption('autoWidth')) {
                promise = promise.then(setOptimalColumnWidthForDocument);
            }

            if (promise.state() === 'pending') {
                docView.enterBusy({ delay: 0, immediate: true });
                docView.waitForAny(promise, function () { docView.leaveBusy(); });
            }
        }

        // initialization -----------------------------------------------------

        this.onInit(function () {
            docModel = self.getModel();
            docView = self.getView();
        });

        this.waitForImportSuccess(function () {
            docView.one('change:usedarea', function () {
                self.executeDelayed(init, { delay: 1000 });
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = docView = null;
        });

    } // class CommandMixin

    // exports ================================================================

    return CommandMixin;

});
