/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/validationcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/utils/arraytemplate',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, SimpleMap, ArrayTemplate, TimerMixin, ModelObject, AttributedModel, Operations, SheetUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var RangeSet = SheetUtils.RangeSet;
    var Range3DArray = SheetUtils.Range3DArray;

    // private global functions ===============================================

    /**
     * Transforms the specified target ranges.
     *
     * @param {RangeArray} targetRanges
     *  The addresses of the target ranges to be transformed.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array of move descriptors that specify how to transform the passed
     *  target ranges.
     *
     * @param {Boolean} [reverse=false]
     *  If set to true, the move descriptors will be processed in reversed
     *  order, and the opposite move operation will be used to transform the
     *  target ranges.
     *
     * @returns {RangeArray}
     *  The transformed target ranges.
     */
    function transformTargetRanges(targetRanges, moveDescs, reverse) {

        // transform the passed ranges, expand the end of the range when inserting cells
        Utils.iterateArray(moveDescs, function (moveDesc) {
            targetRanges = moveDesc.transformRanges(targetRanges, { expandEnd: true, reverse: reverse });
            if (targetRanges.empty()) { return Utils.BREAK; }
        }, { reverse: reverse });

        return targetRanges;
    }

    // class ValidationModel ==================================================

    /**
     * Stores data validation settings for one or more cell ranges in a sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this instance.
     *
     * @param {RangeArray} ranges
     *  The addresses of all cell ranges that contain the data validation
     *  settings.
     *
     * @param {Object} [attrs]
     *  The data validation attribute set.
     */
    var ValidationModel = AttributedModel.extend({ constructor: function (sheetModel, ranges, attrs) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();

        // the target cell ranges covered by the data validation
        var targetRanges = null;

        // the top-left address of the bounging range of the target ranges
        var refAddress = null;

        // the token arrays representing the validation attributes 'value1' and 'value2'
        var tokenArrayMap = new SimpleMap();

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, attrs, { families: 'validation' });

        // private methods ----------------------------------------------------

        /**
         * Updates the token arrays after the validation attributes of this
         * validation model have been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // extract the validation attributes from the passed attribute sets
            newAttributes = (newAttributes || self.getMergedAttributeSet(true)).validation;
            oldAttributes = oldAttributes ? oldAttributes.validation : {};

            // parse a changed formula expression in attribute 'value1' or 'value2'
            tokenArrayMap.forEach(function (tokenArray, propName) {
                if (newAttributes[propName] !== oldAttributes[propName]) {
                    tokenArray.parseFormula('op', newAttributes[propName], { refAddress: refAddress });
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this validation model for
         * the specified sheet.
         *
         * @internal
         *  Used by the class ValidationCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {ValidationModel}
         *  A clone of this validation model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            return new ValidationModel(targetModel, targetRanges, this.getExplicitAttributeSet(true));
        };

        /**
         * Tries to replace unresolved sheet names in the token arrays with
         * existing sheet indexes. Intended to be used after document import to
         * refresh all token arrays that refer to sheets that did not exist
         * during their creation.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.refreshAfterImport = function () {
            var options = { refAddress: refAddress };
            tokenArrayMap.forEach(function (tokenArray) {
                tokenArray.refreshAfterImport(options);
            });
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the addresses of all target cell ranges of this validation
         * model.
         *
         * @returns {RangeArray}
         *  The addresses of all cell ranges.
         */
        this.getRanges = function () {
            return targetRanges.clone(true);
        };

        /**
         * Returns whether the passed cell address is contained by the target
         * cell ranges of this validation model.
         *
         * @param {Address} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this validation model.
         */
        this.containsCell = function (address) {
            return targetRanges.containsAddress(address);
        };

        /**
         * Returns the reference addresses of this validation model (the
         * top-left cell of the bounding range of all target cell ranges of
         * this validation model).
         *
         * @returns {Address}
         *  The address of the reference cell of this validation model.
         */
        this.getRefAddress = function () {
            return refAddress.clone();
        };

        /**
         * Returns the list contents for validations of type 'list' or 'source'
         * according to the formula expression contained in the attribute
         * 'value1'.
         *
         * @param {Address} targetAddress
         *  The target address used to resolve relative references in the
         *  formula expression.
         *
         * @returns {Array<Object>}
         *  The contents of the cells, if this model is a data validation of
         *  type 'list' or 'source', and the formula expression contained in
         *  the attribute 'value1' could be resolved successfully to a list;
         *  otherwise an empty array. Each cell content object will contain the
         *  following properties:
         *  - {Number|String|Boolean|ErrorCode|Null} [value]
         *      The cell value, or formula result (null for blank cells). Will
         *      be omitted for string lists (validation type 'list') to let the
         *      strings being parsed dynamically, resulting in floating-point
         *      numbers with a number format, booleans, or even formula cells.
         *  - {String|Null} display
         *      The display string for the list entry (will be the empty string
         *      for blank cells). The value null represents a cell result that
         *      cannot be formatted to a valid display string with the current
         *      number format of the cell.
         */
        this.queryListContents = function (targetAddress) {

            // resolve source data by validation type
            switch (this.getMergedAttributeSet(true).validation.type) {
                case 'list':
                    var stringList = tokenArrayMap.get('value1').resolveStringList();
                    // convert plain string array to objects with display properties (no values to force parsing the text later)
                    return stringList ? stringList.map(function (text) { return { display: text }; }) : [];

                case 'source':
                    // get the cell range addresses contained in the first token array
                    var result = tokenArrayMap.get('value1').interpretFormula('ref', { refAddress: refAddress, targetAddress: targetAddress });
                    // collect all cell values and display strings, restrict to 1000 entries
                    return (result.value instanceof Range3DArray) ? docModel.getRangeContents(result.value, { display: true, maxCount: 1000 }) : [];
            }

            // default (wrong validation type)
            return [];
        };

        /**
         * Changes the target cell ranges of this validation model.
         *
         * @param {RangeArray} newRanges
         *  The addresses of the new target cell ranges. The old target ranges
         *  of this validation model will be replaced completely.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.setRanges = function (newRanges) {
            if (!targetRanges || !targetRanges.equals(newRanges, true)) {
                targetRanges = newRanges.clone(true);
                refAddress = targetRanges.empty() ? Address.A1.clone() : targetRanges.boundary().start;
            }
            return this;
        };

        /**
         * Changes all formula expressions stored in the validation attributes
         * 'value1' and 'value2' with sheet references, after the sheet
         * collection in the document has been changed.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.transformSheet = function (toSheet, fromSheet) {
            tokenArrayMap.forEach(function (tokenArray) { tokenArray.transformSheet(toSheet, fromSheet); });
            return this;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of this data validation model.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} index
         *  The index of this validation model to be inserted into the document
         *  operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.generateFormulaOperations = function (generator, index, changeDesc) {

            // whether to restore the target ranges (if they cannot be restored implicitly)
            var restoreTargetRanges = false;
            // the properties for the document operation
            var properties = {};
            // the properties for the undo operation
            var undoProperties = {};

            // update the target ranges, if the moved cells are located in the own sheet
            if ((changeDesc.type === 'moveCells') && (sheetModel.getIndex() === changeDesc.sheet)) {

                // restore the entire validation model, if it will be deleted implicitly
                var transformRanges = transformTargetRanges(targetRanges, changeDesc.moveDescs);
                if (transformRanges.empty()) {
                    properties.index = index;
                    _.extend(properties, this.getExplicitAttributeSet(true).validation);
                    generator.generateRangesOperation(Operations.INSERT_VALIDATION, targetRanges, properties, { undo: true });
                    return this; // nothing more to do (do not generate any change operations below)
                }

                // restore the original target ranges on undo, if they cannot be restored implicitly
                var restoredRanges = (targetRanges.length === transformRanges.length) ? transformTargetRanges(transformRanges, changeDesc.moveDescs, true) : null;
                restoreTargetRanges = !restoredRanges || !restoredRanges.equals(targetRanges, true);

                // relocate the formulas according to a changed reference address
                var newRefAddress = restoredRanges ? restoredRanges.boundary().start : null;
                if (newRefAddress && refAddress.differs(newRefAddress)) {
                    changeDesc = _.extend({ relocate: { from: refAddress, to: newRefAddress } }, changeDesc);
                }
            }

            // calculate the new formula expressions for the validation model
            tokenArrayMap.forEach(function (tokenArray, propName) {
                var result = tokenArray.resolveOperation('op', changeDesc);
                if (result) {
                    properties[propName] = result.new;
                    undoProperties[propName] = result.old;
                }
            });

            // generate the change operation with the new formula expressions
            if (!_.isEmpty(properties)) {
                properties.index = index;
                generator.generateSheetOperation(Operations.CHANGE_VALIDATION, properties);
            }

            // generate the undo operation to restore the validation model if required
            if (restoreTargetRanges || !_.isEmpty(undoProperties)) {
                undoProperties.index = index;
                if (restoreTargetRanges) {
                    generator.generateRangesOperation(Operations.CHANGE_VALIDATION, targetRanges, undoProperties, { undo: true });
                } else {
                    generator.generateSheetOperation(Operations.CHANGE_VALIDATION, undoProperties, { undo: true });
                }
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // initialize the target ranges, and calculate the reference address
        this.setRanges(ranges);

        // create the token arrays for the formula expressions in 'value1' and 'value2'
        tokenArrayMap.insert('value1', new TokenArray(sheetModel, 'validation'));
        tokenArrayMap.insert('value2', new TokenArray(sheetModel, 'validation'));

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // destroy all class members
        this.registerDestructor(function () {
            self = docModel = sheetModel = tokenArrayMap = null;
        });

    } }); // class ValidationModel

    // class ValidationModelArray =============================================

    var ValidationModelArray = ArrayTemplate.create(ValidationModel);

    // class ValidationCollection =============================================

    /**
     * Stores data validation settings in a specific sheet. Data validation can
     * be used to restrict the values allowed to be entered in specific cell
     * ranges, and to show tooltips and error messages.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function ValidationCollection(sheetModel) {

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();

        // all model instances with validation settings, as array (in operation order)
        var valModels = new ValidationModelArray();

        // all model instances with validation settings, mapped by target ranges
        var targetRangeSet = new RangeSet();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Inserts the target ranges of the passed validation model into the
         * internal range set.
         *
         * @param {ValidationModel} valModel
         *  The validation model whose target ranges will be inserted into the
         *  range set.
         */
        function insertTargetRanges(valModel) {
            valModel.getRanges().forEach(function (range) {
                range.valModel = valModel;
                targetRangeSet.insert(range);
            });
        }

        /**
         * Removes the target ranges of the passed validation model from the
         * internal range set.
         *
         * @param {ValidationModel} valModel
         *  The validation model whose target ranges will be removed from the
         *  range set.
         */
        function removeTargetRanges(valModel) {
            valModel.getRanges().forEach(function (range) {
                targetRangeSet.remove(range);
            });
        }

        /**
         * Returns the validation model that covers the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {ValidationModel|Null}
         *  The validation for the specified cell. If the cell is not validated
         *  at all, returns null.
         */
        function findModelByAddress(address) {
            var targetRanges = targetRangeSet.findByAddress(address);
            // result is an array with at most one element (target ranges cannot overlap)
            return targetRanges.empty() ? null : targetRanges.first().valModel;
        }

        /**
         * Returns the validation model addressed by the property 'index' in
         * the passed document operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for validations,
         *  expected to contain an integer property 'index'.
         *
         * @returns {Object}
         *  A descriptor for the validation model and its index, in the
         *  following properties:
         *  - {Number} index
         *      The array index of the validation model (the value of the
         *      'index' property in the passed operation).
         *  - {ValidationModel} model
         *      The validation model referenced by the document operation.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid 'index' property, or if
         *  the validation model with such an index does not exist.
         */
        function getModelData(context) {

            // get the index from the operation
            var index = context.getInt('index');

            // check existence of the validation model
            var valModel = valModels[index];
            context.ensure(valModel, 'missing validation');

            return { index: index, model: valModel };
        }

        /**
         * Recalculates the target ranges of all validation models, after cells
         * have been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {

            // simply fill a new target range set while transforming the ranges
            targetRangeSet = new RangeSet();

            // process all validation models in revered order to be able to delete array elements
            valModels.forEachReverse(function (valModel, index) {
                var targetRanges = transformTargetRanges(valModel.getRanges(), [moveDesc]);
                if (targetRanges.empty()) {
                    valModels.splice(index, 1);
                } else {
                    valModel.setRanges(targetRanges);
                    insertTargetRanges(valModel);
                }
            });
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { valModels: valModels };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * contents from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {ValidationCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            cloneData.valModels.forEach(function (valModel) {
                valModel = valModel.clone(sheetModel);
                valModels.push(valModel);
                insertTargetRanges(valModel);
            });
        };

        /**
         * Callback handler for the document operation 'insertValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertValidation' document operation.
         */
        this.applyInsertOperation = function (context) {

            // get and validate the insertion index
            var index = context.getInt('index');
            context.ensure((index >= 0) && (index <= valModels.length), 'invalid index');

            // the target ranges (must not overlap with any other validation models)
            var targetRanges = context.getRangeArray().merge();
            context.ensure(!targetRangeSet.overlaps(targetRanges), 'overlapping target ranges');

            // create and insert the new validation model
            var valModel = new ValidationModel(sheetModel, targetRanges, { validation: context.operation });
            valModels.splice(index, 0, valModel);
            insertTargetRanges(valModel);

            // TODO: notify listeners
        };

        /**
         * Callback handler for the document operation 'deleteValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteValidation' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the validation model addressed by the operation (throws on error)
            var modelData = getModelData(context);
            var valModel = modelData.model;

            // destroy and remove the validation model
            removeTargetRanges(valModel);
            valModels.splice(modelData.index, 1);
            valModel.destroy();

            // TODO: notify listeners
        };

        /**
         * Callback handler for the document operation 'changeValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeValidation' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the validation model addressed by the operation (throws on error)
            var modelData = getModelData(context);
            var valModel = modelData.model;

            // resolve optional 'ranges' property (throws on error), change target ranges
            var targetRanges = context.getOptRangeArray();
            if (targetRanges) {
                removeTargetRanges(valModel);
                context.ensure(!targetRangeSet.overlaps(targetRanges), 'overlapping target ranges');
                valModel.setRanges(targetRanges);
                insertTargetRanges(valModel);
            }

            // change all other validation attributes
            modelData.model.setAttributes({ validation: context.operation });

            // TODO: notify listeners
        };

        // public methods -----------------------------------------------------

        /**
         * Creates an iterator that visits the models of all data validations
         * in this collection.
         *
         * @returns {Object}
         *  An iterator object with the method next(). The result object of the
         *  iterator will contain the following properties:
         *  - {ValidationModel} value
         *      The data validation model currently visited.
         *  - {Number} index
         *      The array index of the data validation model, as used for the
         *      property 'index' in document operations.
         */
        this.createModelIterator = function () {
            return valModels.iterator();
        };

        /**
         * Returns the validation settings for the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Object|Null}
         *  The validation settings of the specified cell, in the following
         *  properties:
         *  - {RangeArray} ranges
         *      The addresses of all related cell ranges containing the same
         *      validation settings.
         *  - {Object} attributes
         *      All validation attributes.
         *  If the cell is not validated at all, returns null.
         */
        this.getValidationSettings = function (address) {
            var valModel = findModelByAddress(address);
            return valModel ? {
                ranges: valModel.getRanges(),
                attributes: valModel.getMergedAttributeSet(true).validation
            } : null;
        };

        /**
         * Returns the list contents for a data validation of type 'list' or
         * 'source' at the specified cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Array<Object>}
         *  The contents of the cells, if the specified cell contains a data
         *  validation model of type 'list' or 'source'; otherwise an empty
         *  array. See method ValidationModel.queryListContents() for details.
         */
        this.queryListContents = function (address) {
            var valModel = findModelByAddress(address);
            return valModel ? valModel.queryListContents(address) : [];
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the data validations in this collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateFormulaOperations = function (generator, changeDesc) {
            return this.iterateArraySliced(valModels, function (valModel, index) {
                valModel.generateFormulaOperations(generator, index, changeDesc);
            }, { delay: 'immediate' });
        };

        // initialization -----------------------------------------------------

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function (alreadyImported) {

            // refresh all formulas after import is finished (formulas may refer to sheets not yet imported)
            if (!alreadyImported) {
                valModels.forEach(function (valModel) {
                    valModel.refreshAfterImport();
                });
            }

            // update the sheet indexes after the sheet collection has been manipulated
            this.listenTo(docModel, 'transform:sheet', function (event, toSheet, fromSheet) {
                valModels.forEach(function (valModel) {
                    valModel.transformSheet(toSheet, fromSheet);
                });
            });

            // update target ranges after moving cells, or inserting/deleting columns or rows
            this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        }, this);

        // destroy all class members
        this.registerDestructor(function () {
            valModels.forEach(function (valModel) { valModel.destroy(); });
            docModel = sheetModel = valModels = targetRangeSet = null;
        });

    } // class ValidationCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ValidationCollection });

});
