/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/operand', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/matrix'
], function (Utils, SheetUtils, FormulaUtils, Matrix) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;

    // class Operand ==========================================================

    /**
     * Represents a single operand stored in the operand stack of a formula
     * token interpreter.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Any} value
     *  The value to be stored as operand. Can be a scalar value (numbers,
     *  strings, boolean values, null), a Date object (will be converted to a
     *  floating-point number immediately), an error code (instance of the
     *  class ErrorCode), a matrix (instance of the class Matrix), a single
     *  cell range address (instance of the class Range3D), or an array of cell
     *  range addresses (instance of the class Range3DArray), or another
     *  instance of this class for copy construction.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {ParsedFormat} [options.format]
     *      A parsed number format associated with the value contained in this
     *      operand. If omitted, this operand does not have a special number
     *      format associated to it (the original number format will not
     *      change).
     *  @param {String} [options.url]
     *      The URL of a hyperlink that will be part of the operand value. The
     *      URL will be carried through the formula result, and can be used by
     *      the user interface to provide a clickable link.
     */
    function Operand(context, value, options) {

        this._context = context;
        this._type = null;
        this._value = null;
        this._format = Utils.getObjectOption(options, 'format', null);
        this._url = Utils.getStringOption(options, 'url', null);

        // initialization -----------------------------------------------------

        // copy construction (options for format and URL passed to constructor win over the operand's settings)
        if (value instanceof Operand) {
            this._type = value._type;
            this._value = value._value;
            if (this._format === null) { this._format = value._format; }
            if (this._url === null) { this._url = value._url; }
            return;
        }

        // cell range addresses passed: validate size
        if ((value instanceof Range3D) || (value instanceof Range3DArray)) {
            value = Range3DArray.get(value);
            if (value.empty()) {
                value = ErrorCode.NULL;
                // continue below, operand becomes an error code (a scalar value)
            } else if (value.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                value = FormulaUtils.UNSUPPORTED_ERROR;
                // continue below, operand becomes an error code (a scalar value)
            } else {
                //ReferenceOperandMixin.call(this, context, value);
                this._type = 'ref';
                this._value = value;
                return;
            }
        }

        // matrix passed: validate matrix size, and all values in the matrix
        if (value instanceof Matrix) {
            FormulaUtils.ensureMatrixSize(value.rows(), value.cols());
            value.transform(context.validateValue, context);
            this._type = 'mat';
            this._value = value;
            return;
        }

        // otherwise: scalar value (as last check, preceding code may have changed the value)
        this._type = 'val';
        this._value = context.validateValue(value);

    } // class Operand

    // public methods ---------------------------------------------------------

    /**
     * Returns the effective data type of this operand.
     *
     * @returns {String}
     *  The effective data type of this operand. Will be one of the following
     *  strings:
     *  - 'val': This operand contains a scalar value (number, string, etc.).
     *  - 'mat': This operand contains a matrix.
     *  - 'ref': This operand contains an array of cell range addresses.
     */
    Operand.prototype.getType = function () {
        return this._type;
    };

    /**
     * Returns the exact unconverted value of this operand.
     *
     * @returns {Any}
     *  The exact unconverted value of this operand.
     */
    Operand.prototype.getRawValue = function () {
        return this._value;
    };

    /**
     * Returns whether this operand contains a scalar value (type 'val').
     *
     * @returns {Boolean}
     *  Whether this operand contains a scalar value (type 'val').
     */
    Operand.prototype.isValue = function () {
        return this._type === 'val';
    };

    /**
     * Returns whether this operand contains a matrix (type 'mat').
     *
     * @returns {Boolean}
     *  Whether this operand contains a matrix (type 'mat').
     */
    Operand.prototype.isMatrix = function () {
        return this._type === 'mat';
    };

    /**
     * Returns whether this operand contains a cell reference (type 'ref').
     *
     * @returns {Boolean}
     *  Whether this operand contains a cell reference (type 'ref').
     */
    Operand.prototype.isReference = function () {
        return this._type === 'ref';
    };

    /**
     * Returns whether this operand represents the empty value (e.g. the second
     * operand in the formula =SUM(1,,2) is empty).
     *
     * @returns {Boolean}
     *  Whether this operand represents the empty value.
     */
    Operand.prototype.isEmpty = function () {
        return this._value === null;
    };

    /**
     * Returns the total size (number of scalar values) of this operand.
     *
     * @returns {Number}
     *  The total size of this operand: The value 1 for scalar operands, the
     *  number of elements for a matrix operand, or the number of cells
     *  (including all blank cells) for a reference operand.
     */
    Operand.prototype.getSize = function () {

        switch (this._type) {
            case 'mat': return this._value.size();
            case 'ref': return this._value.cells();
        }

        // scalar value
        return 1;
    };

    /**
     * Returns the dimensions (width and height) of this operand.
     *
     * @returns {Object}
     *  The dimensions of this operand, in the properties 'rows' and 'cols'.
     *
     * @throws {ErrorCode}
     *  - The #VALUE! error code, if this is a reference operand containing
     *      more than one cell range, or if the cell range refers to multiple
     *      sheets.
     */
    Operand.prototype.getDimensions = function () {

        // return dimensions of scalar and matrix
        switch (this._type) {
            case 'val': return { rows: 1, cols: 1 };
            case 'mat': return { rows: this._value.rows(), cols: this._value.cols() };
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error)
        var range = this._context.convertToRange(this._value, { valueError: true });
        return { rows: range.rows(), cols: range.cols() };
    };

    /**
     * Converts this operand to a scalar value.
     *
     * @param {Number} row
     *  The relative row index, used to resolve a scalar value in a matrix
     *  context. Specifies the row of a matrix, or the relative row index in a
     *  reference operand.
     *
     * @param {Number} col
     *  The relative column index, used to resolve a scalar value in a matrix
     *  context. Specifies the column of a matrix, or the relative column index
     *  in a reference operand.
     *
     * @returns {Any}
     *  The scalar value represented by this operand. Returns an existing
     *  scalar value, or the specified element of a matrix, or the contents of
     *  the specified cell from a cell reference (only if it does not cover the
     *  reference cell of the formula context passed to the constructor).
     *
     * @throws {ErrorCode}
     *  - CIRCULAR_ERROR, if the operand results in a circular reference.
     *  - The #VALUE! error code, if the reference contains more than one
     *      cell range, or if the cell range cannot be resolved to a single
     *      value according to the reference cell.
     */
    Operand.prototype.getValue = function (row, col) {

        // return scalar value, and an element of a matrix
        switch (this._type) {
            case 'val': return this._value;
            case 'mat': return this._value.get(row || 0, col || 0);
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error)
        var range = this._context.convertToValueRange(this._value);

        // pick value from a single cell
        if (range.single()) {
            return this._context.getCellValue(range.sheet1, range.start);
        }

        // the reference cell of the formula context
        var refAddress = this._context.getRefAddress();

        // pick matching cell from a column range (row interval must cover the reference cell)
        if (range.singleCol()) {
            if (!range.containsRow(refAddress[1])) { throw ErrorCode.VALUE; }
            return this._context.getCellValue(range.sheet1, new Address(range.start[0], refAddress[1]));
        }

        // pick matching cell from a row range (column interval must cover the reference cell)
        if (range.singleRow()) {
            if (!range.containsCol(refAddress[0])) { throw ErrorCode.VALUE; }
            return this._context.getCellValue(range.sheet1, new Address(refAddress[0], range.start[1]));
        }

        // range is not left of, right of, above, or below the reference cell
        throw ErrorCode.VALUE;
    };

    /**
     * Converts this operand to a matrix.
     *
     * @returns {Matrix}
     *  The matrix represented by this operand. Returns an existing matrix, or
     *  a scalar value packed into a 1x1 matrix, or the contents of the
     *  referenced cells packed into a matrix. The reference must consist of a
     *  single cell range address that does not cover the reference cell, and
     *  the range must not exceed specific limits in order to prevent
     *  performance problems.
     *
     * @throws {ErrorCode}
     *  - UNSUPPORTED_ERROR, if the resulting matrix would be too large.
     *  - CIRCULAR_ERROR, if the operand results in a circular reference.
     *  - The #VALUE! error code, if the operand contains a reference with more
     *      than one cell range address.
     */
    Operand.prototype.getMatrix = function () {

        // pack scalar value into a 1x1 matrix, or return an existing matrix
        switch (this._type) {
            case 'val': return new Matrix([[this._value]]);
            case 'mat': return this._value;
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error), and return the matrix
        var range3d = this._context.convertToValueRange(this._value);
        return this._context.getCellMatrix(range3d.sheet1, range3d.toRange());
    };

    /**
     * Converts this operand to an unresolved array of cell range addresses.
     *
     * @returns {Range3DArray}
     *  The unresolved cell range addresses contained in this operand.
     *
     * @throws {ErrorCode}
     *  A literal error code carried by this operand; or the #VALUE! error, if
     *  this operand is not reference type.
     */
    Operand.prototype.getRanges = function () {

        // throw #VALUE! error for scalar values and matrixes
        switch (this._type) {
            case 'val':
                throw (this._value instanceof ErrorCode) ? this._value : ErrorCode.VALUE;
            case 'mat':
                throw ErrorCode.VALUE;
            // otherwise, this operand is a reference
        }

        // return the value (always an instance of Range3DArray)
        return this._value;
    };

    /**
     * Returns the parsed number format associated to this operand, as passed
     * to its constructor.
     *
     * @returns {ParsedFormat|Null}
     *  The parsed number format associated to this operand; or null, if no
     *  number format is available.
     */
    Operand.prototype.getFormat = function () {
        return this._format;
    };

    /**
     * Returns the URL of a hyperlink associated to this operand, as passed to
     * its constructor.
     *
     * @returns {String|Null}
     *  The URL of a hyperlink associated to this operand; or null, if no URL
     *  is available.
     */
    Operand.prototype.getURL = function () {
        return this._url;
    };

    /**
     * Returns the string representation of this operand for debug logging.
     */
    Operand.prototype.toString = function () {
        var str = (this._type === 'val') ? FormulaUtils.valueToString(this._value) : this._value.toString();
        if (this._format) { str += ' [' + this._format.category + ']'; }
        return str;
    };

    // exports ================================================================

    return Operand;

});
