/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/matrix', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (IteratorUtils, ErrorCode, FormulaUtils) {

    'use strict';

    // class Matrix ===========================================================

    /**
     * Representation of a matrix literal: a two-dimensional array of constant
     * scalar values (numbers, strings, booleans, and error codes).
     *
     * @constructor
     *
     * @param {Array<Array<Any>>} array
     *  The two-dimensional array of scalars which will become the contents of
     *  this matrix instance. The array MUST contain at least one inner array.
     *  All inner arrays MUST contain at least one value, and all MUST contain
     *  the same number of values.
     */
    function Matrix(array) {

        this._array = array;

    } // class Matrix

    // static methods ---------------------------------------------------------

    /**
     * Creates a new matrix with the specified size, and fills it with the
     * passed constant values.
     *
     * @param {Number} rows
     *  The number of rows contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Number} cols
     *  The number of columns contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value to be inserted into the matrix elements.
     *
     * @returns {Matrix}
     *  The new matrix instance.
     */
    Matrix.create = function (rows, cols, value) {
        return Matrix.generate(rows, cols, _.constant(value));
    };

    /**
     * Creates a new matrix with the specified size, and fills it with the
     * results of the specified generator callback function.
     *
     * @param {Number} rows
     *  The number of rows contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Number} cols
     *  The number of columns contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Function} generator
     *  The generator callback function that will be invoked for every matrix
     *  element. Receives the following parameters:
     *  (1) {Number} row
     *      The zero-based row index of the matrix element.
     *  (2) {Number} col
     *      The zero-based column index of the matrix element.
     *  Note the parameter order (row index before column index), which is the
     *  usual order for mathematical matrixes, but differs from the parameter
     *  order of other methods taking column/row indexes of cell addresses
     *  (column index before row index). The return value of the callback
     *  function MUST be a scalar value used in spreadsheet formulas (numbers,
     *  strings, booleans, error codes, or null), and becomes the value of the
     *  respective matrix element.

     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {Matrix}
     *  The new matrix instance.
     */
    Matrix.generate = function (rows, cols, generator, context) {

        // create the two-dimensional array
        var array = _.times(rows, function (row) {
            return _.times(cols, function (col) {
                return generator.call(context, row, col);
            });
        });

        // create the matrix
        return new Matrix(array);
    };

    /**
     * Creates a new matrix consisting of a single row, and fills it with the
     * passed scalar values.
     *
     * @param {Array<Any>} values
     *  The scalar values to be inserted into the row vector. The array MUST
     *  NOT be empty. For performance, the new matrix takes ownership of this
     *  array.
     *
     * @returns {Matrix}
     *  A new matrix, consisting of one row, containing the elements of the
     *  passed array.
     */
    Matrix.createRowVector = function (values) {
        return new Matrix([values]);
    };

    /**
     * Creates a new matrix consisting of a single column, and fills it with
     * the passed scalar values.
     *
     * @param {Array<Any>} values
     *  The scalar values to be inserted into the column vector.
     *
     * @returns {Matrix}
     *  A new matrix, consisting of one column, containing the elements of the
     *  passed array.
     */
    Matrix.createColVector = function (values) {
        return Matrix.generate(values.length, 1, function (row) { return values[row]; });
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a deep clone of this matrix.
     *
     * @returns {Matrix}
     *  A deep clone of this matrix
     */
    Matrix.prototype.clone = function () {
        return new Matrix(_.invoke(this._array, 'slice'));
    };

    /**
     * Returns the number of rows contained in this matrix.
     *
     * @returns {Number}
     *  The number of rows contained in this matrix.
     */
    Matrix.prototype.rows = function () {
        return this._array.length;
    };

    /**
     * Returns the number of columns contained in this matrix.
     *
     * @returns {Number}
     *  The number of columns contained in this matrix.
     */
    Matrix.prototype.cols = function () {
        return this._array[0].length;
    };

    /**
     * Returns the number of elements contained in this matrix.
     *
     * @returns {Number}
     *  The number of elements contained in this matrix.
     */
    Matrix.prototype.size = function () {
        return this.rows() * this.cols();
    };

    /**
     * Returns the value of the specified element in this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the element. If the matrix consists of a
     *  single row only, this index will be ignored, and the element from the
     *  existing row will be used instead (matrix auto-expansion as used in
     *  spreadsheet formulas).
     *
     * @param {Number} col
     *  The zero-based column index of the element. If the matrix consists of a
     *  single column only, this index will be ignored, and the element from
     *  the existing column will be used instead (matrix auto-expansion as used
     *  in spreadsheet formulas).
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The value of the matrix element.
     *
     * @throws {ErrorCode}
     *  The #N/A error code, if the passed index is located outside the matrix
     *  (except for auto-expansion, see above).
     */
    Matrix.prototype.get = function (row, col) {
        var rows = this.rows(), cols = this.cols();
        // auto-expansion for single row or column vector
        if (rows === 1) { row = 0; }
        if (cols === 1) { col = 0; }
        if ((row < rows) && (col < cols)) { return this._array[row][col]; }
        throw ErrorCode.NA;
    };

    /**
     * Returns the specified row vector of this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the vector. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @returns {Array<Number|String|Boolean|ErrorCode|Null>}
     *  The specified row vector of this matrix.
     */
    Matrix.prototype.getRow = function (row) {
        return this._array[row].slice();
    };

    /**
     * Returns the specified column vector of this matrix.
     *
     * @param {Number} col
     *  The zero-based column index of the vector. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @returns {Array<Number|String|Boolean|ErrorCode|Null>}
     *  The specified column vector of this matrix.
     */
    Matrix.prototype.getCol = function (col) {
        return this._array.map(function (row) { return row[col]; });
    };

    /**
     * Returns the value of the specified element in this matrix.
     *
     * @param {Number} index
     *  The zero-based row-oriented index of a matrix element. MUST be a valid
     *  index (not negative, less than the number of elements in the matrix).
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The value of the matrix element.
     */
    Matrix.prototype.getByIndex = function (index) {
        var cols = this.cols();
        return this._array[Math.floor(index / cols)][index % cols];
    };

    /**
     * Changes the value of the specified element in this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the element. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @param {Number} col
     *  The zero-based column index of the element. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The new value for the matrix element.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.set = function (row, col, value) {
        this._array[row][col] = value;
        return this;
    };

    /**
     * Changes the value of the specified element in this matrix.
     *
     * @param {Number} index
     *  The zero-based row-oriented index of a matrix element. MUST be a valid
     *  index (not negative, less than the number of elements in the matrix).
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The new value for the matrix element.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.setByIndex = function (index, value) {
        var cols = this.cols();
        this._array[Math.floor(index / cols)][index % cols] = value;
        return this;
    };

    /**
     * Creates an iterator that visits all elements contained in this matrix.
     * The elements will be visited row-by-row.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the matrix elements have been visited completely.
     *      No more elements are available; this result object does not contain
     *      any other properties!
     *  - {Number|String|Boolean|ErrorCode|Null} value
     *      The value of the current matrix element.
     *  - {Number} row
     *      The zero-based row index of the matrix element.
     *  - {Number} col
     *      The zero-based column index of the matrix element.
     */
    Matrix.prototype.iterator = function () {

        // combines the results of the outer row iterator, and the inner element iterator
        function createIteratorResult(rowResult, colResult) {
            return { value: colResult.value, row: rowResult.index, col: colResult.index };
        }

        // create a nested iterator, that visits the matrix rows (outer), and the elements in a row (inner)
        var rowIterator = IteratorUtils.createArrayIterator(this._array);
        return IteratorUtils.createNestedIterator(rowIterator, IteratorUtils.createArrayIterator, createIteratorResult);
    };

    /**
     * Invokes the passed callback function for all elements contained in this
     * matrix.
     *
     * @param {Function} callback
     *  The callback function invoked for all elements in this matrix. Receives
     *  the following parameters:
     *  (1) {Number|String|Boolean|ErrorCode|Null} value
     *      The current matrix element.
     *  (2) {Number} row
     *      The zero-based row index of the matrix element.
     *  (3) {Number} col
     *      The zero-based column index of the matrix element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.forEach = function (callback, context) {
        this._array.forEach(function (vector, row) {
            vector.forEach(function (value, col) {
                callback.call(context, value, row, col);
            });
        });
        return this;
    };

    /**
     * Invokes the passed callback function for all elements contained in this
     * matrix, and sets the return values as new element values.
     *
     * @param {Function} callback
     *  The callback function invoked for all elements in this matrix. Receives
     *  the following parameters:
     *  (1) {Number|String|Boolean|ErrorCode|Null} element
     *      The current matrix element.
     *  (2) {Number} row
     *      The zero-based row index of the matrix element.
     *  (3) {Number} col
     *      The zero-based column index of the matrix element.
     *  Must return the new value for the element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.transform = function (callback, context) {
        this.forEach(function (elem, row, col) {
            this._array[row][col] = callback.call(context, elem, row, col);
        }, this);
        return this;
    };

    /**
     * Returns the string representation of this matrix for debug logging.
     *
     * @attention
     *  DO NOT use this method for GUI related code where correct locale
     *  dependent representation of the matrix elements is required.
     *
     * @returns {String}
     *  The string representation of this matrix for debug logging.
     */
    Matrix.prototype.toString = function () {
        return '{' + this._array.map(function (elems) {
            return elems.map(FormulaUtils.valueToString).join(';');
        }).join('|') + '}';
    };

    // exports ================================================================

    return Matrix;

});
