/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/formulagrammar', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/formularesource'
], function (Utils, LocaleData, Formatter, SheetUtils, CellRef, SheetRef, FormulaUtils, FormulaResource) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;

    // settings for all supported formula grammars, mapped by grammar identifiers
    var GRAMMAR_CONFIG = {
        op: { localized: false, locale: LocaleData.LOCALE },
        ui: { localized: true,  locale: LocaleData.LOCALE },
        en: { localized: true,  locale: 'en_US' }
    };

    // prefix for user-defined macro function in the OOXML file format
    var UDF_PREFIX = '_xludf.';

    // character ranges (without brackets) for a whitespace or other non-printable character
    var WHITESPACE_CHARS = '\\s\\x00-\\x1f\\x80-\\x9f';

    // character ranges (without brackets) for the leading character of a name or identifier
    var NAME_LEADING_CHARS = 'A-Z_a-z\\xA1\\xA4\\xA7-\\xA8\\xAA\\xAF-\\xBA\\xBC-\\u02B8\\u02BB-\\u02C1\\u02C7\\u02C9-\\u02CB\\u02CD\\u02D0-\\u02D1\\u02D8-\\u02DB\\u02DD\\u02E0-\\u02E4\\u02EE\\u0370-\\u0373\\u0376-\\u0377\\u037A-\\u037D\\u0386\\u0388-\\u038A\\u038C\\u038E-\\u03A1\\u03A3-\\u03F5\\u03F7-\\u0481\\u048A-\\u0523\\u0531-\\u0556\\u0559\\u0561-\\u0587\\u05D0-\\u05EA\\u05F0-\\u05F2\\u0621-\\u063F\\u0641-\\u064A\\u066E-\\u066F\\u0671-\\u06D3\\u06D5\\u06E5-\\u06E6\\u06EE-\\u06EF\\u06FA-\\u06FC\\u06FF\\u0710\\u0712-\\u072F\\u074D-\\u07A5\\u07B1\\u07CA-\\u07EA\\u07F4-\\u07F5\\u07FA\\u0904-\\u0939\\u093D\\u0950\\u0958-\\u0961\\u0971-\\u0972\\u097B-\\u097F\\u0985-\\u098C\\u098F-\\u0990\\u0993-\\u09A8\\u09AA-\\u09B0\\u09B2\\u09B6-\\u09B9\\u09BD\\u09CE\\u09DC-\\u09DD\\u09DF-\\u09E1\\u09F0-\\u09F1\\u0A05-\\u0A0A\\u0A0F-\\u0A10\\u0A13-\\u0A28\\u0A2A-\\u0A30\\u0A32-\\u0A33\\u0A35-\\u0A36\\u0A38-\\u0A39\\u0A59-\\u0A5C\\u0A5E\\u0A72-\\u0A74\\u0A85-\\u0A8D\\u0A8F-\\u0A91\\u0A93-\\u0AA8\\u0AAA-\\u0AB0\\u0AB2-\\u0AB3\\u0AB5-\\u0AB9\\u0ABD\\u0AD0\\u0AE0-\\u0AE1\\u0B05-\\u0B0C\\u0B0F-\\u0B10\\u0B13-\\u0B28\\u0B2A-\\u0B30\\u0B32-\\u0B33\\u0B35-\\u0B39\\u0B3D\\u0B5C-\\u0B5D\\u0B5F-\\u0B61\\u0B71\\u0B83\\u0B85-\\u0B8A\\u0B8E-\\u0B90\\u0B92-\\u0B95\\u0B99-\\u0B9A\\u0B9C\\u0B9E-\\u0B9F\\u0BA3-\\u0BA4\\u0BA8-\\u0BAA\\u0BAE-\\u0BB9\\u0BD0\\u0C05-\\u0C0C\\u0C0E-\\u0C10\\u0C12-\\u0C28\\u0C2A-\\u0C33\\u0C35-\\u0C39\\u0C3D\\u0C58-\\u0C59\\u0C60-\\u0C61\\u0C85-\\u0C8C\\u0C8E-\\u0C90\\u0C92-\\u0CA8\\u0CAA-\\u0CB3\\u0CB5-\\u0CB9\\u0CBD\\u0CDE\\u0CE0-\\u0CE1\\u0D05-\\u0D0C\\u0D0E-\\u0D10\\u0D12-\\u0D28\\u0D2A-\\u0D39\\u0D3D\\u0D60-\\u0D61\\u0D7A-\\u0D7F\\u0D85-\\u0D96\\u0D9A-\\u0DB1\\u0DB3-\\u0DBB\\u0DBD\\u0DC0-\\u0DC6\\u0E01-\\u0E3A\\u0E40-\\u0E4E\\u0E81-\\u0E82\\u0E84\\u0E87-\\u0E88\\u0E8A\\u0E8D\\u0E94-\\u0E97\\u0E99-\\u0E9F\\u0EA1-\\u0EA3\\u0EA5\\u0EA7\\u0EAA-\\u0EAB\\u0EAD-\\u0EB0\\u0EB2-\\u0EB3\\u0EBD\\u0EC0-\\u0EC4\\u0EC6\\u0EDC-\\u0EDD\\u0F00\\u0F40-\\u0F47\\u0F49-\\u0F6C\\u0F88-\\u0F8B\\u1000-\\u102A\\u103F\\u1050-\\u1055\\u105A-\\u105D\\u1061\\u1065-\\u1066\\u106E-\\u1070\\u1075-\\u1081\\u108E\\u10A0-\\u10C5\\u10D0-\\u10FA\\u10FC\\u1100-\\u1159\\u115F-\\u11A2\\u11A8-\\u11F9\\u1200-\\u1248\\u124A-\\u124D\\u1250-\\u1256\\u1258\\u125A-\\u125D\\u1260-\\u1288\\u128A-\\u128D\\u1290-\\u12B0\\u12B2-\\u12B5\\u12B8-\\u12BE\\u12C0\\u12C2-\\u12C5\\u12C8-\\u12D6\\u12D8-\\u1310\\u1312-\\u1315\\u1318-\\u135A\\u1380-\\u138F\\u13A0-\\u13F4\\u1401-\\u166C\\u166F-\\u1676\\u1681-\\u169A\\u16A0-\\u16EA\\u16EE-\\u16F0\\u1700-\\u170C\\u170E-\\u1711\\u1720-\\u1731\\u1740-\\u1751\\u1760-\\u176C\\u176E-\\u1770\\u1780-\\u17B3\\u17D7\\u17DC\\u1820-\\u1877\\u1880-\\u18A8\\u18AA\\u1900-\\u191C\\u1950-\\u196D\\u1970-\\u1974\\u1980-\\u19A9\\u19C1-\\u19C7\\u1A00-\\u1A16\\u1B05-\\u1B33\\u1B45-\\u1B4B\\u1B83-\\u1BA0\\u1BAE-\\u1BAF\\u1C00-\\u1C23\\u1C4D-\\u1C4F\\u1C5A-\\u1C7D\\u1D00-\\u1DBF\\u1E00-\\u1F15\\u1F18-\\u1F1D\\u1F20-\\u1F45\\u1F48-\\u1F4D\\u1F50-\\u1F57\\u1F59\\u1F5B\\u1F5D\\u1F5F-\\u1F7D\\u1F80-\\u1FB4\\u1FB6-\\u1FBC\\u1FBE\\u1FC2-\\u1FC4\\u1FC6-\\u1FCC\\u1FD0-\\u1FD3\\u1FD6-\\u1FDB\\u1FE0-\\u1FEC\\u1FF2-\\u1FF4\\u1FF6-\\u1FFC\\u2010\\u2013-\\u2016\\u2018\\u201C-\\u201D\\u2020-\\u2021\\u2025-\\u2027\\u2030\\u2032-\\u2033\\u2035\\u203B\\u2071\\u2074\\u207F\\u2081-\\u2084\\u2090-\\u2094\\u2102-\\u2103\\u2105\\u2107\\u2109-\\u2113\\u2115-\\u2116\\u2119-\\u211D\\u2121-\\u2122\\u2124\\u2126\\u2128\\u212A-\\u212D\\u212F-\\u2139\\u213C-\\u213F\\u2145-\\u2149\\u214E\\u2153-\\u2154\\u215B-\\u215E\\u2160-\\u2188\\u2190-\\u2199\\u21D2\\u21D4\\u2200\\u2202-\\u2203\\u2207-\\u2208\\u220B\\u220F\\u2211\\u2215\\u221A\\u221D-\\u2220\\u2223\\u2225\\u2227-\\u222C\\u222E\\u2234-\\u2237\\u223C-\\u223D\\u2248\\u224C\\u2252\\u2260-\\u2261\\u2264-\\u2267\\u226A-\\u226B\\u226E-\\u226F\\u2282-\\u2283\\u2286-\\u2287\\u2295\\u2299\\u22A5\\u22BF\\u2312\\u2460-\\u24B5\\u24D0-\\u24E9\\u2500-\\u254B\\u2550-\\u2574\\u2581-\\u258F\\u2592-\\u2595\\u25A0-\\u25A1\\u25A3-\\u25A9\\u25B2-\\u25B3\\u25B6-\\u25B7\\u25BC-\\u25BD\\u25C0-\\u25C1\\u25C6-\\u25C8\\u25CB\\u25CE-\\u25D1\\u25E2-\\u25E5\\u25EF\\u2605-\\u2606\\u2609\\u260E-\\u260F\\u261C\\u261E\\u2640\\u2642\\u2660-\\u2661\\u2663-\\u2665\\u2667-\\u266A\\u266C-\\u266D\\u266F\\u2C00-\\u2C2E\\u2C30-\\u2C5E\\u2C60-\\u2C6F\\u2C71-\\u2C7D\\u2C80-\\u2CE4\\u2D00-\\u2D25\\u2D30-\\u2D65\\u2D6F\\u2D80-\\u2D96\\u2DA0-\\u2DA6\\u2DA8-\\u2DAE\\u2DB0-\\u2DB6\\u2DB8-\\u2DBE\\u2DC0-\\u2DC6\\u2DC8-\\u2DCE\\u2DD0-\\u2DD6\\u2DD8-\\u2DDE\\u3000-\\u3003\\u3005-\\u3017\\u301D-\\u301F\\u3021-\\u3029\\u3031-\\u3035\\u3038-\\u303C\\u3041-\\u3096\\u309B-\\u309F\\u30A1-\\u30FF\\u3105-\\u312D\\u3131-\\u318E\\u31A0-\\u31B7\\u31F0-\\u321C\\u3220-\\u3229\\u3231-\\u3232\\u3239\\u3260-\\u327B\\u327F\\u32A3-\\u32A8\\u3303\\u330D\\u3314\\u3318\\u3322-\\u3323\\u3326-\\u3327\\u332B\\u3336\\u333B\\u3349-\\u334A\\u334D\\u3351\\u3357\\u337B-\\u337E\\u3380-\\u3384\\u3388-\\u33CA\\u33CD-\\u33D3\\u33D5-\\u33D6\\u33D8\\u33DB-\\u33DD\\u3400-\\u4DB5\\u4E00-\\u9FC3\\uA000-\\uA48C\\uA500-\\uA60C\\uA610-\\uA61F\\uA62A-\\uA62B\\uA640-\\uA65F\\uA662-\\uA66E\\uA680-\\uA697\\uA722-\\uA787\\uA78B-\\uA78C\\uA7FB-\\uA801\\uA803-\\uA805\\uA807-\\uA80A\\uA80C-\\uA822\\uA840-\\uA873\\uA882-\\uA8B3\\uA90A-\\uA925\\uA930-\\uA946\\uAA00-\\uAA28\\uAA40-\\uAA42\\uAA44-\\uAA4B\\uAC00-\\uD7A3\\uE000-\\uF848\\uF900-\\uFA2D\\uFA30-\\uFA6A\\uFA70-\\uFAD9\\uFB00-\\uFB06\\uFB13-\\uFB17\\uFB1D\\uFB1F-\\uFB28\\uFB2A-\\uFB36\\uFB38-\\uFB3C\\uFB3E\\uFB40-\\uFB41\\uFB43-\\uFB44\\uFB46-\\uFBB1\\uFBD3-\\uFD3D\\uFD50-\\uFD8F\\uFD92-\\uFDC7\\uFDF0-\\uFDFB\\uFE30-\\uFE31\\uFE33-\\uFE44\\uFE49-\\uFE52\\uFE54-\\uFE57\\uFE59-\\uFE66\\uFE68-\\uFE6B\\uFE70-\\uFE74\\uFE76-\\uFEFC\\uFF01-\\uFF5E\\uFF61-\\uFFBE\\uFFC2-\\uFFC7\\uFFCA-\\uFFCF\\uFFD2-\\uFFD7\\uFFDA-\\uFFDC\\uFFE0-\\uFFE6';

    // character ranges for the inner character of a name or identifier
    var NAME_INNER_CHARS = '\\d\\?A-Z_a-z\\xA1\\xA4\\xA7-\\xA8\\xAA\\xAF-\\xBA\\xBC-\\u034E\\u0350-\\u0377\\u037A-\\u037D\\u0384-\\u0386\\u0388-\\u038A\\u038C\\u038E-\\u03A1\\u03A3-\\u0523\\u0531-\\u0556\\u0559\\u0561-\\u0587\\u0591-\\u05BD\\u05BF\\u05C1-\\u05C2\\u05C4-\\u05C5\\u05C7\\u05D0-\\u05EA\\u05F0-\\u05F2\\u0600-\\u0603\\u0606-\\u0608\\u060B\\u060E-\\u061A\\u061F\\u0621-\\u063F\\u0641-\\u065E\\u0660-\\u0669\\u066E-\\u06D3\\u06D5-\\u06FF\\u070F-\\u074A\\u074D-\\u07B1\\u07C0-\\u07F6\\u07FA\\u0901-\\u0939\\u093C-\\u094D\\u0950-\\u0954\\u0958-\\u0963\\u0966-\\u096F\\u0971-\\u0972\\u097B-\\u097F\\u0981-\\u0983\\u0985-\\u098C\\u098F-\\u0990\\u0993-\\u09A8\\u09AA-\\u09B0\\u09B2\\u09B6-\\u09B9\\u09BC-\\u09C4\\u09C7-\\u09C8\\u09CB-\\u09CE\\u09D7\\u09DC-\\u09DD\\u09DF-\\u09E3\\u09E6-\\u09FA\\u0A01-\\u0A03\\u0A05-\\u0A0A\\u0A0F-\\u0A10\\u0A13-\\u0A28\\u0A2A-\\u0A30\\u0A32-\\u0A33\\u0A35-\\u0A36\\u0A38-\\u0A39\\u0A3C\\u0A3E-\\u0A42\\u0A47-\\u0A48\\u0A4B-\\u0A4D\\u0A51\\u0A59-\\u0A5C\\u0A5E\\u0A66-\\u0A75\\u0A81-\\u0A83\\u0A85-\\u0A8D\\u0A8F-\\u0A91\\u0A93-\\u0AA8\\u0AAA-\\u0AB0\\u0AB2-\\u0AB3\\u0AB5-\\u0AB9\\u0ABC-\\u0AC5\\u0AC7-\\u0AC9\\u0ACB-\\u0ACD\\u0AD0\\u0AE0-\\u0AE3\\u0AE6-\\u0AEF\\u0AF1\\u0B01-\\u0B03\\u0B05-\\u0B0C\\u0B0F-\\u0B10\\u0B13-\\u0B28\\u0B2A-\\u0B30\\u0B32-\\u0B33\\u0B35-\\u0B39\\u0B3C-\\u0B44\\u0B47-\\u0B48\\u0B4B-\\u0B4D\\u0B56-\\u0B57\\u0B5C-\\u0B5D\\u0B5F-\\u0B63\\u0B66-\\u0B71\\u0B82-\\u0B83\\u0B85-\\u0B8A\\u0B8E-\\u0B90\\u0B92-\\u0B95\\u0B99-\\u0B9A\\u0B9C\\u0B9E-\\u0B9F\\u0BA3-\\u0BA4\\u0BA8-\\u0BAA\\u0BAE-\\u0BB9\\u0BBE-\\u0BC2\\u0BC6-\\u0BC8\\u0BCA-\\u0BCD\\u0BD0\\u0BD7\\u0BE6-\\u0BFA\\u0C01-\\u0C03\\u0C05-\\u0C0C\\u0C0E-\\u0C10\\u0C12-\\u0C28\\u0C2A-\\u0C33\\u0C35-\\u0C39\\u0C3D-\\u0C44\\u0C46-\\u0C48\\u0C4A-\\u0C4D\\u0C55-\\u0C56\\u0C58-\\u0C59\\u0C60-\\u0C63\\u0C66-\\u0C6F\\u0C78-\\u0C7F\\u0C82-\\u0C83\\u0C85-\\u0C8C\\u0C8E-\\u0C90\\u0C92-\\u0CA8\\u0CAA-\\u0CB3\\u0CB5-\\u0CB9\\u0CBC-\\u0CC4\\u0CC6-\\u0CC8\\u0CCA-\\u0CCD\\u0CD5-\\u0CD6\\u0CDE\\u0CE0-\\u0CE3\\u0CE6-\\u0CEF\\u0CF1-\\u0CF2\\u0D02-\\u0D03\\u0D05-\\u0D0C\\u0D0E-\\u0D10\\u0D12-\\u0D28\\u0D2A-\\u0D39\\u0D3D-\\u0D44\\u0D46-\\u0D48\\u0D4A-\\u0D4D\\u0D57\\u0D60-\\u0D63\\u0D66-\\u0D75\\u0D79-\\u0D7F\\u0D82-\\u0D83\\u0D85-\\u0D96\\u0D9A-\\u0DB1\\u0DB3-\\u0DBB\\u0DBD\\u0DC0-\\u0DC6\\u0DCA\\u0DCF-\\u0DD4\\u0DD6\\u0DD8-\\u0DDF\\u0DF2-\\u0DF3\\u0E01-\\u0E3A\\u0E3F-\\u0E4E\\u0E50-\\u0E59\\u0E81-\\u0E82\\u0E84\\u0E87-\\u0E88\\u0E8A\\u0E8D\\u0E94-\\u0E97\\u0E99-\\u0E9F\\u0EA1-\\u0EA3\\u0EA5\\u0EA7\\u0EAA-\\u0EAB\\u0EAD-\\u0EB9\\u0EBB-\\u0EBD\\u0EC0-\\u0EC4\\u0EC6\\u0EC8-\\u0ECB\\u0ECD\\u0ED0-\\u0ED9\\u0EDC-\\u0EDD\\u0F00-\\u0F03\\u0F13-\\u0F39\\u0F3E-\\u0F47\\u0F49-\\u0F6C\\u0F71-\\u0F84\\u0F86-\\u0F8B\\u0F90-\\u0F97\\u0F99-\\u0FBC\\u0FBE-\\u0FCC\\u0FCE-\\u0FCF\\u1000-\\u1049\\u1050-\\u1099\\u109E-\\u10C5\\u10D0-\\u10FA\\u10FC\\u1100-\\u1159\\u115F-\\u11A2\\u11A8-\\u11F9\\u1200-\\u1248\\u124A-\\u124D\\u1250-\\u1256\\u1258\\u125A-\\u125D\\u1260-\\u1288\\u128A-\\u128D\\u1290-\\u12B0\\u12B2-\\u12B5\\u12B8-\\u12BE\\u12C0\\u12C2-\\u12C5\\u12C8-\\u12D6\\u12D8-\\u1310\\u1312-\\u1315\\u1318-\\u135A\\u135F-\\u1360\\u1369-\\u137C\\u1380-\\u1399\\u13A0-\\u13F4\\u1401-\\u166C\\u166F-\\u1676\\u1680-\\u169A\\u16A0-\\u16EA\\u16EE-\\u16F0\\u1700-\\u170C\\u170E-\\u1714\\u1720-\\u1734\\u1740-\\u1753\\u1760-\\u176C\\u176E-\\u1770\\u1772-\\u1773\\u1780-\\u17D3\\u17D7\\u17DB-\\u17DD\\u17E0-\\u17E9\\u17F0-\\u17F9\\u180E\\u1810-\\u1819\\u1820-\\u1877\\u1880-\\u18AA\\u1900-\\u191C\\u1920-\\u192B\\u1930-\\u193B\\u1940\\u1946-\\u196D\\u1970-\\u1974\\u1980-\\u19A9\\u19B0-\\u19C9\\u19D0-\\u19D9\\u19E0-\\u1A1B\\u1B00-\\u1B4B\\u1B50-\\u1B59\\u1B61-\\u1B7C\\u1B80-\\u1BAA\\u1BAE-\\u1BB9\\u1C00-\\u1C37\\u1C40-\\u1C49\\u1C4D-\\u1C7D\\u1D00-\\u1DE6\\u1DFE-\\u1F15\\u1F18-\\u1F1D\\u1F20-\\u1F45\\u1F48-\\u1F4D\\u1F50-\\u1F57\\u1F59\\u1F5B\\u1F5D\\u1F5F-\\u1F7D\\u1F80-\\u1FB4\\u1FB6-\\u1FC4\\u1FC6-\\u1FD3\\u1FD6-\\u1FDB\\u1FDD-\\u1FEF\\u1FF2-\\u1FF4\\u1FF6-\\u1FFE\\u2000-\\u200B\\u2010\\u2013-\\u2016\\u2018\\u201C-\\u201D\\u2020-\\u2021\\u2025-\\u2029\\u202F-\\u2030\\u2032-\\u2033\\u2035\\u203B\\u2044\\u2052\\u205F\\u2070-\\u2071\\u2074-\\u207C\\u207F-\\u208C\\u2090-\\u2094\\u20A0-\\u20B5\\u20D0-\\u20F0\\u2100-\\u214F\\u2153-\\u2188\\u2190-\\u2328\\u232B-\\u23E7\\u2400-\\u2426\\u2440-\\u244A\\u2460-\\u269D\\u26A0-\\u26BC\\u26C0-\\u26C3\\u2701-\\u2704\\u2706-\\u2709\\u270C-\\u2727\\u2729-\\u274B\\u274D\\u274F-\\u2752\\u2756\\u2758-\\u275E\\u2761-\\u2767\\u2776-\\u2794\\u2798-\\u27AF\\u27B1-\\u27BE\\u27C0-\\u27C4\\u27C7-\\u27CA\\u27CC\\u27D0-\\u27E5\\u27F0-\\u2982\\u2999-\\u29D7\\u29DC-\\u29FB\\u29FE-\\u2B4C\\u2B50-\\u2B54\\u2C00-\\u2C2E\\u2C30-\\u2C5E\\u2C60-\\u2C6F\\u2C71-\\u2C7D\\u2C80-\\u2CEA\\u2CFD\\u2D00-\\u2D25\\u2D30-\\u2D65\\u2D6F\\u2D80-\\u2D96\\u2DA0-\\u2DA6\\u2DA8-\\u2DAE\\u2DB0-\\u2DB6\\u2DB8-\\u2DBE\\u2DC0-\\u2DC6\\u2DC8-\\u2DCE\\u2DD0-\\u2DD6\\u2DD8-\\u2DDE\\u2DE0-\\u2DFF\\u2E2F\\u2E80-\\u2E99\\u2E9B-\\u2EF3\\u2F00-\\u2FD5\\u2FF0-\\u2FFB\\u3000-\\u3017\\u301D-\\u302F\\u3031-\\u303C\\u303E-\\u303F\\u3041-\\u3096\\u3099-\\u309F\\u30A1-\\u30FF\\u3105-\\u312D\\u3131-\\u318E\\u3192-\\u31B7\\u31C0-\\u31E3\\u31F0-\\u321E\\u3220-\\u3243\\u3250-\\u32FE\\u3300-\\u4DB5\\u4DC0-\\u9FC3\\uA000-\\uA48C\\uA490-\\uA4C6\\uA500-\\uA60C\\uA610-\\uA62B\\uA640-\\uA65F\\uA662-\\uA672\\uA67C-\\uA67D\\uA67F-\\uA697\\uA700-\\uA78C\\uA7FB-\\uA82B\\uA840-\\uA873\\uA880-\\uA8C4\\uA8D0-\\uA8D9\\uA900-\\uA92E\\uA930-\\uA953\\uAA00-\\uAA36\\uAA40-\\uAA4D\\uAA50-\\uAA59\\uAC00-\\uD7A3\\uD800-\\uD87F\\uDB80-\\uFA2D\\uFA30-\\uFA6A\\uFA70-\\uFAD9\\uFB00-\\uFB06\\uFB13-\\uFB17\\uFB1D-\\uFB36\\uFB38-\\uFB3C\\uFB3E\\uFB40-\\uFB41\\uFB43-\\uFB44\\uFB46-\\uFBB1\\uFBD3-\\uFD3D\\uFD50-\\uFD8F\\uFD92-\\uFDC7\\uFDF0-\\uFDFD\\uFE20-\\uFE26\\uFE30-\\uFE31\\uFE33-\\uFE44\\uFE49-\\uFE52\\uFE54-\\uFE57\\uFE59-\\uFE66\\uFE68-\\uFE6B\\uFE70-\\uFE74\\uFE76-\\uFEFC\\uFF01-\\uFF5E\\uFF61-\\uFFBE\\uFFC2-\\uFFC7\\uFFCA-\\uFFCF\\uFFD2-\\uFFD7\\uFFDA-\\uFFDC\\uFFE0-\\uFFE6\\uFFE8-\\uFFEE';

    // RE pattern for a simple column name in A1 notation.
    // - Group 1: The column name.
    var COL_PATTERN = '([a-z]+)';

    // RE pattern for a simple row name in A1 notation.
    // - Group 1: The row name.
    var ROW_PATTERN = '([0-9]+)';

    // RE pattern for a simple cell address in A1 notation.
    // - Group 1: The column name.
    // - Group 2: The row name.
    var CELL_PATTERN = COL_PATTERN + ROW_PATTERN;

    // RE pattern for a simple cell range address in A1 notation.
    // - Group 1: The name of the first column.
    // - Group 2: The name of the first row.
    // - Group 3: The name of the second column.
    // - Group 4: The name of the second row.
    var RANGE_PATTERN = CELL_PATTERN + ':' + CELL_PATTERN;

    // RE pattern for a simple column interval reference in A1 notation, e.g. C:C.
    // - Group 1: The name of the first column.
    // - Group 2: The name of the second column.
    var COLS_PATTERN = COL_PATTERN + ':' + COL_PATTERN;

    // RE pattern for a simple row interval reference in A1 notation, e.g. 2:2.
    // - Group 1: The name of the first row.
    // - Group 2: The name of the second row.
    var ROWS_PATTERN = ROW_PATTERN + ':' + ROW_PATTERN;

    // RE pattern for an absolute marker (an optional dollar sign) in A1 notation.
    // - Group 1: The absolute marker (empty string, or a dollar sign).
    var ABS_PATTERN = '(\\$)?';

    // RE pattern for a column reference in A1 notation (absolute or relative).
    // - Group 1: The absolute marker.
    // - Group 2: The column name.
    var COL_REF_PATTERN = ABS_PATTERN + COL_PATTERN;

    // RE pattern for a row reference in A1 notation (absolute or relative).
    // - Group 1: The absolute marker.
    // - Group 2: The row name.
    var ROW_REF_PATTERN = ABS_PATTERN + ROW_PATTERN;

    // RE pattern for a cell address in A1 notation (absolute or relative).
    // - Group 1: The absolute marker for the column.
    // - Group 2: The name of the column.
    // - Group 3: The absolute marker for the row.
    // - Group 4: The name of the row.
    var CELL_REF_PATTERN = COL_REF_PATTERN + ROW_REF_PATTERN;

    // RE pattern for a cell range address in A1 notation (absolute or relative).
    // - Group 1: The absolute marker for the first column.
    // - Group 2: The name of the first column.
    // - Group 3: The absolute marker for the first row.
    // - Group 4: The name of the first row.
    // - Group 5: The absolute marker for the second column.
    // - Group 6: The name of the second column.
    // - Group 7: The absolute marker for the second row.
    // - Group 8: The name of the second row.
    var RANGE_REF_PATTERN = CELL_REF_PATTERN + ':' + CELL_REF_PATTERN;

    // RE pattern for a column interval reference in A1 notation (absolute or relative), e.g. $C:$C.
    // - Group 1: The absolute marker for the first column.
    // - Group 2: The name of the first column.
    // - Group 3: The absolute marker for the second column.
    // - Group 4: The name of the second column.
    var COLS_REF_PATTERN = COL_REF_PATTERN + ':' + COL_REF_PATTERN;

    // RE pattern for a row interval reference in A1 notation (absolute or relative), e.g. $2:$2.
    // - Group 1: The absolute marker for the first row.
    // - Group 2: The name of the first row.
    // - Group 3: The absolute marker for the second row.
    // - Group 4: The name of the second row.
    var ROWS_REF_PATTERN = ROW_REF_PATTERN + ':' + ROW_REF_PATTERN;

    // private global functions ===============================================

    /**
     * Returns a regular expression that matches strings in R1C1 notation with
     * optional absolute row and column parts.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @returns {RegExp}
     *  A regular expression that matches absolute references in R1C1 notation.
     *  Either part may be missing completely (references to entire columns or
     *  rows), or the prefix character may be present without index. Matching
     *  strings are for example 'R2C3', 'R2C', 'RC3', 'RC', 'R2', 'C3', 'R',
     *  'C', or the empty string (!).
     */
    var getR1C1RegExp = _.memoize(function (prefixChars) {
        return new RegExp('^(?:' + prefixChars[0] + '(\\d*))?(?:' + prefixChars[1] + '(\\d*))?$', 'i');
    });

    /**
     * Returns whether the passed text is a simple R1C1 reference for the
     * passed prefix characters (either absolute column/row index not
     * enclosed in brackets, or missing column/row index).
     *
     * @param {SpreadsheetModel} docModel
     *  The document model needed to check the maximum column/row index of
     *  sheet names that look like cell addresses.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @param {String} text
     *  The string to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed text is a simple R1C1 reference for the passed
     *  prefix characters.
     */
    function isSimpleR1C1Reference(docModel, prefixChars, text) {

        // do not accept empty strings (would be matched by the regular expression)
        if (text.length === 0) { return false; }

        // check that text matches the regular expression
        var matches = getR1C1RegExp(prefixChars).exec(text);
        if (!matches) { return false; }

        // extract column and row index (row before column in R1C1 notation!)
        var row = matches[1] ? (parseInt(matches[1], 10) - 1) : 0,
            col = matches[2] ? (parseInt(matches[2], 10) - 1) : 0;

        // column and row index must be valid in the document
        return docModel.isValidAddress(new Address(col, row));
    }

    // class FormulaGrammar ===================================================

    /**
     * An instance of this class represents all settings of a specific formula
     * grammar and file format.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @property {String} GRAMMAR
     *  The identifier of the formula grammar, as passed to the constructor.
     *
     * @property {Boolean} UI
     *  Whether the formula grammar is used to handle formula expressions in
     *  the user interface.
     *
     * @property {String} REF_SYNTAX
     *  The identifier of the syntax of cell references:
     *  - 'ooxml' for the syntax used in native and localized OOXML grammars
     *      (references in A1 style, exclamation mark as sheet name separator,
     *      sheets are always absolute, apostrophes enclose entire range of
     *      sheet names, e.g.: ='Sheet1:Sheet 2'!A1:B2).
     *  - 'odfui' for the syntax used in the localized ODF grammar (references
     *      in A1 style, exclamation mark as sheet name separator, sheets can
     *      be absolute or relative, apostrophes enclose single sheet names in
     *      a range of sheets, e.g.: =Sheet1:$'Sheet 2'!A1:B2).
     *  - 'of' for the native OpenFormula syntax (entire reference enclosed in
     *      brackets, references in A1 style, row or column index can be #REF!
     *      errors, period as sheet name separator, sheets can be absolute or
     *      relative, apostrophes enclose single sheet names in a range of
     *      sheets, e.g.: =[Sheet1.A1:$'Sheet 2'.B#REF!]).
     *
     * @property {String} DEC
     *  The decimal separator character.
     *
     * @property {String} SEP
     *  The separator character in function parameter lists.
     *
     * @property {String} MAT_OPEN
     *  The leading character of a matrix literal.
     *
     * @property {String} MAT_ROW
     *  The separator character between rows in a matrix literal.
     *
     * @property {String} MAT_COL
     *  The separator character between row elements in a matrix literal.
     *
     * @property {String} MAT_CLOSE
     *  The trailing character of a matrix literal.
     *
     * @property {Object} RE
     *  A map with regular expressions needed to parse formula expressions.
     */
    function FormulaGrammar(fileFormat, grammarId) {

        // self reference
        var self = this;

        // special behavior depending on file format
        var odf = fileFormat === 'odf';

        // static configuration settings for the passed formula grammar
        var grammarConfig = GRAMMAR_CONFIG[grammarId];

        // the formula resource data according to file format and grammar
        var formulaResource = FormulaResource.create(fileFormat, grammarConfig.locale);

        // whether to used localized strings
        var localized = grammarConfig.localized;

        // the collection of boolean descriptors
        var booleanCollection = formulaResource.getBooleanCollection();

        // the collection of error code descriptors
        var errorCollection = formulaResource.getErrorCollection();

        // the collection of table region descriptors
        var regionCollection = formulaResource.getRegionCollection();

        // the collection of operator descriptors
        var operatorCollection = formulaResource.getOperatorCollection();

        // the collection of function descriptors
        var functionCollection = formulaResource.getFunctionCollection();

        // specific separator/delimiter characters used in regular expressions
        var DEC = formulaResource.getDec(localized);
        var GROUP = formulaResource.getGroup(localized);
        var SEP = formulaResource.getSeparator(localized);
        var TABLE_SEP = SEP;

        // a simple number formatter to generate the string representation of numbers
        var formatter = new Formatter({ dec: DEC, group: GROUP });

        // prefix characters for R1C1 notation
        var PREFIX_CHARS = formulaResource.getR1C1PrefixChars(localized);

        // specific error codes used in regular expressions, or when generating text representations
        var REF_ERROR = errorCollection.getName(ErrorCode.REF.key, localized);
        var NA_ERROR = errorCollection.getName(ErrorCode.NA.key, localized);

        // RE character class for the leading character of a simple sheet name (without apostrophes).
        var SHEET_LEADING_CHAR_CLASS = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]';

        // RE character class for other characters of a simple sheet name (without apostrophes).
        var SHEET_SIMPLE_CHAR_CLASS = '[' + (odf ? '' : '.') + '\\w\\xa1-\\u2027\\u202a-\\uffff]';

        // RE character class for any character of a complex sheet name (enclosed in apostrophes).
        var SHEET_COMPLEX_CHAR_CLASS = '[^\\x00-\\x1f\\x80-\\x9f\\[\\]\'*?:/\\\\]';

        // RE character class for a characters of a simple external file name (without apostrophes).
        var FILE_SIMPLE_CHAR_CLASS = '[\\w.\\\\\\xa1-\\u2027\\u202a-\\uffff]';

        // RE character class for the leading character of a name or identifier (OOXML: allow leading backslash).
        var NAME_LEADING_CHAR_CLASS = '[' + (odf ? '' : '\\\\') + NAME_LEADING_CHARS + ']';

        // RE character class for other characters of a name or identifier (OOXML: allow dots and backslash).
        var NAME_INNER_CHAR_CLASS = '[' + (odf ? '' : '.\\\\') + NAME_INNER_CHARS + ']';

        // RE character class for other characters of a function name (ODF: allow periods in difference to defined names).
        var FUNC_INNER_CHAR_CLASS = '[' + (odf ? '.' : '.\\\\') + NAME_INNER_CHARS + ']';

        // RE pattern for a simple sheet name.
        // - Group 1: The sheet name.
        var SHEET_NAME_SIMPLE_PATTERN = '(' + SHEET_LEADING_CHAR_CLASS + SHEET_SIMPLE_CHAR_CLASS + '*)';

        // RE pattern for a complex sheet name (with embedded double-apostrophes, without surrounding apostrophes).
        // - Group 1: The sheet name.
        var SHEET_NAME_COMPLEX_PATTERN = '(' + SHEET_COMPLEX_CHAR_CLASS + '+(?:(?:\'\')+' + SHEET_COMPLEX_CHAR_CLASS + '+)*)';

        // RE pattern for a simple external file name (enclosed in brackets, without apostrophes).
        // - Group 1: The name of the external file (the part inside the brackets).
        var FILE_NAME_SIMPLE_PATTERN = '(?:\\[(' + FILE_SIMPLE_CHAR_CLASS + '+)\\])';

        // RE pattern for a #REF! error code
        // - Group 1: The matched #REF! error code.
        var REF_ERROR_PATTERN = '(' + _.escapeRegExp(REF_ERROR) + ')';

        // RE look-ahead pattern to exclude an opening parenthesis and an exclamation mark
        // after another pattern. All valid inner characters of names must be excluded too,
        // otherwise the previous groups would match less characters than they should.
        var TERMINATE_REF_PATTERN = '(?!\\(|!|' + NAME_INNER_CHAR_CLASS + ')';

        // RE pattern for a valid single sheet name (without address separator), e.g. Sheet1 in the formula =Sheet1!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name).
        // Either group 1 or group 2 will match the sheet name, the other group will be empty.
        var ABS_SHEET_NAME_PATTERN = '(?:' + SHEET_NAME_SIMPLE_PATTERN + '|\'' + SHEET_NAME_COMPLEX_PATTERN + '\')';

        // RE pattern for a single (implicitly absolute) sheet name (without address separator), e.g. Sheet1 in the formula
        // =Sheet1!A1, or a broken sheet reference after deleting the referenced sheet, e.g. #REF! in the formula =#REF!!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name).
        // - Group 3: The #REF! error code as sheet name.
        // Either group 1 or group 2 will match the sheet name, or group 3 will contain the error code (the other groups will be empty).
        var ABS_SHEET_NAME_OR_REF_PATTERN = '(?:' + ABS_SHEET_NAME_PATTERN + '|' + REF_ERROR_PATTERN + ')';

        // RE pattern for a range of (implicitly absolute) sheet names (without address separator), e.g. Sheet1:Sheet2 in the
        // formula =SUM(Sheet1:Sheet2!A1).
        // - Group 1: The name of the first sheet (simple sheet names).
        // - Group 2: The name of the second sheet (simple sheet names).
        // - Group 3: The name of the first sheet (complex sheet names).
        // - Group 4: The name of the second sheet (complex sheet names).
        // Either group 1 or group 3 will match the first sheet name (the other group will be empty).
        // Either group 2 or group 4 will match the second sheet name (the other group will be empty).
        var ABS_SHEET_RANGE_PATTERN = '(?:' + SHEET_NAME_SIMPLE_PATTERN + ':' + SHEET_NAME_SIMPLE_PATTERN + '|\'' + SHEET_NAME_COMPLEX_PATTERN + ':' + SHEET_NAME_COMPLEX_PATTERN + '\')';

        // RE pattern for a valid single sheet name (with optional absolute marker, without address separator),
        // e.g. $Sheet1 in the formula =$Sheet1!A1.
        // - Group 1: The absolute marker.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // Either group 2 or group 3 will match the sheet name, the other group will be empty.
        var REL_SHEET_NAME_PATTERN = ABS_PATTERN + ABS_SHEET_NAME_PATTERN;

        // RE pattern for a single sheet name (with optional absolute marker, without address separator), e.g. $Sheet1 in the formula
        // =$Sheet1!A1, or a broken sheet reference after deleting the referenced sheet, e.g. $#REF! in the formula =$#REF!!A1.
        // - Group 1: The absolute marker.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // - Group 4: The #REF! error code as sheet name.
        // Either group 2 or group 3 will match the sheet name, or group 4 will contain the error code (the other groups will be empty).
        var REL_SHEET_NAME_OR_REF_PATTERN = ABS_PATTERN + ABS_SHEET_NAME_OR_REF_PATTERN;

        // RE pattern for a range of (implicitly absolute) sheet names (with optional absolute markers, without address separator), e.g.
        // Sheet1:$Sheet2 in the formula =SUM(Sheet1:$Sheet2!A1).
        // - Group 1: The absolute marker of the first sheet.
        // - Group 2: The name of the first sheet (simple sheet name).
        // - Group 3: The name of the first sheet (complex sheet name).
        // - Group 4: The absolute marker of the second sheet.
        // - Group 5: The name of the second sheet (simple sheet name).
        // - Group 6: The name of the second sheet (complex sheet name).
        // Either group 2 or group 3 will match the first sheet name (the other group will be empty).
        // Either group 5 or group 6 will match the second sheet name (the other group will be empty).
        var REL_SHEET_RANGE_PATTERN = REL_SHEET_NAME_PATTERN + ':' + REL_SHEET_NAME_PATTERN;

        // RE pattern for a complete name or identifier.
        // - Group 1: The entire name.
        var NAME_PATTERN = '(' + NAME_LEADING_CHAR_CLASS + NAME_INNER_CHAR_CLASS + '*)';

        // RE pattern for a complete function name.
        // - Group 1: The entire function name.
        var FUNC_PATTERN = '(' + NAME_LEADING_CHAR_CLASS + FUNC_INNER_CHAR_CLASS + '*)';

        // Character ranges (without brackets) for characters that need to be escaped in table column names.
        var TABLE_COL_ESCAPE_CHARS = '\\[\\]\'#' + (localized ? '@' : '');

        // Character ranges (without brackets) for characters of complex table column names. The comma
        // will be a complex table character, if it is also used as separator in the current grammar.
        var TABLE_COL_COMPLEX_CHARS = '\\x00-\\x2B\\x2D-\\x2F\\x3A-\\x40\\x5B-\\x60\\x7B\\x7D\\x7E' + ((SEP === ',') ? ',' : '');

        // RE pattern for a simple column name in a structured table reference.
        // - Group 1: The column name.
        var TABLE_SIMPLE_COL_NAME_PATTERN = '([^' + TABLE_COL_COMPLEX_CHARS + ']+)';

        // RE pattern for a complex column name in a structured table reference.
        // - Group 1: The column name.
        var TABLE_COL_NAME_PATTERN = '((?:[^' + TABLE_COL_ESCAPE_CHARS + ']|\'.)+)';

        // RE pattern for a simple column name without brackets, or any column name with brackets in a
        // structured table reference, e.g. in the formula =Table1[@Col1:[Col 2]]).
        // - Group 1: The simple column name found without brackets.
        // - Group 2: The complex column name found inside brackets, without the brackets.
        var TABLE_MIXED_COL_NAME_PATTERN = '(?:' + TABLE_SIMPLE_COL_NAME_PATTERN + '|\\[' + TABLE_COL_NAME_PATTERN + '\\])';

        // RE pattern for optional whitespace characters in a structured table reference
        // - Group 1: The whitespace characters.
        var TABLE_OPTIONAL_WS_PATTERN = '([' + WHITESPACE_CHARS + ']*)';

        // RE pattern for whitespace characters following a token in a structured table reference. Will only be recognized
        // if specific control characters that are valid inside a tbale reference will follow.
        // - Group 1: The whitespace characters.
        var TABLE_TRAILING_WS_PATTERN = '(?:([' + WHITESPACE_CHARS + ']+)(?=\\[|\\]|' + _.escapeRegExp(TABLE_SEP) + (localized ? '|@' : '') + '|$))?';

        // RE pattern for a column name (absolute or relative) in OpenFormula syntax (#REF! error allowed).
        // - Group 1: The absolute marker.
        // - Group 2: The column name, or the #REF! error code.
        var OF_COL_PATTERN = ABS_PATTERN + '([a-z]+|' + _.escapeRegExp(REF_ERROR) + ')';

        // RE pattern for a row name (absolute or relative) in OpenFormula syntax (#REF! error allowed).
        // - Group 1: The absolute marker.
        // - Group 2: The row name, or the #REF! error code.
        var OF_ROW_PATTERN = ABS_PATTERN + '([0-9]+|' + _.escapeRegExp(REF_ERROR) + ')';

        // RE pattern for a cell address in OpenFormula syntax (with leading period, #REF! errors for column or row allowed).
        // - Group 1: The absolute marker for the column.
        // - Group 2: The name of the column, or the #REF! error code.
        // - Group 3: The absolute marker for the row.
        // - Group 4: The name of the row, or the #REF! error code.
        var OF_CELL_PATTERN = '\\.' + OF_COL_PATTERN + OF_ROW_PATTERN;

        // RE pattern for a cell address with sheet name in OpenFormula syntax (#REF! errors for column or row allowed).
        // - Group 1: The absolute marker for the sheet.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // - Group 4: The #REF! error code as sheet name.
        // - Group 5: The absolute marker for the column.
        // - Group 6: The name of the column, or the #REF! error code.
        // - Group 7: The absolute marker for the row.
        // - Group 8: The name of the row, or the #REF! error code.
        var OF_CELL_3D_PATTERN = REL_SHEET_NAME_OR_REF_PATTERN + OF_CELL_PATTERN;

        // Matches a simple sheet name, or a range of simple sheet names, that do not need to be
        // enclosed in apostrophes. Supports an optional leading reference to an external file
        // (simple name), enclosed in brackets.
        // Examples: 'Sheet1', 'Sheet1:Sheet2', '[file1.xlsx]Sheet1', '[path\to\file1.xlsx]Sheet1:Sheet2'.
        // - Group 1: The name of the external file (the part inside the brackets), may be undefined.
        // - Group 2: The first sheet name.
        // - Group 3: The second sheet name, may be undefined.
        var SIMPLE_SHEET_NAME_RE = new RegExp('^' + FILE_NAME_SIMPLE_PATTERN + '?' + SHEET_NAME_SIMPLE_PATTERN + '(?::' + SHEET_NAME_SIMPLE_PATTERN + ')?$', 'i');

        // Matches a valid label for a defined name.
        var VALID_NAME_RE = new RegExp('^' + NAME_PATTERN + '$', 'i');

        // Matches all characters in table column names that need to be escaped with an apostrophe.
        var TABLE_COL_ESCAPE_RE = new RegExp('[' + TABLE_COL_ESCAPE_CHARS + ']', 'g');

        // Matches a simple table column name that may appear without brackets in a complex structured table reference.
        var SIMPLE_TABLE_COLUMN_NAME_RE = new RegExp('^[^' + TABLE_COL_COMPLEX_CHARS + ']+$', 'i');

        // public properties --------------------------------------------------

        // the grammar identifier
        this.GRAMMAR = grammarId;

        // the user interface mode
        this.UI = localized;

        // the reference syntax identifier
        this.REF_SYNTAX = !odf ? 'ooxml' : localized ? 'odfui' : 'of';

        // separator characters
        this.DEC = DEC;
        this.SEP = SEP;
        this.MAT_OPEN = '{';
        this.MAT_ROW = (odf || localized) ? '|' : ';';
        this.MAT_COL = (odf || localized) ? ';' : ',';
        this.MAT_CLOSE = '}';
        this.TABLE_SEP = TABLE_SEP;

        // map of all regular expressions needed to parse formula expressions
        this.RE = {

            // Any whitespace or other non-printable characters.
            WHITESPACE: new RegExp('^[' + WHITESPACE_CHARS + ']+'),

            // String literals: Enclosed in double quotes, embedded double quotes
            // are represented by a sequence of two double quote characters.
            STRING_LIT: /^(".*?")+/,

            // Boolean literals: Must not be followed by any character valid inside an identifier.
            // This prevents to match 'trueA' or 'false_1' or similar name tokens, as well as TRUE()
            // and FALSE() function calls, or references with sheet names such as 'TRUE!A1'.
            BOOLEAN_LIT: new RegExp('^' + booleanCollection.getAllPattern(localized) + TERMINATE_REF_PATTERN, 'i'),

            // Error code literals (only supported error codes, not any string
            // starting with a hash character).
            ERROR_LIT: new RegExp('^' + errorCollection.getAllPattern(localized) + '(?!' + NAME_INNER_CHAR_CLASS + ')', 'i'),

            // Number literals as fraction with integral part, e.g. '1 2/3'.
            // - Group 1: The leading integer (always non-negative).
            // - Group 2: The numerator (always non-negative).
            // - Group 3: The denominator (always positive).
            FRACTION_LIT: /^(\d+) (\d+)\/(0*[1-9]\d*)/,

            // Opening parenthesis of a matrix literal.
            MATRIX_OPEN: new RegExp('^' + _.escapeRegExp(this.MAT_OPEN)),

            // The column separator in a matrix literal.
            MATRIX_COL_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_COL)),

            // The row separator in a matrix literal.
            MATRIX_ROW_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_ROW)),

            // Closing parenthesis of a matrix literal.
            MATRIX_CLOSE: new RegExp('^' + _.escapeRegExp(this.MAT_CLOSE)),

            // The list/parameter separator character.
            SEPARATOR: new RegExp('^' + _.escapeRegExp(SEP)),

            // All unary and binary operators.
            // - Group 1: The name of the matched operator.
            OPERATOR: new RegExp('^' + operatorCollection.getAllPattern(localized)),

            // The opening parenthesis.
            OPEN: /^\(/,

            // The closing parenthesis.
            CLOSE: /^\)/,

            // A local cell reference, e.g. A1 or $A$1, but without trailing parenthesis,
            // e.g. F2(), and without trailing sheet separator, e.g. F2!A1.
            // - Group 1: The absolute marker for the column.
            // - Group 2: The column name.
            // - Group 3: The absolute marker for the row.
            // - Group 4: The row name.
            CELL_REF: new RegExp('^' + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference with implicitly absolute sheet, e.g. Sheet1!$A$1, but without trailing parenthesis,
            // e.g. Module1!F2() which is a call of the macro function F2() contained in Module1.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the column.
            // - Group 5: The column name.
            // - Group 6: The absolute marker for the row.
            // - Group 7: The row name.
            CELL_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference with relative/absolute sheet, e.g. $Sheet1!$A$1, but without trailing parenthesis,
            // e.g. Module1!F2() which is a call of the macro function F2() contained in Module1.
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            CELL_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_OR_REF_PATTERN + '!' + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            CELL_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$A$1.
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the column.
            // - Group 8: The column name.
            // - Group 9: The absolute marker for the row.
            // - Group 10: The row name.
            CELL_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local range reference, e.g. A1:B2 or $A$1:$B$2, but without trailing parenthesis, e.g. A1:F2(),
            // which is the cell address A1, and the result of the function call F2(), connected with a range operator.
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            RANGE_REF: new RegExp('^' + RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference with implicitly absolute sheet, e.g. Sheet1!$A$1:$B$2, but without trailing parenthesis,
            // e.g. Sheet1!A1:F2().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the first row.
            // - Group 7: The name of the first row.
            // - Group 8: The absolute marker for the second column.
            // - Group 9: The name of the second column.
            // - Group 10: The absolute marker for the second row.
            // - Group 11: The name of the second row.
            RANGE_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference with relative/absolute sheet, e.g. $Sheet1!$A$1:$B$2, but without trailing parenthesis, e.g. Sheet1!A1:F2().
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            RANGE_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_OR_REF_PATTERN + '!' + RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$A$1:$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            RANGE_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$A$1:$A$1.
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the first column.
            // - Group 8: The name of the first column.
            // - Group 9: The absolute marker for the first row.
            // - Group 10: The name of the first row.
            // - Group 11: The absolute marker for the second column.
            // - Group 12: The name of the second column.
            // - Group 13: The absolute marker for the second row.
            // - Group 14: The name of the second row.
            RANGE_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local column reference, e.g. A:B or $C:$C (always with colon).
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the second column.
            // - Group 4: The name of the second column.
            COLS_REF: new RegExp('^' + COLS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column reference with implicitly absolute sheet, e.g. Sheet1!$C:$C (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the second column.
            // - Group 7: The name of the second column.
            COLS_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + COLS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column reference with relative/absolute sheet, e.g. $Sheet1!$C:$C (always with colon).
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            COLS_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_OR_REF_PATTERN + '!' + COLS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column reference in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$C:$C (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            COLS_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + COLS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column reference in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$C:$C (always with colon).
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the first column.
            // - Group 8: The name of the first column.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            COLS_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + COLS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local row reference, e.g. 1:2 or $3:$3 (always with colon).
            // - Group 1: The absolute marker for the first row.
            // - Group 2: The name of the first row.
            // - Group 3: The absolute marker for the second row.
            // - Group 4: The name of the second row.
            ROWS_REF: new RegExp('^' + ROWS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row reference with implicitly absolute sheet, e.g. Sheet1!$3:$3 (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the first row.
            // - Group 5: The name of the first row.
            // - Group 6: The absolute marker for the second row.
            // - Group 7: The name of the second row.
            ROWS_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + ROWS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row reference with relative/absolute sheet, e.g. $Sheet1!$3:$3 (always with colon).
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            ROWS_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_OR_REF_PATTERN + '!' + ROWS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row reference in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            ROWS_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + ROWS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row reference in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$3:$3 (always with colon).
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second row.
            // - Group 10: The name of the second row.
            ROWS_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + ROWS_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A reference error with implicitly absolute sheet reference, e.g. Sheet1!#REF!.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The matched #REF! error code following the sheet name.
            ERROR_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A reference error with relative/absolute sheet reference, e.g. $Sheet1!#REF!.
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The matched #REF! error code following the sheet name.
            ERROR_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_OR_REF_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A reference error in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!#REF!.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The matched #REF! error code following the sheet name.
            ERROR_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A reference error in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!#REF!.
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The matched #REF! error code following the sheet name.
            ERROR_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A local function name (names followed by an opening parenthesis), e.g. SUM().
            // - Group 1: The name of the function.
            FUNCTION_REF: new RegExp('^' + FUNC_PATTERN + '(?=\\()', 'i'),

            // A function name (names followed by an opening parenthesis) with sheet, e.g. Module1!macro().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The name of the function.
            FUNCTION_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + FUNC_PATTERN + '(?=\\()', 'i'),

            // Defined names without sheet reference, e.g. my_name.
            // - Group 1: The label of the defined name.
            NAME_REF: new RegExp('^' + NAME_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // Defined names with sheet reference, e.g. Sheet1!my_name.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The label of the defined name.
            NAME_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_OR_REF_PATTERN + '!' + NAME_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // The opening bracket of a structured table reference.
            // Group 1: The whitespace characters following the opening bracket.
            TABLE_OPEN: new RegExp('^\\[' + TABLE_TRAILING_WS_PATTERN),

            // The closing bracket of a structured table reference.
            // Group 1: The whitespace characters preceding the closing bracket.
            // Group 2: The closing bracket, or an empty string if the bracket is missing at the end of the formula.
            TABLE_CLOSE: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + '(\\]|$)'),

            // Any whitespace or other non-printable characters inside a table reference.
            // Group 1: The whitespace characters preceding the separator.
            // Group 2: The whitespace characters following the separator.
            TABLE_SEPARATOR: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + _.escapeRegExp(TABLE_SEP) + TABLE_TRAILING_WS_PATTERN),

            // A single predefined table region in structured table references (without brackets), followed
            // by a closing bracket, e.g. =Table[#All].
            // - Group 1: The name of the table region, with heading hash character.
            TABLE_SINGLE_REGION_NAME: new RegExp('^' + regionCollection.getAllPattern(localized) + '(?=\\])', 'i'),

            // A single column name (also complex names) in structured table references (without brackets),
            // followed by a closing bracket, e.g. =Table[Col'#2].
            // - Group 1: The column name.
            TABLE_SINGLE_COLUMN_NAME: new RegExp('^' + TABLE_COL_NAME_PATTERN + '(?=\\])', 'i'),

            // A simple column name (no special characters) in structured table references that can be used
            // without brackets in a complex table reference with separators and column ranges.
            // - Group 1: The column name.
            TABLE_SIMPLE_COLUMN_NAME: new RegExp('^' + TABLE_SIMPLE_COL_NAME_PATTERN, 'i'),

            // A predefined table region in structured table references, enclosed in brackets.
            // - Group 1: The name of the table region, without brackets, with heading hash character.
            TABLE_REGION_REF: new RegExp('^\\[' + regionCollection.getAllPattern(localized) + '\\]', 'i'),

            // A column reference in structured table references, enclosed in brackets.
            // - Group 1: The column name, without brackets.
            TABLE_COLUMN_REF: new RegExp('^\\[' + TABLE_COL_NAME_PATTERN + '\\]', 'i'),

            // A column reference in a structured table reference. The column name may occur without brackets
            // (simple column names only), or inside brackets (any column name). Example: the table reference
            // Table1[@Col1] contains the simple column name 'Col1', but the table reference Table1[@[Col 2]]
            // contains the complex column name 'Col 2' in brackets.
            // - Group 1: Leading whitespace characters before the column name.
            // - Group 2: The simple column name found without brackets (undefined for complex).
            // - Group 3: The complex column name without the brackets (undefined for simple).
            TABLE_MIXED_COLUMN_REF: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + TABLE_MIXED_COL_NAME_PATTERN, 'i'),

            // A column range in structured table references, each column name enclosed in brackets.
            // - Group 1: The first column name, without brackets.
            // - Group 2: The second column name, without brackets.
            TABLE_COLUMN_RANGE_REF: new RegExp('^\\[' + TABLE_COL_NAME_PATTERN + '\\]:\\[' + TABLE_COL_NAME_PATTERN + '\\]', 'i'),

            // A column range in a structured table reference. Each column name may occur without brackets
            // (simple column names only), or inside brackets (any column name). Example: the table reference
            // Table1[@Col1:[Col 2]] contains a column range with the simple column name 'Col1', and the
            // complex column name 'Col 2' in brackets.
            // - Group 1: Leading whitespace characters before the column range.
            // - Group 2: The simple name of the first column found without brackets (undefined for complex).
            // - Group 3: The complex name of the first column without the brackets (undefined for simple).
            // - Group 4: The simple name of the second column found without brackets (undefined for complex).
            // - Group 5: The complex name of the second column without the brackets (undefined for simple).
            TABLE_MIXED_COLUMN_RANGE_REF: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + TABLE_MIXED_COL_NAME_PATTERN + ':' + TABLE_MIXED_COL_NAME_PATTERN, 'i'),

            // An at-sign, used in UI grammars as shortcut for [#This Row] in structured table references.
            TABLE_THISROW_REF: /^@/,

            // A local cell reference in OpenFormula syntax, e.g. [.A1], [.$A$1], or [.A$#REF!].
            // - Group 1: The absolute marker for the column.
            // - Group 2: The name of the column, or the #REF! error code.
            // - Group 3: The absolute marker for the row.
            // - Group 4: The name of the row, or the #REF! error code.
            OF_CELL_REF: new RegExp('^\\[' + OF_CELL_PATTERN + '\\]', 'i'),

            // A cell reference with sheet in OpenFormula syntax, e.g. [Sheet1.A1], [$Sheet1.$A$1], or [#REF!.A$#REF!].
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The name of the sheet (simple sheet name).
            // - Group 3: The name of the sheet (complex sheet name).
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The name of the column, or the #REF! error code.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The name of the row, or the #REF! error code.
            OF_CELL_3D_REF: new RegExp('^\\[' + OF_CELL_3D_PATTERN + '\\]', 'i'),

            // A local cell range reference in OpenFormula syntax, e.g. [.A1:.$A$1], or [.A#REF!:.B2].
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column, or the #REF! error code.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row, or the #REF! error code.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column, or the #REF! error code.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row, or the #REF! error code.
            OF_RANGE_REF: new RegExp('^\\[' + OF_CELL_PATTERN + ':' + OF_CELL_PATTERN + '\\]', 'i'),

            // A cell range reference with sheet in OpenFormula syntax, e.g. [Sheet1.A1:.$A$1], or [$Sheet2.A#REF!:.B2].
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The name of the sheet (simple sheet name).
            // - Group 3: The name of the sheet (complex sheet name).
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column, or the #REF! error code.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row, or the #REF! error code.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column, or the #REF! error code.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row, or the #REF! error code.
            OF_RANGE_3D_REF: new RegExp('^\\[' + OF_CELL_3D_PATTERN + ':' + OF_CELL_PATTERN + '\\]', 'i'),

            // A cell range reference in multiple sheets in OpenFormula syntax, e.g. [Sheet1.A1:$Sheet2.$A$1],
            // or [$Sheet2.A#REF!:#REF!.#REF!B2].
            // - Group 1: The absolute marker for the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The #REF! error code as first sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column, or the #REF! error code.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row, or the #REF! error code.
            // - Group 9: The absolute marker for the second sheet.
            // - Group 10: The name of the second sheet (simple sheet name).
            // - Group 11: The name of the second sheet (complex sheet name).
            // - Group 12: The #REF! error code as second sheet name.
            // - Group 13: The absolute marker for the second column.
            // - Group 14: The name of the second column, or the #REF! error code.
            // - Group 15: The absolute marker for the second row.
            // - Group 16: The name of the second row, or the #REF! error code.
            OF_RANGE_CUBE_REF: new RegExp('^\\[' + OF_CELL_3D_PATTERN + ':' + OF_CELL_3D_PATTERN + '\\]', 'i'),

            // A local function name (names followed by an opening parenthesis with optional whitespace), e.g. SUM (),
            // as allowed in the native OpenFormula syntax.
            // - Group 1: The name of the function.
            OF_FUNCTION_REF: new RegExp('^' + FUNC_PATTERN + '(?:[' + WHITESPACE_CHARS + ']*)(?=\\()', 'i')
        };

        // map of all regular expressions needed to parse range lists
        this.RANGE_LIST_RE = {

            // A sequence of white-space characters as separator between two ranges.
            SEP_WS: /^\s+/,

            // A single separator character other than white-space between two ranges.
            SEP_OTHER: /^[,;]/,

            // A cell address, e.g. A1.
            // - Group 1: The column name.
            // - Group 2: The row name.
            CELL: new RegExp('^' + CELL_PATTERN, 'i'),

            // A cell address with sheet, e.g. Sheet1!A1.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The column name.
            // - Group 4: The row name.
            CELL_3D: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + CELL_PATTERN, 'i'),

            // A cell address in multiple sheets, e.g. Sheet1:Sheet2!A1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The column name.
            // - Group 6: The row name.
            CELL_CUBE: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + CELL_PATTERN, 'i'),

            // A range address, e.g. A1:B2.
            // - Group 1: The name of the first column.
            // - Group 2: The name of the first row.
            // - Group 3: The name of the second column.
            // - Group 4: The name of the second row.
            RANGE: new RegExp('^' + RANGE_PATTERN, 'i'),

            // A range address with sheet, e.g. Sheet1!A1:B2.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The name of the first column.
            // - Group 4: The name of the first row.
            // - Group 5: The name of the second column.
            // - Group 6: The name of the second row.
            RANGE_3D: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + RANGE_PATTERN, 'i'),

            // A range address in multiple sheets, e.g. Sheet1:Sheet2!A1:B2.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The name of the first column.
            // - Group 6: The name of the first row.
            // - Group 7: The name of the second column.
            // - Group 8: The name of the second row.
            RANGE_CUBE: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + RANGE_PATTERN, 'i'),

            // A column interval, e.g. A:B (always with colon).
            // - Group 1: The name of the first column.
            // - Group 2: The name of the second column.
            COLS: new RegExp('^' + COLS_PATTERN, 'i'),

            // A column interval with sheet, e.g. Sheet1!C:C (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The name of the first column.
            // - Group 4: The name of the second column.
            COLS_3D: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + COLS_PATTERN, 'i'),

            // A column interval in multiple sheets, e.g. Sheet1:Sheet2!C:C (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The name of the first column.
            // - Group 6: The name of the second column.
            COLS_CUBE: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + COLS_PATTERN, 'i'),

            // A row interval, e.g. 1:2 (always with colon).
            // - Group 1: The name of the first row.
            // - Group 2: The name of the second row.
            ROWS: new RegExp('^' + ROWS_PATTERN, 'i'),

            // A row interval with sheet, e.g. Sheet1!3:3 (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The name of the first row.
            // - Group 4: The name of the second row.
            ROWS_3D: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + ROWS_PATTERN, 'i'),

            // A row interval in multiple sheets, e.g. Sheet1:Sheet2!3:3 (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The name of the first row.
            // - Group 6: The name of the second row.
            ROWS_CUBE: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + ROWS_PATTERN, 'i'),

            // A cell address with sheet in OpenFormula syntax, without brackets, e.g. Sheet1.A1.
            // - Group 1: The name of the sheet (simple sheet name).
            // - Group 2: The name of the sheet (complex sheet name).
            // - Group 3: The name of the column.
            // - Group 4: The name of the row.
            OF_CELL_3D: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '\\.' + CELL_PATTERN, 'i'),

            // A range address with sheet in OpenFormula syntax, with optional leading period,
            // but without brackets, e.g. Sheet1.A1:A1.
            // - Group 1: The name of the sheet (simple sheet name).
            // - Group 2: The name of the sheet (complex sheet name).
            // - Group 3: The name of the first column.
            // - Group 4: The name of the first row.
            // - Group 5: The name of the second column.
            // - Group 6: The name of the second row.
            OF_RANGE_3D: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '\\.' + RANGE_PATTERN, 'i'),

            // A range address in multiple sheets in OpenFormula syntax, without brackets, e.g.
            // Sheet1.A1:Sheet2.A1.
            // - Group 1: The name of the first sheet (simple sheet name).
            // - Group 2: The name of the first sheet (complex sheet name).
            // - Group 3: The name of the first column.
            // - Group 4: The name of the first row.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The name of the second column.
            // - Group 8: The name of the second row.
            OF_RANGE_CUBE: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '\\.' + CELL_PATTERN + ':' + ABS_SHEET_NAME_PATTERN + '\\.' + CELL_PATTERN, 'i')
        };

        FormulaUtils.info('FormulaGrammar(): grammar=' + grammarId + ' config=', this);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed text is a boolean literal according to
         * this formula grammar.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is a boolean literal.
         */
        function isBooleanLiteral(text) {
            return _.isString(booleanCollection.getKey(text, localized));
        }

        /**
         * Returns whether the passed text is the representation of a relative
         * cell reference in A1 notation, or a cell reference in R1C1 notation
         * (either English, e.g. 'R1C1', or according to this formula grammar,
         * e.g. 'Z1S1' in the German UI grammar).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is the representation of a  cell reference
         *  in A1 or R1C1 notation.
         */
        function isReferenceSymbol(docModel, text) {

            // cell address in A1 notation
            var address = Address.parse(text);
            if (address && docModel.isValidAddress(address)) { return true; }

            // cell address in native R1C1 notation (regardless of the grammar)
            if (isSimpleR1C1Reference(docModel, formulaResource.getR1C1PrefixChars(false), text)) { return true; }

            // localized R1C1 references (e.g. the German S1Z1)
            if (localized && isSimpleR1C1Reference(docModel, PREFIX_CHARS, text)) { return true; }

            // the passed text is not a cell reference
            return false;
        }

        /**
         * Returns the original name of the sheet referred by the passed sheet
         * reference.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference. If set to null, an empty string will be
         *  returned to indicate a sheet-local reference.
         *
         * @returns {String|Null}
         *  An empty string, if null has been passed to this method; the
         *  original name of the sheet referred by the passed sheet reference;
         *  or null, if the sheet reference cannot be resolved to a sheet name.
         */
        function getSheetName(docModel, sheetRef) {
            return !sheetRef ? '' : _.isNumber(sheetRef.sheet) ? docModel.getSheetName(sheetRef.sheet) : sheetRef.sheet;
        }

        /**
         * Encodes the passed sheet name, if it is a complex name, and adds an
         * absolute marker if needed.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to recognize simple sheet names.
         *
         * @param {String|Null} sheetName
         *  The sheet name, or null to indicate a sheet reference error.
         *
         * @param {Boolean} abs
         *  Whether the sheet reference is absolute.
         *
         * @returns {String}
         *  The encoded sheet name. If null has been passed as sheet name, the
         *  #REF! error code will be returned. Complex sheet names will be
         *  enclosed in apostrophes. If the absolute flag has been set, and the
         *  sheet name is not empty, a dollar sign will be added before the
         *  (encoded) sheet name.
         */
        function encodeAbsSheetName(docModel, sheetName, abs) {

            // use the #REF! error code for invalid sheets, enclose complex sheet names in apostrophes
            if (!_.isString(sheetName)) {
                sheetName = REF_ERROR;
            } else if (sheetName && !self.isSimpleSheetName(docModel, sheetName)) {
                sheetName = SheetRef.encodeComplexSheetName(sheetName);
            }

            // add the leading absolute marker in ODF mode
            return ((abs && sheetName) ? '$' : '') + sheetName;
        }

        /**
         * Returns the string representation of the passed sheet references for
         * regular token syntax, i.e. as range of sheet names, and with an
         * exclamation mark as sheet name separator.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference. If set to null, an empty string will be
         *  returned.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference. Ignored, if the value null has been
         *  passed to parameter 'sheet1Ref'. If set to null, a single sheet
         *  name as described by the parameter 'sheet1Ref' will be used.
         *
         * @returns {String}
         *  The string representation for the passed sheet references, with a
         *  trailing exclamation mark; or an empty string, if no sheet
         *  references have been passed.
         */
        function generateSheetRangePrefix(docModel, sheet1Ref, sheet2Ref) {

            // empty string for missing sheet reference
            if (!sheet1Ref) { return ''; }

            // convert first sheet name
            var sheet1Name = getSheetName(docModel, sheet1Ref);
            if (!_.isString(sheet1Name)) { return REF_ERROR + '!'; }

            // convert second sheet name
            var sheet2Name = (sheet2Ref && !sheet1Ref.equals(sheet2Ref)) ? getSheetName(docModel, sheet2Ref) : '';
            if (!_.isString(sheet2Name)) { return REF_ERROR + '!'; }

            // the resulting sheet names
            var result = odf ? encodeAbsSheetName(docModel, sheet1Name, sheet1Ref.abs) : sheet1Name;
            if (sheet2Ref) {
                result += ':' + (odf ? encodeAbsSheetName(docModel, sheet2Name, sheet2Ref.abs) : sheet2Name);
            }

            // OOXML: enclose sheet range in apostrophes, if one of the sheet names is complex
            if (!odf && (!self.isSimpleSheetName(docModel, sheet1Name) || (sheet2Name && !self.isSimpleSheetName(docModel, sheet2Name)))) {
                result = SheetRef.encodeComplexSheetName(result);
            }

            return result + '!';
        }

        /**
         * Returns the string representation of the passed sheet reference for
         * the OpenFormula token syntax, i.e. with absolute sheet marker.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference. If set to null, an empty string will be
         *  returned to indicate a sheet-local reference.
         *
         * @returns {String}
         *  The string representation for the passed sheet reference; or an
         *  empty string, if no sheet reference has been passed.
         */
        function generateOFSheetPrefix(docModel, sheetRef) {

            // empty string for missing sheet reference
            if (!sheetRef) { return ''; }

            // convert reference to sheet name, use the #REF! error code for invalid sheets,
            // enclose complex sheet names in apostrophes
            var sheetName = getSheetName(docModel, sheetRef);
            return encodeAbsSheetName(docModel, sheetName, sheetRef.abs) + '.';
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the passed text is a reserved symbol according to
         * this formula grammar. Boolean literals, and strings that look like
         * cell addresses are considered to be reserved symbols. Cell addresses
         * include the representation of a relative cell reference in A1
         * notation, or a cell reference in R1C1 notation (either English, e.g.
         * 'R1C1', or according to this formula grammar, e.g. 'Z1S1' in the
         * German UI grammar).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index of
         *  strings that look like cell addresses.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is a reserved symbol according to this
         *  formula grammar.
         */
        this.isReservedSymbol = function (docModel, text) {
            return isBooleanLiteral(text) || isReferenceSymbol(docModel, text);
        };

        /**
         * Returns whether the passed sheet name is a simple sheet name, i.e.
         * it does not need to be enclosed in a pair of apostrophes in formula
         * expressions. Simple sheet names do not contain any reserved
         * characters (e.g. whitespace, or operators used in formulas), and do
         * not look like other reserved names (e.g. boolean literals, or cell
         * references).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index of
         *  sheet names that look like cell addresses.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.external=false]
         *      If set to true, the passed text may contain a leading simple
         *      file name enclosed in brackets, e.g. '[doc.xlsx]Sheet1', or
         *      even '[path\to\doc.xlsx]Sheet1'.
         *  @param {Boolean} [options.range=false]
         *      If set to true, the passed text may contain a sheet range (two
         *      simple sheet names separated by a colon), e.g. 'Sheet1:Sheet2'.
         *
         * @returns {Boolean}
         *  Whether the passed sheet name is a simple sheet name.
         */
        this.isSimpleSheetName = function (docModel, text, options) {

            // whether to accept leading document name in brackets
            var external = Utils.getBooleanOption(options, 'external', false);
            // whether to accept a single colon separating two simple sheet names
            var range = Utils.getBooleanOption(options, 'range', false);
            // the result of the regular expression test
            var matches = SIMPLE_SHEET_NAME_RE.exec(text);

            // passed text does not match at all
            if (!matches) { return false; }

            // without 'external' flag, external document name is not allowed
            if (matches[1] && !external) { return false; }

            // without 'range' flag, second sheet name is not allowed
            if (matches[3] && !range) { return false; }

            // reserved symbols are not simple sheet names
            if (this.isReservedSymbol(docModel, matches[2])) { return false; }
            if (matches[3] && this.isReservedSymbol(docModel, matches[3])) { return false; }

            return true;
        };

        /**
         * Checks if the passed string can be used as label for a defined name.
         *
         * @param {String} text
         *  The string to be checked.
         *
         * @returns {String}
         *  The empty string, if the passed text is a valid label for a defined
         *  name; otherwise one of the following error codes:
         *  - 'name:empty': The passed label is empty.
         *  - 'name:invalid': The passed label contains invalid characters.
         *  - 'name:address': The passed label would be a valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         */
        this.validateNameLabel = function (docModel, text) {

            // the passed string must not be empty
            if (text.length === 0) { return 'name:empty'; }

            // check that the passed string does not contain invalid characters
            if (!VALID_NAME_RE.test(text)) { return 'name:invalid'; }

            // check that the text does not look like a boolean literal (TODO: own error code?)
            if (!odf && isBooleanLiteral(text)) { return 'name:invalid'; }

            // bug 38786: check that the text does not look like a cell address
            if (isReferenceSymbol(docModel, text)) { return 'name:address'; }

            // the passed text is a valid label for a defined name
            return '';
        };

        /**
         * Converts the passed scalar value to its text representation in
         * formulas according to this formula grammar.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The value to be converted.
         *
         * @returns {String}
         *  The string representation of the passed value.
         */
        this.formatScalar = function (value) {

            switch (FormulaUtils.getScalarType(value)) {

                case FormulaUtils.ScalarType.NUMBER:
                    return !isFinite(value) ? this.getErrorName(ErrorCode.NUM) :
                        FormulaUtils.isZero(value) ? '0' :
                        formatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);

                case FormulaUtils.ScalarType.STRING:
                    // duplicate inner quotes, enclose in quotes
                    return '"' + value.replace(/"/g, '""') + '"';

                case FormulaUtils.ScalarType.BOOLEAN:
                    return this.getBooleanName(value);

                case FormulaUtils.ScalarType.ERROR:
                    return this.getErrorName(value);

                case FormulaUtils.ScalarType.NULL:
                    return '';
            }

            Utils.warn('FormulaGrammar.formatScalar(): unsupported scalar value type');
            return '';
        };

        /**
         * A generic formatter for a cell reference that takes preformatted
         * cell address components, and adds the sheet names of the specified
         * sheet references according to the formula grammar of this instance.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference (e.g. the particle 'Sheet1' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). If set to null, no sheet
         *  names will be inserted into the returned string.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference (e.g. the particle 'Sheet3' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). Ignored, if the value null
         *  has been passed to parameter 'sheet1Ref'. If set to null, a single
         *  sheet name as described by the parameter 'sheet1Ref' will be used
         *  (e.g. 'Sheet1!A1:C3').
         *
         * @param {String} startStr
         *  The text representation of the start cell address (e.g. the text
         *  'A1' for the reference 'Sheet1!A1:C3').
         *
         * @param {String|Null} [endStr]
         *  The text representation of the end cell address (e.g. the text 'C3'
         *  for the reference 'Sheet1!A1:C3'). If omitted, or set to null, or
         *  to an empty string, a single cell address will be generated (e.g.
         *  'Sheet1!A1').
         *
         * @returns {String}
         *  The string representation of the cell range reference.
         */
        this.formatGenericRange = function (docModel, sheet1Ref, sheet2Ref, startStr, endStr) {

            // OpenFormula syntax: sheets of a sheet inetrval will be added individually to the
            // start and end addresses of the range, e.g.: Sheet1.A1:Sheet2.B2
            if (this.REF_SYNTAX === 'of') {

                // the first sheet name with separator character, and the first cell address
                var result = generateOFSheetPrefix(docModel, sheet1Ref) + startStr;

                // add second cell address for sheet ranges, or for cell ranges (repeat
                // single cell address in a sheet range, e.g. [Sheet1.A1:Sheet2.A1])
                if (sheet2Ref || endStr) {
                    result += ':' + generateOFSheetPrefix(docModel, sheet2Ref) + (endStr || startStr);
                }

                return result;
            }

            // start with all sheet names before the cell range address, e.g. Sheet1:Sheet2!A1:C3
            return generateSheetRangePrefix(docModel, sheet1Ref, sheet2Ref) + startStr + (endStr ? (':' + endStr) : '');
        };

        /**
         * Converts the passed reference structures to the text representation
         * of a complete cell range reference in formulas according to this
         * formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference (e.g. the particle 'Sheet1' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). If set to null, no sheet
         *  names will be inserted into the returned string.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference (e.g. the particle 'Sheet3' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). Ignored, if the value null
         *  has been passed to parameter 'sheet1Ref'. If set to null, a single
         *  sheet name as described by the parameter 'sheet1Ref' will be used
         *  (e.g. 'Sheet1!A1:C3').
         *
         * @param {CellRef|Null} cell1Ref
         *  The first cell reference (e.g. the cell address 'A1' in 'A1:C3').
         *  If set to null, the #REF! error code will be returned instead of a
         *  cell range address.
         *
         * @param {CellRef|Null} cell2Ref
         *  The second cell reference (e.g. the cell address 'C3' in 'A1:C3').
         *  If set to null, a single cell address will be returned.
         *
         * @param {Boolean} [rcStyle=false]
         *  If set to true, the reference will be formatted in R1C1 style. If
         *  set to false or omitted, the reference will be formatted in A1
         *  style. MUST NOT be set to true, if this instance represents a
         *  native formula grammar.
         *
         * @returns {String}
         *  The string representation of the cell range reference.
         */
        this.formatReference = function (docModel, sheet1Ref, sheet2Ref, cell1Ref, cell2Ref, rcStyle) {

            // R1C1 notation not supported in native formula grammars (UI only)
            if (!localized && rcStyle) {
                Utils.error('FormulaGrammar.formatReference(): R1C1 notation not supported in native grammar');
                return NA_ERROR;
            }

            // the cell range address (with adjusted column/row indexes)
            var range = cell1Ref ? Range.createFromAddresses(cell1Ref.toAddress(), cell2Ref && cell2Ref.toAddress()) : null;
            // the text representation of the start and end cell address
            var startStr = null, endStr = null;

            // OpenFormula syntax: references are enclosed in brackets, absolute markers for
            // sheets, sheet names are separated by periods, e.g.: [$Sheet1.A1:$'Sheet 2'.B2]
            if (this.REF_SYNTAX === 'of') {

                // the text representation of the start address
                startStr = range ? cell1Ref.refText() : (REF_ERROR + REF_ERROR);
                // start address will always contain a leading period, also without sheet name
                if (!sheet1Ref) { startStr = '.' + startStr; }

                // the text representation of the end address
                if (cell2Ref) {
                    endStr = range ? cell2Ref.refText() : (REF_ERROR + REF_ERROR);
                    // end address will always contain a leading period, also without sheet name
                    if (!sheet2Ref) { endStr = '.' + endStr; }
                }

                // enclose the entire reference into brackets
                return '[' + this.formatGenericRange(docModel, sheet1Ref, sheet2Ref, startStr, endStr) + ']';
            }

            if (!range) {
                // no valid range: generate a #REF! error with (optional) leading sheet name
                startStr = REF_ERROR;
            } else if (cell1Ref.absCol && cell2Ref && cell2Ref.absCol && docModel.isRowRange(range)) {
                // generate row interval (preferred over column interval for entire sheet range)
                startStr = rcStyle ? cell1Ref.rowTextRC(PREFIX_CHARS) : cell1Ref.rowText();
                if (!rcStyle || !range.singleRow() || (cell1Ref.absRow !== cell2Ref.absRow)) {
                    endStr = rcStyle ? cell2Ref.rowTextRC(PREFIX_CHARS) : cell2Ref.rowText();
                }
            } else if (cell1Ref.absRow && cell2Ref && cell2Ref.absRow && docModel.isColRange(range)) {
                // generate column interval
                startStr = rcStyle ? cell1Ref.colTextRC(PREFIX_CHARS) : cell1Ref.colText();
                if (!rcStyle || !range.singleCol() || (cell1Ref.absCol !== cell2Ref.absCol)) {
                    endStr = rcStyle ? cell2Ref.colTextRC(PREFIX_CHARS) : cell2Ref.colText();
                }
            } else {
                // generate range or cell address
                startStr = rcStyle ? cell1Ref.refTextRC(PREFIX_CHARS) : cell1Ref.refText();
                if (cell2Ref) {
                    endStr = rcStyle ? cell2Ref.refTextRC(PREFIX_CHARS) : cell2Ref.refText();
                }
            }

            return this.formatGenericRange(docModel, sheet1Ref, sheet2Ref, startStr, endStr);
        };

        /**
         * Converts the passed sheet reference and label of a defined name to
         * the text representation of a complete defined name reference in
         * formulas according to this formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference (e.g. the particle 'Sheet1' in 'Sheet1!name').
         *  If set to null, no sheet name will be inserted into the returned
         *  string.
         *
         * @param {String} label
         *  The label of the defined name.
         *
         * @returns {String}
         *  The string representation of the defined name.
         */
        this.formatName = function (docModel, sheetRef, label) {
            // no sheet-local names in ODF files
            return (odf && sheetRef) ? NA_ERROR : (generateSheetRangePrefix(docModel, sheetRef) + label);
        };

        /**
         * Converts the passed sheet reference and macro function name to the
         * text representation of a complete macro call in formulas according
         * to this formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference (e.g. the particle 'Sheet1' in the formula
         *  'Sheet1!macro()'). If set to null, no sheet name will be inserted
         *  into the returned string.
         *
         * @param {String} label
         *  The name of a macro function.
         *
         * @returns {String}
         *  The string representation of the macro call.
         */
        this.formatMacro = function (docModel, sheetRef, label) {

            // no sheet-local macros in ODF files
            if (odf) { return sheetRef ? NA_ERROR : label; }

            // OOXML: macro names equal to the internal name of a built-in function will be prefixed
            // Example: the formula '=SUMME(1)+SUM(1)' in a German UI results in the expression 'SUM(1)+_xludf.SUM(1)'
            var hasUdfPrefix = (UDF_PREFIX.length < label.length) && (label.substr(0, UDF_PREFIX.length).toLowerCase() === UDF_PREFIX);

            // add a missing prefix for operation grammer, or remove existing prefix for UI grammar
            if (localized && hasUdfPrefix) {
                label = label.substr(UDF_PREFIX.length);
            } else if (!localized && !hasUdfPrefix && functionCollection.getKey(label, false)) {
                label = UDF_PREFIX + label;
            }

            return generateSheetRangePrefix(docModel, sheetRef) + label;
        };

        /**
         * Returns the name of the passed boolean value.
         *
         * @param {Boolean} value
         *  The boolean value to be converted to its string representation.
         *
         * @returns {String}
         *  The name of the passed boolean value.
         */
        this.getBooleanName = function (value) {
            return booleanCollection.getName(value ? 't' : 'f', localized);
        };

        /**
         * Converts the passed name of a boolean value to the boolean value.
         *
         * @param {String} name
         *  The name of a boolean value (case-insensitive).
         *
         * @returns {Boolean|Null}
         *  The boolean value for the passed name of a boolean value; or null,
         *  if the passed string is not the name of a boolean value.
         */
        this.getBooleanValue = function (name) {
            var key = booleanCollection.getKey(name, localized);
            return key ? (key === 't') : null;
        };

        /**
         * Returns the name of the passed error code.
         *
         * @param {ErrorCode} error
         *  The error code to be converted to its string representation.
         *
         * @returns {String|Null}
         *  The name of the passed error code; or null, if the passed error
         *  code cannot be resolved to a name.
         */
        this.getErrorName = function (error) {
            return errorCollection.getName(error.key, localized);
        };

        /**
         * Converts the passed error name to the error code instance.
         *
         * @param {String} name
         *  The name of an error code (case-insensitive).
         *
         * @returns {ErrorCode|Null}
         *  The error code instance for the passed error name; or null, if the
         *  passed string is not the name of a supported error code.
         */
        this.getErrorCode = function (name) {
            var key = errorCollection.getKey(name, localized);
            return key ? ErrorCode.create(key) : null;
        };

        /**
         * Returns the name of an operator for the passed unique resource key.
         *
         * @param {String} key
         *  The unique resource key of an operator.
         *
         * @returns {String|Null}
         *  The operator name for the passed resource key; or null, if the
         *  passed resource key cannot be resolved to the name of an operator.
         */
        this.getOperatorName = function (key) {
            return operatorCollection.getName(key, localized);
        };

        /**
         * Converts the passed operator name to the unique resource key of the
         * operator.
         *
         * @param {String} name
         *  The name of an operator.
         *
         * @returns {String|Null}
         *  The resource key for the passed operator name; or null, if the
         *  passed string is not the name of a supported operator.
         */
        this.getOperatorKey = function (name) {
            return operatorCollection.getKey(name, localized);
        };

        /**
         * Returns the name of a function for the passed unique resource key.
         *
         * @param {String} key
         *  The unique resource key of a function.
         *
         * @returns {String|Null}
         *  The function name for the passed resource key; or null, if the
         *  passed resource key cannot be resolved to the name of a function.
         */
        this.getFunctionName = function (key) {
            return functionCollection.getName(key, localized);
        };

        /**
         * Converts the passed function name to the unique resource key of the
         * function.
         *
         * @param {String} name
         *  The name of a function (case-insensitive).
         *
         * @returns {String|Null}
         *  The resource key for the passed function name; or null, if the
         *  passed string is not the name of a supported function.
         */
        this.getFunctionKey = function (name) {
            return functionCollection.getKey(name, localized);
        };

        /**
         * Returns the name of a table region for the passed unique resource
         * key.
         *
         * @param {String} key
         *  The unique resource key of a table region.
         *
         * @returns {String|Null}
         *  The name of a table region for the passed resource key; or null, if
         *  the passed resource key cannot be resolved to the name of a table
         *  region.
         */
        this.getTableRegionName = function (key) {
            return regionCollection.getName(key, localized);
        };

        /**
         * Converts the passed name of a table region to the unique resource
         * key of the table region.
         *
         * @param {String} name
         *  The name of a table region (with leading hash character).
         *
         * @returns {String|Null}
         *  The resource key for the passed table region; or null, if the
         *  passed string is not the name of a supported table region.
         */
        this.getTableRegionKey = function (name) {
            return regionCollection.getKey(name, localized);
        };

        /**
         * Returns whether the passed table column name is considered to be
         * simple, i.e. it can be used without brackets in complex structured
         * table references in UI formula grammars.
         *
         * @param {String} colName
         *  The table column name to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed table column name is considered to be simple.
         */
        this.isSimpleTableColumn = function (colName) {
            return SIMPLE_TABLE_COLUMN_NAME_RE.test(colName);
        };

        /**
         * Converts the passed text representation of a table column name (used
         * in formula expressions) to the original column name.
         *
         * @param {String} colName
         *  The encoded name of a table column.
         *
         * @returns {String}
         *  The original table column name.
         */
        this.decodeTableColumn = function (colName) {
            return colName.replace(/'(.)/g, '$1');
        };

        /**
         * Converts the passed table column name to the text representation of
         * a table column reference as used in structured table references in
         * formulas according to this formula grammar.
         *
         * @param {String} colName
         *  The name of a table column.
         *
         * @returns {String}
         *  The string representation of the table column reference.
         */
        this.encodeTableColumn = function (colName) {
            return colName.replace(TABLE_COL_ESCAPE_RE, '\'$&');
        };

        /**
         * Generates the formula expression for a generic subtotal formula.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to generate references.
         *
         * @param {String} funcKey
         *  The resource key of a function.
         *
         * @param {Range} range
         *  The address of the cell range to be inserted into the formula.
         *
         * @param {Boolean} [rcStyle=false]
         *  If set to true, the reference will be formatted in R1C1 style. If
         *  set to false or omitted, the reference will be formatted in A1
         *  style. MUST NOT be set to true, if this instance represents a
         *  native formula grammar.
         *
         * @returns {String}
         *  The formula expression of the subtotal formula.
         */
        this.generateAutoFormula = function (docModel, funcKey, range, rcStyle) {

            var formula = this.getFunctionName(funcKey) + '(';

            if (range) {
                var cell1Ref = new CellRef(range.start[0], range.start[1], false, false);
                var cell2Ref = range.single() ? null : new CellRef(range.end[0], range.end[1], false, false);
                formula += this.formatReference(docModel, null, null, cell1Ref, cell2Ref, rcStyle);
            }

            return formula + ')';
        };

    } // class FormulaGrammar

    // static methods ---------------------------------------------------------

    /**
     * Returns an existing formula grammar singleton from the internal cache,
     * if available; otherwise creates a new instance.
     *
     * @param {String} fileFormat
     *  The identifier of the file format related to the formula resource data
     *  used by the formula grammar.
     *
     * @param {String} grammarId
     *  The identifier of a formula grammar. Supported values are:
     *  - 'op': Fixed token representations as used in operations, and the file
     *      format.
     *  - 'ui': The localized token representations according to the current
     *      UI language of the application.
     *  - 'en': The token representations according to the American English UI
     *      language (locale 'en_US').
     *
     * @returns {FormulaGrammar}
     *  A formula grammar singleton for the passed parameters.
     */
    FormulaGrammar.create = (function () {

        // create a hash key for the memoize() cache
        function getConfigKey(fileFormat, grammarId) {
            return fileFormat + ':' + grammarId;
        }

        // create a new instance of FormulaGrammar
        function createConfig(fileFormat, grammarId) {
            return new FormulaGrammar(fileFormat, grammarId);
        }

        // create a function that caches all created formula grammars
        return _.memoize(createConfig, getConfigKey);
    }());

    // exports ================================================================

    return FormulaGrammar;

});
