/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/matrixfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/matrix'
], function (ErrorCode, FormulaUtils, Matrix) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with matrixes.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, matrixes can be given as literals in formulas,
     * or can be extracted from rectangular cell ranges. The client-side
     * formula engine provides the function signature type 'mat' (and its sub
     * types such as 'mat:num'), and the class Matrix whose instances contain
     * the elements of a matrix in a two-dimensional array.
     *
     *************************************************************************/

    // aggregation options for the SUMPRODUCT function
    var SUMPRODUCT_OPTIONS = {
        valMode: 'exact', // value operands: must be a number
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        emptyParam: true, // empty parameter will lead to #VALUE! (due to 'exact' mode)
        // complexRef: false, // do not accept multi-range and multi-sheet references
        checkSize: 'exact' // operands must have equal width and height
    };

    // aggregation options for other matrix sum functions (SUMX2MY2 etc.)
    var SUM_XY_OPTIONS = {
        valMode: 'exact', // value operands: must be numbers
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        emptyParam: true, // empty parameter will lead to #VALUE! (due to 'exact' mode)
        // complexRef: false, // do not accept multi-range and multi-sheet references
        checkSize: 'size', // operands must have same element count, but ignore width/height (e.g. 2x2 and 1x4)
        sizeMismatchError: ErrorCode.NA // throw #N/A error code if the sizes do not match
    };

    // private global functions ===============================================

    /**
     * Throws an error code, if the passed matrix is not a square matrix (i.e.,
     * its width and height are different).
     *
     * @param {Matrix} matrix
     *  The matrix to be checked.
     *
     * @returns {Number}
     *  The size of a square matrix.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the passed matrix is not a square matrix.
     */
    function ensureSquareMatrix(matrix) {
        var dim = matrix.cols();
        if (dim !== matrix.rows()) { throw ErrorCode.VALUE; }
        return dim;
    }

    /**
     * Returns an identity matrix with the specified size.
     *
     * @param {Number} dim
     *  The resulting size of the identity matrix.
     *
     * @returns {Matrix}
     *  The identity matrix with the specified size.
     */
    function createIdentityMatrix(dim) {
        return Matrix.generate(dim, dim, function (row, col) {
            return (row === col) ? 1 : 0;
        });
    }

    // exports ================================================================

    return {

        MDETERM: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'mat:num',
            resolve: (function () {

                function determ(matrix) {
                    // http://www.mathebibel.de/3x3-determinanten-berechnen
                    // https://www.youtube.com/watch?v=AnezNBuRkEY
                    var dim = ensureSquareMatrix(matrix);
                    return (dim === 1) ? matrix.get(0, 0) : determLaplace(matrix);
                }

                function determLaplace(matrix) {
                    var size = matrix.cols();

                    var firstCol = _.times(size, function (index) {
                        var firstValue = matrix.get(0, index);
                        return (index % 2) ? -firstValue : firstValue;
                    });

                    var result = 0;
                    _.each(firstCol, function (firstValue, index) {
                        var subMatrix = [];
                        _.times(size, function (j) {
                            if (j === index) { return; }
                            var line = _.times(size - 1, function (i) {
                                return matrix.get(i + 1, j);
                            });
                            subMatrix.push(line);
                        });

                        result += firstValue * determ(new Matrix(subMatrix));
                    });
                    return result;
                }

                return determ;
            }())
        },

        MINVERSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:num',
            resolve: function (matrix) {
                // http://blog.acipo.com/matrix-inversion-in-javascript/
                // by  Andrew Ippoliti

                // I use Guassian Elimination to calculate the inverse:
                // (1) 'augment' the matrix (left) by the identity (on the right)
                // (2) Turn the matrix on the left into the identity by elemetry row ops
                // (3) The matrix on the right is the inverse (was the identity matrix)
                // There are 3 elemtary row ops: (I combine b and c in my code)
                // (a) Swap 2 rows
                // (b) Multiply a row by a scalar
                // (c) Add 2 rows

                var dim = ensureSquareMatrix(matrix);

                //create the identity matrix, and a copy of the original
                var identity = createIdentityMatrix(dim);
                var copy = matrix.clone();

                // Perform elementary row operations
                var i = 0, ii = 0, j = 0, e = 0;
                for (i = 0; i < dim; i += 1) {
                    // get the element e on the diagonal
                    e = copy.get(i, i);

                    // if we have a 0 on the diagonal (we'll need to swap with a lower row)
                    if (e === 0) {
                        //look through every row below the i'th row
                        for (ii = i + 1; ii < dim; ii += 1) {
                            //if the ii'th row has a non-0 in the i'th col
                            if (copy.get(ii, i) !== 0) {
                                //it would make the diagonal have a non-0 so swap it
                                for (j = 0; j < dim; j++) {
                                    e = copy.get(i, j);       //temp store i'th row
                                    copy.set(i, j, copy.get(ii, j));//replace i'th row by ii'th
                                    copy.set(ii, j, e);      //repace ii'th by temp
                                    e = identity.get(i, j);       //temp store i'th row
                                    identity.set(i, j, identity.get(ii, j));//replace i'th row by ii'th
                                    identity.set(ii, j, e);      //repace ii'th by temp
                                }
                                //don't bother checking other rows since we've swapped
                                break;
                            }
                        }
                        //get the new diagonal
                        e = copy.get(i, i);
                        //if it's still 0, not invertable (error)
                        if (e === 0) { throw ErrorCode.NUM; }
                    }

                    // Scale this row down by e (so we have a 1 on the diagonal)
                    for (j = 0; j < dim; j++) {
                        copy.set(i, j, copy.get(i, j) / e); //apply to original matrix
                        identity.set(i, j, identity.get(i, j) / e); //apply to identity
                    }

                    // Subtract this row (scaled appropriately for each row) from ALL of
                    // the other rows so that there will be 0's in this column in the
                    // rows above and below this one
                    for (ii = 0; ii < dim; ii++) {
                        // Only apply to other rows (we want a 1 on the diagonal)
                        if (ii === i) { continue; }

                        // We want to change this element to 0
                        e = copy.get(ii, i);

                        // Subtract (the row above(or below) scaled by e) from (the
                        // current row) but start at the i'th column and assume all the
                        // stuff left of diagonal is 0 (which it should be if we made this
                        // algorithm correctly)
                        for (j = 0; j < dim; j++) {
                            copy.set(ii, j, copy.get(ii, j) - e * copy.get(i, j)); //apply to original matrix
                            identity.set(ii, j, identity.get(ii, j) - e * identity.get(i, j)); //apply to identity
                        }
                    }
                }

                //we've done all operations, copy should be the identity
                //matrix identity should be the inverse:
                return identity;
            }
        },

        MMULT: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: 'mat:num mat:num',
            resolve: function (matrix1, matrix2) {

                // width of first matrix must match height of second matrix
                if (matrix1.cols() !== matrix2.rows()) { throw ErrorCode.VALUE; }

                // calculate all elements of the result matrix
                return Matrix.generate(matrix1.rows(), matrix2.cols(), function (row, col) {
                    var elem = 0;
                    for (var i = 0, l = matrix1.cols(); i < l; i += 1) {
                        elem += (matrix1.get(row, i) * matrix2.get(i, col));
                    }
                    return elem;
                });
            }
        },

        MUNIT: {
            category: 'matrix',
            name: { ooxml: '_xlfn.MUNIT' },
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'val:int',
            resolve: function (dim) {
                if (dim < 1) { throw ErrorCode.VALUE; }
                if (dim > Math.min(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)) { throw FormulaUtils.UNSUPPORTED_ERROR; }
                return createIdentityMatrix(dim);
            }
        },

        SUMPRODUCT: {
            category: 'matrix math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                function aggregate(result, numbers) { return result + numbers.reduce(FormulaUtils.multiply, 1); }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUMPRODUCT_OPTIONS);
            }
        },

        SUMX2MY2: {
            category: 'matrix math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'any any',
            resolve: function () {
                function aggregate(result, nums) { return result + nums[0] * nums[0] - nums[1] * nums[1]; }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUM_XY_OPTIONS);
            }
        },

        SUMX2PY2: {
            category: 'matrix math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'any any',
            resolve: function () {
                function aggregate(result, nums) { return result + nums[0] * nums[0] + nums[1] * nums[1]; }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUM_XY_OPTIONS);
            }
        },

        SUMXMY2: {
            category: 'matrix math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'any any',
            resolve: function () {
                function aggregate(result, nums) { var diff = nums[0] - nums[1]; return result + diff * diff; }
                return this.aggregateNumbersParallel(arguments, 0, aggregate, _.identity, SUM_XY_OPTIONS);
            }
        },

        TRANSPOSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:any',
            resolve: function (matrix) {
                return Matrix.generate(matrix.cols(), matrix.rows(), function (row, col) {
                    return matrix.get(col, row);
                });
            }
        }
    };

});
