/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/informationfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/cellref'
], function (ErrorCode, CellRef) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions providing some kind of
     * information about the contents of sheet cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // shortcuts to mathematical functions
    var floor = Math.floor;
    var abs = Math.abs;

    // exports ================================================================

    return {

        CELL: {
            category: 'information',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            recalc: 'always',
            signature: 'val:str ref:single|deps:skip',
            resolve: (function () {

                function formatAddress(context, address) {
                    var cellRef = CellRef.create(address, true, true);
                    context.checkCellRef(cellRef, false);
                    return context.formulaGrammar.formatReference(context.docModel, null, null, cellRef, null, false);
                }

                function getCellCollection(context, sheet) {
                    return context.docModel.getSheetModel(sheet).getCellCollection();
                }

                function getCellAttributes(context, sheet, address) {
                    return getCellCollection(context, sheet).getCellAttributeSet(address);
                }

                function getColorFormat(context, sheet, address) {
                    var numberSections = getCellCollection(context, sheet).getCellParsedFormat(address).numberSections;
                    return ((numberSections.length > 1) && numberSections[1].colorName) ? 1 : 0;
                }

                function getCellFormatCode(context, sheet, address) {
                    var parsedFormat = getCellCollection(context, sheet).getCellParsedFormat(address);
                    var formats = ['G', 'F0', ',0', 'F2', ',2', 'C0', 'C0-', 'C2', 'C2-', 'P0', 'P2', 'S2', 'G', 'D4', 'D1', 'D2', 'D3', 'D5', 'D7', 'D6', 'D9', 'D8'];
                    var formatId = context.numberFormatter.resolveFormatId(parsedFormat.formatCode);
                    return ((formatId !== null) && (formatId <= formats.length)) ? formats[formatId] : 'G';
                }

                function getPrefix(context, sheet, address) {
                    var cellTextAlign = getCellCollection(context, sheet).getCellTextOrientation(address).cssTextAlign;
                    switch (cellTextAlign) {
                        case 'left':
                            return '\'';
                        case 'right':
                            return '"';
                        case 'center':
                            return '^';
                        case 'justify':
                            return '\\';
                        default:
                            return '';
                    }
                }

                function getValueType(context, sheet, address) {
                    var value = context.getCellValue(sheet, address);
                    return (value === null) ? 'b' : (typeof value === 'string') ? 'l' : 'v';
                }

                return function (infoType, range) {

                    // resolve translated or English info type (prefer translations over English names)
                    var paramCollection = this.formulaResource.getCellParamCollection();
                    var paramKey = paramCollection.getKey(infoType, true) || paramCollection.getKey(infoType, false);

                    // get the address of the target cell, use reference address if omitted (in difference to Excel that uses selected cell)
                    var sheet = range ? range.sheet1 : this.getRefSheet();
                    var address = range ? range.start : this.getRefAddress();

                    switch (paramKey) {
                        case 'ADDRESS':
                            return formatAddress(this, address);
                        case 'COL':
                            return address[0] + 1;
                        case 'COLOR':
                            return getColorFormat(this, sheet, address);
                        case 'CONTENTS':
                            return this.getCellValue(sheet, address);
                        case 'FILENAME':
                            return this.app.getFullFileName();
                        case 'FORMAT':
                            return getCellFormatCode();
                        case 'PARENTHESES':
                            throw ErrorCode.NA;
                        case 'PREFIX':
                            return getPrefix(this, sheet, address);
                        case 'PROTECT':
                            return getCellAttributes(this, sheet, address).cell.unlocked ? 0 : 1;
                        case 'ROW':
                            return address[1] + 1;
                        case 'TYPE':
                            return getValueType(this, sheet, address);
                        case 'WIDTH':
                            throw ErrorCode.NA;
                    }
                    throw ErrorCode.VALUE;
                };
            }())
        },

        CURRENT: {
            category: 'information',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.CURRENT' },
            hidden: true,
            minParams: 0,
            maxParams: 0,
            type: 'any'
        },

        'ERROR.TYPE': {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                if (value instanceof ErrorCode) { return value.num; }
                throw ErrorCode.NA;
            }
        },

        'ERRORTYPE.ODF': {
            category: 'information',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.ERRORTYPE' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        INFO: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            recalc: 'always'
        },

        ISBLANK: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isNull // error codes result in FALSE
        },

        ISERR: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                return (value instanceof ErrorCode) && (value !== ErrorCode.NA);
            }
        },

        ISERROR: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) { return value instanceof ErrorCode; }
        },

        ISEVEN: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) === 0;
            }
        },

        ISFORMULA: {
            category: 'information',
            name: { ooxml: '_xlfn.ISFORMULA' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            recalc: 'always',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {

                // always use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // accept reference to own cell (no circular reference error)
                if (this.isRefSheet(range.sheet1) && this.isRefAddress(address)) {
                    return true; // do not check the cell, a new formula has not been inserted yet
                }

                // getCellFormula() returns null for value cells
                return _.isString(this.getCellFormula(range.sheet1, address));
            }
        },

        ISLOGICAL: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isBoolean // error codes and empty cells result in FALSE
        },

        ISNA: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) { return value === ErrorCode.NA; }
        },

        ISNONTEXT: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                return !_.isString(value); // error codes and empty cells result in TRUE
            }
        },

        ISNUMBER: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isNumber // error codes and empty cells result in FALSE
        },

        ISODD: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) !== 0;
            }
        },

        ISREF: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any|deps:skip',
            resolve: function (operator) { return operator.isReference(); } // error codes result in FALSE
        },

        ISTEXT: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isString // error codes and empty cells result in FALSE
        },

        NA: {
            category: 'information',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(ErrorCode.NA)
        },

        STYLE: {
            category: 'information',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.STYLE' },
            hidden: true,
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        TYPE: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any', // resolve cell references to values, accept error codes
            resolve: function (value) {
                // array literals always result in 64 regardless of their contents
                if (this.getOperand(0).isMatrix()) { return 64; }
                // determine type of value parameters and cell contents
                if (_.isNumber(value)) { return 1; }
                if (_.isString(value)) { return 2; }
                if (_.isBoolean(value)) { return 4; }
                if (value instanceof ErrorCode) { return 16; }
                // default (reference to empty cell, or cell not fetched from server)
                return 1;
            }
        }
    };

});
