/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/formulautils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (Utils, Logger, ErrorCode) {

    'use strict';

    // shortcuts to mathematical functions
    var floor = Math.floor;
    var ceil = Math.ceil;
    var round = Math.round;
    var abs = Math.abs;
    var pow = Math.pow;
    var sqrt = Math.sqrt;
    var exp = Math.exp;
    var log = Math.log;

    // mathematical constants
    var PI = Math.PI;
    var SQRT_PI = sqrt(PI);
    var LN10 = Math.LN10;

    // Depth for the taylor expansion (exponent) and continued fraction expansion (number of fractions)
    // to calculate an approximation for erf() and erfc(), see functions erf_taylor() and erfc_cfrac().
    var ERF_DEPTH = 25;

    // The threshold for x values to select between the two expansion algorithms to calculate erf().
    // Using a depth of 25, the minimum distance between both expansions is about 6.6791e-13 at x=1.9785.
    // Therefore, for 0<=x<=1.9785, taylor expansion will be used, and for x>1.9785, the continued
    // fraction expansion will be used.
    var ERF_THRESHOLD = 1.9785;

    // static private functions ===============================================

    /**
     * Creates a special error code for internal usage in the formula engine.
     */
    function makeInternalErrorCode(key, value) {
        var errorCode = new ErrorCode(key);
        errorCode.internal = true;
        errorCode.value = value;
        return errorCode;
    }

    /**
     * Returns the neutral value for the passed scalar value.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  Any scalar value used in spreadsheet formulas.
     *
     * @returns {Number|String|Boolean|Null}
     *  The neutral value for the passed scalar value, if available. Returns
     *  the number zero for any number, the empty string for any string, or the
     *  value false for a boolean value. Otherwise, null will be returned
     *  (especially for error codes which do not have a neutral value).
     */
    function getNeutralValue(value) {
        switch (typeof value) {
            case 'number': return 0;
            case 'string': return '';
            case 'boolean': return false;
        }
        return null; // null for error codes
    }

    /**
     * Compares the passed integers, and returns a signed indicator number.
     *
     * @param {Number} int1
     *  The first integer for comparison. MUST be finite.
     *
     * @param {Number} int2
     *  The second integer for comparison. MUST be finite.
     *
     * @returns {Number}
     *  The number -1, if int1 is less than int2; or the number 1, if int1 is
     *  greater than int2; or the number 0, if both integers are equal.
     */
    function compareIntegers(int1, int2) {
        return (int1 === int2) ? 0 : (int1 < int2) ? -1 : 1;
    }

    /**
     * Rounds the passed number to 15 significant digits.
     *
     * @param {Number} number
     *  The number to be rounded. MUST be finite.
     *
     * @returns {Number}
     *  The rounded number.
     */
    function round15(number) {
        var norm = Utils.normalizeNumber(number);
        return round(norm.mant * 1e14) * FormulaUtils.pow10(norm.exp - 14);
    }

    /**
     * Returns whether the passed numbers are considered to be equal, allowing
     * a slight difference for equally signed numbers.
     *
     * @param {Number} number1
     *  The first number for comparison. MUST be finite.
     *
     * @param {Number} number2
     *  The second number for comparison. MUST be finite.
     *
     * @returns {Boolean}
     *  Whether the passed numbers are considered to be equal.
     */
    function equalNumbers(number1, number2) {

        // shortcut: true, if the numbers are exactly equal
        if (number1 === number2) { return true; }

        // check whether at least one number is zero or denormalized
        var zero1 = FormulaUtils.isZero(number1);
        var zero2 = FormulaUtils.isZero(number2);
        if (zero1 || zero2) { return zero1 === zero2; }

        // false, if the numbers have different signs
        if ((number1 < 0) !== (number2 < 0)) { return false; }

        // performance: false, if the numbers have a decent difference
        var quot = abs((number1 / number2) - 1);
        if (!isFinite(quot) || (quot > 1e-14)) { return false; }

        // round the numbers to 15 significant digits for comparison
        return round15(number1) === round15(number2);
    }

    /**
     * Returns whether the passed strings are considered to be equal.
     *
     * @param {String} string1
     *  The first string for comparison.
     *
     * @param {String} string2
     *  The second string for comparison.
     *
     * @param {Boolean} withCase
     *  If set to true, the strings will be compared case-sensitively.
     *
     * @returns {Boolean}
     *  Whether the passed strings are considered to be equal.
     */
    function equalStrings(string1, string2, withCase) {
        return withCase ? (string1 === string2) : (string1.toUpperCase() === string2.toUpperCase());
    }

    /**
     * Implementation of the taylor expansion for erf() with maximum depth 25.
     * The maximum error to the 'better' results of erf() as computed with
     * taylor expansion, depth 50, is about 2.35e-13 at x=1.9785 (the threshold
     * point between taylor expansion and continued fraction expansion).
     *
     * @param {Number} x
     *  The input parameter. MUST NOT be negative.
     *
     * @returns {Number}
     *  The approximated result of erf(x) for small positive x.
     */
    function erf_taylor(x) {
        var y = x;
        for (var i = 1, j = 3, fact = 1, sgn = -1, y2 = 0; (i <= ERF_DEPTH) && (y !== y2); i += 1, j += 2, fact *= i, sgn = -sgn) {
            y2 = y;
            y += sgn * pow(x, j) / j / fact;
        }
        return 2 * y / SQRT_PI;
    }

    /**
     * Implementation of the continued fraction expansion for erfc() with depth
     * 25. The maximum error to the 'better' results of erfc() as computed with
     * continued fraction expansion, depth 50, is about 4.3e-13 at x=1.9785
     * (the threshold point between taylor expansion and continued fraction
     * expansion).
     *
     * @param {Number} x
     *  The input parameter. MUST NOT be negative.
     *
     * @returns {Number}
     *  The approximated result of erfc(x) for large positive x.
     */
    function erfc_cfrac(x) {
        var y = 0;
        for (var i = ERF_DEPTH, j = (i % 2 + 1); i >= 1; i -= 1, j = 3 - j) {
            y = i / (j * x + y);
        }
        return exp(-x * x) / SQRT_PI / (x + y);
    }

    // static class FormulaUtils ==============================================

    /**
     * The static class FormulaUtils provides low-level helper functions needed
     * to implement the formula engine of a spreadsheet application.
     * Additionally, the class is a console logger bound to the URL hash flag
     * 'spreadsheet:log-formulas', that logs detailed information about the
     * formula parser, compiler, and interpreter.
     */
    var FormulaUtils = {};

    // logger interface -------------------------------------------------------

    Logger.extend(FormulaUtils, { enable: 'spreadsheet:log-formulas', prefix: 'FMLA' });

    // constants --------------------------------------------------------------

    /**
     * Maximum number of function parameters (CalcEngine limitation).
     *
     * @constant
     */
    FormulaUtils.MAX_PARAM_COUNT = 254;

    /**
     * Maximum number of rows supported in matrixes.
     *
     * @constant
     */
    FormulaUtils.MAX_MATRIX_ROW_COUNT = 16;

    /**
     * Maximum number of columns supported in matrixes.
     *
     * @constant
     */
    FormulaUtils.MAX_MATRIX_COL_COUNT = 16;

    /**
     * Maximum number of references in a reference list.
     *
     * @constant
     */
    FormulaUtils.MAX_REF_LIST_SIZE = 1024;

    /**
     * Maximum time available for evaluating a single formula, in milliseconds.
     *
     * @constant
     */
    FormulaUtils.MAX_EVAL_TIME = 2000;

    /**
     * Type identifiers for different types of scalar values used in formulas.
     * The type identifiers represent a natural order for different data types,
     * as used in spreadsheet formulas or for sorting (for example, strings are
     * always considered to be greater than numbers).
     *
     * @constant
     *
     * @property {Number} NULL
     *  Type specifier for the special value null representing a blank cell or
     *  an empty function parameter. Used in situations when null values will
     *  be considered to be less than any other scalar value.
     *
     * @property {Number} NUMBER
     *  Type specifier for floating-point numbers.
     *
     * @property {Number} STRING
     *  Type specifier for strings.
     *
     * @property {Number} BOOLEAN
     *  Type specifier for boolean values.
     *
     * @property {Number} ERROR
     *  Type specifier for error codes (instances of class ErrorCode).
     *
     * @property {Number} NULL_MAX
     *  Type specifier for the special value null representing a blank cell or
     *  an empty function parameter. Used in situations when null values will
     *  be considered to be greater than any other scalar value.
     */
    FormulaUtils.ScalarType = {
        NULL: 0,
        NUMBER: 1,
        STRING: 2,
        BOOLEAN: 3,
        ERROR: 4,
        NULL_MAX: 5
    };

    /**
     * Special error code literal to transport the 'unsupported' state.
     */
    FormulaUtils.UNSUPPORTED_ERROR = makeInternalErrorCode('unsupported', ErrorCode.NA);

    /**
     * Special error code literal to transport the 'circular reference' result.
     */
    FormulaUtils.CIRCULAR_ERROR = makeInternalErrorCode('circular', 0);

    /**
     * Special error code literal for other internal interpreter errors.
     */
    FormulaUtils.INTERNAL_ERROR = makeInternalErrorCode('internal', ErrorCode.NA);

    // public methods ---------------------------------------------------------

    /**
     * Throws the passed value, if it is an error code. Otherwise, the passed
     * value will be returned.
     *
     * @param {Any} value
     *  Any scalar value used in formulas.
     *
     * @returns {Any}
     *  The passed value, if it is not an error code (instance of ErrorCode).
     *
     * @throws {ErrorCode}
     *  The passed value, if it is an error code.
     */
    FormulaUtils.throwErrorCode = function (value) {
        if (value instanceof ErrorCode) { throw value; }
        return value;
    };

    /**
     * Prints the passed error message to the browser console, and throws the
     * internal error code FormulaUtils.INTERNAL_ERROR.
     *
     * @attention
     *  MUST ONLY BE USED to indicate an error in the internal implementation
     *  of the formula engine. NEVER throw this error code for wrong user
     *  input.
     *
     * @throws {ErrorCode}
     *  The FormulaUtils.INTERNAL_ERROR error code.
     */
    FormulaUtils.throwInternal = function (msg) {
        Utils.error(msg);
        throw FormulaUtils.INTERNAL_ERROR;
    };

    /**
     * Resolves the effective recalculation mode for the passed recalculation
     * modes.
     *
     * @param {String|Null} recalc1
     *  The first recalculation mode, either 'always', 'once', or null.
     *
     * @param {String|Null} recalc2
     *  The second recalculation mode, either 'always', 'once', or null.
     *
     * @returns {String|Null}
     *  The effective recalculation mode for the passed recalculation modes.
     */
    FormulaUtils.getRecalcMode = function (recalc1, recalc2) {
        if ((recalc1 === 'always') || (recalc2 === 'always')) { return 'always'; }
        if ((recalc1 === 'once') || (recalc2 === 'once')) { return 'once'; }
        return null;
    };

    // mathematical -----------------------------------------------------------

    /**
     * Rounds the passed number to the next integer towards zero (positive
     * numbers will be rounded down, negative numbers will be rounded up).
     */
    FormulaUtils.trunc = function (number) {
        return (number < 0) ? ceil(number) : floor(number);
    };

    /**
     * Returns the sum of the passed numbers (used for callbacks).
     */
    FormulaUtils.add = function (number1, number2) {
        return number1 + number2;
    };

    /**
     * Returns the product of the passed numbers (used for callbacks).
     */
    FormulaUtils.multiply = function (number1, number2) {
        return number1 * number2;
    };

    /**
     * Returns the quotient of the passed numbers. If the divisor is zero, the
     * error code #DIV/0! will be thrown instead.
     */
    FormulaUtils.divide = function (number1, number2) {
        if (number2 === 0) { throw ErrorCode.DIV0; }
        return number1 / number2;
    };

    /**
     * Returns the floating-point modulo of the passed numbers. If the divisor
     * is zero, the error code #DIV/0! will be thrown instead. If the passed
     * numbers have different signs, the result will be adjusted according to
     * the results of other spreadsheet applications.
     */
    FormulaUtils.modulo = function (number1, number2) {
        if (number2 === 0) { throw ErrorCode.DIV0; }
        var result = number1 % number2;
        return ((result !== 0) && (number1 * number2 < 0)) ? (result + number2) : result;
    };

    /**
     * Returns the power of the passed numbers in an OOXML compliant way. If
     * both numbers are zero, the error code #NUM! will be thrown instead of
     * returning 1. If the base is zero, and the exponent is negative, the
     * error code #DIV/0! will be thrown instead of returning INF.
     */
    FormulaUtils.powerOOXML = function (base, expn) {
        // Math.pow() returns 1 for 0^0; Excel returns #NUM! error
        if ((base === 0) && (expn === 0)) { throw ErrorCode.NUM; }
        if ((base === 0) && (expn < 0)) { throw ErrorCode.DIV0; }
        return pow(base, expn);
    };

    /**
     * Returns the power of the passed numbers in an ODF compliant way. If both
     * numbers are zero, the number 1 will be returned. If the base is zero,
     * and the exponent is negative, the error code #NUM! will be thrown
     * instead of returning INF.
     */
    FormulaUtils.powerODF = function (base, expn) {
        if ((base === 0) && (expn < 0)) { throw ErrorCode.NUM; }
        return pow(base, expn);
    };

    /**
     * An object that contains resolver functions for calculating the power of
     * two numbers, mapped by file format identifiers.
     *
     * @constant
     */
    FormulaUtils.POWER = { ooxml: FormulaUtils.powerOOXML, odf: FormulaUtils.powerODF };

    /**
     * Returns the passed number raised to the power of 10.
     */
    FormulaUtils.pow10 = function (number) {
        return pow(10, number);
    };

    /**
     * Returns the decimal logarithm of the passed number.
     */
    FormulaUtils.log10 = function (number) {
        return log(number) / LN10;
    };

    /**
     * Returns the two-parameter arc tangent of the passed coordinates. If both
     * numbers are zero, the error code #DIV/0! will be thrown instead. The
     * order of the parameters 'x' and 'y' is switched compared to the native
     * method Math.atan2, according to usage in other spreadsheet applications.
     */
    FormulaUtils.arctan2 = function (x, y) {
        if ((x === 0) && (y === 0)) { throw ErrorCode.DIV0; }
        // Spreadsheet applications use (x,y); but Math.atan2() expects (y,x)
        return Math.atan2(y, x);
    };

    /**
     * Returns the hyperbolic sine of the passed number.
     */
    FormulaUtils.sinh = Math.sinh || function (number) {
        return (exp(number) - exp(-number)) / 2;
    };

    /**
     * Returns the hyperbolic cosine of the passed number.
     */
    FormulaUtils.cosh = Math.cosh || function (number) {
        return (exp(number) + exp(-number)) / 2;
    };

    /**
     * Returns the hyperbolic tangent of the passed number.
     */
    FormulaUtils.tanh = Math.tanh || function (number) {
        var pexp = exp(number), nexp = exp(-number);
        return (pexp - nexp) / (pexp + nexp);
    };

    /**
     * Returns the argument resulting in the passed hyperbolic sine.
     */
    FormulaUtils.asinh = Math.asinh || function (number) {
        return log(number + sqrt(number * number + 1));
    };

    /**
     * Returns the argument resulting in the passed hyperbolic cosine.
     */
    FormulaUtils.acosh = Math.acosh || function (number) {
        return log(number + sqrt(number * number - 1));
    };

    /**
     * Returns the argument resulting in the passed hyperbolic tangent.
     */
    FormulaUtils.atanh = Math.atanh || function (number) {
        return log((1 + number) / (1 - number)) / 2;
    };

    /**
     * Returns the argument resulting in the passed hyperbolic cotangent.
     */
    FormulaUtils.acoth = Math.acoth || function (number) {
        return log((number + 1) / (number - 1)) / 2;
    };

    /**
     * Returns the sum of the passed numbers.
     *
     * @param {Array<Numbers>} numbers
     *  The numbers to be added.
     *
     * @returns {Number}
     *  The sum of all passed numbers, or 0 if the passed array is empty.
     */
    FormulaUtils.sum = function (numbers) {
        return numbers.reduce(FormulaUtils.add, 0);
    };

    /**
     * Returns the product of the passed numbers.
     *
     * @param {Array<Numbers>} numbers
     *  The numbers to be multiplied.
     *
     * @returns {Number}
     *  The product of all passed numbers, or 1 if the passed array is empty.
     */
    FormulaUtils.product = function (numbers) {
        return numbers.reduce(FormulaUtils.multiply, 1);
    };

    /**
     * Returns the factorial of the passed number. Throws the #NUM! error code,
     * if the passed number is negative or too large.
     *
     * @param {Number} number
     *  The number to calculate the factorial for.
     *
     * @returns {Number}
     *  The factorial of the passed number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number is negative, or too large.
     */
    FormulaUtils.factorial = (function () {

        // create and return the cache for all available factorials from 0 to 170
        var getFactorials = _.once(function () {
            var FACTORIALS = [];
            var fact = 1;
            while (isFinite(fact)) {
                FACTORIALS.push(fact);
                fact *= FACTORIALS.length;
            }
            return FACTORIALS;
        });

        // public implementation, will cause to create the cached factorials on first call
        return function factorial(number) {
            var fact = getFactorials()[floor(number)];
            if (!fact) { throw ErrorCode.NUM; }
            return fact;
        };
    }());

    /**
     * Returns the binomial coefficient of the passed input values.
     *
     * @param {Number} n
     *  The upper value of the binomial coefficient (size of the set to be
     *  chosen from).
     *
     * @param {Number} k
     *  The lower value of the binomial coefficient (number of choices from the
     *  set).
     *
     * @returns {Number}
     *  The binomial coefficient, if both values are non-negative, and 'n' is
     *  greater than or equal to 'k'.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed values are invalid, or if the
     *  resulting binomial coefficient is too large.
     */
    FormulaUtils.binomial = (function () {

        // cache for different combinations of input parameters
        var BINOM_CACHE = {};
        // maximum input parameters for cache usage
        var CACHE_MAX_NUM = 50;

        // core implementation without error checking
        function coreBinomial(n, k) {

            // the resulting binomial coefficient
            var result = n - k + 1;

            // The function isFinite() is quite expensive, by using two nested loops (the inner
            // loop runs up to 10 iterations without checking the result) the entire algorithm
            // becomes about 30% to 40% faster.
            for (var i = 2; (i <= k) && isFinite(result);) {
                for (var j = 0; (j < 10) && (i <= k); i += 1, j += 1) {
                    result /= i;
                    result *= (n - k + i);
                }
            }
            return result;
        }

        // public implementation with error checking and caching
        function binomial(n, k) {

            // check input values
            n = floor(n);
            k = floor(k);
            if ((n < 0) || (k < 0) || (n < k)) { throw ErrorCode.NUM; }

            // special cases: (n 0)=1 and (n n)=1
            if ((k === 0) || (k === n)) { return 1; }
            // special cases: (n 1)=n and (n n-1)=n
            if ((k === 1) || (k === n - 1)) { return n; }

            // binomial coefficient is symmetric: (n k) = (n n-k); use lower k for performance
            if (k * 2 > n) { k = n - k; }

            // use the cache, if both numbers are in a specific limit (k<=n, see above)
            var result = 0;
            if (n <= CACHE_MAX_NUM) {
                var key = n + ',' + k;
                result = BINOM_CACHE[key];
                // result is always valid (finite) in the cachable limit
                return result ? result : (BINOM_CACHE[key] = coreBinomial(n, k));
            }

            // do not cache larger numbers, check final result (may be infinite)
            result = coreBinomial(n, k);
            if (isFinite(result)) { return result; }
            throw ErrorCode.NUM;
        }

        return binomial;
    }());

    /**
     * Returns the result of the Gauss error function.
     *
     * @param {Number} x
     *  The input value.
     *
     * @returns {Number}
     *  The result value of the Gauss error function.
     */
    FormulaUtils.erf = (function () {

        // Implementation of erf() for non-negative numbers. For all x equal to or greater than 6, the result is
        // assumed to be 1. The real result of erf() is too close to 1 to be represented with the limited
        // precision of JavaScript floating-point numbers.
        function erf_abs(x) {
            return (x >= 6) ? 1 : (x > ERF_THRESHOLD) ? (1 - erfc_cfrac(x)) : erf_taylor(x);
        }

        // Public implementation of erf(). Use the fact that erf() is an odd function: erf(-x)=-erf(x), with a
        // shortcut for the function's zero: erf(0)=0.
        function erf(x) {
            return (x > 0) ? erf_abs(x) : (x < 0) ? -erf_abs(-x) : 0;
        }

        return erf;
    }());

    /**
     * Returns the result of the complementary Gauss error function.
     *
     * @param {Number} x
     *  The input value.
     *
     * @returns {Number}
     *  The result value of the complementary Gauss error function.
     */
    FormulaUtils.erfc = function (x) {
        // Prevent arithmetic underflow for large x: erf(6) returns exactly 1 due to precision limitations of
        // floating-point numbers; but erfc(6)=1-erf(6) is close to but greater than 0.
        return (x > ERF_THRESHOLD) ? erfc_cfrac(x) : (1 - FormulaUtils.erf(x));
    };

    /**
     * Returns the linear slope of a collection of Y values.
     *
     * @param {Array<Numbers>} numbers
     *  The Y values to be processed. The array indexes of the numbers are used
     *  as the X values associated to the array elements.
     *
     * @returns {Number}
     *  The linear slope of the passed numbers.
     */
    FormulaUtils.slope = function (numbers) {
        var meanX = (numbers.length - 1) / 2;
        var meanY = FormulaUtils.sum(numbers) / numbers.length;
        var sumDYX = numbers.reduce(function (sum, y, x) { return sum + (x - meanX) * (y - meanY); }, 0);
        var sumDX2 = meanX * (meanX + 1) * (2 * meanX + 1) / 3;
        return sumDYX / sumDX2;
    };

    // number formats ---------------------------------------------------------

    /**
     * Resolves the number format for adding two formatted numbers.
     *
     * @param {ParsedFormat|Null} parsedFormat1
     *  The first parsed number format, or null to indicate an invalid number
     *  format that cannot be combined with the other number format.
     *
     * @param {ParsedFormat|Null} parsedFormat2
     *  The second parsed number format, or null to indicate an invalid number
     *  format that cannot be combined with the other number format.
     *
     * @returns {ParsedFormat|Null}
     *  One of the passed format categories to be used for the sum of two
     *  formatted numbers; or the value null, if the two format categories
     *  cannot be combined.
     */
    FormulaUtils.combineParsedFormats = (function () {

        var NUMERIC_SET = Utils.makeSet(['number', 'scientific', 'currency', 'percent', 'fraction']);
        var DATETIME_SET = Utils.makeSet(['date', 'time', 'datetime']);

        return function (parsedFormat1, parsedFormat2) {

            // skip invalid categories (null values)
            if (!parsedFormat1 || !parsedFormat2) { return null; }

            // skip standard category
            if (parsedFormat1.isStandard()) { return parsedFormat2; }
            if (parsedFormat2.isStandard()) { return parsedFormat1; }

            // separate handling for numeric categories, and date/time categories
            var isNumeric1 = parsedFormat1.category in NUMERIC_SET;
            var isNumeric2 = parsedFormat2.category in NUMERIC_SET;
            var isDateTime1 = !isNumeric1 && (parsedFormat1.category in DATETIME_SET);
            var isDateTime2 = !isNumeric2 && (parsedFormat2.category in DATETIME_SET);

            // dates and times win over numeric categories
            if (isDateTime1 && isNumeric2) { return parsedFormat1; }
            if (isNumeric1 && isDateTime2) { return parsedFormat2; }

            // first of two numeric categories wins (TODO: more specific precedences?)
            if (isNumeric1 && isNumeric2) { return parsedFormat1; }

            // anything else cannot be combined (e.g. two dates/times)
            return null;
        };
    }());

    // comparison -------------------------------------------------------------

    /**
     * Returns a numeric type identifier for the passed scalar value. The type
     * identifier represents a natural order of different data types, as used
     * in spreadsheet formulas (for example, strings are always considered to
     * be greater than numbers).
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The scalar value to be checked.
     *
     * @param {Boolean} [nullGreater=false]
     *  Whether to consider the null value to be greater than any other value.
     *  By default, the null value is less than any other value.
     *
     * @returns {Number}
     *  A numeric type identifier for the passed scalar value:
     *  - FormulaUtils.ScalarType.NUMBER for a number (including infinite
     *      numbers, and NaN).
     *  - FormulaUtils.ScalarType.STRING for a string.
     *  - FormulaUtils.ScalarType.BOOLEAN for boolean values.
     *  - FormulaUtils.ScalarType.ERROR for error codes.
     *  - FormulaUtils.ScalarType.NULL for the value null, if the option
     *      'nullGreater' has NOT been set to true.
     *  - FormulaUtils.ScalarType.NULL_MAX for the value null, if the option
     *      'nullGreater' has been set to true.
     */
    FormulaUtils.getScalarType = function (value, nullGreater) {
        switch (typeof value) {
            case 'number':  return FormulaUtils.ScalarType.NUMBER;
            case 'string':  return FormulaUtils.ScalarType.STRING;
            case 'boolean': return FormulaUtils.ScalarType.BOOLEAN;
        }
        return (value instanceof ErrorCode) ? FormulaUtils.ScalarType.ERROR :
            nullGreater ? FormulaUtils.ScalarType.NULL_MAX : FormulaUtils.ScalarType.NULL;
    };

    /**
     * Returns whether the passed number is considered to be zero. Denormalized
     * numbers (all numbers with an absolute value less than 2^-1022) will be
     * handled as zero by the formula engine.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is zero, or denormalized. returns false for
     *  positive/negative infinite, or for NaN.
     */
    FormulaUtils.isZero = function (number) {
        return abs(number) < Utils.MIN_NUMBER;
    };

    /**
     * Returns whether the passed numbers are considered to be equal, allowing
     * a slight difference for equally signed numbers. This method may be
     * faster than the method FormulaUtils.compareNumbers() in some situations.
     *
     * @param {Number} number1
     *  The first number for comparison.
     *
     * @param {Number} number2
     *  The second number for comparison.
     *
     * @returns {Boolean}
     *  Whether the passed numbers are both finite, and considered to be equal.
     */
    FormulaUtils.equalNumbers = function (number1, number2) {
        // false, if one of the numbers is not finite
        return isFinite(number1) && isFinite(number2) && equalNumbers(number1, number2);
    };

    /**
     * Returns whether the passed strings are considered to be equal. This
     * method may be faster than the method FormulaUtils.compareStrings() in
     * some situations.
     *
     * @param {String} string1
     *  The first string for comparison.
     *
     * @param {String} string2
     *  The second string for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, the strings will be compared case-sensitively.
     *
     * @returns {Boolean}
     *  Whether the passed strings are considered to be equal.
     */
    FormulaUtils.equalStrings = function (string1, string2, options) {
        var withCase = Utils.getBooleanOption(options, 'withCase', false);
        return equalStrings(string1, string2, withCase);
    };

    /**
     * Returns whether the passed scalar values are considered to be equal.
     * This method may be faster than the method FormulaUtils.compareScalars()
     * in some situations.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first scalar value for comparison.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second scalar value for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, strings will be compared case-sensitively. By
     *      default, the character case of strings will be ignored.
     *  @param {Boolean} [options.nullMode='convert']
     *      Specifies what happens, if one of the passed values is null, and
     *      the other value is not null (two null values are always equal):
     *      - 'convert' (default): The null value will be replaced with the
     *          neutral value of the data type of the other value (zero, empty
     *          string, or false), and these values will be compared. If the
     *          other value is an error code, the null value will not be
     *          converted, but will be considered different to the error code.
     *      - 'less': The null value is different to the other value.
     *      - 'greater': The null value is different to the other value.
     *      The options 'less' and 'greater' are supported for consistency with
     *      the method FormulaUtils.compareScalars().
     *
     * @returns {Boolean}
     *  Whether the passed scalar values are considered to be equal, according
     *  to the passed options.
     */
    FormulaUtils.equalScalars = function (value1, value2, options) {

        // whether to convert a null value to the neutral value of the other parameter
        var convertNull = Utils.getStringOption(options, 'nullMode', 'convert') === 'convert';

        // convert null values if specified
        if (convertNull) {
            if (value1 === null) { value1 = getNeutralValue(value2); }
            if (value2 === null) { value2 = getNeutralValue(value1); }
        }

        // shortcut: simple check for strict equality
        if (value1 === value2) { return true; }

        // get type identifiers (values of different types are never equal)
        var type1 = FormulaUtils.getScalarType(value1);
        var type2 = FormulaUtils.getScalarType(value2);
        if (type1 !== type2) { return false; }

        // compare numbers and strings using the helper methods
        switch (type1) {
            case FormulaUtils.ScalarType.NUMBER: return FormulaUtils.equalNumbers(value1, value2);
            case FormulaUtils.ScalarType.STRING: return FormulaUtils.equalStrings(value1, value2, options);
        }

        // equal booleans, error codes, and null values would have been found above already
        return false;
    };

    /**
     * Compares the passed numbers, and returns a signed indicator number,
     * allowing a slight difference for equally signed numbers.
     *
     * @param {Number} number1
     *  The first number for comparison.
     *
     * @param {Number} number2
     *  The second number for comparison.
     *
     * @returns {Number}
     *  The number -1, if number1 is less than number2; or the number 1, if
     *  number1 is greater than number2; or the number 0, if both numbers are
     *  equal. If the difference of the numbers is less than Utils.MIN_NUMBER
     *  (e.g., due to rounding errors), the numbers are also considered to be
     *  equal. If any of the numbers is NaN or infinite, the result will be the
     *  special value NaN.
     */
    FormulaUtils.compareNumbers = function (number1, number2) {

        // return NaN for any invalid numbers
        if (!isFinite(number1) || !isFinite(number2)) { return Number.NaN; }

        // return comparison specifier for the (finite) numbers
        return equalNumbers(number1, number2) ? 0 : (number1 < number2) ? -1 : 1;
    };

    /**
     * Compares the passed strings lexicographically, and returns a signed
     * indicator number.
     *
     * @param {Boolean} string1
     *  The first string for comparison.
     *
     * @param {Boolean} string2
     *  The second string for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, the strings will be compared case-sensitively.
     *      Capital letters will be considered greater than their lower-case
     *      variants. By default, the character case of the strings will be
     *      ignored.
     *
     * @returns {Number}
     *  The number -1, if string1 is lexicographically less than string2; or
     *  the number 1, if string1 is lexicographically greater than string2;
     *  otherwise the number 0 (the strings are equal).
     */
    FormulaUtils.compareStrings = function (string1, string2, options) {

        // compare the strings regarding the character case
        if (Utils.getBooleanOption(options, 'withCase', false)) {

            // Quick check if the longer string starts exactly with the shorter string. In this case,
            // the longer string is considered greater than the shorter string. This includes the
            // special case that both strings are exactly equal.
            var minLen = Math.min(string1.length, string2.length);
            if (string1.substr(0, minLen) === string2.substr(0, minLen)) {
                return compareIntegers(string1.length, string2.length);
            }

            // The strings differ in at least one character, check the single characters in a loop.
            for (var i = 0; i < minLen; i += 1) {

                // extract the characters at the current position; continue with next character,
                // if both characters are equal
                var c1 = string1[i], c2 = string2[i];
                if (c1 === c2) { continue; }

                // if the upper-case versions of the characters are different, just compare them
                var u1 = c1.toUpperCase(), u2 = c2.toUpperCase();
                if (u1 < u2) { return -1; }
                if (u1 > u2) { return 1; }

                // the first character is less than than second, if and only if the first character
                // is lower-case, AND the second character is upper-case
                return ((c1 !== u1) && (c2 === u2)) ? -1 : 1;
            }

            // The loop should have found a result.
            Utils.error('FormulaUtils.compareStrings(): unexpected end of strings');
            return 0;
        }

        // compare the strings case-insensitively
        string1 = string1.toUpperCase();
        string2 = string2.toUpperCase();
        return (string1 < string2) ? -1 : (string1 > string2) ? 1 : 0;
    };

    /**
     * Compares the passed boolean values, and returns a signed indicator
     * number.
     *
     * @param {Boolean} bool1
     *  The first boolean value for comparison.
     *
     * @param {Boolean} bool2
     *  The second boolean value for comparison.
     *
     * @returns {Number}
     *  The number -1, if bool1 is FALSE, and bool2 is TRUE (bool1 is 'less
     *  than' bool2); or the number 1, if bool1 is TRUE, and bool2 is FALSE
     *  (bool1 is 'greater than' bool2); otherwise the number 0 (the boolean
     *  values are equal).
     */
    FormulaUtils.compareBooleans = function (bool1, bool2) {
        // subtraction operator converts booleans to numbers implicitly
        return bool1 - bool2;
    };

    /**
     * Compares the internal indexes of built-in error codes, and returns a
     * signed indicator number.
     *
     * @param {ErrorCode} errorCode1
     *  The first error code for comparison.
     *
     * @param {ErrorCode} errorCode2
     *  The second error code for comparison.
     *
     * @returns {Number}
     *  The value NaN, if any of the passed error codes does not represent a
     *  built-in error (the property 'num' of the error code is not finite);
     *  the number -1, if the index of the first error code is less than the
     *  index of the second error code; the number 1, if the index of the first
     *  error code is greater than the index of the second error code, or the
     *  number 0, if both error codes have equal indexes.
     */
    FormulaUtils.compareErrorCodes = function (errorCode1, errorCode2) {
        return isFinite(errorCode1.num) && isFinite(errorCode2.num) && compareIntegers(errorCode1.num, errorCode2.num);
    };

    /**
     * Compares the passed scalar values, and returns a signed indicator
     * number.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first scalar value for comparison.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second scalar value for comparison.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.withCase=false]
     *      If set to true, strings will be compared case-sensitively. Capital
     *      letters will be considered greater than their lower-case variants.
     *      By default, the character case of strings will be ignored.
     *  @param {Boolean} [options.nullMode='convert']
     *      Specifies what happens, if one of the passed values is null, and
     *      the other value is not null (two null values are always equal):
     *      - 'convert' (default): The null value will be replaced with the
     *          neutral value of the data type of the other value (zero, empty
     *          string, or false), and these values will be compared. If the
     *          other value is an error code, the null value will not be
     *          converted, but will be considered less than the error code.
     *      - 'less': The null value is less than the other value.
     *      - 'greater': The null value is greater than the other value.
     *
     * @returns {Number}
     *  The number -1, if value1 is considered less than value2; or the number
     *  1, if value1 is considered greater than value2; otherwise the number 0
     *  (the values are equal). Numbers are always less than strings, strings
     *  are always less than boolean values, and boolean values are always less
     *  than error codes. The null value will be treated according to the
     *  option 'nullMode'. Values of the same type will be compared as
     *  described for the methods FormulaUtils.compareNumbers(),
     *  FormulaUtils.compareStrings(), FormulaUtils.compareBooleans(), and
     *  FormulaUtils.compareErrorCodes().
     */
    FormulaUtils.compareScalars = function (value1, value2, options) {

        // behavior for null values
        var nullMode = Utils.getStringOption(options, 'nullMode', 'convert');
        // whether the null value is considered greater than any other value (not for 'convert' mode)
        var nullGreater = nullMode === 'greater';

        // convert null values if specified
        if (nullMode === 'convert') {
            if (value1 === null) { value1 = getNeutralValue(value2); }
            if (value2 === null) { value2 = getNeutralValue(value1); }
        }

        // get type identifiers, compare different types (ignore the actual values)
        var type1 = FormulaUtils.getScalarType(value1, nullGreater);
        var type2 = FormulaUtils.getScalarType(value2, nullGreater);
        if (type1 !== type2) { return compareIntegers(type1, type2); }

        // compare values of equal types
        switch (type1) {
            case FormulaUtils.ScalarType.NUMBER:  return FormulaUtils.compareNumbers(value1, value2);
            case FormulaUtils.ScalarType.STRING:  return FormulaUtils.compareStrings(value1, value2, options);
            case FormulaUtils.ScalarType.BOOLEAN: return FormulaUtils.compareBooleans(value1, value2);
            case FormulaUtils.ScalarType.ERROR:   return FormulaUtils.compareErrorCodes(value1, value2);
        }

        // both values are null
        return 0;
    };

    // matrixes ---------------------------------------------------------------

    /**
     * Returns whether the passed matrix size is inside the limits supported by
     * the formula interpreter.
     *
     * @param {Number} rows
     *  The number of rows in a matrix.
     *
     * @param {Number} cols
     *  The number of columns in a matrix.
     *
     * @returns {Boolean}
     *  Whether the passed matrix size is inside the limits supported by the
     *  formula interpreter.
     */
    FormulaUtils.isValidMatrixSize = function (rows, cols) {
        return (rows > 0) && (rows <= FormulaUtils.MAX_MATRIX_ROW_COUNT) && (cols > 0) && (cols <= FormulaUtils.MAX_MATRIX_COL_COUNT);
    };

    /**
     * Throws an UNSUPPORTED error code, if the passed matrix size is outside
     * the limits supported by the formula interpreter.
     *
     * @param {Number} rows
     *  The number of rows in a matrix.
     *
     * @param {Number} cols
     *  The number of columns in a matrix.
     *
     * @throws {ErrorCode}
     *  The UNSUPPORTED error code, if the passed matrix size is not inside the
     *  limits supported by the formula interpreter.
     */
    FormulaUtils.ensureMatrixSize = function (rows, cols) {
        if (!FormulaUtils.isValidMatrixSize(rows, cols)) {
            throw FormulaUtils.UNSUPPORTED_ERROR;
        }
    };

    // filter aggregation -----------------------------------------------------

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all values matching a filter criterion to the result of the
     * function (COUNTIF, SUMIF, AVERAGEIF). See description of the method
     * FormulaContext.aggregateFiltered() for details about the parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor.
     */
    FormulaUtils.implementFilterAggregation = function (initial, aggregate, finalize, options) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function filterAggregator(sourceRanges, criterion, dataRanges) {

            // convert source ranges operand to a single range (including a circular reference check)
            var sourceRange = this.convertToValueRange(sourceRanges);
            // create an array of criterion descriptors, as expected by FormulaContext.aggregateFiltered()
            var criteriaDescs = [{ range: sourceRange, criterion: criterion }];
            // the data range (no data ranges passed: aggregate source range, or count matching cells)
            var dataRange = !aggregate ? null : dataRanges ? this.convertToValueRange(dataRanges) : sourceRange.clone();

            // ignore original size of the data range, set it to the size of the source range
            if (dataRange) {
                dataRange.end[0] = Math.min(dataRange.start[0] + sourceRange.cols() - 1, this.docModel.getMaxCol());
                dataRange.end[1] = Math.min(dataRange.start[1] + sourceRange.rows() - 1, this.docModel.getMaxRow());
                // shorten the source range, if the data range has been shortened
                sourceRange.end[0] = sourceRange.start[0] + dataRange.cols() - 1;
                sourceRange.end[1] = sourceRange.start[1] + dataRange.rows() - 1;
            }

            // aggregate the data ranges, or count the matching cells (COUNTIF)
            return this.aggregateFiltered(criteriaDescs, dataRange, initial, aggregate, finalize, options);
        };
    };

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all values matching multiple filter criteria to the result of
     * the function (COUNTIFS, SUMIFS, AVERAGEIFS). See description of the
     * method FormulaContext.aggregateFiltered() for more details about the
     * parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor.
     */
    FormulaUtils.implementParallelFilterAggregation = function (initial, aggregate, finalize, options) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function parallelFilterAggregator() {

            // the data ranges to be aggregated (no aggregation callback: count matching cells)
            var dataRange = aggregate ? this.convertToValueRange(arguments[0]) : null;

            // the criteria operands (pairs of range/criterion)
            var criteriaArgs = _.toArray(arguments);
            if (dataRange) { criteriaArgs = criteriaArgs.slice(1); }
            if (criteriaArgs.length % 2) { FormulaUtils.throwInternal('invalid number of operands'); }

            // convert to an array of criterion descriptors, as expected by FormulaContext.aggregateFiltered()
            var criteriaDescs = _.times(criteriaArgs.length / 2, function (index) {
                return {
                    range: this.convertToValueRange(criteriaArgs[2 * index]),
                    criterion: criteriaArgs[2 * index + 1]
                };
            }, this);

            // aggregate the data ranges, or count the matching cells (COUNTIFS)
            return this.aggregateFiltered(criteriaDescs, dataRange, initial, aggregate, finalize, options);
        };
    };

    // debug ------------------------------------------------------------------

    /**
     * Returns the string representation of the passed scalar value for debug
     * logging.
     *
     * @attention
     *  DO NOT use this method for GUI related code where correct locale
     *  dependent representation of the passed value is required.
     *
     * @param {Any} value
     *  The value to be converted to the string representation.
     *
     * @returns {String}
     *  The string representation of the value.
     */
    FormulaUtils.valueToString = function (value) {
        return (typeof value === 'string') ? ('"' + value.replace(/"/g, '""') + '"') :
            (value instanceof Date) ? value.toISOString() :
            (value === null) ? 'null' :
            (value === true) ? 'TRUE' :
            (value === false) ? 'FALSE' :
            value.toString();
    };

    /**
     * Writes all passed tokens to the browser console.
     *
     * @param {String} message
     *  A message string written before the formula tokens.
     *
     * @param {Array} tokens
     *  An array of formula tokens to be written to the browser console. The
     *  array elements can be of any type convertible to a string.
     */
    FormulaUtils.logTokens = FormulaUtils.isLoggingActive() ? function (message, tokens) {
        FormulaUtils.log(message + ': ' + tokens.join(' '));
    } : $.noop;

    // exports ================================================================

    return FormulaUtils;

});
