/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/formulainterpreter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/presetformattable',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/utils/formulaerror',
    'io.ox/office/spreadsheet/model/formula/utils/complex',
    'io.ox/office/spreadsheet/model/formula/utils/matrix',
    'io.ox/office/spreadsheet/model/formula/utils/operand'
], function (Utils, BaseObject, ModelObject, SheetUtils, PresetFormatTable, FormulaUtils, FormulaContext, FormulaError, Complex, Matrix, Operand) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;

    // class InterpreterInstance ==============================================

    /**
     * Resolves a compiled formula token array to the result of the formula.
     * This implementation has been moved outside the class FormulaInterpreter
     * to be able to recursively resolve multiple formulas (e.g. defined
     * names).
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {CompilerNode} rootNode
     *  The root node of the compiled token tree, as returned by the method
     *  FormulaCompiler.compileTokens().
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the method
     *  FormulaInterpreter.interpretTokens().
     */
    var InterpreterInstance = BaseObject.extend({ constructor: function (docModel, rootNode, initOptions) {

        // the number formatter of the document
        var numberFormatter = docModel.getNumberFormatter();

        // the reference sheet
        var refSheet = Utils.getIntegerOption(initOptions, 'refSheet', null);

        // the target reference address
        var targetAddress = Utils.getOption(initOptions, 'targetAddress', Address.A1);

        // whether to recalculate all dirty cell formulas resolved by cell references
        var recalcDirty = Utils.getOption(initOptions, 'recalcDirty', false);

        // the calling context for the operator resolver callback function
        var context = new FormulaContext(docModel, initOptions);

        // the standard number format
        var STANDARD_FORMAT = numberFormatter.getParsedFormat(0);

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private class OperandResolver --------------------------------------

        function OperandResolver() {
            this._operandArray = [];
        } // class OperandResolver

        OperandResolver.prototype.push = function (tokenNode, contextType) {
            this._operandArray.push({ tokenNode: tokenNode, contextType: contextType });
        };

        OperandResolver.prototype.size = function () {
            return this._operandArray.length;
        };

        OperandResolver.prototype.get = function (index) {
            var operandData = this._operandArray[index];
            return (operandData && operandData.operand) || null;
        };

        OperandResolver.prototype.resolve = function (index) {
            var operandData = this._operandArray[index];
            if (!operandData) { return null; }
            return operandData.operand || (operandData.operand = resolveNode(operandData.tokenNode, operandData.contextType));
        };

        // private methods ----------------------------------------------------

        /**
         * Resolves the passed reference token to its result.
         *
         * @param {ReferenceToken} refToken
         *  The formula token representing a cell range reference. If the token
         *  cannot be resolved to a cell range address, this method returns the
         *  #REF! error code.
         *
         * @returns {Range3DArray|ErrorCode}
         *  The cell range addresses contained in the reference, or a #REF!
         *  error code.
         */
        function resolveReference(refToken) {
            var range = refToken.getRange3D(initOptions);
            return range ? new Range3DArray(range) : ErrorCode.REF;
        }

        /**
         * Resolves the passed name token to its result.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {NameToken} nameToken
         *  The formula token representing a defined name. If the name does not
         *  exist in the document, this method returns the #NAME? error code.
         *
         * @returns {Any}
         *  The result value of the defined name (may be values, matrixes, or
         *  references).
         */
        function resolveDefinedName(contextType, nameToken) {

            // resolve the model of a defined name from the token (unknown names result in the #NAME? error code)
            var nameModel = nameToken.resolveNameModel(refSheet);
            if (!nameModel) { return ErrorCode.NAME; }

            // resolve the formula in the defined name (defined names are always relative to cell A1)
            var options = { refSheet: refSheet, refAddress: Address.A1, targetAddress: targetAddress, recalcDirty: recalcDirty };
            return nameModel.getTokenArray().interpretFormula(contextType, options).value;
        }

        /**
         * Resolves the passed table token to its result.
         *
         * @param {TableToken} tableToken
         *  The formula token representing a structured table reference. If the
         *  token cannot be resolved to a range, this method returns the #REF!
         *  error code.
         *
         * @returns {Range3DArray|ErrorCode}
         *  The resulting range referred by thie passed table token, or the
         *  #REF! error code.
         */
        function resolveTableRange(tableToken) {
            var range = tableToken.getRange3D(targetAddress);
            return range ? new Range3DArray(range) : ErrorCode.REF;
        }

        /**
         * Checks that the passed return type, and the context type of the
         * formula or subexpression match. Throws a 'reference' exception, if
         * the context type is 'ref', and the passed type is different from
         * 'ref' or 'any'. Excel rejects such formulas, and even announces a
         * broken file format if such formulas are written into the file.
         * Examples of such invalid formulas are:
         *  =A1:1 (combined number with range operator)
         *  =OFFSET(1,1,1) (passed number to a reference-only parameter)
         *
         * It is possible to use operands and functions with return type
         * 'any' in reference context, e.g.
         *  =A1:name (return type of defined names is 'any')
         *  =A1:IF(TRUE,A1,B1) (return type of IF is 'any').
         * If the defined name or function returns a non-reference value, the
         * formula simply results in the #VALUE! error.
         *
         * @param {String} type
         *  The return type of an operator or function, or the type of a single
         *  operand to be checked. Must be one of:
         *  - 'val': A scalar value (numbers, dates, strings, boolean values,
         *      complex numbers, or error codes).
         *  - 'mat': Constant matrixes, or functions returning matrixes.
         *  - 'ref': Cell references (constant references, reference operators,
         *      functions returning a reference, or the #REF! error code).
         *  - 'any': Any of the above (defined names, a few functions returning
         *      any data type).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens() for
         *  details. If the context type is 'ref', only the return types 'ref'
         *  and 'any' will be accepted. All other return types will cause to
         *  throw an internal 'reference' exception that marks the formula
         *  structure to be ill-formed (see description above).
         *
         * @throws {FormulaError}
         *  The special 'reference' exception, if the passed return type is not
         *  supported in reference context.
         */
        function checkContextType(type, contextType) {
            // context type 'ref': accept 'ref' and 'any' only
            if ((contextType === 'ref') && (type !== 'ref') && (type !== 'any')) {
                throw new FormulaError('expected reference token', 'reference');
            }
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'val', i.e. whether it is a scalar value.
         */
        function isValType(value) {
            if ((value === null) || (value instanceof Date) || (value instanceof Complex) || (value instanceof ErrorCode)) { return true; }
            var type = typeof value;
            return (type === 'number') || (type === 'string') || (type === 'boolean');
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'mat', i.e. whether it is a matrix.
         */
        function isMatType(value) {
            return value instanceof Matrix;
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'ref', i.e. whether it is a single cell range address, or an array
         * of cell range addresses.
         */
        function isRefType(value) {
            return (value instanceof Range3D) || (value instanceof Range3DArray);
        }

        /**
         * Creates an operand object for the passed parser token.
         *
         * @param {CompilerNode} operandNode
         *  The compiler token representing an operand (scalar values, constant
         *  matrixes, cell references, defined names).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the passed token.
         *
         * @throws {ErrorCode}
         *  The special error code UNSUPPORTED_ERROR, if the type of the passed
         *  token is not supported.
         */
        function makeTokenOperand(operandNode, contextType) {

            // the parser token representing the operand value
            var token = operandNode.token;
            // the token type
            var type = null;
            // the value to be inserted into the operand
            var value = null;

            FormulaUtils.log('> resolving operand token: ' + token);

            switch (token.getType()) {
                case 'lit': // scalar literals
                    value = token.getValue();
                    type = (value === ErrorCode.REF) ? 'ref' : 'val';
                    break;
                case 'mat': // matrix literals
                    value = token.getMatrix();
                    type = 'mat';
                    break;
                case 'ref': // cell range references
                    value = resolveReference(token);
                    type = 'ref';
                    break;
                case 'name': // defined names
                    value = resolveDefinedName(contextType, token);
                    type = 'any';
                    break;
                case 'table': // table ranges
                    value = resolveTableRange(token);
                    type = 'ref';
                    break;
                default:
                    FormulaUtils.throwInternal('InterpreterInstance.makeTokenOperand(): unknown token type: "' + token.getType() + '"');
            }

            // check context type (will throw 'reference' exception)
            checkContextType(type, contextType);
            return context.createOperand(value);
        }

        /**
         * Converts a number format category to a parsed number format intended
         * to be returned with the result of an operator or function.
         */
        function getParsedFormat(resolver, category) {

            // resolve the number format from all operands, if 'combine' is specified
            if (category === 'combine') {
                var parsedFormat = STANDARD_FORMAT;
                for (var index = 0; parsedFormat && (index < resolver.size()); index += 1) {
                    var operand = resolver.get(index);
                    var opFormat = (operand && operand.getFormat()) || STANDARD_FORMAT;
                    parsedFormat = FormulaUtils.combineParsedFormats(parsedFormat, opFormat);
                }
                return parsedFormat;
            }

            // convert format category to parsed number format
            var format = null;
            switch (category) {
                case 'percent':
                    format = PresetFormatTable.getPercentId(true); // no decimal places
                    break;
                case 'currency':
                    format = numberFormatter.getCurrencyCode({ red: true });
                    break;
                case 'fraction':
                    format = PresetFormatTable.getFractionId(true); // two-digit fractions
                    break;
                case 'date':
                    format = PresetFormatTable.SYSTEM_DATE_ID;
                    break;
                case 'time':
                    format = PresetFormatTable.getTimeId(null, true); // with seconds
                    break;
                case 'datetime':
                    format = PresetFormatTable.SYSTEM_DATETIME_ID;
                    break;
            }
            return (format === null) ? null : numberFormatter.getParsedFormat(format);
        }

        /**
         * Resolves an operator or function according to the operands and type
         * signature.
         */
        function resolveOperator(descriptor, signature, resolver, row, col) {

            // the resulting operand
            var result = null;

            try {

                // register the operator resolver with the current operands at the formula context, to make the
                // operands accessible to function implementations via public methods of the context
                context.pushOperandResolver(resolver, row, col);

                // resolves the operand values according to the signature (except lazy function parameters)
                var params = signature.map(function (typeData, index) {
                    if (typeData.typeSpec === 'any:lazy') { return null; }
                    return context.getOperand(index, typeData.typeSpec);
                });

                // invoke the resolver callback function of the operator
                FormulaUtils.logTokens('\xa0 resolving with parameters', params);
                result = descriptor.resolve.apply(context, params);

                // function implementations are allowed to return Operand instances for any type
                var value = (result instanceof Operand) ? result.getRawValue() : result;

                // compare result with the specified return type
                var validType = false;
                switch (descriptor.type) {
                    case 'val': validType = isValType(value); break;
                    case 'mat': validType = isMatType(value); break;
                    case 'ref': validType = isRefType(value); break;
                    case 'any': validType = isValType(value) || isMatType(value) || isRefType(value); break;
                    default: FormulaUtils.throwInternal('InterpreterInstance.resolveOperator(): unknown type identifier: "' + descriptor.type + '"');
                }

                // exit immediately if the result does not match the specified type
                if (!validType) {
                    FormulaUtils.throwInternal('InterpreterInstance.resolveOperator(): ' + descriptor.name + ': operator result type does not match operator return type');
                }

                // check maximum evaluation time after each operator to prevent freezing browser
                context.checkEvalTime();

            } catch (error) {
                if (error instanceof ErrorCode) {
                    // operator may throw error codes
                    result = error;
                } else {
                    // re-throw other error codes, or internal JavaScript exceptions
                    throw error;
                }

            } finally {

                // remove the operand resolver from the internal stack of the formula context
                context.popOperandResolver();
            }

            return result;
        }

        /**
         * Processes the passed operator or function call. Calculates the
         * required number of operands from the following compiler tokens, and
         * calculates the result of the operator or function.
         *
         * @param {CompilerNode} subNode
         *  The root node of a compiler token sub-tree representing an operator
         *  or function.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the result of the operator or
         *  function call.
         *
         * @throws {String}
         *  The error code 'missing', if there are not enough tokens available
         *  in the compiled token array.
         */
        function processOperator(subNode, contextType) {

            // immediately throw compiler errors without any further processing
            if (subNode.error) { throw subNode.error; }

            // the parser token representing the operator or function
            var token = subNode.token;
            // the operator/function specification
            var descriptor = subNode.descriptor;
            // the expanded type signature
            var signature = subNode.signature;
            // the operand resolver passed to the formula context
            var resolver = new OperandResolver();
            // matrix size for repeated processing of value operators
            var repeatRows = 0;
            var repeatCols = 0;
            // the resulting operand
            var result = null;

            FormulaUtils.log('> processing operator ' + token + ' with ' + subNode.operands.length + ' parameters');

            // return #NAME? for unknown functions (external functions, macro calls)
            if (!descriptor) { return context.createOperand(ErrorCode.NAME); }

            // check context type (will throw 'reference' exception)
            checkContextType(descriptor.type, contextType);

            // check missing implementation
            if (!_.isFunction(descriptor.resolve)) {
                FormulaUtils.warn('InterpreterInstance.processOperator(): unsupported operator or function "' + token + '"');
                return context.createOperand(FormulaUtils.UNSUPPORTED_ERROR);
            }

            // build the array of operands
            signature.forEach(function (typeData, index) {

                // the resulting context type for the operand
                var opContextType = typeData.baseType;

                // if context type for this operator is matrix, and the current parameter is of type value,
                // and this operator returns a value, pass the matrix context type through to the operand
                // (resolve nested cell references to matrixes instead of single values)
                if ((contextType === 'mat') && (typeData.baseType === 'val') && (descriptor.type === 'val')) {
                    opContextType = 'mat';
                }

                resolver.push(subNode.operands[index], opContextType);
            });

            // special handling for all operands of operators returning values
            if (descriptor.type === 'val') {
                signature.forEach(function (typeData, index) {

                    // convert or preprocess operands, if operator expects a value type
                    if (typeData.baseType === 'val') {

                        // let the operand resolver calculate the resulting operand
                        var operand = resolver.resolve(index);

                        // If outer operator context is matrix, convert cell references to matrixes.
                        // Example: In the formula =MMULT(A1:B2;ABS(A1:B2)), the function MMULT passes
                        // context type matrix to its operands which causes ABS to resolve the cell
                        // reference as matrix instead of value.
                        if ((contextType === 'mat') && operand.isReference()) {
                            operand = resolver._operandArray[index].operand = context.createOperand(operand.getMatrix());
                        }

                        // Check for matrixes in value operands. Resolve operator repeatedly to a matrix,
                        // if it returns values. Example: The plus operator in the formula =SUM({1;2|3;4}+1)
                        // will convert the number 1 to the matrix {1;1|1;1}.
                        if (operand.isMatrix()) {
                            var matrix = operand.getMatrix();
                            repeatRows = Math.max(repeatRows, matrix.rows());
                            repeatCols = Math.max(repeatCols, matrix.cols());
                        }
                    }
                });
            }

            // resolve as matrix, if any parameter with value signature type is a matrix
            if (repeatRows * repeatCols > 0) {

                // restrict to supported matrix size
                if (!FormulaUtils.isValidMatrixSize(repeatRows, repeatCols)) {
                    return context.createOperand(FormulaUtils.UNSUPPORTED_ERROR);
                }

                // build a matrix as result value
                FormulaUtils.info('\xa0 process as ' + repeatRows + 'x' + repeatCols + ' matrix');
                result = Matrix.generate(repeatRows, repeatCols, function (row, col) {
                    return resolveOperator(descriptor, signature, resolver, row, col);
                });

            } else {
                result = resolveOperator(descriptor, signature, resolver, 0, 0);
            }

            // resolve the number format category of the descriptor to a parsed number format
            var parsedFormat = descriptor.format ? getParsedFormat(resolver, descriptor.format) : null;

            // always create and return an instance of class Operand (if 'result' is an operand, it will be copied)
            return context.createOperand(result, { format: parsedFormat });
        }

        /**
         * Returns the result of the specified compiler token. If it is an
         * operand, it will be returned. If it is an operator or function call,
         * its result will be calculated and returned. If the operands of the
         * operator or functions are also operators or functions, their results
         * will be calculated recursively.
         *
         * @param {CompilerNode} tokenNode
         *  A node from a compiled token tree to be processed.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the specified compiler token.
         */
        function resolveNode(tokenNode, contextType) {
            var result = tokenNode.isOperator() ? processOperator(tokenNode, contextType) : makeTokenOperand(tokenNode, contextType);
            FormulaUtils.log('< result of ' + tokenNode + ': ' + result);
            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the token array
         * passed in the constructor.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the formula, as unresolved operand object.
         */
        this.getResult = function (contextType) {
            return resolveNode(rootNode, contextType);
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = rootNode = initOptions = numberFormatter = context = null;
        });

    } }); // class InterpreterInstance

    // class FormulaInterpreter ===============================================

    /**
     * Calculates the result for a compiled tree of compiler token nodes. See
     * class FormulaCompiler for details about compiled token trees.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var FormulaInterpreter = ModelObject.extend({ constructor: function (docModel) {

        // file-format dependent options for value comparison
        var COMPARE_OPTIONS = {
            withCase: docModel.getApp().getFileFormat() === 'odf',
            nullMode: 'convert'
        };

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Compares the passed scalar values according to the current file
         * format. For ODF documents, the character case of strings matters.
         *
         * @param {Number|String|Boolean|Null} value1
         *  The first value to be compared.
         *
         * @param {Number|String|Boolean|Null} value2
         *  The second value to be compared.
         *
         * @returns {Number}
         *  A negative value, if value1 is less than value2; or a positive
         *  value, if value1 is greater than value2; or zero, if both values
         *  are of the same type and are equal. See static function
         *  FormulaUtilscompareScalars() for more details.
         */
        this.compareScalars = function (value1, value2) {
            return FormulaUtils.compareScalars(value1, value2, COMPARE_OPTIONS);
        };

        /**
         * Returns an interpreter result object that represents the passed
         * result value of a formula.
         *
         * @param {Number|String|Boolean|ErrorCode|Matrix|Range3DArray} value
         *  The result value to be converted to the interpreter result object.
         *  If this value is one of the internal error codes defined in the
         *  module FormulaUtils, this method will return a warning result
         *  object, otherwise a regular valid result object.
         *
         * @param {Operand} [operand]
         *  A formula operand containing additional settings to be inserted
         *  into the result object (a parsed number format, and the URL of a
         *  hyperlink, associated to the result value).
         *
         * @returns {Object}
         *  An interpreter result object, containing the following properties:
         *  - {String} type
         *      Set to the value 'valid' for a valid formula result; or to the
         *      value 'warn', if an internal error code has been passed.
         *  - {String} [code]
         *      The internal key of an internal error (type 'warn'). Will be
         *      omitted for a valid formula result (type 'valid').
         *  - {Any} value
         *      The value that has been passed to this method (type 'valid'),
         *      or the fall-back value of an internal error code (type 'warn').
         *  - {ParsedFormat} [format]
         *      A parsed number format associated with the result value. The
         *      property will be omitted for generic unformatted formula
         *      results, or for internal error codes (type 'warn').
         *  - {String} [url]
         *      The URL of a hyperlink associated to the formula result, as
         *      provided by the operand passed to this method. The property
         *      will be omitted for an internal error code (type 'warn').
         */
        this.createResult = function (value, operand) {

            // create a special result object for internal error codes
            if ((value instanceof ErrorCode) && value.internal) {
                return { type: 'warn', code: value.key, value: value.value };
            }

            // resolve the parsed number format, and the URL of a hyperlink
            var format = operand ? operand.getFormat() : null;
            var url = operand ? operand.getURL() : null;

            // create the result object with optional properties
            var result = { type: 'valid', value: value };
            if (format !== null) { result.format = format; }
            if (url !== null) { result.url = url; }
            return result;
        };

        /**
         * Returns an interpreter result object that represents a fatal error.
         *
         * @param {String} code
         *  The error code to be inserted into the result, e.g. 'missing' or
         *  'unexpected'.
         *
         * @returns {Object}
         *  An interpreter result object, containing the property 'type' set to
         *  'error'; the property 'code' set to the passed error code, and the
         *  property 'value' set to the formula error code #N/A.
         */
        this.createErrorResult = function (code) {
            return { type: 'error', code: code, value: ErrorCode.NA };
        };

        /**
         * Calculates the result of the formula represented by the passed
         * compiled token array.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. The following context types are supported:
         *  - 'val': A single value is expected, e.g. in simple cell formulas,
         *      or operators and functions working with simple values (plus
         *      operator, ABS).
         *  - 'mat': A matrix of values is expected, e.g. in matrix formulas,
         *      or functions working on entire matrixes (MMULT).
         *  - 'ref': An unresolved cell reference is expected, e.g. reference
         *      operators (list, intersection, range), or functions calculating
         *      with range addresses instead of the cell contents (OFFSET).
         *  - 'any': Accepts any of the types mentioned above, with minimum
         *      conversion, e.g. the result of defined names, functions passing
         *      one of their original parameters through (IF, CHOOSE), or
         *      functions that iterate over available values in matrixes and
         *      cell references (SUM, PRODUCT).
         *
         * @param {CompilerNode} rootNode
         *  The root node of the compiled token tree, as returned by the method
         *  FormulaCompiler.compileTokens().
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the reference sheet that will be used to resolve
         *      reference tokens and defined names without explicit sheet
         *      reference. If omitted, reference tokens without sheet indexes
         *      will result in #REF! errors.
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated ranges that are located outside the
         *      sheet will be wrapped at the sheet borders.
         *  @param {Boolean} [options.detectCircular=false]
         *      If set to true, trying to dereference a cell reference covering
         *      the target address leads to a circular dependeny error.
         *  @param {Address} [options.recalcDirty=false]
         *      If set to true, all dirty cell formulas referred by the
         *      reference tokens in the passed compiled token tree will be
         *      recalculated recursively using the dependency manager of the
         *      document.
         *
         * @returns {Object}
         *  The result descriptor of the formula interpreter, with the
         *  following properties:
         *  - {String} type
         *      The result type:
         *      - 'valid': A valid result has been calculated. The property
         *          'value' contains the formula result, the property 'format'
         *          may contain a number format category associated to the
         *          result value.
         *      - 'warn': The formula structure is valid, but a formula result
         *          cannnot be calculated. The formula may be inserted into the
         *          document model though. See property 'code' for details.
         *      - 'error': The formula structure is invalid. The formula
         *          expression MUST NOT be inserted into the document model.
         *          See property 'code' for details.
         *  - {String} [code]
         *      A specific code for a warning or fatal error:
         *      - 'circular': A warning that the formula results in a circular
         *          reference (the formula wants to interpret itself again
         *          during calculation, e.g. directly or indirectly via cell
         *          references, or with defined names referring each other).
         *      - 'unsupported': A warning that an unsupported feature,
         *          e.g. a function that has not been implemented yet) was
         *          found in the formula.
         *      - 'internal': A warning that an undetermined internal error has
         *          occurred while interpreting the formula tokens.
         *      - 'missing': A fatal error indicating that something is missing
         *          in the formula structure, e.g. a function was called with
         *          less arguments than required.
         *      - 'unexpected': A fatal error indicating that something in the
         *          formula structure was not expected, e.g. a function was
         *          called with too many arguments.
         *  - {Any} value
         *      The result of the formula. Will be an appropriate error code
         *      (an instance of class ErrorCode) if calculation resulted in a
         *      warning or a fatal error. See property 'type' for details.
         *  - {Number|String} [format]
         *      The identifier of a number format as integer, or a format code
         *      as string, associated with the result value. The property will
         *      be omitted for generic formula results, or for invalid result
         *      types ('warn' and 'error').
         *  - {String} [url]
         *      The URL of a hyperlink associated to the formula result. The
         *      property will be omitted for invalid result types ('warn' and
         *      'error').
         */
        this.interpretTokens = function (contextType, rootNode, options) {

            // the actual interpreter instance, needed for recursive interpretation of e.g. defined names
            var instance = new InterpreterInstance(docModel, rootNode, options);
            // the resulting operand
            var operand = null;
            // the result value extracted from the operand, and the final result object
            var result = null;

            try {

                // calculate the formula, pull formula result operand
                FormulaUtils.takeTime('FormulaInterpreter.interpretTokens(): contextType=' + contextType, function () {
                    operand = instance.getResult(contextType);
                });

                // 'any' context: resolve to actual type of the resulting operand
                contextType = (contextType === 'any') ? operand.getType() : contextType;

                // resolve result operand to result value according to context type
                switch (contextType) {
                    case 'val':
                        result = operand.getValue();
                        // in value context, empty value (reference to empty cell) becomes zero
                        if (result === null) { result = 0; }
                        break;
                    case 'mat':
                        result = operand.getMatrix();
                        break;
                    case 'ref':
                        result = operand.getRanges();
                        break;
                    default:
                        FormulaUtils.throwInternal('FormulaInterpreter.interpretTokens(): unknown context type: "' + contextType + '"');
                }

                // throw error code (special handling for internal errors in catch clause)
                FormulaUtils.throwErrorCode(result);

                // create a complete result object
                result = this.createResult(result, operand);

            } catch (error) {

                // formula errors: invalid formula structure
                if (error instanceof FormulaError) {
                    FormulaUtils.warn('\xa0 error: ' + error.msg);
                    result = this.createErrorResult(error.type);

                // operator may throw error codes as dedicated result
                } else if (error instanceof ErrorCode) {
                    result = this.createResult(error);

                // re-throw internal JS exceptions
                } else {
                    throw error;
                }

            } finally {
                instance.destroy();
            }

            FormulaUtils.info('result: type=' + result.type + ' code=' + result.code + ' value=' + FormulaUtils.valueToString(result.value) + ' format=' + result.format + ' url=' + result.url);
            return result;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class FormulaInterpreter

    // exports ================================================================

    return FormulaInterpreter;

});
