/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/formulacompiler', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/formulaerror',
    'io.ox/office/spreadsheet/model/formula/parser/tokens'
], function (Utils, ModelObject, SheetUtils, FormulaUtils, FormulaError, Tokens) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var LiteralToken = Tokens.LiteralToken;
    var OperatorToken = Tokens.OperatorToken;
    var SheetRefToken = Tokens.SheetRefToken;

    // maps operator precedence categories to operator resource keys
    var OPERATOR_PRECEDENCE_CATEGORIES = {
        cmp: Utils.makeSet(['eq', 'ne', 'lt', 'le', 'gt', 'ge']),
        con: Utils.makeSet(['con']),
        pct: Utils.makeSet(['pct']),
        add: Utils.makeSet(['add', 'sub']),
        mul: Utils.makeSet(['mul', 'div']),
        pow: Utils.makeSet(['pow']),
        list: Utils.makeSet(['list']),
        isect: Utils.makeSet(['isect']),
        range: Utils.makeSet(['range'])
    };

    // private global functions ===============================================

    /**
     * Returns whether the passed formula token is an operator of the specified
     * precedence category.
     *
     * @param {BaseToken|Null} token
     *  The parser token to be checked. May be null, in this case this function
     *  will return false.
     *
     * @param {String} category
     *  An operator precedence category. Must be an existing key in the map
     *  OPERATOR_PRECEDENCE_CATEGORIES.
     *
     * @returns {Boolean}
     *  Whether the passed value is an existing operator token of the specified
     *  precedence category.
     */
    function isOperatorToken(token, category) {
        return _.isObject(token) && token.isType('op') && (token.getValue() in OPERATOR_PRECEDENCE_CATEGORIES[category]);
    }

    // class CompilerNode =====================================================

    /**
     * Represents an operand, an operator, or a function call in the compiled
     * token tree.
     *
     * @property {BaseToken} token
     *  A reference to the original parser token describing the operand, the
     *  operator, or the function call.
     *
     * @property {Array<CompilerNode>|Null} operands
     *  The sub-trees of all operands associated to an operator (value null for
     *  operands, empty array for operators without operands).
     *
     * @property {Object|Null} descriptor
     *  The descriptor with the implementation of an operator or function (null
     *  for operands, and for unknown operators or functions).
     *
     * @property {String|Null} recalc
     *  The recalculation mode of the function represented by this compiler
     *  node, or of any function in the operand sub-trees: Either 'always',
     *  'once', or null.
     *
     * @property {Array<Object>|Null} signature
     *  The type signature of the operands (property 'operands') of an operator
     *  or function (null for operands, and for unknown operators or unknown
     *  functions). Each object in the array contains the following properties:
     *  - {String} typeSpec
     *      The complete type specifier of the parameter as defined in the
     *      operator or function descriptor.
     *  - {String} baseType
     *      The base type of the parameter ('val', 'mat', 'ref', or 'any').
     *  - {String|Null} depSpec
     *      Specifies how to handle a reference operand when collecting the
     *      source dependencies of the formula.
     *
     * @property {FormulaError|Null} error
     *  An error object, if this node represents a formula operator, or a
     *  built-in function (the implementation descriptor in the property
     *  'descriptor' exists), but the number of operands does not match the
     *  definition of the operator or function.
     */
    function CompilerNode(token, operands, descriptor) {

        this.token = token;
        this.operands = operands || null;
        this.descriptor = descriptor || null;
        this.recalc = null;
        this.signature = null;
        this.error = null;

        // initialize extended settings for operators and functions
        this._initialize();

    } // class CompilerNode

    // private methods --------------------------------------------------------

    /**
     * Initializes the property 'error' of this node, if the passed condition
     * is falsy.
     *
     * @param {Any} condition
     *  The condition. If falsy, the property 'error' of this instance will be
     *  set to a new FormulaError.
     *
     * @param {String} message
     *  The message text to be inserted into the formula error.
     *
     * @param {String} code
     *  The internal error code to be inserted into the formula error.
     *
     * @returns {Boolean}
     *  Whether the passed condition was truthy.
     */
    CompilerNode.prototype._ensure = function (condition, message, code) {
        if (!condition) { this.error = new FormulaError(message, code); }
        return !!condition;
    };

    /**
     * Initializes the properties of this instance. Called from constructor.
     */
    CompilerNode.prototype._initialize = function () {

        // operands and operator descriptor must exist
        if (!this.operands || !this.descriptor) { return; }

        // collect the recalculation mode from the operand sub-trees
        this.recalc = this.operands.reduce(function (recalc, operand) {
            return FormulaUtils.getRecalcMode(recalc, operand.recalc);
        }, this.descriptor.recalc);

        // get minimum number of parameters
        var minParams = this.descriptor.minParams;

        // get maximum number of parameters, and length of repeated sequences
        var maxParams = 0, repeatParams = 0;
        if ('maxParams' in this.descriptor) {
            maxParams = this.descriptor.maxParams;
            repeatParams = 1;
        } else {
            repeatParams = ('repeatParams' in this.descriptor) ? this.descriptor.repeatParams : 1;
            maxParams = minParams + Utils.roundDown(FormulaUtils.MAX_PARAM_COUNT - minParams, repeatParams);
        }

        // check operand count
        var params = this.operands.length;
        if (!this._ensure(minParams <= params, 'not enough parameters', 'missing')) { return; }
        if (!this._ensure(params <= maxParams, 'too many parameters', 'unexpected')) { return; }

        // repeated parameter sequences must be complete
        if (!this._ensure((params - minParams) % repeatParams === 0, 'not enough parameters', 'missing')) { return; }

        // build the complete type signature
        var signature = this.descriptor.signature;
        if ((this.operands.length === 0) || (signature.length > 0)) {
            this.signature = _.times(this.operands.length, function (index) {

                // adjust index, if signature is shorter (repeat the last sequence)
                if (signature.length <= index) {
                    index = signature.length - repeatParams + ((index - signature.length) % repeatParams);
                }

                return signature[index];
            });
        }
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this compiler node contains an operator token.
     *
     * @returns {Boolean}
     *  Whether this compiler node contains an operator token.
     */
    CompilerNode.prototype.isOperator = function () {
        return !!this.operands;
    };

    CompilerNode.prototype.toString = function () {
        return this.token + (this.isOperator() ? (':' + this.operands.length) : '');
    };

    // class FormulaCompiler ==================================================

    /**
     * Compiles arrays of formula tokens to a tree structure. This is needed as
     * preparation for interpreting the formula in order to get its current
     * result. All tokens that are not necessary for interpretation will be
     * removed, e.g. whitespace and parentheses.
     *
     * Example: The formula =2+3*(4+5) has been tokenized to the token array
     *  2, PLUS, 3, TIMES, OPEN, 4, PLUS, 5, CLOSE.
     * The compiler will convert this token array to the tree structure
     *  PLUS (2, TIMES (3, PLUS (4, 5))).
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var FormulaCompiler = ModelObject.extend({ constructor: function (docModel) {

        // the descriptors of all supported operators
        var operatorCollection = docModel.getFormulaResource().getOperatorCollection();

        // the descriptors of all supported operators
        var functionCollection = docModel.getFormulaResource().getFunctionCollection();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Compiles the passed tokens from infix notation to a tree structure.
         * Removes all tokens that are not important for interpreting the
         * formula, e.g. white-space tokens, parentheses and parameter
         * separators of function calls, and all other regular parentheses.
         *
         * @param {Array<BaseToken>} tokens
         *  The array of formula tokens, in infix notation.
         *
         * @returns {Object}
         *  The result descriptor of the compilation process, with the
         *  following properties:
         *  - {CompilerNode} root
         *      The root node of the compiled token tree. Will be a single node
         *      containing the #N/A error code as operand, if the compilation
         *      process has failed (see next property 'error').
         *  - {String|Null} error
         *      Null, if the token array has been compiled successfully.
         *      Otherwise, one of the following error codes:
         *      - 'missing': The end of the token array has been reached
         *          unexpectedly (e.g. incomplete formula).
         *      - 'unexpected': A token with an invalid type has been found.
         *  - {Number} missingClose
         *      The number of missing closing parentheses at the end of the
         *      formula expression.
         */
        this.compileTokens = FormulaUtils.profileMethod('FormulaCompiler.compileTokens()', function (tokens) {

            // the array index of the next token to be compiled
            var tokenIndex = 0;
            // whether to stop subexpression at separator token (inside functions)
            var stopAtSep = false;
            // the stack of operand root nodes waiting to be inserted into an operator
            var operandStack = [];
            // number of missing closing parentheses at the end of the formula expression
            var missingClose = 0;

            /**
             * Returns a preview of the next parser token, without increasing
             * the array index.
             *
             * @returns {BaseToken|Null}
             *  The next available parser token; or null, if the end of the
             *  token array has been reached.
             */
            function peekNextRawToken() {
                return tokens[tokenIndex] || null;
            }

            /**
             * Returns the next parser token, and increases the array index.
             *
             * @returns {BaseToken|Null}
             *  The next available parser token; or null, if the end of the
             *  token array has been reached.
             */
            function getNextRawToken() {
                var token = peekNextRawToken();
                if (token) { tokenIndex += 1; }
                return token;
            }

            /**
             * Returns the next non-white-space parser token, and increases the
             * array index.
             *
             * @returns {BaseToken|Null}
             *  The next available parser token; or null, if the end of the
             *  token array has been reached (and the parameter 'required' is
             *  not set).
             */
            function getNextToken() {
                var token = getNextRawToken();
                while (token && token.isType('ws')) { token = getNextRawToken(); }
                return token;
            }

            /**
             * Returns the next non-white-space parser token, and increases the
             * array index. Throws an exception, if no more parser tokens are
             * available in the array.
             *
             * @returns {BaseToken}
             *  The next available parser token.
             *
             * @throws {FormulaError}
             *  A 'missing' exception, if no more parser tokens are available.
             */
            function requireNextToken() {
                var token = getNextToken();
                if (token) { return token; }
                throw new FormulaError('unexpected end of formula', 'missing');
            }

            /**
             * Pushes an operator node onto the operand stack, after pulling
             * the required number of operand nodes.
             */
            function pushOperator(token, params) {
                if (operandStack.length < params) { throw new FormulaError('invalid operand stack size', 'missing'); }
                var operands = (params > 0) ? operandStack.splice(-params) : [];
                var descriptor = token.isType('op') ? operatorCollection.get(token.getValue()) : token.isType('func') ? functionCollection.get(token.getValue()) : null;
                operandStack.push(new CompilerNode(token, operands, descriptor));
            }

            /**
             * Pushes an operand node onto the operand stack.
             */
            function pushOperand(token) {
                operandStack.push(new CompilerNode(token));
            }

            /**
             * Compiles a subexpression by invoking the passed function, and
             * afterwards, inserts the passed token into the compiled token
             * array.
             *
             * @param {Function} compilerFunc
             *  The callback function invoked with the next available token.
             *
             * @param {BaseToken} opToken
             *  The operator token to be inserted into the token array.
             *
             * @param {Number} params
             *  The number of parameters expected by the operator token.
             *
             * @returns {BaseToken|Null}
             *  The token returned by the passed callback function.
             */
            function compileTermAndPush(compilerFunc, opToken, params) {
                var token = compilerFunc(getNextToken());
                pushOperator(opToken, params);
                return token;
            }

            /**
             * Compiles a subexpression starting at the passed token, until the
             * end of the formula, or the specified terminator token.
             */
            function compileExpression(token, newStopAtSep) {
                var oldStopAtSep = stopAtSep;
                stopAtSep = newStopAtSep;
                token = compileCompareTerm(token);
                stopAtSep = oldStopAtSep;
                return token;
            }

            /**
             * Compiles a comparison operator subexpression.
             */
            function compileCompareTerm(token) {
                token = compileConcatTerm(token);
                while (isOperatorToken(token, 'cmp')) {
                    token = compileTermAndPush(compileConcatTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles a string concatenation operator subexpression.
             */
            function compileConcatTerm(token) {
                token = compileAddSubTerm(token);
                while (isOperatorToken(token, 'con')) {
                    token = compileTermAndPush(compileAddSubTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles an addition or subtraction operator subexpression.
             */
            function compileAddSubTerm(token) {
                token = compileMulDivTerm(token);
                while (isOperatorToken(token, 'add')) {
                    token = compileTermAndPush(compileMulDivTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles a multiplication or division operator subexpression.
             */
            function compileMulDivTerm(token) {
                token = compilePowTerm(token);
                while (isOperatorToken(token, 'mul')) {
                    token = compileTermAndPush(compilePowTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles a power operator subexpression.
             */
            function compilePowTerm(token) {
                token = compileUnaryTerm(token);
                while (isOperatorToken(token, 'pow')) {
                    token = compileTermAndPush(compileUnaryTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles a unary prefix/postfix operator subexpression.
             */
            function compileUnaryTerm(token) {
                if (isOperatorToken(token, 'add')) {
                    token = compileTermAndPush(compileUnaryTerm, token, 1);
                } else {
                    token = compileListTerm(token);
                }
                while (isOperatorToken(token, 'pct')) {
                    pushOperator(token, 1);
                    token = getNextToken();
                }
                return token;
            }

            /**
             * Compiles a list operator subexpression.
             */
            function compileListTerm(token) {
                token = compileIntersectTerm(token);
                while (token && (isOperatorToken(token, 'list') || (!stopAtSep && token.isType('sep')))) {
                    if (token.isType('sep')) { token = new OperatorToken('list'); }
                    token = compileTermAndPush(compileIntersectTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles an intersection operator subexpression.
             */
            function compileIntersectTerm(token) {
                token = compileRangeTerm(token);
                while (isOperatorToken(token, 'isect')) {
                    token = compileTermAndPush(compileRangeTerm, token, 2);
                }
                return token;
            }

            /**
             * Compiles a range operator subexpression.
             */
            function compileRangeTerm(token) {
                token = compileOperand(token);
                while (isOperatorToken(token, 'range')) {
                    token = compileTermAndPush(compileOperand, token, 2);
                }
                return token;
            }

            /**
             * Compiles a single operand or function call in a subexpression.
             */
            function compileOperand(token) {

                // an operand must exist
                if (!token) {
                    throw new FormulaError('unexpected end of formula', 'missing');
                }

                // bug 33526: tokens must contain valid sheet references
                if ((token instanceof SheetRefToken) && token.hasSheetError()) {
                    throw new FormulaError('invalid sheet name', 'unexpected');
                }

                // scalar literal, matrix literal, reference, defined name, table reference: push as operand
                if (token.matchesType(/^(lit|mat|ref|name|table)$/)) {
                    pushOperand(token);
                    return getNextToken();
                }

                // opening parenthesis: compile a subexpression up to the closing parenthesis
                if (token.isType('open')) {
                    return compileParentheses();
                }

                // function or macro call: compile parameters enclosed in parentheses
                if (token.isType('func') || token.isType('macro')) {
                    return compileFunction(token);
                }

                throw new FormulaError('operand expected', 'unexpected');
            }

            /**
             * Compiles a complete subexpression in parentheses.
             */
            function compileParentheses() {
                var token = compileExpression(requireNextToken());
                // accept missing closing parenthesis at end of formula
                if (!token) {
                    missingClose += 1;
                } else if (!token.isType('close')) {
                    throw new FormulaError('closing parenthesis expected', 'unexpected');
                }
                return getNextToken();
            }

            /**
             * Compiles a complete function call.
             */
            function compileFunction(token) {

                // the passed function token
                var funcToken = token;
                // current state of the function processor
                var state = 'start';
                // number of parameters in the function
                var params = 0;

                while (state !== 'end') {
                    switch (state) {
                        case 'start':
                            // function start: next token must be an opening parenthesis
                            token = requireNextToken();
                            if (!token.isType('open')) { throw new FormulaError('opening parenthesis expected', 'unexpected'); }
                            state = 'open';
                            break;
                        case 'open':
                            // opening parenthesis: a closing parenthesis may follow directly
                            token = getNextToken();
                            // accept missing closing parenthesis after opening parenthesis at end of formula
                            if (!token) { missingClose += 1; }
                            state = (!token || token.isType('close')) ? 'close' : 'param';
                            break;
                        case 'param':
                            // start of parameter subexpression
                            params += 1;
                            // accept missing closing parenthesis after separator at end of formula
                            if (!token) { missingClose += 1; }
                            if (!token || token.matchesType(/^(sep|close)$/)) {
                                // empty parameter, e.g. in SUM(1,,2)
                                pushOperand(new LiteralToken(null));
                            } else {
                                // compile the parameter subexpression (stop at next separator token)
                                token = compileExpression(token, true);
                                // accept missing closing parenthesis after parameter at end of formula
                                if (!token) {
                                    missingClose += 1;
                                } else if (!token.matchesType(/^(sep|close)$/)) {
                                    throw new FormulaError('parameter separator expected', 'unexpected');
                                }
                            }
                            state = token ? token.getType() : 'close';
                            break;
                        case 'sep':
                            // parameter separator: a valid token must follow
                            token = getNextToken();
                            state = 'param';
                            break;
                        case 'close':
                            // closing parenthesis: finalize the function call
                            pushOperator(funcToken, params);
                            state = 'end';
                            break;
                    }
                }

                return getNextToken();
            }

            try {
                FormulaUtils.logTokens('compiling', tokens);

                // compile the entire token array as subexpression
                var resultToken = compileExpression(requireNextToken());

                // double-check the compiled tokens
                if (operandStack.length === 0) {
                    throw new FormulaError('missing formula expression', 'missing');
                }
                if (operandStack.length > 1) {
                    throw new FormulaError('invalid formula expression', 'unexpected');
                }

                // error, when tokens are left in the passed token array
                if (resultToken || (tokenIndex < tokens.length)) {
                    throw new FormulaError('unexpected garbage at end of formula', 'unexpected');
                }

                FormulaUtils.withLogging(function () {
                    var nodes = [];
                    function pushNode(node) {
                        nodes.push(node);
                        if (node.isOperator()) { node.operands.forEach(pushNode); }
                    }
                    pushNode(operandStack[0]);
                    FormulaUtils.logTokens('compiled', nodes);
                });

            } catch (error) {

                if (error instanceof FormulaError) {
                    FormulaUtils.warn('error: ' + error.message + ', index=' + (tokenIndex - 1));
                    return {
                        root: new CompilerNode(new LiteralToken(ErrorCode.NA)),
                        error: error.type,
                        missingClose: 0
                    };
                }
                throw error;
            }

            return {
                root: operandStack[0],
                error: null,
                missingClose: missingClose
            };
        });

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = operatorCollection = functionCollection = null;
        });

    } }); // class FormulaCompiler

    // exports ================================================================

    return FormulaCompiler;

});
