/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/titlemodel', [
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations'
], function (AttributedModel, Operations) {

    'use strict';

    // class TitleModel =======================================================

    /**
     * A title in a chart object. Can be used as main title of the chart, or as
     * title of a chart axis.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetChartModel} chartModel
     *  The parent chart model containing this title object.
     *
     * @param {String} titleId
     *  The identifier of this title object. Can be 'main' for the main title
     *  of the chart, or 'x', 'y', or 'z' for the respective axis title.
     */
    var TitleModel = AttributedModel.extend({ constructor: function (chartModel, titleId) {

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chartModel.getDocModel(), null, { families: 'text character' });

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this title object.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @returns {TitleModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {
            if (this.hasExplicitAttributes()) {
                var properties = { axis: titleId, attrs: this.getExplicitAttributeSet(true) };
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, properties, { undo: true });
            }
            return this;
        };

    } }); // class TitleModel

    // exports ================================================================

    return TitleModel;

});
