/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/dataseriesmodel', [
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (SimpleMap, AttributedModel, Operations, TokenArray) {

    'use strict';

    var SOURCE_LINK_TYPES = ['names', 'title', 'values', 'bubbles'];

    // class DataSeriesModel ==================================================

    /**
     * Representation of a single data series in a chart object.
     * model.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model that contains the chart object with this data series.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this data series model.
     */
    function DataSeriesModel(sheetModel, initAttributes) {

        // self reference
        var self = this;

        // the document model
        var docModel = sheetModel.getDocModel();

        // token arrays for dynamic source ranges, mapped by source link type
        var tokenArrayMap = new SimpleMap();

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, initAttributes, { families: 'series fill line' });

        // private methods ----------------------------------------------------

        /**
         * Updates the token arrays, after the source link attributes of this
         * data series have been changed.
         */
        function changeAttributesHandler() {

            // the merged series attributes
            var seriesAttrs = self.getMergedAttributeSet(true).series;

            SOURCE_LINK_TYPES.forEach(function (linkType) {

                var sourceLink = seriesAttrs[linkType];

                // create, update, or delete token array according to type of source data
                if (_.isString(sourceLink) && (sourceLink.length > 0)) {
                    // string: parse link formula
                    tokenArrayMap.getOrConstruct(linkType, TokenArray, sheetModel, 'chart').parseFormula('op', sourceLink);
                } else {
                    // else: constant source data, delete token array
                    tokenArrayMap.remove(linkType);
                }
            });
        }

        // public methods -----------------------------------------------------

        this.iterateTokenArrays = function (callback, context) {
            tokenArrayMap.forEach(function (tokenArray, linkType) {
                callback.call(context, tokenArray, linkType);
            });
        };

        this.getRange = function (linkType) {
            return tokenArrayMap.get(linkType).resolveRangeList({ resolveNames: true })[0];
        };

        /**
         * Returns whether any of the source links contained in this data
         * series overlaps with any of the passed cell ranges.
         *
         * @param {Range3DArray|Range3D} ranges
         *  The addresses of the cell ranges, or a single cell range address,
         *  to be checked. The cell range addresses MUST be instances of the
         *  class Range3D with sheet indexes.
         *
         * @returns {Boolean}
         *  Whether any of the passed ranges overlaps with the source links of
         *  this data series.
         */
        this.rangesOverlap = function (ranges) {
            return tokenArrayMap.some(function (tokenArray) {
                return tokenArray.resolveRangeList({ resolveNames: true }).overlaps(ranges);
            });
        };

        /**
         * Tries to replace unresolved sheet names in the token arrays with
         * existing sheet indexes. Intended to be used after document import to
         * refresh all token arrays that refer to sheets that did not exist
         * during their creation.
         *
         * @returns {DataSeriesModel}
         *  A reference to this instance.
         */
        this.refreshAfterImport = function () {
            tokenArrayMap.forEach(function (tokenArray) { tokenArray.refreshAfterImport(); });
            return this;
        };

        /**
         * Changes the formula expressions of all source links with sheet
         * references, after the sheet collection in the document has been
         * changed.
         *
         * @returns {DataSeriesModel}
         *  A reference to this instance.
         */
        this.transformSheet = function (toSheet, fromSheet) {
            tokenArrayMap.forEach(function (tokenArray) { tokenArray.transformSheet(toSheet, fromSheet); });
            return this;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this data series.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @param {Number} index
         *  The index of this data series in the collection of all data series.
         *
         * @returns {DataSeriesModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position, index) {
            var properties = { series: index, attrs: this.getExplicitAttributeSet(true) };
            generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties, { undo: true });
            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source links of this data series.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @param {Number} index
         *  The index of this data series in the collection of all data series.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {DataSeriesModel}
         *  A reference to this instance.
         */
        this.generateFormulaOperations = function (generator, position, index, changeDesc) {

            // collect the changed formula expressions
            var seriesAttrs = {};
            var seriesUndoAttrs = {};
            tokenArrayMap.forEach(function (tokenArray, linkType) {
                var result = tokenArray.resolveOperation('op', changeDesc);
                if (result) {
                    seriesAttrs[linkType] = result.new;
                    seriesUndoAttrs[linkType] = result.old;
                }
            });

            // generate the operations, if any formula expression has changed
            if (!_.isEmpty(seriesAttrs)) {
                var properties = { series: index, attrs: { series: seriesAttrs } };
                generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, properties);
                var undoProperties = { series: index, attrs: { series: seriesUndoAttrs } };
                generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, undoProperties, { undo: true });
            }

            return this;
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = sheetModel = tokenArrayMap = null;
        });

    } // class DataSeriesModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: DataSeriesModel });

});
