/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/condformatcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellvaluecache',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/dateutils',
    'io.ox/office/spreadsheet/model/formula/utils/statutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, SimpleMap, IteratorUtils, TimerMixin, ModelObject, Color, AttributedModel, Operations, SheetUtils, CellValueCache, FormulaUtils, DateUtils, StatUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var RangeArray = SheetUtils.RangeArray;
//    var AddressArray = SheetUtils.AddressArray;

    // limit types (for colorScale, dataBar, and iconSet) that depend on a formula
    var DYNAMIC_LIMIT_TYPES = Utils.makeSet(['formula', 'percent', 'percentile']);

    // maps all supported rule types to type categories
    // - 'value' for rules that use formula expressions from attributes 'value1' and 'value2'
    // - 'self' for rules that work on individual target cells without formula expressions
    // - 'range' for rules that always need to update all target cells if something changes
    var RULE_TYPE_CATEGORIES = {
        formula:        'value',
        between:        'value',
        notBetween:     'value',
        equal:          'value',
        notEqual:       'value',
        less:           'value',
        lessEqual:      'value',
        greater:        'value',
        greaterEqual:   'value',
        contains:       'value',
        notContains:    'value',
        beginsWith:     'value',
        endsWith:       'value',
        yesterday:      'self',
        today:          'self',
        tomorrow:       'self',
        last7Days:      'self',
        lastWeek:       'self',
        thisWeek:       'self',
        nextWeek:       'self',
        lastMonth:      'self',
        thisMonth:      'self',
        nextMonth:      'self',
        lastYear:       'self',
        thisYear:       'self',
        nextYear:       'self',
        blank:          'self',
        notBlank:       'self',
        error:          'self',
        noError:        'self',
        aboveAverage:   'range',
        atLeastAverage: 'range',
        belowAverage:   'range',
        atMostAverage:  'range',
        topN:           'range',
        bottomN:        'range',
        topPercent:     'range',
        bottomPercent:  'range',
        unique:         'range',
        duplicate:      'range',
        colorScale:     'range',
        dataBar:        'range',
        iconSet:        'range'
    };

    // private global functions ===============================================

    /**
     * Transforms the specified target ranges.
     *
     * @param {RangeArray} targetRanges
     *  The addresses of the target ranges to be transformed.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array of move descriptors that specify how to transform the passed
     *  target ranges.
     *
     * @param {Boolean} [reverse=false]
     *  If set to true, the move descriptors will be processed in reversed
     *  order, and the opposite move operation will be used to transform the
     *  target ranges.
     *
     * @returns {RangeArray}
     *  The transformed target ranges, as range array.
     */
    function transformTargetRanges(targetRanges, moveDescs, reverse) {

        // transform the passed ranges, expand the end of the range when inserting cells
        Utils.iterateArray(moveDescs, function (moveDesc) {
            targetRanges = moveDesc.transformRanges(targetRanges, { expandEnd: true, reverse: reverse });
            if (targetRanges.empty()) { return Utils.BREAK; }
        }, { reverse: reverse });

        return targetRanges;
    }

    /**
     * Extracts the rule attributes from the passed document operation.
     *
     * @param {SheetOperationContext} context
     *  A wrapper representing a rule operation.
     *
     * @param {Object} defaultValues
     *  A map with default valus for missing attributes.
     *
     * @returns {Object}
     *  The rule attributes extracted from the passed document operation.
     */
    function getRuleAttributes(context, defaultValues) {
        return {
            type: context.getOptStr('type', defaultValues.type),
            value1: context.getOpt('value1', defaultValues.value1),
            value2: context.getOpt('value2', defaultValues.value2),
            priority: context.getOptInt('priority', defaultValues.priority),
            stop: context.getOptBool('stop', defaultValues.stop),
            colorScale: context.getOptArr('colorScale', defaultValues.colorScale),
            dataBar: context.getOptObj('dataBar', defaultValues.dataBar),
            iconSet: context.getOptObj('iconSet', defaultValues.iconSet),
            attrs: context.getOptObj('attrs', defaultValues.attrs)
        };
    }

    // class RuleModel ========================================================

    /**
     * Stores settings for a single formatting rule of conditionally formatted
     * cell ranges.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The parent sheet model containing this formatting rule.
     *
     * @param {String} ruleId
     *  The unique identifier of this formatting rule. This identifier will not
     *  change during lifetime of the rule model.
     *
     * @param {Object} ruleAttrs
     *  The initial rule attributes, as plain attribute map (not as submap of a
     *  complete attribute set!).
     */
    var RuleModel = AttributedModel.extend({ constructor: function (sheetModel, ruleId, ruleAttrs) {

        // self reference
        var self = this;

        // the collections of the parent sheet model
        var cellCollection = sheetModel.getCellCollection();
        var condFormatCollection = sheetModel.getCondFormatCollection();

        // the spreadsheet document model, and other document objects
        var docModel = sheetModel.getDocModel();
        var numberFormatter = docModel.getNumberFormatter();
        var formulaInterpreter = docModel.getFormulaInterpreter();

        // the token array representing the formula expressions of the various conditions
        var tokenArrayMap = new SimpleMap();

        // the category of the rule type
        var ruleCategory = null;

        // whether the document is based on an OpenDocument file
        var odf = docModel.getApp().isODF();

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, { rule: ruleAttrs }, { families: 'rule' });

        // private methods ----------------------------------------------------

        /**
         * Returns the value cache for the target cell ranges covered by this
         * formatting rule.
         *
         * @returns {CellValueCache}
         *  The value cache for the target cell ranges of this formatting rule.
         */
        function getRangeCache() {
            return condFormatCollection.getRangeCache(ruleId);
        }

        /**
         * Returns the smallest number covered by the target ranges.
         *
         * @param {Boolean} autoZero
         *  If set to true, and the minimum is positive, zero will be returned.
         *
         * @returns {Number|Null}
         *  The smallest number of the covered cells; or null if the target
         *  ranges do not contain any numbers.
         */
        function getMinNumber(autoZero) {
            var min = getRangeCache().getMin();
            return !isFinite(min) ? null : autoZero ? Math.min(0, min) : min;
        }

        /**
         * Returns the largest number covered by the target ranges.
         *
         * @param {Boolean} autoZero
         *  If set to true, and the maximum is negative, zero will be returned.
         *
         * @returns {Number|Null}
         *  The largest number of the covered cells; or null if the target
         *  ranges do not contain any numbers.
         */
        function getMaxNumber(autoZero) {
            var max = getRangeCache().getMax();
            return !isFinite(max) ? null : autoZero ? Math.max(0, max) : max;
        }

        /**
         * Returns a number between smallest and largest number in the cells of
         * the target ranges.
         *
         * @param {Number} percent
         *  The percentage.
         *
         * @returns {Number|Null}
         *  The resulting number; or null if the target ranges do not contain
         *  any numbers.
         */
        function getPercentNumber(percent) {
            var rangeCache = getRangeCache();
            var min = rangeCache.getMin(), max = rangeCache.getMax();
            var number = (max - min) * (percent / 100) + min;
            return isFinite(number) ? number : null;
        }

        /**
         * Returns a number between smallest and largest number in the cells of
         * the target ranges.
         *
         * @param {Number} percentile
         *  The percentile.
         *
         * @returns {Number|Null}
         *  The resulting number; or null if the target ranges do not contain
         *  any numbers.
         */
        function getPercentileNumber(percentile) {
            try {
                var numbers = getRangeCache().getSortedNumbers();
                return StatUtils.getPercentileInclusive(numbers, percentile / 100, true);
            } catch (error) {
                return null;
            }
        }

        /**
         * Parses the passed formula expression into a token array.
         *
         * @param {String} key
         *  The map key of a specific token array.
         *
         * @param {Any} value
         *  The attribute value. Strings will be parsed as formula expression
         *  into the specified token array. All other values will cause to
         *  destroy and remove the token array from the internal map.
         */
        function parseAttributeValue(key, value) {
            if ((typeof value === 'string') && (value.length > 0)) {
                tokenArrayMap.getOrConstruct(key, TokenArray, sheetModel, 'rule').parseFormula('op', value, { refAddress: self.getRefAddress() });
            } else {
                tokenArrayMap.remove(key);
            }
        }

        /**
         * Parses the passed formula expression into a token array.
         *
         * @param {String} key
         *  The map key of a specific token array.
         *
         * @param {Any} value
         *  The attribute value. Strings will be parsed as formula expression
         *  into the specified token array. All other values will cause to
         *  destroy and remove the token array from the internal map.
         */
        function parseDescriptorValue(key, desc) {
            // validate the passed descriptor (only specific limit types depend on a formula, e.g. 'min' and 'max' do not)
            var value = (_.isObject(desc) && _.isString(desc.t) && (desc.t in DYNAMIC_LIMIT_TYPES)) ? desc.v : null;
            parseAttributeValue(key, value);
        }

        /**
         * Updates internal settings after the rule attributes of this rule
         * model have been changed.
         */
        function changeAttributesHandler(event, newAttributeSet, oldAttributeSet) {

            // extract the rule attributes from the passed attribute sets
            var newRuleAttrs = (newAttributeSet || self.getMergedAttributeSet(true)).rule;
            var oldRuleAttrs = oldAttributeSet ? oldAttributeSet.rule : {};
            // whether the rule type has changed
            var typeChanged = newRuleAttrs.type !== oldRuleAttrs.type;

            // delete all token arrays, if the rule type has changed, and detect whether this rule is based on all covered cells
            if (typeChanged) {
                tokenArrayMap.clear();
                ruleCategory = RULE_TYPE_CATEGORIES[newRuleAttrs.type] || null;
            }

            // use formula expressions in 'value1' and 'value2' for simple rules
            if (ruleCategory === 'value') {
                ['value1', 'value2'].forEach(function (propName) {
                    if (typeChanged || (oldRuleAttrs[propName] !== newRuleAttrs[propName])) {
                        parseAttributeValue(propName, newRuleAttrs[propName]);
                    }
                });
                return;
            }

            // parse formula expressions according to the current rule type
            switch (newRuleAttrs.type) {

                // attribute 'colorScale' contains an unlimited number of color stops
                case 'colorScale':
                    var colorScale = newRuleAttrs.colorScale;
                    if (_.isArray(colorScale) && (typeChanged || !_.isEqual(colorScale, oldRuleAttrs.colorScale))) {
                        tokenArrayMap.clear();
                        colorScale.forEach(function (scaleStep, index) {
                            parseDescriptorValue('step' + index, scaleStep);
                        });
                    }
                    break;

                // attribute 'dataBar' contains two formula expressions for minimum/maximum
                case 'dataBar':
                    var dataBar = newRuleAttrs.dataBar;
                    if (_.isObject(dataBar) && (typeChanged || !_.isEqual(dataBar, oldRuleAttrs.dataBar))) {
                        tokenArrayMap.clear();
                        ['r1', 'r2'].forEach(function (propName) {
                            var dataBarRule = dataBar[propName];
                            parseDescriptorValue(propName, dataBarRule);
                        });
                    }
                    break;

                // attribute 'iconSet' contains up to 4 formula expressions
                case 'iconSet':
                    var iconSet = newRuleAttrs.iconSet;
                    if (_.isObject(iconSet) && (typeChanged || !_.isEqual(iconSet, oldRuleAttrs.iconSet))) {
                        tokenArrayMap.clear();
                        var rules = iconSet.r;
                        if (_.isArray(rules) && !_.isEmpty(rules)) {
                            rules.forEach(function (iconSetRule, index) {
                                parseDescriptorValue('rule' + index, iconSetRule);
                            });
                        }
                    }
                    break;
            }
        }

        /**
         * Returns the interpreted result of the specified formula expression.
         *
         * @param {String} key
         *  The map key of a token array.
         *
         * @param {Address} [targetAddress]
         *  The target address needed to interpret the formula (the address of
         *  the cell to interpret the token array for). If omitted, the
         *  reference address of the parent formatting model will be used (for
         *  range rules, such as colorScale, dataBar, and iconSet).
         */
        function interpretCondition(key, targetAddress) {

            // get the specified token array, fall-back to the #N/A error code
            var tokenArray = tokenArrayMap.get(key);
            if (!tokenArray) { return ErrorCode.NA; }

            // calculate the result relative to the target address
            var refAddress = self.getRefAddress();
            var options = { refSheet: sheetModel.getIndex(), refAddress: refAddress, targetAddress: targetAddress || refAddress };
            return tokenArray.interpretFormula('val', options).value;
        }

        /**
         * Implementation for comparison rules. Returns whether the passed cell
         * value and the comparison value resolved from the formula expression
         * of the rule property 'value1' or 'value2' are related to each other
         * as expected.
         */
        function matchesValue(value, address, key, type) {
            var result = formulaInterpreter.compareScalars(value, interpretCondition(key, address));
            switch (type) {
                case 'equal':        return result === 0;
                case 'notEqual':     return result !== 0;
                case 'less':         return result < 0;
                case 'lessEqual':    return result <= 0;
                case 'greater':      return result > 0;
                case 'greaterEqual': return result >= 0;
            }
            return false;
        }

        /**
         * Implementation for interval rules. Returns whether the passed cell
         * value is (not) located in the interval formed by the values resolved
         * from the formula expressions of the rule properties 'value1' and
         * 'value2'.
         */
        function matchesInterval(value, address, matchInside) {
            var isInside = matchesValue(value, address, 'value1', 'greaterEqual') && matchesValue(value, address, 'value2', 'lessEqual');
            return matchInside === isInside;
        }

        /**
         * Implementation for text rules. Returns whether the passed cell value
         * and the comparison string resolved from the formula expression of
         * the rule property 'value1' are matching according to the predicate
         * function.
         */
        function matchesString(value, address, predicate) {
            var textValue = numberFormatter.convertScalarToString(value, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
            if (textValue === null) { return false; }
            var compValue = interpretCondition('value1', address);
            compValue = numberFormatter.convertScalarToString(compValue, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
            return (compValue !== null) && (compValue.length > 0) && predicate(textValue, compValue);
        }

        /**
         * Implementation for date rules. Returns whether the passed cell value
         * is matching according to the exact rule type.
         */
        function matchesDate(value, type, diff) {

            // convert the cell value to a date (returns null on error)
            var date = numberFormatter.convertScalarToDate(value, true);
            if (!date) { return false; }

            var compDate = DateUtils.makeUTCToday();
            switch (type) {
                case 'day':
                    if (diff !== 0) { compDate.setUTCDate(compDate.getUTCDate() + diff); }
                    return (compDate.getUTCFullYear() === date.getUTCFullYear()) && (compDate.getUTCMonth() === date.getUTCMonth()) && (compDate.getUTCDate() === date.getUTCDate());
                case '7days':
                    diff = DateUtils.getDaysDiff(date, compDate);
                    return (diff >= 0) && (diff < 7);
                case 'week':
                    compDate.setUTCDate(compDate.getUTCDate() - compDate.getUTCDay());
                    if (diff !== 0) { compDate.setUTCDate(compDate.getUTCDate() + 7 * diff); }
                    diff = DateUtils.getDaysDiff(compDate, date);
                    return (diff >= 0) && (diff < 7);
                case 'month':
                    if (diff !== 0) { compDate.setUTCMonth(compDate.getUTCMonth() + diff); }
                    return (compDate.getUTCFullYear() === date.getUTCFullYear()) && (compDate.getUTCMonth() === date.getUTCMonth());
                case 'year':
                    if (diff !== 0) { compDate.setUTCYear(compDate.getUTCFullYear() + diff); }
                    return compDate.getUTCFullYear() === date.getUTCFullYear();
            }
            return false;
        }

        /**
         * Implementation for average rules. Returns whether the passed cell
         * value matches the average of all numbers covered by this rule.
         */
        function matchesAverage(value, count, subtract, equal) {

            // try to convert the passed value to a number
            var number = numberFormatter.convertScalarToNumber(value);
            if (number === null) { return false; }

            // get the arithmetic mean of the target ranges
            var rangeCache = getRangeCache();
            var mean = rangeCache.getMean();
            if (!isFinite(mean)) { return false; }

            // add/subtract the standard deviations
            count = (typeof count === 'number') ? Math.floor(Math.max(count, 0)) : 0;
            var devsq = (count === 0) ? 0 : Math.sqrt(rangeCache.getDeviation());
            var average = mean + devsq * (subtract ? -count : count);
            if (!isFinite(average)) { return false; }

            // compare the resulting average with the passed number
            var comp = FormulaUtils.compareNumbers(number, average);
            return (comp === 0) ? equal : subtract ? (comp < 0) : (comp > 0);
        }

        /**
         * Implementation for rank rules. Returns whether the passed cell value
         * matches the rank in the numbers covered by this rule.
         */
        function matchesRank(value, rank, percent, top) {

            // rank rule applies to numbers only, other cells will never match
            if ((typeof value !== 'number') || (typeof rank !== 'number')) { return false; }

            // the sorted numbers in the cell ranges
            var numbers = getRangeCache().getSortedNumbers();
            // the count of the numbers
            var count = numbers.length;

            // calculate a percentual rank (round down, but at least 1, regardless of exact result)
            rank = percent ? Math.max(1, Math.floor(rank / 100 * count)) : Math.floor(rank);

            // compare passed value with the n-th array element
            if (rank < 1) { return false; }
            if (rank > count) { return true; }
            return top ? (numbers[count - rank] <= value) : (value <= numbers[rank - 1]);
        }

        /**
         * Implementation for quantity rules. Returns whether the passed cell
         * is (not) unique in the cells covered by this rule.
         */
        function matchesQuantity(value, matchUnique) {

            // blank cells are neither unique nor duplicate
            if (value === null) { return null; }

            // empty strings (as formula result) are considered to be duplicate to blank cells
            var rangeCache = getRangeCache();
            if ((value === '') && (rangeCache.getBlankCount() > 0)) { return !matchUnique; }

            // the value cache has counted all scalar values
            var isUnique = rangeCache.countScalar(value) === 1;
            return matchUnique === isUnique;
        }

        /**
         * Implementation for blank rules. Returns whether the passed cell
         * value is (not) blank. Strings consisting of space characters only
         * are considered to be blank too.
         */
        function matchesBlank(value, matchBlank) {
            var isBlank = (value === null) || ((typeof value === 'string') && /^ *$/.test(value));
            return matchBlank === isBlank;
        }

        /**
         * Implementation for error rules. Returns whether the passed cell
         * value is (not) an error code.
         */
        function matchesError(value, matchError) {
            var isError = value instanceof ErrorCode;
            return matchError === isError;
        }

        /**
         * Returns the effective number for the specified token array, and a
         * specific result type, for a limiting value in range rules.
         *
         * @param {String} key
         *  The map key of a token array.
         *
         * @param {String} type
         *  One of the following type specifiers:
         *  - 'formula': The passed value will be converted to a number, and
         *      returned.
         *  - 'min': The minimum of the numbers of the parent formatting model
         *      will be returned. The passed cell value will be ignored.
         *  - 'max': The maximum of the numbers of the parent formatting model
         *      will be returned. The passed cell value will be ignored.
         *  - 'percent': The passed cell value will be interpreted as percent,
         *      and a scaled number between the minimum and maximum of the
         *      parent formatting model will be returned.
         *  - 'percentile': The passed cell value will be interpreted as
         *      percentile, and a scaled number between the minimum and maximum
         *      of the parent formatting model will be returned.
         *
         * @returns {Number|Null}
         *  The resulting number according to the specified type; or null, if
         *  no valid number could be resolved.
         */
        function getLimitForType(key, type) {

            // ignore the passed cell value for minimum/maximum
            switch (type) {
                case 'min':     return getMinNumber(false);
                case 'max':     return getMaxNumber(false);
                case 'automin': return getMinNumber(true);
                case 'automax': return getMaxNumber(true);
            }

            // try to convert the passed scalar to a number
            var value = interpretCondition(key);
            var number = docModel.getNumberFormatter().convertScalarToNumber(value);
            if (number === null) { return null; }

            switch (type) {
                case 'formula':    return number;
                case 'percent':    return getPercentNumber(number);
                case 'percentile': return getPercentileNumber(number);
            }

            Utils.error('RuleModel.getLimitForType(): invalid type "' + type + '"');
            return null;
        }

        /**
         * Returns the effective formatting attributes for a scalar cell value,
         * and the passed color scale settings, according to the numbers
         * covered by the parent formatting rule.
         *
         * @param {Any} value
         *  A scalar cell value of any type.
         *
         * @returns {Object|Null}
         *  The cell attribute with the effective fill color for the passed
         *  cell value and color scale properties; or null, if no fill color
         *  could be calculated.
         */
        function getColorScaleAttributes(value) {

            function createAttributes(jsonColor) {
                return _.isObject(jsonColor) ? { cell: { fillColor: jsonColor } } : null;
            }

            // cell value must be a real number (no type conversion)
            if (typeof value !== 'number') { return null; }

            // check validity of the color scale properties
            var colorScale = self.getMergedAttributeSet(true).rule.colorScale;
            if (!_.isArray(colorScale) || (colorScale.length < 2) || !colorScale.every(_.isObject)) { return null; }

            // calculate the effective limits of all color steps
            var limits = colorScale.map(function (colorStep, index) {
                return getLimitForType('step' + index, colorStep.t);
            });

            // exit if a simgle limit could not be calculated
            if (limits.some(function (limit) { return limit === null; })) { return null; }

            // sort the limits (without reordering the color stops!)
            limits = _.sortBy(limits);

            // return start color, if the cell value is less than or equal to the lowest limit
            if (value <= limits[0]) { return createAttributes(colorScale[0].c); }

            // return end color, if the cell value is greater than or equal to the highest limit
            if (value >= _.last(limits)) { return createAttributes(_.last(colorScale).c); }

            // find the index of the first color stop that follows or equals the cell value
            var index = Utils.findFirstIndex(limits, function (limit) { return value <= limit; }, { sorted: true });

            // return the color stop value, if the cell value is exactly equal to the limit
            if (limits[index] === value) { return createAttributes(colorScale[index].c); }

            // create a mixed color, according to the start and end of the interval
            var ratio = (value - limits[index - 1]) / (limits[index] - limits[index - 1]);
            var color1 = Color.parseJSON(colorScale[index - 1].c);
            var color2 = Color.parseJSON(colorScale[index].c);
            var mixedColor = color1.resolveMixed(color2, ratio, 'fill', docModel.getTheme());
            return createAttributes({ type: 'rgb', value: mixedColor.hex });
        }

        /**
         * Return the properties to render a data bar in the specified cell.
         *
         * @param {Any} value
         *  A scalar cell value of any type.
         *
         * @returns {Object|Null}
         *  If the passed cell value is a number, the rendering properties for
         *  a data bar will be returned; otherwise null.
         */
        function getDataBarProperties(value) {

            // data bars only for number cells (no type conversion)
            if (typeof value !== 'number') { return null; }

            // the rule settings containing the original data bar properties
            var dataBar = self.getMergedAttributeSet(true).rule.dataBar;
            if (!_.isObject(dataBar) || !_.isObject(dataBar.r1) || !_.isObject(dataBar.r2)) { return null; }

            // resolve the interval of the color scale
            var limit1 = getLimitForType('r1', dataBar.r1.t);
            var limit2 = getLimitForType('r2', dataBar.r2.t);
            if ((limit1 === null) || (limit2 === null)) { return null; }

            // the resulting data bar properties
            return { dataBar: {
                min: Math.min(limit1, limit2),
                max: Math.max(limit1, limit2),
                num: value,
                color: dataBar.c,
                showValue: !!dataBar.s
            } };
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this rule model for the
         * specified sheet.
         *
         * @internal
         *  Used by the class CondFormatCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The sheet model instance that will own the cloned rule model
         *  returned by this method.
         *
         * @returns {RuleModel}
         *  A clone of this rule model, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {
            var ruleAttrs = this.getExplicitAttributeSet(true).rule;
            return new RuleModel(targetModel, ruleId, ruleAttrs);
        };

        /**
         * Tries to replace unresolved sheet names in the token arrays with
         * existing sheet indexes. Intended to be used after document import to
         * refresh all token arrays that refer to sheets that did not exist
         * during their creation.
         *
         * @returns {RuleModel}
         *  A reference to this instance.
         */
        this.refreshAfterImport = function () {
            var options = { refAddress: this.getRefAddress() };
            tokenArrayMap.forEach(function (tokenArray) {
                tokenArray.refreshAfterImport(options);
            });
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the sheet model instance that contains the collection with
         * this rule model.
         *
         * @returns {SheetModel}
         *  The parent sheet model instance.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the unique identifier of this formatting rule, as used in
         * document operations.
         *
         * @returns {String}
         *  The unique identifier of this formatting rule.
         */
        this.getRuleId = function () {
            return ruleId;
        };

        /**
         * Returns the priority of this rule model. The less this value, the
         * earlier this rule model will be evaluated compared to all other rule
         * models in a conditional formatting collection.
         *
         * @returns {Number}
         *  The priority of this rule model.
         */
        this.getPriority = function () {
            return this.getMergedAttributeSet(true).rule.priority;
        };

        /**
         * Returns the addresses of all target ranges covered by this rule.
         *
         * @returns {RangeArray}
         *  The addresses of all target ranges.
         */
        this.getTargetRanges = function () {
            return condFormatCollection.getTargetRanges(ruleId);
        };

        /**
         * Returns the reference addresses of the target ranges covered by this
         * instance. The reference address is the address of the top-left cell
         * of the bounding range of all target cell ranges.
         *
         * @returns {Address}
         *  The address of the reference cell of the target ranges.
         */
        this.getRefAddress = function () {
            return condFormatCollection.getRefAddress(ruleId);
        };

        /**
         * Returns whether this rule needs all its covered cells for evaluating
         * if it applies to a specific cell value. For example, 'average' rules
         * use the arithmetic mean of all covered number cells. After changing
         * one of these cells, all other cells covered by this rule need to be
         * checked whether they apply to the new arithmetic mean.
         *
         * @returns {Boolean}
         *  Whether this rule needs all its covered cells for evaluating if it
         *  applies to a specific cell value.
         */
        this.isRangeRule = function () {
            return ruleCategory === 'range';
        };

        /**
         * Returns the target position to be used to resolve the dependencies
         * of the formulas contained in this rule model. For range rules (see
         * method RuleModel.isRangeRule() for details), this is the reference
         * address of the parent formatting model (the formulas will always be
         * evaluated relative to the reference address), otherwise the target
         * ranges of the formatting model will be returned (the formulas will
         * be evaluated relative to each cell in the target ranges).
         *
         * @returns {Address|RangeArray}
         *  The target position to resolve the formula dependencies, according
         *  to the type of this rule.
         */
        this.getDependencyTarget = function () {
            return this.isRangeRule() ? this.getRefAddress() : this.getTargetRanges();
        };

        /**
         * Returns an iterator that visits all token arrays of this rule. The
         * token arrays will be visited in no specific order.
         *
         * @returns {Object}
         *  An iterator object that implements the standard EcmaScript iterator
         *  protocol, i.e. it provides the method next() that returns a result
         *  object with the following properties:
         *  - {Boolean} done
         *      If set to true, the token arrays have been visited completely.
         *      No more token arrays are available; this result object does not
         *      contain any other properties!
         *  - {TokenArray} value
         *      The token array currently visited.
         *  - {String} key
         *      The internal map key of the token array currently visited.
         */
        this.createTokenArrayIterator = function () {
            return tokenArrayMap.iterator();
        };

        /**
         * Returns the explicit formatting attributes contained in this rule
         * model that will be applied to a cell if this rule matches its value.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in this instance, which MUST NOT be
         *  modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object}
         *  The explicit formatting attributes that will be applied if this
         *  rule matches the cell value.
         */
        this.getMatchAttributeSet = function (direct) {
            var attributeSet = this.getMergedAttributeSet(true).rule.attrs;
            return direct ? attributeSet : _.copy(attributeSet, true);
        };

        /**
         * Returns the formatting attributes to be applied at the specified
         * target cell, if the conditions of this rule are matching.
         *
         * @param {Address} address
         *  The address of the cell to calculate the attributes for. MUST be a
         *  cell that is covered by the parent formatting model.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Object|Null} attributeSet
         *      The attribute set with all character and cell formatting
         *      attributes to be applied to the cell. Will be null, if the rule
         *      does not match the passed cell value, or for a data bar rule
         *      that returns rendering properties only.
         *  - {Object|Null} renderProps
         *      Additional optional rendering properties, e.g. for data bars.
         *  - {Boolean} stop
         *      Whether to abort the evalution of following formatting rules.
         */
        this.resolveRule = function (address) {

            // the settings of this rule
            var ruleAttrs = this.getMergedAttributeSet(true).rule;
            // the current value of the specified cell
            var value = cellCollection.getCellValue(address);
            // whether the cell value matches the conditions (always false for the advanced range rules colorScale, DataBar, and iconSet)
            var matches = false;
            // the cell formatting attributes to be applied at the cell for this rule
            var attributeSet = null;
            // additional rendering properties, e.g. for data bars
            var renderProps = null;

            // process all supported rule types
            switch (ruleAttrs.type) {

                // formula expression: rule matches if result is truthy
                case 'formula':
                    value = interpretCondition('value1', address);
                    matches = numberFormatter.convertScalarToBoolean(value);
                    break;

                // interval rules: value must be inside or outside an interval
                case 'between':
                    matches = matchesInterval(value, address, true);
                    break;
                case 'notBetween':
                    matches = matchesInterval(value, address, false);
                    break;

                // comparison rules: compare cell value with formula expression in 'value1'
                case 'equal':
                case 'notEqual':
                case 'less':
                case 'lessEqual':
                case 'greater':
                case 'greaterEqual':
                    matches = matchesValue(value, address, 'value1', ruleAttrs.type);
                    break;

                // text rules
                case 'contains':
                    // TODO: case-insensitivity, pattern matching
                    matches = matchesString(value, address, function (text, comp) { return text.indexOf(comp) >= 0; });
                    break;
                case 'notContains':
                    // TODO: case-insensitivity, pattern matching
                    matches = matchesString(value, address, function (text, comp) { return text.indexOf(comp) < 0; });
                    break;
                case 'beginsWith':
                    matches = matchesString(value, address, function (text, comp) { return text.substr(0, comp.length) === comp; });
                    break;
                case 'endsWith':
                    matches = matchesString(value, address, function (text, comp) { return text.substr(-comp.length) === comp; });
                    break;

                // date rules
                case 'yesterday':
                    matches = matchesDate(value, 'day', -1);
                    break;
                case 'today':
                    matches = matchesDate(value, 'day', 0);
                    break;
                case 'tomorrow':
                    matches = matchesDate(value, 'day', 1);
                    break;
                case 'last7Days':
                    matches = matchesDate(value, '7days', 0);
                    break;
                case 'lastWeek':
                    matches = matchesDate(value, 'week', -1);
                    break;
                case 'thisWeek':
                    matches = matchesDate(value, 'week', 0);
                    break;
                case 'nextWeek':
                    matches = matchesDate(value, 'week', 1);
                    break;
                case 'lastMonth':
                    matches = matchesDate(value, 'month', -1);
                    break;
                case 'thisMonth':
                    matches = matchesDate(value, 'month', 0);
                    break;
                case 'nextMonth':
                    matches = matchesDate(value, 'month', 1);
                    break;
                case 'lastYear':
                    matches = matchesDate(value, 'year', -1);
                    break;
                case 'thisYear':
                    matches = matchesDate(value, 'year', 0);
                    break;
                case 'nextYear':
                    matches = matchesDate(value, 'year', 1);
                    break;

                // above/below average rules
                case 'aboveAverage':
                    matches = matchesAverage(value, ruleAttrs.value1, false, false);
                    break;
                case 'atLeastAverage':
                    matches = matchesAverage(value, ruleAttrs.value1, false, true);
                    break;
                case 'belowAverage':
                    matches = matchesAverage(value, ruleAttrs.value1, true, false);
                    break;
                case 'atMostAverage':
                    matches = matchesAverage(value, ruleAttrs.value1, true, true);
                    break;

                // top/bottom rules
                case 'topN':
                    matches = matchesRank(value, ruleAttrs.value1, false, true);
                    break;
                case 'topPercent':
                    matches = matchesRank(value, ruleAttrs.value1, true, true);
                    break;
                case 'bottomN':
                    matches = matchesRank(value, ruleAttrs.value1, false, false);
                    break;
                case 'bottomPercent':
                    matches = matchesRank(value, ruleAttrs.value1, true, false);
                    break;

                // quantity rules
                case 'unique':
                    matches = matchesQuantity(value, true);
                    break;
                case 'duplicate':
                    matches = matchesQuantity(value, false);
                    break;

                // cell type rules
                case 'blank':
                    matches = matchesBlank(value, true);
                    break;
                case 'notBlank':
                    matches = matchesBlank(value, false);
                    break;
                case 'error':
                    matches = matchesError(value, true);
                    break;
                case 'noError':
                    matches = matchesError(value, false);
                    break;

                // complex range rules
                case 'colorScale':
                    // resolve the attributes (fill color) directly, do not set 'matches' (colorScale cannot stop rule evaluation)
                    attributeSet = getColorScaleAttributes(value);
                    break;
                case 'dataBar':
                    // do not set 'matches' (dataBar cannot stop rule evaluation)
                    renderProps = getDataBarProperties(value);
                    break;
                case 'iconSet':
                    // do not set 'matches' (iconSet cannot stop rule evaluation)
                    // TODO
                    break;
            }

            // if a regular rule matches the passed cell value, return the formatting attributes of this rule
            if (matches) {
                attributeSet = this.getMatchAttributeSet(); // returns a deep copy of the attributes
            }

            // ODF: resolve the explicit attributes of a style sheet (but do not add the default attribute values)
            if (attributeSet && ('styleId' in attributeSet)) {
                if (odf) {
                    attributeSet = docModel.getCellStyles().getMergedAttributes(attributeSet, { skipDefaults: true });
                }
                delete attributeSet.styleId;
            }

            return {
                attributeSet: attributeSet,
                renderProps: renderProps,
                stop: matches && ruleAttrs.stop
            };
        };

        /**
         * Changes all formula expressions stored in the rule attributes, after
         * the sheet collection of the document has been changed.
         *
         * @returns {RuleModel}
         *  A reference to this instance.
         */
        this.transformSheet = function (toSheet, fromSheet) {
            tokenArrayMap.forEach(function (tokenArray) { tokenArray.transformSheet(toSheet, fromSheet); });
            return this;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operation to restore this formatting rule after
         * it has been deleted.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} rangesStr
         *  The string representation of the target ranges of this rule, as
         *  used in document operations.
         *
         * @returns {RuleModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, rangesStr) {

            // initial operation properties
            var properties = { id: ruleId, ranges: rangesStr };

            // add the rule attributs that are different to the defaults
            var ruleAttrs = this.getExplicitAttributeSet(true).rule;
            var ruleDefAttrs = docModel.getDefaultAttributes('rule');
            _.each(ruleAttrs, function (ruleAttr, attrName) {
                if (!_.isEqual(ruleAttr, ruleDefAttrs[attrName])) {
                    properties[attrName] = ruleAttr;
                }
            });

            generator.generateSheetOperation(Operations.INSERT_CFRULE, properties, { undo: true });
            return this;
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the formula expressions of this formatting rule.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @param {String|Null} newRangesStr
         *  The new string representation of the target ranges of this rule to
         *  be added to the document operation.
         *
         * @param {String|Null} oldRangesStr
         *  The string representation of the current target ranges of this rule
         *  to be added to the undo operation.
         *
         * @returns {RuleModel}
         *  A reference to this instance.
         */
        this.generateFormulaOperations = function (generator, changeDesc, newRangesStr, oldRangesStr) {

            // add the 'ranges' property to the operations if specified
            var properties = newRangesStr ? { ranges: newRangesStr } : {};
            var undoProperties = oldRangesStr ? { ranges: oldRangesStr } : {};

            // update formula expressions according to the current rule type
            var ruleAttrs = this.getMergedAttributeSet(true).rule;
            switch (ruleAttrs.type) {

                // attribute 'colorScale' contains an unlimited number of color stops
                case 'colorScale':
                    var oldColorScale = ruleAttrs.colorScale;
                    if (_.isArray(oldColorScale)) {
                        var newColorScale = oldColorScale.slice();
                        newColorScale.forEach(function (scaleStep, index) {
                            if (!_.isObject(scaleStep)) { return; }
                            tokenArrayMap.with('step' + index, function (tokenArray) {
                                var result = tokenArray.resolveOperation('op', changeDesc);
                                if (result) {
                                    scaleStep = newColorScale[index] = _.clone(scaleStep);
                                    scaleStep.v = result.new;
                                    properties.colorScale = newColorScale;
                                    undoProperties.colorScale = oldColorScale;
                                }
                            });
                        });
                    }
                    break;

                // attribute 'dataBar' contains two formula expressions for minimum/maximum
                case 'dataBar':
                    var oldDataBar = ruleAttrs.dataBar;
                    if (_.isObject(oldDataBar)) {
                        var newDataBar = _.clone(oldDataBar);
                        ['r1', 'r2'].forEach(function (propName) {
                            var dataBarRule = newDataBar[propName];
                            if (!_.isObject(dataBarRule)) { return; }
                            tokenArrayMap.with(propName, function (tokenArray) {
                                var result = tokenArray.resolveOperation('op', changeDesc);
                                if (result) {
                                    dataBarRule = newDataBar[propName] = _.clone(dataBarRule);
                                    dataBarRule.v = result.new;
                                    properties.dataBar = newDataBar;
                                    undoProperties.dataBar = oldDataBar;
                                }
                            });
                        });
                    }
                    break;

                // attribute 'iconSet' contains up to 4 formula expressions
                case 'iconSet':
                    var oldIconSet = ruleAttrs.iconSet;
                    if (_.isObject(oldIconSet)) {
                        // TODO
                    }
                    break;

                // attributes 'value1' and 'value2' for all other regular rules
                default:
                    if (ruleCategory === 'value') {
                        ['value1', 'value2'].forEach(function (propName) {
                            tokenArrayMap.with(propName, function (tokenArray) {
                                var result = tokenArray.resolveOperation('op', changeDesc);
                                if (result) {
                                    properties[propName] = result.new;
                                    undoProperties[propName] = result.old;
                                }
                            });
                        });
                    }
            }

            if (!_.isEmpty(properties)) {
                properties.id = ruleId;
                generator.generateSheetOperation(Operations.CHANGE_CFRULE, properties);
            }
            if (!_.isEmpty(undoProperties)) {
                undoProperties.id = ruleId;
                generator.generateSheetOperation(Operations.CHANGE_CFRULE, undoProperties, { undo: true });
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // destroy all class members
        this.registerDestructor(function () {
            self = docModel = sheetModel = cellCollection = null;
            numberFormatter = formulaInterpreter = tokenArrayMap = null;
        });

    } }); // class RuleModel

    // class RangeDescriptor ==================================================

    /**
     * A small helper structure that stores the target ranges of one or more
     * formatting rules, as well as the formatting rules based on these target
     * ranges.
     *
     * @constructor
     *
     * @property {RangeArray} ranges
     *  The target ranges, as array of range address objects.
     *
     * @property {String} rangesKey
     *  A unique key for the target ranges, intened to be used as map key.
     *
     * @property {Address} refAddress
     *  The reference address of the target ranges, i.e. to top-left address of
     *  the bounding range of all target ranges. Used as reference address for
     *  formulas in formatting rules.
     *
     * @property {CellValueCache} valueCache
     *  The cache containing all cell values covered by the target ranges.
     *
     * @property {SimpleMap<RuleModel>} ruleModelMap
     *  All rule models covering the target ranges, mapped by their UIDs.
     *
     * @param {Number} rangeRules
     *  The number of range rules in the rule model map.
     */
    function RangeDescriptor(sheetModel, ranges, rangesKey) {

        this.ranges = ranges;
        this.rangesKey = rangesKey;
        this.refAddress = ranges.boundary().start;
        this.valueCache = new CellValueCache(sheetModel, ranges);
        this.ruleModelMap = new SimpleMap();
        this.rangeRules = 0;

        // private properties
        this._numberFormatter = sheetModel.getDocModel().getNumberFormatter();

        // refresh the target ranges after change events of the value cache (range rules only)
        this.valueCache.on('change:cells', function () {
            if (this.rangeRules > 0) { sheetModel.refreshRanges(ranges); }
        }.bind(this));

    } // class RangeDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Updates further internal settings after the specified rule model has
     * been inserted into the conditional formatting collection.
     */
    RangeDescriptor.prototype.registerRuleModel = function (ruleModel) {
        // update the range rule count
        if (ruleModel.isRangeRule()) { this.rangeRules += 1; }
        // update the transient number formats
        var cellAttrs = Utils.getObjectOption(ruleModel.getMatchAttributeSet(true), 'cell', null);
        this._numberFormatter.registerTransientFormat(cellAttrs);
    };

    /**
     * Updates further internal settings before the specified rule model will
     * be removed from the conditional formatting collection.
     */
    RangeDescriptor.prototype.unregisterRuleModel = function (ruleModel) {
        // update the range rule count
        if (ruleModel.isRangeRule()) { this.rangeRules -= 1; }
        // update the transient number formats
        var cellAttrs = Utils.getObjectOption(ruleModel.getMatchAttributeSet(true), 'cell', null);
        this._numberFormatter.unregisterTransientFormat(cellAttrs);
    };

    /**
     * Inserts the passed rule model into this range descriptor.
     */
    RangeDescriptor.prototype.insertRuleModel = function (ruleModel) {
        // insert the rule model from the map
        this.ruleModelMap.insert(ruleModel.getRuleId(), ruleModel);
        // update internal settings for the rule depending on the rule attributes
        this.registerRuleModel(ruleModel);
    };

    /**
     * Removes the passed rule model from this range descriptor, but does not
     * destroy it.
     */
    RangeDescriptor.prototype.removeRuleModel = function (ruleModel) {
        // update internal settings for the rule depending on the rule attributes
        this.unregisterRuleModel(ruleModel);
        // remove the rule model from the map
        this.ruleModelMap.remove(ruleModel.getRuleId());
    };

    /**
     * Destroys all rule models contained in this instance, and the value
     * cache.
     */
    RangeDescriptor.prototype.destroy = function () {
        this.ruleModelMap.forEach(function (ruleModel) { ruleModel.destroy(); });
        this.valueCache.destroy();
        this.ranges = this.refAddress = this.valueCache = this.ruleModelMap = null;
        this._numberFormatter = null;
    };

    // class CondFormatCollection =============================================

    /**
     * Stores conditional formatting settings in a specific sheet. Conditional
     * formatting can be used to modify specific formatting attributes of one
     * or more cell ranges in a sheet automatically.
     *
     * Triggers the following events:
     * - 'insert:rule'
     *      After a new rule has been inserted into a formatting model of this
     *      collection. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RuleModel} ruleModel
     *          The model of the new formatting rule.
     * - 'delete:rule'
     *      Before an existing rule will be deleted from a formatting model in
     *      this collection. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RuleModel} ruleModel
     *          The model of the formatting rule to be deleted.
     * - 'change:rule'
     *      After the attributes of a rule have been changed. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RuleModel} ruleModel
     *          The model of the changed formatting rule.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var CondFormatCollection = ModelObject.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();
        var formulaParser  = docModel.getFormulaParser();

        // all target ranges descriptors (instances of RangeDescriptor), mapped by the range keys
        var rangeDescMap = new SimpleMap();

        // all rule models and their ranges models, mapped by their operation identifier
        var ruleDescMap = new SimpleMap();

        // whether the document is based on an OpenDocument file
        var odf = docModel.getApp().isODF();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns an existing descriptor for target ranges of a formatting
         * rule, or creates a new range descriptor.
         */
        function getOrCreateRangeDesc(targetRanges) {
            return rangeDescMap.getOrCreate(targetRanges.key(), function (rangesKey) {
                return new RangeDescriptor(sheetModel, targetRanges, rangesKey);
            });
        }

        /**
         * Destroys the passed range descriptor, and all associated rule
         * models, and removes it from the internal containers.
         */
        function deleteRangeDesc(rangeDesc) {

            // remove the remaining rule models from the global map
            rangeDesc.ruleModelMap.forEach(function (ruleModel, ruleId) {
                ruleDescMap.remove(ruleId);
            });

            // remove the descriptor from the global map, and destroy it
            rangeDescMap.remove(rangeDesc.rangesKey);
            rangeDesc.destroy();
        }

        /**
         * Inserts the rule model into the passed range descriptor.
         */
        function insertRuleModel(rangeDesc, ruleModel) {
            rangeDesc.insertRuleModel(ruleModel);
        }

        /**
         * Removes the rule model from the passed range descriptor, but does
         * not destroy it. If the range descriptor becomes empty, it will be
         * destroyed though.
         */
        function removeRuleModel(rangeDesc, ruleModel) {
            rangeDesc.removeRuleModel(ruleModel);
            // remove and destroy the range descriptor if no other rules remain
            if (rangeDesc.ruleModelMap.empty()) {
                deleteRangeDesc(rangeDesc);
            }
        }

        /**
         * Creates a new rule model, and inserts it into the internal rule
         * containers.
         *
         * @param {String} ruleId
         *  The unique identifier for the new rule model.
         *
         * @param {RangeArray} targetRanges
         *  The target ranges covered by the new formatting rule.
         *
         * @param {Function} generator
         *  A callback function that MUSt construct and return a new formatting
         *  rule for the passed identifier and target ranges.
         *
         * @returns {RuleModel}
         *  The new rule model returned by the generator callback function.
         */
        function createRuleModel(ruleId, targetRanges, generator) {

            // create a new cache model for the target ranges on demand
            var rangeDesc = getOrCreateRangeDesc(targetRanges);

            // first, create the descriptor entry for the new rule in the global map, so that the
            // rule model can resolve its own target ranges by identifier during construction
            var ruleDesc = { ruleModel: null, rangeDesc: rangeDesc };
            ruleDescMap.insert(ruleId, ruleDesc);

            // create and insert the new rule model
            ruleDesc.ruleModel = generator();

            // associate the rule with the target ranges, update the number of range rules
            insertRuleModel(rangeDesc, ruleDesc.ruleModel);
            return ruleDesc.ruleModel;
        }

        /**
         * Extracts the target ranges form the 'ranges' property of a document
         * operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {Boolean} [optional=false]
         *  Whether the operation property is optional.
         *
         * @returns {RangeArray|Null}
         *  The array of cell range addresses, if available; or null, if the
         *  operation property is optional and missing.
         *
         * @throws {OperationError}
         *  If the property is required but missing, or if the property value
         *  cannot be parsed to an array of range addresses.
         */
        function getTargetRanges(context, optional) {

            // extract the formula expression for the range list from the operation
            var formula = optional ? context.getOptStr('ranges', null) : context.getStr('ranges');
            if (!formula) { return null; }

            // parse the expression to an array of cell range addresses
            // Bug 46080: LO exports range lists with commas or semicolons as separator (instead of spaces) to XLSX
            // (see https://bugs.documentfoundation.org/show_bug.cgi?id=99947 for details)
            // Bug 45877: ODF inserts sheet names into the range list, ignore them silently
            var ranges = formulaParser.parseRangeList('op', formula, { extSep: true, skipSheets: odf });
            context.ensure(ranges, 'invalid target ranges');
            return ranges;
        }

        /**
         * Returns the string representation of the passed target ranges, as
         * used in document operations.
         *
         * @param {RangeArray|Range} targetRanges
         *  range to get the string representation
         * @param {String|Null} [sheetName]
         *  A custom sheet name to be added to the range list in ODF mode.
         *
         * @returns {String}
         *  The string representation of the passed target ranges, as used in
         *  document operations.
         */
        function formatTargetRanges(targetRanges, sheetName) {
            // in ODF documents, the sheet name is part of the target ranges
            sheetName = odf ? (sheetName || sheetModel.getName()) : null;
            return formulaParser.formatRangeList('op', targetRanges, { sheetName: sheetName });
        }

        /**
         * Changes the target ranges of the passed formatting rule.
         */
        function changeTargetRanges(rangeDesc, targetRanges, ruleModel) {

            // get or create the range descriptor for the target ranges of the rule
            var newRangeDesc = getOrCreateRangeDesc(targetRanges);
            if (rangeDesc === newRangeDesc) { return false; }

            // helper function that moves a single rule model to the new target ranges
            function updateRuleModel(ruleModel) {

                // move the rule model to the new range descriptor
                removeRuleModel(rangeDesc, ruleModel);
                insertRuleModel(newRangeDesc, ruleModel);

                // update target range descriptor in the rule descriptor
                var ruleDesc = ruleDescMap.get(ruleModel.getRuleId());
                ruleDesc.rangeDesc = newRangeDesc;
            }

            // process all rule models (or the passed single explicit rule model)
            if (ruleModel) {
                updateRuleModel(ruleModel);
            } else {
                rangeDesc.ruleModelMap.forEach(updateRuleModel);
            }

            return true;
        }

        /**
         * Updates the sheet indexes after the sheet collection has been
         * manipulated in the spreadsheet document.
         */
        function transformSheetHandler(event, toSheet, fromSheet) {
            ruleDescMap.forEach(function (ruleDesc) {
                ruleDesc.ruleModel.transformSheet(toSheet, fromSheet);
            });
        }

        /**
         * Recalculates the target ranges of all validation models, after cells
         * have been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {

            // Collect all changed target ranges. Bug 46442: Do not manipulate the target ranges in-place while
            // iterating the target range descriptors, this may cause to move updated rule models to wrong
            // target ranges, after they have been moved into target ranges not yet processed in the loop.
            var pendingRangeDescs = [];

            // calculate the new target ranges for all range descriptors
            rangeDescMap.forEach(function (rangeDesc) {

                // the transformed target ranges
                var targetRanges = transformTargetRanges(rangeDesc.ranges, [moveDesc]);

                // if the range array becomes empty (all cells deleted), delete the rules
                if (targetRanges.empty()) {

                    // notify event listeners, then destroy the entire descriptor with all rules
                    rangeDesc.ruleModelMap.forEach(function (ruleModel) {
                        self.trigger('delete:rule', ruleModel);
                    });
                    deleteRangeDesc(rangeDesc);

                } else {

                    // store new target ranges (bug 46442: no in-place manipulation of the target range descriptors)
                    pendingRangeDescs.push({ rangeDesc: rangeDesc, targetRanges: targetRanges });
                }
            });

            // in the end, move the rule models to the new target ranges, and notify event listeners
            pendingRangeDescs.forEach(function (entry) {
                // rescue the rule model map which will be cleared by changeTargetRanges()
                var ruleModelMap = entry.rangeDesc.ruleModelMap;
                // update the target ranges for all rule models
                changeTargetRanges(entry.rangeDesc, entry.targetRanges);
                // notify listeners about the changed rule models
                ruleModelMap.forEach(function (ruleModel) {
                    self.trigger('change:rule', ruleModel);
                });
            });
        }

        /**
         * Returns a unique rule ID for a new Rule
         * @param {String} ruleType
         *  Internal used Excel ruleTypes: 'A' old conditional format rule type, 'B' new rule type spreadsheetml_2009_9 or 'C' if it is a rule with the old and the new type.
         *
         * @returns {String} the new rule ID
         *  If the rule type is valid (A,B or C) the id looks like e.g. 'A{id}', otherwise 'id'
         */
        function getUniqueRuleId(ruleType) {
            var ruleId = null;
            var addRuleType = (ruleType === 'A') || (ruleType === 'B') || (ruleType === 'C');
            do {
                ruleId = _.uniqueId(Date.now()) + '0011001100110011001100110011110011';
                if (addRuleType) {
                    ruleId = ruleType + '{' + ruleId.slice(0, 8) + '-' + ruleId.slice(8, 12) + '-' + ruleId.slice(12, 16) + '-' + ruleId.slice(16, 20) + '-' + ruleId.slice(20, 32) + '}';
                }
            } while (ruleDescMap.has(ruleId));
            return ruleId;
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { rangeDescMap: rangeDescMap };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * contents from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {CondFormatCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            cloneData.rangeDescMap.forEach(function (rangeDesc) {
                var targetRanges = rangeDesc.ranges.clone(true);
                rangeDesc.ruleModelMap.forEach(function (ruleModel, ruleId) {
                    createRuleModel(ruleId, targetRanges, function () {
                        return ruleModel.clone(sheetModel);
                    });
                });
            });
        };

        /**
         * Callback handler for the document operation 'insertCFRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertCFRule' document operation.
         */
        this.applyInsertOperation = function (context) {

            // the rule identifier must not be used yet in the entire collection
            var ruleId = context.getStr('id');
            context.ensure(!ruleDescMap.has(ruleId), 'conditional formatting rule exists already');

            // the target ranges (sorted to ensure map hits using range array keys)
            var targetRanges = getTargetRanges(context);

            // the attributes of the rule
            var ruleAttrs = getRuleAttributes(context, docModel.getDefaultAttributes('rule'));

            // create and insert the new rule model
            var ruleModel = createRuleModel(ruleId, targetRanges, function () {
                return new RuleModel(sheetModel, ruleId, ruleAttrs);
            });

            // notify event listeners
            this.trigger('insert:rule', ruleModel);
        };

        /**
         * Callback handler for the operation 'deleteCFRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteCFRule' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the descriptor of the formatting rule
            var ruleId = context.getStr('id');
            var ruleDesc = ruleDescMap.get(ruleId, null);
            context.ensure(ruleDesc, 'missing conditional formatting rule');
            var ruleModel = ruleDesc.ruleModel;

            // notify event listeners
            self.trigger('delete:rule', ruleModel);

            // remove the rule model from its range descriptor
            removeRuleModel(ruleDesc.rangeDesc, ruleModel);

            // remove the rule model from the map, destroy the rule model
            ruleDescMap.remove(ruleId);
            ruleModel.destroy();
        };

        /**
         * Callback handler for the operation 'changeCFRule'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeCFRule' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the descriptor of the formatting rule
            var ruleId = context.getStr('id');
            var ruleDesc = ruleDescMap.get(ruleId, null);
            context.ensure(ruleDesc, 'missing conditional formatting rule');
            var ruleModel = ruleDesc.ruleModel;

            // update internal settings for the rule depending on the rule attributes
            ruleDesc.rangeDesc.unregisterRuleModel(ruleModel);

            // the new cell formatting attributes of the rule
            var ruleAttrs = getRuleAttributes(context, ruleModel.getMergedAttributeSet(true).rule);
            var changedAttrs = ruleModel.setAttributes({ rule: ruleAttrs });

            // update internal settings for the rule depending on the rule attributes
            ruleDesc.rangeDesc.registerRuleModel(ruleModel);

            // update target ranges (optional operation property)
            var targetRanges = getTargetRanges(context, true);
            var changedRanges = targetRanges && changeTargetRanges(ruleDesc.rangeDesc, targetRanges, ruleModel);

            // notify event listeners
            if (changedAttrs || changedRanges) {
                this.trigger('change:rule', ruleModel);
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the rule model with the specified identifier.
         *
         * @param {String} ruleId
         *  The identifier of a formatting rule.
         *
         * @returns {RuleModel|Null}
         *  The rule model with the specified identifier; or null, if there is
         *  no rule with the given identifier.
         */
        this.getRuleModel = function (ruleId) {
            var ruleDesc = ruleDescMap.get(ruleId, null);
            return ruleDesc ? ruleDesc.ruleModel : null;
        };

        /**
         * Creates an iterator that visits all rule models contained in this
         * collection.
         *
         * @returns {Object}
         *  An iterator object with the method next(). The result object of the
         *  iterator will contain the following properties:
         *  - {RuleModel} value
         *      The rule model currently visited.
         *  - {String} key
         *      The identifier of the rule model, as used for the property 'id'
         *      in document operations.
         */
        this.createModelIterator = function () {
            return IteratorUtils.createTransformIterator(ruleDescMap.iterator(), 'ruleModel');
        };

        /**
         * Returns the model instance representing the target cell ranges
         * covered by the formatting rule with the specified identifier.
         *
         * @param {String} ruleId
         *  The unique identifier of a formatting rule.
         *
         * @returns {CellValueCache|Null}
         *  The value cache for the target ranges covered by the specified
         *  formatting rule; or null, if the passed identifier is invalid.
         */
        this.getRangeCache = function (ruleId) {
            var ruleDesc = ruleDescMap.get(ruleId, null);
            return ruleDesc ? ruleDesc.rangeDesc.valueCache : null;
        };

        /**
         * Returns the addresses of all target ranges covered by the formatting
         * rule with the specified identifier.
         *
         * @returns {RangeArray|Null}
         *  The addresses of all target ranges of the specified formatting
         *  rule; or null, if the passed identifier is invalid.
         */
        this.getTargetRanges = function (ruleId) {
            var ruleDesc = ruleDescMap.get(ruleId, null);
            return ruleDesc ? ruleDesc.rangeDesc.ranges.clone(true) : null;
        };

        /**
         * Returns the reference addresses of the specified formatting rule.
         * The reference address is the address of the top-left cell of the
         * bounding range of all target cell ranges covered by that rule.
         *
         * @returns {Address|Null}
         *  The address of the reference cell of the specified formatting rule;
         *  or null, if the passed identifier is invalid.
         */
        this.getRefAddress = function (ruleId) {
            var ruleDesc = ruleDescMap.get(ruleId, null);
            return ruleDesc ? ruleDesc.rangeDesc.refAddress.clone() : null;
        };

        /**
         * Returns the formatting attributes to be applied at the specified
         * cell, and other settings, for all matching rules in this collection.
         *
         * @param {Address} address
         *  The address of the cell to calculate the attributes for.
         *
         * @returns {Object|Null}
         *  Null, if there are no active formatting rules covering the passed
         *  cell address; otherwise a result object with the following
         *  properties:
         *  - {Object|Null} attributeSet
         *      The attribute set with all character and cell formatting
         *      attributes to be applied to the cell. Will be null, if no rule
         *      matches and provides formatting attributes to be applied.
         *  - {Object|Null} renderProps
         *      Additional optional rendering properties, e.g. for data bars.
         */
        this.resolveRules = function (address) {

            // nothing to do in an empty collection
            if (rangeDescMap.empty()) { return null; }

            // collect all rule models covering the specified cell
            var ruleModels = rangeDescMap.reduce([], function (ruleModels, rangeDesc) {
                if (rangeDesc.ranges.containsAddress(address)) {
                    ruleModels = ruleModels.concat(rangeDesc.ruleModelMap.values());
                }
                return ruleModels;
            });
            if (ruleModels.length === 0) { return null; }

            // sort the entire list of rule models by priority (ignore their parent formatting models)
            ruleModels = _.sortBy(ruleModels, function (ruleModel) { return ruleModel.getPriority(); });

            // collect the settings of all matching rules
            var attributeSet = {};
            var renderProps = {};
            ruleModels.some(function (ruleModel) {

                // resolve the current rule, nothing to do, if it does not match at all
                var ruleResult = ruleModel.resolveRule(address);
                if (!ruleResult) { return; }

                // merge the attributes (prefer existing attributes already found before)
                if (ruleResult.attributeSet) {
                    attributeSet = docModel.extendAttributes(ruleResult.attributeSet, attributeSet);
                }

                // merge the rendering properties (prefer existing properties already found before)
                if (ruleResult.renderProps) {
                    renderProps = _.extend(ruleResult.renderProps, renderProps);
                }

                // abort evaluation of other rules if specified by the rule
                return ruleResult.stop;
            });

            // do not create a result object, if no active rule covers the passed address
            if (_.isEmpty(attributeSet)) { attributeSet = null; }
            if (_.isEmpty(renderProps)) { renderProps = null; }
            return (attributeSet || renderProps) ? { attributeSet: attributeSet, renderProps: renderProps } : null;
        };

        /**
         * Returns the complete rules to be applied at the specified
         * cell.
         *
         * @param {Range} range
         *  The range to get all rules for.
         *
         * @returns {[RuleModel]|Null}
         *  Null, if there are no rules covering the passed range
         */
        this.getRules = function (range) {

            // nothing to do in an empty collection
            if (rangeDescMap.empty()) { return null; }

            // collect all rule models covering the specified cell
            var ruleModels = rangeDescMap.reduce([], function (ruleModels, rangeDesc) {
                if (rangeDesc.ranges.overlaps(range)) {
                    ruleModels = ruleModels.concat(rangeDesc.ruleModelMap.values());
                }
                return ruleModels;
            });

            if (ruleModels.length === 0) { return null; }

            return ruleModels;
        };

        /**
         * Refreshes all target ranges of all existing formatting rules.
         *
         * @returns {CondFormatCollection}
         *  A reference to this instance.
         */
        this.refreshRanges = function () {

            // collect all target ranges covered by the rules in this collection
            var targetRanges = new RangeArray();
            rangeDescMap.forEach(function (rangeDesc) {
                targetRanges.append(rangeDesc.ranges);
            });

            // let the sheet model trigger the refresh event
            sheetModel.refreshRanges(targetRanges);
            return this;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update and restore
         * the formula expressions of the conditional formatting rules in this
         * collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateFormulaOperations = function (generator, changeDesc) {

            // update the target ranges, if the operation is located in the own sheet
            var ownSheet = sheetModel.getIndex() === changeDesc.sheet;
            // update the target ranges in ODF (stored with sheet name) when the sheet has been renamed
            var refresh = ownSheet && odf && (changeDesc.type === 'renameSheet');
            // transform the target ranges when moving cells in the sheet
            var transform = ownSheet && (changeDesc.type === 'moveCells');

            // process the rules of the different target ranges separately
            return this.iterateSliced(rangeDescMap.iterator(), function (rangeDesc) {

                // the original target ranges covered by the rules currently processed
                var targetRanges = rangeDesc.ranges;
                // whether to restore the current target ranges in the undo operation
                var restore = refresh;

                // transform the target ranges, if the moved cells are located in the own sheet
                if (transform) {

                    // restore all formatting rules, if the target ranges will be deleted implicitly
                    var transformRanges = transformTargetRanges(targetRanges, changeDesc.moveDescs);
                    if (transformRanges.empty()) {
                        var rangesStr = formatTargetRanges(targetRanges);
                        return this.iterateSliced(rangeDesc.ruleModelMap.iterator(), function (ruleModel) {
                            ruleModel.generateRestoreOperations(generator, rangesStr);
                        }, { delay: 'immediate' });
                    }

                    // restore the original target ranges on undo, if they cannot be restored implicitly
                    var restoredRanges = (targetRanges.length === transformRanges.length) ? transformTargetRanges(transformRanges, changeDesc.moveDescs, true) : null;
                    restore = restore || !restoredRanges || !restoredRanges.equals(targetRanges, true);

                    // relocate the formulas according to a changed reference address
                    var newRefAddress = restoredRanges ? restoredRanges.boundary().start : null;
                    if (newRefAddress && rangeDesc.refAddress.differs(newRefAddress)) {
                        changeDesc = _.extend({ relocate: { from: rangeDesc.refAddress, to: newRefAddress } }, changeDesc);
                    }
                }

                // generate the range lists (as strings) to be added to the operations for all rules
                var newRangesStr = refresh ? formatTargetRanges(targetRanges, changeDesc.sheetName) : null;
                var oldRangesStr = restore ? formatTargetRanges(targetRanges) : null;

                // generate all necessary operations for the formula expressions of the rule models
                return this.iterateSliced(rangeDesc.ruleModelMap.iterator(), function (ruleModel) {
                    ruleModel.generateFormulaOperations(generator, changeDesc, newRangesStr, oldRangesStr);
                }, { delay: 'immediate' });
            }, { delay: 'immediate' });
        };

        /**
         * Generates and applies insert conditional format rule operations for all passed ranges.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Object>} condFormats
         *  An array of conditional formatting rules which contains the target
         *  ranges, the excel rule type see@ getUniqueRuleId(ruleType) and the
         *  rule attributes for the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateCondFormatOperations = function (generator, condFormats) {

            return self.iterateArraySliced(condFormats, function (condFormat) {

                var ranges = RangeArray.get(condFormat.targetRanges).unify(),
                    ruleID = getUniqueRuleId(condFormat.ruleType);

                // initial operation properties
                var properties = { id: ruleID, ranges: formatTargetRanges(ranges) };

                // add the rule attributes to the operation properties
                _.extend(properties, condFormat.attrs.rule);

                generator.generateSheetOperation(Operations.DELETE_CFRULE, { id: ruleID }, { undo: true });
                generator.generateSheetOperation(Operations.INSERT_CFRULE, properties);
            }, { delay: 'immediate' });
        };

        /**
         * Generates and applies change conditional formatting operations, if the given range overlaps
         * a rule range. The give range will be subtracted from the rule range, the result is a smaller range,
         * n new rule ranges or the rule will be removed.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Range} range to change or delete a conditional formatting rule.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateReduceCondFormatRangeOperations = function (generator, range) {

            return this.iterateSliced(rangeDescMap.iterator(), function (rangeDesc) {
                return this.iterateSliced(rangeDesc.ruleModelMap.iterator(), function (ruleModel) {
                    if (ruleModel.getTargetRanges().overlaps(range)) {
                        var newRanges = ruleModel.getTargetRanges().difference(range),
                            ruleID = ruleModel.getRuleId(),
                            ruleRanges = formatTargetRanges(ruleModel.getTargetRanges());

                        if (newRanges.length > 0) {
                            // change the conditional formatting rule
                            generator.generateSheetOperation(Operations.CHANGE_CFRULE, { id: ruleID, ranges: ruleRanges }, { undo: true });
                            generator.generateSheetOperation(Operations.CHANGE_CFRULE, { id: ruleID, ranges: formatTargetRanges(newRanges) });
                        } else {
                            var properties = { id: ruleID, ranges: ruleRanges };
                            _.extend(properties,  ruleModel.getExplicitAttributes().rule);

                            generator.generateSheetOperation(Operations.INSERT_CFRULE, properties, { undo: true });
                            generator.generateSheetOperation(Operations.DELETE_CFRULE, { id: ruleModel.getRuleId() });
                        }
                    }
                }, { delay: 'immediate' });

            }, { delay: 'immediate' });
        };

        // initialization -----------------------------------------------------

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function (alreadyImported) {

            // reparse all formulas after import is finished (formulas may refer to sheets not yet imported)
            if (!alreadyImported) {
                ruleDescMap.forEach(function (ruleDesc) {
                    ruleDesc.ruleModel.refreshAfterImport();
                });
            }

            // update the sheet indexes after the sheet collection has been manipulated
            this.listenTo(docModel, 'transform:sheet', transformSheetHandler);

            // update target ranges after moving cells, or inserting/deleting columns or rows
            this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        }, this);

        // destroy all class members
        this.registerDestructor(function () {
            rangeDescMap.forEach(deleteRangeDesc);
            self = docModel = formulaParser = sheetModel = rangeDescMap = ruleDescMap = null;
        });

    } }); // class CondFormatCollection

    // exports ================================================================

    return CondFormatCollection;

});
