/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/view/view', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/view/editview',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/textdialogs',
    'io.ox/office/textframework/view/popup/universalcontextmenu',
    'io.ox/office/presentation/view/presentationcontrols',
    'io.ox/office/presentation/view/slidepreviewmixin',
    'io.ox/office/presentation/view/slidepane',
    'gettext!io.ox/office/presentation/main',
    'less!io.ox/office/presentation/view/style'
], function (Utils, Forms, AttributeUtils, EditView, AttributesToolTip, Config, Position, DOM, Labels, Dialogs, UniversalContextMenu, Controls, SlidePreviewMixin, SlidePane, gt) {

    'use strict';

    var // predefined zoom factors
        ZOOM_FACTORS = Utils.SMALL_DEVICE ? [100, 150, 200] : [35, 50, 75, 100, 150, 200, 300, 400, 600, 800];

    // class PresentationView =========================================================

    /**
     * The presentation view.
     *
     * Triggers the events supported by the base class EditView, and the
     * following additional events:
     * - 'change:zoom': When the zoom level for the displayed document was changed.
     *
     * @constructor
     *
     * @extends EditView
     *
     * @param {TextApplication} app
     *  The application containing this view instance.
     *
     * @param {TextModel} docModel
     *  The document model created by the passed application.
     */
    function PresentationView(app, docModel) {

        var // self reference
            self = this,

            // the root node of the entire application pane
            appPaneNode = null,

            // the scrollable document content node
            contentRootNode = null,

            // the app content node (child of contentRootNode)
            appContentNode = null,

            // the page node
            pageNode = null,

            //page content node
            pageContentNode = null,

            // the 'slide pane' reference
            slidepane = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // the vertical position of the application pane
            verticalScrollPosition = 0,

            // debug operations for replay
            replayOperations = null,

            // maximum osn for debug operations for replay
            replayOperationsOSN = null,

            // current zoom type (percentage or keyword)
            zoomType = (Utils.COMPACT_DEVICE) ? 'slide' : 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100,

            // outer width of page node, used for zoom calculations
            pageNodeWidth = 0,

            // outer height of page node, used for zoom calculations
            pageNodeHeight = 0,

            // the generic content menu for all types of contents
            contextMenu = null,

            // whether a vertical scroll event needs to be blocked
            // -> this is the case, if the mouse button is also pressed at the same time
            blockVerticalScrollEvent = false,

            // the handler for the vertical scroll bar events
            refreshVerticalScrollBar = $.noop,

            // storage for last know touch position
            // (to open context menu on the correct position)
            touchX = null, touchY = null,

            initOptions = {
                initHandler: initHandler,                 // modified by `SlidePreviewMixin` via function composition.
                initDebugHandler: initDebugHandler,
                initGuiHandler: initGuiHandler,
                initDebugGuiHandler: initDebugGuiHandler,
                grabFocusHandler: grabFocusHandler,
                contentScrollable: true,
                contentMargin: 30,
                enablePageSettings: true,
                userLayerShowOwnColor: true
            };

        // base constructor ---------------------------------------------------

        EditView.call(this, app, docModel, searchHandler, initOptions);

        // private methods ----------------------------------------------------

        function handleSlideSizeChange(/*evt, data*/) {
            self.setZoomType('slide');
        }

        /**
         * Handler for slide changes.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.newSlideId='']
         *      The id of the standard slide that will be activated.
         *  @param {String} [options.newLayoutId='']
         *      The id of the layout slide that will be activated.
         *  @param {String} [options.newMasterId='']
         *      The id of the master slide that will be activated.
         *  @param {String} [options.oldSlideId='']
         *      The id of the standard slide that will be deactivated.
         *  @param {String} [options.oldLayoutId='']
         *      The id of the layout slide that will be deactivated.
         *  @param {String} [options.oldMasterId='']
         *      The id of the master slide that will be deactivated.
         *  @param {Boolean} [options.isMasterChange=false]
         *      Whether only the master slide for a layout slide was changed. In this case the
         *      'options.newLayoutId' is not set, because this is also possible for inactive
         *      slides.
         */
        function handleSlideChange(event, options) {

            var // the id of the standard slide that will be activated
                slideId = Utils.getStringOption(options, 'newSlideId', ''),
                // the id of the layout slide that will be activated
                layoutId = Utils.getStringOption(options, 'newLayoutId', ''),
                // the id of the master slide that will be activated
                masterId = Utils.getStringOption(options, 'newMasterId', ''),
                // the id of the standard slide that will be deactivated
                oldSlideId = Utils.getStringOption(options, 'oldSlideId', ''),
                // the id of the layout slide that will be deactivated
                oldLayoutId = Utils.getStringOption(options, 'oldLayoutId', ''),
                // the id of the master slide that will be deactivated
                oldMasterId = Utils.getStringOption(options, 'oldMasterId', ''),
                // whether drawings layout slide shall be visible
                showLayoutSlide = slideId ? docModel.isFollowMasterShapeSlide(slideId) : true,
                // whether drawings of master slide shall be visible (can only be visible, if showLayoutSlide is true)
                showMasterSlide = showLayoutSlide ? (layoutId ? docModel.isFollowMasterShapeSlide(layoutId) : true) : false,
                // whether a (not active) layout slide got a new master slide assigned (multi selection in master view side pane)
                // -> in this case the layout ID is not defined, but the master must stay not selectable.
                isMasterChange = Utils.getBooleanOption(options, 'isMasterChange', false);

            // deactivating the old slide (together with master and layout slide)
            if (oldSlideId) { docModel.getSlideById(oldSlideId).addClass('invisibleslide'); }
            if (oldLayoutId) { docModel.getSlideById(oldLayoutId).addClass('invisibleslide'); }
            if (oldMasterId) { docModel.getSlideById(oldMasterId).addClass('invisibleslide'); }

            // activating one slide again (together with master and layout slide)
            if (slideId) { docModel.getSlideById(slideId).removeClass('invisibleslide'); }
            if (layoutId) { docModel.getSlideById(layoutId).removeClass('invisibleslide'); }
            if (masterId) { docModel.getSlideById(masterId).removeClass('invisibleslide'); }

            // making layout and master slide non-selectable, if required
            if (layoutId) { docModel.getSlideById(layoutId).toggleClass('notselectable', slideId !== ''); }
            if (masterId) { docModel.getSlideById(masterId).toggleClass('notselectable', layoutId !== '' || isMasterChange); }

            // taking care of empty place holder drawings that need border and template text
            handleEmptyPlaceHolderVisibility(event, slideId || layoutId || masterId);

            // handling the visibility of the drawings of the layout and the master slide
            // -> taking care, that a layout slide (slideId is empty), does not hide itself.
            handleLayoutMasterVisibility(layoutId, masterId, showLayoutSlide, showMasterSlide);
        }

        /**
         * Handler for all slide attribute changes.
         */
        function handleSlideAttributeChange() {
            // updating a possible change of attribute 'followMasterPage'
            if (docModel.isImportFinished()) { updateFollowMasterPageChange(); }
        }

        /**
         * After changes of slide attributes, it is required to react to changes
         * of 'followMasterShapes' attributes of slides.
         */
        function updateFollowMasterPageChange() {

            var // the id of the active slide
                slideId = docModel.getActiveSlideId(),
                // the id of the layout slide
                layoutId = null,
                // the id of the master slide
                masterId = null,
                // whether drawings of layout slide shall be visible
                showLayoutSlide = true,
                // whether drawings of master slide shall be visible
                showMasterSlide = true;

            if (!slideId) { return; } // this can happen during loading

            if (docModel.isStandardSlideId(slideId)) {
                layoutId = docModel.getLayoutSlideId(slideId);
                showLayoutSlide = docModel.isFollowMasterShapeSlide(slideId);
                if (layoutId) {
                    masterId = docModel.getMasterSlideId(layoutId);
                    // master slide can only be visible, if layout slide is visible, too
                    showMasterSlide = showLayoutSlide ? docModel.isFollowMasterShapeSlide(layoutId) : false;
                }
            } else if (docModel.isLayoutSlideId(slideId)) {
                masterId = docModel.getMasterSlideId(slideId);
                showMasterSlide = slideId ? docModel.isFollowMasterShapeSlide(slideId) : true;
            }

            // handling the visibility of the drawings of the layout and the master slide
            handleLayoutMasterVisibility(layoutId, masterId, showLayoutSlide, showMasterSlide);
        }

        /**
         * Handler for a removal of the active slide. In this it might be necessary, to hide also
         * the underlying master and layout slides.
         *
         * Info: The options object supports the properties 'id', 'layoutId' and 'masterId'. This
         *       is generated by the public model class function 'getSlideIdSet'.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.id='']
         *      The id of the standard slide that is removed.
         *  @param {String} [options.layoutId='']
         *      The id of the layout slide that will be removed or needs to be made invisible.
         *  @param {String} [options.masterId='']
         *      The id of the layout slide that will be removed or needs to be made invisible.
         */
        function hideActiveSlideCompletely(event, options) {

            var // the id of the standard slide that will be activated
                slideId = Utils.getStringOption(options, 'id', ''),
                // the id of the layout slide that will be activated
                layoutId = Utils.getStringOption(options, 'layoutId', ''),
                // the id of the master slide that will be activated
                masterId = Utils.getStringOption(options, 'masterId', '');

            // deactivating the old slide (together with master and layout slide)
            if (slideId && docModel.getSlideById(slideId)) { docModel.getSlideById(slideId).addClass('invisibleslide'); }
            if (layoutId && docModel.getSlideById(layoutId)) { docModel.getSlideById(layoutId).addClass('invisibleslide'); }
            if (masterId && docModel.getSlideById(masterId)) { docModel.getSlideById(masterId).addClass('invisibleslide'); }

            handleLayoutMasterVisibility(layoutId, masterId, false, false);
        }

        /**
         * Helper function to handle the visibility of the layout and the master slide. This is
         * required, if the slide attribute 'followMasterShapes' is modified. It is important, that
         * not the complete slide gets 'display: none' but only the drawing children. Otherwise
         * the background would also vanish. But that would not be the correct behavior.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {Boolean} showLayoutSlide
         *  Whether the specified layout slide shall be made visible or not.
         *
         * @param {Boolean} showMasterSlide
         *  Whether the specified master slide shall be made visible or not.
         */
        function handleLayoutMasterVisibility(layoutId, masterId, showLayoutSlide, showMasterSlide) {
            if (layoutId) { docModel.getSlideById(layoutId).toggleClass('invisibledrawings', !showLayoutSlide); }
            if (masterId) { docModel.getSlideById(masterId).toggleClass('invisibledrawings', !showMasterSlide); }
        }

        /**
         * Handling the visibility of empty place holder drawings, before a slide is made visible. If
         * it was inserted before, it might be possible that the class 'emptyplaceholdercheck' is still
         * set. This functionality just before showing the slide is especially important for remote clients,
         * where it is otherwise not possible to detect, when this handling of empty place holder needs
         * to be done.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {String} id
         *  The slide ID.
         */
        function handleEmptyPlaceHolderVisibility(event, id) {

            var // the jQuerified slide node
                slide = null;

            if (id && event.type === 'change:slide:before') {

                slide = docModel.getSlideById(id);

                if (slide.hasClass('emptyplaceholdercheck')) {
                    docModel.getDrawingStyles().handleAllEmptyPlaceHolderDrawingsOnSlide(slide);
                    slide.removeClass('emptyplaceholdercheck');
                }
            }
        }

        /**
         * Getting the larger of the two values 'content root node height' and the 'height of a slide' (in pixel).
         *
         * @returns {Number}
         *  The larger of the two values 'content root node height' and the 'height of a slide' (in pixel).
         */
        function getScrollStepHeight() {
            return Math.max(Utils.round((pageNode.outerHeight() * zoomFactor) / 100, 1), contentRootNode.height());
        }

        /**
         * Handling the change of the active view. It is necessary to set the height
         * of the app content node, so that the vertical scrollbar can be adapted.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         *  Optional parameters:
         *  @param {Number} [options.sliceCount=1]
         *      The number of slides in the activated view.
         */
        function handleActiveViewChange(event, options) {

            var // the number of slides in the activated view
                slideCount = Utils.getNumberOption(options, 'slideCount', 1);

            // setting the height of the app content node, so that the vertical scroll bar is adapted.
            appContentNode.height(Utils.round(getScrollStepHeight() * slideCount, 1));
        }

        /**
         * Hander for the event 'update:verticalscrollbar'. This is triggered
         * after a change of the active slide.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         *  Optional parameters:
         *  @param {Number} [options.pos=0]
         *      The index of the slide inside its container.
         */
        function handleVerticalScrollbarUpdate(event, options) {

            var // the index of the slide in its container
                index = Utils.getNumberOption(options, 'pos', 0),
                // the old margin-top caused by the slide position
                oldMarginTop = pageNode.data('scrollTopMargin') || 0,
                // the new margin-top caused by the slide position
                newMarginTop = Utils.getElementCssLength(pageNode, 'margin-top') - oldMarginTop + (getScrollStepHeight() * index);

            pageNode.css('margin-top', newMarginTop + 'px');
            contentRootNode.scrollTop(newMarginTop); // setting the vertical scroll bar position
            pageNode.data('scrollTopMargin', newMarginTop);
            verticalScrollPosition = newMarginTop;
        }

        /**
         * Moves the browser focus focus to the editor root node.
         */
        function grabFocusHandler(options) {
            if (Utils.TOUCHDEVICE) {
                var changeEvent = Utils.getOption(options, 'changeEvent', { sourceEventType: 'click' });
                var softKeyboardOpen = Utils.isSoftKeyboardOpen();
                var sourceType = changeEvent.sourceEventType;
                var node = docModel.getNode();

                if (sourceType === 'tap' && !softKeyboardOpen) {
                    Utils.focusHiddenNode();
                } else {
                    Utils.trigger('change:focus', node, node, null);
                    node.focus();
                }
            } else {
                docModel.getNode().focus();
            }
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the text
         * selection.
         */
        function scrollToSelection() {

            var // the boundaries of the text cursor
                boundRect = docModel.getSelection().getFocusPositionBoundRect();

            // make cursor position visible in application pane
            if (boundRect) {
                self.scrollToPageRectangle(boundRect);
            }
        }

        /**
         * Handler for scroll events at the content root node.
         */
        function scrollInContentRootNode() {

            var // the current scrollPosition
                scrollY = contentRootNode.scrollTop(),
                // the scroll direction
                downwards = scrollY > verticalScrollPosition,
                // the additional top margin that is required because of the scroll bar
                marginTop = 0,
                // the height of the visible page
                realPageHeight = 0,
                // whether the top border of the page is visible
                topBorderVisible = false,
                // whether the top border of the page is visible
                bottomBorderVisible = false,
                // the height of the content root node
                contentRootNodeHeight = 0,
                // the top offset of the content root node
                contentRootNodeTopOffset = 0,
                // the top offset of the page node
                pageNodeTopOffset = 0;

            if (blockVerticalScrollEvent) { return; }  // mouse down, vertical scrolling blocked

            if (scrollY === verticalScrollPosition) { return; }

            verticalScrollPosition = scrollY;  // saving the value for the next scroll event
            realPageHeight = Utils.round((pageNode.outerHeight() * zoomFactor) / 100, 1);
            pageNodeTopOffset = pageNode.offset().top;
            contentRootNodeHeight = contentRootNode.height();
            contentRootNodeTopOffset = contentRootNode.offset().top;

            topBorderVisible = pageNodeTopOffset > contentRootNodeTopOffset;
            bottomBorderVisible = (pageNodeTopOffset + realPageHeight) < (contentRootNodeTopOffset + contentRootNodeHeight);

            // switching to the next or previous slide
            if (downwards) {
                if (bottomBorderVisible) {
                    if (!docModel.changeToNeighbourSlide()) {
                        // restoring the previous position (scroll down on last slide)
                        if (contentRootNodeHeight > realPageHeight) {
                            marginTop = pageNode.data('scrollTopMargin') || 0;
                            contentRootNode.scrollTop(marginTop); // setting the vertical scroll bar position
                            verticalScrollPosition = marginTop;
                        } else {
                            contentRootNode.scrollTop(verticalScrollPosition);
                        }
                    }
                } else {
                    // saving vertical scroll position (required for the last slide)
                    verticalScrollPosition = contentRootNode.scrollTop();
                }
            } else if (!downwards && topBorderVisible) {
                docModel.changeToNeighbourSlide({ next: false });
            }

        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            docModel.getSelection().registerInsertText(options);

            //fixing the problem with an upcoming soft-keyboard
            if (Utils.TOUCHDEVICE) { scrollToSelection(); }
        }

        /**
         * Handles the 'contextmenu' event for the complete browser.
         */
        function globalContextMenuHandler(event) {
            if (_.isUndefined(event)) { return false; }

            var // was the event triggered by keyboard?
                isGlobalTrigger = null,
                // node, on which the new event should be triggered
                triggerNode     = null,
                // search inside this node for the allowed targets
                windowNode      = app.getWindowNode(),
                // parent elements on which 'contextmenu' is allowed (inside the app node)
                winTargetAreas  = ['.commentlayer', '.page', '.textdrawinglayer', '.popup-content'],
                // global target areas
                globalTargetAreas  = ['.popup-container > .popup-content'],
                // was one of these targets clicked
                targetClicked   = false,
                // the last cached key event
                lastKeyEvent = Utils.getContextKey(),
                // the slide pane container as a jQuery object
                slidePaneContainer = self.getSlidePane().getSlidePaneContainer(),
                // the selection object
                selection = docModel.getSelection();

            // detect if the context event was invoked by keyboard -> when the last keydown event before the
            // context event matches the following conditions it's a keyboard event:
            // the event is not null and the last key was the 'context' key (93) or f10 (121) and the key event
            // was invoked max. 150 ms before this context event (so the key was pressed directly before the context event)
            if ((lastKeyEvent !== null) && (lastKeyEvent.keyCode === 93 || lastKeyEvent.keyCode === 121) && (event.timeStamp - lastKeyEvent.timeStamp < 150)) {
                isGlobalTrigger = true;
            }

            /* context menu detection for the slide pane */

            // Bugfix for IE 11 only (can be deleted otherwise): In IE 11 the focus is the clicked element,
            // while in all other browsers it's the element with a tab index (here '.slide-pane-container').
            // So we need to set the focus to the '.slide-pane-container' when a child element from it was clicked.
            if (_.browser.IE < 12) {
                if (Utils.containsNode(self.getSlidePane().getNode(), event.target) === true) {
                    slidePaneContainer.focus();
                }
            }

            // when the focus is on the slidepane, trigger the contextmenu on the slide pane, when the following cases are NOT fulfilled:
            if (Utils.isElementNode(Utils.getActiveElement(), '.slide-pane-container')) {

                // On COMPACT_DEVICES a different method in the slidepane.js is used to trigger the context menu.
                // That means taphold is not used on the slide pane to trigger the context menu, because this interaction is used for drag and drop on COMPACT_DEVICES.
                if (Utils.COMPACT_DEVICE) {
                    return false;
                }

                // when event.target is NOT inside the slide pane and when it's also NOT triggered by keyboard, don't open it // TODO check comment and if
                if (Utils.containsNode(slidePaneContainer, event.target, { allowEqual: true }) === false && !isGlobalTrigger) {
                    return false;
                }

                // trigger an event on the slide pane. that invokes a context menu on the activeSlide. when it's triggered by keyboard open the menu at the side
                slidePaneContainer.trigger(new $.Event('documents:contextmenu', { sourceEvent: event, stickToSlide: isGlobalTrigger }));
                // we have a detected a context menu for the slide pane, therefore return false here
                return false;
            }

            /* context menu detection for the windowNode */

            // prevent context menu on key to open again
            if (Utils.isElementNode(Utils.getActiveElement(), '.popup-content')) {
                return false;
            }

            // check if the event target is inside of one of the allowed containers
            _.each(winTargetAreas, function (selector) {
                var parentNode = windowNode.find(selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });
            _.each(globalTargetAreas, function (selector) {
                var parentNode = $('body > ' + selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });

            // get out here, if no allowed target was clicked, or the global trigger
            // was fired (via keyboard)
            if (!targetClicked && !isGlobalTrigger) { return false; }

            // if a drawing is selected
            if (docModel.isDrawingSelected()) {
                if ($(event.target).is('.tracker')) { return false; }
                // get the (first) selected drawing as trigger node
                triggerNode = (selection.isMultiSelectionSupported() && selection.isMultiSelection()) ? selection.getFirstSelectedDrawingNode() : $(selection.getSelectedDrawing()[0]);

            } else {
                var windowSelection = window.getSelection();

                // get the current focus node to trigger on it
                _.each($('.popup-content'), function (popupContentNode) {
                    if (Utils.containsNode(popupContentNode, event.target)) { triggerNode = $(event.target); }
                });

                if (_.isNull(triggerNode) && windowSelection.type !== 'None') {
                    triggerNode = $(window.getSelection().focusNode.parentNode);

                }

                if (_.isNull(triggerNode)) {
                    Utils.warn('TextView.globalContextMenuHandler(): No "triggerNode" found. Contextmenu will not be open!');
                    return;
                }
            }

            // if we have a last known touch position, use it
            if (touchX && touchY) {
                event.pageX = touchX;
                event.pageY = touchY;

            // in case of global trigger (via keyboard), or x/y are both "0"
            // locate a meaningful position
            } else if (isGlobalTrigger || (event.pageX === 0 && event.pageY === 0) || (_.isUndefined(event.pageX) && _.isUndefined(event.pageY))) {
                // find the position of the trigger node
                var xy = Position.getPixelPositionToRootNodeOffset($('html'), triggerNode);
                // and set pageX/pageY manually
                event.pageX = (xy.x + (triggerNode.width() / 2));
                event.pageY = (xy.y + (triggerNode.height() / 2));
            }

            // trigger our own contextmennu event on the found trigger node
            triggerNode.trigger(new $.Event('documents:contextmenu', { sourceEvent: event }));

            return false;
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // deferred and debounced scroll handler, registering insertText operations immediately
                scrollToSelectionDebounced = self.createDebouncedMethod(registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500, infoString: 'Text: View: scrollToSelection' });

            // register the global event handler for 'contextmenu' event
            if (Utils.COMPACT_DEVICE) {
                app.registerGlobalEventHandler($(document), 'taphold', globalContextMenuHandler);
                app.registerGlobalEventHandler($(document), 'contextmenu', function () { return false; });
            } else {
                app.registerGlobalEventHandler($(document), 'contextmenu', globalContextMenuHandler);
            }

            // to save the position from 'touchstart', we have to register some global eventhandler
            // we need this position to open the contextmenu on the correct location
            if (Utils.IOS || _.browser.Android) {
                app.registerGlobalEventHandler($(document), 'touchstart', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = event.originalEvent.changedTouches[0].pageX;
                    touchY = event.originalEvent.changedTouches[0].pageY;
                });
                app.registerGlobalEventHandler($(document), 'touchend', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = touchY = null;
                });
            }

            // initialize other instance fields
            appPaneNode = self.getAppPaneNode();
            contentRootNode = self.getContentRootNode();
            appContentNode = self.getAppContentNode();

            // add the SlidePane
            self.addPane(slidepane = new SlidePane(self));

            // insert the editor root node into the application pane
            self.insertContentNode(docModel.getNode());

            // Page node, used for zooming
            pageNode = docModel.getNode();

            //page content node, used for zooming on small devices
            pageContentNode = $(pageNode).children('div.pagecontent');

            // handle scroll position when hiding/showing the entire application
            self.listenTo(app.getWindow(), {
                beforehide: function () {
                    saveScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    self.executeDelayed(function () { appPaneNode.css('opacity', 1); }, undefined, 'Text: View: Setting opacity');
                }
            });

            // handle selection change events
            self.listenTo(docModel, 'selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor (but not, if only a remote selection was updated (39904))
                if (!Utils.getBooleanOption(options, 'updatingRemoteSelection', false)) { scrollToSelectionDebounced(options); }
                var slideID = docModel.getActiveSlideId();
                var layoutOrMaster = docModel.isLayoutOrMasterId(slideID);
                var userData = {
                    selections: selection.getRemoteSelectionsObject(),
                    target: layoutOrMaster ? slideID : undefined
                };
                if (docModel.isHeaderFooterEditState()) {
                    userData.targetIndex = docModel.getPageLayout().getTargetsIndexInDocument(docModel.getCurrentRootNode(), docModel.getActiveTarget());
                }
                // send user data to server
                app.updateUserData(userData);

            });

            // slide size has changed - e.g. orientation change.
            self.listenTo(docModel, 'slidesize:change', handleSlideSizeChange);

            // disable drag&drop for the content root node to prevent dropping images which are loaded by the browser
            self.listenTo(self.getContentRootNode(), 'drop dragstart dragenter dragexit dragover dragleave', false);
            // Handle switch of slide inside standard view or master/layout view
            self.listenTo(docModel, 'change:slide:before', handleSlideChange);
            // Handle the removal of the active slide
            self.listenTo(docModel, 'removed:activeslide', hideActiveSlideCompletely);
            // Handle the change of a layout slide for a document slide
            self.listenTo(docModel, 'change:layoutslide', handleSlideChange);
            // Handle switch of followMasterShapes attribute
            self.listenTo(docModel, 'change:slideAttributes', handleSlideAttributeChange);

            // store the values of page width and height with paddings, after loading is finished
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            // quick fix for bug 42893
            if (_.browser.IE) { Forms.addDeviceMarkers(app.getWindowNode()); }
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            var // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1;

            // collect operations
            function collectOperations(event, operations, external) {
                if (docModel.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) { return; }

                delayedRecordingActive = true;
                self.executeDelayed(function () {
                    var // full JSON string
                        fullJSONString = replayOperations.getFieldValue(),
                        // json part string to be appended
                        jsonPartString = '',
                        // current part to be processed
                        operations = pendingOperations.splice(0, 20),
                        // first entry
                        firstEntry = (fullJSONString.length === 0);

                    _(operations).each(function (entry) {

                        // skipping operations, that were 'optimized' in 'sendActions' in the
                        // optimizeOperationHandler. Otherwise 'merged operations' are still
                        // followed by the operations, that will not be sent to
                        // the server, because they were removed by the 'optimizeOperationHandler'.
                        var skipOperation = false,
                            opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                        // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                        // only happen for internal operations (task 29601)
                        if (!entry.external) {
                            if (_.isNumber(opl) && (opl > 1)) {
                                oplCounter = opl;
                            } else if (oplCounter > 1) {
                                oplCounter--;
                                skipOperation = true;
                            }
                        }

                        if (!skipOperation) {
                            if (!firstEntry) {
                                jsonPartString += ',';
                            }
                            jsonPartString += JSON.stringify(entry.operation);
                            firstEntry = false;
                        }
                    });

                    if (fullJSONString.length >= 2) {
                        if (fullJSONString.charAt(0) === '[') {
                            fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                        }
                    }
                    fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                    replayOperations.setValue(fullJSONString);

                    delayedRecordingActive = false;
                    if (pendingOperations.length > 0) {
                        recordOperations();
                    }
                }, 50, 'Text: View: Record operations');
            }

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', 'Current selection')
                .addDebugInfoNode('selection', 'type', 'Type of the selection')
                .addDebugInfoNode('selection', 'start', 'Start position')
                .addDebugInfoNode('selection', 'end', 'End position')
                .addDebugInfoNode('selection', 'target', 'Target position')
                .addDebugInfoNode('selection', 'dir', 'Direction')
                .addDebugInfoNode('selection', 'attrs1', 'Explicit attributes of cursor position')
                .addDebugInfoNode('selection', 'attrs2', 'Explicit attributes (parent level)')
                .addDebugInfoNode('selection', 'style', 'Style sheet attributes');

            // log all selection events, and new formatting after operations
            self.listenTo(docModel, 'selection operations:success', function () {

                function debugAttrs(position, ignoreDepth, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    position = _.initial(position, ignoreDepth);
                    var element = (position && position.length) ? Position.getDOMPosition(docModel.getCurrentRootNode(), position, true) : null;
                    if (element && element.node) {
                        var attrs = AttributeUtils.getExplicitAttributes(element.node);
                        markup = '<span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>' + Utils.escapeHTML(Utils.stringifyForDebug(attrs)) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                function debugStyle(family, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    var styleId = docModel.getAttributes(family).styleId;
                    if (styleId) {
                        var attrs = docModel.getStyleCollection(family).getStyleAttributeSet(styleId);
                        markup = '<span>family=' + family + '</span> <span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>id=' + Utils.escapeHTML(styleId) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                var selection = docModel.getSelection();
                operationsPane
                    .setDebugInfoText('selection', 'type', selection.getSelectionType() + (selection.isMultiSelection() ? ' multi selection (' + selection.getMultiSelection().length + ')' : ''))
                    .setDebugInfoText('selection', 'start', selection.isMultiSelection() ? selection.getListOfPositions() : selection.getStartPosition().join(', '))
                    .setDebugInfoText('selection', 'end', selection.isMultiSelection() ? selection.getListOfPositions({ start: false }) : selection.getEndPosition().join(', '))
                    .setDebugInfoText('selection', 'target', selection.getRootNodeTarget())
                    .setDebugInfoText('selection', 'dir', selection.isTextCursor() ? 'cursor' : selection.isBackwards() ? 'backwards' : 'forwards');

                debugAttrs(selection.getStartPosition(), 0, 'attrs1');
                debugAttrs(selection.getStartPosition(), 1, 'attrs2');
                debugStyle('paragraph', 'style');
            });

            // operations to replay
            replayOperations = new Controls.TextField({ tooltip: _.noI18n('Paste operations here'), select: true });

            // operations to replay
            replayOperationsOSN = new Controls.TextField({ tooltip: _.noI18n('Define OSN for last executed operation (optional)'), width: 50 });

            // record operations after importing the document
            self.waitForImport(function () {
                self.listenTo(docModel, 'operations:after', collectOperations);
            });

            // process debug events triggered by the model
            self.listenTo(docModel, 'debug:clipboard', function (event, content) {
                clipboardPane.setClipboardContent(content);
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler(viewMenuGroup) {

            var fontToolBar             = null,
                fontStyleToolBar        = null,
                fontColorToolBar        = null,
                alignmentToolBar        = null,
                fillBorderToolBar       = null,
                listSettingsToolBar     = null,
                insertToolBar           = null,
                slideToolBar            = null,
                tableToolBar            = null,
                tableBorderToolBar      = null,
                tableStyleToolBar       = null,
                drawingToolBar          = null,
                drawingAnchorToolBar    = null,
                drawingFillToolBar      = null,
                drawingLineToolBar      = null,
                drawingGroupToolBar      = null,
                reviewToolBar           = null,
              //changeTrackingToolBar   = null,

                btnInsertTextframe      = null,

                pickFontFamily          = new Controls.FontFamilyPicker(self),
                pickFontSize            = new Controls.FontSizePicker(),
                pickLineHeight          = new Controls.LineHeightPicker(),

              //layoutSlidePicker       = new Controls.LayoutSlidePicker(self),

                pickBulletListStyle     = new Controls.BulletListStylePicker(self, docModel, { noListItemId: docModel.getNoListItemId('bullet') }),
                pickNumberedListStyle   = new Controls.NumberedListStylePicker(self, docModel, { noListItemId: docModel.getNoListItemId('numbering') }),
                pickTableSize           = new Controls.TableSizePicker({ label: /*#. a table in a text document */ gt.pgettext('text-doc', 'Table'), tooltip: /*#. insert a table in a text document */ gt.pgettext('text-doc', 'Insert a table'), maxCols: Config.MAX_TABLE_COLUMNS, maxRows: Config.MAX_TABLE_ROWS, smallerVersion: { css: { width: 50 } } }),
                pickLanguage            = new Controls.LanguagePicker({ width: 200 }),
                pickTextColor           = new Controls.TextColorPicker(self),
                insertField             = new Controls.PresentationDateNumberPicker(self),

                btnBold                 = new Controls.Button(Labels.BOLD_BUTTON_OPTIONS),
                btnItalic               = new Controls.Button(Labels.ITALIC_BUTTON_OPTIONS),
                btnUnderline            = new Controls.Button(Labels.UNDERLINE_BUTTON_OPTIONS),
                btnStrike               = new Controls.Button(Labels.STRIKEOUT_BUTTON_OPTIONS),
                btnReset                = new Controls.Button(Labels.CLEAR_FORMAT_BUTTON_OPTIONS),
                btnDecindent            = new Controls.Button({ icon: 'docs-list-dec-level', tooltip: /*#. indentation of lists (one list level up) */ gt('Demote one level'), dropDownVersion: { label: /*#. indentation of lists (one list level up) */ gt('Demote one level') } }),
                btnIncindent            = new Controls.Button({ icon: 'docs-list-inc-level', tooltip: /*#. indentation of lists (one list level down) */ gt('Promote one level'), dropDownVersion: { label: /*#. indentation of lists (one list level down) */ gt('Promote one level') } }),
                btnInsertImage          = new Controls.ImagePicker(self),
                btnInsertHyperlink      = new Controls.Button(Labels.HYPERLINK_BUTTON_OPTIONS),
                btnInsertRow            = new Controls.Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }),
                btnInsertColumn         = new Controls.Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }),
                btnDeleteRow            = new Controls.Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }),
                btnDeleteColumn         = new Controls.Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }),
                btnDeleteDrawing        = new Controls.Button(Labels.DELETE_DRAWING_BUTTON_OPTIONS),
                btnSpelling             = new Controls.Button({ icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true, dropDownVersion: { label: gt('Check spelling') } }),
                btnInsertFooter         = new Controls.Button(Labels.FOOTER_BUTTON_OPTIONS),

                grpVertAlign            = new Controls.TextPositionGroup(),

                alignmentList           = new Controls.ParagraphAlignmentPicker({ dropDownVersion: { visible: true } }),
                alignmentGroup          = new Controls.ParagraphAlignmentGroup({ smallerVersion: { pendant: alignmentList } });

            // -----------------------------------------------------
            // TABS
            //      prepare all tabs (for normal or combined panes)
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                self.createToolBarTab('format',  { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/text',       priority: 1 });
                self.createToolBarTab('insert',  { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable',            priority: 3 });
                self.createToolBarTab('slide',   { label: Labels.SLIDE_HEADER_LABEL,  visibleKey: 'document/editable',            priority: 3 });
                self.createToolBarTab('table',   { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/show/tabletab' });
                self.createToolBarTab('drawing', { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawing', priority: 2 });
                self.createToolBarTab('review',  { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'document/editable',            priority: 20, sticky: true });

            } else {
                self.createToolBarTab('format',    { label: Labels.FONT_HEADER_LABEL,   visibleKey: 'document/editable/text' });
                self.createToolBarTab('paragraph', { label: Labels.PARAGRAPH_LABEL,     visibleKey: 'document/editable/text' });
                self.createToolBarTab('insert',    { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/text' });
                self.createToolBarTab('slide',     { label: Labels.SLIDE_HEADER_LABEL,  visibleKey: 'document/editable/text' });
                self.createToolBarTab('table',     { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/show/tabletab' });
                self.createToolBarTab('drawing',   { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawing', priority: 2 });
                if (Config.SPELLING_ENABLED) {
                    self.createToolBarTab('review',         { label: Labels.REVIEW_HEADER_LABEL,         visibleKey: 'document/editable', priority: 20 });
                }
                if (!app.isODF()) {
                    self.createToolBarTab('changetracking', { label: Labels.CHANGETRACKING_HEADER_LABEL, visibleKey: 'document/editable', sticky: true });
                }
                self.createToolBarTab('comment',   { label: gt.pgettext('menu-title', 'Comments'), visibleKey: 'document/editable' });
            }

            // -----------------------------------------------------
            // TOOLBARS
            //      prepare all toolbars (for normal or combined panes)
            // -----------------------------------------------------
            fontToolBar                 = self.createToolBar('format', { priority: 1, prepareShrink: true, icon: 'fa-font', tooltip: Labels.FONT_HEADER_LABEL });
            fontStyleToolBar            = self.createToolBar('format', { priority: 2, prepareShrink: true, icon: 'docs-font-bold', caretTooltip: gt('More font styles'), splitBtn: { key: 'character/bold', options: Labels.BOLD_BUTTON_OPTIONS } });
            fontColorToolBar            = self.createToolBar('format', { priority: 3, prepareShrink: true, icon: 'docs-color', tooltip: gt('Colors') });

            if (!self.panesCombined()) {
                alignmentToolBar        = self.createToolBar('format', { priority: 4 });
                fillBorderToolBar       = self.createToolBar('format', { priority: 4, hideable: true, prepareShrink: true, icon: 'docs-border-h', tooltip: gt('Paragraph formatting') });
                listSettingsToolBar     = self.createToolBar('format', { priority: 5, prepareShrink: true, icon: 'docs-lists', tooltip: Labels.LIST_SETTINGS_LABEL });
            } else {
                alignmentToolBar        = self.createToolBar('paragraph', { priority: 1 });
                fillBorderToolBar       = self.createToolBar('paragraph', { priority: 2, prepareShrink: true, icon: 'docs-border-h', tooltip: gt('Paragraph formatting') });
                listSettingsToolBar     = self.createToolBar('paragraph', { priority: 3, prepareShrink: true, icon: 'docs-lists', tooltip: Labels.LIST_SETTINGS_LABEL });
            }

            insertToolBar               = self.createToolBar('insert');
            slideToolBar                = self.createToolBar('slide');

            tableToolBar                = self.createToolBar('table');
            tableBorderToolBar          = self.createToolBar('table', { priority: 1, hideable: true, classes: 'noSeparator' });
            tableStyleToolBar           = self.createToolBar('table', { priority: 2, hideable: true });

            drawingToolBar              = self.createToolBar('drawing');

            if (!self.panesCombined()) {
                drawingAnchorToolBar    = self.createToolBar('drawing', { visibleKey: 'document/editable/anydrawing' });
                drawingGroupToolBar     = self.createToolBar('drawing', { visibleKey: 'document/editable/anydrawing' });
                drawingLineToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/line' });
                drawingFillToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/fill', hideable: true });
            }

            if (!self.panesCombined()) {
                reviewToolBar           = self.createToolBar('review', { visibleKey: 'document/spelling/available', icon: 'docs-online-spelling', caretTooltip: /*#. tooltip for a drop-down menu with functions for reviewing the document (spelling, change tracking) */ gt('More review actions'), splitBtn: { key: 'document/onlinespelling', options: { toggle: true, tooltip: gt('Check spelling') } }, priority: 2, prepareShrink: true });
            } else {
                if (Config.SPELLING_ENABLED) {
                    reviewToolBar       = self.createToolBar('review', { visibleKey: 'document/spelling/available', icon: 'docs-online-spelling', caretTooltip: /*#. tooltip for a drop-down menu with functions for reviewing the document (spelling, change tracking) */ gt('More review actions'), splitBtn: { key: 'document/onlinespelling', options: { toggle: true, tooltip: gt('Check spelling') } } });
                }
              //changeTrackingToolBar   = self.createToolBar('changetracking');
            }

            // -----------------------------------------------------
            // CONTROLS
            //      add all controls
            // -----------------------------------------------------

            // FONT ------------------------------------------

            fontToolBar
                .addGroup('character/fontname', pickFontFamily)
                .addGap()
                .addGroup('character/fontsize', pickFontSize);

            fontStyleToolBar
                .addGroup('character/bold', btnBold)
                .addGroup('character/italic', btnItalic)
                .addGroup('character/underline', btnUnderline)
                .addSeparator({ classes: 'noVerticalSeparator' })
                .addGroup('character/format',
                    new Controls.CompoundButton(self, {
                        icon: 'docs-font-format',
                        tooltip: gt('More font styles'),
                        dropDownVersion: { visible: false }
                    })
                    .addGroup('character/strike', btnStrike)
                    .addSeparator()
                    .addGroup('character/vertalign', grpVertAlign)
                    .addSeparator()
                    .addGroup('character/reset', btnReset)
                )
                .addGroup('character/strike', btnStrike.clone({ dropDownVersion: { visible: true } }))
                .addSeparator({ classes: 'hidden' })
                .addGroup('character/vertalign', grpVertAlign.clone({ dropDownVersion: { visible: true } }))
                .addSeparator({ classes: 'hidden' })
                .addGroup('character/reset', btnReset.clone({ dropDownVersion: { visible: true } }));

            fontColorToolBar
                .addGroup('character/color',     pickTextColor);

            // PARAGRAPH ------------------------------------------

            alignmentToolBar
                .addGroup('paragraph/alignment', alignmentGroup)
                .addGroup('paragraph/alignment', alignmentList);

            fillBorderToolBar
                .addGroup('paragraph/lineheight', pickLineHeight);

            listSettingsToolBar
                .addGroup('paragraph/list/bullet', pickBulletListStyle)
                .addGap()
                .addGroup('paragraph/list/numbered', pickNumberedListStyle)
                .addGap()
                .addGroup('paragraph/list/decindent', btnDecindent)
                .addGroup('paragraph/list/incindent', btnIncindent);

            // INSERT ------------------------------------------

            insertToolBar
              //.addGroup(null,
              //    new Controls.CompoundButton(self, {
              //      //icon: 'docs-para-line-spacing-100',
              //      //tooltip: Labels.PARAGRAPH_AND_LINE_SPACING_LABEL,
              //      //dropDownVersion: { label: Labels.PARAGRAPH_AND_LINE_SPACING_LABEL }
              //
              //        icon: 'fa-clone',
              //        tooltip: _.noI18n('Insert a new slide'),
              //        dropDownVersion: { label: _.noI18n('Insert a new slide') }
              //        label: gt('New slide'),
              //
              //        updateCaptionMode: 'none',
              //        toggle: true,
              //        itemDesign: 'grid',
              //        gridColumns: 5,
              //        smallerVersion: { hideLabel: true }
              //    })
              //    .addSectionLabel(Labels.LINE_SPACING_LABEL)
              //    .addGroup('paragraph/lineheight', pickLineHeight)
              //
              //    .addSectionLabel(Labels.PARAGRAPH_SPACING_LABEL)
              //    .addGroup('paragraph/spacing', pickParagraphSpacing)
              //)
              //.addGap()
                .addGroup('layoutslidepicker/insertslide', new Controls.LayoutSlidePicker(self, {

                    // overwrites
                    tooltip: _.noI18n('Insert a new slide'),
                    label: gt('New slide'),

                    // specialized additional parameter(s)
                    splitValue: 'blank_slide'
                }))
                .addSeparator()
                .addGroup('table/insert', pickTableSize, { visibleKey: 'table/insert/available' })
                .addSeparator()
                .addGroup('image/insert/dialog', btnInsertImage)
                .addSeparator();

            if (!self.panesCombined()) {
                btnInsertTextframe = new Controls.Button(Labels.INSERT_TEXTFRAME_BUTTON_OPTIONS);

                insertToolBar
                    .addGroup('textframe/insert', btnInsertTextframe)
                    .addSeparator();

                insertToolBar
                    .addGroup('shape/insert', new Controls.ShapeTypePicker(self))
                    .addSeparator();
            }

            insertToolBar.addGroup('character/hyperlink/dialog', btnInsertHyperlink);

            if (!self.panesCombined()) {
                var btnInsertTab         = new Controls.Button({ icon: 'docs-insert-tab', label: gt('Tab stop'), tooltip: /*#. insert a horizontal tab stop into the text */ gt('Insert tab stop'), smallerVersion: { hideLabel: true } }),
                    btnInsertBreak       = new Controls.Button({ icon: 'docs-insert-linebreak', label: gt('Line break'),  tooltip: /*#. insert a manual line break into the text */ gt('Insert line break'), smallerVersion: { hideLabel: true } });

                insertToolBar
                    .addGroup('character/insert/tab',   btnInsertTab)
                    .addGroup('character/insert/break', btnInsertBreak)
                    .addGroup('document/insertfooter', btnInsertFooter)
                    .addGroup('document/insertfield', insertField);
            }

            // SLIDE ---------------------------------

            slideToolBar
                .addGroup('slide/deleteslide', new Controls.Button({ icon: 'fa-trash-o', tooltip: _.noI18n('Deleting the slide') }))
                .addSeparator()
                .addGroup('debug/followmastershapes', new Controls.Button({ icon: 'fa-file-image-o', toggle: true, tooltip: _.noI18n('Follow master shapes') }))
                .addGroup('debug/hiddenslide', new Controls.Button({ icon: 'fa-adjust', toggle: true, tooltip: _.noI18n('Hiding the slide') }))
                .addGroup('debug/resetbackground', new Controls.Button({ icon: 'fa-square-o', tooltip: _.noI18n('Remove background') }))
                .addGroup('debug/setbackgroundcolor', new Controls.FillColorPicker(self, { icon: 'docs-para-fill-color', tooltip: _.noI18n('Slide background color'), title: _.noI18n('Slide background color'), dropDownVersion: { label: _.noI18n('Slide background color') } }))
                .addGroup('debug/setbackgroundimage', new Controls.ImagePicker(self, 'debug/setbackgroundimage'))
                .addSeparator()
                .addGroup('debug/normalslideview', new Controls.Button({ icon: 'fa-file-powerpoint-o', tooltip: _.noI18n('Normal slide view') }))
                .addGroup('debug/masterslideview', new Controls.Button({ icon: 'fa-file-code-o', tooltip: _.noI18n('Master slide view / Layout slide view') }))
                .addSeparator()
                .addGroup('debug/previousslide', new Controls.Button({ icon: 'fa-arrow-left', tooltip: _.noI18n('Previous slide') }))
                .addGroup('debug/nextslide', new Controls.Button({ icon: 'fa-arrow-right', tooltip: _.noI18n('Next slide') }))
                .addSeparator();

            // TABLE ---------------------------------

            tableToolBar
                .addGroup('table/insert/row', btnInsertRow)
                .addGroup('table/delete/row', btnDeleteRow)
                .addGap()
                .addGroup('table/insert/column', btnInsertColumn)
                .addGroup('table/delete/column', btnDeleteColumn)
                .addGap();

            if (!app.isODF() && !self.panesCombined()) {

                var btnTableSplit       = new Controls.Button({ icon: 'docs-table-split', tooltip: /*#. split current table */ gt('Split table') }),
                    pickFillColorTable  = new Controls.FillColorPicker(self, { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }),
                    pickBorderModeTable = new Controls.BorderModePicker({ tooltip: Labels.CELL_BORDERS_LABEL, showInsideHor: true, showInsideVert: true }),
                    pickBorderWidth     = new Controls.BorderWidthPicker({ tooltip: gt('Cell border width') }),
                    pickTableStyle      = new Controls.TableStylePicker(self);

                tableToolBar
                    .addGroup('table/split', btnTableSplit)
                    .addSeparator()
                    .addGroup('table/fillcolor', pickFillColorTable)
                    .addGap()
                    .addGroup('table/cellborder', pickBorderModeTable);
                tableBorderToolBar
                    .addGroup('table/borderwidth', pickBorderWidth);
                tableStyleToolBar
                    .addGroup('table/stylesheet', pickTableStyle);
            }

            // DRAWING ------------------------------------------

            drawingToolBar
                .addGroup('drawing/delete', btnDeleteDrawing);

            if (!self.panesCombined()) {

                var cBoxToggleAutoFit         = new Controls.CheckBox({ label: gt('Autofit'), boxed: true, tooltip: gt('Turn automatic height resizing of text frames on or off') }),
                    cBoxToggleAutoTextHeight  = new Controls.CheckBox({ label: gt('Shrink text'), boxed: true, tooltip: gt('Turn automatic shrink text on overflow on or off') }),
                    pickBorderPresetStyle     = new Controls.BorderPresetStylePicker(app.isODF() ? Labels.BORDER_ODF_PRESET_STYLES : Labels.BORDER_OOXML_PRESET_STYLES),
                    pickBorderColor           = new Controls.BorderColorPicker(self, { label: gt('Border color'), smallerVersion: { hideLabel: true } }),
                    pickFillColorDrawing      = new Controls.FillColorPicker(self, { label: gt('Background color'),  icon: 'docs-drawing-fill-color', smallerVersion: { hideLabel: true } }),
                    pickVerticalTextAlignment = new Controls.VerticalAlignmentPicker(),
                    drawingOrder              = new Controls.DrawingOrder(self, { icon: 'docs-z-order' }),
                    cBoxToggleLockRatio       = new Controls.CheckBox({ label: gt('Lock ratio'), boxed: true, tooltip: gt('Lock or unlock the drawing aspect ratio'), ambiguous: true }),
                    drawingAlignment          = new Controls.DrawingAlignment(self, { icon: 'empty' });

                drawingAnchorToolBar
                    .addGap()
                    .addGroup('drawing/textframeautofit', cBoxToggleAutoFit)
                    .addGroup('drawing/textframeautotextheight', cBoxToggleAutoTextHeight)
                    .addGap()
                    .addGroup('drawing/lockratio', cBoxToggleLockRatio)
                    .addGap()
                    .addGroup('drawing/verticalalignment', pickVerticalTextAlignment)
                    .addGap()
                    .addGroup('drawing/order', drawingOrder)
                    .addGap()
                    .addGroup('drawing/align', drawingAlignment);

                drawingGroupToolBar
                    .addGap()
                    .addGroup('drawing/group', new Controls.Button({ icon: 'fa-object-group', toggle: true, tooltip: Labels.GROUP_DRAWING_TOOLTIP }))
                    .addGroup('drawing/ungroup', new Controls.Button({ icon: 'fa-object-ungroup', toggle: true, tooltip: Labels.UNGROUP_DRAWING_TOOLTIP }));

                drawingLineToolBar
                    .addGroup('drawing/border/style', pickBorderPresetStyle)
                    .addGroup('drawing/border/color', pickBorderColor);

                drawingFillToolBar
                    .addGroup('drawing/fill/color', pickFillColorDrawing);
            }

            // REVIEW ------------------------------------------

            if (Config.SPELLING_ENABLED) {
                reviewToolBar
                    .addGroup('document/onlinespelling', btnSpelling)
                    .addSeparator()
                    .addGroup('character/language', pickLanguage);
            }

            // the 'View' drop-down menu
            viewMenuGroup
                .addSectionLabel(Labels.ZOOM_LABEL)
                .addGroup('view/zoom/dec', new Controls.Button(Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom/inc', new Controls.Button(Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom', new Controls.PercentageLabel(), { inline: true })
                .addGroup('view/zoom/type', new Controls.Button({ label: Labels.ZOOM_SCREEN_WIDTH_LABEL, value: 'slide' }))
                .addSectionLabel(Labels.OPTIONS_LABEL)
                // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                .addGroup('view/toolbars/show', new Controls.CheckBox(Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                .addGroup('view/toolbars/sidebar', new Controls.CheckBox(Labels.SHOW_SLIDEBAR_CHECKBOX_OPTIONS))
                .addGroup('document/users', new Controls.CheckBox(Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS));

            // event handling

            /**
             * Handels the resizing from the slide on a 'refresh:layout' event.
             */
            function refreshLayoutHandler() {

                var // the remote selection object
                    remoteSelection = docModel.getRemoteSelection(),
                    // temporary page node width value
                    tempPageNodeWidth = pageNode.outerWidth();

                // fix for #33829 - if padding is reduced, dependent variables have to be updated
                pageNodeWidth = tempPageNodeWidth;
                pageNodeHeight = pageNode.outerHeight();

                if (zoomType === 'slide') {
                    self.executeDelayed(function () { self.setZoomType('slide'); }, undefined, 'Text: View: Setting zoom type');
                }

                // it might be necessary to update the slide position at the vertical scroll bar
                refreshVerticalScrollBar();

                // clean and redraw remote selections
                remoteSelection.renderCollaborativeSelections();
            }

            // Note: The delay should be higher than resizeWindowHandlerDebounced() in slidepane.js to prevent flickering while resizing with 'fit to width'.
            var refreshLayoutDebounced = this.createDebouncedMethod($.noop, refreshLayoutHandler, { delay: 100 });
            self.on('refresh:layout', refreshLayoutDebounced);
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler(viewMenuGroup) {

            viewMenuGroup
                .addGroup('debug/pagebreaks/toggle', new Controls.CheckBox({ label: _.noI18n('Show page breaks') }))
                .addGroup('debug/draftmode/toggle', new Controls.CheckBox({ label: _.noI18n('View in draft mode') }))
                .addGroup('debug/useFastLoad', new Controls.Button({ icon: 'fa-fighter-jet', label: _.noI18n('Use fast load'), toggle: true }));

            self.createToolBar('debug')
                .addGroup('document/cut',   new Controls.Button({ icon: 'fa-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                .addGroup('document/copy',  new Controls.Button({ icon: 'fa-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                .addGroup('document/paste', new Controls.Button({ icon: 'fa-paste', tooltip: _.noI18n('Paste from clipboard') }));

            self.createToolBar('debug')
                .addGroup('debug/recordoperations', new Controls.Button({ icon: 'fa-dot-circle-o', iconStyle: 'color:red;', toggle: true, tooltip: _.noI18n('Record operations') }))
                .addGroup('debug/replayoperations', new Controls.Button({ icon: 'fa-play-circle-o', tooltip: _.noI18n('Replay operations') }))
                .addGroup(null, replayOperations)
                .addGroup(null, replayOperationsOSN);
        }

        /**
         * Executes a search/replace operation, as requested by the base class
         * EditView.
         */
        function searchHandler(command, settings) {
            var search = docModel.getSearchHandler();
            switch (command) {
                case 'search:start':
                    return search.quickSearch(settings.query, 0);
                case 'search:next':
                    return search.selectNextSearchResult();
                case 'search:prev':
                    return search.selectPreviousSearchResult();
                case 'search:end':
                    return search.removeHighlighting();
                case 'replace:next':
                    return search.replaceSelectedSearchResult(settings.replace);
                case 'replace:all':
                    return search.searchAndReplaceAll(settings.query, settings.replace);
                default:
                    Utils.error('TextView.searchHandler(): unsupported search command "' + command + '"');
                    return 'internal';
            }
        }

        /**
         * calculate the optimal document scale for the slide.
         * It considers width and hight so that the whole slide fits to the visible screen.
         */
        function calculateOptimalDocumentScale() {

            var // the slide in presentation
                documentWidth = pageNode.outerWidth(),
                documentHeight = pageNode.outerHeight(),

                parentWidth = appPaneNode.width(),
                parentHeight = appPaneNode.height(),

                // the visible area, more or less...
                appContent = appPaneNode.find('.app-content'),
                scaledWidth = null,
                scaledHeight = null,

                // "3" because of the rounding issues
                outerDocumentMarginX = 3 + Utils.getElementCssLength(appContent, 'margin-left') + Utils.getElementCssLength(appContent, 'margin-right'),
                outerDocumentMarginY = 3 + Utils.getElementCssLength(appContent, 'margin-top') + Utils.getElementCssLength(appContent, 'margin-bottom'),

                // the scale ratio
                ratio;

            //landscape
            if (documentWidth > documentHeight) {

                ratio = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMarginX)));
                scaledHeight = (documentHeight * ratio);

                // when image is still not fitting stage - then we need to resize it by smaller dimension
                if (scaledHeight > (parentHeight - outerDocumentMarginY)) {
                    ratio = ((parentHeight * parentHeight) / (documentHeight * (parentHeight + outerDocumentMarginY)));
                }
            //portrait
            } else {

                ratio = ((parentHeight * parentHeight) / (documentHeight * (parentHeight + outerDocumentMarginY)));
                scaledWidth = (documentWidth * ratio);

                // when image is still not fitting stage - then we need to resize it by smaller dimension
                if (scaledWidth > (parentWidth - Utils.SCROLLBAR_WIDTH - 4 - outerDocumentMarginX)) {
                    ratio = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMarginX)));
                }
            }

            Utils.log('calculateOptimalDocumentScale', pageNode.outerWidth());
            Utils.log('ratio', ratio);
            return Math.floor(ratio * 100);
        }

        function setAdditiveMargin() {
            if (Utils.IOS) {
                //extra special code for softkeyboard behavior, we nee more space to scroll to the end of the textfile
                var dispHeight = Math.max(screen.height, screen.width);

                pageNode.css('margin-bottom', (Utils.getElementCssLength(pageNode, 'margin-bottom') + dispHeight / 3) + 'px');

                //workaround for clipped top of the first page (20)
                pageNode.css('margin-top', (Utils.getElementCssLength(pageNode, 'margin-top') + 20) + 'px');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} options
         *  @param {Boolean} [options.forceToTop=false]
         *   Whether the specified node shall be at the top border of the
         *   visible area of the scrolling node.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node, options) {
            if (Utils.getBooleanOption(options, 'regardSoftkeyboard', false)) {
                //fix for Bug 39979 on Ipad text is sometimes under the keyboard
                if (!Utils.IOS || !Utils.isSoftKeyboardOpen()) { delete options.regardSoftkeyboard; }
            }
            Utils.scrollToChildNode(contentRootNode, node, _.extend(options || {}, { padding: 15 }));
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect) {
            var options = { padding: 15 };

            Utils.scrollToPageRectangle(contentRootNode, pageRect, options);
            return this;
        };

        /**
         * Returns the slide pane.
         *
         * @returns {SlidePane}
         *  The slidepane pane of this application.
         */
        this.getSlidePane = function () {
            return slidepane;
        };

        /**
         * Provides the debug operations as string for the debugging feature
         * replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field, or null if debug mode is
         *  not active.
         */
        this.getDebugOperationsContent = function () {
            var // the operations in the required json format
                json = null;

            if (replayOperations) {
                json = replayOperations.getFieldValue();

                try {
                    json = json.length ? JSON.parse(json) : null;
                    if (_.isObject(json) && json.operations) {
                        json = json.operations;
                    }
                    return json;
                } catch (e) {
                    // error handling outside this function
                }
            }
            return null;
        };

        /**
         * Setting the maximum OSN for the operations from the operation player, that will
         * be executed.
         *
         * @returns {Number|Null}
         *  The OSN of the last operation that will be executed from the operation player.
         */
        this.getDebugOperationsOSN = function () {

            var // the operations in the required json format
                number = null;

            if (replayOperationsOSN) {

                number = replayOperationsOSN.getFieldValue();

                number = parseInt(number, 10);

                if (!(_.isNumber(number) && _.isFinite(number))) {
                    number = null;
                    replayOperationsOSN.setValue('');
                }
            }

            return number;
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'slide' (slide size is aligned to width and hight of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in percent.
         */
        this.getZoomFactor = function () {
            return zoomFactor;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'slide' (slide size is aligned to width and hight of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            Utils.log('setZoomType');
            var optimalLevel = calculateOptimalDocumentScale();

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'slide') {
                if (docModel.isDraftMode()) {
                    optimalLevel = 100;
                }
                if (Math.abs(zoomFactor - optimalLevel) > 1.1) {
                    this.increaseZoomLevel(optimalLevel);
                }
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                // the parent node of the page
                appContentRoot = $(appPaneNode).find('.app-content'),
                // an additional padding that is required to display comment layer, if overflow hidden is set
                commentLayerWidth = 0;

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (docModel.isDraftMode()) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                if (zoomInsertion < 100 && docModel.isDraftMode()) {
                    pageContentNode.css({ width: '100%' });
                } else {
                    pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                }
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    // there needs to be a padding, so that comments will still be visible
                    if (!docModel.getCommentLayer().isEmpty()) {
                        commentLayerWidth = docModel.getCommentLayer().getAppContentPaddingWidth();
                        appContentRoot.css({ paddingLeft: commentLayerWidth, paddingRight: commentLayerWidth });
                    }
                } else {
                    appContentRoot.css({ paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                // pageNode.css({
                //    'zoom': zoomInsertion / 100
                // });

                zoomType = zoomFactor;
                scrollToSelection();
            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @param {Boolean} option.clean
         *  Switching between draft and normal mode requires zoom cleanup
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor, clean) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                appContentRoot = $(appPaneNode).find('.app-content'),
                // an additional padding that is required to display comment layer, if overflow hidden is set
                commentLayerWidth = 0;

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if ((docModel.isDraftMode()) && !clean) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' + (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    // there needs to be a padding, so that comments will still be visible
                    if (!docModel.getCommentLayer().isEmpty()) {
                        commentLayerWidth = docModel.getCommentLayer().getAppContentPaddingWidth();
                        appContentRoot.css({ paddingLeft: commentLayerWidth, paddingRight: commentLayerWidth });
                    }
                } else {
                    appContentRoot.css({ paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                //  pageNode.css({
                //      'zoom': zoomInsertion / 100
                //  });

                zoomType = zoomFactor;
                scrollToSelection();

            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Creates necessary margin values when using transform scale css property
         *
         */
        this.recalculateDocumentMargin = function (options) {
            var
                marginAddition = zoomFactor < 99 ? 3 : 0,
                recalculatedMargin = (pageNodeWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition,
                // whether the (vertical) scroll position needs to be restored
                keepScrollPosition = Utils.getBooleanOption(options, 'keepScrollPosition', false),
                // an additional right margin used for comments that might be set at the page node
                commentMargin = pageNode.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageNode.data(DOM.COMMENTMARGIN_CLASS) : 0;

            if (zoomFactor !== 100) {
                // #34735 page width and height values have to be updated
                pageNodeWidth = pageNode.outerWidth();
                pageNodeHeight = pageNode.outerHeight();
            }

            if (_.browser.Chrome && zoomFactor > 100) {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (30 / (zoomFactor / 100)) + 'px ' + recalculatedMargin + 'px'
                });
            } else {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (pageNodeHeight * (zoomFactor / 100 - 1) + marginAddition) + 'px ' + recalculatedMargin + 'px'
                });
            }

            setAdditiveMargin();

            // in the end scroll to cursor position/selection (but not after event 'refresh:layout', that uses the option 'keepScrollPosition')
            if (!keepScrollPosition) { scrollToSelection(); }
        };

        /**
         * Rejects an attempt to edit the text document (e.g. due to reaching
         * the table size limits).
         *
         * @param {String} cause
         *  The cause of the rejection.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.rejectEditTextAttempt = function (cause) {

            switch (cause) {
                case 'tablesizerow':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further rows.') });
                    break;

                case 'tablesizecolumn':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further columns.') });
                    break;
            }

            return this;
        };

        this.showPageSettingsDialog = function () {
            return new Dialogs.PageSettingsDialog(this).show();
        };

        /**
         * Removes the drawinganchor marker of all paragraphs.
         */
        this.clearVisibleDrawingAnchor = function () {
            docModel.getCurrentRootNode().find('.visibleAnchor').removeClass('visibleAnchor').find('.anchorIcon').remove();
        };

        /**
         * Function, that is only required, because it might be called from field manager.
         * This is not required in presentation app.
         */
        this.hideFieldFormatPopup = function () {
            // TODO: Remove this function asap
        };

        /**
         * Function, that is only required, because it might be called in undo:after.
         * This is not required in presentation app.
         */
        this.getChangeTrackPopup = function () {
            return { hide: $.noop };
        };

        /**
         *
         * This is not  yet required in presentation app.
         */
        this.showFieldFormatPopup = function () {
            return { hide: $.noop };
        };

        /**
         *
         * This is not  yet required in presentation app.
         */
        this.hideFieldFormatPopup = function () {
            return { hide: $.noop };
        };

        /**
         * Sets the marker for the anchor
         *
         * @param {HTMLElement|jQuery} drawing
         *  The DOM element for that the anchor will be made
         *  visible.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         */
        this.setVisibleDrawingAnchor = function (drawing, attributes) {
            var oxoPosition = null,
                paragraph = null;

            // get out here, no attributes given
            if (_.isEmpty(attributes)) { return; }

            // remove all visible anchors first
            self.clearVisibleDrawingAnchor();

            // inline drawing get no visible anchor
            if (!_.has(attributes, 'drawing') || !_.has(attributes.drawing, 'inline') || attributes.drawing.inline) { return; }

            oxoPosition = Position.getOxoPosition(docModel.getCurrentRootNode(), drawing, 0);
            paragraph = $(Position.getParagraphElement(docModel.getCurrentRootNode(), _.initial(oxoPosition)));

            $(paragraph).addClass('visibleAnchor').prepend('<i class="docs-position-anchor anchorIcon helper">');
        };

        /**
         * After successful import the scroll handler for the vertical scrolling can be registered.
         */
        this.enableVerticalScrollBar = function () {

            /**
             * After a change of the layout, the height of the content node need
             * to be updated and also the margin-top of the page node.
             */
            refreshVerticalScrollBar = function (event, options) {

                // do nothing, if the document was reloaded with a snapshot
                if (Utils.getBooleanOption(options, 'documentReloaded', false)) { return; }

                // setting the height of the app content node, so that the vertical scroll bar is adapted.
                handleActiveViewChange(null, { slideCount: app.getModel().getActiveViewCount() });

                // also updating the top margin at the page for the active slide
                handleVerticalScrollbarUpdate(null, { pos: app.getModel().getActiveSlideIndex() });
            };

            // Registering vertical scroll handler
            self.listenTo(self.getContentRootNode(), 'scroll', scrollInContentRootNode);
            // Handle an update of a zoom change
            self.listenTo(self, 'change:zoom', function () {
                pageNode.removeData('scrollTopMargin'); // the saved value cannot be reused
                refreshVerticalScrollBar();
            });
            // Handle switch of active view
            self.listenTo(docModel, 'change:activeView', handleActiveViewChange);
            // Handle an update of the vertical scroll bar
            self.listenTo(docModel, 'update:verticalscrollbar', handleVerticalScrollbarUpdate);
            // inserting or deleting a slide requires also an update of the vertical scroll bar
            self.listenTo(docModel, 'inserted:slide removed:slide moved:slide', refreshVerticalScrollBar);

            // avoiding scrolling together with mouse down
            self.listenTo(self.getContentRootNode(), 'mousedown touchstart', function () { blockVerticalScrollEvent = true; });
            self.listenTo(self.getContentRootNode(), 'mouseup touchend', function () { blockVerticalScrollEvent = false; });

            // refreshing once the vertical scroll bar after successful loading
            refreshVerticalScrollBar();
        };

        // initialization -----------------------------------------------------

        // create the context menu
        contextMenu = new UniversalContextMenu(this, { delay: (_.browser.IE) ? 1 : 200 });

        // store the values of page width and height with paddings, after loading is finished
        this.waitForImportSuccess(function () {
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            if (Utils.COMPACT_DEVICE) {
                self.executeDelayed(function () {
                    var optimalDocWidth = calculateOptimalDocumentScale();
                    if (optimalDocWidth < 100) {
                        zoomType = 'slide';
                    }
                }, undefined, 'Text: View: Setting zoom type, compact device');
            }

            if (Utils.IOS) {
                var appContent = pageNode.parent();
                appContent.attr('contenteditable', true);

                var listenerList = docModel.getListenerList();
                pageNode.off(listenerList);
                appContent.on(listenerList);

                //we dont want the focus in the document on the beginning!
                Utils.focusHiddenNode();
            }

            // quick fix for bug 40053
            if (_.browser.Safari) { Forms.addDeviceMarkers(app.getWindowNode()); }

        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            app = self = replayOperations = replayOperationsOSN = docModel = null;
            appPaneNode = contentRootNode = pageNode = pageContentNode = contextMenu = null;
        });

        // presentation view mixins -------------------------------------------

        /**
         * Additional preview specific behavior for the presentation view.
         *
         * `SlidePreviewMixin` provides a set of specific functionality
         *  that is just for rendering the preview of any presentation
         *  slide that most recently has been changed.
         */
        SlidePreviewMixin.call(this, app, initOptions); // provide the presentations view context and inject
                                                        // `app` and `initOptions` each as additional state.

    } // class PresentationView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: PresentationView });
});
