/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/presentation/view/slidepane', [

    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/textframework/utils/keycodeutils',
    'io.ox/office/tk/forms',

    'io.ox/office/baseframework/view/pane',
    'io.ox/office/presentation/view/popup/slidepanecontextmenu',
    'io.ox/office/textframework/utils/dom',

    'io.ox/office/tk/utils/tracking'

], function (Utils, KeyCodes, KeyCodeUtils, Forms, Pane, SlidePaneContextMenu, DOM, Tracking) {

    'use strict';

    // class SlidePane =======================================================

    /**
     * The side pane in presentation to select slides.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {PresentationView} docView
     *  The presentation view containing this instance.
     */
    function SlidePane(docView) {

        var debugSlidePane = false;  // only for debug

        var // self reference
            self = this,
            // the application instance
            app = docView.getApp(),
            // app Model
            appModel = app.getModel(),
            // the slide-pane-container node
            slidePaneContainer = $('<div class ="slide-pane-container" tabindex="1" > </div>'),
            // drag&drop marker
            slidePlacer = $('<div class ="slidePlacer" style = "position: absolute; top: 0px; visibility: hidden; left: 10px; width: 100px; background: black; height: 3px"> </div>'), //TODO in styles in less
            // context menu for the slide pane
            contextMenu = null,
            // minWidth from the slide pane in px
            minWidth = 90,
            // maxWidth from the slide pane in px
            maxWidth = 400,
            // slide pane width in percent on tablets, landscape and portrait can have different values
            slidePaneTabletWidth = { landscapeWidth: 14, portraitWidth: 16 },
            // slide pane width in percent on small devices, landscape and portrait can have different values
            slidePaneSmallDevicesWidth = { landscapeWidth: 20, portraitWidth: 25 },

            // the width of the displayed thumbnail (.slideContent) in px at scale 1
            slideContentWidth = null,
            // the height of the displayed thumbnail (.slideContent) in px at scale 1
            slideContentHeight = 100,

            css_value__preview_scale = 'scale(1)',

            // on touch there is no visible scrollbar, therefore the width is reduced to 6 for a nicer layout
            scrollbarWidth = (!Utils.COMPACT_DEVICE) ? 18 : 6,
            // this set the offset after the slidePane thumbnail to the right in px at scale 1
            offsetRight = 10  + scrollbarWidth,
            // current offset before the slidePane thumbnail to the left
            currentOffsetLeft = null,  //TODO
            // depending on the current view the defaultOffSet is used as the currentOffsetLeft - must be equal to the 'margin-left' value from the less
            defaultOffsetLeftForMasterView = 25,
            defaultOffsetLeftForNormalView = 35,
            // total offset in px from the slidePane and slideContentWidth
            offSet = null, // .middle 35 padding left + 10 / offset to right ( 178+35+10 = 223) //TODO comment
            // slidePane width in px at scale 1
            slidePaneWidthAtScale1 = null,
            // get the windows width in which the app suite is shown
            windowWidth = null,
            // the markup for the icon that is shown on hidden slides
            hiddenMarkup = Forms.createIconMarkup('fa-minus-circle', { classes: 'hiddenSlideIcon' }),
            // name of class that marks a slide as hidden in the DOM
            hiddenClassName = 'hiddenSlide',
            // the blue print for the slideContent that is inside a slide
            slideContentBluePrint = $('<div class="slideContent" />'),
            // the blue print for a single slide
            slideBluePrint = $([
                '<div class ="slide-container">',
                '<div class = "leftSide" />',
                '<div class = "middle">',
                '<div class="slidePaneThumbnail"> ',
                hiddenMarkup,
                '</div></div></div>'].join('')),

            handleVisibleSlideRangeNotificationDebounced = $.noop,
          //handleSlideRangeVisibilityDebounced          = $.noop,

            forceRerenderSlidePaneThumbnailsDebounced    = $.noop,

            currentSlideRangeForNotification = [0, 0],
          //currentSlideRangeForVisibility   = [0, 0],

            // if a drag and drop interaction is active at the moment
            dragInteractionActiveFlag = false,
            // flag that indicates a possible drag&drop that should be checked later
            pendingDragInteractionFlag = false,
            // flag that indicates clearing the selection should be checked later
            pendingClearSelectionFlag = false,
            // the slide number were the drag and drop starts
            dragStartSlideNumber,
            // the slide number which is selected while the drag interaction in ongoing (can be from scrolling or mouse/touch movement)
            dragMoveSlideNumber,
            // flag if the context menu should be opened on TOUCH from a second click on the selected slide
            doubleTouchOnSelectedTarget,
            // the slide hight at drag start
            currentSlideHeightPx,
            // y-position at drag start
            dragStartPosY,
            // y-position while dragging
            dragMovePosY,
            // flag if there was a touchend before the touch hold to cancel the hold
            touchHoldWasCanceled = false,
            // if the drag and drop was started by mouse or touch
            clickedByTouch,
            // number of slides in the slide pane
            slideCount,
            // flag if the slide pane context menu is open
            contextIsOpen = false,
            // timer to determine if we have a touch hold
            timer = null,
            // current scale factor for slide thumbnails
            factor,
            // containing IDs from a possible multiselection
            slideSelection = [];

        // base constructor ---------------------------------------------------

        Pane.call(this, docView, { position: 'left', classes: 'slide-pane standard-design', resizable: (!(Utils.COMPACT_DEVICE)), maxSize: maxWidth, minSize: minWidth });

        // private methods ----------------------------------------------------

        /**
         * Returns the screen width on COMPACT_DEVICES or the browser width on other devices.
         * Note: On Android the orientation change event is too early, so that we would get a
         * wrong width when we would use 'Utils.getPageSize().width'.
         *
         * @returns {Number}
         *   The screen width on COMPACT_DEVICES or the browser width in percent.
         */
        function getWindowWidth() {
            if (Utils.COMPACT_DEVICE) {
                return Utils.getScreenWidth();
            } else {
                return Utils.getPageSize().width;
            }
        }

        /**
         * Returns the width in percent which should be used on COMPACT_DEVICES.
         * It considers the device orientation and gives back different value for landscape and portrait.
         *
         * @returns {Number}
         *   The width in percent according to the current orientation.
         */
        function getWidthOnCompactDevices() {
            if (Utils.SMALL_DEVICE) {
                return Utils.isPortrait() ? slidePaneSmallDevicesWidth.portraitWidth : slidePaneSmallDevicesWidth.landscapeWidth;
            } else {
                return Utils.isPortrait() ? slidePaneTabletWidth.portraitWidth : slidePaneTabletWidth.landscapeWidth;
            }
        }

        /**
         * Set the slide pane width to a given percent value.
         *
         * @param {Number} widthInPercent
         *  The width in percent.
         */
        function setSizeSlidePaneInPercent(widthInPercent) {
            // round to get integer values for pixels
            var widthInPx  =  Math.round(widthInPercent * (windowWidth / 100));
            setSizeSlidePaneInPx(widthInPx);
        }

        /**
         * Set the slide pane width to a given width in pixel. The width is restricted to a min and max width.
         *
         * Note: Triggering this function causes a 'pane:resize' event in the pane.js, which invokes
         * the 'setSizeSlidePaneThumbnails' function here to adapt all sizing values in the slide pane.
         *
         * @param {Number} widthInPx
         *  The width in pixel.
         */
        function setSizeSlidePaneInPx(widthInPx) {
            // restrict to min and max width
            self.setSize(Utils.minMax(widthInPx, minWidth, maxWidth));
        }

        /**
         * Get the current slide pane width in percent.
         *
         * @returns {Number}
         *   The slide pane width in percent.
         */
        function getWidthSlidePaneInPercent() {

            var widthInPercent = (100 / windowWidth) * getWidthSlidePane();
            return widthInPercent;
        }

        /**
         * Get the current slide pane width in pixel.
         *
         * @returns {Number}
         *   The slide pane width in pixel.
         */
        function getWidthSlidePane() {
            return self.getSize().width;
        }

        /**
         * Calculates values to set dimensions for the slide pane
         * for a given slide ratio.
         *
         * @param {Number} slideRatio
         *  The ratio from the current slide.
         *
         *  @param {Boolean} isMasterView
         *  A flag if the normal or master/slide view is active.
         */
        function calcSlidePaneDimensionsForSlideRatio(slideRatio, isMasterView) {
            slideContentWidth = slideRatio * slideContentHeight;
            //offSet = currentOffsetLeft + offsetRight;
            //slidePaneWidthAtScale1 = slideContentWidth + offSet;
            setNewSlidePaneOffset(isMasterView);
        }

        /**
         * Helper that (re)computes the currently valid scaling multiplier for any preview.
         * Preview thumbnail must fit to the 'slideContentHeight' at scale 1 from the slide pane,
         * so that the big slide out of the page fits in the small thumbnail container in the slide pane.
         */
        function computeInitialPreviewScaling() {
            css_value__preview_scale = 'scale(' + (Math.round(100000 * (slideContentHeight / Utils.convertHmmToLength(appModel.getSlideDocumentSize().height, 'px', 1))) / 100000) + ')';
        }

        /**
         * Helper that applies the currently valid scaling multiplier to any preview.
         */
        function applyPreviewScaling($preview) {
            $preview.css('transform-origin', '0 0');
            $preview.css('transform', css_value__preview_scale);
        }

        /**
         * Sets the visible left offset between thumbnail and border in the slide pane depended on the view (normal/master view).
         * It calculates all values which are depended on the 'currentOffsetLeft' to set the right dimensions for the slide pane.
         *
         *  @param {Boolean} isMasterView
         *  A flag if the normal or master/slide view is active.
         */
        function setNewSlidePaneOffset(isMasterView) {
            currentOffsetLeft = isMasterView ? defaultOffsetLeftForMasterView : defaultOffsetLeftForNormalView;
            offSet = currentOffsetLeft + offsetRight;
            slidePaneWidthAtScale1 = slideContentWidth + offSet;
        }
        // ------ render functions ------

        /**
         * Render the Slidepane.
         *
         * @param {Boolean} isMasterView
         *  A flag if the normal or master/slide view is active.
         */
        function renderSlidePane(isMasterView) {

            // clear the DOM for new rendering
            slidePaneContainer.empty();

            // classify standard view state - somehow connected to BUG-44952 [https://bugs.open-xchange.com/show_bug.cgi?id=44952]
            slidePaneContainer[isMasterView ? 'removeClass' : 'addClass']('standard-view');

            // get slideIdList from the model
            var slideIdList = isMasterView ? appModel.getMasterSlideOrder() : appModel.getStandardSlideOrder();
            // to cache the attached nodes
            var nodes = [];

            // the html template
            var templ = null;

            // render the slideIdList and insert the slide id as an attribute
            slideIdList.forEach(function (slideId, idx/*, list*/) {

                // get the blue print for one slide
                templ = slideBluePrint.clone();

                templ = createSingleSlideMarkup(templ, slideId, isMasterView);

                // visually associate each slide with its ordinal number
                templ.children('.leftSide').html('<span>' + (idx + 1) + '</span> ');

                // attach preview when it already exists in the preview-store
                insertPreviewThumbnailInSlide(slideId, templ);

                // if the current slide (see 'slideId') is hidden, attach a hide class
                if (appModel.isHiddenSlide(slideId)) { templ.addClass(hiddenClassName); }

                // add the slide to the final jQuery object
                nodes.push(templ);
            });
            slidePaneContainer.append(nodes);

            if (debugSlidePane) { Utils.log('::renderSlidePane:::', 'slideIdList', slideIdList); }
        }

        /**
        *  Inserting a preview from the preview-store in it's slide container. When it doesn't exist
        *  (e.g. isn't finished in the preview-store), nothing is added.
        */
        function insertPreviewThumbnailInSlide(slideId, $slideContainer) {
            // use preview when it exists, otherwise do nothing
            var $preview = docView.getPreview(slideId);
            if ($preview) {

                // scale the preview to fit to the in it's container at slidepane scale 1
                applyPreviewScaling($preview);

                // in case this method needs to be called by id only without being able to provide a current container reference.
                $slideContainer = $slideContainer || slidePaneContainer.children('[data-container-id="' + slideId + '"]').eq(0);

                $preview.prependTo($slideContainer.find('.slideContent').empty());
            }
        }

        /**
        *  Updating all slide previews that has been changed. These slide(s) Ids are delivered in the 'slideIdList'.
        *
        */
        function updatePreviewThumbnails(event, sideIdList) {
        //  // update visible preview slides only
        //  sideIdList = _.intersection(sideIdList, getVisibleSlideIdList());   // unfortunately this does break user experience.

            // anonymous function expression due to `insertPreviewThumbnailInSlide` featuring an arguments api
            // that does not match with the arguments api of array iterator methods.

            sideIdList.forEach(function (slideId/*, idx, list*/) { insertPreviewThumbnailInSlide(slideId); });
        }

        /**
        * Returning the markup (without the preview, that happens at a different function) for a single slide.
        * Master and layout slides have a special class. All slides have their Id attached.
        */
        function createSingleSlideMarkup(templ, slideId, isMasterView) {
            // set classes for master/layout view
            if (isMasterView) { // TODO other method to get the view here
                if (appModel.isLayoutSlideId(slideId)) {
                    templ.addClass('layout');
                } else {
                    templ.addClass('master');
                }
            }
            // attach the id to the .slide-container
            DOM.setTargetContainerId(templ, slideId);

            templ.find('.slidePaneThumbnail').eq(0).append(slideContentBluePrint.clone());

            return templ;
        }

        /**
         * Renders new slides indices from '1' to 'n' for all slides in the current view. This is necessary when a slide
         * is deleted, moved or inserted for example,.
         */
        function renderSlideIndexNumbers() {
            // new slide numbers
            slidePaneContainer.find('.leftSide').each(function (i, value) {
                // +1 because we have a 0 based index but want slide numbers
                $(value).html('<span>' + (i + 1) + '</span>');
            });
        }

        /**
         * Set a slide hidden in the slide pane. It adds/remove a hiddenSlide class according to the state.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} options
         *  Parameters:
         *
         *  @param {Boolean} [options.slideId]
         *   The ID from the slide for which the attribute was modified.
         *
         *  @param {Boolean} [options.hiddenStateChangede]
         *   A flag if the hidden state from the given slide was changed or not.
         *
         *  @param {Boolean} [options.isHidden]
         *   A flag if the given slide is hidden or not.
         *
         */
        function renderSlideHidden(event, options) {

            // only change the class when the state was really changed
            if (options.hiddenStateChanged) {

                // add/remove class according to the given state
                if (options.isHidden) {
                    slidePaneContainer.children('div[data-container-id = "' + options.slideId + '"]').addClass(hiddenClassName);
                } else {
                    slidePaneContainer.children('div[data-container-id = "' + options.slideId + '"]').removeClass(hiddenClassName);
                }
            }
        }

        /**
         * Select the slide to a given 'id' in the GUI.
         *
         * @param {String} id
         *  The slide id.
         */
        function guiSelectSlide(id) {
            // cache the old slide
            var oldSlide =  slidePaneContainer.find('.selected').attr('data-container-id');

            // unselect all slides
            slidePaneContainer.children('.selected').removeClass('selected');

            // select the given slide
            var selectedSlide = slidePaneContainer.children('div[data-container-id = "' + id + '"]');

            // it might happen, that there are no slides to select in the view so 'id' would be not valid - prevent that
            if (selectedSlide.length > 0) {

                selectedSlide.addClass('selected');

                // scroll to the current selected slide in the slidepane
                Utils.scrollToChildNode(slidePaneContainer, selectedSlide);

                // The 'slideSelection' needs to be updated when a different slide is selected/active.
                // - For a single selection (just one slide selected) it happens here. In this case the old
                //   slide must be deleted and the new must be added to reflect this change.
                // - For a multiselection the handling is different and it happens at various handler function for the multiselection.
                if ((!(multiSelectionActive())) && (slideSelection.indexOf(id) === -1)) {
                    slideSelection = _.without(slideSelection, oldSlide);
                    slideSelection.push(id);
                }

            }

            if (debugSlidePane) { Utils.log('::guiSelectSlide::', id); }
        }

        // ------ handler functions ------

        /**
        * Move a slide node from a given index to a new index position in DOM.
        *
        * @param: {Event} event
        * The event which triggered this function.
        *
        * @param {Object} movedSlideData
        *  Parameters:
        *
        *  @param {Number} from
        *   The current index of the moved slide ID.
        *
        *  @param {Number} to
        *  The new index of the moved slide ID.
        *
        *  @param {Boolean} isMasterView
        *  Whether the moved slide is in the standard or the master/layout container.
        */
        function handleMovedSlide(event, movedSlideData) {
            if (debugSlidePane) { Utils.log('::: movedSlideData', event, movedSlideData, slidePaneContainer.children().eq(movedSlideData.from), slidePaneContainer.children().eq(movedSlideData.to)); }

            // only change the view when the move operation is from the same active view
            // (important when the active view was changed and undo is called after that)
            if (appModel.isMasterView() === movedSlideData.isMasterView) {

                // all slide containers
                var slideContainer = slidePaneContainer.children('.slide-container');

                var from = slideContainer.eq(movedSlideData.from)[0];
                var to = null;

                // calculate the correct index
                if (movedSlideData.from > movedSlideData.to) {
                    to = slideContainer.eq(movedSlideData.to)[0];
                } else {
                    to = slideContainer.eq(movedSlideData.to + 1)[0];
                }
                // move the slide in DOM
                from.parentNode.insertBefore(from, to);

                renderSlideIndexNumbers(); //TODO maybe debounce

                // scroll to the moved slide, which is also the selected one, when it's outside the viewport (important for moving via keyboard arrows)
                Utils.scrollToChildNode(slidePaneContainer, slidePaneContainer.children('.selected')); //TODO maybe debounce
            }
        }

        function handleInsertSlide(event, initOptions) {
            //console.log(initOptions);

            // do nothing, if the document was reloaded with a snapshot
            if (Utils.getBooleanOption(initOptions, 'documentReloaded', false)) { return; }

            var insertPos = appModel.getSortedSlideIndexById(initOptions.id);
            var isMasterView = Utils.getBooleanOption(initOptions, 'isMasterView', false);

            var templ = createSingleSlideMarkup(slideBluePrint.clone(), initOptions.id, isMasterView);

            var targetNode = slidePaneContainer.children().eq(insertPos);

            // when no slides (targetNode) exists just append to the slidePaneContainer
            if (targetNode.length > 0) {
                targetNode.before(templ);
            } else {
                slidePaneContainer.append(templ);
            }

            //slidePaneContainer.insertBefore(templ, slidePaneContainer.children().eq(insertPos));

            // restore slide pane thumbnail size after rendering
            setSizeSlidePaneThumbnails(getWidthSlidePane()); // TODO maybe debounce
            //element.parentNode.insertBefore(newElement, element);
            renderSlideIndexNumbers();  //TODO maybe debounce
            clearSlidesInMultiSelectionDebounced();
        }

        function handleDeleteSlide(event, initOptions) {
            slidePaneContainer.children('div[data-container-id = "' + initOptions.id + '"]').remove();
            renderSlideIndexNumbers(); //TODO maybe debounce
            clearSlidesInMultiSelectionDebounced();
        }

        /**
         * Adapts the slide thumbnail size according to to the slide pane width.
         * Note: The thumbnail size is changed automatically everytime when the slide pane width is changed
         * due to the 'pane:resize' event. 'Normal', 'master' and 'layout' thumbnails have a different layout depending
         * on the activeView.
         *
         * @param {Number} slidePaneWidthinPx
         *  The slide pane width in pixel.
         */
        function setSizeSlidePaneThumbnails(slidePaneWidthinPx) {

            var layoutSlideFactor = 0.8;
            factor = ((slidePaneWidthinPx - offSet) / (slidePaneWidthAtScale1 - offSet));

            // use integer values to prevent pixel rounding errors by the browser (noticed only in win/firefox)
            var currentWidth  = Math.round(factor * slideContentWidth);
            var currentHeight = Math.round(factor * slideContentHeight);

            // for master view
            if (appModel.isMasterView()) {

                var slideLayout = slidePaneContainer.find('.layout').find('.slidePaneThumbnail');
                var slideMaster = slidePaneContainer.find('.master').find('.slidePaneThumbnail');

                var layoutSlideWidth = Math.round(currentWidth * layoutSlideFactor);
                var layoutSlideHeight = Math.round(currentHeight * layoutSlideFactor);
                var offsetToKeepSameHeightAfterLayoutScale = (currentHeight - layoutSlideHeight);
                var offsetLeftForLayoutSlides = (currentWidth - layoutSlideWidth);

                slideMaster.css({ width: currentWidth, height: currentHeight });
                slideMaster.children('.slideContent').css('transform', 'scale(' + factor + ')');

                slideLayout.css({ width: layoutSlideWidth, height: layoutSlideHeight, marginTop: (offsetToKeepSameHeightAfterLayoutScale * 0.5), marginBottom: (offsetToKeepSameHeightAfterLayoutScale * 0.5), marginLeft: offsetLeftForLayoutSlides });
                slideLayout.children('.slideContent').css('transform', 'scale(' + factor * layoutSlideFactor + ')');

                // resize slidePlacer to the current scale //TODO offset
                slidePlacer.css({ width: currentWidth + (currentOffsetLeft - 10) });

            // for normal view
            } else {
                var slideNormal = slidePaneContainer.find('.slidePaneThumbnail');
                slideNormal.css({ width: currentWidth, height: currentHeight });
                slideNormal.children('.slideContent').css('transform', 'scale(' + factor + ')');
                // resize slidePlacer to the current scale //TODO offset
                slidePlacer.css({ width: currentWidth + (currentOffsetLeft - 10) });
            }

            if (debugSlidePane) { Utils.log('setSizeSlidePaneThumbnails', slidePaneWidthinPx); }

        }

        function forceRerenderSlidePaneThumbnails() {
          //Utils.log('+++ forceRerenderSlidePaneThumbnails +++');
            var
                $collection = $('.slidePaneThumbnail > .slideContent');

            $collection.toArray().forEach(function (elmNode) {
                $(elmNode).parent().append(elmNode);
            });
        }

        /**
         * Sets the active slide to the clicked/touched slide in the model via the controller. When the active slide
         * is changed in the model fires and event automatically, which invokes observeActiveslide() to update the GUI.
         *
         * Also: On COMPACT_DEVICES tapping on a selected slide invokes the context menu.
         *
         * @param {event} event
         *  The click/touch event from the user to select the slide.
         *
         * @param {String} type
         *  If the event was triggered by 'mouse' or touch. //TODO
         *
         */
        function setActiveSlideforClickedTarget(event, type) {
            // reset it here
            doubleTouchOnSelectedTarget = false;

            // get the 'clicked' slide-container
            var selected = $(event.target).closest('.slide-container');

            // on TOUCH the context menu is invoked by a tap on the selected slide
            if (!(type === 'mouse')) {
                // the focus should be on the slidepane when the slide pane is touched
                slidePaneContainer.focus();
                // if the touched target is already selected, invoke the context menu
                if (selected.hasClass('selected')) {
                    doubleTouchOnSelectedTarget = true;
                }
            }

            // set the 'clicked' activeSlide in the model via controller
            if (selected.attr('data-container-id') !== undefined) {  // is undefined when clicking on the margin from the slide-container
                docView.executeControllerItem('slidePaneSetActiveSlide', selected.attr('data-container-id'), { preserveFocus: true });
            }

            if (debugSlidePane) { Utils.log('::setActiveSlideforClickedTarget:: target ID', selected.attr('data-container-id')); }
        }

        /**
         * Key handler for the slide pane.
         *
         * @param {event} event
         *  The key handler event.
         */
        function keyHandler(event) {

            if (KeyCodes.matchKeyCode(event, 'UP_ARROW'))   { clearSlidesInMultiSelection(); docView.executeControllerItem('slidePaneSlideUp', '', { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW')) { clearSlidesInMultiSelection(); docView.executeControllerItem('slidePaneSlideDown', '', { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'HOME'))       { clearSlidesInMultiSelection(); docView.executeControllerItem('slidePaneShowFirstSlide', '', { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'END'))        { clearSlidesInMultiSelection(); docView.executeControllerItem('slidePaneShowLastSlide', '', { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'UP_ARROW', { ctrlOrMeta: true }))   { docView.executeControllerItem('slide/moveslideupwards', getSelection(), { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { ctrlOrMeta: true })) { docView.executeControllerItem('slide/moveslidedownwards', getSelection(), { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'UP_ARROW', { shift: true }))   { handleSelectionOneStepUpDown(true); }
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { shift: true })) { handleSelectionOneStepUpDown(false); }
            if (KeyCodes.matchKeyCode(event, 'HOME', { shift: true })) { selectAllSlidesUpDownFromActiveSlide(true); }
            if (KeyCodes.matchKeyCode(event, 'END', { shift: true })) { selectAllSlidesUpDownFromActiveSlide(false); }
            if (KeyCodes.matchKeyCode(event, 'DELETE')) { docView.executeControllerItem('slide/deleteslide', '', { preserveFocus: true }); }
            if (KeyCodes.matchKeyCode(event, 'BACKSPACE')) { docView.executeControllerItem('slide/deleteslide', '', { preserveFocus: true }); }

            if (KeyCodeUtils.isCopyKeyEventKeyDown(event) || KeyCodeUtils.isPasteKeyEventKeyDown(event) || KeyCodeUtils.isCutKeyEventKeyDown(event)) {
                // copy and paste key combos needs to get through
                appModel.getSelection().setFocusIntoClipboardNode();
                appModel.grabClipboardFocus(appModel.getSelection().getClipboardNode());
            } else {
                // important!: needs event.preventDefault() for all keys, otherwise the focus may be lost
                event.preventDefault();
            }
        }

        /**
         * Expand the selection for normal or layout slides to the first or last valid slide.
         * The start position it the currently active slide.
         *
         * @param {Boolean} upwards
         *  Whether the selection should be expanded up or downwards.
         */
        function selectAllSlidesUpDownFromActiveSlide(upwards) {
            var indexActiveSlide  = appModel.getSortedSlideIndexById(appModel.getActiveSlideId());
            var slidesToFirstOrLastPosition = upwards ? indexActiveSlide : appModel.getActiveViewCount() - indexActiveSlide - 1;
            _.times(slidesToFirstOrLastPosition, function () {
                handleSelectionOneStepUpDown(upwards, { setSelectionInGui: false });
            });
            // add classes for all selected slides
            setSelectedSlides();
        }

        /**
         * Expanding the selection for normal or layout slides one step up or downwards. Master slides are not
         * a valid target here, so only consider the next normal or layout slide.
         *
         * It's possible to suppress the final GUI update. This is useful, when this function is called
         * multiple times in a different function and the GUI update should be called there to prevent
         * unnecessary GUI updates.
         *
         *
         * @param {Boolean} upwards
         *  Whether the selection should be expanded up or downwards.
         *
         * @param {Object} [initOptions]
         *  Optional parameters:
         *  @param {Boolean} [initOptions.setSelectionInGui=true]
         *  Whether the GUI update, to make the new selected slides visible, should be called in this function or not.
         */
        function handleSelectionOneStepUpDown(upwards, initOptions) {

            var // the active slide id
                active = appModel.getActiveSlideId(),
                // whether the GUI should be updated in this function or not
                setSelectionInGui = Utils.getBooleanOption(initOptions, 'setSelectionInGui', true);

            function selectNextValidSlide(upwards, addToSelection) {

                var nextSlide;

                // remove the currently active slide from selection
                if (!addToSelection) {
                    slideSelection = _.without(slideSelection, active);
                }

                // find next slide that is not a master
                nextSlide = appModel.getNextValidSlideIndex(active, { upwards: upwards });

                // when a valid next slide was found set it active
                if (nextSlide !== -1) {
                    docView.executeControllerItem('slidePaneSetActiveSlide', appModel.getIdOfSlideOfActiveViewAtIndex(nextSlide), { preserveFocus: true });

                    if (addToSelection) {
                        // if not in array push it
                        if (slideSelection.indexOf(active) === -1) {
                            slideSelection.push(active);
                        }
                        // push the new slide if not in array
                        // Important: It MUST be used the new, currently active slide,
                        // so don't use the cached 'active' here!
                        if (slideSelection.indexOf(appModel.getActiveSlideId()) === -1) {
                            slideSelection.push(appModel.getActiveSlideId()); // new active
                        }
                    }
                }
            }

            // do nothing if the currently active slide is a master slide
            if (!appModel.isMasterSlideId(active)) {

                // upwards
                if (upwards) {
                    // add to selection when the active slide is above the lowest slide
                    if (Number(_.first(getSelection().sort(function (a, b) { return a - b; }), 1)) >= Number(convertIdsToIndex([active]))) {
                        // add
                        selectNextValidSlide(upwards, true);
                    } else {
                        // remove
                        selectNextValidSlide(upwards, false);
                    }
                } else {
                // downwards
                    // add to selection when the active slide is below the highest slide
                    if (Number(_.first(getSelection().sort(function (b, a) { return a - b; }), 1)) <= Number(convertIdsToIndex([active]))) {
                        // add
                        selectNextValidSlide(upwards, true);
                    } else {
                        // remove
                        selectNextValidSlide(upwards, false);
                    }
                }

                if (setSelectionInGui) {
                    // add classes for all selected slides
                    setSelectedSlides();
                }
            }
        }

        // set the width after the slide pane was resized by the user
        function resizeFinishedHandler() {
            docView.executeControllerItem('slidePaneSetSize', getWidthSlidePaneInPercent());
        }

        // Note: The delay should be lower than refreshLayoutDebounced() in view.js to prevent flickering while resizing with 'fit to width'.
        var resizeWindowHandlerDebounced = self.createDebouncedMethod($.noop, function () {

            // temporary save the slide pane width in percent for the windowWidth before the resize
            var wpercent = getWidthSlidePaneInPercent();

            // must update the page width value here
            windowWidth = getWindowWidth();

            // set it to the same width in percent like before the windows resize
            setSizeSlidePaneInPercent(wpercent);
        }, { delay: 50 });

        /**
         * Observe the activeSlide in the model.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} viewCollector
         *  Object with information about the currently active and used slides.
         */
        function observeActiveSlide(event, viewCollector) {

            // update the slidePane GUI with the currently active slide
            guiSelectSlide(viewCollector.activeSlideId);

            if (debugSlidePane) { Utils.log('::observeActiveSlide:: ', viewCollector); }
        }

        /**
         * Render the slidePane again when the activeView is changed.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} initOptions
         *  Optional parameters:
         *
         *  @param {Boolean} [initOptions.isMasterView=false]
         *   A flag if the normal or master/slide view is active.
         *
         */
        function handleChangeActiveView(event, initOptions) {
            var
                isMasterView = Utils.getBooleanOption(initOptions, 'isMasterView', false);

            // TODO def value for force new
            renderSlidePane(isMasterView);
            // important to change the offset from the thumbnail depending on the activeView
            setNewSlidePaneOffset(isMasterView);
            // restore slide pane thumbnail size after rendering
            setSizeSlidePaneThumbnails(getWidthSlidePane()); // TODO
            // important: clear slide selection on change to layoutview/normalview
            slideSelection = [];

            handleVisibleSlideRangeNotification();

            if (debugSlidePane) { Utils.log('::handleChangeActiveView::  ', initOptions); }
        }

        // clear the multiselection
        function clearSlidesInMultiSelection() {
            var selectedSlideId = slidePaneContainer.find('.selected').attr('data-container-id');

            if (multiSelectionActive()) {
                slideSelection = [];

                // check if undefined to prevent not valid values in the selection
                if (selectedSlideId) {
                    // add just the active slide to the selection again
                    slideSelection.push(selectedSlideId);
                }
            }
            slidePaneContainer.find('.slide-container').removeClass('selection');
        }

        // clear the multiselection debounced
        var clearSlidesInMultiSelectionDebounced = self.createDebouncedMethod($.noop, function () {
            clearSlidesInMultiSelection();
        }, { delay: 200 });

        // add classes for all selected slides
        function setSelectedSlides() {
            slidePaneContainer.find('.slide-container').removeClass('selection');
            // only add classes if we have a multiselection active
            if (multiSelectionActive()) {
                _.each(slideSelection,  function (id) { slidePaneContainer.children('div[data-container-id = "' + id + '"]').addClass('selection'); });
            }
        }

        // get the slide id from it's index in the slide pane // //TODO use this? getSortedSlideIndexById??
        function getSlideIdFromIndex(index) {
            return slidePaneContainer.find('.slide-container').eq(index).attr('data-container-id');

        }

        function trackingHandler(event) {

            var // the target that is clicked
                selected = $(event.target).closest('.slide-container'),
                // indicate that the click was outside of the valid area
                clickedOnSlideContainerElement = true;

            switch (event.type) {

                case 'tracking:start':
                    // only for debug
                    if (debugSlidePane) { Utils.log('::START', event.target, event, event.pageX, 'slideSelection:', slideSelection); }

                    // flag that indicates clearing the selection should be checked later
                    pendingClearSelectionFlag = false;
                    // flag that indicates a possible drag&drop that should be checked later
                    pendingDragInteractionFlag = false;
                    // check at first click if a context menu is open
                    contextIsOpen = contextMenu.isVisible();

                    // cancel the timer on start so that the last running timer is aborted (e.g. when clicking very fast on the slide)
                    if (timer) { timer.abort(); }

                    // check that the tracking:start is not triggered on the scrollbar, it would start a unwanted drag&drop when scrolling (mousedown->drag scrollbar)
                    if (!(selected.is((slidePaneContainer.children('.slide-container'))))) { clickedOnSlideContainerElement = false; }

                    // number of currently existing slides in the slide pane
                    slideCount = slidePaneContainer.children('.slide-container').length;
                    // reset the slideNumber for moving when starting a new drag interaction
                    dragMoveSlideNumber = null;
                    // set interaction type 'touch' or 'mouse' for this drag&drop interaction
                    clickedByTouch = event.trackingType === 'touch';
                    // save y-position in the slide pane when drag starts
                    dragStartPosY = calcPos(event);
                    // the first drag move position is the start position, in case there is no movement (touchstart -> touchend)
                    dragMovePosY = dragStartPosY;
                    // get the current height at drag start for later calculations
                    currentSlideHeightPx = slidePaneContainer.children('.slide-container').outerHeight(true); //TODO first?
                    // get the slide where the drag starts
                    dragStartSlideNumber = calcSlideNumberAtPosY(true, currentSlideHeightPx, dragStartPosY, slideCount);
                    // it must be possible to calculate the position with 'calcSlideNumberAtPosY()' after the last slide
                    // to insert slides after the last slide, but that would be not valid start position, so cancel a starting drag&drop in this case
                    // also clear the multiselection when clicked below the last slide //TODO therefore maybe calc on drop?
                    if (dragStartSlideNumber >= slideCount) { appModel.trigger('slidepane:clearselection'); return ''; }

                    // [TOUCH]: to start drag&drop on TOUCH hold
                    if (clickedByTouch) {

                        // cancel drag & drop here when the user has no edit rights
                        if (!appModel.getEditMode()) { return ''; }

                        // reset the flag to init value
                        touchHoldWasCanceled = false;

                        // check if a long tap should be triggered after 750 ms
                        timer = self.executeDelayed(function () {

                            //if (debugSlidePane) { Utils.log('TOUCH IN HOLD', !touchHoldWasCanceled, dragStartPosY, dragMovePosY, ((Math.abs(dragStartPosY - dragMovePosY)) < 5), ((dragStartSlideNumber === dragMoveSlideNumber) || dragMoveSlideNumber === null)); }

                            //if the touch hold was not canceled before 750ms and the finger stays on the start slide (not moved more than 10 px, not moved outside or not moved at all)
                            if (!touchHoldWasCanceled && ((Math.abs(dragStartPosY - dragMovePosY)) < 5) && ((dragStartSlideNumber === dragMoveSlideNumber) || dragMoveSlideNumber === null)) {

                                // no drag&drop on master slides
                                if (isMasterSlideClicked(selected)) { return ''; }

                                dragInteractionActiveFlag = true;
                                setActiveSlideforClickedTarget(event, event.trackingType); //or clickedBytouch TODO
                                slidePlacer.css({ top: calcTopPosForSlide(dragStartSlideNumber, currentSlideHeightPx), visibility: 'visible'  }); // - 1 px offset to center TODO is just a workaround here

                            }
                        }, { delay: 750 });
                    }
                    // [MOUSE] + ctrl/cmd for multiselection
                    if (!clickedByTouch && !Utils.IOS && ((_.browser.MacOS ? event.metaKey : event.ctrlKey) === true) && clickedOnSlideContainerElement) {

                        // no multiselection if clicked slides is a master or when a master slide is active
                        if (!(isMasterSlideClicked(selected) || isMasterSlidesInSelection())) {
                            handleMultiSelectionOnClickedTarget(selected, dragStartSlideNumber);
                        }
                    }

                    // [MOUSE] + shift for multiselection
                    if (!clickedByTouch && !Utils.IOS && event.shiftKey === true && clickedOnSlideContainerElement) { // TODO maybe block when meta is true?

                        // no multiselection if clicked slides is a master or when a master slide is active
                        if (!(isMasterSlideClicked(selected) || isMasterSlidesInSelection())) {
                            selectSlideRangeFromActiveSlide(dragStartSlideNumber);
                        }
                    }

                    // [MOUSE] handler for a normal mousedown in the slidePane  //TODO ios mouse comment and OSX
                    if (!clickedByTouch && !Utils.IOS && ((_.browser.MacOS ? event.metaKey : event.ctrlKey) === false) && event.shiftKey === false  && clickedOnSlideContainerElement) {

                        if (multiSelectionActive()) {
                            // case 1) in a multiselection: only clear the selection and change the active slide when the
                            // target is NOT a selected slide or active slide
                            if (!(selected.hasClass('selection') || selected.hasClass('selected'))) {
                                appModel.trigger('slidepane:clearselection');
                                setActiveSlideforClickedTarget(event, event.trackingType); //or clickedBytouch TODO

                            // case 2) in a multiselection: when we have a active selection and click on a selected slide, the
                            // selection MUST be cleared on tracking:end (therefore pending), otherwise drag&drop is not possible
                            } else {
                                pendingClearSelectionFlag = true;
                            }

                        // case 3) no multiselection: clicking on the slide should select it
                        } else {
                            setActiveSlideforClickedTarget(event, event.trackingType); //or clickedBytouch TODO
                        }

                        // cancel drag & drop here when the user has no edit rights
                        if (!appModel.getEditMode()) { return ''; }

                        // no drag&drop on master slides
                        if (isMasterSlideClicked(selected)) { return ''; }

                        // important: we have a pending drag interaction in every three cases from above
                        pendingDragInteractionFlag = true;
                    }

                    if (debugSlidePane) { Utils.log('START ', dragStartSlideNumber); }
                    break;

                case 'tracking:move':
                    // position while moving
                    dragMovePosY = calcPos(event);
                    // we need it for later cases
                    dragMoveSlideNumber = calcSlideNumberAtPosY(clickedByTouch, currentSlideHeightPx, dragMovePosY, slideCount);

                    // only process moving when a drag&drop interaction is ongoing or pending
                    if (dragInteractionActiveFlag || pendingDragInteractionFlag) {

                        // stops scroll on TOUCH while dragging
                        if (event.originalEvent.cancelable) { event.originalEvent.preventDefault(); }

                        // move the sliderPlacer to the top or bottom of a slide at the current position (for touch only top position)
                        slidePlacer.css({ top: (calcTopPosForSlide(dragMoveSlideNumber, currentSlideHeightPx)) });

                        // MOUSE: make sliderPlacer visible for MOUSE interaction after the cursor has moved 5px
                        if (!clickedByTouch && !Utils.IOS && (Math.abs(dragStartPosY - dragMovePosY)) > 5) {
                            slidePlacer.css({ visibility: 'visible' });
                            // dragInteraction is not active until the slide really moved by mouse
                            dragInteractionActiveFlag = true;
                        }
                        if (debugSlidePane) { Utils.log(':::move', dragMoveSlideNumber, event, event.pageY, dragMovePosY, currentSlideHeightPx); }
                    }
                    break;

                case 'tracking:end':
                    // set flag to cancel the touch hold
                    touchHoldWasCanceled = true;
                        // there can't be a drag interaction after tracking:end so set to false again
                        //pendingDragInteractionFlag = false;

                    // very important to prevent ghost clicks on touch (jquery mobile)
                    // it must be prevented on touchend, because on touchstart it would disable scrolling
                    if (event.originalEvent.cancelable) { event.originalEvent.preventDefault(); }

                    // use the last dragMoveSlideNumber, when there was no move event use the start position
                    var dragEndSlideNumber = (dragMoveSlideNumber === null) ? dragStartSlideNumber : dragMoveSlideNumber;

                    // TOUCH: when no drag interaction was active, check on touchend if the context menu should be opened
                    // (not when moved more than 5px and if the touch start is more than 500ms in the past)

                    //console.log('tapp', !dragInteractionActiveFlag, clickedByTouch, ((Math.abs(dragStartPosY - dragMovePosY)) < 5), (Math.abs(event.timeStamp - event.startTime))); //TODO
                    if (!dragInteractionActiveFlag && clickedByTouch && ((Math.abs(dragStartPosY - dragMovePosY)) < 5) && (Math.abs(event.timeStamp - event.startTime) < 500)) {

                        setActiveSlideforClickedTarget(event, event.trackingType);

                        if (doubleTouchOnSelectedTarget && ((dragStartSlideNumber - dragEndSlideNumber) === 0)) {

                            // check if there is a context menu open, when not trigger it
                            if (!contextIsOpen) {
                                // open context menu for the slide pane
                                slidePaneContainer.trigger(new $.Event('documents:contextmenu', { sourceEvent: event, stickToSlide: true }));
                            }
                        }
                    }

                    // resolves the pending clear selection on mouseup, when clicked on a already selected slide in a multiselection
                    if (!dragInteractionActiveFlag && pendingClearSelectionFlag) {
                        appModel.trigger('slidepane:clearselection');
                        setActiveSlideforClickedTarget(event, event.trackingType); //or clickedBytouch TODO
                    }

                   // when a drag interaction was active, finalize the drag & drop interaction here
                    if (dragInteractionActiveFlag) {
                        slidePlacer.css({ visibility: 'hidden' });

                        // only for debug
                        if (debugSlidePane) { Utils.log('BEFORE MOVE', slideSelection, getSelection(), 'start', dragStartSlideNumber, 'end', dragEndSlideNumber); }

                        docView.executeControllerItem('slide/move', { indexStart: getSelection(), indexEnd: dragEndSlideNumber }, { preserveFocus: true }); // TODO preservefocus here or not?

                        // drag & drop finished
                        dragInteractionActiveFlag = false;
                    }

                    if (debugSlidePane) { Utils.log(':::move END', dragStartSlideNumber, dragEndSlideNumber,  event); }
                    break;

                case 'tracking:scroll':
                    if (debugSlidePane) { Utils.log(':::scroll', dragInteractionActiveFlag, event); }

                    // only process if a drag&drop interaction is ongoing
                    if (dragInteractionActiveFlag) {
                        if (event.scrollY) {
                            slidePaneContainer.scrollTop(slidePaneContainer.scrollTop() + event.scrollY);

                            // scrolling changes the position in the slide pane, therefore we need to update the dragMoveSlideNumber
                            dragMoveSlideNumber = calcSlideNumberAtPosY(clickedByTouch, currentSlideHeightPx, calcPos(event), slideCount);
                            // move the sliderPlacer to the top or bottom of a slide at the current position (for touch only top position)
                            slidePlacer.css({ top: (calcTopPosForSlide(dragMoveSlideNumber, currentSlideHeightPx)) });
                        }
                    }
                    break;

                case 'tracking:cancel':

                    // cancel drag & drop
                    dragInteractionActiveFlag = false;
                    // set flag to cancel the touch hold
                    touchHoldWasCanceled = true;

                    slidePlacer.css({ visibility: 'hidden' });
                    break;
            }
        }

        // y-position from the mouse/touch in the slide pane
        function calcPos(event) {
            return event.pageY + slidePaneContainer.scrollTop() - slidePaneContainer.offset().top;
        }
        // pos in px for a given slide
        function calcTopPosForSlide(slideNumber, slideHeight) {
            var topPosSlide = ((slideNumber * slideHeight) - slidePaneContainer.scrollTop());

            // MASTER/LAYOUT VIEW: we need to correct the offset between differently scaled master and layout
            // slides to get the centered position between them (e.g. take a look at the slidePlacer position while drag&drop)
            if (appModel.isMasterView()) {

                var slideContainer = slidePaneContainer.children('.slide-container');

                // when we are on a layout slide before a master slide - additionally the first slide MUST not be corrected
                if (slideContainer.eq(slideNumber).hasClass('master') && slideNumber !== 0) {
                    //topPosSlide =  topPosSlide - (5 * factor);
                    topPosSlide -= (5 * factor);
                }
                // when we are on a layout slide slide after a master slide
                if (slideContainer.eq(slideNumber - 1).hasClass('master')) {
                    //topPosSlide =  topPosSlide + (5 * factor);
                    topPosSlide += (5 * factor);
                }
            }

            return topPosSlide;
        }

        function calcSlideNumberAtPosY(useSlideBorderToCalc, slideHeight, posY, slideCount) {
            // use the top/bottom or the middle from the slide to calculate the slide number
            var slideNumber = useSlideBorderToCalc ? Math.floor(posY / slideHeight) : Math.round(posY / slideHeight);

            // limit to first and last slide number to get valid values
            if (slideNumber < 0) { slideNumber = 0; }
            if (slideNumber > slideCount) {
                slideNumber = slideCount;
            }

            return slideNumber;
        }

        // returns if we currently have a multiselection
        function multiSelectionActive() {
            return (slideSelection.length > 1);
        }

        // returns if a master slide in in the current selection //TODO maybe model functions and not jquery
        function isMasterSlidesInSelection() {
            return slidePaneContainer.find('.selected').hasClass('master');
        }

        // returns if the target is a master slide
        function isMasterSlideClicked(selected) {  //get TODO doppelt? checkSlideAtIndexIsMaster?? maybe model functions and not jquery
            return selected.hasClass('master');
        }

        // handle clicks on a target slide (Selected = dom, dragStartSlideNumber = index) and decides
        // if a selection should be cleared completely, deleted or added for the target slide
        function handleMultiSelectionOnClickedTarget(selected, dragStartSlideNumber) {
            // deselect
            if (selected.hasClass('selected')) {
                if (multiSelectionActive()) {
                    slideSelection = _.without(slideSelection, getSlideIdFromIndex(dragStartSlideNumber));
                    docView.executeControllerItem('slidePaneSetActiveSlide', String(_.last(slideSelection, 1)), { preserveFocus: true });
                }

            } else {
                // if not in array push it
                if (slideSelection.indexOf(getSlideIdFromIndex(dragStartSlideNumber)) === -1) {
                    slideSelection.push(getSlideIdFromIndex(dragStartSlideNumber));
                    docView.executeControllerItem('slidePaneSetActiveSlide', getSlideIdFromIndex(dragStartSlideNumber), { preserveFocus: true });

                // if in object delete it
                } else {
                    slideSelection = _.without(slideSelection, getSlideIdFromIndex(dragStartSlideNumber));
                }
            }
            // add classes for all selected slides
            setSelectedSlides();
        }

        // Selects a range of slides from the active slide to the given index ('dragStartSlideNumber').
        // Master slides are not selected when they are in between the start and end point.
        function selectSlideRangeFromActiveSlide(dragStartSlideNumber) {

            var // all slide-containers in the active view
                containers = slidePaneContainer.children('.slide-container'),
                // slide index were the selection starts
                selectionStart = null,
                // marker for selection end
                selectionMarker = dragStartSlideNumber;

            /* if (false) {
                // give the last element to selectionStart as start as Index
                selectionStart = containers.filter(_.last(slideSelection, 1)).index();
            } else {*/

            // index from the active slide
            selectionStart = containers.filter('.selected').index();

            // upwards
            if (selectionStart > selectionMarker) {
                while (selectionStart >= selectionMarker) {

                    if (slideSelection.indexOf(getSlideIdFromIndex(selectionStart)) !== -1) {
                        slideSelection = _.without(slideSelection, getSlideIdFromIndex(selectionStart));
                    }
                    // don't add it when the index is a master slide
                    if (!checkSlideAtIndexIsMaster(selectionStart)) {
                        slideSelection.push(getSlideIdFromIndex(selectionStart));
                        containers.eq(selectionStart).addClass('selection'); // TODO  setSelectedSlides(); ?
                    }
                    selectionStart--;
                }
            // downwards
            } else {
                while (selectionStart <= selectionMarker) {
                    if (slideSelection.indexOf(getSlideIdFromIndex(selectionStart)) !== -1) {
                        slideSelection = _.without(slideSelection, getSlideIdFromIndex(selectionStart));
                    }
                    // don't add it when the index is a master slide
                    if (!checkSlideAtIndexIsMaster(selectionStart)) {
                        slideSelection.push(getSlideIdFromIndex(selectionStart));
                        containers.eq(selectionStart).addClass('selection'); // TODO setSelectedSlides(); ?
                    }
                    selectionStart++;
                }
            }

            docView.executeControllerItem('slidePaneSetActiveSlide', getSlideIdFromIndex(dragStartSlideNumber), { preserveFocus: true });
            // add classes for all selected slides
            setSelectedSlides();
        }

        // check if the the slide at the given index is a master // TODO CHECK MODEL is exist if not move to model
        function checkSlideAtIndexIsMaster(index) {
            if (appModel.isMasterSlideId(appModel.getIdOfSlideOfActiveViewAtIndex(index))) {
                return true;
            } else {
                return false;
            }
        }

        // convert an selection array with IDs to index positions and return it
        function convertIdsToIndex(slideSelection) {

            var arrayIndex = [];
        // old   _.each(slideSelection, function (id) { arrayIndex.push(slidePaneContainer.children('div[data-container-id = "' + id + '"]').index()); });
            _.each(slideSelection, function (id) {
                arrayIndex.push(appModel.getSortedSlideIndexById(id));
            });

            return arrayIndex;
        }

        /**
         * Returning the slide selection in the slide pane.
         *
         * @returns {Number[]}
         *  An Array containing the position as index from all selected slides in the slide pane.
         */
        function getSelection() {
            //functionCheckModel();
            return convertIdsToIndex(slideSelection);
        }

        // function refreshSlideRangeVisibility(firstVisibleIndex, lastVisibleIndex) {
        //     var
        //         $children = slidePaneContainer.children(),
        //         itemCount = $children.length,
        //         idx       = (firstVisibleIndex - 1);
        //
        //     while (++idx <= lastVisibleIndex) {
        //         $children.eq(idx).css('visibility', '');
        //     }
        //     idx = -1;
        //
        //     while (++idx < firstVisibleIndex) {
        //         $children.eq(idx).css('visibility', 'hidden');
        //     }
        //     idx = lastVisibleIndex;
        //
        //     while (++idx < itemCount) {
        //         $children.eq(idx).css('visibility', 'hidden');
        //     }
        // }

        function getVisibleSlideRange() {
            var
                $elm          = slidePaneContainer,
                elm           = $elm[0],

                boxHeight     = elm.offsetHeight,
                scrollHeight  = elm.scrollHeight,
                scrollTop     = elm.scrollTop,

                $children     = $elm.children(),

                itemCount     = $children.length,
                itemHeight,

                scrollBottom,

                firstIndex,
                lastIndex,

                slideRange    = [];

            if (itemCount) {
                itemHeight    = $children[itemCount - 1].offsetHeight; // last item's height

                scrollBottom  = (scrollHeight - boxHeight - scrollTop);

                firstIndex    = Math.floor(scrollTop / itemHeight);
                lastIndex     = itemCount - 1 - Math.floor(scrollBottom / itemHeight);

                slideRange    = [firstIndex, lastIndex];
            }
        //  if (debugSlidePane) {
        //    Utils.log('+++ on scroll +++ getVisibleSlideRange - boxHeight, scrollHeight, scrollTop, scrollBottom, itemCount, itemHeight : ', boxHeight, scrollHeight, scrollTop, scrollBottom, itemCount, itemHeight);
        //  }
        //  if (debugSlidePane) {
        //    Utils.log('+++ on scroll +++ getVisibleSlideRange - firstIndex, lastIndex : ', firstIndex, lastIndex);
        //  }
            return slideRange;
        }

        function getVisibleSlideIdList(slideRange) {

            slideRange = (_.isArray(slideRange) && slideRange) || getVisibleSlideRange();

            return (
                (slideRange.length === 2)

                ? (appModel[appModel.isMasterView() ? 'getMasterSlideOrder' : 'getStandardSlideOrder']().slice(slideRange[0], (slideRange[1] + 1)))
                : []
            );
        }

        function handleVisibleSlideRangeNotification() {
            var
                slideRange  = getVisibleSlideRange(),
                firstIndex  = slideRange[0],
                lastIndex   = slideRange[1];

            if (
                (slideRange.length === 2) &&
                ((firstIndex !== currentSlideRangeForNotification[0]) || (lastIndex !== currentSlideRangeForNotification[1]))
            ) {
                currentSlideRangeForNotification = [firstIndex, lastIndex];

                self.trigger('visiblesliderange:changed', currentSlideRangeForNotification, getVisibleSlideIdList(currentSlideRangeForNotification));
            }
        }

        function handleInitialPreviewThumbRendering() {
          //if (debugSlidePane) { Utils.log('+++ slidepane :: handleInitialPreviewThumbRendering +++ '); }
            Utils.log('+++ slidepane :: handleInitialPreviewThumbRendering +++ ');

            handleVisibleSlideRangeNotification();
        }

    //  function handleSlideRangeVisibility() {
    //      var
    //          slideRange  = getVisibleSlideRange(),
    //          firstIndex  = slideRange[0],
    //          lastIndex   = slideRange[1];
    //
    //      if (
    //          (slideRange.length === 2) &&
    //          ((firstIndex !== currentSlideRangeForVisibility[0]) || (lastIndex !== currentSlideRangeForVisibility[1]))
    //      ) {
    //         currentSlideRangeForVisibility = [firstIndex, lastIndex];
    //
    //         refreshSlideRangeVisibility(firstIndex, lastIndex);
    //     }
    // }

        function handleSlideSizeChange(evt, data) {
            Utils.info('+++ "slidesize:change" +++ [evt, data] : ', evt, data);

            // init values for the slide pane to a given slide ratio
            calcSlidePaneDimensionsForSlideRatio(data.slideRatio, appModel.isMasterView());

            computeInitialPreviewScaling(); // recalculate any preview's css scaling value `css_value__preview_scale`

            // set initial slide pane width ...
            setSizeSlidePaneThumbnails(getWidthSlidePane()); // -> intial width was already set in 'setSidePaneWidthDuringLoad'

            /**
             *  the next two lines are supposed to be the best solution from a model point of view.
             *
             *  nevertheless due to performance issues not only a container's size/width measures need to be reassigned
             *  but also some of its content css values (that of course are tasks that still belong to and will be continued
             *  to be handled by the slidepreview mixin)
             */
        //  currentSlideRangeForNotification = [0, 0];  //  - reset to initial state in order to retrieve properly
        //  handleVisibleSlideRangeNotification();      //    rerendered previews independently from the current
        //                                              //    view and scroll state

            // ... do also apply the new scaling value to each preview within its new width/height context.
            slidePaneContainer.find('.slideContent').children('.page').toArray().forEach(function (elm) {

                $(elm).css(this);             // - apply current page size context provided by invoking `getCurrentPageSize`.
                applyPreviewScaling($(elm));  // - apply the new scaling value.

            }, (function getCurrentPageSize() {
                var
                    $pageNode = appModel.getNode(),
                    elmPage   = $pageNode[0];

                return {
                    width:  (elmPage.offsetWidth  + 'px'),
                    height: (elmPage.offsetHeight + 'px')
                };

            }()));

            forceRerenderSlidePaneThumbnailsDebounced();
        }

        /**
         * Initialization from the SlidePane. The activeSlide and activeView is set directly by the event.
         *
         * Info: The width of the slide pane was already set during 'updateDocumentFormatting' to avoid
         *       jumping of the already visible slide, when the slide pane is inserted.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} initOptions
         *  Optional parameters:
         *
         *  @param {Boolean} [initOptions.activeSlideId]
         *   The currently active slide, at the moment the SlidePane was invoked by the source event.
         *
         *  @param {Boolean} [initOptions.isMasterView=false]
         *   A flag if the normal or master/slide view is active, at the moment the SlidePane was invoked by the source event.
         *
         */
        function initSlidePane(event, initOptions) {
            if (debugSlidePane) { Utils.log('::initSlidePane::Start', !(Utils.IOS || Utils.CHROME_ON_ANDROID)); }

            var // the activeSlide id
                activeSlideId = Utils.getStringOption(initOptions, 'activeSlideId'),
                // if normal oder master/slide view is active
                isMasterView = Utils.getBooleanOption(initOptions, 'isMasterView', false),
                // slide ratio from the loaded presentation
                slideRatio = Utils.getNumberOption(initOptions, 'slideRatio');

            // init values for the slide pane to a given slide ratio
            calcSlidePaneDimensionsForSlideRatio(slideRatio, isMasterView);

            computeInitialPreviewScaling();

            // init the windows width
            windowWidth = getWindowWidth();

            //init important listeners
            initListener();

            // render the html
            renderSlidePane(isMasterView, true);

            // set initial slide pane width
            setSizeSlidePaneThumbnails(getWidthSlidePane()); // -> intial width was already set in 'setSidePaneWidthDuringLoad'

            // since there is no change:slide' event after the initialization, the selection must be set here initially.
            guiSelectSlide(activeSlideId);

            contextMenu = new SlidePaneContextMenu(docView, slidePaneContainer);
        }

        /**
         * Initialization from event listener after initialization.
         */
        function initListener() {

            // slide size has changed - e.g. orientation change.
            self.listenTo(appModel, 'slidesize:change', handleSlideSizeChange);

            // slide change
            self.listenTo(appModel, 'change:slide', observeActiveSlide);

            // activeView change (between normal and master/layout)
            self.listenTo(appModel, 'change:activeView', handleChangeActiveView);
            self.listenTo(appModel, 'moved:slide', handleMovedSlide);

            // update the view after insert/remove slide
            self.listenTo(appModel, 'inserted:slide', handleInsertSlide);
            self.listenTo(appModel, 'removed:slide', handleDeleteSlide);
            // when a slide is set hidden/unhidden
            self.listenTo(appModel, 'change:slideAttributes', renderSlideHidden);
            self.listenTo(docView, 'previewupdate:after', updatePreviewThumbnails);

            // handle key events on the slidepane (when focused)
            slidePaneContainer.on('keydown', keyHandler);

            // this is important mostly for Safari, to get cut, copy, paste events on slidepane
            slidePaneContainer.on('beforecopy beforecut beforepaste', function () {
                appModel.getSelection().setFocusIntoClipboardNode();
                appModel.grabClipboardFocus(appModel.getSelection().getClipboardNode());
            });

            // mousemove have to be preventDefault() or the focus can be lost on some mouse interactions on the pane
            slidePaneContainer.on('mousemove', function (e) { e.preventDefault(); });

            // when a slide is clicked with the right mouse button set the focus to the slide pane and select the slide TODO check contextmenu is opened in view
            slidePaneContainer.on('contextmenu', function (e) {
                // not on compact devices
                if (!Utils.COMPACT_DEVICE) {

                    if (Utils.isElementNode($(e.target), '.slide-pane-container')) {
                        appModel.trigger('slidepane:clearselection');
                        slidePaneContainer.focus();
                        docView.executeControllerItem('slidePaneSetLastSlideActive', '', { preserveFocus: true });
                    } else {

                        if (!($(e.target).closest('.slide-container').hasClass('selection'))) {
                            appModel.trigger('slidepane:clearselection');
                            slidePaneContainer.focus();
                            setActiveSlideforClickedTarget(e, 'mouse');
                        }
                    }
                }
            });

            self.listenTo(appModel, 'slidepane:clearselection', function () {
                clearSlidesInMultiSelection();
            });

            // To calculate and set the thumbnail size according to the pane width when the pane is resized.
            // When the pane is resized, the thumbnails need to be scaled accordingly.
            self.listenTo(self, 'pane:resize', function () {

                setSizeSlidePaneThumbnails(getWidthSlidePane());

                handleVisibleSlideRangeNotification();

                forceRerenderSlidePaneThumbnailsDebounced();
            });

            // listen to the tracking:end on the .resizer-node to get an info that the resize is finished, after that set this value in the document
            self.getNode().children('.resizer').on('tracking:end', resizeFinishedHandler);

            // scale the slide pane width in relation to the browser size
            if (!Utils.COMPACT_DEVICE) {
                self.listenTo($(window), 'resize', resizeWindowHandlerDebounced);
            }

            // slide pane layout according to the orientation
            if (Utils.COMPACT_DEVICE) {
                self.listenTo($(window), 'orientationchange', function () {

                    // must update the page width value here
                    windowWidth = getWindowWidth();

                    setSizeSlidePaneInPercent(getWidthOnCompactDevices());
                });
            }

            // handle clicks on the slide in the slidePane
            //Forms.touchAwareListener({ node: slidePaneContainer }, setActiveSlideforClickedTarget);
            //self.listenTo(slidePaneContainer, 'touchstart', setActiveSlideforClickedTarget);

            slidePaneContainer.on('tracking:start tracking:move tracking:end tracking:scroll tracking:cancel', trackingHandler);

            forceRerenderSlidePaneThumbnailsDebounced    = self.createDebouncedMethod($.noop, forceRerenderSlidePaneThumbnails, { delay: 500, infoString: 'slidepane :: forceRerenderSlidePaneThumbnailsDebounced' });

            handleVisibleSlideRangeNotificationDebounced = self.createDebouncedMethod($.noop, handleVisibleSlideRangeNotification, { delay: 500, infoString: 'slidepane :: handleVisibleSlideRangeNotificationDebounced' });
          //handleSlideRangeVisibilityDebounced          = self.createDebouncedMethod($.noop, handleSlideRangeVisibility,          { delay:  50, infoString: 'slidepane :: handleVisibleSlideRangeNotificationDebounced' });

            slidePaneContainer[0].addEventListener('scroll', handleVisibleSlideRangeNotificationDebounced, false);
          //slidePaneContainer[0].addEventListener('scroll', handleSlideRangeVisibilityDebounced,          false);

            Tracking.enableTracking(slidePaneContainer, {
                autoScroll: true,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 5,
                maxSpeed: 300,
                acceleration: 1.5
            });
        }

        /**
         * Getting the slidePaneContainer.
         *
         * @returns {jQuery}
         *   The slidePaneContainer as a jQuery object.
         */
        this.getSlidePaneContainer = function () {
            return slidePaneContainer;
        };

        /**
         * Setting the side pane width in an early state during loading the document.
         * This is necessary during 'updateDocumentFormatting' to avoid jumping of the
         * already visible slide, when the slide pane is inserted.
         *
         * @param {Number} width
         *  The slide pane width from the document, on COMPACT_DEVICES use a fixed width.
         */
        this.setSidePaneWidthDuringLoad = function (width) {
            width = Utils.COMPACT_DEVICE ? getWidthOnCompactDevices() : width; //TODO maybe rename to slidePaneWidth
            setSizeSlidePaneInPx(Utils.round(width * (getWindowWidth() / 100), 1));
        };

        this.getVisibleSlideRange = function () {
            return getVisibleSlideRange();
        };

        this.getVisibleSlideIdList = function (slideRange) {
            return getVisibleSlideIdList(slideRange);
        };

        /**
         * Returning the slide selection in the slide pane.
         *
         * @returns {Number[]}
         *  An Array containing the position as index from all selected slides in the slide pane.
         */
        this.getSlidePaneSelection = function () {
            return getSelection();
        };

        //function functionCheckModel() {
        //
        //    _.each(slideSelection, function (id) {
        //        if (slidePaneContainer.children('div[data-container-id = "' + id + '"]').index() !== appModel.getSortedSlideIndexById(id)) { console.error('MODEL VIEW ERROR'); }
        //    });

        //}

        // initialization -----------------------------------------------------

        // add the main HTML container to the slidePane
        self.getNode().append(slidePaneContainer);
        self.getNode().append(slidePlacer);
            //$('#io-ox-windowmanager-pane').append(slidePlacer);

        // needed for Safari to get beforepaste event
        if (_.browser.Safari) {
            self.getNode().addClass('safari-userselect-text');
        }

        // init the slidepane
        self.listenTo(appModel, 'slideModel:init', initSlidePane);

        // render preview item into each visible slide item of slidepane
        // when all slide previews are ready
        self.listenTo(docView, 'slidepreview:init:after', handleInitialPreviewThumbRendering);

        // destroy class members on destruction
        this.registerDestructor(function () {
            if (contextMenu) { contextMenu.destroy(); }
            self = app = docView = appModel = slidePaneContainer = contextMenu = null;
        });

    } // class StatusPane

    // exports ================================================================

    // derive this class from class ToolPane
    return Pane.extend({ constructor: SlidePane });

});
