/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/pagehandlermixin', [
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/components/table/tableresize',
    'io.ox/office/drawinglayer/view/drawingframe'
], function (DOM, Position, Hyperlink, TableResize, DrawingFrame) {

    'use strict';

    // mix-in class PageHandlerMixin ======================================

    /**
     * A mix-in class for the document model class PresentationModel providing
     * the event handlers for the main page used in a presentation document.
     *
     * @constructor
     *
     * @param {Application} app
     *  The application containing this instance.
     */
    function PageHandlerMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * The handler for the 'mousedown' and the 'touchstart' event in the presentation
         * application.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         */
        function pageProcessMouseDownHandler(event) {

            var // whether the user has privileges to edit the document
                readOnly = self.getEditMode() !== true,
                // the slide node
                slideRootNode = null,
                // jQuerified target of the event
                $eventTarget = $(event.target),
                // a table cell node
                tableCell = $eventTarget.closest(DOM.TABLE_CELLNODE_SELECTOR),
                // a table cell point
                tableCellPoint = null,
                // drawing start and end position for selection
                startPosition = null, endPosition = null,
                // mouse click on a drawing node
                drawingNode = null,
                // move handler node for drawings
                moveNode = null,
                // mouse click on a table resize node
                resizerNode = null,
                // the id of an optionally activated slide node
                activatedSlideId = '',
                // the currently activated node
                activeRootNode = self.getCurrentRootNode(),
                // whether the right mouse-button was clicked, or not
                rightClick = (event.button === 2);

            // fix for not clearing annoying selection range in Chrome on click
            if (_.browser.Chrome && self.getSelection().hasRange() && !rightClick) { window.getSelection().empty(); }

            // expanding a waiting double click to a triple click
            if (self.getDoubleClickEventWaiting()) {
                self.setTripleClickActive(true);
                event.preventDefault();
                return;
            }

            // -> if the click happened on the slide, the selection box already handles the event
            if (DOM.isSlideNode($eventTarget)) {
                event.preventDefault();
                return;
            }

            // activation mouse down
            self.setActiveMouseDownEvent(true);
            // clear a possible multi selection in the slide pane
            self.trigger('slidepane:clearselection');

            // make sure that the clickable parts in the hyperlink popup are processed by the browser
            if (Hyperlink.isClickableNode($eventTarget, readOnly)) {
                return;
            } else if (Hyperlink.isPopupOrChildNode($eventTarget)) {
                // don't call selection.processBrowserEvent as we don't have
                // a valid position in the document if we process mouse event
                // on the popup
                event.preventDefault();
                return;
            }

            // Fix for 29409: preventing default to avoid grabbers in table cells or inline components
            if ((_.browser.IE === 11) &&
                ($eventTarget.is('div.cell') ||
                DOM.isInlineComponentNode(event.target) ||
                (event.target.parentNode && DOM.isInlineComponentNode(event.target.parentNode)))) {
                event.preventDefault(); // -> not returning !
            }

            if (tableCell.length > 0) {
                // saving initial cell as anchor cell for table cell selection
                // -> this has to work also in read-only mode
                tableCellPoint = DOM.Point.createPointForNode(tableCell);
                self.getSelection().setAnchorCellRange(new DOM.Range(tableCellPoint, new DOM.Point(tableCellPoint.node, tableCellPoint.node + 1)));
            }

            // tabulator handler (handle in processMouseUp event)
            if ($eventTarget.closest(DOM.TAB_NODE_SELECTOR).length > 0) { return; }

            // checking for a selection on a drawing node. But if this is a text frame drawing node, this is not a drawing
            // selection, if the click happened on the text frame sub element.
            drawingNode = $eventTarget.closest(DrawingFrame.NODE_SELECTOR);

            if (drawingNode.length > 0) {

                // check, if a drawing multiple selection can be created or expanded or decreased or removed
                if ((event.shiftKey || event.ctrlKey) && self.getSelection().checkMutipleSelection(drawingNode)) {
                    // another drawing already selected -> creating, expanding or removing a multiple drawing selection
                    self.getSelection().handleMultipleSelection(drawingNode);
                    event.preventDefault();
                    return;
                }

                // presentation specific code
                if (DOM.isInsideSlideContainerNode(drawingNode)) {  // TODO: This should always be the case

                    slideRootNode = drawingNode.closest(DOM.SLIDECONTAINER_SELECTOR);

                    // handling unselectable drawings in presentation
                    if (slideRootNode.children('.slide').is('.notselectable')) {
                        event.preventDefault();
                        return;
                    }

                    self.getSelection().setNewRootNode(slideRootNode);
                    activatedSlideId = DOM.getTargetContainerId(slideRootNode);
                    self.setActiveTarget(activatedSlideId);
                }

                if (DrawingFrame.isOnlyBorderMoveableDrawing(drawingNode) && $eventTarget.closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0 && !(rightClick && self.getSelection().isMultiSelection())) {
                    drawingNode = $();
                }
            } else {
                // no drawing selected -> click on slide
                if (DOM.isSlideNode($eventTarget) && DOM.isSlideContainerNode($eventTarget.parent())) {
                    slideRootNode = $eventTarget.parent();
                    self.getSelection().setNewRootNode(slideRootNode);
                    activatedSlideId = DOM.getTargetContainerId(slideRootNode);
                    self.setActiveTarget(activatedSlideId);
                }
            }

            // deactivating active target, but not if it was activated before (presentation specific code)
            if (self.getActiveTarget() !== '' && !activatedSlideId) {
                self.getSelection().setNewRootNode(self.getNode());
                self.setActiveTarget('');
            }

            // in read only mode allow text selection only (and drawing selection, task 30041)
            if (readOnly && drawingNode.length === 0) {
                self.getSelection().processBrowserEvent(event);
                return;
            }

            // checking for a selection on a resize node
            resizerNode = $eventTarget.closest(DOM.RESIZE_NODE_SELECTOR);

            if (drawingNode.length > 0) {

                if (event.type !== 'touchstart') {
                    // prevent default click handling of the browser
                    event.preventDefault();
                }

                // set focus to the document container (may be located in GUI edit fields)
                app.getView().grabFocus();

                // do nothing if the drawing is already selected (prevent endless
                // recursion when re-triggering the event, to be able to start
                // moving the drawing immediately, see below)
                // Also create a drawing selection, if there is an additional text frame selection,
                // that also has a text selection. In this case the text selection is removed.
                if (!DrawingFrame.isSelected(drawingNode) || self.getSelection().isAdditionalTextframeSelection()) {

                    // select the drawing
                    startPosition = Position.getOxoPosition(activeRootNode, drawingNode[0], 0);
                    endPosition = Position.increaseLastIndex(startPosition);
                    self.getSelection().setTextSelection(startPosition, endPosition, { event: event });

                    // The following triggering of the event is necessary to move a drawing without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the drawing. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a drawing already with the first mousedown/touchstart.
                    // For moving a drawing, a tracker node was created and registered for 'tracking:start' in the
                    // preceeding selection.drawDrawingSelection() call. This additional mousedown/touchstart
                    // generates a 'tracking:start' event, that will be handled by the tracker node.
                    // The event target for this event in 'processMouseDown' could be the 'img' or the 'div.drawing'. But
                    // it has to be sent to the tracker. Therefore it is necessary to adapt the event target, before
                    // triggering it again.
                    moveNode = drawingNode.find('.tracker');
                    if (moveNode.length > 0) {
                        event.target = moveNode[0];
                        moveNode.trigger(event);
                    }
                }

                return event.type === 'touchstart';

            } else if (resizerNode.length > 0) {

                if (!DOM.isActiveResizeNode(resizerNode)) {
                    // Because the event is triggered again to avoid a second click, it is necessary to mark the resize node
                    // as 'active'. Otherwise the event would be triggered endlessly.

                    // prevent default click handling of the browser
                    event.preventDefault();
                    // set focus to the document container (may be located in GUI edit fields)
                    app.getView().grabFocus();
                    // draw the resizer
                    TableResize.drawTableCellResizeSelection(self, app.getView(), app.getWindowNode(), resizerNode);

                    // send initial mouse down event to the registered handlers
                    // The following triggering of the event is necessary to resize the table without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the resize node. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a resize node already with the first mousedown/touchstart.
                    resizerNode.trigger(event);
                }
            } else {

                // clicked somewhere else: calculate logical selection from browser selection, after browser has processed the mouse event.
                self.getSelection().processBrowserEvent(event);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the handler function that is used for process mouse down and tochstart
         * events on the page node.
         *
         * @returns {Function}
         *  The handler function for the 'mousedown' and the 'touchstart' event.
         */
        this.getPageProcessMouseDownHandler = function () {
            return pageProcessMouseDownHandler;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class PageHandlerMixin

    // exports ================================================================

    return PageHandlerMixin;

});
