/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/presentation/model/dynfontsizemixin', [
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/utils',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations'
], function (DOM, Utils, Position, Operations) {

    'use strict';

    var DEFAULT_FONT_SIZE = 16;
    var MIN_FONT_SIZE = 4;

    var INDEX_ARRAY = [0, 1, 2, 3, 4, 5];

    // mixin DynFontSizeMixin =========================================================

    function DynFontSizeMixin() {

        var self = this;

        // all drawings nodes that need to be updated
        var drawingQueue = $();

        // private methods ----------------------------------------------------

        function updateStep(textframecontent, textframe, maxHeight, lastHeight, i) {
            var currentFontSize = parseFloat(textframecontent.css('font-size'));
            var lineReduction = parseFloat(textframecontent.attr('linereduction')) || 0;

            // dont use jquery postion, because it is cached!!!
            // var currentHeight = lastP.offsetTop + lastP.offsetHeight;

            var currentHeight = 0;
            textframe.children().get().forEach(function (para) {
                currentHeight += $(para).outerHeight(true);
            });

            // console.log('diff', maxHeight, currentHeight);
            /*
            console.log('iter', i, 'currentHeight', currentHeight, 'maxHeight', maxHeight, 'currentFontSize', currentFontSize);
            if (i === 6) {
                return null;
            }
            */
            if (currentHeight < maxHeight && currentFontSize === DEFAULT_FONT_SIZE && lineReduction === 0) {
                //all is fine
                // console.log('all is fine');
                return null;
            } else if (currentHeight < maxHeight * 1.02 && currentHeight > maxHeight * 0.95) {
                //all is scaled but fine in 95% - 102%
                // console.log('all is scaled but fine');
                return null;
            } else {
                var diff = maxHeight / currentHeight;

                if (currentFontSize <= 4 && diff <= 1) {
                    // console.log('early cut out');
                    return null;
                } else {

                    if (diff > 1) {
                        diff = diff * 0.5 + 0.5;
                    } else if (lastHeight === currentHeight && diff > 0.5) {
                        diff = diff * 0.5 + 0.25;
                    }
                    // console.log('have to scale', diff, i);

                    var newFont = diff * currentFontSize;
                    var newLineReduction = lineReduction;
                    if (diff < 1) {
                        newLineReduction += 8;
                        newLineReduction = Math.min(20, newLineReduction);
                    }

                    if (i === 0 && (currentFontSize !== DEFAULT_FONT_SIZE || lineReduction !== 0)) {
                        newFont = DEFAULT_FONT_SIZE;
                        newLineReduction = 0;
                    }

                    newFont = Math.min(newFont, DEFAULT_FONT_SIZE);
                    newFont = Math.max(newFont, MIN_FONT_SIZE);
                    newLineReduction = Math.min(newLineReduction, 20);

                    textframecontent.css('font-size', newFont + 'px');
                    textframecontent.attr('font-scale', newFont / DEFAULT_FONT_SIZE);
                    textframecontent.attr('linereduction', newLineReduction);
                }
            }
            return currentHeight;
        }

        function updateDrawing(drawing) {
            var $drawing = $(drawing);
            var textframecontent = $drawing.find('>.textframecontent');
            var initialFontSize = parseFloat(textframecontent.css('font-size'));
            var initialLineReduction = parseFloat(textframecontent.attr('linereduction'));
            var textFrame = textframecontent.find('>.textframe');

            if (!textFrame.children().length) {
                //empty drawing
                return;
            }

            // console.log('updateDrawing', textframecontent.attr('class'));
            if (textframecontent.hasClass('autoresizeheight') || !textframecontent.hasClass('autoresizetext')) {
                if (initialFontSize !== DEFAULT_FONT_SIZE || initialLineReduction !== 0) {
                    textframecontent.css('font-size', DEFAULT_FONT_SIZE + 'px');
                    textframecontent.attr('font-scale', null);
                    textframecontent.attr('linereduction', null);
                }
                if (textframecontent.hasClass('autoresizeheight')) {
                    var drawingHeight = $drawing.data('drawingHeight') || 0;
                    var currentHeight = $drawing.height();

                    if (Math.abs(drawingHeight - currentHeight) > 1) {

                        $drawing.data('drawingHeight', currentHeight);

                        var drawingAttrs = {
                            height: Utils.convertLengthToHmm(currentHeight, 'px'),
                            width: Utils.convertLengthToHmm($drawing.width(), 'px'),
                            left: Utils.convertLengthToHmm(parseFloat(drawing.style.left), 'px'),
                            top: Utils.convertLengthToHmm(parseFloat(drawing.style.top), 'px')
                        };
                        return $.when({ attrs: { drawing: drawingAttrs }, drawing: drawing });
                    }
                }
                return;
            }

            //var maxHeight = textframecontent.height();
            //var maxHeight = parseFloat(drawing[0].style.height);
            var maxHeight = textframecontent.data('shapeHeight');
            if (!maxHeight) {
                // console.warn('noooo', textframecontent[0]);
                return;
            }
            maxHeight = maxHeight - (parseFloat(textFrame[0].style.paddingTop) || 0) - (parseFloat(textFrame[0].style.paddingBottom) || 0);
            var lastHeight = -1;

            function innerUpdate(i) {
                if (lastHeight === null) {
                    return;
                }
                lastHeight = updateStep(textframecontent, textFrame, maxHeight, lastHeight, i);
            }

            function returnFontSize() {
                var newFontScale = parseFloat(textframecontent.css('font-size'));
                var newLineReduction = parseFloat(textframecontent.attr('linereduction'));

                if (Math.abs(newFontScale - initialFontSize) < 0.001 && Math.abs(newLineReduction - initialLineReduction) < 0.001) {
                    return;
                }
                return {
                    attrs: {
                        shape: {
                            fontScale: newFontScale / DEFAULT_FONT_SIZE,
                            lineReduction: newLineReduction / 100,
                            autoResizeText: true,
                            autoResizeHeight: false,
                            noAutoResize: false
                        }
                    },
                    drawing: drawing
                };
            }

            var def = null;
            if (_.browser.IE) {
                def = self.iterateArraySliced(INDEX_ARRAY, innerUpdate, { slice: 1, infoString: 'DynFontSizeMixin:updateDrawing()' });
            } else {
                _.each(INDEX_ARRAY, innerUpdate);
                def = $.when(true);
            }

            return def.then(returnFontSize);
        }

        function handleContainerVisibility(slide, invisibleWithReflow) {
            var makeVisible = !slide.hasClass('invisibleslide');
            self.handleContainerVisibility(slide, { invisibleWithReflow: invisibleWithReflow, makeVisible: makeVisible });
        }

        /**
         * Updating some specified drawings in a specified slide. During loading a document
         * the slides are formatted at all. In that context also the font size is set and it is not
         * necessary to specify the drawings, because all of them are updated.
         *
         * After the document is loaded, it is only necessary to update those drawings, who got new
         * attributes or whose content has changed. In this case, it is necessary to specify the
         * drawings, so that not all drawings of a slide need to be updated.
         *
         * @param {jQuery} slide
         *  The slide of the drawing
         *
         * @param {String} slideID
         *  The slideID of the slide
         *
         * @param {jQuery} [drawings]
         *  this drawings will be updated.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after all drawings are updated.
         */
        function updateSlide(slide, slideID, drawings) {

            // ignoring layout or master slides
            if (!self.isStandardSlideId(slideID)) {
                return;
            }

            var defs = [];

            handleContainerVisibility(slide, true);

            _.each(drawings, function (drawing) {
                var def = updateDrawing(drawing);
                if (def) { defs.push(def); }
            });

            if (defs.length) {
                return $.when.apply($, defs).then(function () {
                    handleContainerVisibility(slide, false);
                    return arguments;
                });
            } else {
                handleContainerVisibility(slide, false);
            }
        }

        // The drawing element(s) as DOM node or jQuery object.
        function registerDrawing(newDrawings) {
            // store the new drawing(s) in the collection (jQuery keeps the collection unique)
            if (newDrawings) { drawingQueue = drawingQueue.add(newDrawings); }
        }

        function updateDynFontSizeInDrawings() {
            var promise = updateDynFontSizeForDrawings(drawingQueue);
            drawingQueue = $();
            return promise;
        }

        // updating the specified drawings
        //
        // Info: This function handles only drawings on document slides, not on layout
        //       or master slides. Therefore the 'target' is always NOT defined.
        //
        //  Because of the 'target'
        function updateDynFontSizeForDrawings(drawings) {
            var defs = [];
            var slideID = null;
            // iterating over all drawings
            _.each(drawings, function (drawing) {
                var slide = $(drawing).closest(DOM.SLIDE_SELECTOR);
                slideID = self.getSlideId(slide);

                // skip slides if they are not formatted at all -> asking slide format manager
                if (!self.getSlideFormatManager().isUnformattedSlide(slideID)) {
                    var change = updateSlide(slide, slideID, $(drawing));
                    if (change) { defs.push(change); }
                }
            });

            if (defs.length) {
                return $.when.apply($, defs).then(function () {
                    //controller update must be called for changed font-size in the toolbar
                    self.getApp().getController().update();

                    if (!self.getEditMode()) {
                        return;
                    }
                    var ops = [];

                    _.flatten(arguments).forEach(function (change) {
                        if (!change) {
                            return;
                        }

                        var position = Position.getOxoPosition(self.getRootNode(slideID), change.drawing);
                        ops.push({
                            noUndo: true,
                            name: Operations.SET_ATTRIBUTES,
                            start: position,
                            attrs: change.attrs,
                            target: self.getOperationTargetBlocker()
                        });

                        // Info: Adding the operationsTargetBlocker is necessary because the operation is not
                        //       executed on the active slide. Therefore the target must not be added automatically
                        //       by the finalizeOperations handler.
                        //       Because all affected slides are document slides, it is never necessary to add any
                        //       target. If this is required, this target must be set explicitely to the operation.
                    });
                    self.applyOperations(ops);

                });
            } else {
                return $.when();
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Handling the property 'autoResizeHeight' of a selected text frame node.
         *
         * @param {Boolean} autoResizeHeight
         *  Whether the property 'autoResizeHeight' of a selected text frame shall be
         *  enabled or disabled.
         */
        self.handleTextFrameAutoFit = function (autoResizeHeight, autoResizeText) {

            var selection = self.getSelection();
            // the operations generator
            var generator = self.createOperationsGenerator();
            // the options for the setAttributes operation
            var operationOptions = {};
            // a selected text frame node or the text frame containing the selection
            var textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            // collecting the attributes for the operation
            operationOptions.attrs =  {};
            operationOptions.attrs.shape = {};

            operationOptions.attrs.shape.autoResizeText   = autoResizeText || false;
            operationOptions.attrs.shape.autoResizeHeight = autoResizeHeight || false;
            operationOptions.attrs.shape.noAutoResize = (!autoResizeText && !autoResizeHeight);

            if (selection.isAdditionalTextframeSelection()) {
                operationOptions.start = Position.getOxoPosition(self.getNode(), textFrame, 0);
            } else {
                operationOptions.start = selection.getStartPosition();
            }

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            self.applyOperations(generator);

            self.updateDynFontSizeDebounced(textFrame);
        };

        self.updateDynFontSizeDebounced = self.createDebouncedMethod(registerDrawing, updateDynFontSizeInDrawings, { delay: 500, infoString: 'Presentation: DynfontSizeMixin.updateDynFontSizeDebounced' });

        self.getPreFlushDocumentPromise = function () { return updateDynFontSizeInDrawings(); };

        // initialization -----------------------------------------------------

    } // mixin DynFontSizeMixin

    // exports ================================================================

    return DynFontSizeMixin;
});
