/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/slidestyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/editframework/model/stylecollection'
], function (Utils, Image, StyleCollection) {

    'use strict';

    var // definitions for slide attributes
        DEFINITIONS = {

            /**
             * Whether the drawings from layout or master slide shall be visible.
             */
            followMasterShapes: { def: true },

            /**
             * Whether the date place holder will be inherited.
             */
            isDate: { def: false },

            /**
             * Whether the header place holder will be inherited.
             */
            isFooter: { def: false },

            /**
             * Whether the footer place holder will be inherited.
             */
            isHeader: { def: false },
            /**
             * Whether the slide number place holder will be inherited.
             */
            isSlideNum: { def: false },

            /**
             * Whether the drawings from layout or master slide shall be visible.
             */
            hidden: { def: false },

            /**
             * The type of a layout slide
             */
            type: { def: '' }
        };

    // Info for the properties 'isDate', 'isFooter', 'isHeader' and 'isSildeNum'.
    // In OOXML file there is an 'hf' element. If this does not exist, all values are set
    // to false. This is the default that the client uses. But if there is at least one child
    // (for example 'isFooter') set explicitely to 'false', all other (not defined) children
    // are set automatically to true (the default changes). This needs to be handled by the
    // filter for the loading operations of the document. If the client changes one of these
    // values in the UI, all four values need to be set.

    // private global functions ===============================================

    // class SlideStyles =======================================================

    /**
     * Contains the style sheets for page formatting attributes. The CSS
     * formatting will be read from and written to the slide container elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function SlideStyles(docModel) {

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'slide', {
            families: 'fill',
            formatHandler: updateSlideFormatting
        });

        // private functions --------------------------------------------------

        /**
         * Updating the slide background node with the specified attributes of the family 'fill'.
         *
         * @param {jQuery} slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         */
        function updateBackground(slide, mergedAttributes) {

            var // the fill attributes of the slide
                fillAttrs = mergedAttributes.fill,
                // the CSS properties for the slide
                cssProps = {};

            // calculate the CSS fill attributes
            switch (fillAttrs.type) {
                case 'none':
                    // clear everything: color and bitmaps
                    cssProps.background = '';
                    break;

                case 'solid':
                    // use 'background' compound attribute to clear bitmaps
                    cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill');
                    break;

                case 'bitmap':
                    // stretching the drawing, but keeping the ratio
                    docModel.getImageSize(fillAttrs.imageUrl).then(function (size) {

                        var // the ratio of width and height of the slide
                            slideRatio = docModel.getSlideRatio(),
                            // the ratio of width and height of the drawing
                            drawingRatio = Utils.round(size.width / size.height, 0.01),
                            // the background size value for the width
                            width = '100%',
                            // the background size value for the width
                            height = '100%';

                        if (slideRatio > drawingRatio) {
                            // the drawing needs to be cropped at top and bottom -> height need to be increased
                            height = (Utils.round(slideRatio / drawingRatio, 0.01) * 100) + '%';
                        } else {
                            // the drawing needs to be cropped at left and right sides -> width need to be increased
                            width = (Utils.round(drawingRatio / slideRatio, 0.01) * 100) + '%';
                        }

                        cssProps.background = 'url("' + Image.getFileUrl(docModel.getApp(), fillAttrs.imageUrl)  + '")';
                        cssProps['background-position'] = 'center';
                        cssProps['background-size'] = width + ' ' + height;
                        slide.css(cssProps);
                    });
                    break;

                default:
                    Utils.warn('SlideStyles.setSlideBackground(): unknown fill type "' + fillAttrs.type + '"');
            }

            // apply the fill attributes
            slide.css(cssProps);
        }

        /**
         * Will be called for every page whose attributes have been changed.
         *
         * @param {jQuery} slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         */
        function updateSlideFormatting(slide, mergedAttributes) {

            var // the slide id
                slideId = docModel.getSlideId(slide),
                // getting the old 'hidden' state (the model is already updated)
                oldHiddenState = (slide.attr('isHidden') === 'true'),
                // whether the hidden state of a slide was changed
                hiddenStateChanged = (mergedAttributes.slide.hidden !== oldHiddenState);

            // updating the slide background
            updateBackground(slide, mergedAttributes);

            // setting a marker at the slide, so that changes can be recognized
            if (hiddenStateChanged) { slide.attr('isHidden', mergedAttributes.slide.hidden); }

            // inform the view, that the merged slide attributes (including 'fill') need to be updated
            docModel.trigger('change:slideAttributes', { slideId: slideId, hiddenStateChanged: hiddenStateChanged, isHidden: mergedAttributes.slide.hidden });
        }

        // public methods -----------------------------------------------------

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('slide', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class SlideStyles

    // constants --------------------------------------------------------------

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: SlideStyles });

});
