/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/drawingstyles', [
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom'
], function (PresentationUtils, Utils, AttributeUtils, Border, DrawingStyleCollection, DrawingFrame, DOM) {

    'use strict';

    var // definitions for presentation-specific drawing attributes
        DEFINITIONS = {

            /**
             * Margin from top border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin from bottom border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin from left border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin from right border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginRight: { def: 0 },

            /**
             * Style, width and color of the left drawing border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the right drawing border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the top drawing border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the bottom drawing border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border));
                }
            },

            // object anchor --------------------------------------------------

            /**
             * The distance of the drawing to the left slide border, as number
             * in 1/100 of millimeters.
             */
            left: { def: 0 },

            /**
             * The distance of the drawing to the top slide border, as number
             * in 1/100 of millimeters.
             */
            top: { def: 0 },

            /**
             * Width of the drawing, as number in 1/100 of millimeters. The
             * value 0 will be interpreted as 100% width of the parent element.
             */
            width: { def: 0 },

            /**
             * Height of the drawing, as number in 1/100 of millimeters.
             */
            height: { def: 0 },

            // drawing rotation -------------------------------------------------------

            rotation: {
                def: 0,
                format: function (element, degree) {
                    if (_.isNumber(degree) && degree !== 0) {
                        element.css({ transform: 'rotate(' + degree + 'deg)' }).addClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME);
                    } else {
                        element.css({ transform: 'rotate(0deg)' }).removeClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME);
                    }
                }
            },

            // locking aspect ratio for resizing
            noChangeAspect: {
                def: false
            }

        };

    // class PresentationDrawingStyles ================================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends DrawingStyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function PresentationDrawingStyles(docModel) {

        var // self reference
            self = this,
            // collector for the place holder attributes in layout or master slides
            placeHolderCollector = {};

        // base constructor ---------------------------------------------------

        DrawingStyleCollection.call(this, docModel, {
            families: 'presentation line fill image shape geometry changes',
            formatHandler: updateDrawingFormatting,
            layoutHandler: addLayoutAttributes
        });

        // private methods ----------------------------------------------------

        /**
         * Setting the width of the drawing.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingWidth(drawing, attributes) {

            if (!DrawingFrame.isGroupedDrawingFrame(drawing)) {
                if (attributes.drawing.width === 0) {
                    drawing.width('100%');
                } else {
                    drawing.width(Utils.convertHmmToLength(attributes.drawing.width, 'px', 1));
                }
            }
        }

        /**
         * Setting the height of the drawing. This needs to be done, because it might be
         * dependent from other attribute values.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingHeight(drawing, attributes) {

            if (!(DrawingFrame.isGroupedDrawingFrame(drawing) || attributes.shape.autoResizeHeight)) {
                drawing.height(Utils.convertHmmToLength(attributes.drawing.height, 'px', 1));
            } else if (attributes.shape.autoResizeHeight) {
                // -> no setting of height for grouped drawings and for shapes with property autoResizeHeight
                drawing.height(''); // if 'autoResizeHeight' is set, an explicit height must be removed
            }
        }

        /**
         * Helper function to update attributes of children of drawings. This is only used for text frames yet.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node as jQuery object, whose children need an update of element formatting.
         */
        function handleChildAttributes(drawing) {

            var // the explicit drawing attributes
                explicitDrawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                // the paragraph style sheet container
                paragraphStyles = null,
                // the container for the paragraphs and tables inside the drawing
                textFrameNode = null;

            // check, if the empty text frame needs an artificial border and a default text
            self.handleEmptyPlaceHolderDrawing(drawing, explicitDrawingAttrs);

            // checking the fill color (task 36385)
            if (explicitDrawingAttrs && explicitDrawingAttrs.fill) {

                paragraphStyles = docModel.getStyleCollection('paragraph');
                textFrameNode = DrawingFrame.getTextFrameNode(drawing);

                Utils.iterateSelectedDescendantNodes(textFrameNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                    paragraphStyles.updateElementFormatting(paragraph);
                }, undefined, { children: true });
            }
        }

        /**
         * Handling drawing that are absolute positioned. This are all drawings that are not in-line
         * with text. This is especially important, if the drawing anchor was modified. In this case
         * some clean up at the node is required.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map with the merged drawing attributes, as name/value pairs.
         *
         * @param {Number} borderWidth
         *  The width of the drawing of the border in 1/100 mm.
         *
         * @param {Number} halfBorderWidth
         *  The half width of the drawings border in 1/100 mm.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         */
        function assignDrawingAttributes(drawing, mergedAttributes) {

            var // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
//                // offset from top of slide element, in 1/100 mm
//                topOffset = drawingAttributes.top,
//                // margins to be applied at the drawing
//                topMargin = 0, bottomMargin = 0, leftMargin = 0, rightMargin = 0,
                // the type of place holder for presentation drawings
                placeHolderType = mergedAttributes.presentation.phType,
                // the index of place holder for presentation drawings
                placeHolderIndex = mergedAttributes.presentation.phIndex;

            // calculate drawing margins
//            topMargin = Utils.minMax(drawingAttributes.marginTop, 0, topOffset > 0 ? topOffset : 0);
//            bottomMargin = Math.max(drawingAttributes.marginBottom, 0);
//            leftMargin = Math.max(drawingAttributes.marginLeft, 0);
//            rightMargin = Math.max(drawingAttributes.marginRight, 0);

            // drawing is absolutely positioned
            drawing.addClass('absolute');

            // marking presentation placeholder drawings
            drawing.attr('placeholdertype', placeHolderType ? placeHolderType : null); // toggle attribute
            drawing.attr('placeholderindex', _.isNumber(placeHolderIndex) ? placeHolderIndex : null); // toggle attribute

            // apply CSS formatting to drawing node
            // -> adding the half border width to the margin (border is drawn with a canvas)
            drawing.css({
                left: Utils.convertHmmToLength((drawingAttributes.left - drawingAttributes.marginLeft), 'px', 1),
                top: Utils.convertHmmToLength((drawingAttributes.top - drawingAttributes.marginTop), 'px', 1)
            });
        }

        /**
         * Adding attributes of master and layout slide into the merge attributes. This has to happen
         * before the explicit attributes are set into the specified attribute set.
         *
         * @param {jQuery} element
         *  The node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets, but without explicit attributes.
         */
        function addLayoutAttributes(element, attributes) {

            var // the attribute set of the place holder elements in layout and/or master slide
                placeHolderAttributes = null,
                // the explicit attributes at the specified element
                elemAttrs = AttributeUtils.getExplicitAttributes(element),
                // the IDs of document slide, layout slide and master slide
                id = null, layoutId = null, masterId = null,
                // an object containing the place holder type and index as property
                keySet = null;

            if (PresentationUtils.isPlaceHolderAttributeSet(elemAttrs)) {

                id = DOM.getTargetContainerId(element.parent()); // checking only direct children of slide
                keySet = self.getValidTypeIndexSet(elemAttrs);

                if (keySet) {
                    if (docModel.isStandardSlideId(id)) {
                        layoutId = docModel.getLayoutSlideId(id);
                        masterId = docModel.getMasterSlideId(layoutId);
                        placeHolderAttributes = getMergedPlaceHolderAttributes(layoutId, masterId, keySet.type, keySet.index);

                    } else if (docModel.isLayoutSlideId(id)) {
                        masterId = docModel.getMasterSlideId(id);
                        placeHolderAttributes = getMergedPlaceHolderAttributes(null, masterId, keySet.type, keySet.index);
                    }
                }
            }

            // merging the new place holder attributes into the specified attribute set
            if (placeHolderAttributes) { docModel.extendAttributes(attributes || {}, placeHolderAttributes); }
        }

        /**
         * Will be called for every drawing node whose attributes have been changed. Repositions
         * and reformats the drawing according to the passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            // nothing to do, if paragraph is a group content object
            // -> in this case the group is responsible for the correct attributes
            if (DrawingFrame.isGroupContentNode(drawing.parent())) {
                drawing.addClass('absolute grouped');  // absolute positioned drawing inside its group!
                // set generic formatting to the drawing frame
                DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);
                // simply do nothing, let the group make further attribute settings
                return;
            }

            // setting the drawing height
            setDrawingHeight(drawing, mergedAttributes);

            // setting the drawing width
            setDrawingWidth(drawing, mergedAttributes);

            // setting the drawing attributes except width and height (those are already set)
            assignDrawingAttributes(drawing, mergedAttributes);

            // set generic formatting to the drawing frame
            DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);

            // update formatting of all paragraphs inside the text frame content node (36385)
            if (DrawingFrame.isTextFrameShapeDrawingFrame(drawing)) { handleChildAttributes(drawing); }
        }

        /**
         * Saving the specified drawing attributes in the model container.
         *
         * This function can be called directly from insertDrawing operation. In that
         * case the attributes already contain the full set of properties from the
         * families 'drawing' and 'presentation'.
         *
         * @param {String} target
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Number} index
         *  The place holder index.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        function savePlaceHolderAttributes(target, type, index, attrs, options) {

            var // whether list style handling is required
                listStyle = attrs && attrs.listStyle,
                // the level specified in the list styles
                level = null,
                // a helper object for merging list style attributes
                drawingAttrs = null,
                // the old place holder attributes
                oldAttrs = null,
                // whether an update of dependent slide shall be triggered immediately
                saveOldAttrs = Utils.getBooleanOption(options, 'saveOldAttrs', false);

            if (!(target && type && _.isNumber(index))) { return null; }

            placeHolderCollector[target] = placeHolderCollector[target] || {};
            placeHolderCollector[target][type] = placeHolderCollector[target][type] || {};
            placeHolderCollector[target][type][index] = placeHolderCollector[target][type][index] || {};

            // optionally saving a deep copy of the old value in the return object for later usage
            if (saveOldAttrs) { oldAttrs = _.copy(placeHolderCollector[target][type][index], true); }

            // merging the attributes, that are not list styles ('autoclear' is required for undo, so that attributes can be removed)
            docModel.extendAttributes(placeHolderCollector[target][type][index], attrs, { autoClear: true });

            // special handling for list style attribute at drawings
            if (listStyle) {
                for (level in listStyle) {
                    drawingAttrs = placeHolderCollector[target][type][index];
                    drawingAttrs.listStyle = drawingAttrs.listStyle || {};
                    drawingAttrs.listStyle[level] = drawingAttrs.listStyle[level] || {};
                    docModel.extendAttributes(drawingAttrs.listStyle[level], listStyle[level], { autoClear: true });
                }
            }

            // returning an object with the complete registration information
            return { target: target, placeHolderType: type, placeHolderIndex: index, attrs: attrs, oldAttrs: oldAttrs };
        }

        /**
         * Getting the merged place holder attributes from master slide and layout slide.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {String} placeHoderType
         *  The drawing place holder type.
         *
         * @param {Number} placeHoderIndex
         *  The drawing place holder index.
         *
         * @returns {Object}
         *  An object containing the attributes from the layout slide and the master slide. If attributes are defined
         *  at both slide, the layout slide overwrites the values from the master slide.
         */
        function getMergedPlaceHolderAttributes(layoutId, masterId, placeHolderType, placeHolderIndex) {

            var // the layout slide attributes
                layoutAttrs = layoutId ? self.getPlaceHolderAttributes(layoutId, placeHolderType, placeHolderIndex) : null,
                // the master slide attributes
                masterAttrs = masterId ? self.getPlaceHolderAttributes(masterId, placeHolderType, placeHolderIndex) : null,
                // the attributes of layout and master slide
                attributes = null;

            // merging the collected attributes from master and layout slides and drawings (never overwrite attribute set)
            if (masterAttrs) { attributes = docModel.extendAttributes({}, masterAttrs); }
            if (layoutAttrs) { attributes = attributes ? docModel.extendAttributes(attributes, layoutAttrs) : docModel.extendAttributes({}, layoutAttrs); }

            return attributes;
        }

        /**
         * Handling the visualization of empty text frames without explicit border. In the
         * standard view it is necessary to assign a dotted line to these empty text frames.
         * In the layout or master view, such a dotted line is always assigned to the place
         * holder drawings.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Object} [explicitDrawingAttrs]
         *  The (explicit) attributes of the drawing node. If they are not specified,
         *  the explicit attributes at the specified drawing node are evaluated.
         */
        function handleBorderForEmptyTextFrames(drawing, explicitDrawingAttrs) {
            // set class DrawingFrame.EMPTYTEXTFRAME_CLASS at empty text frames without border
            $(drawing).children('.content').toggleClass(DrawingFrame.EMPTYTEXTFRAME_CLASS, DOM.isEmptyTextframeWithoutBorder(drawing, explicitDrawingAttrs, { ignoreTemplateText: true }));
            // setting 'ignoreTemplateText' to true, so that an already inserted template text does not 'fill' the drawing.
            // -> it must still be empty, if the drawing is formatted twice.
        }

        /**
         * Handling a default text that is inserted into empty place holder drawings.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @returns {Boolean}
         *  Whether the default template text was inserted into the drawing.
         */
        function handleTemplateText(drawing) {
            return docModel.addTemplateTextIntoTextSpan(drawing);
        }

        /**
         * Emptying the complete place holder model. This is necessary, if a new snapshot is applied.
         */
        function emptyPlaceHolderModel() {
            placeHolderCollector = {};
        }

        // public methods -----------------------------------------------------

        /**
         * Saving the drawing attributes in a container, if the drawing is a place holder
         * element.
         *
         * @param {String} target
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node, that gets the specified attributes assigned.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target is not set (no master or layout slide) or the attributes
         *  contain no place holder information (no place holder drawing), null is
         *  returned.
         */
        this.savePlaceHolderAttributes = function (target, attrs, drawing, options) {

            var // an object containing the place holder type and index as property
                keySet = null;

            if (!target || !PresentationUtils.isPlaceHolderDrawing(drawing)) { return null; }

            keySet = self.getValidTypeIndexSet(AttributeUtils.getExplicitAttributes(drawing));

            // target must be defined and the attributes must have phType or phIndex specified
            // -> this is a placeholder drawing in master or layout slide -> the attributes are saved in the model.
            return savePlaceHolderAttributes(target, keySet.type, keySet.index, attrs, options);
        };

        /**
         * Receiving the attributes for a drawing with place holder attribute. In this case the attributes from the
         * parent layers can be used. If the drawing is inside a 'standard' slide, the attributes might be defined
         * in the layout or the master slide. If the drawing is inside a layout slide, the missing attributes might
         * be defined in the master slide.
         *
         * Info: For master slides the index needs to be ignored. This can easily be seen for the place holder types
         *       'ftr', 'dt' and 'sldNum'. In the default master slide they have not the same indices as used in the
         *       layout slides.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Number} index
         *  The place holder index.
         *
         * @returns {Object|Null}
         *  The attributes defined for the specified target and place holder type.
         */
        this.getPlaceHolderAttributes = function (target, type, index) {

            var // the attributes assigned to target and type
                attributes = null,
                // the available indices for this type of place holder
                indices = null;

            // switching type, if not supported on master slide
            if (docModel.isMasterSlideId(target) && PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES[type]) { type = PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES[type]; }

            if (placeHolderCollector && placeHolderCollector[target] && placeHolderCollector[target][type]) {
                attributes = placeHolderCollector[target][type];

                if (attributes[index]) {
                    attributes = attributes[index];
                } else {
                    if (docModel.isMasterSlideId(target)) {
                        // if this is a master slide, the index can be ignored
                        indices = _.keys(attributes);
                        if (indices && indices.length > 0) {
                            attributes = attributes[indices[0]]; // using the first key
                        }
                    }
                }
            }

            return attributes ? _.clone(attributes) : null;
        };

        /**
         * Receiving all place holder attributes that are stored for a specific master or layout node.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @returns {Object|Null}
         *  An object containing all place holder attributes defined for the specified layout/master
         *  slide. Or null, if this slide contains no place holder attributes.
         */
        this.getAllPlaceHolderAttributesForId = function (target) {
            return placeHolderCollector[target] || null;
        };

        /**
         * Receiving the complete list style object. This is necessary, if a snap shot is created, because these
         * list styles are not saved in the data object of the drawing. The other attributes are all saved in the
         * data-object, so that they can be restored later.
         *
         * Info: This function shall only be used for snap shot creation.
         *
         * @returns {Object}
         *  The collector object for the list style attributes assigned to place holder drawings.
         */
        this.getAllDrawingListStyleAttributes = function () {

            var // the object containing the drawing list styles
                allDrawingListStyles = null;

            // iterating over target, type and index
            _.each(_.keys(placeHolderCollector), function (target) {
                _.each(_.keys(placeHolderCollector[target]), function (type) {
                    _.each(_.keys(placeHolderCollector[target][type]), function (index) {

                        var drawingAttrs = placeHolderCollector[target][type][index];

                        if (drawingAttrs.listStyle) {
                            allDrawingListStyles = allDrawingListStyles || {};
                            allDrawingListStyles[target] = allDrawingListStyles[target] || {};
                            allDrawingListStyles[target][type] = allDrawingListStyles[target][type] || {};
                            allDrawingListStyles[target][type][index] = allDrawingListStyles[target][type][index] || {};

                            allDrawingListStyles[target][type][index].listStyle = _.copy(drawingAttrs.listStyle, true);
                        }

                    });
                });
            });

            return allDrawingListStyles;
        };

        /**
         * Setting the full object for the drawing list styles. This is necessary after applying a snapshot.
         *
         * Info: This function shall only be used for applying snapshot.
         *
         * @param {Object} listStyles
         *  The object containing the full set of drawing list styles. This was typically saved before using
         *  the function 'getAllDrawingListStyleAttributes'.
         */
        this.setAllDrawingListStyleAttributes = function (listStyles) {

            // iterating over target, type and index
            _.each(_.keys(listStyles), function (target) {
                _.each(_.keys(listStyles[target]), function (type) {
                    _.each(_.keys(listStyles[target][type]), function (index) {

                        var drawingAttrs = listStyles[target][type][index];

                        if (drawingAttrs.listStyle) {
                            placeHolderCollector = placeHolderCollector || {};
                            placeHolderCollector[target] = placeHolderCollector[target] || {};
                            placeHolderCollector[target][type] = placeHolderCollector[target][type] || {};
                            placeHolderCollector[target][type][index] = placeHolderCollector[target][type][index] || {};

                            placeHolderCollector[target][type][index].listStyle = _.copy(drawingAttrs.listStyle, true);
                        }
                    });
                });
            });
        };

        /**
         * Emptying the complete place holder model. This is necessary, if a new snapshot is applied.
         */
        this.emptyPlaceHolderModel = function () {
            emptyPlaceHolderModel();
        };

        /**
         * Receiving a valid set for the properties 'type' and 'index', so that the model can be accessed.
         * Typically the specified attributes parameter is the set of explicit drawing attributes.
         *
         * @param {Object} attrs
         *  The attributes object. This are typically the explicit attributes set at a drawing.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'type' and 'index'. 'type' is a string, that describes
         *  the place holder type, 'index' is a number describing the index specified for the given
         *  place holder type.
         *  If the specified attribute does not contain a 'presentation' property or inside this
         *  'presentation' property neither 'phType' nor 'phIndex' are defined, null is returned.
         */
        this.getValidTypeIndexSet = function (attrs) {

            var // the place holder type
                type = null,
                // the place holder index
                index = null;

            // it is necessary that at least one of phType or phIndex is defined in the attributes
            if (!attrs || !attrs.presentation || !('phType' in attrs.presentation || 'phIndex' in attrs.presentation)) { return null; }

            type = Utils.getStringOption(attrs.presentation, 'phType', PresentationUtils.getDefaultPlaceHolderType());
            index = Utils.getIntegerOption(attrs.presentation, 'phIndex', PresentationUtils.getDefaultPlaceHolderIndex());

            return { type: type, index: index };
        };

        /**
         * Getting the list styles for a specified drawing that are saved in the model. Or null, if the
         * specified drawing is no place holder drawing or if no list styles exist for this drawing.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node to be checked.
         *
         * @returns {Object|Null}
         *  The list styles for the specified drawing that are saved in the model. Or null, if the
         *  specified drawing is no place holder drawing or if no list styles exist for this drawing.
         */
        this.getAllListStylesAtPlaceHolderDrawing = function (drawing) {

            var // an object containing target, type and index for the specified drawing
                descriptor = null,
                // the drawing list style object
                drawingListStyles = null;

            // getting target, type and index
            descriptor = PresentationUtils.getDrawingModelDescriptor(drawing);

            if (!descriptor) { return drawingListStyles; }  // this might not be a place holder drawing

            // the descriptor object must contain the keys 'target', 'type' and 'index' for the specified drawing

            if (placeHolderCollector[descriptor.target] &&
                placeHolderCollector[descriptor.target][descriptor.type] &&
                placeHolderCollector[descriptor.target][descriptor.type][descriptor.index] &&
                placeHolderCollector[descriptor.target][descriptor.type][descriptor.index].listStyle) {
                drawingListStyles = { listStyle: _.copy(placeHolderCollector[descriptor.target][descriptor.type][descriptor.index].listStyle, true) };
            }

            return drawingListStyles;
        };

        /**
         * Checking, whether a specified place holder type is allowed on a slide with a specified ID.
         * This function is especially used to find, if place holders of type 'dt', 'ftr' or 'sldNum'
         * are to be inserted on a slide or not. For all other place holder types, true is returned.
         *
         * @param {String} id
         *  The slide ID.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Object} [attrs]
         *  The attributes of family 'slide'. If these attributes are not specified, the specified id
         *  is used to determine this attributes. For performance reasons, this attributes can be given
         *  into this function, if they are already available.
         *
         * @returns {Boolean}
         *  Whether a specified place holder type is allowed on a slide with a specified ID. This value
         *  can only be false for place holders of type 'dt', 'ftr' or 'sldNum'.
         */
        this.isSupportedPlaceHolderForSlideId = function (id, phType, attrs) {

            var // whether the slide with the specified ID allows inserting of place holder drawings of the specified type
                isSupported = true,
                // the merged slide attributes, to determine, which place holder drawings will be inserted
                mergedSlideAttributes = attrs || docModel.getMergedSlideAttributesByFamily(id, 'slide');

            // The place holder drawings of type 'dt', 'ftr' and 'sldNum' are not always created
            // on the new slide. This is specified at the layout and/or master slide.
            if ((phType === 'dt' && !mergedSlideAttributes.isDate) ||
                (phType === 'ftr' && !mergedSlideAttributes.isFooter) ||
                (phType === 'sldNum' && !mergedSlideAttributes.isSlideNum)) {
                isSupported = false;
            }

            return isSupported;
        };

        /**
         * Helper function to find different and identical place holder drawings in two
         * specified layout slides.
         *
         * The return object contains three properties, that have an array as value. The
         * property names are 'onlyFirst', 'onlySecond' and 'both'.
         *
         * The values are arrays, that contain objects, that describe a place holder drawing
         * by its type and its index. Therefore these objects have the properties 'type'
         * and 'index'.
         *
         * So the properties 'onlyFirst', 'onlySecond' and 'both' describe the place holder
         * drawings that exist only on the first slide, only on the second slide or in both
         * slides.
         *
         * @param {String} firstId
         *  The ID of the first layout slide.
         *
         * @param {String} secondId
         *  The ID of the second layout slide.
         *
         * @param {Object} firstSlideAttrs
         *  The attributes of 'slide' family for the first slide
         *
         * @param {Object} secondSlideAttrs
         *  The attributes of 'slide' family for the second slide
         *
         * @returns {Object}
         *  A comparison object with the properties 'onlyFirst', 'onlySecond' and 'both' that
         *  describe the place holder drawings that exist only on the first slide, only on the
         *  second slide or in both slides.
         */
        this.comparePlaceHolderSet = function (firstId, secondId, firstSlideAttrs, secondSlideAttrs) {

            var // the comparison object with the properties 'onlyFirst', 'onlySecond' and 'both'
                comparison = { onlyFirst: [], onlySecond: [], both: [] },
                // the saved place holder attributes for the first specified layout ID
                first = self.getAllPlaceHolderAttributesForId(firstId),
                // the saved place holder attributes for the second specified layout ID
                second = self.getAllPlaceHolderAttributesForId(secondId);

            _.each(_.keys(first), function (type) {
                if (self.isSupportedPlaceHolderForSlideId(firstId, type, firstSlideAttrs)) {
                    _.each(_.keys(first[type]), function (index) {
                        if (second && second[type] && second[type][index] && self.isSupportedPlaceHolderForSlideId(secondId, type, secondSlideAttrs)) {
                            comparison.both.push({ type: type, index: parseInt(index, 10) });
                        } else {
                            comparison.onlyFirst.push({ type: type, index: parseInt(index, 10) });
                        }
                    });
                }
            });

            _.each(_.keys(second), function (type) {
                if (self.isSupportedPlaceHolderForSlideId(secondId, type, secondSlideAttrs)) {
                    _.each(_.keys(second[type]), function (index) {
                        if (!first || !first[type] || !first[type][index]) {
                            comparison.onlySecond.push({ type: type, index: parseInt(index, 10) });
                        }
                    });
                }
            });

            // some magic conversion:
            // title <-> ctrTitle
            // body <-> subTitle

            return comparison;
        };

        /**
         * After a change of the slide layout it is possible that the new
         * layout contains some other types for the place holder drawings
         * than the old layout. Some types are converted into each other.
         * For example a place holder drawing of type 'title' can be
         * interpreted as 'ctrTitle' after changing the layout slide.
         *
         * The allowed conversions are listed in the constant:
         * PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES.
         *
         * The parameter 'comparePlaceHolder' is an object created by the
         * function 'PresentationDrawingStyles.comparePlaceHolderSet'.
         * It has to contain the properties 'onlyFirst', 'onlySecond'
         * and 'both'.
         *
         * This parameter object is modified within this function.
         *
         * @param {Object} comparePlaceHolder
         *  The object that describe the differences in the place holder
         *  drawings of two layout slides. This object is generated by the
         *  function 'PresentationDrawingStyles.comparePlaceHolderSet' and
         *  contains the properties 'onlyFirst', 'onlySecond' and 'both'.
         *  This object is modified within this function.
         *
         * @returns {Object[]}
         *  An array of objects. The objects inside this array describe
         *  the allowed switches of drawing place holder. Every object
         *  contains the properties 'from' and 'to'. The values are
         *  the place holder drawing info objects, that contain the
         *  type and index for a place holder drawing.
         */
        this.analyzeComparePlaceHolder = function (comparePlaceHolder) {

            var // an array of supported switches for the place holder drawings
                switches = [];

            // helper function to find the specified type in the collector
            function foundType(type, allPlaceHolders) {
                return _.find(allPlaceHolders, function (placeHolder) {
                    return placeHolder.type === type;
                });
            }

            if (comparePlaceHolder && comparePlaceHolder.onlyFirst.length > 0 && comparePlaceHolder.onlySecond.length > 0) {

                _.each(PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES, function (value, key) {

                    var first = foundType(key, comparePlaceHolder.onlyFirst),
                        second = foundType(value, comparePlaceHolder.onlySecond);

                    // searching the key in onlyFirst and the value in onlySecond
                    if (first && second) {

                        // adding to switches object
                        switches.push({ from: first, to: second });

                        // removing items from collectors
                        comparePlaceHolder.onlyFirst = _.without(comparePlaceHolder.onlyFirst, first);
                        comparePlaceHolder.onlySecond = _.without(comparePlaceHolder.onlySecond, second);
                    }
                });
            }

            return switches;
        };

        /**
         * Collecting all place holder drawings on the specified slide, that have no definition in the
         * layout slide. This can happen after switching the layout slide. It is possible, that the new
         * layout slide has less place holder drawing definitions.
         *
         * @param {String} id
         *  The ID of the specified slide.
         *
         * @returns {jQuery}
         *  A jQuery collection of all place holder drawings, that have no definition in the layout slide.
         *  This can be drawings of any type, not only of type 'body'.
         */
        this.getAllUnassignedPhDrawings = function (id) {
            return self.getAllPlaceHolderDrawingsOnSlideByIndex(id, self.getMsDefaultPlaceHolderIndex());
        };

        /**
         * Helper function for updating all place holder drawings on a
         * specified slide.
         *
         * @param {Node|jQuery} slide
         *  The (jQueryfied) slide node.
         */
        this.updatePlaceHolderDrawings = function (slide) {

            // simply update all drawings that are place holder types
            _.each($(slide).children('div[placeholdertype]'), function (drawing) {

                // updating the drawing itself
                self.updateElementFormatting($(drawing));

                // ... and updating the paragraphs
                _.each($(drawing).find(DOM.PARAGRAPH_NODE_SELECTOR), function (para) {
                    // updating all paragraphs, that have a modified attribute set
                    docModel.getParagraphStyles().updateElementFormatting($(para));
                    docModel.updateListsDebounced($(para));
                });

            });

        };

        /**
         * Handling the visualization of empty text frames for a complete slide.
         *
         * @param {Node|jQuery} slide
         *  The slide node for that the visualization is done for all empty
         *  text frames.
         */
        this.handleAllEmptyPlaceHolderDrawingsOnSlide = function (slide) {
            _.each($(slide).children('div.drawing'), function (drawing) {
                self.handleEmptyPlaceHolderDrawing(drawing);
            });
        };

        /**
         * Handling the visualization of empty text frames. In the standard view it is
         * necessary to assign a dotted line to these empty text frames and to insert
         * a template text.
         * In the layout or master view, such a dotted line is always assigned to the place
         * holder drawings.
         * In PowerPoint only place holder drawings get the dotted border (even if
         * they have a background color). Empty text frames do not get a border.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Object} [attributes]
         *  The (explicit) attributes of the drawing node. If they are not specified,
         *  the explicit attributes at the specified drawing node are evaluated.
         */
        this.handleEmptyPlaceHolderDrawing = function (drawing, attributes) {

            var // whether the default template text was inserted into the drawing
                addedText = false,
                // the id of the slide containing the specified drawing
                slideId = null;

            if (PresentationUtils.isPlaceHolderDrawing(drawing)) {
                handleBorderForEmptyTextFrames(drawing, attributes || AttributeUtils.getExplicitAttributes(drawing));
                addedText = handleTemplateText(drawing);
                if (addedText && docModel.isMasterView()) {
                    slideId = docModel.getSlideId($(drawing).parent());
                    if (slideId && !docModel.getSlideFormatManager().isUnformattedSlide(slideId)) {
                        docModel.forceTriggerOfSlideStateUpdateEvent(slideId);
                    }
                }

            }
        };

        /**
         * Collecting from a specified slide all place holder drawings with the specified
         * type and index. If the index is not specified, all drawings with the specified
         * type will be returned.
         *
         * @param {String} id
         *  The ID of the slide.
         *
         * @param {String|String[]} type
         *  The type(s) of the drawing. It is possible to determine several types. This is a
         *  convenience function, so that subtitle can be changed, too, if a body was modified.
         *  In this case the parameter 'index' will be ignored.
         *
         * @param {String|Number} [index]
         *  The index of the drawing. If not specified, all drawings with the specified type
         *  with any index will be collected.
         *
         * @returns {jQuery}
         *  A collection of all place holder drawings, that are direct children of
         *  the slide.
         */
        this.getAllPlaceHolderDrawingsOnSlideBySelector = function (id, type, index) {

            var // the slide node
                slide = docModel.getSlideById(id),
                // the selector specific to the place holder type and index. This is used to find the drawing
                // that need an update of formatting (the model is already updated).
                selector = '',
                // whether the index will be ignored
                ignoreIndexValue = false;

            if (_.isArray(type)) {
                _.each(type, function (oneType) {
                    if (selector) { selector += ', '; }
                    selector += 'div[placeholdertype=' + oneType + ']';
                    if (oneType === 'body') { selector += '[placeholderindex], div[placeholderindex]:not([placeholdertype])'; }
                });
            } else {

                ignoreIndexValue = (!_.isNumber(index) && !_.isString(index));  // check if the index is specified
                selector = 'div[placeholdertype=' + type + ']';

                // if type is body, the 'placeholderindex' must be correct, but 'placeholdertype' might not be set
                if (type === 'body') {
                    if (ignoreIndexValue) {
                        selector += '[placeholderindex], div[placeholderindex]:not([placeholdertype])';
                    } else {
                        selector += '[placeholderindex=' + index + '], div[placeholderindex=' + index + ']:not([placeholdertype])';
                    }
                }
            }

            return slide.children(selector);
        };

        /**
         * Collecting from a specified slide all place holder drawing of specified
         * type and index.
         *
         * @param {String} id
         *  The ID of the slide.
         *
         * @param {String} type
         *  The type of the drawing.
         *
         * @param {String|Number} index
         *  The index of the drawing.
         *
         * @returns {jQuery}
         *  A collection of the place holder drawings with the specified index. All those
         *  drawings are direct children of the slide. This function can be used to find
         *  all unassigned place holder drawings with index
         *  'PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX'.
         *  This drawings can be drawings of any type, not only of type 'body'.
         */
        this.getAllPlaceHolderDrawingsOnSlideByIndex = function (id, index) {

            var // the slide node
                slide = docModel.getSlideById(id),
                // the selector specific to the place holder index.
                selector = 'div[placeholderindex=' + index + ']';

            return slide.children(selector);
        };

        /**
         * Getter for the list of allowed drawing type switches during change of layout.
         *
         * @returns {Object}
         *  The object that contains all allowed drawing type switches during a change
         *  of the layout of a slide.
         */
        this.getListOfAllowedDrwawingTypeSwitches = function () {
            return PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES;
        };

        /**
         * Getter for the constant value of 'PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX'.
         *
         * @returns {Number}
         *  The default place holder index that is used in PP for place holder drawings
         *  that cannot be assigned to place holder drawing of current layout slide.
         */
        this.getMsDefaultPlaceHolderIndex = function () {
            return PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX;
        };

        /**
         * Getter for the master slide type converter.
         *
         * @returns {Object}
         *  The object describing the master slide type conversions.
         */
        this.getMasterSlideTypeConverter = function () {
            return _.copy(PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES);
        };

        /**
         * Setting list style attributes at the drawing node for non-place holder drawings (45279).
         * For place holder drawings there is a model, but it is also possible, that the
         * list styles are directly defined at a non-placeholder drawing node. In this case,
         * it is necessary to save this data directly at the node.
         *
         * Info: Using this data object must be an exception. It is only valid for non
         *       place holder drawings with directly set list styles. For all other drawings
         *       the model in the object 'placeHolderCollector' must be used.
         *
         * @param {jQuery|Node} drawingNode
         *  The drawing node.
         *
         * @param {Object} attrs
         *  The attributes object.
         */
        this.setDrawingListStyleAttributes = function (drawingNode, attrs) {
            if (attrs && attrs.listStyle && !PresentationUtils.isPlaceHolderDrawing(drawingNode)) {
                $(drawingNode).data('listStyle', attrs.listStyle);
            }
        };

        /**
         * Getting the list style attributes directly from the drawing node for non-place holder
         * drawings (45279). For place holder drawings there is a model, but it is also possible,
         * that the list styles are directly defined at a non placeholder drawing node. In this case,
         * it is necessary to read this data directly from the node.
         *
         * @param {jQuery|Node} drawingNode
         *  The drawing node.
         *
         * @param {Object} attrs
         *  The attributes object.
         */
        this.getDrawingListStyleAttributes = function (drawingNode) {
            return $(drawingNode).data('listStyle');
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('drawing', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class PresentationDrawingStyles

    // constants --------------------------------------------------------------

    /**
     * Helper object with allowed switches of drawing types during
     * a change of a layout slide.
     */
    PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES = {
        title: 'ctrTitle',
        ctrTitle: 'title',
        body: 'subTitle',
        subTitle: 'body'
    };

    /**
     * Helper object with forced switches for receiving the place holder
     * attributes from a master slide. Example: A master never has a place
     * holder of type 'subTitle'. Instead the 'body' must be used.
     *
     * Info: Please also check 'SlideAttributesMixin.LIST_TYPE_CONVERTER'.
     */
    PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES = {
        subTitle: 'body',
        ctrTitle: 'title'
    };

    /**
     * The default place holder index for drawings of type 'body'.
     */
    PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX = 4294967295;

    // exports ================================================================

    // derive this class from class DrawingStyleCollection
    return DrawingStyleCollection.extend({ constructor: PresentationDrawingStyles });

});
