/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/portal/collection/templatecollection', [
    'io.ox/office/portal/model/templatemodel'
], function (TemplateModel) {

    'use strict';

    var templateCollection = Backbone.Collection.extend({

        model: TemplateModel,

        /**
         * Fills this collection with passed array of template objects.
         *
         * @param {Array} templateList
         *  An array of template objects (infostore file descriptors)
         */
        setData: function (templateList) {

            var self = this;

            // populate the collection with template objects and
            // invokes an 'update' event that re-renders the view
            self.set(templateList);

        },

        /**
         * Get templates according to passed template type.
         *
         * @param {String} appBaseName
         *  The base name of the current application.
         *
         * @param {String = 'all'} source
         *  valid types: 'all', 'user', 'admin' or 'global'. Defaults to 'all' if omitted.
         *
         * @returns {Array}
         *  An array of template models.
         */
        getTemplates: function (appBaseName, source) {

            var models = this.models;

            source = source || 'all';

            function templateFilter(templateModel) {
                if (source === 'all') { return templateModel.get('type') === appBaseName; }
                return templateModel.get('source') === source && templateModel.get('type') === appBaseName;
            }

            return _.filter(models, templateFilter);

        },

        /**
         * Comparator function for sorting templates according to their filenames alphabetically.
         *
         * @param {Backbone.Model} model
         *  a template model.
         *
         * @returns {String}
         *  returns the values to be sorted.
         */
        comparator: function (model) {
            return model.get('filename').toLowerCase();
        }

    });

    return templateCollection;

});
