/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editview', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/app/extensionregistry',
    'io.ox/office/baseframework/view/baseview',
    'io.ox/office/baseframework/view/baselabels',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/editframework/view/toolbartabcollection',
    'io.ox/office/editframework/view/toppane',
    'io.ox/office/editframework/view/edittoolpane',
    'io.ox/office/editframework/view/searchpane',
    'io.ox/office/editframework/view/operationspane',
    'io.ox/office/editframework/view/clipboardpane',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/editframework/view/editdialogs',
    'io.ox/office/editframework/view/mobilesearchgroup',
    'io.ox/office/editframework/view/popup/userslayermenu',
    'gettext!io.ox/office/editframework/main',
    'less!io.ox/office/editframework/view/editstyle'
], function (Utils, ExtensionRegistry, BaseView, BaseLabels, Config, ToolBarTabCollection, TopPane, EditToolPane, SearchPane, OperationsPane, ClipboardPane, Labels, Controls, Dialogs, MobileSearchGroup, UsersLayerMenu, gt) {

    'use strict';

    var // default search configuration
        DEFAULT_SEARCH_SETTINGS = {
            // current search query
            query: '',
            // whether 'query' is a regular expression
            regexp: false,
            // the replacement string
            replace: ''
        },

        // beneath this minimum screen width or height, the tool/tab bars should be combined
        // Nexus 7 = min resolution 601
        PANES_COMBINED_MIN_SIZE = 610,

        // combined panes mode via URL flag in debug mode
        DEBUG_PANES_COMBINED = Config.getDebugUrlFlag('office:panes-combined');

    // class EditView =========================================================

    /**
     * Base class for the application view of all editor applications.
     *
     * @constructor
     *
     * @extends BaseView
     *
     * @param {EditApplication} app
     *  The application containing this view instance.
     *
     * @param {EditModel} docModel
     *  The document model created by the passed application.
     *
     * @param {Function} searchHandler
     *  A callback function that will implement searching for a specific query
     *  string entered in the search/replace tool bar. Receives the following
     *  parameters:
     *  (1) {String} command
     *      The search command to be executed. See description of the method
     *      EditView.executeSearchOperation() for details.
     *  (2) {Object} settings
     *      The current search settings. See the description of the method
     *      EditView.getSearchSettings() for details about the properties
     *      contained in this object.
     *  The function may return a promise, if the search action involves
     *  asynchronous operations. The view will be locked until the promise has
     *  been resolved or rejected. The promise may be rejected with an error
     *  code string which may result in an alert message box shown in the user
     *  interface. Alternatively, the function may return an explicit error
     *  code string synchronously (without using a promise). The function will
     *  be called in the context of this view instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  constructor of the base class BaseView, with the following additions
     *  and differences:
     *  @param {Function} [initOptions.initDebugHandler]
     *      A callback handler function called for additional initialization of
     *      debug settings. Will be called in debug mode ONLY. Receives the
     *      following parameters:
     *      (1) {OperationsPane} operationsPane
     *          The view pane containing the operations log, and additional
     *          debugging information to be displayed.
     *      (2) {ClipboardPane} clipboardPane
     *          The view pane with the HTML mark-up contained in the browser
     *          clipboard.
     *      May return a promise to be able to run asynchronous code during
     *      initialization.
     *  @param {Function} [initOptions.initGuiHandler]
     *      A callback handler function called to initialize the graphical
     *      elements of this view instance, after the document has been
     *      imported. Receives the following parameters:
     *      (1) {CompoundButton} viewMenuGroup
     *          The 'View' drop-down menu shown in the tool pane, containing
     *          all view settings for the document.
     *      May return a promise to be able to run asynchronous code during
     *      initialization.
     *  @param {Function} [initOptions.initDebugGuiHandler]
     *      A callback handler function called to add graphical elements for
     *      debugging purposes to this view instance. Will be called in debug
     *      mode ONLY. Will be called after the GUI initialization handler (see
     *      option 'initGuiHandler'). Receives the following parameters:
     *      (1) {CompoundButton} viewMenuGroup
     *          The 'View' drop-down menu shown in the tool pane, containing
     *          all view settings for the document.
     *      (2) {CompoundButton} actionMenuGroup
     *          The 'Actions' drop-down menu shown in the debug tool bar,
     *          containing action commands for debugging.
     *      May return a promise to be able to run asynchronous code during
     *      initialization.
     *  @param {Boolean} [initOptions.enablePageSettings=false]
     *      If set to true, the file toolbar tab will contain a page toolbar,
     *      containing a page settings button to show a page settings dialog
     *      for the current document.
     */
    function EditView(app, docModel, searchHandler, initOptions) {

        var // self reference
            self = this,

            // the tool bar tab collection
            toolBarTabs = null,

            // the top-level view pane containing global controls and the tool bar tabs
            topPane = null,

            // the main tool pane below the top-level view pane containing the tool bars
            toolPane = null,

            // the search/replace pane below the tool pane
            searchGroup = null,

            // the current search configuration
            searchSettings = _.clone(DEFAULT_SEARCH_SETTINGS),

            // collaborator list
            collaboratorPopup = null,

            // show combined Panes
            panesCombined = false,

            userLayerShowOwnColor = Utils.getBooleanOption(initOptions, 'userLayerShowOwnColor', false);

        // base constructor ---------------------------------------------------

        BaseView.call(this, app, docModel, Utils.extendOptions(initOptions, {
            initHandler: initHandler,
            initGuiHandler: initGuiHandler,
            classes: 'io-ox-office-edit-main'
        }));

        // private methods ----------------------------------------------------

        /**
         * Builds generic contents of this edit view instance: the side pane,
         * the overlay panes, and the debug view panes. After that, calls the
         * passed callback handler to build the application-specific view
         * elements.
         */
        function initHandler() {

            // the operations pane (debug mode only)
            var operationsPane = null;
            // the clipboard pane (debug mode only)
            var clipboardPane = null;

            // determine whether the panes are combined or not
            panesCombined = DEBUG_PANES_COMBINED || (Utils.TOUCHDEVICE && (Math.min(Utils.getScreenWidth(), Utils.getScreenHeight()) <= PANES_COMBINED_MIN_SIZE));

            // disable internal browser handles on tables and drawings after edit mode has changed
            self.listenTo(docModel, 'change:editmode', function () { self.disableBrowserEditControls(); });

            // create the tool bars (initialize themselves internally)
            self.addPane(topPane = new TopPane(self, toolBarTabs));
            self.addPane(toolPane = new EditToolPane(self, toolBarTabs, { landmark: false }));

            if (panesCombined) {
                searchGroup = new MobileSearchGroup(self);
            } else {
                searchGroup = new SearchPane(self);
                self.addPane(searchGroup);
            }

            // deactivate tool bar tabs while tool pane is hidden
            toolPane.on('pane:hide', function () { toolBarTabs.activateTab(null); });

            // hide tool bars after double click on tab buttons
            topPane.getNode().find('.group.tab-group').on('dblclick', function () {
                toolPane.hide();
            });

            // hide search pane when application is in internal error state
            this.listenTo(app, 'docs:state:error', function () { searchGroup.hide(); });

            // update controller immediately when toggling the collaborator pop-up menu
            this.listenTo(collaboratorPopup, 'popup:show popup:hide', function () {
                app.getController().update();
            });

            // initialize debug panes before other custom panes of the application
            if (Config.DEBUG) {

                // create and insert the clipboard pane, hide it initially
                clipboardPane = new ClipboardPane(self);
                clipboardPane.hide();
                self.addPane(clipboardPane);

                // create and insert the operations pane, show it according to URL flag
                operationsPane = new OperationsPane(self);
                if (!Config.getDebugUrlFlag('office:operations-pane')) { operationsPane.hide(); }
                self.addPane(operationsPane);

                // always show the operations pane when an internal application error occurs
                self.listenTo(app, 'docs:state:error', function () { operationsPane.show(); });

                // create controller items for debug functionality
                app.getController().registerDefinitions({
                    'debug/operationspane': {
                        parent: 'debug/enabled',
                        get: function () { return operationsPane.isVisible(); },
                        set: function (state) { operationsPane.toggle(state); }
                    },
                    'debug/clipboardpane': {
                        parent: 'debug/enabled',
                        get: function () { return clipboardPane.isVisible(); },
                        set: function (state) { clipboardPane.toggle(state); }
                    },
                    'debug/highlight': {
                        parent: 'debug/enabled',
                        get: function () { return app.getWindowNode().hasClass('debug-highlight'); },
                        set: function (state) { app.getWindowNode().toggleClass('debug-highlight', state); }
                    },
                    'debug/realtime/trigger': {
                        parent: 'debug/enabled',
                        set: function (type) { app.debugTriggerRealtimeEvent(type); }
                    },
                    'debug/realtime/sendinvalidoperation': {
                        parent: ['debug/enabled', 'document/editable'],
                        set: function (options) { app.debugSendInvalidOperation(options); }
                    },
                    'debug/operations/apply': {
                        parent: ['debug/enabled', 'document/editable'],
                        set: function (operation) { docModel.applyOperations(operation); }
                    },
                    'debug/slowsave': {
                        parent: 'debug/enabled',
                        enable: function () { return Config.getFlag('debugslowsave'); },
                        get: function () { return Config.getFlag('debugslowsave') && Config.getFlag('debugslowsaveuser'); },
                        set: function (state) { Config.set('debugslowsaveuser', state); }
                    },
                    'debug/view/busy': {
                        parent: 'debug/enabled',
                        set: function (cancelable) { return cancelable ? self.createAbortablePromise($.Deferred()) : $.Deferred(); },
                        cancel: true
                    },
                    'debug/quit': {
                        parent: 'debug/enabled',
                        set: function (type) { app.debugQuit(type); }
                    }
                });
            }

            // call initialization handler passed by sub class
            var result = Utils.getFunctionOption(initOptions, 'initHandler', $.noop).call(self);

            // call debug initialization handler passed by sub class
            if (Config.DEBUG) {
                result = $.when(result).then(_.bind(Utils.getFunctionOption(initOptions, 'initDebugHandler', $.noop), self, operationsPane, clipboardPane));
            }

            return result;
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler() {

            var
                renameToolbar = null,
                saveToolbar   = null,
                pageToolbar   = null,
                commonToolbar = null,
                actionToolbar = null,

                btnDownload        = new Controls.Button({ icon: Labels.DOWNLOAD_ICON, tooltip: gt('Download'), classes: 'link-style', dropDownVersion: { label: gt('Download') } }),
                btnPrint           = new Controls.Button({ icon: 'fa-print', tooltip: gt('Print as PDF'), classes: 'link-style', dropDownVersion: { label: gt('Print as PDF') } }),

                btnSendMail        = null,
                btnSendMailKey     = 'document/sendmail',

              // no configuration at all, automatically always defaults to the file-attachment build-in.
              //sendMailAttachmentConfig    = { attachMode: 'attachment' },
                sendMailPDFAttachmentConfig = { attachMode: 'pdf-attachment' },
                sendMailInlineHTMLConfig    = { attachMode: 'inline-html' },

                canConvertToPDFAttachment   = app.canConvertTo('pdf-mail-attachment'),
                canConvertToInlineHTML      = app.canConvertTo('inline-html-document'),

                btnReload          = new Controls.Button({ icon: 'fa-repeat', label: gt('Reload'), tooltip: gt('Reload document'), classes: 'link-style' }),
                btnAcquireEdit     = new Controls.AcquireEditButton(self, { classes: 'link-style' }),
                enablePageSettings = Utils.getBooleanOption(initOptions, 'enablePageSettings', false),
                btnShare           = new Controls.ShareButton(self, gt('Share'));

            /**
             *  provide different appearances of "send mail" button(s) in dependency
             *  of an application's set of available converters that each on their
             *  own do provide functionality for converting the current document's
             *  (entire) content.
             */
            if (canConvertToPDFAttachment || canConvertToInlineHTML) {

                btnSendMail        = new Controls.CompoundButton(self, {
                    visibleKey:      'document/sendmail/menu',
                    icon:            'fa-envelope-o',
                    classes:         'link-style',
                    tooltip:         BaseLabels.SEND_AS_MAIL_LABEL,
                    dropDownVersion: { label: BaseLabels.SEND_AS_MAIL_LABEL }
                })
                .addGroup('document/sendmail', new Controls.Button({ label: BaseLabels.ATTACH_AS_FILE_TO_MAIL_LABEL }));

                if (canConvertToPDFAttachment) {
                    btnSendMail
                        .addGroup('document/sendmail/pdf-attachment', new Controls.Button({ value: sendMailPDFAttachmentConfig, label: BaseLabels.ATTACH_AS_PDF_TO_MAIL_LABEL }));
                }
                if (canConvertToInlineHTML) {
                    btnSendMail
                        .addGroup('document/sendmail/inline-html', new Controls.Button({ value: sendMailInlineHTMLConfig, label: BaseLabels.SEND_CONTENT_AS_NEW_MAIL_LABEL }));
                }

                btnSendMailKey = 'document/sendmail/menu';

            } else { // defaults to the file-attachment build-in.

                btnSendMail        = new Controls.Button({
                    icon:            'fa-envelope-o',
                    classes:         'link-style',
                    tooltip:         BaseLabels.SEND_AS_MAIL_LABEL,
                    dropDownVersion: { label: BaseLabels.SEND_AS_MAIL_LABEL }
                });

                btnSendMailKey = 'document/sendmail';
            }

            // -----------------------------------------------------
            // TABS
            //      prepare all tabs (for normal or combined panes)
            // -----------------------------------------------------
            self.createToolBarTab('file', { label: Labels.FILE_HEADER_LABEL, priority: 10 });

            // -----------------------------------------------------
            // TOOLBARS
            //      prepare all toolbars (for normal or combined panes)
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                renameToolbar   = self.createToolBar('file');
                saveToolbar     = self.createToolBar('file', { priority: 2, visibleKey: 'app/bundled' });
            }

            if (enablePageSettings) {
                pageToolbar = self.createToolBar('file', { priority: 3, label: gt('Page'), classes: 'link-style', prepareShrink: true });
            }

            commonToolbar       = self.createToolBar('file', { priority: 3, label: gt('Actions'), classes: 'link-style', prepareShrink: true });
            actionToolbar       = self.createToolBar('file', { priority: 1, tooltip: gt('actions'), classes: 'link-style' });

            // -----------------------------------------------------
            // CONTROLS
            //      add all controls
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                var fileNameField       = new Controls.FileNameField(self),
                    cBoxAutosave        = new Controls.CheckBox({ label: gt('AutoSave') }),
                    btnSaveAs           = new Controls.Button({ label: gt('Save as'), value: 'file' }),
                    btnSaveAsTemplate   = new Controls.Button({ label: gt('Save as template'), value: 'template' }),
                    btnExportAsPdf      = new Controls.Button({ label: gt('Export as PDF'), value: 'pdf' });        // DOCS-122 :: misinterpreted story - [https://jira.open-xchange.com/browse/DOCS-122] :: but hereby saved for good.

                renameToolbar
                    .addGroup('document/rename', fileNameField);

                saveToolbar
                    .addGroup('view/saveas/menu', new Controls.CompoundButton(self, { label: gt('Save in Drive'), classes: 'link-style', smallerVersion: { icon: 'fa-save', hideLabel: true } })
                        .addGroup('document/saveas/dialog', btnSaveAs)
                        .addGroup('document/saveas/dialog', btnSaveAsTemplate)
                        .addSeparator()
                        .addGroup('document/exportaspdf/dialog', btnExportAsPdf)                                    // DOCS-122 :: misinterpreted story - [https://jira.open-xchange.com/browse/DOCS-122] :: but hereby saved for good.
                        .addSeparator()
                        .addGroup('document/autosave', cBoxAutosave)
                    );

                // register double-click handler for the application launcher
                self.waitForImportSuccess(function () {
                    $('#io-ox-topbar .launcher[data-app-guid="' + app.guid + '"] a.apptitle').on('dblclick', function () {
                        if (fileNameField.isEnabled()) {
                            self.executeControllerItem('view/toolbars/tab', 'file', { focusTarget: fileNameField });
                        } else {
                            self.grabFocus();
                        }
                    });
                });
            }

            if (pageToolbar) {
                if (enablePageSettings) {
                    pageToolbar.addGroup('document/pagesettings', new Controls.Button({
                        icon: 'docs-page-layout',
                        tooltip: gt('Page settings'),
                        classes: 'link-style',
                        dropDownVersion: { label: gt('Page settings') }
                    }));
                }
            }

            commonToolbar
                .addGroup('document/download', btnDownload)
                .addGap()
                .addGroup('document/print', btnPrint)
                .addGap()
                // provide different appearance of "send mail" button(s) in dependency of an application's ([app]'s) send mail operations/methods.
                .addGroup(btnSendMailKey, btnSendMail)
                .addGroup('document/share', btnShare);

            actionToolbar
                .addGroup('document/acquireeditOrPending', btnAcquireEdit, { visibleKey: 'document/acquireedit' })
                .addGroup('document/reload', btnReload, { visibleKey: 'document/reload' });

            // call initialization handler passed by sub class
            var result = Utils.getFunctionOption(initOptions, 'initGuiHandler', $.noop).call(self, topPane.getViewMenuGroup());

            // debug initialization
            if (Config.DEBUG) {
                result = $.when(result).then(initDebugGuiHandler);
            }

            // immediately activate the first available tool bar tab
            return $.when(result).done(function () { toolBarTabs.activateFirstTab(); });
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler() {

            var // the 'View' drop-down menu
                viewMenuGroup = topPane.getViewMenuGroup(),
                // the drop-down menu with debug actions
                actionMenuGroup = null;

            // add view options
            viewMenuGroup
                .addSectionLabel(_.noI18n('Debug'))
                .addGroup('debug/operationspane', new Controls.CheckBox({ label: _.noI18n('Show operations panel') }))
                .addGroup('debug/clipboardpane',  new Controls.CheckBox({ label: _.noI18n('Show clipboard panel') }))
                .addGroup('debug/console',        new Controls.CheckBox({ label: _.noI18n('Show overlay log console') }))
                .addGroup('debug/highlight',      new Controls.CheckBox({ label: _.noI18n('Highlight DOM elements') }));

            // create the drop-down list for the debug actions
            actionMenuGroup = new Controls.CompoundButton(self, { icon: 'fa-wrench', label: Labels.ACTIONS_LABEL, tooltip: _.noI18n('Additional debug actions'), classes: 'link-style' })
                // register realtime debug actions
                .addSectionLabel(_.noI18n('Offline mode'))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Switch to offline mode'),      value: 'offline' }))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Switch to online mode'),       value: 'online' }))
                .addSectionLabel(_.noI18n('Realtime connection'))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Simulate connection reset'),   value: 'reset' }))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Simulate connection timeout'), value: 'timeout' }))
                .addGroup('debug/realtime/trigger', new Controls.Button({ label: _.noI18n('Simulate connection hangup'),  value: 'hangup' }))
                // register debug actions for operations
                .addSectionLabel(_.noI18n('Operations'))
                .addGroup('debug/operations/apply',              new Controls.Button({ label: _.noI18n('Apply operation with invalid name'), value: { name: '_invalid_operation_' } }))
                .addGroup('debug/operations/apply',              new Controls.Button({ label: _.noI18n('Apply operation with invalid position'), value: { name: 'insertText', start: [987654321, 0], text: 'test' } }))
                .addGroup('debug/realtime/sendinvalidoperation', new Controls.Button({ label: _.noI18n('Send operation with invalid OSN'), value: { operation: { name: 'delete', start: [987654321, 0] }, badOsn: true } }))
                .addGroup('debug/realtime/sendinvalidoperation', new Controls.Button({ label: _.noI18n('Send invalid operation'), value: { operation: { name: 'createError', start: [987654321, 0] } } }))
                // register debug actions for busy screen
                .addSectionLabel(_.noI18n('Busy blocker'))
                .addGroup('debug/view/busy', new Controls.Button({ label: _.noI18n('Show permanent busy screen'), value: false }))
                .addGroup('debug/view/busy', new Controls.Button({ label: _.noI18n('Show busy screen with Cancel button'), value: true }))
                // register debug actions for application quit
                .addSectionLabel(_.noI18n('Quit'))
                .addGroup('debug/quit', new Controls.Button({ label: _.noI18n('Close document with unsaved changes'), value: 'unsaved' }))
                .addGroup('debug/quit', new Controls.Button({ label: _.noI18n('Close document with server error'),    value: 'error' }));

            searchGroup.handleTabListButtons();

            // create a new tool bar tab for debugging
            self.createToolBarTab('debug', { label: _.noI18n('Debug'), visibleKey: 'debug/enabled', priority: 9999 });
            self.createToolBar('debug').addGroup(null, actionMenuGroup);

            // create a new tool bar for debugging containing debug options
            self.createToolBar('debug')
                .addGroup('debug/slowsave', new Controls.Button({ icon: 'fa-floppy-o', tooltip: _.noI18n('Slow down save document (' + Config.get('debugslowsavetime', 0) + 's)'), toggle: true }));

            // call initialization handler passed by sub class
            return Utils.getFunctionOption(initOptions, 'initDebugGuiHandler', $.noop).call(self, viewMenuGroup, actionMenuGroup);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the top pane containing the tool bar tab buttons, and other
         * global control elements.
         *
         * @returns {TopPane}
         *  The top pane of this application.
         */
        this.getTopPane = function () {
            return topPane;
        };

        /**
         * Returns the tool pane containing all existing tool bars.
         *
         * @returns {EditToolPane}
         *  The tool pane of this application.
         */
        this.getToolPane = function () {
            return toolPane;
        };

        /**
         * Returns the collection containing data for all registered tool bar
         * tabs.
         *
         * @returns {ToolBarTabCollection}
         *  The collection containing data for all registered tool bar tabs.
         */
        this.getToolBarTabs = function () {
            return toolBarTabs;
        };

        /**
         * Creates a new tab button in the top view pane used to control the
         * visibility of one or more tool bar components in the tool pane of
         * the application view.
         *
         * @param {String} tabId
         *  The unique identifier for the new tab button.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  ToolBarTabCollection.createTab() used to control the behavior and
         *  visibility of the tab, and RadioGroup.createOptionButton() used to
         *  create the tab button control (especially label and icon settings).
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.createToolBarTab = function (tabId, options) {
            toolBarTabs.createTab(tabId, options);
            return this;
        };

        /**
         * Creates a new tool bar in the tool pane of the application view.
         *
         * @param {String} tabId
         *  The identifier of the tab button the tool bar will be associated
         *  with.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  EditToolPane.createToolBar().
         *
         * @returns {ToolBar}
         *  The new tool bar instance.
         */
        this.createToolBar = function (tabId, options) {
            return toolPane.createToolBar(tabId, options);
        };

        /**
         * Returns the search group.
         *
         * @returns {SearchPane|MobileSearchGroup}
         *  The search pane of this application.
         */
        this.getSearchGroup = function () {
            return searchGroup;
        };

        /**
         * Returns whether the search pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the search pane is currently visible.
         */
        this.isSearchActive = function () {
            return searchGroup.isVisible();
        };

        /**
         * Returns the tool pane.
         *
         * @returns {EditToolPane}
         *  The tool pane of this application.
         */
        this.getEditToolPane = function () {
            return toolPane;
        };

        /**
         * Registers a label text and optional icon for a specific user defined
         * application state.
         *
         * @param {String} state
         *  The identifier of an application state.
         *
         * @param {String} label
         *  The label text to be shown while the application state is active.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.icon]
         *      The CSS class of an icon to be shown in front of the label.
         *  @param {Boolean} [options.busy=false]
         *      If set to true, a rotating busy icon will be shown in front of
         *      the label. This option overrides the option 'icon'.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.registerStatusCaption = function (state, label, options) {
            topPane.getAppStatusLabel().registerCaption(state, label, options);
            return this;
        };

        /**
         * Creates a new editable container node for clipboard functionality,
         * and inserts it into the hidden container node of the application.
         *
         * @returns {jQuery}
         *  The new empty clipboard container node.
         */
        this.createClipboardNode = function () {
            // clipboard node must be content-editable to allow system copy-to-clipboard
            // clipboard node must contain 'user-select-text' to be focusable in Chrome
            var clipboardNode = $('<div class="clipboard user-select-text" contenteditable="true" data-focus-role="clipboard">');
            this.insertHiddenNodes(clipboardNode);
            return clipboardNode;
        };

        /**
         * Disables the internal resize handles shown for of tables, drawing
         * objects, etc. provided by the browser edit mode (content-editable
         * elements).
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.disableBrowserEditControls = function () {
            // execute delayed to prevent conflicts with ongoing event handling
            this.executeDelayed(function () {
                try {
                    // disable FireFox table manipulation handlers
                    window.document.execCommand('enableObjectResizing', false, false);
                    window.document.execCommand('enableInlineTableEditing', false, false);
                } catch (ex) {
                }
            }, undefined, 'EditFramework: disableBrowserEditControls');
            return this;
        };

        /**
         * Closes the passed dialog automatically, if the document switches to
         * read-only mode.
         *
         * @param {ModalDialog} dialog
         *  The dialog instance that will be closed on read-only mode.
         *
         * @param {Function} [closeCallback]
         * A callback handler function called if document switches to read-only mode
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.closeDialogOnReadOnlyMode = function (dialog, closeCallback) {

            // close the dialog if document switches to read-only mode
            function changeEditModeHandler(event, editMode) {
                if (!editMode) {
                    dialog.close();
                    if (_.isFunction(closeCallback)) {
                        closeCallback();
                    }
                }
            }

            // listen to read-only mode, close dialog automatically
            this.listenTo(docModel, 'change:editmode', changeEditModeHandler);

            // unregister listener after closing the dialog
            dialog.on('close', function () {
                self.stopListeningTo(docModel, 'change:editmode', changeEditModeHandler);
            });

            return this;
        };

        /**
         * Shows a modal query dialog with Yes/No buttons (an instance of the
         * class ModalQueryDialog). The dialog will be closed automatically, if
         * the application loses edit rights.
         *
         * @param {String|Null} title
         *  The title text for the dialog; or null for a dialog without title.
         *
         * @param {String} message
         *  The message text shown in the dialog body.
         *
         * @param {Object} [options]
         *  Additional options passed to the dialog constructor.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved after pressing
         *  the Yes button; or rejected after pressing the No button, or losing
         *  edit rights.
         */
        this.showQueryDialog = function (title, message, options) {

            // create a query dialog instance
            var dialog = new Dialogs.ModalQueryDialog(message, _.extend({}, options, { title: title }));

            // close dialog automatically when losing edit rights
            this.closeDialogOnReadOnlyMode(dialog);

            // show the dialog, return its promise
            return dialog.show();
        };

        /**
         * Toggles the visibility of the collaborator popup
         *
         * @param {Boolean} visibility
         */
        this.toggleCollaboratorPopup = function (visibility) {
            collaboratorPopup.toggle(visibility);
        };

        /**
         * Gets the instance of the collaborator popup
         *
         * @returns {UsersLayerMenu}
         */
        this.getCollaboratorPopup = function () {
            return collaboratorPopup;
        };

        /**
         * Shows a text input dialog that allows to enter a new file name for
         * this document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the document has been renamed
         *  successfully, or rejected when the dialog has been canceled, or
         *  renaming the document has failed.
         */
        this.showRenameDialog = function () {

            var // the dialog instance
                dialog = new Dialogs.ModalInputDialog({
                    title: gt('Rename document'),
                    okLabel: gt('Rename'),
                    value: this.getShortFileName(),
                    placeholder: gt('Document name')
                });

            // close dialog automatically after losing edit rights
            this.closeDialogOnReadOnlyMode(dialog);

            // show the dialog, and try to rename the document
            return dialog.show().then(function () {
                return app.rename(dialog.getText());
            });
        };

        /**
         * Saves the current document to the specified folder using the
         * current file name without extension and prefixing it with
         * the phrase 'Copy of'.
         *
         * @param {String} [type='file']
         *  The target file type. The following file types are supported:
         *  - 'file' (default): Save with the current extension.
         *  - 'template': Save as template file.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  file has been copied successfully; or that will be rejected, if
         *  copying the file has failed.
         */
        this.showSaveAsDialog = function (type) {

            var // the current file name
                oldFileName = app.getShortFileName(),
                // the new file name
                newFileName = null,
                // template extension
                templateExt = null,
                // title of the dialog
                title = null,
                // the dialog instance
                dialog = null;

            // prepare file extension and dialog title
            if (type === 'template') {

                newFileName = oldFileName;

                // use original extension as fallback - possible if a template
                // document should be stored via save as as template.
                templateExt = ExtensionRegistry.getTemplateExtensionForFileName(app.getFullFileName()) || app.getFileExtension();

                //#. %1$s is the file extension used for the template file
                //#, c-format
                title = gt('Save as template (%1$s)', _.noI18n(templateExt));

                // create the dialog
                dialog = new Dialogs.SaveAsTemplateDialog(this, {
                    title: title,
                    value: newFileName
                });
            } else if (type === 'pdf') {  // DOCS-122 :: misinterpreted story - [https://jira.open-xchange.com/browse/DOCS-122] :: but hereby saved for good.

            //  newFileName = oldFileName + '.pdf';
                newFileName = oldFileName;
                var
                    len     = newFileName.length,
                    idx     = newFileName.lastIndexOf('.');

                newFileName = newFileName.substring(0, ((idx >= 0) ? idx : len)) + '.pdf';
                title = gt('Export as PDF');

                // create the dialog
                dialog = new Dialogs.SaveAsFileDialog({ title: title, value: newFileName, preselect: app.getFileParameters().folder_id, exportAsPdf: true });

            } else {

                //#. %1$s is the name of the original file used to create a new copy
                //#, c-format
                newFileName = gt('Copy of %1$s', _.noI18n(oldFileName));

                title = gt('Save as');

                // create the dialog
                dialog = new Dialogs.SaveAsFileDialog({ title: title, value: newFileName, preselect: app.getFileParameters().folder_id });
            }

            // show the dialog, and perform the save action
            return dialog.show().then(function () {
                return app.saveDocumentAs(dialog.getText(), dialog.getSelectedFolderId(), type);
            });
        };

        /**
         * Shows a restore document dialog to the user, where he/she can select
         * the target folder and file name of the restored document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  file has been copied successfully; or that will be rejected, if
         *  copying the file has failed.
         */
        this.showRestoreDialog = function () {

            var // the current file name
                oldFileName = app.getShortFileName(),
                // the new file name
                newFileName = null,
                // title of the dialog
                title = null,
                // the dialog instance
                dialog = null;

            //#. %1$s is the name of the original file used to create a new copy
            //#, c-format
            newFileName = gt('Restore of %1$s', _.noI18n(oldFileName));

            title = gt('Restore document');

            // create the dialog
            dialog = new Dialogs.SaveAsFileDialog({ title: title, value: newFileName, preselect: app.getFileParameters().folder_id });

            // show the dialog, and perform the save action
            return dialog.show().then(function () {
                return { text: dialog.getText(), folder_id: dialog.getSelectedFolderId() };
            });
        };

        /**
         * return whether tool-/tabbars should be combined or not
         *
         * @returns {Boolean}
         *  'true' for combining the bars
         */
        this.panesCombined = function () {
            return panesCombined;
        };

        /**
         * Returns the effective zoom factor of the document. This method MUST
         * be overwritten by the respective applications sub classes.
         *
         * @attention
         *  Existence of this method is expected by code in base modules, e.g.
         *  the drawing layer.
         *
         * @returns {Number}
         *  The effective zoom factor for the document.
         */
        this.getZoomFactor = function () {
            Utils.error('EditView.getZoomFactor(): missing implementation');
            throw new Error('abstract method called');
        };

        // undo manager -------------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return docModel.getUndoManager().getUndoCount() > 0;
        };

        /**
         * Applies the topmost undo action on the undo stack.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            return docModel.getUndoManager().undo();
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack.
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return docModel.getUndoManager().getRedoCount() > 0;
        };

        /**
         * Applies the topmost redo action on the redo stack.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            return docModel.getUndoManager().redo();
        };

        // search/replace -----------------------------------------------------

        /**
         * Returns the current search settings, as currently configured in the
         * search pane.
         *
         * @returns {Object}
         *  The current search settings, with the following properties:
         *  - {String} query
         *      The search query string.
         *  - {Boolean} regexp
         *      Whether the query string shall be interpreted as regular
         *      expression, instead of literal search string.
         */
        this.getSearchSettings = function () {
            return _.clone(searchSettings);
        };

        /**
         * Changes a property in the search configuration, and initiates a new
         * search session with the current settings by invoking the search
         * handler passed to the constructor.
         *
         * @param {String} name
         *  The name of the property to be changed in the search configuration.
         *  MUST be the name of a supported search configuration property.
         *
         * @param {Any} value
         *  The new value for the configuration property.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when
         *  searching with the new configuration has finished, or rejected with
         *  an error code that may lead to an alert message box.
         */
        this.changeSearchSettings = function (newSettings) {
            _.extend(searchSettings, newSettings);
            return this;
        };

        /**
         * Invokes the search/replace handler passed to the constructor of this
         * instance.
         *
         * @param {String} command
         *  The search/replace command to be executed. Will be one of the
         *  following commands:
         *  - 'search:start'
         *      Start a new search/replace session. Will be sent after entering
         *      some text in the query input field, or changing any search
         *      options which influence the matching document contents.
         *  - 'search:prev'
         *      Search for the previous occurrence of the current query string.
         *  - 'search:next'
         *      Search for the next occurrence of the current query string.
         *  - 'search:end'
         *      Finish the current search/replace session, remove all feedback
         *      for search/replace in the user interface. Will be sent after
         *      closing the search pane.
         *  - 'replace:next'
         *      Replace the next occurrence of the current query string with
         *      the replacement text.
         *  - 'replace:all'
         *      Replace all occurrences of the current query string with the
         *      replacement text.
         *
         * @param {Object} [newSettings]
         *  New properties for the search configuration object. Will be
         *  inserted into the configuration before invoking the serahc handler.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  search handler has finished the specified operation, or that will
         *  be rejected on any error.
         */
        this.executeSearchOperation = function (command, newSettings) {

            var // the original result of the search handler
                result = null,
                // the promise returned by this method
                promise = null;

            // change the specified configuration properties
            if (_.isObject(newSettings)) { this.changeSearchSettings(newSettings); }

            // invoke the search handler
            result = searchHandler.call(this, command, _.clone(searchSettings));

            // convert error code (non-empty strings) to rejected promise
            promise = (_.isString(result) && result) ? $.Deferred().reject(result) : $.when(result);

            // TODO: show alert messages for generic errors
            promise.fail(function (result) {
                Utils.error('executeSearchOperation failed', result);
            });

            return promise;
        };

        // initialization -----------------------------------------------------

        // create class members relying on complete view
        toolBarTabs = new ToolBarTabCollection(this);
        collaboratorPopup = new UsersLayerMenu(this, { showOwnColor: userLayerShowOwnColor });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            toolBarTabs.destroy();
            collaboratorPopup.destroy();
            app = searchHandler = initOptions = self = docModel = null;
            toolBarTabs = topPane = toolPane = searchGroup = null;
            collaboratorPopup = null;
        });

    } // class EditView

    // exports ================================================================

    // derive this class from class BaseView
    return BaseView.extend({ constructor: EditView });

});
