/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/autostylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/operations',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/attributedmodel'
], function (Utils, SimpleMap, TimerMixin, ModelObject, Operations, AttributeUtils, AttributedModel) {

    'use strict';

    // class AutoStyleModel ===================================================

    /**
     * The representation of an auto-style, used as entries in an auto-style
     * collection.
     *
     * @constructor
     *
     * @extends AttributedModel
     */
    var AutoStyleModel = AttributedModel.extend({ constructor: function (docModel, styleFamily, attributeSet) {

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, attributeSet, { styleFamily: styleFamily, listenToStyles: false });

    } }); // class AutoStyleModel

    // class AutoStyleCollection ==============================================

    /**
     * A collection for all auto-styles of a specific attribute family in a
     * document. Auto-styles are used to collect a set of formatting attributes
     * under a specific identifier (similar to style sheets), in order to
     * prevent to store large attribute set objects at many small objects in a
     * document. Auto-styles are always immutable, and they cannot be removed
     * from this collection.
     *
     * Instances will trigger the following events:
     *  - 'insert:autostyle': After a new auto-style has been inserted into
     *      this collection. The event handler receives the identifier of the
     *      new auto-style.
     *  - 'delete:autostyle': After an auto-style has been deleted from this
     *      collection. The event handler receives the identifier of the
     *      deleted auto-style.
     *  - 'change:autostyle': After the formatting attributes of an auto-style
     *      in this collection have been changed. The event handler receives
     *      the identifier of the changed auto-style.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {String} styleFamily
     *  The main attribute family represented by the auto-styles contained in
     *  this collection. Used to resolve the attribute 'styleId' of an
     *  auto-style which refers to its parent style sheet.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.standardPrefix='a']
     *      The standard prefix used to generate the identifiers for new
     *      auto-styles (the prefix, followed by an increasing integer index).
     *      Also used for 'strict name mode' (see option 'strictMode' below).
     *  @param {Boolean} [initOptions.strictMode=false]
     *      If set to true, the 'insertAutoStyle' operations for creating new
     *      auto-styles  MUST use identifiers with the prefix specified in the
     *      option 'standardPrefix', followed by an integer index that strictly
     *      increases by one per new auto-style, starting at the index 0.
     *      Example: The first auto-style inserted into this collection MUST
     *      have the identifier 'a0' when using the standard prefix 'a', the
     *      second auto-style MUST have the identifier 'a1', and so on. By
     *      default, auto-styles can use all possible non-empty identifiers, as
     *      long as they are unique. Additionally, in non-strict mode, this
     *      collection will accept the identifiers of real style sheets when
     *      querying for auto-style attributes etc. In this case, the names of
     *      existing style sheets will not be used as identifiers for new
     *      auto-styles generated by this collection.
     *  @param {Boolean} [initOptions.keepComplete=false]
     *      If set to true, the explicit attribute sets in the auto-styles will
     *      be kept complete. All missing or deleted attributes will be filled
     *      automatically with the default values.
     *  @param {Function} [initOptions.updateHandler]
     *      A callback function that will be invoked for every new auto-style
     *      inserted into this collection, for every auto-style that will be
     *      deleted from this collection, and after the merged attribute set of
     *      an auto-style has been changed due to a modified style sheet. This
     *      callback function will be invoked before triggering any events to
     *      external listeners, and is intended to be used by subclasses of
     *      this class to perform internal updates before broadcasting the
     *      collection change. Receives the following parameters:
     *      (1) {String} styleId
     *          The identifier of the new, modified, or deleted auto-style.
     *      (2) {String} type
     *          The type of the auto-style operation. Will be one of 'insert',
     *          'delete', or 'change'.
     *      (3) {Boolean} isStyleSheet
     *          Whether changing a real style sheet has caused to invoke the
     *          callback function, instead of an auto-style contained in this
     *          collection. Can only happen in non-strict mode where the
     *          identifiers of style sheets can be used additionally to the
     *          identifiers of auto-style to resolves attribute sets etc. See
     *          description of the option 'strictMode' above for more details.
     */
    function AutoStyleCollection(docModel, styleFamily, initOptions) {

        // the style sheet collection for the passed style family
        var styleSheets = docModel.getStyleCollection(styleFamily);

        // the names of all supported attribute families, as flag set
        var supportedFamilySet = styleSheets.getSupportedFamilies();

        // all registered auto-styles, mapped by unique identifier
        var autoStyleMap = new SimpleMap();

        // the serialized attribute keys of all auto-styles, mapped by identifier
        var attrKeysByStyleId = new SimpleMap();

        // all auto-style identifiers, mapped by serialized attribute key
        var styleIdsByAttrKey = new SimpleMap();

        // identifier of the default auto style
        var defaultStyleId = '';

        // the default prefix for new identifiers, and for the strict mode
        var standardPrefix = Utils.getStringOption(initOptions, 'standardPrefix', 'a', true);

        // a regular expression matching style identifiers in strict mode,
        // the first capturing group is the index following the prefix
        var standardIdRegExp = new RegExp('^' + standardPrefix + '(0|[1-9]\\d*)$');

        // whether to use the strict mode for identifiers with increasing indexes
        var strictMode = Utils.getBooleanOption(initOptions, 'strictMode', false);

        // whether to keep the explicit attribute sets complete
        var keepComplete = Utils.getBooleanOption(initOptions, 'keepComplete', false);

        // callback function for new or changed auto-styles
        var updateHandler = Utils.getFunctionOption(initOptions, 'updateHandler', _.noop).bind(this);

        // expected index for the next inserted auto-style identifier
        var nextStyleIndex = 0;

        // base constructors --------------------------------------------------

        ModelObject.call(this, docModel);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        function deleteAttributeKey(styleId) {
            var oldAttrKey = attrKeysByStyleId.remove(styleId);
            styleIdsByAttrKey.remove(oldAttrKey);
        }

        /**
         * Additional processing for a new or changed auto-style.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style.
         *
         * @param {String} attrKey
         *  A unique map key for the formatting attributes of the auto-style.
         *
         * @param {Boolean} insert
         *  Whether the specified auto-style has been inserted (true), or
         *  changed (false).
         */
        function updateAutoStyle(styleId, attrKey, insert) {

            // update the attribute key mappings
            attrKeysByStyleId.insert(styleId, attrKey);
            styleIdsByAttrKey.insert(attrKey, styleId);

            // invoke the user-defined callback handler
            updateHandler(styleId, insert ? 'insert' : 'change', false);
        }

        /**
         * Creates a new auto-style, and inserts it into this collection.
         *
         * @param {String} styleId
         *  The unique identifier of the new auto-style.
         *
         * @param {Object} attributeSet
         *  The formatting attributes for the new auto-style.
         */
        function createAutoStyle(styleId, attributeSet) {

            // create and insert a new auto-style instance
            var autoStyle = new AutoStyleModel(docModel, styleFamily, attributeSet);
            autoStyleMap.insert(styleId, autoStyle);

            // initial update for the new auto-style
            var attrKey = Utils.stringifyJSON(attributeSet);
            updateAutoStyle(styleId, attrKey, true);

            // update the auto-style again, if the attributes have been changed (due to a changed style sheet)
            autoStyle.on('change:attributes', function () {

                // delete the old attribute key mapping
                deleteAttributeKey(styleId);

                // calculate the new attribute key, and update the internal maps
                var newAttrKey = Utils.stringifyJSON(autoStyle.getExplicitAttributeSet(true));
                updateAutoStyle(styleId, newAttrKey, false);
            });
        }

        /**
         * Processes the passed attribute set for an auto-style according to
         * the option 'keepComplete' passed to the constructor of this
         * collection. In 'complete mode', all missing attributes in the passed
         * attribute set will be filled with the default attribute values.
         *
         * @param {Object} [attributeSet]
         *  An (incomplete) attribute set for an auto-style to be processed. If
         *  omitted, the merged attribute set of the default style will be
         *  returned in 'complete mode'.
         *
         * @returns {Object|Null}
         *  In 'complete mode', the completed attribute set with all attributes
         *  from the style sheet, otherwise the original unmodified object
         *  passed to this method; or null, if the parameter has been omitted.
         */
        function getEffectiveAttributeSet(attributeSet) {

            // return a complete merged attribute set in 'complete mode'
            if (keepComplete) {
                var defAttributeSet = styleSheets.getDefaultStyleAttributeSet();
                return attributeSet ? docModel.extendAttributes(defAttributeSet, attributeSet, { clone: true }) : defAttributeSet;
            }

            // default to null, if parameter is missing
            return attributeSet || null;
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'insertAutoStyle'.
         * Inserts a new auto-style into this collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertAutoStyle' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation, or the same identifier for an auto-style
         *  is used multiple times.
         */
        this.applyInsertAutoStyleOperation = function (context) {

            // check that the passed operation represents the correct style family
            context.ensure(context.getStr('type') === styleFamily, 'invalid attribute family for auto-style');

            // the unique identifier of the auto-style
            var styleId = context.getStr('styleId');
            context.ensure(!autoStyleMap.has(styleId), 'auto-style \'%s\' exists already', styleId);

            // set default auto-style identifier
            if (context.getOptBool('default')) {
                context.ensure(!defaultStyleId, 'multiple default auto styles \'%s\' and \'%s\'', defaultStyleId, styleId);
                defaultStyleId = styleId;
            }

            // check the expected style identifier for strict mode, otherwise update the index variables
            if (strictMode) {
                context.ensure(styleId === standardPrefix + nextStyleIndex, 'invalid name for auto-style \'%s\'', styleId);
                // in strict mode, the first auto-style MUST be the default auto-style
                if (nextStyleIndex === 0) { context.ensure(defaultStyleId, 'missing leading default auto-style'); }
                nextStyleIndex += 1;
            } else {
                // always keep the index above the maximum index of a standard style name, but ignore all other style names
                // (this allows to generate auto-style names without needing to search for an unused name)
                var matches = standardIdRegExp.exec(styleId);
                if (matches) { nextStyleIndex = Math.max(nextStyleIndex, parseInt(matches[1], 10) + 1); }
            }

            // the attribute set for the auto-style
            var attributeSet = getEffectiveAttributeSet(context.getOptObj('attrs'));

            // create and store the new auto-style instance
            createAutoStyle(styleId, attributeSet);

            // notify the listeners
            this.trigger('insert:autostyle', styleId);
        };

        /**
         * Callback handler for the document operation 'changeAutoStyle'.
         * Changes the formatting attributes of an existing auto-style in this
         * collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'changeAutoStyle' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyChangeAutoStyleOperation = function (context) {

            // check that the passed operation represents the correct style family
            context.ensure(context.getStr('type') === styleFamily, 'invalid attribute family for auto-style');

            // remove an existing auto-style (immediately fail for missing auto-style)
            var styleId = context.getStr('styleId');
            var autoStyle = autoStyleMap.remove(styleId);
            context.ensure(autoStyle, 'auto-style \'%s\' does not exist', styleId);

            // the new attribute set for the auto-style (must exist)
            var attributeSet = getEffectiveAttributeSet(context.getObj('attrs'));

            // delete the attribute key settings of the old auto-style
            deleteAttributeKey(styleId);

            // destroy the old auto-style (detach listeners etc.)
            autoStyle.destroy();

            // create a new auto-style instance (do not care to delete/replace old explicit attributes)
            createAutoStyle(styleId, attributeSet);

            // notify the listeners
            this.trigger('change:autostyle', styleId);
        };

        /**
         * Callback handler for the document operation 'deleteAutoStyle'.
         * Deletes an existing auto-style from this collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'deleteAutoStyle' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyDeleteAutoStyleOperation = function (context) {

            // check that the passed operation represents the correct style family
            context.ensure(context.getStr('type') === styleFamily, 'invalid attribute family for auto-style');

            // remove an existing auto-style (immediately fail for missing auto-styles)
            var styleId = context.getStr('styleId');
            var autoStyle = autoStyleMap.remove(styleId);
            context.ensure(autoStyle, 'auto-style \'%s\' does not exist', styleId);

            // in strict naming mode, only the last auto-style can be deleted
            if (strictMode) {
                var lastIndex = nextStyleIndex - 1;
                context.ensure(styleId === standardPrefix + lastIndex, 'cannot delete auto-style \'%s\'', styleId);
                nextStyleIndex -= 1;
            }

            // invoke the user-defined callback handler
            updateHandler(styleId, 'delete', false);

            // delete the attribute key settings of the old auto-style
            deleteAttributeKey(styleId);

            // destroy the old auto-style (detach listeners etc.)
            autoStyle.destroy();

            // notify the listeners
            this.trigger('delete:autostyle', styleId);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the style family name of this auto-style collection.
         *
         * @returns {String}
         *  The style family name of this style sheet collection.
         */
        this.getStyleFamily = function () {
            return styleFamily;
        };

        /**
         * Returns the collection of style sheets auto-style collection is
         * based on.
         *
         * @returns {StyleCollection}
         *  The collection of style sheets auto-style collection is based on.
         */
        this.getStyleCollection = function () {
            return styleSheets;
        };

        /**
         * Returns the identifier of the default auto-style in this collection.
         *
         * @returns {String}
         *  The identifier of the default auto-style; or the empty string, if
         *  this collection does not contain an explicit default auto-style.
         */
        this.getDefaultStyleId = function () {
            return defaultStyleId;
        };

        /**
         * Returns the effective auto-style identifier for the passed
         * identifier. The empty string will be replaced with the identifier of
         * the default auto-style registered for this collection.
         *
         * @param {String} styleId
         *  The identifier of an auto-style.
         *
         * @returns {String}
         *  The identifier of the default auto-style registered for this
         *  collection, if the parameter is an empty string; otherwise the
         *  passed identifier.
         */
        this.getEffectiveStyleId = function (styleId) {
            return styleId || defaultStyleId;
        };

        /**
         * Returns whether the passed auto-style identifier is the default
         * auto-style in this collection, or an empty string (which can always
         * be used to refer to the default auto-style).
         *
         * @param {String} styleId
         *  The identifier of an auto-style.
         *
         * @returns {Boolean}
         *  Whether the passed identifier is the default auto-style of this
         *  collection, or the empty string.
         */
        this.isDefaultStyleId = function (styleId) {
            return this.getEffectiveStyleId(styleId) === defaultStyleId;
        };

        /**
         * Returns whether the passed auto-style identifiers can be considered
         * to be equal.
         *
         * @param {String|Null} styleId1
         *  The first identifier of an auto-style; or the value null for
         *  situations where no auto-style is available.
         *
         * @param {String|Null} styleId2
         *  The second identifier of an auto-style; or the value null for
         *  situations where no auto-style is available.
         *
         * @returns {Boolean}
         *  Whether the passed identifiers are considered equal. If both passed
         *  identifiers refer to the default auto-style, true will be returned
         *  regardless of the actual value of the two parameters (the empty
         *  string vs. the real name of the default auto-style). If both values
         *  are null, true will be returned.
         */
        this.areEqualStyleIds = function (styleId1, styleId2) {
            if (styleId1 === styleId2) { return true; }
            var isString1 = typeof styleId1 === 'string';
            var isString2 = typeof styleId2 === 'string';
            return (isString1 && isString2 && (this.getEffectiveStyleId(styleId1) === this.getEffectiveStyleId(styleId2))) || (!isString1 && !isString2);
        };

        /**
         * Returns whether this collection contains the specified auto-style.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @returns {Boolean}
         *  Whether this collection contains the specified auto-style.
         */
        this.hasAutoStyle = function (styleId) {
            return autoStyleMap.has(this.getEffectiveStyleId(styleId));
        };

        /**
         * Returns the model of the specified auto-style.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @returns {AutoStyleModel|Null}
         *  The auto-style with the specified identifier; or null, if the
         *  passed identifier cannot be resolved to an existing auto-style.
         */
        this.getAutoStyle = function (styleId) {
            return autoStyleMap.get(this.getEffectiveStyleId(styleId), null);
        };

        /**
         * Returns the merged attribute set of the specified auto-style.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @returns {Object}
         *  A complete merged attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, additionally
         *  containing the style sheet reference in the 'styleId' property),
         *  containing values for all supported attributes, collected from the
         *  specified auto-style, and all its parent style sheets, up to the
         *  map of default attributes.
         *  ATTENTION! For efficiency, a reference to the cached attribute set
         *  will be returned. This object MUST NOT be changed!
         */
        this.getMergedAttributeSet = function (styleId) {

            // the specified auto-style (null, if the passed identifier cannot be resolved)
            var autoStyle = this.getAutoStyle(styleId);

            // return the merged attributes of an existing auto-style
            if (autoStyle) { return autoStyle.getMergedAttributeSet(true); }

            // in non-strict mode, allow to use the identifiers of real style sheets,
            // otherwise return the default attributes only
            return strictMode ? styleSheets.getDefaultStyleAttributeSet() : styleSheets.getStyleAttributeSet(styleId);
        };

        /**
         * Returns the explicit attribute set of the specified auto-style.
         *
         * @param {String} styleId
         *  The unique identifier of an auto-style. The empty string can be
         *  used for the default auto-style of this collection.
         *
         * @returns {Object}
         *  The (incomplete) explicit attribute set (a map of attribute maps
         *  with name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property) for
         *  all explicitly set attributes.
         *  ATTENTION! For efficiency, a reference to the cached attribute set
         *  will be returned. This object MUST NOT be changed!
         */
        this.getExplicitAttributeSet = function (styleId) {

            // the specified auto-style (null, if the passed identifier cannot be resolved)
            var autoStyle = this.getAutoStyle(styleId);

            // return the explicit attributes of an existing auto-style
            if (autoStyle) { return autoStyle.getExplicitAttributeSet(true); }

            // in non-strict mode, allow to use the identifiers of real style sheets,
            // otherwise return an empty object
            return strictMode ? {} : styleSheets.getStyleSheetAttributeMap(styleId);
        };

        /**
         * Returns the merged attribute set of the default auto-style of this
         * collection.
         *
         * @returns {Object}
         *  The complete merged attribute set of the default auto-style of this
         *  collection.
         *  ATTENTION! For efficiency, a reference to the cached attribute set
         *  will be returned. This object MUST NOT be changed!
         */
        this.getDefaultAttributeSet = function () {
            return this.getMergedAttributeSet(defaultStyleId);
        };

        /**
         * Returns the identifier of an auto-style that contains the formatting
         * attributes of an existing auto-style, merged with the passed
         * attribute set. If the auto-style does not exist yet, it will be
         * created, and the appropriate remote operations will be inserted into
         * the passed generators.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operation.
         *
         * @param {String|Null} styleId
         *  The unique identifier of an auto-style. The value null, or an empty
         *  string can be used for the default auto-style of this collection.
         *
         * @param {Object} attributeSet
         *  The formatting attributes to be merged over the attributes of the
         *  specified auto-style.
         *
         * @returns {String}
         *  The identifier of the resulting auto-style containing the merged
         *  formatting attributes.
         */
        this.generateAutoStyleOperation = function (generator, styleId, attributeSet) {

            // the specified auto-style (null, if the passed identifier cannot be resolved)
            var autoStyle = this.getAutoStyle(styleId);

            // merge the explicit attributes of the auto-style with the passed attributes
            var resultAttributeSet = autoStyle ? autoStyle.getExplicitAttributeSet() : {};
            _.each(supportedFamilySet, function (flag, family) {

                // check if new attributes exist in the passed attribute set
                var sourceAttributes = attributeSet[family];
                if (!_.isObject(sourceAttributes)) { return; }

                // attribute definitions of the current family
                var definitions = docModel.getAttributeDefinitions(family);
                // target attribute map for the current family
                var resultAttributes = resultAttributeSet[family] || (resultAttributeSet[family] = {});

                // update the attribute map with the new attribute values
                _.each(sourceAttributes, function (value, name) {
                    if (AttributeUtils.isRegisteredAttribute(definitions, name)) {
                        AttributeUtils.updateAttribute(resultAttributes, name, value);
                    }
                });

                // remove empty attribute value maps completely
                if (_.isEmpty(resultAttributes)) {
                    delete resultAttributeSet[family];
                }
            });

            // add the identifier of a new style sheet
            if ('styleId' in attributeSet) {
                resultAttributeSet.styleId = attributeSet.styleId;
            }

            // finalize the attribute set
            resultAttributeSet = getEffectiveAttributeSet(resultAttributeSet);

            // early exit, if this collection contains an auto-style with the exact same attributes
            var attrKey = Utils.stringifyJSON(resultAttributeSet);
            var newStyleId = styleIdsByAttrKey.get(attrKey, null);
            if (newStyleId !== null) { return newStyleId; }

            // create an identifier for the new auto-style (in non-strict mode, do not use names of existing style sheets)
            newStyleId = standardPrefix + nextStyleIndex;
            if (!strictMode) {
                while (styleSheets.containsStyleSheet(newStyleId)) {
                    nextStyleIndex += 1;
                    newStyleId = standardPrefix + nextStyleIndex;
                }
            }

            // create the document operation to insert the auto-style
            var properties = { type: styleFamily, styleId: newStyleId, attrs: resultAttributeSet };
            generator.generateOperation(Operations.INSERT_AUTOSTYLE, properties, { apply: false });

            // prepend the undo operation (in strict naming mode, auto-styles MUST be deleted in reversed order)
            var undoProperties = { type: styleFamily, styleId: newStyleId };
            generator.generateOperation(Operations.DELETE_AUTOSTYLE, undoProperties, { undo: true, prepend: true });

            return newStyleId;
        };

        // initialization -----------------------------------------------------

        // in non-strict mode, listen to changes in the style sheet collection, and invoke the update handler
        if (!strictMode) {
            this.listenTo(styleSheets, 'insert:stylesheet', function (event, styleId) { updateHandler(styleId, 'insert', true); });
            this.listenTo(styleSheets, 'delete:stylesheet', function (event, styleId) { updateHandler(styleId, 'delete', true); });
            this.listenTo(styleSheets, 'change:stylesheet', function (event, styleId) { updateHandler(styleId, 'change', true); });
        }

        // destroy class members on destruction
        this.registerDestructor(function () {
            autoStyleMap.forEach(function (autoStyle) { autoStyle.destroy(); });
            docModel = styleSheets = supportedFamilySet = null;
            autoStyleMap = attrKeysByStyleId = styleIdsByAttrKey = null;
        });

    } // class AutoStyleCollection

    // exports ================================================================

    return ModelObject.extend({ constructor: AutoStyleCollection });

});
