/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/editframework/app/editapplication', [

    'io.ox/core/tk/doc-converter-utils',

    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/driveutils',

    'io.ox/office/baseframework/utils/errorcode',
    'io.ox/office/baseframework/utils/errorcontext',
    'io.ox/office/baseframework/utils/errorcontextdata',
    'io.ox/office/baseframework/utils/clienterror',
    'io.ox/office/baseframework/utils/infostate',
    'io.ox/office/baseframework/utils/errormessages',
    'io.ox/office/baseframework/utils/infomessages',
    'io.ox/office/baseframework/app/baseapplication',
    'io.ox/office/baseframework/app/extensionregistry',

    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/editframework/app/rtconnection',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/editframework/utils/operationutils',

    'gettext!io.ox/office/editframework/main'

], function (ConverterUtils, Utils, IO, Dialogs, DriveUtils, ErrorCode, ErrorContext, ErrorContextData, ClientError, InfoState, ErrorMessages, InfoMessages, BaseApplication, ExtensionRegistry, Config, RTConnection, Labels, OperationUtils, gt) {

    'use strict';

    var // maximal time for synchronization
        MAX_SECONDS_FOR_SYNC = 30,

        // not locked id
        NOT_LOCKED_ID = -1,

        // max number of pending remote updates acquire edit rights is possible
        MAX_NUM_OF_PENDING_UPDATES_FOR_EDITRIGHTS = 1;

    // private global functions ===============================================

    /**
     * Extracts and validates the actions array from the passed data object of
     * a server response.
     *
     * @param {Object} data
     *  The response data object of the server request.
     *
     * @returns {Array|Undefined}
     *  The actions array, if existing, otherwise undefined.
     */
    function getActionsFromData(data) {

        var // whether all actions are arrays of objects
            validActions = _.isArray(data.actions) && data.actions.every(_.isObject);

        if (!validActions && ('actions' in data)) {
            Utils.error('EditApplication.getActionsFromData(): invalid actions', data.actions);
        }

        // check that all array elements are objects
        return validActions ? data.actions : undefined;
    }

    /**
     * Extends the message data with specific actions for the display
     * function.
     *
     * @param {Object} messageData
     *  The message data to be used for the yell method provided by the
     *  view. This object will be extended dependent on the action property
     *  and application.
     */
    function extendMessageDataWithActions(messageData) {
        if (_.isObject(messageData) && _.isString(messageData.action)) {
            // special handling for actions to be shown in the error/warning message
            switch (messageData.action) {
                case 'reload': messageData.action = { itemKey: 'document/reload', icon: Labels.RELOAD_ICON, label: Labels.RELOAD_LABEL }; break;
                case 'acquireedit': messageData.action = { itemKey: 'document/acquireeditOrPending', icon: Labels.EDIT_ICON, label: Labels.EDIT_LABEL }; break;
                default: delete messageData.action; break;
            }
        }
    }

    // class EditApplication ==================================================

    /**
     * The base class for all OX Documents applications allowing to edit a
     * document.
     *
     * Triggers the events supported by the base class BaseApplication, and the
     * following additional events:
     *  - 'docs:state':
     *      When the state of the application has changed. The event handlers
     *      receive the current application state, as also returned by the
     *      method 'EditApplication.getState()'.
     *  - 'docs:state:<STATE_ID>':
     *      When the state of the application has changed to the specified
     *      identifier <STATE_ID>. Will be triggered after the generic
     *      'docs:state' event. All identifiers returned by the method
     *      'EditApplication.getState()' are supported. Example: when the state
     *      'offline' is reached, the application will trigger a 'docs:state'
     *      event, and a 'docs:state:offline' event.
     *  - 'docs:update':
     *      When the application receives an update push message from the
     *      real-time framework. The event handlers receive the complete data
     *      object of the update message.
     *  - 'docs:editrights':
     *      An event related to acquiring the edit rights while the document is
     *      in read-only mode. The event handlers receive the actual state:
     *      - 'acquire': After sending the server request to acquire edit
     *          rights for the document.
     *      - 'accept': The server has accepted the request to acquire edit
     *          right for the document. The document will switch to editing
     *          mode soon.
     *      - 'decline': The server has rejected the request to acquire edit
     *          right for the document. The document will remain in read-only
     *          mode.
     *  - 'docs:editrights:<STATE_ID>':
     *      Will be triggered after a generic 'docs:editrights' event, where
     *      <STATE_ID> is one of the states described above.
     *  - 'docs:users':
     *      After the list of editor clients for this document has changed.
     *      Event handlers receive an array with all active editor clients. See
     *      method EditApplication.getActiveClients() for details.
     *  - 'docs:users:selection':
     *      After the selection of one of the editor clients for this document
     *      has changed. Because this can happen very often, this event was
     *      separated from 'docs:users'.
     *      Event handlers receive an array with all active editor clients. See
     *      method EditApplication.getActiveClients() for details.
     *  - 'fastload:done':
     *      After all fast load strings have been applied to the DOM, but before
     *      operations are handled.
     *
     * @constructor
     *
     * @extends BaseApplication
     *
     * @param {Function} ModelClass
     *  The constructor function of the document model class. See the
     *  BaseApplication base class constructor for details.
     *
     * @param {Function} ViewClass
     *  The constructor function of the view class. See the BaseApplication
     *  base class constructor for details.
     *
     * @param {Function} ControllerClass
     *  The constructor function of the controller class. See the
     *  BaseApplication base class constructor for details.
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method):
     *  @param {String} launchOptions.action
     *      Controls how to connect the application to a document file. The
     *      following actions are supported:
     *      - 'load': Tries to load the file described in the launch option
     *          'launchOptions.file'.
     *      - 'new': Tries to create a new document. Optionally, the new file
     *          will be a clone of an existing file described in the launch
     *          option 'launchOptions.templateFile'.
     *      - 'convert': Tries to convert the file described in the launch
     *          option 'launchOptions.templateFile' to a native file format.
     *  @param {Object} [launchOptions.file]
     *      The descriptor of the file to be loaded. Used by the action 'load'
     *      only.
     *  @param {String} [launchOptions.folderId]
     *      The identifier of the folder a new file will be created in. Used by
     *      the actions 'new' and 'convert'.
     *  @param {Object} [launchOptions.templateFile]
     *      The descriptor of an existing file used as template for the new
     *      file. Used by the actions 'new' and 'convert'.
     *  @param {Boolean} [launchOptions.preserveFileName=false]
     *      If set to true, the name of the template file will be preserved
     *      when converting a file to a native file format. Otherwise, a
     *      default file name will be generated. Used by the action 'convert'
     *      only.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  BaseApplication. Additionally, the following options are supported:
     *  @param {Object} [initOptions.newDocumentParams]
     *      Additional parameters that will be inserted into the AJAX request
     *      sent to create a new empty document on the server.
     *  @param {Boolean} [initOptions.applyActionsDetached=false]
     *      If set to true, the application pane will be detached from the DOM
     *      while the import actions will be applied. This may improve the
     *      import performance of large documents.
     *  @param {Function} [initOptions.preProcessHandler]
     *      A function that will be called before the document operations will
     *      be downloaded and applied, with the progress bar already being
     *      visible. Will be called in the context of this application. Must
     *      return a promise that will be resolved or rejected after the
     *      document has been prepared.
     *  @param {Function} [initOptions.postProcessHandler]
     *      A function that will be called after the document operations have
     *      been downloaded and applied successfully, and that post-processes
     *      the document contents while the progress bar is still visible. Will
     *      be called in the context of this application. Must return a
     *      promise that will be resolved or rejected after the document has
     *      been post-processed.
     *  @param {Function} [initOptions.postProcessHandlerStorage]
     *      Load performance: Similar to postProcessHandler. A function that
     *      will be called after the document has been loaded from the local
     *      storage. Will be called in the context of this application. Must
     *      return a promise that will be resolved or rejected after the
     *      document has been post-processed.
     *  @param {Function} [initOptions.fastEmptyLoadHandler]
     *      A function that will be called when importing an empty document in
     *      a special fast way is supported. Will be called in the context of
     *      this application. Must return a promise that will be resolved or
     *      rejected after the document has been loaded quickly.
     *  @param {Function} [initOptions.previewHandler]
     *      A function that can be used to show an early preview of the
     *      imported document, while the import is still running. Receives the
     *      preview data object sent from the server as first parameter. Must
     *      return a Boolean value stating whether displaying the preview
     *      succeeded. On success, the view will leave the busy state. Will be
     *      called in the context of this application instance.
     *  @param {Function} [initOptions.importFailedHandler]
     *      A function that will be called when importing the document has
     *      failed for any reason (either connection problems, or while
     *      applying the operations). Will be called in the context of this
     *      application.
     *  @param {Function} [initOptions.optimizeOperationsHandler]
     *      A function that can be used to optimize the operations before they
     *      are sent to the server. Receives an operations array as first
     *      parameter, and must return an operations array. Will be called in
     *      the context of this application instance.
     *  @param {Function} [initOptions.mergeCachedOperationsHandler]
     *      A function that can be used to merge the operations and cached before they
     *      are sent to the server. Receives an operations array as first
     *      parameter, cached buffer of operations as second parameter,
     *      and must return an operations array. Will be called in
     *      the context of this application instance.
     *  @param {Function} [initOptions.operationFilter]
     *      A function that can be used to filter the operations that are
     *      executed during importing the document. Will be called in the
     *      context of the model instance.
     *  @param {Function} [initOptions.prepareFlushHandler]
     *      A function that will be called before sending all pending actions,
     *      and flushing the document to the source file. Will be called in the
     *      context of this application. Receives a string identifier as first
     *      parameter specifying the cause of the flush request:
     *      - 'download': The document will be prepared for downloading or
     *          printing.
     *      - 'email': The document will be prepared to be sent as e-mail.
     *      - 'quit': The document is about to be closed.
     *      May return a promise that will be resolved/rejected as soon as the
     *      application has finished all preparations for flushing the
     *      document.
     *  @param {Function} [initOptions.prepareLoseEditRightsHandler]
     *      A function that will be called if the server sends a request to
     *      leave the edit mode. Will be called in the context of this
     *      application. May return a promise that will be resolved or rejected
     *      as soon as the application can safely switch to read-only mode.
     *  @param {Function} [initOptions.prepareRenameHandler]
     *      A function that will be called if the client wants to rename
     *      the documents name. Will be called in the context of this
     *      application. May return a promise that will be resolved or rejected
     *      as soon as the application can safely send the rename request.
     *  @param {Number} [initOptions.sendActionsDelay=0]
     *      The duration in milliseconds used to debounce passing new pending
     *      actions to the realtime connection.
     *  @param {Number} [initOptions.realTimeDelay=700]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     *  @param {Boolean} [initOptions.useStorage=false]
     *      Load performance: Whether the local storage shall be used for
     *      storing the document.
     *  @param {Boolean} [initOptions.supportsOnlineSync=false]
     *      Whether the application supports synchronization to enable seamless
     *      editing of a document after a offline/online transition.
     */
    var EditApplication = BaseApplication.extend({ constructor: function (ModelClass, ViewClass, ControllerClass, launchOptions, initOptions) {

        var // self reference
            self = this,

            // the document model instance
            docModel = null,

            // application view: contains panes, tool bars, etc.
            docView = null,

            // the unique identifier of this application
            clientId = null,

            // the name of the current user loading the document for operations
            clientOperationName = null,

            // connection to the realtime framework
            rtConnection = null,

            // buffer for actions not yet sent to the server
            actionsBuffer = [],

            // new action collecting multiple operations in an undo group
            pendingAction = null,

            // whether actions are currently sent to the server
            sendingActions = false,

            // last time operations where sent, to use for a forced delay when calling flushhandler
            sentActionsTime = 0,

            // preprocessing of the document before applying import operations
            preProcessHandler = Utils.getFunctionOption(initOptions, 'preProcessHandler'),

            // post-processing of the document after successfully applying import operations
            postProcessHandler = Utils.getFunctionOption(initOptions, 'postProcessHandler'),

            // Load performance: A simplified post process handler, that is called after using document from local storage
            postProcessHandlerStorage = Utils.getFunctionOption(initOptions, 'postProcessHandlerStorage'),

            // fast load empty documents: display page after first server request
            fastEmptyLoadHandler = Utils.getFunctionOption(initOptions, 'fastEmptyLoadHandler'),

            // callback for displaying an early preview while import is still running
            previewHandler = Utils.getFunctionOption(initOptions, 'previewHandler'),

            // post-processing of the document when import has failed
            importFailedHandler = Utils.getFunctionOption(initOptions, 'importFailedHandler', $.noop),

            // callback to flush the document before accessing its source
            prepareFlushHandler = Utils.getFunctionOption(initOptions, 'prepareFlushHandler', $.noop),

            // callback to prepare the document before it goes into read-only mode
            prepareLoseEditRightsHandler = Utils.getFunctionOption(initOptions, 'prepareLoseEditRightsHandler', $.noop),

            // callback to prepare the document before the rename request is sent to the backend
            prepareRenameHandler = Utils.getFunctionOption(initOptions, 'prepareRenameHandler', $.noop),

            // callback function that implements merging of actions to reduce data transfer to the server
            optimizeOperationsHandler = Utils.getFunctionOption(initOptions, 'optimizeOperationsHandler'),

            // callback function that implements merging of current and cached operations, that represent update of complex date fields
            mergeCachedOperationsHandler = Utils.getFunctionOption(initOptions, 'mergeCachedOperationsHandler'),

            // Load Performance: Only selected operations need to be executed, if document can be loaded from local storage
            operationFilter = Utils.getFunctionOption(initOptions, 'operationFilter'),

            // application specific delay in milliseconds, until operations are sent to the server
            sendActionsDelay = Utils.getNumberOption(initOptions, 'sendActionsDelay', 0),

            // whether the application supports multiple selection
            isMultiSelectionApp = Utils.getBooleanOption(initOptions, 'isMultiSelectionApp', true),

            // current application state (see method EditApplication.getState())
            appState = null,

            // user-defined application state that temporarily overrides the application state
            userState = null,

            // whether the application is currently connected to the server
            connected = false,

            // cached name of the user currently editing the document
            oldEditUser = null,

            // if any operations where generated by the own document model and sent to the server
            locallyModified = false,

            // if any operations where generated by a remote application
            remotelyModified = false,

            // current server request to rename the edited document
            renamePromise = null,

            // whether the file has been renamed locally
            locallyRenamed = false,

            // whether the file has been move to other folder or to trash
            movedAway = false,

            // whether application is locked globally (no file permissions)
            locked = false,

            // document file is locked by another user (lock set via OX Drive)
            //documentFileIsLocked = false,

            // document file is locked by the user
            documentFileLockedByUser = null,

            // whether application is locked globally (internal unrecoverable error)
            internalError = false,

            // whether the user had edit rights before the offline handler was called
            onlineEditRights = false,

            // whether the connection was at least once offline
            offlineHandlerTriggered = false,

            // the update attributes used by server responses
            updateDataAttributes = ['editUserId', 'editUser', 'hasErrors', 'writeProtected', 'activeUsers', 'locked', 'lockedByUser', 'lockedByUserId', 'failSafeSaveDone', 'wantsToEditUser', 'wantsToEditUserId'],

            // old edit user id
            oldEditClientId = null,

            // whether we are in a prepare losing edit rights state
            inPrepareLosingEditRights = false,

            // the user who gets the edit rights
            wantsToEditUser = null,

            // the user who last time wanted to get the edit rights
            oldWantsToEditUser = null,

            // whether the documents of the application can be saved in the local storage
            localStorageApp = Utils.getBooleanOption(initOptions, 'localStorageApp', false),

            // the required version of the local storage
            requiredStorageVersion = Utils.getIntegerOption(initOptions, 'requiredStorageVersion', 0),

            // the supported version of the local storage
            supportedStorageVersion = Utils.getIntegerOption(initOptions, 'supportedStorageVersion', 0),

            // reading configuration whether using local browser storage is configured
            // Additionally checking, if the 'Stay signed in' check box is checked or not
            saveFileInLocalStorage = Config.USE_LOCAL_STORAGE && (ox.secretCookie === true),

            // whether the local storage can be used to save the file
            isLocalStorageSupported = Utils.isLocalStorageSupported(),

            // whether the document was loaded from the local storage
            loadedFromStorage = false,

            // maps client identifiers to unique and constant integral indexes
            clientIndexes = {},

            // next free unique client index (0 is used for this client)
            nextClientIndex = 1,

            // all clients working on this document (including the local client, including inactive clients)
            editClients = [],

            // all active clients working on this document (less than 60 seconds inactivity)
            activeClients = [],

            // pending remote updates queue
            pendingRemoteUpdates = [],

            // object for logging several time stamps
            performanceLogger = {},

            // start time of preview phase (import still running but with enabled view)
            previewStartTime = null,

            // time stamp when import finishes
            importFinishedTime = 0,

            // server sent us a fast empty doc response with document data
            fastEmpty = false,

            // whether the generic fail handler was executed
            failureAlreadyEvaluated = false,

            // Synchronization support
            // supports online sync or not
            supportsOnlineSync = Utils.getBooleanOption(initOptions, 'supportsOnlineSync', false),

            // must sync after online again, with mode ('editSync', 'viewSync')
            mustSyncNowMode = null,

            // received backend sync states
            syncStateReceived = null,

            // already shown warnings
            alreadyShownWarning = {},

            // unique list of document authors (e.g. change tracking, comments, collaboration...)
            documentAuthorsList = [],

            // allow or disallow registering of operations
            distribute = true,

            // block registration of operations in 'operations:success'
            operationsBlock = false,

            // cache the operations during prohibition of registering operations (date fields on document load)
            cacheBuffer = [],

            // rename and reload promise
            renameAndReloadPromise = null,

            // document restore id
            docRestoreId = null,

            // document restore started
            docRestore = false,

            // enabled/disabled document restore function
            restoreDocEnabed = Config.ENABLED_RESTORE_DOCUMENT,

            // specifies if reload should en-/disabled
            reloadEnabled = true,

            // rescue document are loaded read-only
            rescueDocReadOnly = Config.RESCUE_DOCUMENT_READONLY;

        // base constructor ---------------------------------------------------

        BaseApplication.call(this, ModelClass, ViewClass, ControllerClass, importHandler, launchOptions, _.extend({}, initOptions, {
            initFileHandler: initFileHandler,
            flushHandler: flushDocumentHandler,
            messageDataHandler: extendMessageDataWithActions
        }));

        // private methods ----------------------------------------------------

        /**
         * Extracts the update data from the passed data object of a
         * server response.
         *
         * @param {Object} data
         *  The response data object of a server request.
         *
         * @returns {Object}
         *  An object which contains only the update attributes, if
         *  existing, otherwise undefined.
         */
        function getUpdateAttributesFromData(data) {

            // the return object
            var updateData = {};
            // the found attributes
            var attrs = _.intersection(_.keys(data), updateDataAttributes);

            // copy only the known update attributes
            _.each(attrs, function (key) {
                updateData[key] = data[key];
            });

            return updateData;
        }

        /**
         * Triggers the 'docs:state' event, and updates the 'data-app-state'
         * attribute at the root DOM node of this application.
         *
         * @param {String} [oldState]
         *  Optionally the old state can be specified, so that listeners will
         *  be informed about the old state, too.
         */
        function triggerChangeState(oldState) {
            var state = self.getState();
            self.setRootAttribute('data-app-state', state);
            self.trigger('docs:state', state, oldState);
            self.trigger('docs:state:' + state);
        }

        /**
         * Recalculates the application state. Triggers a 'docs:state' event,
         * if the application state has been changed.
         */
        function updateState() {

            // calculate the new application state
            var newAppState = internalError ? 'error' :
                    _.isNumber(previewStartTime) ? 'preview' :
                    !connected ? 'offline' :
                    (locked || !docModel.getEditMode()) ? 'readonly' :
                    self.hasUnsavedChanges() ? 'sending' :
                    locallyModified ? 'ready' :
                    'initial';

            // the old state
            var oldState = null;

            // trigger event if state has been changed
            if (appState !== newAppState) {
                oldState = appState;
                appState = newAppState;
                triggerChangeState(oldState);
            }
        }

        /**
         * Updates the enable/disable function states dependent on the provided
         * message data object.
         *
         * @param {Object} messageData
         *  An object containing several properties which are used to display a
         *  message popup. Dependend on these properties the function determines
         *  the state of certain other functions.
         */
        function updateFunctionStates(messageData) {
            reloadEnabled = _.isObject(messageData) && _.isObject(messageData.action) && _.isString(messageData.action.itemKey) && (messageData.action.itemKey === 'document/reload');
        }

        /**
         *
         */
        function getInfoStateOptions() {
            var // the options object filled with context data
                options = {
                    wantsToEditUser: wantsToEditUser,
                    oldWantsToEditUser: oldWantsToEditUser,
                    documentFileLockedByUser: documentFileLockedByUser,
                    fullFileName: self.getFullFileName(),
                    oldEditUser: oldEditUser
                };

            return options;
        }

        /**
         * Shows a read-only alert banner according to the current state of the
         * application. If the document is locked due to missing editing
         * permissions, shows an appropriate alert, otherwise shows the name of
         * the current editing user.
         *
         * @param {String} [message]
         *  A custom message to be shown in the alert banner. If omitted, a
         *  default message will be generated according to the error/info state.
         */
        function showReadOnlyAlert(message, duration) {

            var // alert type
                type = 'info',
                // the alert title
                headline = gt('Read-Only Mode'),
                // state to check
                checkState = null,
                // message data
                messageData = null,
                // options
                options = null;

            if (_.isString(message)) {
                messageData = { type: type, message: message, headline: headline };
                if (!isNaN(duration)) { messageData.duration = duration; }
                messageData = _.extend({}, messageData, { action: 'acquireedit' });
            } else {
                options = getInfoStateOptions();
                checkState = self.getErrorState();
                if (_.isObject(checkState) && (checkState.isError() || checkState.isWarning())) {
                    messageData = ErrorMessages.getMessageData(checkState, options);
                } else {
                    checkState = self.getInfoState();
                    if (_.isString(checkState)) {
                        messageData = InfoMessages.getMessageData(checkState, options);
                    }
                }
            }

            self.extendMessageData(messageData);
            // show the alert banner
            docView.yell(messageData);
        }

        /**
         * In debug mode, the global configuration value for 'useLocalStorage' can be overwritten
         * with the local configuration value of 'debugUseLocalStorage'.
         */
        var checkDebugUseLocalStorage = Config.DEBUG ? function () {
            saveFileInLocalStorage = self.getUserSettingsValue('debugUseLocalStorage', saveFileInLocalStorage);
        } : $.noop;

        /**
         * Adding two specific key-value pairs to the performance logger object.
         * For a given key name the absolute time and the duration compared
         * to the launch time is saved.
         *
         * @param {String} key
         *  The key for the performance logger object.
         *
         * @param {Number} [time]
         *  The logged time for the performance logger object.
         */
        var addTimePerformance = Config.LOG_PERFORMANCE_DATA ? function (key, time) {
            var localTime = time || _.now(),
                absoluteKey = key + 'Absolute',
                durationKey = key + 'Duration';
            addToPerformanceLogger(absoluteKey, localTime);
            addToPerformanceLogger(durationKey, localTime - self.getLaunchStartTime());
        } : $.noop;

        /**
         * Handles a lost network connection. Switches the document model to
         * read-only mode.
         */
        function connectionOfflineHandler() {
            offlineHandlerTriggered = true;
            if (connected && !internalError && !self.isInQuit()) {
                onlineEditRights = docModel.getEditMode();
                connected = false;

                // Sets the client lock mode to ensure that the user is not able
                // to do anything via the UI while we try to synchronize.
                if (!mustSyncNowMode) {
                    locked = true;
                    docModel.setEditMode(false);
                    updateState();
                    self.setInternalError(ClientError.WARNING_READONLY_DUE_TO_OFFLINE, ErrorContext.OFFLINE);
                } else {
                    // a off-line notification while synchronizing means
                    // that we give up and set internal error to true
                    self.setInternalError(ClientError.ERROR_OFFLINE_WHILE_SYNCHRONIZING, ErrorContext.OFFLINE);
                }
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Connection offline handler called by Realtime!');
            }
        }

        /**
         * Handles if we get a network connection again. Sets the internal
         * connected flag to true if our internal error state is not set.
         */
        function connectionOnlineHandler() {
            // RT-Library sometimes calls online although the state has not changed,
            // therefore we also check the offlineHandlerTriggered flag. See
            // #27831 for more information.
            if (offlineHandlerTriggered && !connected && !internalError && !self.isInQuit()) {
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Connection online handler called by Realtime!');

                connected = true;
                // check for synchronization
                if (supportsOnlineSync) {
                    RTConnection.log('Client supports synchronization - starting to sync with server');

                    // if application supports synchronization we have
                    // to check what to do next - the editor needs to
                    // synchronize while the other clients just reset
                    // locked.
                    if (onlineEditRights) {
                        // edit synchronization must be done
                        self.setInternalError(ClientError.WARNING_SYNC_AFTER_OFFLINE, ErrorContext.SYNCHRONIZATION);
                        RTConnection.log('Client is editor - synchronizing server with current client state');
                        mustSyncNowMode = 'editSync';
                        syncStateReceived = null;
                    } else {
                        // view synchronization must be done
                        self.setInternalError(ClientError.WARNING_SYNC_FOR_VIEWER, ErrorContext.SYNCHRONIZATION);
                        RTConnection.log('Client is viewer - waiting for server messages to unlock view');
                        mustSyncNowMode = 'viewSync';
                    }

                    // setup timeout to prevent waiting forever
                    self.executeDelayed(function () {
                        checkSynchronizationState(null, null, null, { cause: 'timeout' });
                    }, MAX_SECONDS_FOR_SYNC * 1000, 'EditApplication: checkSynchronizationState');

                    self.repeatDelayed(function () {
                        // If necessary, send periodic sync requests - wait for
                        // pending requests sent first
                        if (mustSyncNowMode && !sendingActions) {
                            rtConnection.sync();
                        }
                    }, { delay: 0, repeatDelay: ((MAX_SECONDS_FOR_SYNC * 1000) / 12), cycles: 10 }, 'EditApp: send sync requests');
                }

                updateState();
                showReadOnlyAlert();
            }
        }

        /**
         * Handles if a connection has been timed out and was reset. The
         * document is set to read-only and the user has to reopen it to
         * edit it again.
         */
        function connectionResetHandler() {
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Reset for connection called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                self.setInternalError(ClientError.ERROR_CONNECTION_RESET_RECEIVED, ErrorContext.CONNECTION);
            }
            // check reset message while we are synchronizing
            if (mustSyncNowMode) {
                checkSynchronizationState(null, null, null, { cause: 'reset' });
            }
        }

        /**
         * Handles communication with server failed due to a time-out
         * failure. By default this handler triggers the restore
         * documents function and shows an error message.
         */
        function connectionTimeoutHandler() {
            var // error code
                errorCode = null;

            // more logging for RT to better see who has the edit rights
            RTConnection.log('Connection time out called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                errorCode = new ErrorCode(ClientError.ERROR_CONNECTION_TIMEOUT);
                errorCode.setErrorContext(ErrorContext.CONNECTION);

                // try to restore the document data or set error
                self.restoreDocument(errorCode);
            }
        }

        /**
         * Handles problems with the connection instance on the server side.
         * The handler is called if the connection instance was closed  due
         * to the fact that the server node has been shutdown. The document
         * must be reopened to be accessible again. Can also happen if the
         * server-side runs into a timeout and removes our instance from the
         * document connection.
         */
        function connectionNotMemberHandler() {
            // more logging for RT
            RTConnection.log('NotMember for connection called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                self.setInternalError(ClientError.ERROR_CONNECTION_NOT_MEMBER, ErrorContext.CONNECTION);
            }

            // check not member message while we are synchronizing
            if (mustSyncNowMode) {
                checkSynchronizationState(null, null, null, { cause: 'notMember' });
            }
        }

        /**
         * Handles problems with the processing of stanzas. By default this
         * handler triggers the restore documents function and shows an
         * error message.
         */
        function connectionProcessingStanzaFailed() {
            var // error code
                errorCode = null;

            // more logging for RT to better see who has the edit rights
            RTConnection.log('Processing Stanza failed called by Realtime!');

            if (!self.isLocked() && !self.isInQuit()) {
                errorCode = new ErrorCode(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE);
                errorCode.setErrorContext(ErrorContext.CONNECTION);

                // try to restore the document data or set error
                self.restoreDocument(errorCode);
            }
        }

        /**
         * Handles the situation that we have sent actions to the server but
         * we are not the editor registered at the server. Therefore the server
         * doesn't accept our actions and sends us a hangup notification for
         * reloading. Our document contains changes that are not part of the
         * shared document on the server!
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function connectionHangupHandler(event, data) {

            var // the extracted error code from the server response
                errorCode = null;

            if (!self.isInQuit()) {
                errorCode = new ErrorCode(data);
                if (!errorCode.isError()) {
                    errorCode = new ErrorCode(ClientError.ERROR_CONNECTION_HANGUP);
                }

                // output error message to debug console
                Utils.warn('EditApplication.connectionHangupHandler: Error code "' + errorCode.getCodeAsConstant() + '"');

                errorCode.setErrorContext(ErrorContext.SYNCHRONIZATION);
                // try to restore the document data or set error
                self.restoreDocument(errorCode, data);
            }

            // more logging for RT to better see who has the edit rights
            RTConnection.log('Server sent a hangup message due to inconsistency between client/server, client osn = ' + docModel.getOperationStateNumber());
        }

        /**
         * Handles the situation that the server notifies us that another
         * client wants to receive the edit rights. Therefore we are forced
         * to switch to read-only mode and flush our buffer. If that happened
         * we have to notify the server that the edit rights switch can be
         * accomplished.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function handlePrepareLosingEditRights(event, data) {

            function checkForPendingOperations() {

                if (!self.hasUnsavedChanges()) {

                    // just to be sure that nothing is left
                    sendActions().then(function () {
                        // After sending the actions we can acknowledge that
                        // we are ready to lose our edit rights.
                        rtConnection.canLoseEditRights();
                        oldEditUser = wantsToEditUser || oldEditUser;
                        oldWantsToEditUser = wantsToEditUser;
                        wantsToEditUser = '';
                        inPrepareLosingEditRights = false;
                        self.getController().update();
                        // more logging for RT to better see who has the edit rights
                        RTConnection.log('sent canLoseEditRights');
                    });
                } else {
                    // wait again some time and try again
                    self.executeDelayed(checkForPendingOperations, 2000, 'EditApplication: checkForPendingOperations 1');
                }
            }

            /**
             * Sends periodically alive messages to the backend to extend
             * the time out for switching edit rights.
             *
             * @return {Boolean|Object} specifies if the repetition calling
             *  this function should be stopped or not. Utils.BREAK will stop
             *  the repetition.
             */
            function sendAliveForExtendedSwichtingTime() {
                // send alive messages as long as we are in preparations and
                // the backend haven't forced the switch
                if ((inPrepareLosingEditRights) && (clientId === oldEditClientId)) {
                    rtConnection.alive();
                    // continue to send alive messages
                    return true;
                } else {
                    // we are done - alive messages shouldn't be sent
                    return Utils.BREAK;
                }
            }

            if (self.isInQuit()) { return; }

            inPrepareLosingEditRights = true;
            self.repeatDelayed(sendAliveForExtendedSwichtingTime, { repeatDelay: 2000, delay: 0 });

            // Switch edit mode to false, this must happen before any asynchronous
            // calculation is done!
            docModel.setEditMode(false);

            // Set the information message code correctly to enable users
            // to see the message as soon as possible and use it if the
            // users tries to edit during the switching process (46273)
            self.setCurrentInfoState(InfoState.INFO_PREPARE_LOSING_EDIT_RIGHTS);

            updateState();
            self.getController().update();
            wantsToEditUser = getUserNameByRTId(Utils.getStringOption(data, 'wantsToEditUserId', ''));

            // invoke callback for document specific preparations
            $.when(prepareLoseEditRightsHandler.call(self))
            .always(function () {

                RTConnection.log('handle PrepareLosingEditRights');
                self.executeDelayed(checkForPendingOperations, 1000, 'EditApplication: checkForPendingOperations 2');
            });
        }

        /**
         * Handles the request from the server to send a alive message
         * and that we need more time for the "edit rights"-switch process.
         *
         * @param  {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param  {Object} data
         *  The data sent by the backend. Contains the latest status
         *  information.
         */
        function aliveHandler(event, data) {
            var // osn
                osn = 0;

            if (self.isInQuit()) { return; }

            RTConnection.log('handleAlive');
            if (inPrepareLosingEditRights) {
                osn = Utils.getOptionalInteger('serverOSN', data, 0);
                rtConnection.alive();
                RTConnection.log('handled alive request from backend, osn = ' + self.getModel().getOperationStateNumber() + ', server osn = ' + osn);
            }
        }

        function getUserNameByRTId(id) {
            var client = _.find(editClients, function (editClient) { return id === editClient.clientId; }) || {};
            return client.userName || '';
        }

        /**
         * Triggers an event after the edit rights for the document have been
         * tried to acquire.
         */
        function triggerEditRightsEvent(editState) {
            self.trigger('docs:editrights', editState).trigger('docs:editrights:' + editState);
        }

        /**
         * Handles the situation that this client will receive the
         * edit rights as soon as the transfer from the editor
         * has been completed.
         */
        function handleAcceptedAcquireEditRights() {
            if (!self.isInQuit()) {
                self.setCurrentInfoState(InfoState.INFO_EDITRIGHTS_ARE_TRANSFERED);
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('handle AccepteAcquireEditRights');
            triggerEditRightsEvent('accept');
        }

        /**
         * Handles the situation that this client tries to acquire the
         * edit rights but someone else already did and the process is
         * still in progress.
         */
        function handleDeclinedAcquireEditRights(event, data) {
            var // error from the server answer
                error = null,
                // info state for the ui message
                info = null;

            if (!self.isInQuit()) {
                error = new ErrorCode(data);

                switch (error.getCodeAsConstant()) {
                    case 'SWITCHEDITRIGHTS_NOT_POSSIBLE_SAVE_IN_PROGRESS_WARNING': info = InfoState.INFO_EDITRIGHTS_SAVE_IN_PROGRESS; break;
                    case 'SWITCHEDITRIGHTS_IN_PROGRESS_WARNING': info = InfoState.INFO_ALREADY_TRANSFERING_EDIT_RIGHTS; break;
                    default: info = InfoState.INFO_ALREADY_TRANSFERING_EDIT_RIGHTS; break;
                }
                self.setCurrentInfoState(info);
            }

            // more logging for RT to better see who has the edit rights
            RTConnection.log('handle DeclinedAcquireEditRights');
            triggerEditRightsEvent('decline');
        }

        /**
         * Handles the notification from the server-side where the client
         * should show informtion messages to the user. The purposes of this
         * notification is NOT intended to process errors.
         */
        function handleInfoMessage(event, data) {
            var // error from the erver
                error = null,
                // info state for the ui message
                info = null;

            if (!self.isInQuit()) {
                error = new ErrorCode(data);

                // treat warnings as info messages in this special context
                if (error.isWarning()) {

                    switch (error.getCodeAsConstant()) {
                        case 'SWITCHEDITRIGHTS_NEEDS_MORE_TIME_WARNING': info = InfoState.INFO_EDITRIGHTS_PLEASE_WAIT; break;
                        default: info = InfoState.INFO_EDITRIGHTS_PLEASE_WAIT; break;
                    }
                    self.setCurrentInfoState(info);
                }
            }
        }

        /**
         * Handles the notification that an asynchronous rename has been
         * processed. It must be checked that the process was successful or
         * not. In case of a success the code has to trigger a reload of the
         * renamed document. An asynchronous rename is only triggered, if the
         * storage system uses intransparent file-ids. It that case the file-id
         * changes and the client needs to open the document using the new
         * file-id.
         */
        function handleRenameAndReload(event, data) {
            var // error from the erver
                error = null,
                // the new file id of the renamed file
                newFileId = null,
                // the new file name of the renamed file
                newFileName = null,
                // a new file descriptor
                newFile = {};

            if (!self.isInQuit()) {
                error = new ErrorCode(data);

                if (error.isError()) {
                    if (renameAndReloadPromise) {
                        renameAndReloadPromise.reject(data);
                    }
                } else {
                    newFileId = Utils.getStringOption(data, 'fileId', null);
                    newFileName = Utils.getStringOption(data, 'fileName', null);

                    if (newFileId && newFileName) {
                        if (renameAndReloadPromise) {
                            renameAndReloadPromise.resolve(data);
                        }

                        newFile.id = newFileId;
                        newFile.filename = newFileName;
                        //no file ending
                        newFile.title =  Utils.getFileBaseName(newFileName);
                        self.updateFileDescriptor(newFile);

                        // Adapt launch options to prevent removing the file as it
                        // could be empty and new or a template file.
                        // See #32719 "Save as with an empty document: => Load Error"
                        launchOptions.action = 'load';
                        delete launchOptions.template;

                        // close the current document and restart with new one
                        self.reloadDocument();
                    } else if (renameAndReloadPromise) {
                        renameAndReloadPromise.reject(data);
                    }
                }
            }
        }

        /**
         * Handles the situation that this client detects that no update has
         * been received since a certain amount of time were we sent an
         * "applyactions" message.
         * This can happen if the server side code is not able to reply
         * on our requests. Currently we just show a warning to the
         * user.
         */
        function handleNoUpdateFromServer() {
            if (docRestore) {
                // Prevent to show info/warnings or errors that are related
                // to the real-time connection, if we are in the
                // middle of a restore process. A restore request is sent
                // as a "normal" AJAX-request and it's possible that we
                // are not affected.
                return;
            }

            if (!self.isInQuit()) {
                self.setInternalError(ClientError.WARNING_CONNECTION_INSTABLE, ErrorContext.CONNECTION);
            }

            // more logging for RT to better see that the server doesn't send updates
            RTConnection.log('handle NoUpdateFromServer');
        }

        /**
         * Handles the situation if the server detects problems while storing the latest
         * changes to the document. Normally this means that a _ox file was created. It's
         * also possible that just a warning has to be shown (in case the old document
         * content cannot be preserved in a backup file). This function provides feedback
         * to the user to notify her that there is a problem.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function handleFlushErrorInfo(event, data) {

            var // extract error information
                errorCode = new ErrorCode(data);

            if (!self.isInQuit() && errorCode.isError()) {
                self.setInternalError(errorCode, ErrorContext.FLUSH);
            }

            // more logging for RT to better see that the server doesn't send updates
            RTConnection.log('handle flushErrorInfo from server');
        }

        /**
         * Updates the edit state of the application according to the passed data of
         * a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyEditState(data) {

            var // unique identifier of current user with edit rights
                editClientId = Utils.getStringOption(data, 'editUserId', ''),
                // display name of current user with edit rights
                editUser = getUserNameByRTId(editClientId),
                // current state of document edit mode
                oldEditMode = docModel.getEditMode(),
                // new state of document edit mode
                newEditMode = oldEditMode,
                // the file descriptor of this application
                file = self.getFileDescriptor(),
                // client OSN
                clientOSN = docModel.getOperationStateNumber(),
                // server OSN
                serverOSN = Utils.getIntegerOption(data, 'serverOSN', -1),
                // possible error/warning code sent via update
                errorCode  = new ErrorCode(data),
                // locked by user id
                lockedByUserId = Utils.getIntegerOption(data, 'lockedByUserId', NOT_LOCKED_ID),
                // write protected state (will only be sent once - must be ignored if property is not set)
                writeProtected = Utils.getBooleanOption(data, 'writeProtected', null),
                // rescue document state
                rescueDoc = Utils.getBooleanOption(data, 'rescueDoc', null),
                // sync information received - must be evaluated by an editor in off-/online transition
                syncData = Utils.getObjectOption(data, 'sync', null);

            // no state information or internal error may have been set while request was running
            if (internalError || (_.intersection(_.keys(data), updateDataAttributes).length === 0)) {
                return;
            }

            if (editClientId !== oldEditClientId) {
                RTConnection.log('Edit rights changed, new editing user id: ' + editClientId);
                // Bug 32043: Set edit mode only if the edit user has changed, otherwise it's
                // possible that we switch back to read/write mode although we switched
                // programmatically to read-only mode due to 'preparelosingeditrights'
                newEditMode = (clientId === editClientId);
                self.setCurrentInfoState(InfoState.INFO_USER_IS_CURRENTLY_EDIT_DOC, { showMessage: false });
            }

            // check if we have to update the files view (fail safe save done on server)
            if (Utils.getBooleanOption(data, 'failSafeSaveDone', false)) {
                DriveUtils.propagateChangeFile(file);
            }

            // internal server error: stop editing completely, show permanent error banner
            if (Utils.getBooleanOption(data, 'hasErrors', false)) {
                self.setInternalError(ClientError.ERROR_GENERAL_SYNCHRONIZATION_ERROR, ErrorContext.SYNCHRONIZATION);
                RTConnection.error('synchronization to the server has been lost.');
                return;
            }

            if (!locked) {
                if (rescueDocReadOnly && _.isBoolean(rescueDoc) && rescueDoc) {
                    // special document format detected - rescue documents should always be read-only!
                    locked = true;
                    docModel.setEditMode(false);
                    self.setInternalError(ClientError.WARNING_RESCUE_DOC_CANNOT_BE_CHANGED, ErrorContext.READONLY);
                } else if (_.isBoolean(writeProtected) && writeProtected) {
                    // ATTENTION: The write protected property is only sent once as it's
                    // client specific. Don't evaluate it, if the property is missing -
                    // especially don't use a boolean default value!!
                    locked = true;
                    docModel.setEditMode(false);
                    self.setInternalError(ClientError.WARNING_NO_PERMISSION_FOR_DOC, ErrorContext.READONLY);
                } else if ((lockedByUserId !== NOT_LOCKED_ID) && (ox.user_id !== lockedByUserId)) {
                    locked = true;
                    docModel.setEditMode(false);
                    self.setInternalError(ClientError.WARNING_DOC_IS_LOCKED, ErrorContext.READONLY);
                    documentFileLockedByUser = Utils.getStringOption(data, 'lockedByUser', '');
                }
            }

            // update edit mode (not if document is locked due to missing write permissions)
            if (!locked) {
                // set edit mode at document model
                RTConnection.log('Edit mode: ' + newEditMode);
                docModel.setEditMode(newEditMode);

                // Remember old edit user id to check for edit rights instead of using editMode
                // which can be misleading if we in the process of switching the edit rights.
                // There is a time where the client still has edit rights but is in read-only
                // mode to prevent the user from changing the document further. All changes which
                // have been done should be sent to the server in time!
                if (connected && editClientId && (editClientId !== oldEditClientId)) {
                    oldEditClientId = editClientId;
                    oldEditUser = editUser;
                }

                // Show success banner, if edit rights gained after read-only
                // mode, or after connection re-established, but not directly
                // after loading the document (oldEditMode === null).
                if (newEditMode && (oldEditMode === false)) {
                    self.setCurrentInfoState(InfoState.INFO_EDITRIGHTS_RECEIVED);
                }

                // Make consistency check if we just have lost the edit rights
                // and our OSN is higher than the server OSN. The old editor
                // must never be in front of operation, as that would mean that
                // operations have not been sent to the server side. E.g. a
                // timeout scenario occurred.
                if (connected && !newEditMode && oldEditMode && (serverOSN !== -1) && (clientOSN > serverOSN)) {
                    self.setInternalError(ClientError.ERROR_GENERAL_SYNCHRONIZATION_ERROR, ErrorContext.SYNCHRONIZATION);
                    RTConnection.error('synchronization between client and server is broken. Switching editor into read-only mode. New edit mode: ' + newEditMode + 'Old edit mode: ' + oldEditMode + ', Server OSN: ' + serverOSN + ', Client OSN: ' + clientOSN);
                }

                // Show warning if edit rights have been lost, or are still missing
                // after connection has been re-established, or if the edit user has changed.
                if (!newEditMode && (!connected || oldEditMode || ((editUser !== oldEditUser) && editUser.length))) {
                    oldEditUser = editUser;
                    self.setCurrentInfoState(InfoState.INFO_USER_IS_CURRENTLY_EDIT_DOC);
                }

                // Show a possible error/warning sent via update
                if (errorCode.isError() || errorCode.isWarning()) {
                    self.setInternalError(errorCode, ErrorContext.GENERAL);
                }
            }

            // check for a possible synchronization
            checkSynchronizationState(editClientId, clientOSN, serverOSN, null, syncData);

            // application is now in connected state
            connected = true;
            updateState();
        }

        /**
         * Manages the current synchronization state after a off-/online
         * transition.
         *
         * @param {String} editClientId
         *  The current editClientId received from the latest update message.
         *
         * @param {Number} clientOSN
         *  The client osn, can be null.
         *
         * @param {Number} serverOSN
         *  The server osn, can be null.
         *
         * @param {Object} failedData
         *  An optional object describing what went wrong, if other handlers
         *  detected a bad synchronization state. Can be null.
         *  @param {String} [failedData.cause]
         *      A string describing the root cause why the synchronization
         *      has failed. Must be set, if failedData is provided.
         *
         * @param {Object} syncData
         *  An optional synchronization data object, which must be evaluated
         *  by an editor which wants to switch on edit rights. Contains sync
         *  information from the backend that made document consistency checks.
         */
        function checkSynchronizationState(editClientId, clientOSN, serverOSN, failedData, syncData) {
            var // error code to be set
                errorCode = null,
                // possible sync error code
                syncError = null;

            // Start synchronization after we received an update message which
            // is an indication that we have a clean connection. We also must be
            // sure that all pending operations stored in the RT queue were sent
            // to the backend.
            if (mustSyncNowMode) {

                RTConnection.log('Checking synchronization state with server');

                // Handle a detected failed synchronization state here - other
                // bad dynamic states are detected within this function.
                if (failedData) {
                    switch (failedData.cause) {
                        case 'timeout':
                            // in case of time out show error message and give up!
                            RTConnection.log('Synchronization not possible - maybe connection instance is gone');
                            errorCode = new ErrorCode(ClientError.ERROR_SYNCHRONIZATION_TIMEOUT);
                            errorCode.setErrorContext(ErrorContext.SYNCHRONIZATION);
                            mustSyncNowMode = null;
                            break;

                        case 'notMember':
                        case 'reset':
                        case 'error:disposed':
                            // real-time is not able to recover easily - for now give up!
                            RTConnection.log('Synchronization not possible - connection instance is gone or rt-state broken');
                            self.setCurrentInfoState();
                            errorCode = new ErrorCode(ClientError.ERROR_SYNCHRONIZATION_NOT_POSSIBLE);
                            errorCode.setErrorContext(ErrorContext.SYNCHRONIZATION);
                            mustSyncNowMode = null;
                            break;
                    }

                    self.restoreDocument(errorCode);
                }

                switch (mustSyncNowMode) {
                    case 'editSync':
                        // set sync state, if we received a state from the
                        // backend
                        if (_.isObject(syncData)) {
                            syncStateReceived = syncData;
                        }

                        // Ignore updates which are not up to date with the server osn
                        // until we reached the timeout.
                        if ((editClientId === oldEditClientId) && (clientOSN !== serverOSN)) {
                            RTConnection.log('Synchronization in progess server osn != client osn');
                            return;
                        }

                        if (editClientId === oldEditClientId) {
                            if (clientOSN === serverOSN) {
                                if (_.isObject(syncStateReceived)) {
                                    syncError = new ErrorCode(syncStateReceived);

                                    if (!syncError.isError()) {
                                        RTConnection.log('Synchronization successfully completed');
                                        locked = false;
                                        docModel.setEditMode(true);
                                        self.setCurrentInfoState(InfoState.INFO_SYNCHRONIZATION_SUCCESSFUL);
                                    } else {
                                        RTConnection.log('Synchronization detected bad document consistency state');
                                        syncError.setErrorContext(ErrorContext.SYNCHRONIZATION);
                                        self.restoreDocument(syncError);
                                    }
                                } else {
                                    // wait until we receive the sync state from the
                                    // backend or reached the timeout
                                    return;
                                }
                            } else {
                                // We are not editor anymore - therefore no easy sync is
                                // possible. Just give up!
                                RTConnection.log('Synchronization not possible - document has been changed');
                                errorCode = new ErrorCode(ClientError.ERROR_SYNCHRONIZATION_DOC_CHANGED);
                                errorCode.setErrorContext(ErrorContext.SYNCHRONIZATION);
                                self.restoreDocument(errorCode);
                            }
                        } else {
                            // We are not editor anymore - therefore no easy sync is
                            // possible. Just give up!
                            RTConnection.log('Synchronization not possible - editor changed');
                            self.setCurrentInfoState();
                            errorCode = new ErrorCode(ClientError.ERROR_SYNCHRONIZATION_LOST_EDIT_RIGHTS);
                            errorCode.setErrorContext(ErrorContext.SYNCHRONIZATION);
                            self.restoreDocument(errorCode);
                        }
                        break;
                    case 'viewSync':
                        RTConnection.log('Synchronization for viewer completed');
                        locked = false;
                        break;
                }

                mustSyncNowMode = null;
            }
        }

        /**
         * Updates the file state of the application according to the passed
         * data of a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyFileState(data) {

            var // current file descriptor
                file = self.getFileDescriptor(),
                // the new file name
                newFileName = Utils.getStringOption(data, 'fileName', ''),
                // the new file version
                newFileVersion = Utils.getStringOption(data, 'fileVersion', ''),
                // the changed components of the file descriptor
                changedFile = {};

            // file name changed after a 'rename' operation
            if ((newFileName.length > 0) && (newFileName !== Utils.getStringOption(file, 'filename'))) {
                changedFile.filename = newFileName;
            }

            // file version changed after remote client closes the modified file
            if ((newFileVersion.length > 0) && (newFileVersion !== Utils.getStringOption(file, 'version'))) {
                changedFile.version = newFileVersion;
            }

            if (!internalError && !_.isEmpty(changedFile)) {
                self.updateFileDescriptor(changedFile);
                self.getController().update();
                updateState();
            }
        }

        /**
         * Check, whether the selection is the only different object
         * in the two specified arrays containing the information about
         * the clients.
         *
         * @param {Object[]} clients1
         *  An array with user data for clients.
         *
         * @param {Object[]} clients2
         *  An array with user data for clients.
         *
         * @returns {Boolean}
         *  Whether the two specified arrays with client data are identical,
         *  if the selection of the clients is ignored.
         */
        function onlySelectionChanged(clients1, clients2) {

            var reducedClients1 = null,
                reducedClients2 = null;

            if (clients1.length !== clients2.length) { return false; } // simple shortcut

            reducedClients1 = _.copy(clients1, true);
            reducedClients2 = _.copy(clients2, true);

            // deleting the selection objects
            _.each(reducedClients1, function (client) {
                if (client.userData && client.userData.selections) { delete client.userData.selections; }
                if (client.userData && client.userData.ranges) { delete client.userData.ranges; } // 46976
            });

            _.each(reducedClients2, function (client) {
                if (client.userData && client.userData.selections) { delete client.userData.selections; }
                if (client.userData && client.userData.ranges) { delete client.userData.ranges; }
            });

            return _.isEqual(reducedClients1, reducedClients2);
        }

        /**
         * Updates the user data of the application according to the passed
         * data of a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        var applyUserData = this.createSynchronizedMethod(function (data) {

            var // reference to the old array of active clients, for comparison
                oldActiveClients = activeClients,
                // editClients could be generated defered because of UserAPI use
                editClientsPromises = [];

            // trigger a 'docs:users' event only if server data contains a user list
            if (!_.isArray(data.activeUsers)) { return $.when(); }

            // update the list of edit clients (all active and inactive)
            _.each(data.activeUsers, function (client) {
                var valid = _.isObject(client) && _.isString(client.userId) && _.isNumber(client.id);
                if (!valid) {
                    Utils.warn('EditApplication.applyUserData(): missing identifier');
                } else {
                    //we always fetch the name, because the realtime name can be shown in a different way (than Word)
                    editClientsPromises.push(Utils.getUserInfo(client.id).then(function (info) {
                        client.userDisplayName = info.displayName;
                        client.userOperationName = info.operationName;
                        return client;
                    }));
                }
            });

            return $.when.apply($, editClientsPromises).done(function () {

                var // whether only the selection changed
                    // -> in this case only the remote selection needs to be repainted.
                    // -> all other listeners to 'docs:users' can ignore the event (46610).
                    onlySelectionChange = false;

                // translate raw server data to internal representation of the client
                editClients = _.map(arguments, function (client) {

                    var // the new internal client descriptor
                        clientDesc = {};

                    // add identifiers and user name to the descriptor
                    clientDesc.clientId = client.userId; // server sends real-time client identifier as 'userId'
                    clientDesc.userId = client.id; // server sends user identifier as 'id'
                    clientDesc.userName = client.userDisplayName;
                    clientDesc.userOperationName = client.userOperationName;
                    clientDesc.guest = client.guest;

                    // add other client properties
                    clientDesc.active = (client.active === true) || (_.isNumber(client.durationOfInactivity) && (client.durationOfInactivity < 60));
                    clientDesc.remote = clientId !== clientDesc.clientId;
                    clientDesc.editor = oldEditClientId === clientDesc.clientId;

                    // add user data if existing
                    if (_.isObject(client.userData)) {
                        clientDesc.userData = client.userData;
                    }

                    // store unknown clients into the index map
                    if (!(clientDesc.clientId in clientIndexes)) {
                        clientIndexes[clientDesc.clientId] = nextClientIndex;
                        nextClientIndex += 1;
                    }

                    // add client index and color scheme index into the descriptor
                    clientDesc.clientIndex = clientIndexes[clientDesc.clientId];
                    clientDesc.colorIndex = self.getAuthorColorIndex(clientDesc.userOperationName);

                    return clientDesc;
                });

                // create the list of active edit clients (less than 60 seconds inactivity)
                activeClients = _.where(editClients, { active: true });

                // sort by unique client index for comparison
                activeClients = _.sortBy(activeClients, 'clientIndex');

                // notify listeners with all active clients
                if (!_.isEqual(oldActiveClients, activeClients)) {
                    Utils.info('EditApplication.applyUserData(): activeClients=', activeClients);

                    // checking whether only the selection has changed
                    onlySelectionChange = onlySelectionChanged(oldActiveClients, activeClients);

                    if (!onlySelectionChange) { self.trigger('docs:users', activeClients); }

                    self.trigger('docs:users:selection', activeClients);
                }
            });
        });

        /**
         * Applies actions and their operations passed with an 'update' message
         * received from a remote editor of the current document in a
         * background task.
         *
         * @param {Object} data
         *  The data of a server notification message.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all document actions have
         *  been applied.
         */
        function applyRemoteActions(data) {

            if (internalError) {
                RTConnection.error('EditApplication.applyRemoteActions(): failed due to internal error');
                return $.Deferred().reject();
            }

            if (!_.isObject(data)) {
                Utils.error('EditApplication.applyRemoteActions(): invalid data, expecting object:', data);
                return $.Deferred().reject();
            }

            var // all actions contained in the passed update message data
                actions = getActionsFromData(data);

            // do nothing, if the update message does not contain any actions
            if (!_.isArray(actions) || (actions.length === 0)) {
                return $.when();
            }

            // more RT debug logging to better find problems
            RTConnection.log('EditApplication.applyRemoteActions(): last remote action: ' + JSON.stringify(_.last(actions)));

            var // apply all actions asynchronously
                promise = docModel.applyActions(actions, { external: true });

            // remember that actions have been received from the server
            promise.done(function () { remotelyModified = true; });

            // TODO: We have to set internalError to true. Our DOM is not in sync anymore!
            promise.fail(function () {

                // workaround for Bug 46778
                if (!self || self.isInQuit()) { return; }

                RTConnection.error('EditApplication.applyRemoteActions(): failed');
            });

            return promise;
        }

        /**
         * Use the remote actions and the current application state to detect that
         * we have to trigger restore document. This can happen in several
         * situations. E.g. a short-offline phase were the edit rights switched and
         * the document has been modified. Handlers are called in arbitrary order by
         * real-time, etc.
         *
         * @param {Array} actions
         *  Actions received from the remote-side.
         *
         * @returns {Boolean}
         *  TRUE if the restore document function must be triggered or FALSE otherwise.
         */
        function checkStateForMustTriggerRestoreDocument(actions) {
            var // result of this function
                restore = false,
                // client osn retrieved from our model
                clientOSN = docModel.getOperationStateNumber(),
                // first remote operation osn
                firstRemoteOSN = -1;

            if (restoreDocEnabed && (_.isArray(actions) && (actions.length > 0))) {
                if ((mustSyncNowMode === 'editSync') && rtConnection.hasPendingActions()) {
                    // we were the editor, have pending actions and are currently synchronizing due to off-/online transition
                    // receive remote actions
                    restore = true;
                } else if (_.isObject(actions[0] && _.isArray(actions[0].operations))) {
                    // check the client/remote op osn to detect erronous state
                    firstRemoteOSN = actions[0].operations[0].osn;
                    restore = (clientOSN !== firstRemoteOSN);
                }
            }

            return restore;
        }

        /**
         * Applies the passed application state and actions passed with an
         * 'update' push message received from the real-time framework. Actions
         * will be applied in a background loop to prevent unresponsive browser
         * and warning messages from the browser. If actions from older
         * 'update' messages are still being applied, the new message data
         * passed to this method will be cached, and will be processed after
         * the background task for the old actions is finished.
         *
         * @param {Object} data
         *  The message data received from 'update' messages sent by the
         *  real-time framework.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.notify=false]
         *      If set to true, the application will trigger a 'docs:update'
         *      event with the message data, after the actions have been
         *      applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the message data
         *  has been processed.
         */
        var applyUpdateMessageData = this.createSynchronizedMethod(function (data, options) {

            RTConnection.log('EditApplication.applyUpdateMessageData(): received update message:', data);

            // the resulting promise
            var promise = null;
            // received actions via update
            var actions = getActionsFromData(data);

            // check our current state and decide to continue our stop now and request to
            // restore the document
            if (checkStateForMustTriggerRestoreDocument(actions)) {
                // Don't continue to process this update message, because it will fail.
                // Just start to provide the user the possibility to restore his/her
                // document with restoreDocument().
                self.restoreDocument(new ErrorCode(ClientError.ERROR_SYNCHRONIZATION_LOST_EDIT_RIGHTS));
                return;
            }

            // apply all actions asynchronously
            promise = applyRemoteActions(data);

            // additional processing: edit mode, file descriptor, user data, etc.
            promise.done(function () {

                // update application state (read-only mode, etc.)
                applyEditState(data);

                // update application state (fileName, fileVersion)
                applyFileState(data);

                // update user data (user's current selection, etc.)
                applyUserData(data);

                // remove message data from pending queue
                pendingRemoteUpdates.shift();
                RTConnection.log('EditApplication.applyUpdateMessageData(): update processed, pending = ' + pendingRemoteUpdates.length);

                // notify all listeners if specified
                if (Utils.getBooleanOption(options, 'notify', false)) {
                    self.trigger('docs:update', data);
                }
            });

            return promise;
        }, 'EditApplication: applyUpdateMessageData');

        /**
         * Callback executed by the base class BaseApplication to initialize
         * the file descriptor. Creates a new document or converts an existing
         * document, if specified in the launch options.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the new document has been
         *  created successfully. Otherwise, the promise will be rejected.
         */
        function initFileHandler() {

            var // parameters passed to the server request
                requestParams = {},
                // whether to convert the file to a native format
                convert = false;

            // create new document if required
            switch (Utils.getStringOption(launchOptions, 'action', 'load')) {
                case 'load':
                    // load existing document, current file must exist
                    //getFile for new realtime-id in data.file
                    return DriveUtils.getFile(launchOptions.file).then(function (file) {
                        launchOptions.file = file;
                        self.updateFileDescriptor(file);
                    });
                case 'new':
                    _.extend(requestParams, Utils.getObjectOption(initOptions, 'newDocumentParams'));
                    break;
                case 'convert':
                    convert = true;
                    break;
                default:
                    Utils.error('EditApplication.initFileHandler(): unsupported launch action');
                    return $.Deferred().reject();
            }

            var // file options
                templateFile = Utils.getObjectOption(launchOptions, 'templateFile'),
                preserveFileName = Utils.getBooleanOption(launchOptions, 'preserveFileName', false),
                targetFilename = Utils.getStringOption(launchOptions, 'target_filename', /*#. default base file name (without extension) for new OX Documents files (will be extended to e.g. 'unnamed(1).docx') */ gt('unnamed')),
                targetFolderId = Utils.getStringOption(launchOptions, 'target_folder_id', ''),

                // origin of a file attachment stored as template file in Drive (e.g. a mail)
                origin = Utils.getObjectOption(templateFile, 'origin'),

                // the resulting promise returned by this function
                promise = null;

            // sets document into internal error state, returns a rejected promise with an appropriate error descriptor
            function createErrorResponse(errorCode) {

                self.setInternalError(errorCode, ErrorContext.CREATEDOC);
                return $.Deferred().reject({ cause: convert ? 'convert' : 'create' });
            }

            addTimePerformance('createNewDocStart');

            // create the URL options
            _.extend(requestParams, {
                action: 'createdefaultdocument',
                document_type: self.getDocumentType(),
                target_filename: targetFilename,
                target_folder_id: targetFolderId,
                preserve_filename: preserveFileName,
                convert: convert
            });

            if (launchOptions.template) {
                _.extend(requestParams, {
                    action: 'createfromtemplate',
                    file_id: templateFile.id
                });
            }

            // additional parameters according to creation mode
            if (_.isObject(templateFile)) {
                _.extend(requestParams, {
                    file_id: templateFile.id,
                    version: templateFile.version
                });
            }

            // show alert banners after document has been imported successfully
            self.waitForImportSuccess(function () {

                var // the target folder, may differ from the folder in the launch options
                    // Whether the file has been created in another folder. We
                    // have to check the template option which always use the
                    // user folder as target. In that case no message should be shown.
                    copied = (launchOptions.template) ? false : (targetFolderId !== Utils.getStringOption(self.getFileDescriptor(), 'folder_id'));

                // For a file created from a template we have to check the
                // converted attribute in the file descriptor to be sure
                // that it has been converted to another file format or not.
                convert = (launchOptions.template) ? self.getFileDescriptor().converted : convert;
                if (convert) {
                    if (copied) {
                        self.setCurrentInfoState(InfoState.INFO_DOC_CONVERT_STORED_IN_DEFFOLDER);
                    } else {
                        self.setCurrentInfoState(InfoState.INFO_DOC_CONVERTED_AND_STORED);
                    }
                } else if (copied) {
                    self.setCurrentInfoState(InfoState.INFO_DOC_CREATED_IN_DEFAULTFOLDER);
                }
            });

            //optional values for changing "insertField" operation by url-hashs assigned as "fields" or "sourcefields" if it is a csv file
            var fields = _.url.hash('fields');
            var sourcefields = _.url.hash('fields-source');
            if (fields || sourcefields) {
                _.url.hash('fields', null);
                _.url.hash('fields-source', null);
                requestParams.fields = fields;
                requestParams.sourcefields = sourcefields;
            }

            // send the server request to create the new document
            promise = self.sendRequest(IO.FILTER_MODULE_NAME, requestParams);

            // check response whether it contains an error code
            promise = promise.then(function (response) {
                addTimePerformance('createNewDocAfterRequest');
                // forward response data if no error occured, otherwise create a rejected promise with a GUI message
                var error = new ErrorCode(response);
                return error.isError() ? createErrorResponse(error) : response;
            }, function () {
                // server request failed completely, create generic response data
                return createErrorResponse();
            });

            //getFile for new realtime-id in data.file
            promise = promise.then(function (data) {
                return DriveUtils.getFile(data.file).then(function (file) {
                    data.file = file;
                    return data;
                });
            });

            // process data descriptor returned by the server
            promise = promise.then(function (data) {
                // callback for promise piping: leaves busy mode, forwards original state and result
                function leaveBusyEarly(response) {
                    var resolved = this.state() === 'resolved';
                    return self.leaveBusyDuringImport(!resolved).then(
                        // leaving busy state succeeded: return original state and response
                        function () {
                            addTimePerformance('afterLeaveBusyEarly');
                            return resolved ? response : $.Deferred().reject(response);
                        },
                        // leaving busy state failed: always reject the promise chain
                        function () { return resolved ? arguments[0] : response; }
                    );
                }

                if (!_.isObject(data.file)) {
                    Utils.error('EditApplication.initFileHandler(): missing file descriptor');
                    return $.Deferred().reject();
                }

                // add the origin to the file descriptor (e.g. mail descriptor whose attachment has been copied to Drive)
                if (origin) { data.file.origin = origin; }
                data.file.source = 'drive';

                // determine if server sent an extended response with initial document contents
                fastEmpty = _.isString(data.htmlDoc) && _.isArray(data.operations) && (data.operations.length > 0);
                if (fastEmpty && fastEmptyLoadHandler) {
                    // invoke fast-load handler provided by the application
                    return fastEmptyLoadHandler.call(self, data.htmlDoc, { operations: data.operations })
                        // leave busy mode early (also if import handler fails)
                        .then(leaveBusyEarly, leaveBusyEarly)
                        // forward response data to resulting piped promise
                        .then(function () { return data; });
                }

                // forward returned data to piped promise
                return data;
            });

            // return the final piped promise with the file descriptor as response
            return promise.then(function (data) { return data.file; });
        }

        /**
         * Loads the document described in the current file descriptor.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the document has been loaded,
         *  or rejected when an error has occurred.
         */
        function importHandler() {

            var // time for initializing document
                initializationTime = 0,
                // Performance: Whether stored data can be used for loading the document
                useStorageData = false,
                // connect update messages
                connectUpdateMessages = [],
                // update with document operations
                asynchronousLoadFinished = $.Deferred(),
                // the promise that will be resolved with importing the document
                importPromise = null,
                // connect deferred
                connectDeferred = $.Deferred();

            // shows a log message with a duration in the browser console
            function logDuration(duration, message1, message2) {
                Utils.info('EditApplication.importHandler(): ' + message1 + ' in ' + duration + 'ms' + (_.isString(message2) ? (' (' + message2 + ')') : ''));
            }

            // set busy mode during asynchronous file loading
            function enterBusyMode() {

                docView.enterBusy({
                    cancelHandler: _.bind(self.quit, self),
                    immediate: true,
                    showFileName: true,
                    warningLabel: gt('Sorry, your document is very large. It will take some time to be loaded.'),
                    warningDelay: 10000
                });
            }

            // invokes the passed callback function that must return a promise, and logs the time until resolving the callback promise
            function logCallback(callback, message, timerLogKey, count) {

                var // profiling: start time of the callbck invocation
                    lapTime = _.now(),
                    // the promise used for logging the time for the callback
                    logPromise = null;

                addTimePerformance(timerLogKey + 'Start', lapTime);

                logPromise = _.isFunction(callback) ? callback.call(self) : $.when();
                logPromise.fail(function () { Utils.warn('EditApplication.importHandler(): ' + message + ' FAILED!'); });

                logPromise = logPromise.then(function (response) {

                    var endTime = _.now(),
                        totalTime = endTime - lapTime,
                        timePerElem = (count > 0) ? Utils.round(totalTime / count, 0.001) : 0;

                    logDuration(totalTime, message, (count > 0) ? (count + ' elements, ' + timePerElem + 'ms per element') : null);
                    addTimePerformance(timerLogKey + 'End', endTime);

                    // bug 32520: reject the chain of deferreds when closing document during import
                    return self.isInQuit() ? $.Deferred().reject({ cause: 'closing' }) : response;
                });

                return logPromise;
            }

            // Function to handle critical RT notifications while loading the document.
            function handleCriticalMsgWhileLoading(event) {
                if (_.isObject(event)) {
                    switch (event.type) {
                        case 'reset':
                            self.setInternalError(ClientError.ERROR_CONNECTION_RESET_RECEIVED, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_CONNECTION_RESET_RECEIVED);
                            break;
                        case 'offline':
                            self.setInternalError(ClientError.ERROR_OFFLINE_WHILE_LOADING, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_OFFLINE_WHILE_LOADING);
                            break;
                        case 'timeout':
                            self.setInternalError(ClientError.ERROR_CONNECTION_TIMEOUT, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_CONNECTION_TIMEOUT);
                            break;
                        case 'error:join':
                            self.setInternalError(ClientError.ERROR_CONNECTION_JOIN_FAILED, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_CONNECTION_JOIN_FAILED);
                            break;
                        case 'error:disposed':
                            self.setInternalError(ClientError.ERROR_CONNECTIONINSTANCE_DISPOSED, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_CONNECTIONINSTANCE_DISPOSED);
                            break;
                        case 'error:stanzaProcessingFailed':
                            self.setInternalError(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE);
                            break;
                        default:
                            self.setInternalError(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE, ErrorContext.LOAD);
                            connectDeferred.reject(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE);
                            break;
                    }
                }
            }

            // Function to collect update data while connecting to/importing document.
            // Must handle error messages, too in case of asynchronous loading.
            function collectUpdateEvent(event, data) {
                if (data && data.finalLoadUpdate) {  // 'finalLoadUpdate' shows, that the asynchronous loading is finished
                    if (data.hasErrors) {
                        asynchronousLoadFinished.reject(data);
                    } else {
                        asynchronousLoadFinished.resolve(data);
                        RTConnection.log('Final load update received for asynchronous load');
                    }
                } else {
                    connectUpdateMessages.push(data);
                    RTConnection.log('Update received during document import');
                }
            }

            // After being successfully connected to real-time framework and after all actions are
            // downloaded, in 'processDocumentActions' function the following steps happen:
            //
            // 1. check for errors, local storage, fast load and fast empty documents
            // 2. calling application specific pre process handler
            // 3. applying all actions
            // 4. calling application specific post process handler
            // 5. finalizing the import process
            //
            function processDocumentActions(data) {

                var // the initial action from the server response
                    actions = getActionsFromData(data),
                    // 'update' messages received by answer from connect
                    docUpdateMessages = [],
                    // update data retrieved from connect message
                    updateData = getUpdateAttributesFromData(data),
                    // the error data sent with the server response (no indication of an error)
                    error = new ErrorCode(data),
                    // the result passed with a rejected Deferred object
                    errorResult = null,
                    // the key under which the data might be stored in browser cache
                    loadStorageKey = null,
                    // the key containing the version of the saved string in the storage
                    storageVersionKey = null,
                    // the saved storage version
                    storageVersion = 0,
                    // the key under which the OSN and the file version for the file are stored in browser cache
                    loadStorageKeyOSN = null, storageFileVersionKey = null,
                    // the supported extensions for the local storage
                    supportedExtensions = null,
                    // a counter for the supported extensions
                    counter = 0,
                    // the promise for pre processing the document loading
                    preProcessPromise = null;

                // called for normal doc loading, and for fast load of empty doc
                // after postprocessing is done. Applies edit state and update message data
                function finalizeImport(data, docUpdateMessages) {

                    // register handlers for hangup and switching edit rights notifications
                    rtConnection.on({
                        hangup: connectionHangupHandler,
                        preparelosingeditrights: handlePrepareLosingEditRights,
                        declinedacquireeditrights: handleDeclinedAcquireEditRights,
                        acceptedacquireeditrights: handleAcceptedAcquireEditRights,
                        noUpdateFromServer: handleNoUpdateFromServer,
                        flusherrorinfo: handleFlushErrorInfo,
                        infomessage: handleInfoMessage,
                        renameandreload: handleRenameAndReload
                    });

                    // update application state (read-only mode, etc.)
                    applyEditState(data);
                    addTimePerformance('editMode');

                    // extract the document restore ID which must be provided, if the client
                    // wants to request a document restore
                    docRestoreId = Utils.getStringOption(data.syncInfo, 'restore_id', null);

                    // apply all 'update' messages collected during import (wait for the import promise,
                    // otherwise methods that work with the method isImportFinished() will not run correctly)
                    self.waitForImportSuccess(function () {
                        // first apply actions received on connect() answer
                        RTConnection.log('Applying document update messages, count = ' + docUpdateMessages.length);
                        _.each(docUpdateMessages, function (updateData) { applyUpdateMessageData(updateData); });
                        // second apply actions received during connect() and document import
                        RTConnection.log('Applying update messages collected during connect/import, count = ' + connectUpdateMessages.length);
                        _.each(connectUpdateMessages, function (updateData) { applyUpdateMessageData(updateData); });
                        // reconnect 'update' events for direct message processing, forward event
                        rtConnection.off('update', collectUpdateEvent).on('update', function (event, updateData) {
                            // public method triggers 'docs:update' event
                            self.applyUpdateMessageData(updateData);
                        });
                        // remove temporary reset handler and set our global one
                        rtConnection.off('reset', handleCriticalMsgWhileLoading).on('reset', connectionResetHandler);
                        rtConnection.off('offline', handleCriticalMsgWhileLoading).on('offline', connectionOfflineHandler);
                        rtConnection.off('timeout', handleCriticalMsgWhileLoading).on('timeout', connectionTimeoutHandler);
                        rtConnection.off('error:stanzaProcessingFailed', handleCriticalMsgWhileLoading).on('handleCriticalMsgWhileLoading', connectionProcessingStanzaFailed);

                        updateState();
                    });

                    return $.when();
                }

                // loading document from local storage, if possible
                function handleLocalStorage() {

                    // helper function to clear content in local storage
                    function clearLocalStorage() {
                        _.each(supportedExtensions, function (oneExtensionDefintion) { localStorage.removeItem(loadStorageKey + oneExtensionDefintion.extension); });
                        localStorage.removeItem(loadStorageKeyOSN);
                        localStorage.removeItem(storageVersionKey);
                        localStorage.removeItem(storageFileVersionKey);
                    }

                    if (window.Storage && localStorageApp && saveFileInLocalStorage && isLocalStorageSupported && data.syncInfo &&
                        data.syncInfo.fileVersion && data.syncInfo['document-osn'] && _.isFinite(data.syncInfo['document-osn']) &&
                        data.syncInfo.fileId && data.syncInfo.folderId &&
                        _.isFunction(docModel.setFullModelNode) && _.isFunction(docModel.getSupportedStorageExtensions)
                    ) {

                        loadStorageKey = data.syncInfo.fileId + '_' + data.syncInfo.folderId;
                        storageFileVersionKey = loadStorageKey + '_FILEVERSION';
                        storageVersionKey = loadStorageKey + '_STORAGEVERSION';
                        loadStorageKeyOSN = loadStorageKey + '_OSN';

                        supportedExtensions = docModel.getSupportedStorageExtensions();

                        // checking the supported storage version
                        storageVersion = (localStorage.getItem(storageVersionKey) && parseInt(localStorage.getItem(storageVersionKey), 10)) || 1;

                        if (storageVersion >= requiredStorageVersion) {

                            Utils.info('EditApplication.importHandler(): document-osn=' + data.syncInfo['document-osn'] + ', document-rev=' + data.syncInfo.fileVersion);

                            // Support for localStorage -> comparing OSN and file version
                            if ((localStorage.getItem(storageFileVersionKey) && localStorage.getItem(storageFileVersionKey) === data.syncInfo.fileVersion) &&
                                ((localStorage.getItem(loadStorageKeyOSN)) && parseInt(localStorage.getItem(loadStorageKeyOSN), 10) === parseInt(data.syncInfo['document-osn'], 10))) {

                                Utils.iterateArray(supportedExtensions, function (oneExtensionDefintion) {

                                    var // one specific local storage key
                                        oneLocalStorageKey = loadStorageKey + oneExtensionDefintion.extension,
                                        // the options for setFullModelNode
                                        options = { usedLocalStorage: true };

                                    if (_.isObject(oneExtensionDefintion.additionalOptions)) { _.extend(options, oneExtensionDefintion.additionalOptions); }

                                    if (localStorage.getItem(oneLocalStorageKey)) {
                                        Utils.info('EditApplication.importHandler(): loading document from local storage. Key: ' + oneLocalStorageKey + ' Length: ' + localStorage.getItem(oneLocalStorageKey).length);
                                        loadedFromStorage = true;
                                        useStorageData = true;
                                        if (counter === 0) { docModel.setLoadDocumentOsn(parseInt(data.syncInfo['document-osn'], 10)); }
                                        docModel.setFullModelNode(localStorage.getItem(oneLocalStorageKey), options);
                                        counter++;
                                    } else {
                                        Utils.info('EditApplication.importHandler(): optional local storage value not found. Key: ' + oneLocalStorageKey);
                                        if (!oneExtensionDefintion.optional) {
                                            Utils.info('EditApplication.importHandler(): file in local storage not found. Key: ' + loadStorageKey + '. But found OSN key: ' + loadStorageKeyOSN);
                                            clearLocalStorage();
                                            return Utils.BREAK;
                                        }
                                    }
                                });

                            } else {
                                Utils.info('EditApplication.importHandler(): file in local storage not found. Key: ' + loadStorageKey);
                                clearLocalStorage();
                            }
                        } else {
                            Utils.info('EditApplication.importHandler(): version of file in local storage not supported. Key: ' + loadStorageKey);
                            clearLocalStorage();
                        }
                    }

                }

                // loading document with fast load, if possible
                function handleFastLoad() {

                    var // the html string sent from the server
                        htmlDoc = null,
                        // a helper string for header or footer and comments
                        unionString = '';

                    //if fastLoad has an error & debug is activated we want to yell it loud
                    if (!data.htmlDoc && Config.DEBUG && !fastEmpty && docModel.setFullModelNode && self.getController().getItemValue('debug/useFastLoad') && (self.getFileDescriptor() && !Utils.stringEndsWith(self.getFileDescriptor().filename, '_ox'))) {
                        docView.yell({ type: 'error', headline: _.noI18n('Error in fastload'), message: _.noI18n('you should have a look in the backend logfiles ("Exception while creating the generic html document")') });
                    }

                    if (!loadedFromStorage && data.htmlDoc && _.isFunction(docModel.setFullModelNode) && !fastEmpty) { // if doc is empty, this was already called
                        Utils.info('EditApplication.importHandler(): load complete HTML document, osn: ', data.syncInfo['document-osn']);
                        docModel.setLoadDocumentOsn(parseInt(data.syncInfo['document-osn'], 10));
                        //docModel.setFullModelNode(data.htmlDoc);

                        // markup is in form of object, parse and get data from mainDocument
                        htmlDoc = JSON.parse(data.htmlDoc);

                        if (htmlDoc.mainDocument) {
                            docModel.setFullModelNode(htmlDoc.mainDocument);
                        }
                        // if there is header/footer data, restore that, too
                        if (htmlDoc.headerFooter) {
                            _.each(htmlDoc.headerFooter, function (element) {
                                unionString += element;
                            });
                            docModel.setFullModelNode(unionString, { headerFooterLayer: true });
                            docModel.getPageLayout().headerFooterCollectionUpdate();
                        }
                        // if there is comments data, restore that, too
                        if (htmlDoc.comments) {
                            unionString = '';
                            _.each(htmlDoc.comments, function (element) {
                                unionString += element;
                            });
                            docModel.setFullModelNode(unionString, { commentLayer: true });
                        }

                        // inform listener, that fastload is done (38562)
                        docModel.trigger('fastload:done');
                    }

                }

                // detecting errors from server (and client)
                function handleErrorResults() {

                    var // an error object
                        errorObject = null;

                    if (Utils.getBooleanOption(data, 'hasErrors', false)) {
                        Utils.error('EditApplication.importHandler(): server side error');
                        errorObject = { cause: 'server' };
                    } else if (Utils.getBooleanOption(data, 'operationError', false)) {
                        Utils.error('EditApplication.importHandler(): operation failure');
                        errorObject = { cause: data.message };
                    } else if (error.isError()) {
                        Utils.error('EditApplication.importHandler(): server side error, error code = ' + error.getCodeAsConstant());
                        errorObject = { cause: 'server' };
                    } else if (clientId.length === 0) {
                        Utils.error('EditApplication.importHandler(): missing client identifier');
                        errorObject = { cause: 'noclientid' };
                    } else if (!_.isArray(actions) || (actions.length === 0)) {
                        Utils.error('EditApplication.importHandler(): missing actions');
                        errorObject = { cause: 'noactions' };
                    }

                    return errorObject;
                }

                // receiving the 'operation' name of the current user
                function getOperationName() {
                    return Utils.getUserInfo(ox.user_id).done(function (info) {
                        clientOperationName = info.operationName;
                    });
                }

                // get the own unique client identifier
                clientId = Utils.getStringOption(data, 'clientId', '');
                clientIndexes[clientId] = 0;

                // check error conditions, leaving further processing immediately in error case
                errorResult = handleErrorResults();

                if (_.isObject(errorResult)) {
                    // return rejected Deferred object with error information
                    errorResult = _.extend(errorResult, genericImportFailedHandler(data));
                    return $.Deferred().reject(errorResult);
                }

                // saveFileInLocalStorage can be overwritten with user setting in debug mode
                checkDebugUseLocalStorage();

                // checking, if the file can be loaded from storage (or another source)
                handleLocalStorage();

                // checking, if the file can be loaded with fast load
                handleFastLoad();

                // connect sends us the first update event together with the action data
                docUpdateMessages.push(updateData);

                // shortcut for fast empty documents
                if (fastEmpty) { // we already have preprocessing, actions and formatting applied, so we finalize doc importing
                    // osn is -1 and we need to set it to correct value, server osn is used
                    docModel.setOperationStateNumber(parseInt(data.serverOSN, 10));
                    return finalizeImport(data, docUpdateMessages).then(getOperationName);
                }

                // invoke pre-process callback function passed to the constructor of this application
                preProcessPromise = logCallback(preProcessHandler, 'preprocessing finished', 'preProcessing');

                // setting the full name of the client in 'getOperationName'
                preProcessPromise = preProcessPromise.then(getOperationName).then(function () {

                    var // some options for applying actions
                        actionOptions = { external: true, useStorageData: useStorageData },
                        // total number of operations, for logging
                        operationCount = Config.DEBUG ? Utils.getSum(actions, function (action) {
                            return Utils.getArrayOption(action, 'operations', []).length;
                        }) : 0,
                        // the promise for applying actions
                        applyActionsPromise = null;

                    // callback function passed to the method invokeCallbackWithProgress()
                    function applyActions() {
                        var detachAppPane = Utils.getBooleanOption(initOptions, 'applyActionsDetached', false);
                        if (detachAppPane) { docView.detachAppPane(); }
                        return docModel.applyActions(actions, actionOptions).always(function () {
                            if (detachAppPane && docView && !docView.destroyed) { docView.attachAppPane(); }
                        });
                    }

                    // Load performance: Execute only specified operations during load of document, if the document can be loaded from local storage
                    if (useStorageData && operationFilter) { actionOptions.filter = operationFilter; }

                    // apply actions at document model asynchronously, update progress bar
                    applyActionsPromise = logCallback(applyActions, 'operations applied', 'applyActions', operationCount);

                    applyActionsPromise = applyActionsPromise.then(function () {

                        var // the promise for post processing the document loading
                            postProcessPromise = null;

                        // After successfully applied all actions we must have a valid operation state number.
                        // Otherwise there is a synchronization error and we have to stop processing the document
                        // further showing an error message.
                        if (docModel.getOperationStateNumber() < 0) {
                            return $.Deferred().reject({ cause: 'applyactions', headline: gt('Synchronization Error') });
                        }

                        // successfully applied all actions, at least one
                        // -> invoke post-process callback function passed to the constructor of this application

                        postProcessPromise = logCallback(useStorageData ? postProcessHandlerStorage : postProcessHandler, 'postprocessing finished', 'postProcessing');

                        postProcessPromise = postProcessPromise.then(function () {
                            return finalizeImport(data, docUpdateMessages);
                        }, function (response) {
                            return _.extend({ cause: 'postprocess' }, response); // failure: post-processing failed
                        });

                        return postProcessPromise;

                    }, function (response) {
                        // failure: applying actions failed
                        // RT connection already destroyed when quitting while loading
                        if (rtConnection) {
                            rtConnection.off('update', collectUpdateEvent);
                        }
                        return _.extend({ cause: 'applyactions' }, response);
                    });

                    return applyActionsPromise;

                }, function (response) {
                    // failure (pre-processing failed): adjust error response
                    return _.extend({ cause: 'preprocess' }, response);
                });

                return preProcessPromise;
            }

            // Processing the data received from the server connect
            // -> check for errors from server
            // -> check for special case in which preview data were sent from server
            // -> synchronize call of 'processDocumentActions' in the synchronous and asynchronous loading process
            function processConnectData(data) {

                var // extract error object from the server response
                    error = new ErrorCode(data),
                    // extract optional syncLoad property from the server response
                    syncLoad = Utils.getBooleanOption(data, 'syncLoad', false),
                    // data for an early preview of the imported document (must contain some operations)
                    previewData = Utils.getObjectOption(data, 'preview'),
                    // the promise for the preprocess callback
                    localPromise = $.when();

                // helper function to unify synchronous and asynchronous loading processes
                // -> only this function triggers the worker 'processDocumentActions' in a defined state
                function prepareProcessDocumentActions() {

                    var // a locally used promise
                        helperPromise = null;

                    // synchronizes with a finalLoad update that is required to finalize a asynchronous loading.
                    function handleAsynchronousLoading() {
                        return asynchronousLoadFinished.then(function (asyncData) {
                            //asyncLoadingFinished = true;
                            return processDocumentActions(asyncData);
                        }, function (asyncData) {
                            //asyncLoadingFinished = true;
                            return genericImportFailedHandler(_.extend({ cause: 'asyncload' }, asyncData));
                        });
                    }

                    if (syncLoad || error.isError()) {
                        helperPromise = processDocumentActions(data); // -> direct call of 'processDocumentActions'
                    } else {
                        // waiting for asynchronous update with additional operations
                        // -> deferred call of 'processDocumentActions' after asynchronous load is finished
                        helperPromise = logCallback(handleAsynchronousLoading, 'asynchronous import', 'asyncLoad');
                    }

                    return helperPromise;
                }

                // helper function with preview information. In this scenario it is necessary to call the preprocess
                // handler (the same that is called in 'processDocumentActions'), before operations are applied.
                function handlePreviewData() {

                    var // the promise for handling the preview data
                        previewPromise = logCallback(preProcessHandler, 'preview preprocessing finished', 'previewPreProcessing');

                    previewPromise = previewPromise.then(function () {

                        // avoid that the preprocess handler is called again
                        preProcessHandler = null;

                        // apply the operations
                        var promise = docModel.applyActions({ operations: data.preview.operations }, { external: true });

                        // handle result of applying the preview operations
                        promise = promise.then(function () {
                            // invoke the preview handler after the operations have been applied
                            // TODO: treat false as error, or continue silently in busy mode?
                            return previewHandler.call(self, previewData) ? self.leaveBusyDuringImport() : $.when();
                        }, function () {
                            // error during applying operations (bug 36639)
                            data.operationError = true;
                            data.message = gt('An unrecoverable error occurred while modifying the document.');
                            syncLoad = true; // misusing for error handling
                            return $.when();
                        });

                        return promise;
                    });

                    return previewPromise;
                }

                // checking the server answer for preview data
                if (previewData && _.isArray(previewData.operations)) {
                    localPromise = handlePreviewData(); // special synchronous handling to show document preview
                }

                // checking server answer: Is this synchronous load, asynchronous load?
                localPromise = localPromise.then(prepareProcessDocumentActions);

                return localPromise;
            }

            // starting initialization
            addTimePerformance('importDocStart');

            // during opened undo group, collect all operations in the same action
            docModel.getUndoManager().on('undogroup:open', function () {
                pendingAction = createAction();
            });

            // undo group closed: send the pending action to the server
            docModel.getUndoManager().on('undogroup:close', function () {
                if (pendingAction.operations.length > 0) {
                    registerAction(pendingAction);
                }
                pendingAction = null;
            });

            // collect all operations generated by the document model, send them to the server
            docModel.on('operations:success', function (event, operations, external) {
                // ignore external operations
                if (!external && !operationsBlock) {
                    if (_.isObject(pendingAction)) {
                        // if inside an undo group, push all operations into the pending action
                        RTConnection.log('operations:success - push operations into pending action');
                        pendingAction.operations = pendingAction.operations.concat(operations);
                    } else {
                        // otherwise, generate and send a self-contained action
                        registerAction(createAction(operations));
                    }
                }
            });

            // on prepare to lose edit rights, send cached operations
            docModel.on('cacheBuffer:flush', function () {
                RTConnection.log('cacheBuffer:flush received - send cached operations');
                if (_.isFunction(mergeCachedOperationsHandler) && cacheBuffer.length) {
                    actionsBuffer = mergeCachedOperationsHandler.call(self, actionsBuffer, cacheBuffer);
                    sendActions();
                }
            });

            // error occurred while applying operations
            docModel.on('operations:error', function () {
                if (self.isImportFinished()) {
                    self.setInternalError(ClientError.ERROR_WHILE_MODIFYING_DOCUMENT, ErrorContext.GENERAL);
                    RTConnection.error('An unrecoverable error occurred while modifying the document');
                } else {
                    self.setInternalError(ClientError.ERROR_WHILE_LOADING_DOCUMENT, ErrorContext.GENERAL);
                    RTConnection.error('An unrecoverable error occurred while loading the document');
                }
            });

            // disable dropping of images onto the application background area
            self.getWindowNode().on('drop', false);

            // create the real-time connection to the server, register event listeners
            rtConnection = new RTConnection(self, initOptions);

            rtConnection.on({
                online: connectionOnlineHandler,
                offline: handleCriticalMsgWhileLoading,
                timeout: handleCriticalMsgWhileLoading,
                'error:notMember': connectionNotMemberHandler,
                reset: handleCriticalMsgWhileLoading,
                'error:stanzaProcessingFailed': handleCriticalMsgWhileLoading,
                'error:joinFailed': handleCriticalMsgWhileLoading,
                'error:disposed': handleCriticalMsgWhileLoading,
                update: collectUpdateEvent,
                alive: aliveHandler
            });

            // immediately show the busy blocker screen
            if (!fastEmpty) { enterBusyMode(); }

            // log the initalization time needed before import actually starts
            logDuration(initializationTime, 'initialization finished');

            // connect to server, receive initial operations of the document
            importPromise = logCallback(function () {

                var // parameters for connect
                    connectOptions = {
                        fastEmpty: fastEmpty,
                        useLocalStorage: saveFileInLocalStorage,
                        app: self.getDocumentType(),
                        newDocument: (Utils.getStringOption(launchOptions, 'action') === 'new')
                    };

                rtConnection.connect(connectOptions).then(function (data) {
                    connectDeferred.resolve(data);
                }, function (data) {
                    connectDeferred.reject(data);
                });

                return connectDeferred.promise();

            }, 'actions downloaded', 'connect');

            // analyzing data received from the server after connection is established.
            // The generic fail handler needs to react to all failures in processConnectData (37484))
            importPromise = importPromise.then(processConnectData).then(null, genericImportFailedHandler);

            importPromise.always(function () {
                // TODO: show a success message if the preview was active too long?
                previewStartTime = null;
                updateState();
                addTimePerformance('leaveBusy');
            });

            return importPromise;
        }

        /**
         * Generic import failed handler which handles general errors that are
         * not application dependent. Application specific errors will be
         * handled by an optional importFailedHandler which can be set using
         * 'initOptions.importFailedHandler'.
         *
         * @param {Object} response
         *  The server response containing internal error information that can
         *  be used to create UI specific strings.
         *
         * @returns {Object}
         *  The result object extended with user information to be presented to
         *  the user describing what is the root cause of the error.
         */
        function genericImportFailedHandler(response) {

            if (self.isInQuit()) {
                response = { cause: 'quit' };
            }

            // this function must only run once
            // TODO: this is necessary because of call inside 'function processDocumentActions(data)'
            // -> this needs to be improved
            if (failureAlreadyEvaluated) { return { cause: '' }; }

            var // specific error code sent by the server
                error = new ErrorCode(response),
                // the error code constant as string
                constant = error.getCodeAsConstant(),
                // temporary object to extract possible RT error
                tmpError = null,
                // possible error code provided by low-level layers
                code = null;

            internalError = true;
            importFailedHandler.call(self, response);

            response = _.clone(response);
            switch (constant) {
                case 'GENERAL_FILE_NOT_FOUND_ERROR': movedAway = true; break;
                case 'NO_ERROR':
                    // in case we haven't received a office backend error we have to
                    // check for a possible real-time error
                    tmpError = ErrorCode.extractRealtimeError(response);
                    if (tmpError && tmpError.code && tmpError.prefix === 'RT_STANZA') {
                        RTConnection.log('ImportFailedHandler: RT error detected: RT_STANZA-' + tmpError.code);
                        switch (tmpError.code) {
                            case 6: // STANZA_INTERNAL_SERVER_ERROR
                            case 1009: // RESULT_MISSING
                            case 1014: // RESPONSE_AWAIT_TIMEOUT
                                error = new ErrorCode(ClientError.ERROR_LOAD_DOC_FAILED_SERVER_TOO_BUSY);
                                break;
                            case 1010: // GROUP_DISPOSED
                                error = new ErrorCode(ClientError.ERROR_CONNECTIONINSTANCE_DISPOSED);
                                break;
                            case 1016: // COMPONENT_HANDLE_CREATION_DENIED
                                error = new ErrorCode(ClientError.ERROR_SERVER_DENIES_CONNECTION);
                                break;
                            default:
                                error = new ErrorCode(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE);
                        }
                    }
                    break;
            }

            // Try to extract error information from the possible properties
            // 'cause' (used by our internal system) or 'error' used by the
            // rt-core client-framework.
            tmpError = response.cause || response.error;
            if (!error.isError() && _.isString(tmpError)) {
                // handle more general errors here, which use the properties 'cause'/'error'
                // provided by our own code or http.js (used by RT core framework for
                // communication)
                switch (tmpError) {
                    case 'bad server component':
                        error = new ErrorCode(ClientError.ERROR_BAD_SERVER_COMPONENT_DETECTED);
                        break;
                    case 'timeout': //#40441
                        error = new ErrorCode(ClientError.ERROR_LOAD_DOC_FAILED_SERVER_TOO_BUSY);
                        break;
                    case '0 simulated fail':
                        error = new ErrorCode(ClientError.ERROR_CONNECTION_RESET_RECEIVED);
                        break;
                }

                // Unfortunately http.js provides error information in a very strange way,
                // (sometimes using translated strings in error: and no code: prop)
                // Therefore we have to check a second property (which is sometimes
                // used by http.js)
                if (!error.isError()) {
                    code = _.isString(response.code) ? response.code : '';
                    switch (code) {
                        case '0000': // Server did not send a response, see io.ox/core/http.js
                            error = new ErrorCode(ClientError.ERROR_CONNECTION_TIMEOUT);
                            break;
                    }
                }
            }

            // always have a fallback error
            if (!error.isError()) {
                error = new ErrorCode(ClientError.ERROR_UNKNOWN_REALTIME_FAILURE);
            }

            // Bug 37898: No need to set an internal error during application shut-down.
            if (!self.isInQuit()) {
                // Use showMessage false to prevent setInternalError to show the error message.
                // The base view uses a importFailed handler to show the message in a more
                // general way.
                self.setInternalError(error, ErrorContext.LOAD, null, { showMessage: false });
            }

            // not running this function twice
            failureAlreadyEvaluated = true;

            return response;
        }

        /**
         * Creates and returns a new action containing operations and
         * additional data.
         *
         * @param {Array} [operations]
         *  The operations contained by the new action. If omitted, an empty
         *  array will be inserted into the action.
         *
         * @returns {Object}
         *  The new action.
         */
        function createAction(operations) {
            return { operations: operations || [] };
        }

        /**
         * Sends all actions that are currently stored in the actions buffer.
         * If called repeatedly while still sending actions, the new actions
         * buffered in the meantime will be stored in another internal buffer,
         * and will be sent afterwards, until no new actions have been inserted
         * in the actions buffer anymore.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the last real-time message
         *  containing new actions has been acknowledged, or rejected if any
         *  real-time message has failed.
         */
        var sendActions = (function () {

            var // the Deferred object that will be returned to the caller
                resultDef = null;

            // sends the current actions buffer (calls itself recursively)
            function sendActionsBuffer() {

                if (RTConnection.debug && inPrepareLosingEditRights && (actionsBuffer && actionsBuffer.length > 0)) {
                    // we need to debug operations while we are in the process of switching edit rights
                    var operations = actionsBuffer[actionsBuffer.length - 1].operations;
                    if (operations && (operations.length > 0)) {
                        Utils.log('EditApplication.sendActionsBuffer(): last action:', _.last(operations));
                    }
                }

                // try to reduce the number of actions and operations that need to be sent to the server
                // testing optimizations because of problems with second undo/redo
                if (optimizeOperationsHandler) {
                    actionsBuffer = optimizeOperationsHandler.call(self, actionsBuffer);
                }

                // send all existing actions
                rtConnection.sendActions(actionsBuffer)
                .done(function () {
                    if (internalError) {
                        RTConnection.error('EditApplications.sendActionsBuffer(): sent successful, but internal error detected!');
                        resultDef.reject();
                    } else if (actionsBuffer.length === 0) {
                        resultDef.resolve();
                    } else {
                        sendActionsBuffer();
                    }
                })
                .fail(function (response) {
                    RTConnection.error('EditApplications.sendActionsBuffer(): rtConnection.sendActions promise failed!');
                    resultDef.reject(response);
                });

                actionsBuffer = [];
            }

            return function () {

                RTConnection.log('EditApplication.sendActions(): entered');

                // a real-time message is currently running: return the current
                // result Deferred object that will be resolved after all messages
                if (sendingActions) {
                    RTConnection.log('EditApplication.sendActions(): sending in progress - using current promise');
                    return $.when(resultDef);
                }

                // internal error: immediately return with rejected Deferred
                if (internalError) {
                    RTConnection.error('EditApplication.sendActions(): internal error detected - promise rejected!');
                    return $.Deferred().reject();
                }

                // no new actions: return immediately
                if (actionsBuffer.length === 0) {
                    RTConnection.log('EditApplication.sendActions(): no need to send action buffer empty');
                    return $.when();
                }

                // prepare the result Deferred object (will be kept unresolved until
                // all actions have been sent, also including subsequent iterations
                resultDef = $.Deferred().always(function () {
                    RTConnection.log('EditApplication.sendActions(): current send promise resolved - sendingActions = false');
                    sendingActions = false;
                    sentActionsTime = _.now();
                    updateState();
                });
                sendingActions = true;
                updateState();

                // start sending the actions (will call itself recursively)
                sendActionsBuffer();
                return $.when(resultDef);
            };

        }()); // end of local scope for sendActions() method

        /**
         * Registers the passed action to be sent to the server. All registered
         * actions will be collected and sent automatically in a debounced
         * call to the sendActions() method.
         *
         * @param {Object} action
         *  A new action to be sent to the server.
         */
        var registerAction = (function () {

            // direct callback: called every time registerAction() is called, push action to array
            function storeAction(action) {

                // whether an exisiting cache buffer was emptied
                var usedCacheBuffer = false;

                RTConnection.withLogging(function () {
                    var lastOp = (action.operations.length > 0) ? JSON.stringify(_.last(action.operations)) : null;
                    RTConnection.log('EditApplication.storeAction(): ' + (lastOp ? ('last operation: ' + lastOp) : 'no operations provided'));
                });

                if (mergeCachedOperationsHandler && cacheBuffer.length > 0) {
                    actionsBuffer = mergeCachedOperationsHandler.call(self, actionsBuffer, cacheBuffer);
                    mergeCachedOperationsHandler = null;
                    cacheBuffer = [];
                    usedCacheBuffer = true;
                }

                if (!internalError && (action.operations.length > 0 || usedCacheBuffer)) {
                    if (action.operations.length > 0) { actionsBuffer.push(action); }
                    locallyModified = true;
                    updateState();
                }
            }

            // create the debounced method to store and send the actions
            var storeAndSendDebounced = self.createDebouncedMethod(storeAction, sendActions, {
                delay: sendActionsDelay,
                maxDelay: 5 * sendActionsDelay,
                infoString: 'EditApplication: sendActions'
            });

            // the resulting registerAction() method returned from local scope
            function registerAction(action) {
                if (distribute) {
                    storeAndSendDebounced(action);
                } else {
                    RTConnection.log('evaluateAction: distribute = false, operations must be cached');
                    cacheBuffer.push(action);
                }
            }

            return registerAction;
        }());

        /**
         * Sends all pending actions and flushes the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the real-time message has been
         *  acknowledged.
         */
        function flushDocumentHandler(reason) {

            // do nothing if application is already in internal error state
            if (internalError) { return $.Deferred().reject(); }

            // first, call the flush preparation handler passed to the constructor
            var promise = $.when(prepareFlushHandler.call(self, reason));

            // force sending all pending actions
            promise = promise.then(sendActions);

            // finally, do the RT flush
            promise = promise.then(function () { return rtConnection.flushDocument(); });

            // fix Bug 46125 & hack for RT1
            // FIXME: remove hack when using RT2
            promise = promise.then(function () {
                var timeDiff = 5000 - (_.now() - sentActionsTime);
                if (timeDiff > 0) { return self.executeDelayed(Utils.NOOP, timeDiff); }
            });

            return promise;
        }

        /**
         * The handler function that will be called when the application is
         * asked to be closed. If the edited document has unsaved changes, a
         * dialog will be shown asking whether to save or drop the changes, or
         * to continue editing.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the application can be closed
         *  (either if it is unchanged, or the user has chosen to save or lose
         *  the changes), or will be rejected if the application will continue
         *  to run (user has cancelled the dialog, save operation failed).
         */
        function beforeQuitHandler() {

            // the deferred returned to the framework
            var deferred = $.Deferred();

            // don't call sendActions() when we are offline or in an error state
            if (self.isOffline() || internalError) {
                // rejected promise would keep application alive
                deferred.resolve();
                return deferred.promise();
            }

            // first, call the flush preparation handler passed to the constructor
            var promise = $.when(prepareFlushHandler.call(self, 'quit'));

            // send pending actions to the server
            promise = promise.then(function () {

                // maximum time for sendActions (in ms)
                var maxDelay = 5000;
                // the deferred returned by sendActions
                var sendPromise = sendActions();

                if (sendPromise.state() === 'pending') {

                    // prevent endless waiting for sendActions to complete
                    var sendActionsTimer = self.executeDelayed(function () {
                        if (sendPromise.state() === 'pending') {
                            deferred.resolve({ cause: 'timeout' });
                        }
                    }, maxDelay, 'EditApplication: beforeQuitHandler');

                    // This must not take more than specified in maxDelay
                    sendPromise.always(function () {
                        sendActionsTimer.abort();
                    });
                }

                return sendPromise;
            });

            // resolve the deferred if sending fails (rejected promise would keep application alive)
            promise.fail(function () { deferred.resolve(); });

            // show a dialog box, if there are still unsaved changes
            promise.done(function () {

                // still pending actions: ask user whether to close the application
                if (self.hasUnsavedChanges() && (clientId === oldEditClientId)) {

                    var queryPromise = docView.showQueryDialog(null, gt('This document contains unsaved changes. Do you really want to close?'));
                    queryPromise.done(deferred.resolve.bind(deferred)); // Answer 'Yes': close application
                    queryPromise.fail(deferred.reject.bind(deferred)); // Answer 'No': keep application alive

                } else {
                    // all actions saved, close application without dialog
                    deferred.resolve();
                }
            });

            return deferred.promise();
        }

        /**
         * Called before the application will be really closed.
         *
         * @param {String} reason
         *  The reason for quitting the application, as passed to the method
         *  BaseApplication.quit().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected if the application can
         *  be savely closed.
         */
        function quitHandler(reason) {

            var // the resulting deferred object
                def = null,
                // the application window
                win = self.getWindow(),
                // the file descriptor of this application (bug 34464: may be
                // null, e.g. when quitting during creation of a new document)
                file = self.getFileDescriptor(),
                // whether to show a progress bar (enabled after a short delay)
                showProgressBar = false,
                // recent file data
                recentFileData = null;

            // evaluates the result of the 'closedocument' call
            function checkCloseDocumentResult(data) {

                var // the resulting deferred object
                    def = $.Deferred(),
                    // the error code in the response
                    errorCode = new ErrorCode(data),
                    // additional data for the error data
                    additionalData = {},
                    // restored file id, if provided
                    restoredFileId = null,
                    // restored file name if provided
                    restoredFileName = null,
                    // restored fole id, if provided
                    restoredFolderId = null,
                    // restored error code
                    restoredErrorCode = null;

                // no error/warning, forward response data to piped callbacks
                if (!errorCode.isError() && !errorCode.isWarning()) {
                    return data;
                }

                // retrieve the restored file name / folder id in case there was a restore operation
                // during the flushDocument
                restoredFileId = Utils.getStringOption(data, ErrorContextData.RESTORED_FILEID, null);
                restoredFileName = Utils.getStringOption(data, ErrorContextData.RESTORED_FILENAME, null);
                restoredFolderId = Utils.getStringOption(data, ErrorContextData.RESTORED_FOLDERID, null);
                restoredErrorCode = Utils.getStringOption(data, ErrorContextData.RESTORED_ERRORCODE, 'BACKUPDOCUMENT_RESTORE_DOCUMENT_WRITTEN');
                if (_.isString(restoredFileId) && _.isString(restoredFileName) && _.isString(restoredFolderId)) {
                    if (restoredErrorCode === ErrorCode.CONSTANT_NO_ERROR) {
                        // no error on restore means that we have to show the restore success error
                        restoredErrorCode = new ErrorCode({ errorClass: ErrorCode.ERRORCLASS_ERROR, error: 'BACKUPDOCUMENT_RESTORE_DOCUMENT_WRITTEN' });
                    } else {
                        // create error code object with error class and code
                        restoredErrorCode = new ErrorCode({ errorClass: ErrorCode.ERRORCLASS_ERROR, error: restoredErrorCode });
                    }

                    additionalData[ErrorContextData.RESTORED_FILENAME] = restoredFileName;
                    DriveUtils.getPath(restoredFolderId).always(function (paths) {
                        additionalData[ErrorContextData.RESTORED_PATHNAME] = DriveUtils.preparePath(paths);
                        additionalData[ErrorContextData.ORIGINAL_FILEID] = self.getFileDescriptor().id;

                        // update file descriptor to enable 'reload' to load the restored document file
                        self.updateFileDescriptor({ id: restoredFileId, folder_id: restoredFolderId });
                        self.setInternalError(restoredErrorCode, ErrorContext.CLOSE, errorCode, { additionalData: additionalData });
                    });
                } else {
                    // all other cases show an error immediately
                    self.setInternalError(errorCode, ErrorContext.CLOSE, null, { additionalData: additionalData });
                }

                if (docModel && docView) {
                    // Bug 32989: Register a new quit handler that will reject the pending
                    // deferred object (the base application has deleted all old quit handlers
                    // from its internal list, including this handler currently running). This
                    // covers execution of app.immediateQuit(), e.g. when logging out while
                    // this document is still waiting for the user to click the Quit button.
                    // Simply reject the deferred but do not return it, otherwise the logout
                    // process will be aborted.
                    self.registerQuitHandler(function () { def.reject(); });

                    // The original app.quit() method has been modified to ensure that the
                    // quit code does not run multiple times, e.g. by double-clicking the Quit
                    // button. Here, the application will return to idle state, and the user
                    // has to click the Quit button again to really close the application. Thus,
                    // calling app.quit() has to reject the deferred object in order to ensure
                    // that the application really shuts down.
                    self.quit = (function (parentQuit) {
                        return function () {
                            def.reject();
                            return parentQuit.call(self);
                        };
                    }(self.quit));

                    // back to idle to allow final copy&paste
                    win.idle();
                    def.always(function () { win.busy(); });
                } else {
                    def.reject();
                }

                if (errorCode.getDescription()) {
                    Utils.log(errorCode.getDescription());
                }

                return def.promise();
            }

            // called in case the 'closedocument' call has been rejected
            function closeDocumentResultFailed(data) {
                return checkCloseDocumentResult(data);
            }

            // updating the progress bar during saving file in storage
            function progressHandler(progress) {
                if (showProgressBar) { docView.updateBusyProgress(progress); }
            }

            // trying to save the file in the local storage
            function saveFileInStorage(data) {

                var // a timer for measuring conversion time
                    startTime = null,
                    // the key used in the local storage
                    saveKey = null,
                    // the key used in the local storage to save the OSN, file version and storage version
                    saveKeyOSN = null, versionKey = null, storageVersionKey = null,
                    // the resulting deferred object
                    def = null,
                    // whether the document was modified
                    newVersion = (locallyModified || locallyRenamed || remotelyModified),
                    // the operation state number of this local client
                    clientOSN = docModel.getOperationStateNumber(),
                    // a counter for the different strings in the local storage
                    counter = 0,
                    // the maximum number of previous versions, that are checked to be deleted
                    maxOldVersions = 20,
                    // a collector for all extensions used in the local storage
                    allExtensions = ['_OSN', '_FILEVERSION', '_STORAGEVERSION'];

                // helper function to clear content in local storage
                function clearLocalStorage(options) {

                    var // whether old registry entries shall be removed
                        migration = Utils.getBooleanOption(options, 'migration', false);

                    // deleting all currently used registry entries
                    _.each(allExtensions, function (oneExtension) { localStorage.removeItem(saveKey + oneExtension); });

                    // Migration code: Removing old registry values, where the file version was part of the base key (these are no longer used)
                    if (migration) {
                        if (_.isFinite(data.syncInfo.fileVersion)) {
                            _.each(_.range((data.syncInfo.fileVersion > maxOldVersions) ? (data.syncInfo.fileVersion - maxOldVersions) : 0, data.syncInfo.fileVersion), function (number) {
                                _.each(allExtensions, function (oneExtension) { localStorage.removeItem(saveKey + '_' + number + oneExtension); });
                            });
                        } else {
                            localStorage.removeItem(saveKey + '_' + data.syncInfo.fileVersion);
                            localStorage.removeItem(saveKey + '_CURRENTVERSION');
                        }
                    }
                }

                // saveFileInLocalStorage can be overwritten with user setting in debug mode
                checkDebugUseLocalStorage();

                // saving file in local storage, using the file version returned from the server (if storage can be used)
                if ((newVersion || !loadedFromStorage) && data && data.syncInfo && window.Storage &&
                    localStorageApp && saveFileInLocalStorage && isLocalStorageSupported && _.isFunction(docModel.getFullModelDescription) &&
                    data.syncInfo.fileVersion && data.syncInfo.fileId && data.syncInfo.folderId && data.syncInfo['document-osn'] &&
                    _.isFinite(data.syncInfo['document-osn'], 10) && (clientOSN === parseInt(data.syncInfo['document-osn'], 10))
                ) {

                    def = $.Deferred();

                    // show the progress bar, if saving takes more than 1.5 seconds
                    self.executeDelayed(function () { showProgressBar = true; }, 1500, 'EditApplication: showing progress bar');

                    // create the string describing the document to save it in local storage
                    self.executeDelayed(function () {

                        def.notify(0.1);

                        // creating the converted document to be stored in local storage
                        startTime = _.now();

                        docModel.getFullModelDescription(def, 0.1, 0.9)
                        .always(function () {

                            // defining the base key in the local storage
                            saveKey = data.syncInfo.fileId + '_' + data.syncInfo.folderId;

                            // expanding this list, some layers might no longer be used
                            if (_.isFunction(docModel.getSupportedStorageExtensions)) {
                                _.each(docModel.getSupportedStorageExtensions(), function (oneExtension) {
                                    if (oneExtension && _.isString(oneExtension.extension)) { allExtensions.push(oneExtension.extension); }
                                });
                            }

                            // removing old content in the local storage (all possible layers need to be removed, even if they are no longer used)
                            clearLocalStorage({ migration: true });
                        })
                        .done(function (docStringList) {

                            // and finally writing all strings from each layer into the registry
                            Utils.iterateArray(docStringList, function (oneStringObject) {

                                var // the html string for one layer
                                    docString = oneStringObject.htmlString,
                                    // the save key for this one layer
                                    localSaveKey = saveKey + oneStringObject.extension;

                                Utils.info('quitHandler, created document string with size: ' + docString.length + '. Time for conversion: ' + (_.now() - startTime) + ' ms.');

                                try {
                                    if (counter === 0) {
                                        // defining keys used in the local storage
                                        saveKeyOSN = saveKey + '_OSN';
                                        versionKey = saveKey + '_FILEVERSION';
                                        storageVersionKey = saveKey + '_STORAGEVERSION';
                                        // saving the keys in the local storage
                                        localStorage.setItem(saveKeyOSN, data.syncInfo['document-osn']);
                                        localStorage.setItem(storageVersionKey, supportedStorageVersion);
                                        localStorage.setItem(versionKey, data.syncInfo.fileVersion);
                                    }
                                    // saving string in the cache, if there is sufficient space
                                    localStorage.setItem(localSaveKey, docString);

                                    // checking, if saving was successful
                                    if (localStorage.getItem(localSaveKey) && (localStorage.getItem(localSaveKey).length === docString.length)) {
                                        Utils.info('quitHandler, successfully saved document in local storage with key: ' + localSaveKey);
                                    } else {
                                        clearLocalStorage();
                                        Utils.info('quitHandler, failed to save document in local storage (1).');
                                        return Utils.BREAK;
                                    }

                                    if (counter === 0) {
                                        if ((localStorage.getItem(saveKeyOSN)) && (parseInt(localStorage.getItem(saveKeyOSN), 10) === data.syncInfo['document-osn'])) {
                                            Utils.info('quitHandler, successfully saved OSN for document in local storage with key: ' + saveKeyOSN + ' : ' + data.syncInfo['document-osn']);
                                        } else {
                                            clearLocalStorage();
                                            Utils.info('quitHandler, failed to save OSN for document in local storage (1).');
                                            return Utils.BREAK;
                                        }
                                    }

                                    counter++;

                                } catch (ex) {
                                    // do nothing, simply not using local storage
                                    clearLocalStorage();
                                    Utils.info('quitHandler: failed to save document and/or document OSN in local storage.');
                                    return Utils.BREAK;
                                }
                            });
                        })
                        .always(function () {
                            def.notify(1.0);
                            def.resolve(data);
                        });

                    }, undefined, 'EditApplication: getFullModelDescription');

                }

                return (def === null) ? data : def.promise();
            }

            function propagateChangedFile() {
                return DriveUtils.propagateChangeFile(file);
            }

            // window may be gone already, e.g. durung browser unload
            if (win) { win.busy(); }

            // whether the document is modified and needs a new version
            var modified = locallyModified || locallyRenamed || remotelyModified || movedAway;

            if (file && (modified || Config.LOG_PERFORMANCE_DATA) || (!((Utils.getStringOption(launchOptions, 'action') === 'new') || launchOptions.template) && (editClients.length <= 1))) {
                recentFileData = { file: _.extend({ last_opened: Date.now() }, file), app: self.getDocumentType() };
                // send performance data to server
                if (Config.LOG_PERFORMANCE_DATA) {
                    addGlobalPerformanceInfo();
                    recentFileData.performanceData = performanceLogger;
                }
            }

            // Don't call closeDocument when we are offline, in an error state or
            // the document has not been modified.
            if (file && rtConnection && !self.isOffline() && !internalError) {
                // notify server to save/close the document file
                def = rtConnection.closeDocument(recentFileData).then(checkCloseDocumentResult, closeDocumentResultFailed).then(saveFileInStorage).progress(progressHandler);
            } else {
                // error/offline: it's not safe to make remote calls therefore simply close rtConnection
                def = $.when();
            }

            // disconnect real-time connection after document is closed
            // ATTENTION: it be MUST ensured that rtConnection.destroy() is called
            // before model, view and controller are destroyed
            // rtConnection implementation relies on this fact, breaking this will
            // result in problems especially on shutdown/closing documents
            if (rtConnection) {
                def.always(function () {
                    rtConnection.destroy();
                    rtConnection = null;
                });
            }

            // bug 34464: do nothing else, if the new document has not been created yet (no file descriptor)
            if (file) {
                if (modified) {
                    // propagate changed document file to Files application (also in case of an error)
                    def = def.then(propagateChangedFile, propagateChangedFile);
                // bug 35797: do not delete an empty file, if user has requested to reload the document
                } else if ((reason !== 'reload') && ((Utils.getStringOption(launchOptions, 'action') === 'new') || launchOptions.template) && (editClients.length <= 1) && !launchOptions.keepFile) {
                    // Delete new document in Files application if no changes have been made at all or
                    // the file has been created from a template and not via edit as new.
                    // Shared documents should never be deleted and also a quit due to reload
                    // should preserve a new file.
                    def = def.then(function () { return DriveUtils.purgeFile(file); });
                }
            }

            return def;
        }

        /**
         * Adds the provided message data to the pending update queue and sets
         * the new editUser as the current editor for all pending updates as
         * long as the editor is not this client.
         * This prevents that pending message data lead to a wild changing
         * of editors, if the browser catches up processing the pending updates.
         * Therefore we manipulate the pending updates and set the current editor.
         * This must NOT be done, if the current editor is this client, otherwise
         * the edit rights are enabled although the osn is not up-to-date!!
         *
         * @param {Object} data
         *  The message data to be applied. Must be in the exact same format as
         *  also received from 'update' messages sent by the real-time
         *  framework.
         */
        function setAndUpdatePendingRemoteUpdates(data) {

            RTConnection.log('pendingUpdates: updates queued, pending = ' + (pendingRemoteUpdates.length + 1));

            if ((pendingRemoteUpdates.length > 1) && (clientId !== data.editUserId)) {
                RTConnection.log('pendingUpdates: pending updates must be changed to user: ' + data.editUserId + ' now.');
                var extendData = { editUser: data.editUser, editUserId: data.editUserId };
                _.each(pendingRemoteUpdates, function (entryData) {
                    _.extend(entryData, extendData);
                });
            }

            // add the latest message data to the pending update queue
            pendingRemoteUpdates.push(data);
        }

        /**
         * Adding one key-value pair to the performance logger object.
         *
         * @param {String} key
         *  The key for the performance logger object.
         *
         * @param {any} value
         *  The value for the performance logger object.
         */
        function addToPerformanceLogger(key, value) {
            performanceLogger[key] = value;
        }

        /**
         * Adding global performance information that will be sent
         * to the server.
         *
         */
        function addGlobalPerformanceInfo() {
            // logging some additional performance information
            _.each(self.getTimerLogCollector(), function (time, key) {
                addTimePerformance(key, time);
            });

            addTimePerformance('launchStart', self.getLaunchStartTime());
            addTimePerformance('triggerImportSuccess', importFinishedTime);
            addToPerformanceLogger('LocalStorage', loadedFromStorage);
            addToPerformanceLogger('FastEmptyLoad', fastEmpty);
            addToPerformanceLogger('user-agent', navigator.userAgent);
            addToPerformanceLogger('platform', navigator.platform);
            addToPerformanceLogger('user', ox.user);
            addToPerformanceLogger('version', ox.version);
            addToPerformanceLogger('server', ox.abs);
        }

        /**
         * Returns the index of an author from the document authors list. If
         * the author name is not known yet, it will be inserted into the list.
         *
         * @param {String} author
         *  The name of an author of this document.
         *
         * @returns {Number}
         *  The array index of the author in the document authors list.
         */
        function getAuthorIndex(author) {
            var index = _.indexOf(documentAuthorsList, author);
            if (index < 0) {
                index = documentAuthorsList.length;
                documentAuthorsList.push(author);
            }
            return index;
        }

        // protected methods --------------------------------------------------

        /**
         * Retrieves the number of pending remote updates.
         *
         * @return {Integer}
         *  The number of pending remote updates waiting for client processing.
         */
        this.getNumOfPendingRemoteUpdates = function () {
            return pendingRemoteUpdates.length;
        };

        /**
         * Leaves the busy mode while importing the document has not been
         * finished.
         *
         * @internal
         *  Must not be called after importing the document is finished.
         *
         * @param {Boolean} [globalFailure=false]
         *  If set to true, the application is in an internal error state, and
         *  the view will not be initialized further.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the GUI initialization handler
         *  has finished.
         */
        this.leaveBusyDuringImport = _.once(function (globalFailure) {

            // remember start time of preview mode
            previewStartTime = _.now();
            addTimePerformance('previewStart');
            updateState();

            // leave busy mode early
            return docView.showAfterImport(globalFailure);
        });

        // public methods -----------------------------------------------------

        /**
         * Creates a method that debounces multiple invocations, and that waits
         * until the 'action processing mode' of the document model is
         * currently active, i.e. the model currently applies operation actions
         * (needed e.g. in complex undo or redo operations). If the document
         * does not process any operations, the created method behaves like a
         * regular debounced method as created by the method
         * TimerMixin.createDebouncedMethod().
         *
         * @param {BaseObject} target
         *  The instance used as calling context for the passed callback
         *  function. Additionally, the lifetime of the target object will be
         *  taken into account. If the object will be destroyed before the
         *  operation actions have been applied, the deferred callback function
         *  will not be called anymore.
         *
         * @param {Function} directCallback
         *  The callback function to be called debounced, after the document
         *  model has applied the current operation actions.
         *
         * @param {Function} deferredCallback
         *
         * @returns {Function}
         *  The debounced method, bound to the lifetime of the passed target
         *  object. Intended to be used as (private or public) instance method
         *  of the passed target object!
         */
        this.createDebouncedActionsMethodFor = function (target, directCallback, deferredCallback, options) {

            var // whether the created method already waits for document operations
                waiting = false;

            // adding the string for asynchronous logging information to the options
            if (!options) { options = {}; }
            if (!options.infoString) { options.infoString = 'EditApplication: invokeDeferredCallback'; }

            // defers the invocation of the passed deferred callback if actions are being applied
            function invokeDeferredCallback() {

                // nothing to do, if the target object has been destroyed in the meantime
                if (target.destroyed) { return; }

                // invoke deferred callback directly, if no operations are applied currently
                if (!docModel.isProcessingActions()) {
                    deferredCallback.call(target);
                    return;
                }

                // nothing to do, if the method is already waiting for the operation promise
                if (waiting) { return; }
                waiting = true;

                // wait for the 'action processing mode' to finish
                target.waitForAny(docModel.getActionsPromise(), function () {
                    // handler will not be invoked, if target will be destroyed in the meantime
                    waiting = false;
                    deferredCallback.call(target);
                });
            }

            // create and return the special debounced method waiting for action processing mode
            return this.createDebouncedMethod(_.bind(directCallback, target), invokeDeferredCallback, options);
        };

        /**
         * Returns the file format of the edited document.
         */
        this.getFileFormat = function () {
            // TODO: remove the fallback to OOXML if the file format is known at construction time (no hasFileDescriptor() check anymore)
            var fileName = this.hasFileDescriptor() ? this.getFullFileName() :
                (launchOptions && launchOptions.templateFile) ? launchOptions.templateFile.filename : '';
            var fileFormat = fileName ? ExtensionRegistry.getFileFormat(fileName) : null;
            return fileFormat || 'ooxml';
        };

        /**
         * Returns whether the edited document is an Office-Open-XML file.
         */
        this.isOOXML = function () {
            return this.getFileFormat() === 'ooxml';
        };

        /**
         * Returns whether the edited document is an OpenDocument file.
         */
        this.isODF = function () {
            return this.getFileFormat() === 'odf';
        };

        /**
         * Returns whether the edited document is a template document file.
         */
        this.isTemplateFile = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isTemplate(this.getFullFileName());
        };

        /**
         * Returns whether the edited document may contain macro scripts.
         */
        this.isScriptable = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isScriptable(this.getFullFileName());
        };

        /**
         * Returns the own unique real-time client identifier.
         *
         * @returns {String}
         *  The own unique real-time client identifier.
         */
        this.getClientId = function () {
            return clientId;
        };

        /**
         * Returns the unique real-time identifier of the client with edit
         * rights.
         *
         * @returns {String}
         *  The unique real-time identifier of the client with edit rights.
         */
        this.getEditClientId = function () {
            return oldEditClientId;
        };

        /**
         * Returns the own client name, can be used by operations
         *
         * @returns {String}
         *  The own operation name.
         */
        this.getClientOperationName = function () {
            return clientOperationName;
        };

        /**
         * Returns all pending operations stored in the actions
         * buffer maintained by the application.
         *
         * @returns {Array}
         *  The pending operations queued by the application and
         *  which have not been sent to the real-time connection.
         */
        this.getPendingActions = function () {
            return actionsBuffer;
        };

        /**
         * Returns the array of all active clients currently editing this
         * document. Inactive clients (clients without activity for more than
         * 60 seconds) will not be included in this list.
         *
         * @returns {Array}
         *  The active clients. Each array element is a client descriptor
         *  object with the following properties:
         *  - {String} client.clientId
         *      The unique real-time session identifier of the client.
         *  - {Number} client.clientIndex
         *      A unique integral zero-based index for the client that will
         *      never change during the lifetime of this application.
         *  - {Number} client.userId
         *      The identifier of the user (teh same user will have the same
         *      identifier on different remote clients).
         *  - {String} client.userName
         *      The name of the user intended to be displayed in the UI.
         *  - {Boolean} client.remote
         *      Whether the entry represents a remote client (true), or the
         *      local client (false).
         *  - {Boolean} client.editor
         *      Whether the respective client has edit rights.
         *  - {Number} client.colorIndex
         *      A one-based index into the color scheme used to visualize the
         *      client (e.g. user list, or remote selections).
         *  - {Object} [client.userData]
         *      Optional data for the user that has been set by the respective
         *      client, e.g. the current selection.
         */
        this.getActiveClients = function () {
            return activeClients;
        };

        /**
         * Returns whether this document has been modified locally (true after
         * any operations where generated and applied by the own document model
         * and sent to the server).
         *
         * @returns {Boolean}
         *  Whether this document has been modified locally.
         */
        this.isLocallyModified = function () {
            return locallyModified;
        };

        /**
         * Returns whether we are in the process of switching edit rights.
         */
        this.isPrepareLosingEditRights = function () {
            return inPrepareLosingEditRights;
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout=20000]
         *  The delay time in milliseconds, after the Promise returned by this
         *  method will be rejected, if the query is still pending.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the answer of the server
         *  request, or rejected on error or timeout.
         */
        this.sendRealTimeQuery = function (action, timeout) {
            if (!_.isNumber(timeout)) { timeout = 20000; }
            return rtConnection.sendQuery(action, timeout);
        };

        /**
         * Sends an action POST request in a specific format to the server.
         * ATTENTION:
         * This method is bound to a working RT connection as the
         * processing on the server-side must be done in a RT environment.
         * Therefore this method uses the client-side RT framework to
         * provide necessary data via the used http-request.
         *
         * @param {String} action
         *  The action identifier, inserted as 'action' property into the
         *  request POST data.
         *
         * @param {Object} data
         *  The data object, inserted as 'requestdata' property into the
         *  request POST data.
         *
         * @param {Object} [options]
         *  Optional parameters. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The abortable promise representing the action request.
         */
        this.sendActionRequest = function (action, data, options) {
            options = _.extend({ method: 'POST' }, options); // default to POST
            // ATTENTION: The property "rtdata" is essential and MUST be sent
            // from the client. Currently the unique real-time id and session
            // must be part of the rtdata json object.
            return this.sendFileRequest(IO.FILTER_MODULE_NAME,
                { action: action, requestdata: JSON.stringify(data), rtdata: JSON.stringify({ rtid: rtConnection.getUuid(), session: ox.session }) },
                options);
        };

        /**
         * Sends a server query (an action request with the 'action' property
         * set to the value 'query') in a specific format to the server.
         *
         * @param {String} type
         *  The type of the query, inserted as 'type' property into the request
         *  POST data.
         *
         * @param {Object} data
         *  The data object, inserted as 'requestdata' property into the
         *  request POST data.
         *
         * @param {Object} [options]
         *  Optional parameters. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The abortable promise representing the query request.
         */
        this.sendQueryRequest = function (type, data, options) {
            return this.sendActionRequest('query', _.extend({}, data, { type: type }), options);
        };

        /**
         * Applies the passed application state, actions, and other application
         * data. Actions will be applied in a background loop to prevent
         * unresponsive browser and warning messages from the browser. If
         * actions from older 'update' messages are still being applied, the
         * new data passed to this method will be cached, and will be processed
         * after the background task for the old actions is finished. The
         * application will trigger a 'docs:update' event with the passed data,
         * after the actions have been applied.
         *
         * @param {Object} data
         *  The message data to be applied. Must be in the exact same format as
         *  also received from 'update' messages sent by the real-time
         *  framework.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the application
         *  data has been processed.
         */
        this.applyUpdateMessageData = function (data) {
            if (!_.isObject(data)) {
                Utils.error('EditApplication.applyUpdateMessageData(): invalid response (object expected):', data);
                this.setInternalError(ClientError.ERROR_WHILE_MODIFYING_DOCUMENT, ErrorContext.GENERAL);
                return $.Deferred().reject();
            }
            setAndUpdatePendingRemoteUpdates(data);
            return applyUpdateMessageData(data, { notify: true });
        };

        /**
         * Sends a log message via the real-time connection if logging is
         * enabled.
         *
         * @param {String} message
         *  The message to be logged on the server-side.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.sendLogMessage = Config.LOG_ERROR_DATA ? function (message) {
            if (rtConnection) { rtConnection.sendLogMessage(message); }
            return this;
        } : Utils.NOOP;

        /**
         * Renames the file currently edited by this application.
         *
         * @param {String} shortName
         *  The new short file name (without extension).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the file has been renamed
         *  successfully; or that will be rejected, if renaming the file has
         *  failed.
         */
        this.rename = function (shortName) {

            var
                file = this.getFileDescriptor(),
                // the original file name
                oldFileName = Utils.getStringOption(file, 'filename', this.getFullFileName()),
                oldSanitizedFilename = Utils.getStringOption(file, 'com.openexchange.file.sanitizedFilename', oldFileName),
                // the new file name (trim NPCs and spaces at beginning and end, replace embedded NPCs)
                newFileName = Utils.trimAndCleanString(shortName),
                // we fetch the currentfocus in IE because it could get lost
                currentFocus = Utils.getActiveElement();

            // return running request (should not happen anyway!)
            if (renamePromise) {
                Utils.warn('EditDocument.rename(): multiple calls, ignoring subsequent call');
                return renamePromise;
            }

            // empty name (after trimming)
            if (newFileName.length === 0) {
                return $.Deferred().reject();
            }

            // name does not change
            if (newFileName === this.getShortFileName()) {
                return $.when();
            }

            // Bug 27724: immediately update file descriptor
            newFileName += '.' + this.getFileExtension();
            this.updateFileDescriptor({ filename: newFileName });

            // set needed promise in case of asynchronous rename necessary
            renameAndReloadPromise = $.Deferred();
            // prepare the rename (makes sure that view settings are sent
            // before we start to rename the document). we also have to
            // make sure that pending requests are sent.
            renamePromise = $.when(prepareRenameHandler.call(self)).then(sendActions).then(function () {
                return self.sendActionRequest('renamedocument', { target_filename: newFileName });
            });

            // filter response by the 'error' property containing an error code
            renamePromise = renamePromise.then(function (data) {

                var // the result of the rename can be extract from the error propety
                    error = new ErrorCode(data);

                // check result as it can be possible that the final result is sent via a RT
                // notification
                if (error.isWarning() && error.getCodeAsConstant() === 'RENAMEDOCUMENT_SAVE_IN_PROGRESS_WARNING') {
                    // we have an asynchronous rename due to storage limitations, where we
                    // also need to reload the renamed document
                    return renameAndReloadPromise.promise();
                } else {
                    // forward result data on success, otherwise reject with error code
                    return (!error.isError()) ? data : $.Deferred().reject(error);
                }
            });

            // renaming succeeded: prevent deletion of new empty file, update file descriptor
            renamePromise.done(function (data) {
                locallyRenamed = true;
                self.updateFileDescriptor({
                    filename: Utils.getStringOption(data, 'fileName', oldFileName),
                    'com.openexchange.file.sanitizedFilename': Utils.getStringOption(data, 'com.openexchange.file.sanitizedFilename', oldSanitizedFilename)
                });
            });

            // renaming failed: show an appropriate warning message
            renamePromise.fail(function (response) {

                var // message data for docView.yell
                    messageData = null,
                    // error code
                    errorCode = ErrorCode.isErrorCode(response) ? response : new ErrorCode(response);

                if (response !== 'abort') {
                    errorCode.setErrorContext(ErrorContext.RENAME);
                    messageData = ErrorMessages.getMessageData(errorCode);
                    Utils.warn('EditApplication.rename() error: ' + errorCode.getErrorText());
                    docView.yell(messageData);
                }

                // back to old file name
                self.updateFileDescriptor({ filename: oldFileName });
            });

            // clean up after renaming has finished
            renamePromise.always(function () {
                renamePromise = null;

                //after renaming the file, the appsuite spinner fetches the focus
                //so we have to get it back
                if (_.browser.IE && currentFocus) { currentFocus.focus(); }
            });
            return renamePromise;
        };

        /**
         * Returns whether there are actions left that have not been sent
         * successfully to the server.
         *
         * @attention
         *  Do not rename this method. The OX core uses it to decide whether to
         *  show a warning before the browser refreshes or closes the page.
         *
         * @returns {Boolean}
         *  Whether the application currently caches unsaved actions.
         */
        this.hasUnsavedChanges = function () {
            return sendingActions || !!renamePromise || (actionsBuffer.length > 0) || _.isObject(pendingAction);
        };

        /**
         * Returns a promise that waits until all pending actions with their
         * operations have seen sent to the server.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all pending actions have been
         *  saved. If no actions are currently pending, the returned promise
         *  will be resolved already.
         */
        this.saveChanges = function () {
            return sendActions();
        };

        /**
         * Sets a temporary user-defined application state.
         *
         * @param {String|Null} newState
         *  The new user-defined application state; or null to return to the
         *  regular internal application state.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.setUserState = function (newState) {
            if (!this.isInternalError() && (userState !== newState)) {
                userState = newState;
                triggerChangeState();
            }
            return this;
        };

        /**
         * Returns the current application state.
         *
         * @returns {String}
         *  A string representing the application state. Will be one of the
         *  following values:
         *  - 'error': internal error, application locked permanently.
         *  - 'offline': network connection missing.
         *  - 'readonly': the document cannot be changed.
         *  - 'initial': the document has not been changed yet.
         *  - 'ready': all local changes have been sent to the server.
         *  - 'sending': local changes are currently being sent to the server.
         *  If a specific user defined state has been set using the method
         *  EditApplication.setUserState(), it will be returned instead, unless
         *  the application is in internal error state.
         */
        this.getState = function () {
            return (userState && !this.isInternalError()) ? userState : appState;
        };

        /**
         * Returns whether the application is currently in offline state, i.e.
         * whether the method EditApplication.getState() returns the state
         * 'offline'.
         *
         * @returns {Boolean}
         *  Whether the application is currently in offline state.
         */
        this.isOffline = function () {
            return appState === 'offline';
        };

        /**
         * Returns whether the application is in the internal error state, i.e.
         * whether the method EditApplication.getState() returns the state
         * 'error'.
         *
         * @returns {Boolean}
         *  Whether the application is in internal error state.
         */
        this.isInternalError = function () {
            return appState === 'error';
        };

        /**
         * Sets this application into the internal error/warning mode,
         * dependent on the error/warning code and shows the error message to
         * the user.
         *
         * @param {String|Object} errorCode
         *  An error code string constant, either from the client-side (see
         *  ClientError) or from the server-side.
         *
         * @param {String} context
         *  An error context string constant defined by ErrorContext.
         *
         * @param {String|Object} cause
         *  An error code string constant, either from the client-side (see
         *  ClientError) or from the server-side, which describe the root
         *  cause of an error. While errorCode describes the error to be
         *  shown, cause is the root cause of the error state. If this is null
         *  errorCode is used as root cause.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.showMessage=true]
         *      If set to false, the application won't show any message to the
         *      user. This is applicable, if a handler takes over the
         *      responsibility to show the error/warning message.
         *  @param {Object} [option.additionalData=null]
         *      If set, this object can be used to provide context dependent
         *      data to the error message (e.g. file name etc.).
         */
        this.setInternalError = function (errorCode, context, cause, options) {

            var // the error code object
                newError = ErrorCode.isErrorCode(errorCode) ? errorCode : new ErrorCode(errorCode),
                // the old error code object
                oldError = null,
                // message data
                messageData = null,
                // text for logging
                logError = null,
                // show warning
                showWarning = true,
                // additional data
                additionalData = Utils.getObjectOption(options, 'additionalData', {}),
                // the cause of the error, use errorCode as cause if not provided
                causeError = ErrorCode.isErrorCode(cause) ? cause : (_.isString(cause) ? new ErrorCode(cause) : newError);

            if (_.isString(context)) {
                newError.setErrorContext(context);
            }

            // check, if we have already a set error
            oldError = self.getErrorState();
            if (!_.isObject(oldError) || !oldError.isError()) {
                // old errors should not be overwritten
                self.setErrorState(newError);
            }

            logError = self.getErrorState();
            if (ErrorCode.isErrorCode(logError)) {
                Utils.error('Internal error: ' + logError.getErrorText());
            }

            // Set states only for true errors
            if (newError.isError()) {
                internalError = true;
                connected = false;
                locked = true;
                docModel.setEditMode(false);
            } else {
                // warnings should be shown once, check this using map
                showWarning = !_.has(alreadyShownWarning, newError.getCodeAsConstant());
                alreadyShownWarning[newError.getCodeAsConstant()] = true;
            }
            updateState();

            messageData = ErrorMessages.getMessageData(newError, _.extend({ causeError: causeError }, getInfoStateOptions(), additionalData));
            self.extendMessageData(messageData);

            updateFunctionStates(messageData);
            if (Utils.getBooleanOption(options, 'showMessage', true) && showWarning) {
                docView.yell(messageData);
            }
        };

        /**
         * Sets the current application information state, which is used to
         * provide information to the user.
         *
         * @param {String} infoState
         *  a info state defined by the InfoState class.
         *
         * @param {Object} [options]
         *  A optional object, which contains additional properties for the
         *  information message. If null or undefined, the implementation
         *  provides a default options object, which will be used by the
         *  internal info state => information message mapping function.
         *  Optional parameters:
         *  @param {Boolean} [options.showMessage=true]
         *      If set to false, the application won't show any
         *      message to the user. This is applicable, if a
         *      handler takes over the responsibility to show
         *      the error/warning message.
         */
        this.setCurrentInfoState = function (infoState, options) {

            var // error message
                messageData = null,
                // show message option
                showMessage = Utils.getBooleanOption(options, 'showMessage', true);

            self.setInfoState(infoState);
            if (showMessage) {
                messageData = InfoMessages.getMessageData(infoState, _.extend({}, getInfoStateOptions(), options));
                self.extendMessageData(messageData);
                if (_.isObject(messageData)) {
                    docView.yell(messageData);
                }
            }
        };

        /**
         * Returns whether the application is locked for editing. Reasons for
         * the lock may be missing edit permissions for the document file, or
         * an unrecoverable internal application error.
         *
         * @returns {Boolean}
         *  Whether editing the document is locked.
         */
        this.isLocked = function () {
            return internalError || locked;
        };

        /**
         * Returns whether acquiring edit rights is currently possible (the
         * document must be in read-only mode, and no internal application
         * error has occurred).
         *
         * @returns {Boolean}
         *  Whether acquiring edit rights is currently possible.
         */
        this.isAcquireEditRightsEnabled = function () {
            return _.isObject(rtConnection) && !this.isLocked() && !this.isOffline() && !docModel.getEditMode();
        };

        /**
         * Returns whether acquiring edit rights is currently possible (the
         * document must be in read-only mode, no internal application
         * error has occurred, and no pending updates are available).
         *
         * @returns {Boolean}
         *  Whether acquiring edit rights is currently possible.
         */
        this.isAcquireEditRightsEnabledPendingUpdates  = function () {
            return this.isAcquireEditRightsEnabled() && this.getNumOfPendingRemoteUpdates() <= MAX_NUM_OF_PENDING_UPDATES_FOR_EDITRIGHTS;
        };

        /**
         * Tries to acquire edit rights for the document. Edit rights will be
         * updated automatically while synchronizing operations with the
         * server. Multiple calls of this function will be throttled with a
         * delay of 3 seconds (first call will be executed immediately).
         */
        this.acquireEditRights = _.throttle(function () {
            if (this.isAcquireEditRightsEnabled()) {
                rtConnection.acquireEditRights();
                triggerEditRightsEvent('acquire');
            }
        }, 3000);

        /**
         * Returns whether reload is currently enabled or not.
         *
         * @returns {Boolean}
         *  Whether reloaded is enabled or not.
         */
        this.isReloadEnabled = function () {
            return reloadEnabled;
        };

        /**
         * Returns whether the auto-save feature is currently enabled.
         *
         * @returns {Boolean}
         *  Whether the auto-save feature is currently enabled.
         */
        this.isAutoSaveEnabled = function () {
            // auto-save is always enabled, trying to toggle it results in an alert box (see below)
            return true;
        };

        /**
         * Enables or disables the auto-save feature.
         *
         * @param {Boolean} state
         *  Whether to enable or disable the auto-save feature.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.toggleAutoSave = function (/*state*/) {
            // auto-save is always enabled, trying to toggle it results in an alert box
            docView.yell({
                type: 'info',
                headline: gt('What is AutoSave?'),
                message: gt('In OX Documents all your changes are saved automatically through AutoSave. It is not necessary to save your work manually.')
            });
            return this;
        };

        /**
         * Saves the current document to a specified folder using the provided
         * file name (without extension). If saving the current document
         * to the new folder is successful the function automatically
         * loads the copied document, which restarts the application.
         *
         * @param {String} shortName
         *  The new short file name (without extension).
         *
         * @param {Number} targetFolderId
         *  The target folder id where the document or template should be saved
         *
         * @param {String} [type='file']
         *  The target file type. The following file types are supported:
         *  - 'file' (default): Save with the current extension.
         *  - 'template': Save as template file.
         *  - 'pdf': Export (rename, convert and save) as PDF into selected target folder.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the file has been saved
         *  successfully; or that will be rejected, if saving the file has
         *  failed.
         */
        this.saveDocumentAs = function (shortName, targetFolderId, type) {

            var // the resulting deferred object
                def = null,
                // the new file name (trim NPCs and spaces at beginning and end, replace embedded NPCs)
                newFileName = _.isString(shortName) ? Utils.trimAndCleanString(shortName) : '',
                // folder path for yell
                fullPathName = null;

            // empty name (after trimming)
            if (newFileName.length === 0) {
                return $.Deferred().reject();
            }

            // send the server request
            if (type === 'pdf') {           // DOCS-122 :: misinterpreted story - [https://jira.open-xchange.com/browse/DOCS-122] :: but hereby saved for good.
              //exportAsPdf: type === 'pdf'
                var fileDescriptor  = self.getFileDescriptor();

                def = DriveUtils.getFileModelFromDescriptor(fileDescriptor).then(function (fileModel) {
                    return ConverterUtils.sendConverterRequest(fileModel, {
                        documentformat: type,
                        saveas_filename: newFileName,
                        saveas_folder_id: targetFolderId
                    });
                });
            } else {                        // default.

                def = this.sendActionRequest('copydocument', {
                    file_id: self.getFileParameters().id,
                    target_filename: newFileName,
                    asTemplate: type === 'template',
                    target_folder_id: targetFolderId
                });
            }

            // filter response by the 'error' property containing an error code
            def = def.then(function (data) {

                var // error state of the request
                    error = new ErrorCode(data);

                // forward result data on success, otherwise reject with error code
                return (!error.isError()) ? data : $.Deferred().reject(error);
            });

            //file path for yelling
            def = def.then(function (data) {

                // force Drive to reload data of targetFolder
                DriveUtils.deleteFromCache(targetFolderId);

                return DriveUtils.getPath(targetFolderId).then(function (paths) {
                    fullPathName = DriveUtils.preparePath(paths);

                  //Utils.info('+++ EditApplication :: saveDocumentAs :: thenable : fullPathName, data : ', fullPathName, data);
                    return data;
                });
            });

            if (type === 'pdf') {           // DOCS-122 :: misinterpreted story - [https://jira.open-xchange.com/browse/DOCS-122] :: but hereby saved for good.

                // intercept data flow via a new deferred object
                // in order to support both special use cases, ...

                def = def.then(function (data) {
                    var
                        interceptor = $.Deferred();

                    if ('cause' in data) {

                        // ... an own error switch that passes its data
                        // into the generic deferred failure handling ...
                        interceptor.reject(data);

                    } else {
                        // ... and the success handling
                        // that notifies users about
                        // 'export as pdf' has succeeded.
                        interceptor.resolve(data);

                        var
                            fileName = Utils.getStringOption(data, 'filename', newFileName);

                        self.setCurrentInfoState(InfoState.INFO_DOC_SAVED_IN_FOLDER, { fullFileName: fileName, fullPathName: fullPathName, type: 'success', duration: 10000 });
                    }
                    return interceptor;
                });

            } else {                        // default.

                // renaming succeeded: prevent deletion of new empty file, update file descriptor
                def.then(function (data) {

                    var // target folder of the copied document
                        folderId = Utils.getStringOption(data, 'folder_id', null),
                        // new file name of the copied document
                        fileName = Utils.getStringOption(data, 'filename', newFileName);

                    docModel.setEditMode(false);

                    // Templates are always saved in the user directory. Notify users about this
                    if (type === 'template') {
                        self.setCurrentInfoState(InfoState.INFO_DOC_SAVED_AS_TEMPLATE, { fullFileName: fileName, fullPathName: fullPathName, type: 'success', duration: 10000 });
                    }

                    var newId = Utils.getStringOption(data, 'id', null);
                    var newFolder_id = targetFolderId || folderId;

                  //Utils.info('+++ EditApplication :: saveDocumentAs :: thenable : folderId, fileName, newId, newFolder_id : ', folderId, fileName, newId, newFolder_id);

                    return DriveUtils.getFile({ id: newId, folder_id: newFolder_id }).then(function (newFile) {

                        //no file ending
                        newFile.title =  Utils.getFileBaseName(newFile.filename);
                        self.updateFileDescriptor(newFile);

                        // Adapt launch options to prevent removing the file as it
                        // could be empty and new or a template file.
                        // See #32719 "Save as with an empty document: => Load Error"
                        launchOptions.action = 'load';
                        delete launchOptions.template;

                      //Utils.info('+++ EditApplication :: saveDocumentAs :: thenable : DriveUtils.getFile :: newFile : ', newFile);

                        // close the current document and restart with new one
                        self.reloadDocument();
                    });
                });
            }

            // copy the document failed: show an appropriate warning message
            def.fail(function (error) {

                var // the message data retrieved from getMessageData
                    messageData = null;

                // make sure we always have a valid error object
                error = ErrorCode.isErrorCode(error) ? error : new ErrorCode(ErrorCode.GENERAL_ERROR);
                error.setErrorContext(ErrorContext.SAVEAS);
                messageData = ErrorMessages.getMessageData(error, getInfoStateOptions());
                Utils.warn('EditApplication.saveDocumentAs(): Error code "' + error.getCodeAsConstant() + '"');

                // Use docView.yell to show message - this is only a temporary error and
                // cannot be set via setInternalError()
                docView.yell({ type: messageData.type, headline: messageData.headline, message: messageData.message });
            });

            return def.promise();
        };

        /**
         * Closes this application instance, and launches a new application
         * instance with the document that is currently edited.
         */
        this.reloadDocument = function () {

            var // the file descriptor of the current document
                file = this.getFileDescriptor(),
                // full filename
                fullFileName = this.getFullFileName(),
                // the edit application module identifier
                editModule = ExtensionRegistry.getEditModule(fullFileName);

            if (!editModule) {
                Utils.error('EditApplication.reloadDocument(): unknown application type');
                return this;
            }

            if (!ExtensionRegistry.isNative(fullFileName)) {
                Utils.error('EditApplication.reloadDocument(): unsupported document type');
                return this;
            }

            // Close this application, and wait for quit to finish. This is required
            // to prevent that the framework detects a running application for the
            // edit document and rejects launching the new application instance.
            return this.quit('reload').done(function () {
                // additional delay before launching, for safety
                _.delay(function () {
                    ox.launch(editModule + '/main', { action: 'load', file: file });
                }, 1000);
            });
        };

        /**
         * Tries to restore the document content using the backup-data stored on
         * the backend. The client is able to provide own actions, to be merged into
         * the backup-data/operation queue. If restore document is disabled, this
         * function just sets the error state and returns.
         *
         * @param {ErrorCode} errorCode
         *  The error code which occurred previous and is the reason to restore
         *  the document.
         *
         * @param {Object} [options]
         *  Additional options to control the behaviour of the function.
         *  @param {Array} [options.actions]
         *      The client operations to be stored with the restore document
         *      function. The restore document functions try to merge the operations
         *      to restore the document content to prevent data loss.
         *
         */
        this.restoreDocument = function (errorCode, options) {
            var // possible operations stored by the client or provided by the
                // backend.
                sentActionsPending = Utils.getArrayOption(options, 'actions', null),
                // file descriptor
                file = self.getFileDescriptor(),
                // actions that must be provided to the restore function
                actionsPending = null;

            /**
             * Tries to merge the actions retrieved from the real-time connection and
             * the actions retrieved from the action buffer of the application.
             *
             * @param {Object|Array} sentActions
             *  An actions object or array with operations objects from the
             *  real-time connection which are pending.
             *
             * @param {Object|Array} appActions
             *  An actions object or array with operations objects from the
             *  application instance stored in the actionsBuffer.
             *
             * @returns {Array}
             *  An array with operations or null if no operations are available
             *  to be applied for restore document.
             */
            function mergeActions(sentActions, appActions) {
                var // result actions object
                    actions = {},
                    // sent but not ack operations array
                    sentActionsArray = OperationUtils.getActionsArray(sentActions),
                    // pending application operations array
                    appActionsArray = OperationUtils.getActionsArray(appActions);

                if (_.isArray(sentActionsArray) && _.isArray(appActionsArray)) {
                    actions = sentActionsArray.concat(appActionsArray);
                } else {
                    // fall-back use the sent actions
                    actions = sentActionsArray;
                }

                return actions;
            }

            // in case we don't support restore document - just set the error
            // code and return
            if (!restoreDocEnabed) {
                self.setInternalError(errorCode);
                return;
            }

            // No need to call restoreDocument more than once, but there are
            // several notifications which should result in a restoreDocument.
            // Therefore we protecr us from being called more than once.
            if (docRestore) { return; }

            // Set doc restore to true - we don't need to reset it as the view
            // is read-only and won't change anymore.
            docRestore = true;

            if (_.isNull(sentActionsPending)) {
                // In case we don't get specific pending operations we try to
                // recover them using the real-time connection and application's
                // action buffer.
                actionsPending = mergeActions(rtConnection.getPendingActions(), self.getPendingActions());
            } else {
                actionsPending = sentActionsPending;
            }

            // set editor to internal error, read-only mode and lock it
            locked = true;
            docModel.setEditMode(false);
            updateState();
            internalError = true;

            var saveRestorePromise =
                    self.sendFileRequest(IO.FILTER_MODULE_NAME,
                            { action: 'saveaswithbackup',
                              file_id: file.id,
                              target_filename: file.filename,
                              target_folder_id: file.folder_id,
                              restore_id: docRestoreId,
                              operations: _.isNull(actionsPending) ? actionsPending : JSON.stringify(actionsPending) },
                            { method: 'POST' });

            saveRestorePromise.then(function (result) {
                var // error code from the saveaswithbackup
                    restoreErrorCode = new ErrorCode(result),
                    // deferred to determine path
                    deferred = $.Deferred(),
                    // file id of the restored file
                    fileId = Utils.getStringOption(result, ErrorContextData.RESTORED_FILEID, null),
                    // file name of the restored file
                    fileName = Utils.getStringOption(result, ErrorContextData.RESTORED_FILENAME, null),
                    // folder id containing the restored file
                    folderId = Utils.getStringOption(result, ErrorContextData.RESTORED_FOLDERID, null),
                    // the path name of the restored file
                    pathName = null,
                    // extended context data for error processing
                    additionalData = {};

                if (_.isObject(result) && _.isString(fileName) && _.isString(folderId)) {
                    DriveUtils.getPath(folderId).always(function (paths) {
                        pathName = DriveUtils.preparePath(paths);
                        deferred.resolve(pathName);
                    });
                } else {
                    deferred.resolve(null);
                }

                deferred.always(function (data) {
                    var // support reload
                        updateFileDescriptor = _.isString(data);

                    if (!restoreErrorCode.isError()) {
                        // no error -> provide the success restore error code
                        restoreErrorCode = new ErrorCode({ errorClass: ErrorCode.ERRORCLASS_ERROR, error: 'BACKUPDOCUMENT_RESTORE_DOCUMENT_WRITTEN' });
                    }

                    // set necessary context data for error processing/output
                    additionalData[ErrorContextData.RESTORED_FILENAME] = fileName;
                    additionalData[ErrorContextData.RESTORED_PATHNAME] = pathName;
                    additionalData[ErrorContextData.ORIGINAL_FILEID] = self.getFileDescriptor().id;
                    if (updateFileDescriptor) {
                        // to support to 'reload' a new / restored document file we have to
                        // update the stored file descriptor
                        self.updateFileDescriptor({ id: fileId, folder_id: folderId });
                    }
                    self.setInternalError(restoreErrorCode, errorCode.getErrorContext(), errorCode, { additionalData: additionalData });
                });
            }, function () {
                self.setInternalError(errorCode);
            });
        };

        /**
         * Rejects an attempt to edit the document (e.g. due to received
         * keyboard events in read-only mode). The application will show an
         * appropriate warning alert banner to the user, and will update the
         * internal state according to the cause of the rejection.
         *
         * @param {String} [cause='edit']
         *  The cause of the rejection. If omitted, a simple edit attempt has
         *  been rejected, without any further influence on the application
         *  state.
         *  'image':
         *    Inserting an image into the document has failed.
         *  'siri':
         *    A siri event has been detected and the application is
         *    set to interal error mode = true. The user has to reload the
         *    document to make changes.
         *  'android-ime':
         *    A ime notification has been detected and the application is set
         *    to internal error mode = true. The user has to reload the
         *    document to make changes.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.rejectEditAttempt = function (cause) {

            switch (cause) {
                case 'image':
                    docView.yell({ type: 'warning', message: gt('The image could not be inserted.') });
                    break;

                case 'formula':
                    docView.yell({ type: 'warning', message: gt('The formula could not be inserted.') });
                    break;

                case 'siri':
                    self.setInternalError(ClientError.ERROR_SIRI_NOT_SUPPORTED, ErrorContext.GENERAL);
                    break;

                case 'loadingInProgress':
                    self.setCurrentInfoState(InfoState.INFO_LOADING_IN_PROGRESS);
                    break;

                case 'android-ime':
                    self.setInternalError(ClientError.ERROR_ANDROID_IME_NOT_SUPPORTED_KEYBOARD, ErrorContext.GENERAL);
                    break;

                default:
                    if (self.isImportFinished()) {
                        showReadOnlyAlert();
                    } else {
                        self.setCurrentInfoState(InfoState.INFO_LOADING_IN_PROGRESS);
                    }
            }

            return this;
        };

        /**
         * Calls the realtime update user data function debounced.
         *
         * @param {Object} userData
         *  the user data, containing e.g. user selection etc.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.updateUserData = (function () {

            var // the cached user data from the last call of this method
                cachedData = null;

            // direct callback: store user data of last invocation
            function storeUserData(userData) {
                cachedData = userData;
                return this;
            }

            // deferred callback: send last cached user data to server
            function sendUserData() {
                // bug 32520: do not send anything when closing document during import
                if (rtConnection && !internalError && !self.isInQuit()) {
                    rtConnection.updateUserData('updateuserdata', cachedData);
                }
            }

            return self.createDebouncedMethod(storeUserData, sendUserData, { delay: 500, maxDelay: 1000, infoString: 'EditApplication: sendUserData' });
        }());

        /**
         * Returns the scheme color index to be used to render information for
         * an author from the document authors list. If the author name is not
         * known yet, it will be inserted into the list.
         *
         * @param {String} author
         *  The name of an author of this document.
         *
         * @returns {Number}
         *  The one-based scheme color index of the author in the document
         *  authors list.
         */
        this.getAuthorColorIndex = function (author) {
            return Utils.getSchemeColor(getAuthorIndex(author));
        };

        /**
         * Adds the passed names of authors to the document authors list, and
         * keeps the list entries unique.
         *
         * @param {Array<String>} authors
         *  Author names to be added to the document authors list.
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.addAuthors = function (authors) {
            // _.union() retains element order of documentAuthorsList
            documentAuthorsList = _.union(documentAuthorsList, authors);
            return this;
        };

        /**
         * Stops registering of operations during callback function.
         *
         * @param {Function} callback
         *  The callback function to be executed during blocking of operations distribution.
         */
        this.stopOperationDistribution = function (callback) {
            distribute = false;
            if (_.isFunction(callback)) {
                callback.call(self);
            }
            distribute = true;
        };

        /**
         * Returns whether the registration of operations inside 'operations:success'
         * is blocked.
         *
         * @returns {Boolean}
         *  Whether the registration of operations inside 'operations:success' is blocked.
         */
        this.isOperationsBlockActive = function () {
            return operationsBlock;
        };

        /**
         * Blocks registration of operations in 'operations:success' listener. This is especially
         * useful, if the user cancelled a long running process of asynchronous executed
         * operations.
         *
         * @param {Function} callback
         *  The callback function to be executed during blocking of operations.
         */
        this.enterBlockOperationsMode = function (callback) {
            operationsBlock = true;
            // emptying an already filled pending action object
            // -> this could be filled by a delete action before a paste action
            if (pendingAction && pendingAction.operations && pendingAction.operations.length > 0) {
                // marking the last operations in the list of locally executed operations
                self.trigger('operations:remove', pendingAction.operations.length);
                // empty the list of already locally executed operations
                pendingAction = createAction();
            }
            // calling the callback function
            if (_.isFunction(callback)) {
                callback.call(self);
            }
            operationsBlock = false;
        };

        /**
         * Whether the application supports multiple selections.
         *
         * @returns {Boolean}
         *  Returns whether this application supported multiple selection.
         */
        this.isMultiSelectionApplication = function () {
            return isMultiSelectionApp;
        };

        // debug --------------------------------------------------------------

        /**
         * Triggers the specified event at the realtime connection instance.
         *
         * @attention
         *  Only available for debug purposes.
         */
        this.debugTriggerRealtimeEvent = Config.DEBUG ? function (type) {
            rtConnection.trigger(type);
            return this;
        } : $.noop;

        /**
         * Sends a invalid operation to the server side. Depending on the
         * options different invalid operations can be created and sent
         * to the server side.
         *
         * @param {Object} options
         *  options data to control what kind of invalid operation is sent
         *  to the server side.
         *  @param {Object} options.operation
         *   An operation object to be sent to the server side.
         *  @param {Boolean} options.badOsn [optional]
         *   Use a bad osn for the operation.
         *
         * @attention
         *  Only available for debug purposes.
         */
        this.debugSendInvalidOperation = Config.DEBUG ? function (options) {
            var // bad osn option
                badOsn = Utils.getBooleanOption(options, 'badOsn', false),
                // operations array
                opsArray = [],
                // osn to be used for the operation
                osn = badOsn ? docModel.getOperationStateNumber() + 1000 : docModel.getOperationStateNumber(),
                // operation to be sent
                operation = Utils.getObjectOption(options, 'operation', null);

            if (operation) {
                operation = _.extend(operation, { osn: osn, opl: 1 });
                opsArray.push(operation);
                if (!badOsn) {
                    docModel.setOperationStateNumber(osn + 1);
                }
                rtConnection.sendActions([{ operations: opsArray }]);
            }
            return this;
        } : $.noop;

        /**
         * Quits the application with additional debug actions.
         *
         * @attention
         *  Only available for debug purposes.
         *
         * @param {String} type
         *  The debug action type to be performed. The following actions are
         *  supported:
         *  - 'unsaved': Shows the 'unsaved changes' dialog.
         *  - 'error': Simulates a server error while trying to shut down.
         */
        this.debugQuit = Config.DEBUG ? function (type) {
            switch (type) {
                case 'unsaved':
                    var originalMethod = self.hasUnsavedChanges;
                    this.hasUnsavedChanges = _.constant(true);
                    this.quit().fail(function () { self.hasUnsavedChanges = originalMethod; });
                    break;
                case 'error':
                    // simulate server error after 2 seconds
                    rtConnection.closeDocument = function () {
                        var def = $.Deferred();
                        _.delay(function () { def.reject(); }, 2000);
                        return def;
                    };
                    this.quit();
                    break;
            }
        } : $.noop;

        /**
         *
         * @returns {Object}
         *  returns any content for this name
         */
        this.getLaunchOption = function (name) {
            return Utils.getOption(launchOptions, name);
        };

        // initialization -----------------------------------------------------

        // initialization after construction
        this.onInit(function () {

            // get references to MVC instances after construction
            docModel = this.getModel();
            docView = this.getView();

            // store exact time stamp when import finishes (regardless of the result)
            this.waitForImport(function () {
                importFinishedTime = _.now();
            });

            // calculate initial application state
            updateState();

            // update application state when model changes read-only state
            // (this includes all kinds of internal errors)
            docModel.on('readonly', updateState);

            // check whether editing is globally disabled for this application type
            if (!ExtensionRegistry.supportsEditMode(self.getDocumentType())) {
                locked = true;
                docModel.setEditMode(false);
                // Bug 44262
                showReadOnlyAlert(gt('Support starts with Chrome %1$d, Firefox %2$d, IE %3$d, and Safari %4$d.', 49, 43, 11, 8), -1);
            } else if (!(self.attributes && self.attributes.name === 'io.ox/office/spreadsheet') && _.browser.MacOS && _.browser.Chrome && !Utils.supportedChromeVersionOnMacForText()) {
                locked = true;
                docModel.setEditMode(false);
                // Bug 48777 & Bug 48784
                showReadOnlyAlert(gt('Support starts with Chrome %1$d, Firefox %2$d, IE %3$d, and Safari %4$d.', 53, 43, 11, 8), -1);
            }

            if (localStorageApp && saveFileInLocalStorage) {
                Utils.info('EditApplication.onInit(): saving document to local storage is active');
                if (isLocalStorageSupported) {
                    Utils.info('EditApplication.onInit(): current browser supports local storage');
                } else {
                    Utils.warn('EditApplication.onInit(): current browser does not support local storage');
                }
            }
        }, this);

        // set quit handlers
        this.registerBeforeQuitHandler(beforeQuitHandler);
        this.registerQuitHandler(quitHandler);

        // Selenium logging: notify start of import process
        this.waitForImportStart(function () {
            Utils.logSelenium('app/loading', 'pending');
        });

        // Selenium logging: notify start of preview phase during import
        this.once('docs:state:preview', function () {
            Utils.logSelenium('app/preview', 'pending');
        });

        // Selenium logging: notify end of import/preview
        this.waitForImport(function () {
            Utils.logSelenium('app/preview', 'resolved');
            Utils.logSelenium('app/loading', 'resolved');
        });

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = initOptions = self = docModel = docView = null;
            rtConnection = actionsBuffer = pendingAction = cacheBuffer = null;
            preProcessHandler = postProcessHandler = postProcessHandlerStorage = null;
            fastEmptyLoadHandler = previewHandler = importFailedHandler = null;
            prepareFlushHandler = prepareLoseEditRightsHandler = optimizeOperationsHandler = mergeCachedOperationsHandler = operationFilter = null;
            renamePromise = clientIndexes = editClients = activeClients = null;
            pendingRemoteUpdates = performanceLogger = null;
        });

    } }); // class EditApplication

    // exports ================================================================

    return EditApplication;

});
