/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/drawinglayer/view/drawingframe', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/lib/canvasjs.min',
    'io.ox/office/editframework/utils/border',
    'less!io.ox/office/drawinglayer/view/drawingstyle'
], function (Utils, Forms, IO, DateUtils, LocaleData, Canvas, AttributeUtils, Labels, CanvasJS, Border) {

    'use strict';

    var // the CSS class name of the drawing content node
        CONTENT_CLASS = 'content',

        // the CSS class used to crop images
        CROPPING_CLASS = 'cropping-frame',

        // the CSS class name of the selection root node
        SELECTION_CLASS = 'selection',

        // the CSS class name for active tracking mode
        TRACKING_CLASS = 'tracking-active',

        CULTURE_INFO = null,

        PRESET_GEOMETRIES = null;

    // private global functions ===============================================

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    function getSelectionNode(drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    }

    /**
     * Calculates the offset and size of the bitmap in an image frame for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} frameSize
     *  Visible size (with or height) of the drawing frame containing the image
     *  bitmap, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top).
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom).
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap node (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(app, frameSize, leadingCrop, trailingCrop) {

        // ODF: absolute cropping length (in 1/100mm), OOXML: percentage
        if (app.isODF()) {
            leadingCrop /= frameSize;
            trailingCrop /= frameSize;
        } else {
            leadingCrop /= 100;
            trailingCrop /= 100;
        }

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop away more than 90% of the bitmap
        if (totalCrop > 0.9) {
            leadingCrop *= (0.9 / totalCrop);
            trailingCrop *= (0.9 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = frameSize / (1 - leadingCrop - trailingCrop);
        offset = size * leadingCrop;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    /**
     * Returns the path rendering mode for the specified flags, as expected by
     * the rendering methods of a canvas context.
     *
     * @param {Boolean} hasFill
     *  Whether the shape area will be filled.
     *
     * @param {Boolean} hasLine
     *  Whether the shape outline will be rendered.
     *
     * @returns {String|Null}
     *  The path rendering mode as string, if either of the passed flags is
     *  true, otherwise null to indicate that nothing needs to be drawn.
     */
    function getRenderMode(hasFill, hasLine) {
        return hasFill ? (hasLine ? 'all' : 'fill') : (hasLine ? 'stroke' : null);
    }

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = '.' + DrawingFrame.NODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing the content nodes in a drawing frame.
     *
     * @constant
     */
    DrawingFrame.CONTENT_NODE_SELECTOR = '.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes as place holder.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_CLASS = 'placeholder';

    /**
     * A jQuery selector that matches nodes representing a place holder element
     * of type 'div'.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_SELECTOR = 'div.' + DrawingFrame.PLACEHOLDER_CLASS;

    /**
     * A jQuery selector that matches content nodes of group drawing objects.
     *
     * @constant
     */
    DrawingFrame.GROUPCONTENT_SELECTOR = DrawingFrame.NODE_SELECTOR + '[data-type="group"]>.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_CLASS = 'autoresizeheight';

    /**
     * A jQuery selector that matches nodes representing a content node inside a
     * test frame that automatically resizes its height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZEHEIGHT_CLASS;

    DrawingFrame.AUTORESIZETEXT_CLASS = 'autoresizetext';

    DrawingFrame.AUTORESIZETEXT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZETEXT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE = 'verticalalign';

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_CLASS = 'odftextframe';

    /**
     * A jQuery selector that matches nodes representing a text frame node that
     * are 'classical' text frames in odf format.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_SELECTOR = 'div.' + DrawingFrame.ODFTEXTFRAME_CLASS;

    /**
     * The CSS class used to mark empty text frames.
     *
     * @constant
     */
    DrawingFrame.EMPTYTEXTFRAME_CLASS = 'emptytextframe';

    /**
     * The name of the jQuery data attribute that stores the unique node
     * identifier of a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_UID = 'uid';

    /**
     * The name of the jQuery data attribute that stores the drawing model
     * instance for a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_MODEL = 'model';

    /**
     * A jQuery selector that matches elements representing a text frame element
     * inside a drawing frame of type shape.
     */
    DrawingFrame.TEXTFRAME_NODE_SELECTOR = 'div.textframe';

    /**
     * The CSS class used to mark text frame content nodes inside a
     * drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS = 'textframecontent';

    /**
     * A jQuery selector that matches nodes representing a text frame content
     * element inside a drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR = 'div.' + DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS;

    /**
     * The CSS class used to mark drawing nodes inside a group container.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_CLASS = 'grouped';

    /**
     * A jQuery selector that matches nodes representing a grouped element.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_SELECTOR = '.' + DrawingFrame.GROUPED_NODE_CLASS;

    /**
     * The CSS class used to mark canvas nodes of a drawing.
     *
     * @constant
     */
    DrawingFrame.CANVAS_CLASS = 'canvasgeometry';

    /**
     * A jQuery selector that matches nodes representing a canvas element.
     *
     * @constant
     */
    DrawingFrame.CANVAS_NODE_SELECTOR = 'canvas.' + DrawingFrame.CANVAS_CLASS;

    /**
     * The CSS class used to mark unsupported drawing nodes.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_CLASS = 'unsupported';

    /**
     * A jQuery selector that matches nodes representing unsupported drawings.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_SELECTOR = '.' + DrawingFrame.UNSUPPORTED_CLASS;

    /**
     * A class name that matches elements representing a rotated drawing.
     */
    DrawingFrame.ROTATED_DRAWING_CLASSNAME = 'rotated-drawing';

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node without specific contents.
     *
     * @param {String} type
     *  The type of the drawing.
     *
     * @param {DrawingModel|String} model
     *  The drawing model the new drawing frame will be based on. Alternatively
     *  (for applications not supporting drawing models), this value can be a
     *  string with the type of the drawing object.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (model) {

        var // the drawing type identifier
            type = _.isObject(model) ? model.getType() : model,
            // whether the drawing object is a group
            isGroup = type === 'group',
            // the content node
            contentNode = $('<div class="' + CONTENT_CLASS + '">'),
            // the resulting drawing frame node
            drawingFrame = $('<div class="' + DrawingFrame.NODE_CLASS + '" data-type="' + type + '" contenteditable="' + (isGroup && !_.browser.IE && !_.browser.Firefox) + '">');

        // add data attributes, event handlers, and the content node
        drawingFrame
            .data(DrawingFrame.DATA_UID, 'frame' + _.uniqueId())
            .data(DrawingFrame.DATA_MODEL, _.isObject(model) ? model : null)
            .on('dragstart', false)
            .append(contentNode);

        return drawingFrame;
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().attr('data-type');
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame.
     */
    DrawingFrame.isDrawingContentNode = function (node) {
        return $(node).is('.' + CONTENT_CLASS);
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame
     * that has no content.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame that has
     *  no content.
     */
    DrawingFrame.isEmptyDrawingContentNode = function (node) {
        return DrawingFrame.isDrawingContentNode(node) && $(node).children().not(DrawingFrame.CANVAS_NODE_SELECTOR).length === 0;
    };

    /**
     * Returns whether the passed node is a placeholder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a place holder node (of a drawing frame).
     */
    DrawingFrame.isPlaceHolderNode = function (node) {
        return $(node).is(DrawingFrame.PLACEHOLDER_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape'.
     */
    DrawingFrame.isShapeDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'shape';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape' AND
     * that is used as text frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape' and it is used
     *  as text frame.
     */
    DrawingFrame.isTextFrameShapeDrawingFrame = function (node) {
        return DrawingFrame.isShapeDrawingFrame(node) && DrawingFrame.getTextFrameNode(node).length > 0;
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'group'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'group'.
     */
    DrawingFrame.isGroupDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'group';
    };

    /**
     * Returns whether the passed node is a drawing frame that is grouped inside a
     * group.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a grouped drawing frame.
     */
    DrawingFrame.isGroupedDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && $(node).is(DrawingFrame.GROUPED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that can be moved only
     * in the region near the border. In its center, it can contain text frames.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that cannot be moved by clicking
     *  in its center.
     */
    DrawingFrame.isOnlyBorderMoveableDrawing = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) || DrawingFrame.isGroupDrawingFrame(node);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is NOT marked as
     * 'classical' odf text frame, so that there is only reduced text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is NOT marked as
     *  'classical' odf text frame, so that there is only reduced text functionality
     *  available.
     */
    DrawingFrame.isReducedOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !$(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is marked as
     * 'classical' odf text frame, so that there is no text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is marked as
     *  'classical' odf text frame, so that there is no reduced text functionality
     *  available.
     */
    DrawingFrame.isFullOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && $(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node that's rotated by some angle.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node that has rotation applied.
     */
    DrawingFrame.isRotatedDrawingNode = function (node) {
        return node && $(node).hasClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME);
    };

    /**
     * Returns whether the passed node contains table(s).
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node contains table.
     */
    DrawingFrame.nodeContainsTable = function (node) {
        return node && $(node).find('table').length > 0;
    };

    /**
     * Returns the number of drawing frames inside a specified drawing group. Only
     * the direct children are counted, not the children in sub groups.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Number|Null}
     *  The number of drawing frames inside a specified drawing group
     */
    DrawingFrame.getGroupDrawingCount = function (node) {
        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }
        return $(node).children().first().children().length;
    };

    /**
     * Returns the child drawing frame of a drawing group at a specified
     * position. Or null, if it cannot be determined.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Number} [number]
     *  The zero based number of the child to be returned.
     *
     * @returns {Node|Null}
     *  The child drawing frame of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getGroupDrawingChildren = function (node, number) {
        if (!DrawingFrame.isGroupDrawingFrame(node) || !_.isNumber(number)) { return null; }
        return $(node).children().first()[0].childNodes[number];
    };

    /**
     * Returns all drawing frames inside a specified drawing group. This does
     * not only contain direct children, but also the children of further groups
     * inside the group. If the option 'onlyDirectChildren' is set to true,
     * only the direct children are returned.
     * Returns null, if the specified node is not a group drawing node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Object} [options]
     *  A map with options to control the selection. The following options are
     *  supported:
     *  @param {Boolean} [options.onlyDirectChildren=false]
     *      If set to true, only those drawing nodes are returned, that are
     *      direct children of the group.
     *
     * @returns {Node|Null}
     *  The child drawing frames of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getAllGroupDrawingChildren = function (node, options) {

        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }

        if (Utils.getBooleanOption(options, 'onlyDirectChildren', false)) {
            return $(node).children().children(DrawingFrame.NODE_SELECTOR);
        } else {
            return $(node).find(DrawingFrame.NODE_SELECTOR);
        }
    };

    /**
     * Returns the closest drawing node of a given node (for example a canvas.border node).
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getDrawingNode = function (node) {

        var // the group content node is the direct child of the group node itself
            drawingNode = $(node).closest(DrawingFrame.NODE_SELECTOR);

        return drawingNode ? drawingNode : null;
    };

    /**
     * Returns the closest drawing node of a given node that is a text frame shape
     * drawing node. If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The text frame drawing node, or null, if it cannot be determined.
     */
    DrawingFrame.getClosestTextFrameDrawingNode = function (node) {

        var // the closest drawing node that is ancestor of the specified node
            drawingNode = DrawingFrame.getDrawingNode(node);

        if (!drawingNode) { return null; }

        return DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) ? drawingNode : null;
    };

    /**
     * Returns the closest drawing group node of a drawing node that is grouped.
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The grouped drawing node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to determine the group node. The following options are
     *  supported:
     *  @param {Boolean} [options.farthest=false]
     *      If set to true, that group node is returned, that is the highest in
     *      the dom. This is necessary, because groups can be included into groups.
     *      Otherwise the closest group node is returned.
     *  @param {Node} [options.rootNode]
     *      If the option 'farthest' is set to true, the root node must be determined,
     *      until which the ancestors will be searched.
     *
     * @returns {jQuery|Null}
     *  The closest or farthest drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getGroupNode = function (node, options) {

        var // the group content node is the direct child of the group node itself
            groupContentNode = null,
            // the root node, until which the ancestors are searched. This is only
            // used, if the option 'farthest' is set to true
            rootNode = null;

        if (Utils.getBooleanOption(options, 'farthest', false)) {
            rootNode = Utils.getOption(options, 'rootNode');
            if (rootNode) { groupContentNode = $(Utils.findFarthest(rootNode, node, DrawingFrame.GROUPCONTENT_SELECTOR)); }
        } else {
            groupContentNode = $(node).closest(DrawingFrame.GROUPCONTENT_SELECTOR);
        }

        return (groupContentNode && groupContentNode.length) > 0 ? groupContentNode.parent() : null;
    };

    /**
     * Returns the container node of a drawing frame, that contains all
     * top-level content nodes (paragraphs and tables).
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The drawing frame DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed table cell that contains all
     *  top-level content nodes (paragraphs and tables).
     */
    DrawingFrame.getTextFrameNode = function (drawingNode) {
        return $(drawingNode).find('> * > ' + DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframe element.
     */
    DrawingFrame.isTextFrameNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame content <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframecontent element.
     */
    DrawingFrame.isTextFrameContentNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * automatically resizes its height. In this case the content node contains a
     * specific class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an automatically resizing text frame drawing node.
     */
    DrawingFrame.isAutoResizeHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autoresizeheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    DrawingFrame.isAutoTextHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autortextheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZETEXT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * has a fixed height.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text frame drawing node with fixed height.
     */
    DrawingFrame.isFixedHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has NOT the class 'autoresizeheight'
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames as container for additional drawings.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div groupcontent element.
     */
    DrawingFrame.isGroupContentNode = function (node) {
        return $(node).is(DrawingFrame.GROUPCONTENT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a unsupported drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a unsupported drawing element.
     */
    DrawingFrame.isUnsupportedDrawing = function (drawingFrame) {
        return $(drawingFrame).hasClass(DrawingFrame.UNSUPPORTED_CLASS);
    };

    /**
     * Returns the model of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Object|Null}
     *  The model object of the specified drawing frame, as specified while the
     *  frame has been created.
     */
    DrawingFrame.getModel = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_MODEL);
    };

    /**
     * Returns the unique identifier of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The unique identifier of the specified drawing frame, as specified
     *  while the frame has been created.
     */
    DrawingFrame.getUid = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_UID);
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + CONTENT_CLASS);
    };

    /**
     * Returns the Canvas node (canvas) of the specified drawing frame
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The canvas node of the specified drawing frame.
     */
    DrawingFrame.getCanvasNode = function (drawingFrame) {
        return $(drawingFrame).find(DrawingFrame.CANVAS_NODE_SELECTOR);
    };

    /**
     * Returns whether the specified node is a canvas node (canvas)
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  Whether the specified node is a canvas node.
     */
    DrawingFrame.isCanvasNode = function (node) {
        return $(node).is(DrawingFrame.CANVAS_NODE_SELECTOR);
    };

    /**
     * Clears and returns the root content node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The empty root content node of the specified drawing frame.
     */
    DrawingFrame.getAndClearContentNode = function (drawingFrame) {
        return DrawingFrame.getContentNode(drawingFrame).empty().removeClass(DrawingFrame.PLACEHOLDER_CLASS).attr('style', '');
    };

    /**
     * Increasing the height of a group drawing frame, that contains text frames
     * with auto resize height functionality. Those text frames expand automatically
     * the height of the drawing, if required.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The text frame drawing node with auto resize height functionality.
     */
    DrawingFrame.updateDrawingGroupHeight = function (drawingNode) {

        // comparing lower border of auto resize text frame node with the group

        var // the offset of the bottom border in pixel
            drawingBottom = 0,
            // an optional group node surrounding the specified text frame drawing node
            groupNode = null,
            // the offset of the group border in pixel
            groupBottom = 0,
            // the new height of the group drawing node
            newHeight = 0,
            // whether the group iteration need to be continued
            doContinue = true;

        // helper function to find the pixel positions of all bottom borders inside a group
        function getLowestChildDrawingBottom(node) {

            var // the lowest bottom pixel position of all child drawings inside the specified node
                lowestBottom = null;

            // iterating over all children of the specified (group) node
            _.each(node.children(), function (oneDrawing) {

                var // the bottom pixel position of one child node
                    localBottom = Utils.round($(oneDrawing).offset().top + $(oneDrawing).height(), 1);

                // the larger the value, the deeper the drawing
                if (lowestBottom === null || localBottom > lowestBottom) { lowestBottom = localBottom; }
            });

            return lowestBottom;
        }

        // iterating over all groups
        while (doContinue && DrawingFrame.isGroupedDrawingFrame(drawingNode)) {

            // drawingBottom = $(drawingNode).offset().top + $(drawingNode).height();
            groupNode = DrawingFrame.getGroupNode(drawingNode);
            groupBottom = groupNode ? Utils.round(groupNode.offset().top + groupNode.height(), 1) : null;

            drawingBottom = getLowestChildDrawingBottom($(drawingNode).parent());

            // comparing the bottom borders of group and specified text frame
            if (_.isNumber(groupBottom) && _.isNumber(drawingBottom) && (groupBottom !== drawingBottom)) {
                // increasing or decreasing the height of the group without operation
                newHeight = Utils.round(groupNode.height() + (drawingBottom - groupBottom), 1);
                groupNode.height(newHeight);
                drawingNode = groupNode;
            } else {
                doContinue = false;
            }

        }
    };

    /**
     * Prepares a specified drawing frame for text insertion. This requires
     * that the content node inside drawing frame is cleared and then a new
     * text frame child is appended. This new text frame element is returned
     * and can be used as parent for paragraphs, tables, ... .
     * Only drawingFrames of type 'shape' allow this conversion.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The new text frame 'div' element, that can be used as container for
     *  the text content.
     */
    DrawingFrame.prepareDrawingFrameForTextInsertion = function (drawingFrame) {

        // only drawing frames of type 'shape' can be converted
        if (!DrawingFrame.isShapeDrawingFrame(drawingFrame)) { return $(); }

        var // the content node of the drawing
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the text frame node inside the content node, parent of paragraphs and tables
            // -> required for setting cursor type, distance to content node and click event
            textFrameNode = $('<div>').addClass('textframe').attr('contenteditable', true);

        // making content node visible by assigning a border via class 'textframecontent'
        contentNode.addClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS).append(textFrameNode);
        // in IE the drawing node itself must have contenteditable 'false', otherwise the
        // grabbers will be visible. Other browsers have to use the value 'true'.
        if (!_.browser.IE && !_.browser.Firefox) { $(drawingFrame).attr('contenteditable', true); }

        // returning the new created text frame node
        return textFrameNode;
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the empty content node
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = Math.max(0, drawingFrame.outerWidth() - 2),
            innerHeight = Math.max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(Math.min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(Math.min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(Math.min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .addClass(DrawingFrame.PLACEHOLDER_CLASS)
            .css({
                padding: Math.floor(verticalPadding) + 'px ' + Math.floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div>')
                    .addClass('abs background-icon')
                    .css('line-height', innerHeight + 'px')
                    .append(Forms.createIconMarkup(Labels.getDrawingTypeIcon(type), { style: 'font-size:' + pictureIconSize + 'px' })),
                $('<p>').text(_.noI18n(name) || Labels.getDrawingTypeLabel(type))
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(_.noI18n(description)));
        }
    };

    // formatting -------------------------------------------------------------

    /**
     * Updates the CSS formatting of the passed drawing frame, according to the
     * passed generic formatting attributes.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Object} mergedAttributes
     *  The drawing attribute set (a map of attribute maps as name/value pairs,
     *  keyed by the 'drawing' attribute family), containing the effective
     *  attribute values merged from style sheets and explicit attributes.
     */
    DrawingFrame.updateFormatting = function (app, drawingFrame, mergedAttributes) {

        var // the document model
            docModel = app.getModel(),
            // the drawing attributes of the passed attribute set
            drawingAttributes = mergedAttributes.drawing,
            // value of the 'transform' CSS attribute, e.g. for flipped drawings
            transformations = '',
            // type of the drawing object: 'image', ...
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the content node inside the drawing frame
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the replacement data for unsupported drawings
            replacementData = Utils.getStringOption(drawingAttributes, 'replacementData', ''),
            // rendering result (false to fall-back to placeholder graphic)
            rendered = false;

        // first remove the class which marks this drawing as unsupported
        drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);

        // helper function to get all node attributes (TODO: This needs to be valid for all applications)
        function getAllDrawingAttributes(node, family) {

            var // the drawing style collection
                styleCollection = docModel.getStyleCollection('drawing'),
                // the element specific attributes
                elementAttributes = styleCollection ? styleCollection.getElementAttributes(node) : null,
                // the family specific attributes of the element
                familyAttributes = elementAttributes ? elementAttributes[family] : null;

            return familyAttributes;
        }

        // creates an image node in the content node with the specified source URL
        function createImageNodeFromUrl(srcUrl) {
            return app.createImageNode(srcUrl, { timeout: 15000 }).done(function (imgNode) {
                var cropNode = $('<div class="' + CROPPING_CLASS + '">').append(imgNode);
                DrawingFrame.getAndClearContentNode(drawingFrame).append(cropNode);
            });
        }

        // creates an image node or an SVG image according to the passed image data
        function createImageNodeFromData(imageData) {
            if (/^data:/.test(imageData)) {
                return createImageNodeFromUrl(imageData);
            }
            if (/^<svg/.test(imageData)) {
                DrawingFrame.getAndClearContentNode(drawingFrame)[0].innerHTML = imageData;
                return $.when();
            }
            Utils.warn('DrawingFrame.createImageNodeFromData() - unsupported source data for the image object');
            return $.Deferred().reject();
        }

        // updates all formatting for unsupported objects
        function updateUnsupportedFormatting() {
            if (replacementData.length === 0) {
                // insert default replacement nodes, if no replacement data is available
                DrawingFrame.insertReplacementNodes(drawingFrame, drawingAttributes.name, drawingAttributes.description);
            } else if (!drawingFrame.data('has-replacement')) {
                // replacement data MUST NOT change at runtime
                createImageNodeFromData(replacementData);
                drawingFrame.data('has-replacement', true);
            }
            // add class to mark this drawing as a unsupported drawing.
            drawingFrame.addClass(DrawingFrame.UNSUPPORTED_CLASS);
        }

        // updating the size attributes at an image inside a drawing (also supported inside a drawing group)
        function setImageCssSizeAttributes(imageParent, drawingWidth, drawingHeight, drawingAttrs, imageAttrs) {

            var // IE requires to swap cropping settings if image is flipped
                swapHorCrop = _.browser.IE && drawingAttrs.fliph,
                swapVertCrop = _.browser.IE && drawingAttrs.flipv,
                // effective cropping values (checking for 0, because explicit attributes might be used)
                cropLeft = swapHorCrop ? (imageAttrs.cropRight || 0) : (imageAttrs.cropLeft || 0),
                cropRight = swapHorCrop ? (imageAttrs.cropLeft || 0) : (imageAttrs.cropRight || 0),
                cropTop = swapVertCrop ? (imageAttrs.cropBottom || 0) : (imageAttrs.cropTop || 0),
                cropBottom = swapVertCrop ? (imageAttrs.cropTop || 0) : (imageAttrs.cropBottom || 0),
                // horizontal offset/size of cropped bitmaps, as CSS attributes
                horizontalSettings = calculateBitmapSettings(app, drawingWidth, cropLeft, cropRight),
                // vertical offset/size of cropped bitmaps, as CSS attributes
                verticalSettings = calculateBitmapSettings(app, drawingHeight, cropTop, cropBottom);

            imageParent.find('img').css({
                left: horizontalSettings.offset,
                top: verticalSettings.offset,
                width: horizontalSettings.size,
                height: verticalSettings.size
            });
        }

        // updates all formatting for image objects
        function updateImageFormatting() {

            var // special attributes for images
                imageAttributes = mergedAttributes.image,
                // the promise that handles the loading of the image
                imageLoadPromise = null;

            // updates attributes of the image node after loading
            // -> but not for grouped drawings. Those drawings are handled by the group container
            function updateImageAttributes(options) {

                var // current width of the drawing frame, in 1/100 mm
                    drawingWidth = Utils.convertLengthToHmm(drawingFrame.width(), 'px'),
                    // current height of the drawing frame, in 1/100 mm
                    drawingHeight = Utils.convertLengthToHmm(drawingFrame.height(), 'px'),
                    // whether the call of this function is asynchronous
                    isAsync = Utils.getBooleanOption(options, 'isAsync', true),
                    // whether the visibility was changed (sometimes required in async mode)
                    visibilityChanged = false;

                if ((drawingWidth > 0) && (drawingHeight > 0)) {

                    if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
                        // For grouped drawings, it is useless to set the image attributes directly,
                        // because these attributes are dependent from the group itself.
                        // -> after the image is loaded (deferred!), the image attributes need to be set again
                        // to be correct for the complete group.
                        // The call of resizeDrawingsInGroup() is only required after loading the document for
                        // grouped drawings when loading happens without fast load and without
                        // local storage.
                        if (isAsync) {
                            if (docModel.useSlideMode()) { visibilityChanged = docModel.handleContainerVisibility(drawingFrame, { makeVisible: true }); }
                            resizeDrawingsInGroup(DrawingFrame.getGroupNode(drawingFrame));
                            if (visibilityChanged) { docModel.handleContainerVisibility(drawingFrame, { makeVisible: false }); }
                        }
                    } else {
                        setImageCssSizeAttributes(contentNode, drawingWidth, drawingHeight, drawingAttributes, imageAttributes);
                    }
                    // remove class which marks this drawing as unsupported
                    drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);
                    // update the border
                    updateLineFormatting();
                }
            }

            // create the image or SVG node (considered to be constant at runtime)
            if (contentNode.find('>.' + CROPPING_CLASS).children().length === 0) {
                // start with replacement graphics (image data will be loaded asynchronously)
                updateUnsupportedFormatting();
                if (imageAttributes.imageData.length > 0) {
                    imageLoadPromise = createImageNodeFromData(imageAttributes.imageData).done(updateImageAttributes);
                } else if (imageAttributes.imageUrl.length > 0) {
                    // convert relative URL to an application-specific absolute URL
                    imageLoadPromise = createImageNodeFromUrl(/:\/\//.test(imageAttributes.imageUrl) ? imageAttributes.imageUrl : app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: imageAttributes.imageUrl })).done(updateImageAttributes);
                } else {
                    Utils.warn('DrawingFrame.updateFormatting() - missing source data for the image object');
                    imageLoadPromise = $.when();
                }
            } else {
                // update attributes of an existing image element
                updateImageAttributes({ isAsync: false });
                imageLoadPromise = $.when();
            }

            imageLoadPromise.then(function () {
                if (docModel.useSlideMode()) { docModel.trigger('image:loaded', drawingFrame); } // this might be interesting for other apps, too
            });

            // image successfully rendered
            return true;
        }

        // updates chart formatting
        function updateChartFormatting() {

            var // the drawing object model
                model = DrawingFrame.getModel(drawingFrame),
                // the chart rendering engine (lazy initialization, see below)
                renderer = drawingFrame.data('chart-renderer'),
                // unique DOM element identifier for the renderer
                chartId = null,
                drawingNode = null;

            // drawing object model must exist for chart objects
            if (!model) { return false; }

            // lazy creation of the renderer
            if (!renderer) {
                chartId = 'io-ox-documents-chart-' + DrawingFrame.getUid(drawingFrame);
                contentNode.addClass('chartholder');

                // localization for CanvasJS
                if (!CULTURE_INFO) {
                    CULTURE_INFO = {
                        decimalSeparator: LocaleData.DEC,
                        digitGroupSeparator: LocaleData.GROUP,
                        shortDays: DateUtils.getLocaleDaysShort(),
                        days: DateUtils.getLocaleDays(),
                        months: DateUtils.getLocaleMonths(),
                        shortMonths: DateUtils.getLocaleMonthsShort()
                    };
                    CanvasJS.addCultureInfo('en', CULTURE_INFO);
                }

                drawingNode = $('<div>');
                drawingNode.addClass('chartnode');
                drawingNode.attr('id', chartId);
                contentNode.append(drawingNode);

                // really important call, because of a bug in canvasJS
                model.resetData();

                renderer = new CanvasJS.Chart(chartId, model.getModelData());
                drawingFrame.data('chart-renderer', renderer);
                model.firstInit();
            }

            // render the chart object
            model.updateRenderInfo();
            if (model.hasDataPoints()) {
                try {
                    renderer.render();
                } catch (ex) {
                    Utils.exception(ex, 'chart rendering error');
                }
            }
            // background color of model data is already prepared for css
            contentNode.css('backgroundColor', model.getModelData().cssBackgroundColor);

            // chart successfully rendered
            return true;
        }

        /**
         * Updates the fill formatting of the drawing frame.
         *
         * @param {jQuery} [node]
         *  An optional jQuery node, to which the css properties are
         *  assigned. If it is not set, the content node inside the
         *  drawing frame is used. Setting this node, is only required
         *  for drawing groups, where all children need to be modified.
         */
        function updateFillFormatting() {

            var // the document model
                docModel = app.getModel(),
                // the CSS properties for the drawing frame
                cssProps = {},
                // the node, whose properties will be set
                cssNode = contentNode,
                // the fill attributes
                fillAttrs = mergedAttributes.fill;

            if (fillAttrs) {
                // calculate the CSS fill attributes
                switch (fillAttrs.type) {
                    case 'none':
                        // clear everything: color and bitmaps
                        cssProps.background = '';
                        break;
                    case 'solid':
                        // use 'background' compound attribute to clear bitmaps
                        cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill');
                        break;
                    default:
                        Utils.warn('DrawingFrame.updateFillFormatting(): unknown fill type "' + fillAttrs.type + '"');
                }

                // apply the fill attributes
                cssNode.css(cssProps);
            }
        }

        /**
         * Updates the border formatting of the drawing frame.
         *
         * @param {jQuery} [node]
         *  An optional jQuery node, to which the css properties are
         *  assigned. If it is not set, the content node inside the
         *  drawing frame is used. Setting this node, is only required
         *  for drawing groups, where all children need to be modified.
         */
        function updateLineFormatting(node) {

            var // gets the correct node to work on (parent() used at grouped drawings)
                currentNode = node ? DrawingFrame.getDrawingNode(node) : drawingFrame,
                // the line attributes
                lineAttrs = node ? AttributeUtils.getExplicitAttributes(currentNode).line : mergedAttributes.line;

            // Explicit attributes are not sufficient, if default line attributes are used (36892)
            if (node && (!lineAttrs || !lineAttrs.type)) { lineAttrs = getAllDrawingAttributes(node, 'line'); }

            if (lineAttrs && lineAttrs.type) {
                // calculate the border properties
                switch (lineAttrs.type) {
                    case 'none':
                        // clear everything
                        DrawingFrame.clearCanvas(currentNode);
                        break;
                    case 'solid':
                        DrawingFrame.drawBorder(app, currentNode, lineAttrs);
                        break;
                    default:
                        Utils.warn('DrawingFrame.updateLineFormatting(): unknown line type "' + lineAttrs.type + '"');
                }
            }
        }

        /**
         * Updating the 'shape' attributes at the drawing.
         *
         * @param {jQuery} drawing
         *  The jQueryfied drawing node.
         */
        function updateShapeFormatting(node) {

            var // gets the correct node to work on (parent() used at grouped drawings)
                currentNode = node ? DrawingFrame.getDrawingNode(node) : drawingFrame,
                // the explicit attributes of the handled node
                explicitAttributes = AttributeUtils.getExplicitAttributes(currentNode),
                // the shape attributes
                shapeAttrs = node ? explicitAttributes.shape : mergedAttributes.shape,
                // the line attributes
                lineAttrs = node ? explicitAttributes.line : mergedAttributes.line,
                // the fill attributes
                fillAttrs = node ? explicitAttributes.fill : mergedAttributes.fill,
                // shape geometry
                geometryAttrs = explicitAttributes.geometry ? explicitAttributes.geometry : mergedAttributes.geometry,
                // the canvas wrapper
                canvas = null,
                // the text frame node of the drawing
                textFrame = DrawingFrame.getTextFrameNode(currentNode),
                // the default left, top and bottom padding inside the shape (in 1/100 mm)
                leftDefaultPadding = 250, rightDefaultPadding = 250, topDefaultPadding = 125, bottomDefaultPadding = 125;

            // applying text padding & vertical alignment
            if (textFrame && textFrame.length > 0) {
                if (shapeAttrs) {
                    if (shapeAttrs.anchor) {
                        textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE, shapeAttrs.anchor);
                    }
                    if (!geometryAttrs) {
                        textFrame.css({ padding:
                            Utils.convertHmmToCssLength(shapeAttrs.paddingTop ? shapeAttrs.paddingTop : bottomDefaultPadding, 'px', 1) + ' ' +
                            Utils.convertHmmToCssLength(shapeAttrs.paddingRight ? shapeAttrs.paddingRight : leftDefaultPadding, 'px', 1) + ' ' +
                            Utils.convertHmmToCssLength(shapeAttrs.paddingBottom ? shapeAttrs.paddingBottom : bottomDefaultPadding, 'px', 1) + ' ' +
                            Utils.convertHmmToCssLength(shapeAttrs.paddingLeft ? shapeAttrs.paddingLeft : leftDefaultPadding, 'px', 1)
                        });
                    }
                }
            }
            // setting the value for automatic vertical resizing of the shape.
            if (shapeAttrs && shapeAttrs.autoResizeHeight) {
                contentNode.addClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
                // updating the height of a surrounding group node
                if (DrawingFrame.isGroupedDrawingFrame(contentNode.parent())) { DrawingFrame.updateDrawingGroupHeight(contentNode.parent()); }
            } else {
                contentNode.removeClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
            }

            if (docModel.useSlideMode()) {
                // setting the value for automatic text height in the shape.
                if (shapeAttrs && shapeAttrs.autoResizeText) {
                    contentNode.css('font-size', (shapeAttrs.fontScale * 16) + 'px');
                    contentNode.attr('font-scale', shapeAttrs.fontScale);
                    contentNode.attr('lineReduction', Math.round(Math.min(0.2, shapeAttrs.lineReduction) * 100));
                    contentNode.addClass(DrawingFrame.AUTORESIZETEXT_CLASS);
                } else {
                    contentNode.css('font-scale', null);
                    contentNode.css('font-size', null);
                    contentNode.attr('lineReduction', null);
                    contentNode.removeClass(DrawingFrame.AUTORESIZETEXT_CLASS);
                }
            }

            if (!geometryAttrs) {
                updateFillFormatting();
                updateLineFormatting(node);
            } else {
                // the canvas wrapper of the border node
                var lineWidth = 1,
                    snapRect = { left: 0, top: 0, width: currentNode.first().width(), height: currentNode.first().height() },
                    boundRect = DrawingFrame.calcBoundRectPixel(snapRect),
                    guideCache = {},
                    drawGeometry = function (geometry) {

                        var visibilityChanged = false;

                        if (!docModel || docModel.destroyed) { return; } // the model might have been destroyed in the meantime

                        if (lineAttrs) {
                            lineWidth = Math.max(Utils.convertHmmToLength(lineAttrs.width, 'px', 1), 1);
                        }
                        // taking care of special text rects
                        if (geometry.textRect && textFrame && textFrame.length > 0) {
                            var getTextRectValue = function (value, defaultValue) {
                                    return value ? Utils.round(DrawingFrame.getGeometryValue(value, snapRect, geometry.avList, geometry.gdList, guideCache, 1000), 1) : defaultValue;
                                },
                                left = getTextRectValue(geometry.textRect.l, snapRect.left) + Utils.convertHmmToLength((shapeAttrs && shapeAttrs.paddingLeft) ? shapeAttrs.paddingLeft : leftDefaultPadding, 'px', 1),
                                top = getTextRectValue(geometry.textRect.t, snapRect.top) + Utils.convertHmmToLength((shapeAttrs && shapeAttrs.paddingTop) ? shapeAttrs.paddingTop : topDefaultPadding, 'px', 1),
                                right = getTextRectValue(geometry.textRect.r, (snapRect.left + snapRect.width) - 1) - Utils.convertHmmToLength((shapeAttrs && shapeAttrs.paddingRight) ? shapeAttrs.paddingRight : rightDefaultPadding, 'px', 1),
                                paddingBottom = (snapRect.height - getTextRectValue(geometry.textRect.b, (snapRect.top + snapRect.height) - 1)) + top +
                                    Utils.convertHmmToLength((shapeAttrs && shapeAttrs.paddingTop) ? shapeAttrs.paddingTop : topDefaultPadding, 'px', 1) +
                                    Utils.convertHmmToLength((shapeAttrs && shapeAttrs.paddingBottom) ? shapeAttrs.paddingBottom : bottomDefaultPadding, 'px', 1);

                            // changed from "paddingBottom" to "height - padding" & "marginBottom", so textframe does not gets bigger than its parent drawing (needed for selection)
                            textFrame.css({ left: left + 'px', top: top + 'px', width: (right - left) + 'px', height: 'calc(100% - ' + paddingBottom + 'px)', marginBottom: paddingBottom + 'px' });

                            // IE workarounds for vertical aligment @see style.less div.textframe
                            if (_.browser.IE && shapeAttrs && shapeAttrs.anchor !== 'top') {
                                var offset = shapeAttrs.anchor === 'bottom' ? '100%' : '50%';
                                textFrame.css('top', 'calc(' + offset + ' + ' + top + 'px)');
                            }

                            // the textframe been changed so also the snaprect, we have to reset the guideCache and also
                            // reset the new snaprect
                            guideCache = {};

                            if (docModel.useSlideMode()) { visibilityChanged = docModel.handleContainerVisibility(currentNode.first(), { makeVisible: true }); }
                            snapRect = { left: 0, top: 0, width: currentNode.first().width(), height: currentNode.first().height() };
                            if (visibilityChanged) { docModel.handleContainerVisibility(currentNode.first(), { makeVisible: false }); }

                        }
                        boundRect = DrawingFrame.calcBoundRectPixel(snapRect);

                        canvas = DrawingFrame.initializeCanvas(currentNode, boundRect, lineWidth);

                        canvas.render(function (context) {

                            if (('pathList' in geometry) && (geometry.pathList.length > 0)) {
                                DrawingFrame.drawShape(docModel, context, snapRect, geometry, fillAttrs, lineAttrs, guideCache);
                                return;
                            }

                            var hasFill = fillAttrs && (fillAttrs.type !== 'none');
                            if (hasFill) {
                                context.setFillStyle(docModel.getCssColor(fillAttrs.color, 'fill'));
                            }

                            var hasLine = lineAttrs && (lineAttrs.type !== 'none');
                            if (hasLine) {
                                var color = docModel.getCssColor(lineAttrs.color, 'line');
                                var pattern = Border.getBorderPattern(lineAttrs.style, lineWidth);
                                context.setLineStyle({ style: color, width: lineWidth, pattern: pattern });
                            }

                            var mode = getRenderMode(hasFill, hasLine);
                            if (mode) { context.drawRect(boundRect, mode); }
                        });
                    };
                if ('presetShape' in geometryAttrs) {
                    var presetGeo = PRESET_GEOMETRIES[geometryAttrs.presetShape],
                        geo = _.defaults(_.clone(geometryAttrs), presetGeo);
                    if (presetGeo.avList && geo.avList) {
                        _.defaults(geo.avList, presetGeo.avList);
                    }
                    drawGeometry(geo);
                } else {
                    drawGeometry(geometryAttrs);
                }
            }
        }

        // updates the class at the drawing, that is used to find standard odt text frames (and not shapes with text)
        function updateTextFrameState() {

            var // the explicit drawing attributes (not using merge attributes, because they always contain a style id)
                attrs = AttributeUtils.getExplicitAttributes(drawingFrame);

            if (attrs && attrs.styleId && _.isString(attrs.styleId)) {
                drawingFrame.addClass(DrawingFrame.ODFTEXTFRAME_CLASS);
            } else {
                drawingFrame.removeClass(DrawingFrame.ODFTEXTFRAME_CLASS);
            }

        }

        /**
         * updates the size and position of all children of a drawing group
         *
         *  @param {Boolean} [options.onlyResize=false]
         *   If set to true, only resizing of the grouped drawings is handled.
         *   This is especially useful, after the document was loaded.
         */
        function resizeDrawingsInGroup(drawing, options) {

            var // the direct drawing children of the drawing group
                allChildren = DrawingFrame.getAllGroupDrawingChildren(drawing, { onlyDirectChildren: true }),
                // the explicit attributes at the group node (should be used for precision)
                explicitGroupAttributes = AttributeUtils.getExplicitAttributes(drawing),
                // the explicit drawing attributes
                explicitDrawingAttributes = explicitGroupAttributes && explicitGroupAttributes.drawing,
                // whether the children of a group only need to be resized
                // -> if onlyResize is set, the 'mergedAttrinbutes' of DrawingFrame.updateFormatting are invalid
                onlyResize = Utils.getBooleanOption(options, 'onlyResize', false),
                // the width of the drawing group element in 1/100 mm
                groupWidth = (explicitDrawingAttributes && _.isNumber(explicitDrawingAttributes.width)) ? explicitDrawingAttributes.width : Utils.convertLengthToHmm(drawing.children().first().width(), 'px'),
                // the height of the drawing group element in  1/100 mm
                groupHeight = (explicitDrawingAttributes && _.isNumber(explicitDrawingAttributes.height)) ? explicitDrawingAttributes.height : Utils.convertLengthToHmm(drawing.children().first().height(), 'px'),
                // the maximum width of the children
                maxWidth = 0,
                // the maximum height of the children
                maxHeight = 0,
                // all horizontal offsets
                allHorzOffsets = [],
                // all vertical offsets
                allVertOffsets = [],
                // the horizontal offset of one drawing
                offsetHorz = 0,
                // the vertical offset of one drawing
                offsetVert = 0,
                // the horizontal stretch factor for the children
                horzFactor = 1,
                // the vertical stretch factor for the children
                vertFactor = 1;

            // updates the size and position of one drawing child inside a drawing group
            function handleDrawingInsideDrawingGroup(drawing, attrs, horzFactor, vertFactor, offsetX, offsetY) {

                var // the attributes at the drawing property
                    drawingAttrs = attrs.drawing,
                    // the required css attributes
                    cssAttrs = {},
                    // an optional horizontal resize factor
                    horzResizeFactor = horzFactor || 1,
                    // an optional vertical resize factor
                    vertResizeFactor = vertFactor || 1,
                    // type of the drawing object inside the group: 'image', ...
                    type = DrawingFrame.getDrawingType(drawing);

                // updates attributes of the image node of a drawing frame inside a drawing group container
                function updateImageAttributesInGroup() {

                    var // current width of the image, in 1/100 mm
                        drawingWidth = Utils.round(drawingAttrs.width * horzResizeFactor, 1),
                        // current height of the image, in 1/100 mm
                        drawingHeight = Utils.round(drawingAttrs.height * vertResizeFactor, 1);

                    setImageCssSizeAttributes(drawing.children().first(), drawingWidth, drawingHeight, drawingAttrs, attrs.image);
                }

                // set the CSS attributes for the drawing
                if (_.isNumber(drawingAttrs.left)) { cssAttrs.left = Utils.convertHmmToCssLength((drawingAttrs.left - offsetX) * horzResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.top)) { cssAttrs.top = Utils.convertHmmToCssLength((drawingAttrs.top - offsetY) * vertResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.width)) { cssAttrs.width = Utils.convertHmmToCssLength(drawingAttrs.width * horzResizeFactor, 'px', 1); }

                if (_.isNumber(drawingAttrs.height) && !DrawingFrame.isAutoResizeHeightDrawingFrame(drawing)) { // not setting height for auto resize drawings
                    cssAttrs.height = Utils.convertHmmToCssLength(drawingAttrs.height * vertResizeFactor, 'px', 1);
                }

                drawing.css(cssAttrs);

                if (!onlyResize) {
                    // updating the size of the images inside the drawing
                    if (type === 'image') { updateImageAttributesInGroup(drawing); }

                    // update border
                    if (type !== 'group') { updateShapeFormatting(drawing); }
                }
            }

            // iterating over all drawing children of the group
            if (allChildren.length > 0) {

                // comparing width and height of the group with all values of the children
                // -> the aim is, that all children fit optimal into the group
                // -> this needs to be set after loading and after resizing

                // getting the maximum value of the children for 'left + width' and 'top + height')
                _.each(allChildren, function (drawing) {

                    var // the explicit drawing attributes
                        drawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                        // the horizontal expansion of one drawing
                        currentHorz = 0,
                        // the vertical expansion of one drawing
                        currentVert = 0;

                    if (drawingAttrs && drawingAttrs.drawing) {
                        if (_.isNumber(drawingAttrs.drawing.left)) { currentHorz = drawingAttrs.drawing.left; }
                        if (_.isNumber(drawingAttrs.drawing.width)) { currentHorz += drawingAttrs.drawing.width; }
                        if (_.isNumber(drawingAttrs.drawing.top)) { currentVert = drawingAttrs.drawing.top; }
                        if (_.isNumber(drawingAttrs.drawing.height)) { currentVert += drawingAttrs.drawing.height; }

                        if (currentHorz > maxWidth) { maxWidth = currentHorz; }
                        if (currentVert > maxHeight) { maxHeight = currentVert; }

                        // collecting all horizontal and vertical offset positions
                        allHorzOffsets.push(_.isNumber(drawingAttrs.drawing.left) ? drawingAttrs.drawing.left : 0);
                        allVertOffsets.push(_.isNumber(drawingAttrs.drawing.top) ? drawingAttrs.drawing.top : 0);
                    }
                });

                if (maxWidth > 0 || maxHeight > 0) {

                    offsetHorz = Math.min.apply(null, allHorzOffsets);
                    offsetVert = Math.min.apply(null, allVertOffsets);

                    // now checking values for maxWidth and maxHeight
                    horzFactor = Utils.round(groupWidth / (maxWidth - offsetHorz + 1), 0.001);
                    vertFactor = Utils.round(groupHeight / (maxHeight - offsetVert + 1), 0.001);

                    // getting the maximum value of the children for 'left + width' and 'top + height')
                    _.each(allChildren, function (drawing) {
                        handleDrawingInsideDrawingGroup($(drawing), AttributeUtils.getExplicitAttributes(drawing), horzFactor, vertFactor, offsetHorz, offsetVert);
                    });
                }
            }
        }

        // used for fix for Bug 45323
        // special behavior of presentation files
        // Shapes do ignore their horizontal flip state if parent group has standard transform
        // or completely inverted transform
        // textDirection info is set always but only checked in presentation if horizontal flip is true
        var standardTextDirection = false;

        if (drawingAttributes) {

            // convert to single jQuery object
            drawingFrame = $(drawingFrame).first();

            var flipH = drawingAttributes.flipH;
            var flipV = drawingAttributes.flipV;
            if (flipH && docModel.useSlideMode() && type === 'shape' && drawingFrame.parent().data('standardTextDirection')) {
                //disable horizontal flip because of special presentation behavior
                flipH = false;
            }

            // vertically or horizontally flipped drawings
            if (flipH && flipV) {
                transformations = 'scale(-1)';
                standardTextDirection = true;
            } else if (flipH) {
                transformations = 'scaleX(-1)';
            } else if (flipV) {
                transformations = 'scaleY(-1)';
            } else {
                standardTextDirection = true;
            }
        }

        contentNode.data('standardTextDirection', standardTextDirection);

        // always set the CSS attribute, to be able to remove existing transformations
        Utils.setCssAttributeWithPrefixes(contentNode, 'transform', transformations);

        // update drawing frame specific to the drawing type
        switch (type) {
            case 'image':
                updateLineFormatting();
                rendered = updateImageFormatting();
                break;
            case 'ole':
                rendered = updateImageFormatting();
                break;
            case 'chart':
                rendered = updateChartFormatting();
                break;
            case 'shape':
                updateShapeFormatting();
                if (app.isODF()) { updateTextFrameState(); }
                // the following line has to be removed to activate shape geometry in text documents
                rendered = DrawingFrame.isTextFrameShapeDrawingFrame(drawingFrame) || app.getName() === 'io.ox/office/presentation';
                if (!_.browser.IE && !_.browser.Firefox && rendered) { $(drawingFrame).attr('contenteditable', true); }  // TODO: Not good here, but required for fast load (36246)
                break;
            case 'group':
                // the group needs to take care of the size of all children
                resizeDrawingsInGroup(drawingFrame);
                if (!_.browser.IE && !_.browser.Firefox) { $(drawingFrame).attr('contenteditable', true); }  // TODO: Not good here, but required for fast load
                rendered = true;
                break;
        }

        // during loading of document, an additional resize of drawings in groups might be required
        // -> during working with document, operations are only generated for the group itself.
        if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
            resizeDrawingsInGroup(DrawingFrame.getGroupNode(drawingFrame), { onlyResize: true });
        }

        // add replacement in case the drawing type is not supported, or if rendering was not successful
        if (!rendered) {
            updateUnsupportedFormatting();
        }

    };

    /**
     * returns the wrapped canvas for the current node.
     *
     * @param {jQuery} [node]
     */
    DrawingFrame.getWrappedCanvas = function (currentNode) {

        var // the canvasNode inside the drawing frame
            canvasNode = DrawingFrame.getCanvasNode(currentNode),
            // the canvas wrapper
            canvas = null;

        if (canvasNode.length > 0) {
            canvas = new Canvas({ classes: DrawingFrame.CANVAS_CLASS, node: canvasNode });
        } else {
            canvas = new Canvas({ classes: DrawingFrame.CANVAS_CLASS });
            DrawingFrame.getContentNode(currentNode).prepend(canvas.getNode());
        }
        return canvas;
    };

    /**
     * a initialized canvas is returned for the current node
     *
     * @param {jQuery} [node]
     */
    DrawingFrame.initializeCanvas = function (currentNode, rectangle, borderSize) {
        var // drawing-type
            type = DrawingFrame.getDrawingType(currentNode),
            // the canvas wrapper of the border node
            canvas          = DrawingFrame.getWrappedCanvas(currentNode),
            // border size
            halfBorderSize  = Math.ceil(borderSize * 0.5),
            // size to calculate with (different for images and shapes)
            calcSize        = (type === 'image') ? borderSize : halfBorderSize,
            // canvas start (left, top)
            left            = rectangle.left - halfBorderSize,
            top             = rectangle.top - halfBorderSize,
            // width and height of both (canvas and canvas-node)
            width           = rectangle.width + 2 * calcSize,
            height          = rectangle.height + 2 * calcSize,
            // canvasnode start (left, top)
            nodeLeft        = rectangle.left - calcSize,
            nodeTop         = rectangle.top - calcSize;

        // initialize canvas
        canvas.initialize({ left: left, top: top,
                            width: width, height: height });

        // set position of the canvas element
        canvas.getNode().css({ left: nodeLeft, top: nodeTop,
                               width: width, height: height });
        return canvas;
    };

    /*
     * returns the bounding rectangle in pixel of the drawing.. this is
     * necessary because the shape might exceed the initial node size
    */
    DrawingFrame.calcBoundRectPixel = function (snapRect) {
        return snapRect;
    };

    DrawingFrame.drawShape = function (docModel, context, snapRect, geometryAttrs, fillAttrs, lineAttrs, guideCache) {

        var avList = geometryAttrs.avList,
            gdList = geometryAttrs.gdList;

        if (!geometryAttrs.pathList) {
            return;
        }

        function getValue(value) {
            return DrawingFrame.getGeometryValue(value, snapRect, avList, gdList, guideCache, 1000);
        }

        function shadeColor(color, percent) {
            var f = parseInt(color.slice(1), 16),
                t = percent < 0 ? 0 : 255,
                p = Math.abs(percent),
                R = f >> 16,
                G = f >> 8 & 0x00FF,
                B = f & 0x0000FF;
            return '#' + (0x1000000 + (Math.round((t - R) * p) + R) * 0x10000 + (Math.round((t - G) * p) + G) * 0x100 + (Math.round((t - B) * p) + B)).toString(16).slice(1);
        }

        geometryAttrs.pathList.forEach(function (path) {

            // initialize context with proper fill attributes for the next path
            var hasFill = fillAttrs && (fillAttrs.type !== 'none') && (path.fillMode !== 'none');
            if (hasFill) {
                var fillColor = docModel.getCssColor(fillAttrs.color, 'fill');
                switch (path.fillMode) {
                    case 'lighten':
                        fillColor = shadeColor(fillColor, 0.40);
                        break;
                    case 'lightenLess':
                        fillColor = shadeColor(fillColor, 0.20);
                        break;
                    case 'darken':
                        fillColor = shadeColor(fillColor, -0.40);
                        break;
                    case 'darkenLess':
                        fillColor = shadeColor(fillColor, -0.20);
                        break;
                }
                context.setFillStyle(fillColor);
            }

            // initialize context with proper line attributes for the next path
            var hasLine = lineAttrs && (lineAttrs.type !== 'none') && (path.isStroke !== false);
            if (hasLine) {
                var lineColor = docModel.getCssColor(lineAttrs.color, 'line');
                var lineWidth = Math.max(Utils.convertHmmToLength(lineAttrs.width, 'px', 1), 1);
                var pattern = Border.getBorderPattern(lineAttrs.style, lineWidth);
                context.setLineStyle({ style: lineColor, width: lineWidth, pattern: pattern });
            }

            var mode = getRenderMode(hasFill, hasLine);
            if (mode && path.commands) {
                var newPath = context.createPath(),
                    widthPx = snapRect.width,
                    heightPx = snapRect.height,
                    logWidth = ('width' in path) ? path.width : widthPx,
                    logHeight = ('height' in path) ? path.height : heightPx,
                    fXScale = widthPx / logWidth,
                    fYScale = heightPx / logHeight,
                    cx = 0,
                    cy = 0;

                if (logWidth > 0 && logHeight > 0) {
                    path.commands.forEach(function (command) {
                        var pts = null;
                        var length = null;
                        var i = 0;
                        switch (command.c) {
                            case 'moveTo':
                                cx = getValue(command.x) * fXScale;
                                cy = getValue(command.y) * fYScale;
                                newPath.moveTo(cx, cy);
                                break;
                            case 'lineTo':
                                cx = getValue(command.x) * fXScale;
                                cy = getValue(command.y) * fYScale;
                                newPath.lineTo(cx, cy);
                                break;
                            case 'arcTo':
                                var rx = getValue(command.wr) * fXScale,
                                    ry = getValue(command.hr) * fYScale,
                                    stAng = (getValue(command.stAng) / 60000) * Math.PI / 180,
                                    swAng = getValue(command.swAng),

                                    /*
                                     * calculates the point on an arc that corresponds to the given angle (ms like)
                                     */
                                    calcArcPoint = function (rx, ry, angle) {
                                        var wt1 = rx * Math.sin(angle),
                                            ht1 = ry * Math.cos(angle),
                                            dx1 = rx * Math.cos(Math.atan2(wt1, ht1)),
                                            dy1 = ry * Math.sin(Math.atan2(wt1, ht1));

                                        return { x: dx1, y: dy1 };
                                    },
                                    // calculating new currentPoint
                                    startPoint = calcArcPoint(rx, ry, stAng),
                                    endPoint = calcArcPoint(rx, ry, stAng + swAng / 10800000 * Math.PI),
                                    calcAngle = function (x, y) {
                                        if (rx > ry) {
                                            return Math.atan2(y * rx / ry, x);
                                        } else {
                                            return Math.atan2(y, x * ry / rx);
                                        }
                                    },
                                    eStAng = calcAngle(startPoint.x, startPoint.y),
                                    eEndAng = calcAngle(endPoint.x, endPoint.y);

                                // check if a full circle has to be drawn
                                if (Math.abs(swAng) >= 21600000) {
                                    swAng = -1;
                                    eEndAng = eStAng + 0.0001;
                                }
                                newPath.ellipseTo(cx, cy, rx, ry, eStAng, eEndAng, swAng < 0);

                                cx = cx + endPoint.x - startPoint.x;
                                cy = cy + endPoint.y - startPoint.y;
                                break;
                            case 'close':
                                newPath.close();
                                break;
                            case 'quadBezierTo':
                                if (command.pts) {
                                    pts = command.pts;
                                    length = pts.length;

                                    if (length % 2 === 0) {
                                        for (i = 0; i < length; i += 2) {
                                            newPath.quadraticCurveTo(getValue(pts[i].x) * fXScale, getValue(pts[i].y) * fYScale,
                                                    cx = getValue(pts[i + 1].x) * fXScale, cy = getValue(pts[i + 1].y) * fYScale);
                                        }
                                    }
                                }
                                break;
                            case 'cubicBezierTo':
                                if (command.pts) {
                                    pts = command.pts;
                                    length = pts.length;

                                    if (length % 3 === 0) {
                                        for (i = 0; i < length; i += 3) {
                                            newPath.bezierCurveTo(getValue(pts[i].x) * fXScale, getValue(pts[i].y) * fYScale,
                                                                  getValue(pts[i + 1].x) * fXScale, getValue(pts[i + 1].y) * fYScale,
                                                                  cx = getValue(pts[i + 2].x) * fXScale, cy = getValue(pts[i + 2].y) * fYScale);
                                        }
                                    }
                                }
                                break;
                        }
                    });
                    context.drawPath(newPath, mode);
                }
            }
        });
        return;
    };

    DrawingFrame.getGeometryValue = function (value, snapRect, avList, gdList, guideCache, maxGdNumber) {
        if (typeof value === 'number') {
            return value;
        }
        if (value in guideCache) {
            return guideCache[value];
        }
        switch (value) {
            // 3 x 360° / 4 = 270°
            case '3cd4':
                return 270 * 60000;
            // 3 x 360° / 8 = 135°
            case '3cd8':
                return 135 * 60000;
            // 5 x 360° / 8 = 225°
            case '5cd8':
                return 225 * 60000;
            // 7 x 360° / 8 = 315°
            case '7cd8':
                return 315 * 60000;
            // bottom
            case 'b':
                return snapRect.top + (snapRect.height - 1);
            // 360° / 2 = 180°
            case 'cd2':
                return 180 * 60000;
            // 360° / 4 = 90°
            case 'cd4':
                return 90 * 60000;
            // 360° / 8 = 45°
            case 'cd8':
                return 45 * 60000;
            // horizontal center
            case 'hc':
                return snapRect.left + snapRect.width * 0.5;
            // height
            case 'h':
                return snapRect.height;
            // height / 2
            case 'hd2':
                return snapRect.height / 2;
            // height / 3
            case 'hd3':
                return snapRect.height / 3;
            // height / 4
            case 'hd4':
                return snapRect.height / 4;
            // height / 5
            case 'hd5':
                return snapRect.height / 5;
            // height / 6
            case 'hd6':
                return snapRect.height / 6;
            // height / 8
            case 'hd8':
                return snapRect.height / 8;
            // left
            case 'l':
                return snapRect.left;
            // long side
            case 'ls':
                return Math.max(snapRect.width, snapRect.height);
            // right
            case 'r':
                return snapRect.left + (snapRect.width - 1);
            // short side
            case 'ss':
                return Math.min(snapRect.width, snapRect.height);
            // short side / 2
            case 'ssd2':
                return Math.min(snapRect.width, snapRect.height) / 2;
            // short side / 4
            case 'ssd4':
                return Math.min(snapRect.width, snapRect.height) / 4;
            // short side / 6
            case 'ssd6':
                return Math.min(snapRect.width, snapRect.height) / 6;
            // short side / 8
            case 'ssd8':
                return Math.min(snapRect.width, snapRect.height) / 8;
            // short side / 16
            case 'ssd16':
                return Math.min(snapRect.width, snapRect.height) / 16;
            // short side / 32
            case 'ssd32':
                return Math.min(snapRect.width, snapRect.height) / 32;
            // top
            case 't':
                return snapRect.top;
            // vertical center
            case 'vc':
                return snapRect.top + snapRect.height * 0.5;
            // width
            case 'w':
                return snapRect.width;
            // width / 2
            case 'wd2':
                return snapRect.width / 2;
            // width / 4
            case 'wd4':
                return snapRect.width / 4;
            // width / 5
            case 'wd5':
                return snapRect.width / 5;
            // width / 6
            case 'wd6':
                return snapRect.width / 6;
            // width / 8
            case 'wd8':
                return snapRect.width / 8;
            // width / 10
            case 'wd10':
                return snapRect.width / 10;
            // width / 32
            case 'wd32':
                return snapRect.width / 32;
        }
        // the value is not a constant value, next check if this is a guide name
        if (avList && value in avList) {
            return avList[value];
        }
        if (gdList) {
            for (var i = 0; i < gdList.length; i++) {
                if (gdList[i].name === value) {
                    if (i > maxGdNumber) {
                        return 0.0;
                    }
                    var result = DrawingFrame.calcGeometryValue(gdList[i], snapRect, avList, gdList, guideCache, i - 1);
                    guideCache[value] = result;
                    return result;
                }
            }
        }
        // we never should come here ...
        if (!$.isNumeric(value)) {
            return 0.0;
        }
        // this should be normal float/int value
        return parseFloat(value);
    };

    DrawingFrame.calcGeometryValue = function (gd, snapRect, avList, gdList, guideCache, maxGdNumber) {

        function p(p) {
            return parseFloat(DrawingFrame.getGeometryValue(gd[p], snapRect, avList, gdList, guideCache, maxGdNumber));
        }

        var p0, p1, p2;

        switch (gd.op) {
            // Multiply Divide Formula
            // Arguments: 3 (fmla="*/ x y z")
            // Usage: "*/ x y z" = ((x * y) / z) = value of this guide
            case '*/':
                return (p('p0') * p('p1')) / p('p2');

            // Add Subtract Formula
            // Arguments: 3 (fmla="+- x y z")
            // Usage: "+- x y z" = ((x + y) - z) = value of this guide
            case '+-':
                return (p('p0') + p('p1')) - p('p2');

            //  Add Divide Formula
            // Arguments: 3 (fmla="+/ x y z")
            // Usage: "+/ x y z" = ((x + y) / z) = value of this guide
            case '+/':
                return (p('p0') + p('p1')) / p('p2');

            // If Else Formula
            // Arguments: 3 (fmla="?: x y z")
            // Usage: "?: x y z" = if (x > 0), then y = value of this guide,
            // else z = value of this guide
            case '?:':
                return (p('p0') > 0) ? p('p1') : p('p2');

            // Absolute Value Formula
            // Arguments: 1 (fmla="abs x")
            // Usage: "abs x" = if (x < 0), then (-1) * x = value of this guide
            // else x = value of this guide
            case 'abs':
                return Math.abs(p('p0'));

            // ArcTan Formula
            // Arguments: 2 (fmla="at2 x y")
            // Usage: "at2 x y" = arctan(y / x) = value of this guide
            case 'at2':
                return (10800000 * Math.atan2(p('p1'), p('p0'))) / Math.PI;

            // Cosine ArcTan Formula
            // Arguments: 3 (fmla="cat2 x y z")
            // Usage: "cat2 x y z" = (x*(cos(arctan(z / y))) = value of this guide
            case 'cat2':
                return p('p0') * (Math.cos(Math.atan2(p('p2'), p('p1'))));

            // Cosine Formula
            // Arguments: 2 (fmla="cos x y")
            // Usage: "cos x y" = (x * cos( y )) = value of this guide
            case 'cos':
                return p('p0') * Math.cos(Math.PI * p('p1') / 10800000);

            // Maximum Value Formula
            // Arguments: 2 (fmla="max x y")
            // Usage: "max x y" = if (x > y), then x = value of this guide
            // else y = value of this guide
            case 'max':
                return Math.max(p('p0'), p('p1'));

            // Minimum Value Formula
            // Arguments: 2 (fmla="min x y")
            // Usage: "min x y" = if (x < y), then x = value of this guide
            // else y = value of this guide
            case 'min':
                return Math.min(p('p0'), p('p1'));

            // Modulo Formula
            // Arguments: 3 (fmla="mod x y z")
            // Usage: "mod x y z" = sqrt(x^2 + b^2 + c^2) = value of this guide
            case 'mod':
                p0 = p('p0');
                p1 = p('p1');
                p2 = p('p2');
                return Math.sqrt(p0 * p0 + p1 * p1 + p2 * p2);

            // Pin To Formula
            // Arguments: 3 (fmla="pin x y z")
            // Usage: "pin x y z" = if (y < x), then x = value of this guide
            // else if (y > z), then z = value of this guide
            // else y = value of this guide
            case 'pin':
                p0 = p('p0');
                p1 = p('p1');
                p2 = p('p2');
                return (p1 < p0) ? p0 : (p1 > p2) ? p2 : p1;

            // Sine ArcTan Formula
            // Arguments: 3 (fmla="sat2 x y z")
            // Usage: "sat2 x y z" = (x*sin(arctan(z / y))) = value of this guide
            case 'sat2':
                return p('p0') * (Math.sin(Math.atan2(p('p2'), p('p1'))));

            // Sine Formula
            // Arguments: 2 (fmla="sin x y")
            // Usage: "sin x y" = (x * sin( y )) = value of this guide
            case 'sin':
                return p('p0') * Math.sin(Math.PI * p('p1') / 10800000);

            // Square Root Formula
            // Arguments: 1 (fmla="sqrt x")
            // Usage: "sqrt x" = sqrt(x) = value of this guide
            case 'sqrt':
                return Math.sqrt(p('p0'));

            // Tangent Formula
            // Arguments: 2 (fmla="tan x y")
            // Usage: "tan x y" = (x * tan( y )) = value of this guide
            case 'tan':
                return p('p0') * Math.tan(Math.PI * p('p1') / 10800000);

            // Literal Value Formula
            // Arguments: 1 (fmla="val x")
            // Usage: "val x" = x = value of this guide
            case 'val':
                return p('p0');
        }
        return 0.0;
    };

    // draws the border to the canvas
    DrawingFrame.drawBorder = function (app, currentNode, lineAttrs) {

        var // the document model
            docModel = app.getModel(),
            // border width
            lineWidth = Math.max(Utils.convertHmmToLength(lineAttrs.width, 'px', 1), 1),
            halfBorderSize = Math.ceil(lineWidth * 0.5),
            // dimensions of the canvas element, special behavior for text frames. Their border covers their boundary
            cWidth = currentNode.first().width(),
            cHeight = currentNode.first().height(),
            // the canvas wrapper of the border node
            canvas = DrawingFrame.initializeCanvas(currentNode, { left: 0, top: 0, width: cWidth, height: cHeight }, lineWidth);

        canvas.render(function (context, width, height) {
            var color = docModel.getCssColor(lineAttrs.color, 'line'),
                pattern = Border.getBorderPattern(lineAttrs.style, lineWidth);

            context.setLineStyle({ style: color, width: lineWidth, pattern: pattern });

            var drawX = 0,
                drawY = 0,
                drawW = (width - 2 * halfBorderSize),
                drawH = (height - 2 * halfBorderSize);

            context.drawRect(drawX, drawY, drawW, drawH, 'stroke');
        });
    };

    // clears the canvas
    DrawingFrame.clearCanvas = function (currentNode) {

        var // the border canvas node inside the drawing frame
            canvasNode = DrawingFrame.getCanvasNode(currentNode),
            // the canvas wrapper of the border node
            canvas = null;

        if (canvasNode.length > 0) {
            canvas = new Canvas({ classes: DrawingFrame.CANVAS_CLASS, node: canvasNode });
            canvas.initialize({ width: canvasNode.width(), height: canvasNode.height() });
        }
    };

    /**
     * Returns rotation angle of given drawing node.
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Number|null}
     *  Angle of rotation, or null if no rotation is set.
     */
    DrawingFrame.getDrawingRotationAngle = function (docModel, drawingNode) {
        var attrs = null;
        var rotation = null;
        if (DrawingFrame.isRotatedDrawingNode(drawingNode)) {
            // first trying to fetch direct attributes of the element
            attrs = AttributeUtils.getExplicitAttributes(drawingNode);
            rotation = attrs && attrs.drawing && attrs.drawing.rotation || null;
            if (!rotation) {
                // if not fetched, get from inherited attributes
                attrs = docModel.getDrawingStyles().getElementAttributes(drawingNode);
                rotation = attrs && attrs.drawing && attrs.drawing.rotation || null;
            }
        }
        return rotation;
    };

    // selection --------------------------------------------------------------

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return Forms.isSelectedNode(drawingFrame);
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *  @param {Number} [options.scaleHandles=1]
     *      the usable handles are scaled or themselves
     *      (without changing the size of the whole selection)
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        var // HTML mark-up for the selection
            markup = null;

        // create new selection nodes if missing
        if (!DrawingFrame.isSelected(drawingFrame)) {

            // create a new selection node
            markup = '<div class="' + SELECTION_CLASS + '">';

            // create the border nodes
            markup += '<div class="borders">';
            _.each('tblr', function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create resizer handles
            markup += '<div class="resizers">';
            _.each(['t', 'tr', 'r', 'br', 'b', 'bl', 'l', 'tl'], function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            if (options.rotatable) {
                // create rotate handle bar
                markup += '<div class="rotate-handle"></div>';
            }

            // create the tracker node
            markup += '<div class="tracker"></div>';

            // close the selection node
            markup += '</div>';

            $(drawingFrame).first().append(markup);
        }

        // update state classes at the drawing frame
        $(drawingFrame).first()
            .addClass(Forms.SELECTED_CLASS)
            .toggleClass('movable', Utils.getBooleanOption(options, 'movable', false))
            .toggleClass('resizable', Utils.getBooleanOption(options, 'resizable', false));

        DrawingFrame.updateScaleHandles(drawingFrame, options);

        return getSelectionNode(drawingFrame);
    };

    /**
     * updates the additional nodes displayed while a drawing frame is selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Number} [options.scaleHandles=1]
     *      the usable handles are scaled or themselves
     *      (without changing the size of the whole selection)
     *      range is 1 - 3
     */
    DrawingFrame.updateScaleHandles = function (drawingFrame, options) {

        var // scaleHandles for setting fontsize to scale inner elements
            scaleHandles = Utils.getNumberOption(options, 'scaleHandles', 1, 1, 3);

        getSelectionNode(drawingFrame).css('font-size', (scaleHandles * 0.1) + 'rem');
    };

    /**
     * Updates mouse pointers for the resizers of the drawing frame, according to the rotation angle of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Number} angle
     *  Rotation angle of the drawing, in degrees.
     */
    DrawingFrame.updateResizersMousePointers = function (drawingFrame, angle) {
        var resizeHandles, shift;
        var pointers = ['n-resize', 'ne-resize', 'e-resize', 'se-resize', 's-resize', 'sw-resize', 'w-resize', 'nw-resize'];

        if (_.isNumber(angle)) {
            angle %= 360;
            angle += 22.5; // start offset
            shift = ~~(angle / 45); // 8 handles makes 8 areas with 45 degrees each
            resizeHandles = drawingFrame.find('.resizers').children();
            _.each(resizeHandles, function (handle, index) {
                var pointerInd = (index + shift) % pointers.length;
                $(handle).css('cursor', pointers[pointerInd]);
            });
        }
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        $(drawingFrame).first().removeClass(Forms.SELECTED_CLASS + ' movable resizable');
        getSelectionNode(drawingFrame).remove();
    };

    /**
     * Sets or resets the CSS class representing the active tracking mode for
     * the specified drawing frames.
     *
     * @param {HTMLElement|jQuery} drawingFrames
     *  The root node of a single drawing frame, or multiple drawing frames in
     *  a jQuery collection. This method changes the tracking mode of all
     *  drawing frames at once.
     *
     * @param {Boolean} active
     *  Whether tracking mode is active.
     */
    DrawingFrame.toggleTracking = function (drawingFrames, active) {
        $(drawingFrames).find('>.' + SELECTION_CLASS).toggleClass(TRACKING_CLASS, active === true);
    };

    /**
     * Returns the tracking direction stored in a resizer handle node of a
     * drawing frame selection box.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String|Null}
     *  If the passed node is a resizer handle of a drawing frame selection
     *  box, returns its direction identifier (one of 'l', 'r', 't', 'b', 'tl',
     *  'tr', 'bl', or 'br'). Otherwise, returns null.
     */
    DrawingFrame.getResizerHandleType = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR + '>.' + SELECTION_CLASS + '>.resizers>[data-pos]') ? $(node).attr('data-pos') : null;
    };

    // exports ================================================================

    var promise = IO.loadJSON('io.ox/office/drawinglayer/view/presetGeometries');
    promise.done(function (preset) { PRESET_GEOMETRIES = preset; });

    return promise.then(_.constant(DrawingFrame));

});
