/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/toolpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/pane'
], function (Utils, Pane) {

    'use strict';

    // class ToolPane =========================================================

    /**
     * A view pane attached to a border of the application window that adds
     * specific styling to the form controls contained in its view components.
     * Provides a leading, a centered, and a trailing area (according to the
     * position of the pane) as target nodes for the view components.
     *
     * The method ToolPane.addViewComponent() accepts a new option 'targetArea'
     * that can be set to the values 'leading', 'trailing', and 'center' (the
     * default value).
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {BaseView} docView
     *  The document view instance containing this tool pane.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  Pane.
     */
    var ToolPane = Pane.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // area nodes, mapped by type identifier
        var areaNodes = {};

        // base constructor ---------------------------------------------------

        Pane.call(this, docView, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed component into this tool pane.
         */
        function componentInserter(component, options) {

            // the target area passed in the options
            var targetArea = Utils.getStringOption(options, 'targetArea', 'center');
            // special position of component
            var specialPosition = Utils.getOption(options, 'insertBefore', false);

            // insert the component root node (fall-back to center area)
            if (!(targetArea in areaNodes)) { targetArea = 'center'; }

            if (specialPosition) {
                specialPosition.getNode().before(component.getNode());
            } else {
                areaNodes[targetArea].append(component.getNode());
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the fixed-sized root node of the specified tool pane area.
         *
         * @param {String} targetArea
         *  The identifier of the tool pane area: one of 'leading', 'trailing',
         *  and 'center'.
         *
         * @returns {jQuery}
         *  The fixed-sized root node of the specified tool pane area.
         */
        this.getAreaWrapperNode = function (targetArea) {
            return (areaNodes[targetArea] || $()).parent();
        };

        /**
         * Returns the inner and outer sizes of all tool pane areas.
         *
         * @returns {Object}
         *  An object with the following properties:
         *  - {Number} total
         *      Total available size of the tool pane.
         *  - {Object} leading
         *      Content and area size of the leading area, in the following
         *      properties:
         *      - {Number} leading.content
         *          The effective size of the area contents, in pixels.
         *      - {Number} leading.area
         *          The fixed size of the area node itself, in pixels.
         *  - {Object} center
         *      Content and area size of the center area. See the property
         *      'leading' for details.
         *  - {Object} trailing
         *      Content and area size of the trailing area. See the property
         *      'leading' for details.
         */
        this.getAreaSizes = function () {

            // the name of the size property for the area nodes
            var sizeProp = this.isVerticalPosition() ? 'width' : 'height';
            // the resulting sizes
            var areaSizes = { total: Utils.getFloorNodeSize(self.getNode())[sizeProp] };

            _.each(areaNodes, function (areaNode, areaType) {
                areaSizes[areaType] = {
                    content: Utils.getCeilNodeSize(areaNode)[sizeProp],
                    area: areaNode.parent()[sizeProp]()
                };
            });

            return areaSizes;
        };

        /**
         * Updates the sizes of the area nodes according to the sizes of their
         * content nodes.
         *
         * @returns {ToolPane}
         *  A reference to this instance.
         */
        this.updateAreaSizes = function () {

            // the name of the size property for the area nodes
            var sizeProp = this.isVerticalPosition() ? 'width' : 'height';
            // the current sizes of the area nodes
            var areaSizes = this.getAreaSizes();

            // do not let leading and trailing areas overlap (trailing pane wins over leading over center)
            areaSizes.trailing.area = Math.min(areaSizes.trailing.content, areaSizes.total);
            areaSizes.leading.area = Math.min(areaSizes.leading.content, areaSizes.total - areaSizes.trailing.area);
            areaSizes.center.area = areaSizes.total - areaSizes.leading.area - areaSizes.trailing.area;

            // set the resulting sizes at the wrapper nodes (important for top/bottom panes)
            _.each(areaNodes, function (areaNode, areaType) {
                areaNode.parent()[sizeProp](areaSizes[areaType].area);
            });

            return this;
        };

        // initialization -----------------------------------------------------

        // initialization of the DOM root node of this tool pane
        (function (rootNode) {

            // add CSS marker class for special formatting; initially hide all contents
            rootNode.addClass('tool-pane hidden-contents');

            // removes the CSS marker class that hides the pane contents during import
            function showPaneContents() {
                self.executeDelayed(function () {
                    rootNode.removeClass('hidden-contents');
                }, 1500);
            }

            // show pane contents after import, and during import in preview state
            this.waitForImport(showPaneContents);
            docView.getApp().once('docs:state:preview', showPaneContents);

            // prevent unwanted scrolling on iOS
            rootNode.on('touchmove', function (event) {
                event.preventDefault();
            });
        }.call(this, this.getNode()));

        // create component area nodes
        ['leading', 'center', 'trailing'].forEach(function (areaType) {
            areaNodes[areaType] = $('<div class="area-container">');
            self.getNode().append($('<div class="float-wrapper ' + areaType + '">').append(areaNodes[areaType]));
        });

        // insert new view components into the area nodes
        this.registerComponentInserter(componentInserter);

        // update size of the areas after the view pane has been resized
        this.on('pane:layout', function () { self.updateAreaSizes(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(areaNodes, 'remove');
            self = docView = initOptions = areaNodes = null;
        });

    } }); // class ToolPane

    // exports ================================================================

    return ToolPane;

});
