/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/popup/contextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/view/popup/compoundmenu'
], function (Utils, TimerMixin, CompoundMenu) {

    'use strict';

    // class ContextMenu ======================================================

    /**
     * An instance of this class represents a context menu bound to a specific
     * DOM element. Right-clicking that DOM element, or pressing specific keys
     * (CONTEXT key) or key combinations (e.g. Shift+F10) while the element is
     * focused or contains the browser focus will show this context menu.
     *
     * Instances of this class trigger the following additional events:
     * - 'contextmenu:prepare'
     *      After a browser event (right mouse click, keyboard shortcut, etc.)
     *      has been caught that causes to show this context menu. Event
     *      handlers receive the usual jQuery event object as first parameter,
     *      with the additional property 'sourceEvent' set to the originating
     *      jQuery event object (the mouse click or keyboard event). If an
     *      event handler calls the method preventDefault() at the event
     *      object, the context menu will not be shown.
     *
     * @constructor
     *
     * @extends CompoundMenu
     * @extends TimerMixin
     *
     * @param {BaseView} docView
     *  The document view instance containing this context menu.
     *
     * @param {HTMLObject|jQuery} sourceNode
     *  The DOM element used as event source for context events (right mouse
     *  clicks, or 'contextMenu' events for specific keyboard shortcuts). If
     *  this object is a jQuery collection, uses the first DOM element it
     *  contains.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.selector]
     *      A jQuery selector for specific descendant elements of the passed
     *      event source node. The context menu will only be shown, if the
     *      mouse or keyboard events originate from the selected nodes.
     *  @param {String|Array<String>} [initOptions.enableKey]
     *      The key of a controller item, or an array of controller item keys,
     *      that specify whether the context menu is enabled. Uses the enabled
     *      state of the controller items registered for the specified keys.
     *      All specified items must be enabled to enable the context menu. If
     *      omitted, the context menu will always be enabled.
     *  @param {Boolean} [initOptions.mouseDown=false]
     *      If set to true, the context menu will be shown instantly after the
     *      'mousedown' event for the right mouse button. If omitted or set to
     *      false, the context menu will be opened after the 'mouseup' and its
     *      related 'contextmenu' event.
     */
    function ContextMenu(docView, sourceNode, initOptions) {

        var // self reference
            self = this,

            // jQuery selector for descendant nodes this context menu is restricted to
            selector = Utils.getStringOption(initOptions, 'selector'),

            // controller keys specifying whether the context menu is enabled
            enableKeys = Utils.getOption(initOptions, 'enableKey'),

            // delay to open the context menu
            delay = Utils.getNumberOption(initOptions, 'delay'),

            // special handler to invoke
            contextMenuHandlers = [],
            detectHandlers = [],
            resetHandlers = [];

        // base constructor ---------------------------------------------------

        CompoundMenu.call(this, docView, _.extend({ anchorPadding: 0, autoHideGroups: true }, initOptions));

        // add timer methods (needs the registerDestructor() method)
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Invokes the callback function passed to the constructor, and shows
         * the context menu at the correct position.
         */
        function showContextMenu(event) {
            var // the (cancelable) jQuery prepare event
                prepareEvent = new $.Event('contextmenu:prepare', { sourceEvent: event.sourceEvent }),
                xy = self.getXY(event);

            // notify listeners (also if the context menu is disabled, e.g. to select
            // an element on click, although the context menu will not occur)
            self.trigger(prepareEvent);

            // do not show the context menu, if an event listener has vetoed, or if
            // the controller item bound to the context menu is disabled
            if (prepareEvent.isDefaultPrevented() || !self.isEnabled()) { return; }

            function executeShow() {
                // set the click position as anchor, and show the context menu
                self.hide().setAnchor({ left: xy.pageX, top: xy.pageY, width: 1, height: 1 }).show();
            }

            if (delay) {
                self.executeDelayed(executeShow, delay);
            } else {
                executeShow();
            }
        }

        /**
         * Detects some states we must know in the following event stack
         *
         * @param {String} defaultEvent
         *  The type which triggers the event (mouse, keyboard or touch)
         */
        function detectState() {
            detectHandlers.forEach(function (func) {
                func.call(this);
            }, this);
        }

        /**
         * Resets all detected states
         *
         * @param {Object} [options]
         *  Optional parameters. Used in some resetHandlers.
         */
        function resetState(options) {
            resetHandlers.forEach(function (func) {
                func.call(this, options);
            }, this);
        }

        // EVENT-Handler ------------------------------------------------------

        /**
         * Handles 'keydown' events of the source node.
         */
        function keyDownHandler() {
            detectState();
        }

        /**
         * Handles 'keyup' events of the source node.
         */
        function keyUpHandler() {
            resetState();
        }

        /**
         * Handles 'mousedown' events of the source node. Shows the context
         * menu, if the option 'mouseDown' has been passed to the constructor.
         */
        function mouseDownHandler() {
            detectState();
        }

        /**
         * Handles 'contextmenu' events of the source node, triggered after
         * right-clicking the anchor node, or pressing specific keys that cause
         * showing a context menu.
         */
        function contextMenuHandler(event) {

            // bug 38770: fail-safety: prevent context menus in busy state
            if (docView.isBusy()) { return false; }

            var sourceEvent = event.sourceEvent;

            // if x/y-coordinate not set, calculate them both by last known
            // touch position
            if (!event.pageX || !event.pageY) {
                if (!_.isNull(sourceEvent)) {
                    event.pageX = sourceEvent.pageX;
                    event.pageY = sourceEvent.pageY;

                } else {
                    var clickedElement = $(sourceEvent.target),
                        offset = clickedElement.offset(),
                        width = clickedElement.width(),
                        height = clickedElement.height();

                    event.pageX = (offset.left + (width / 2));
                    event.pageY = (offset.top + (height / 2));
                }
            }

            contextMenuHandlers.forEach(function (func) {
                func.call(self, event);
            });

            showContextMenu(event);

            event.preventDefault();
            event.stopPropagation();
            return false;
        }

        /**
         * Handles 'mousedown' events of the source node. Shows the context
         * menu, if the option 'mouseDown' has been passed to the constructor.
         */
        function mouseUpHandler() {
            resetState({ resetTextSelection: true });
        }

        /**
         * Hides the context menu if any of the controller items specified with
         * the constructor option 'enableKey' has been disabled.
         */
        function controllerChangeHandler(changedItems) {

            // nothing to do, if the context menu is not visible
            if (!self.isVisible()) { return; }

            var // whether to hide the context menu due to changed controller item states
                hideMenu = enableKeys.some(function (enableKey) {
                    return (enableKey in changedItems) && (changedItems[enableKey].enabled === false);
                });

            if (hideMenu) {
                self.hide();
                docView.grabFocus();
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether this context menu is currently enabled (whether it
         * will be shown after right-clicking the anchor node, or using the
         * keyboard).
         *
         * @returns {Boolean}
         *  Whether this context menu is currently enabled.
         */
        this.isEnabled = function () {
            var controller = docView.getApp().getController();
            return enableKeys.every(controller.isItemEnabled, controller);
        };

        /**
         * Register handler for 'contextmenu' event. Invoked in the
         * contextMenuHandler.
         *
         * @param {Function} func
         *  The function which should be invoked.
         */
        this.registerContextMenuHandler = function (func) {
            contextMenuHandlers.push(func);
        };

        /**
         * Register handler for 'detect' method. Invoked in the
         * detectState method.
         *
         * @param {Function} func
         *  The function which should be invoked.
         */
        this.registerDetectHandler = function (func) {
            detectHandlers.push(func);
        };

        /**
         * Register handler for 'reset' method. Invoked in the
         * resetState method.
         *
         * @param {Function} func
         *  The function which should be invoked.
         */
        this.registerResetHandler = function (func) {
            resetHandlers.push(func);
        };

        /**
         * Return the default pageX / pageY of the event
         *
         * @returns {Object}
         *  Object with 'pageX' and 'pageY' values
         */
        this.getXY = function (event) {
            return {
                pageX: event.pageX,
                pageY: event.pageY
            };
        };

        // initialization -----------------------------------------------------

        this.getNode().addClass('context-menu');

        // listen to relevant context events on the source node
        sourceNode.on({
            mousedown: mouseDownHandler,
            mouseup: mouseUpHandler,
            keydown: keyDownHandler,
            keyup: keyUpHandler,
            'documents:contextmenu': contextMenuHandler
        }, selector);

        // always use an array of parent keys (as strings)
        enableKeys =
            _.isArray(enableKeys) ? _.filter(enableKeys, _.isString) :
            _.isString(enableKeys) ? [enableKeys] : [];

        // hide displayed context menu, if controller item changes to disabled state
        if (enableKeys.length > 0) {
            this.handleChangedControllerItems(controllerChangeHandler);
        }

        // fix for Bug 46769
        self.on('popup:hide', function () { docView.grabFocus(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = sourceNode = initOptions = self = null;
        });

    } // class ContextMenu

    // exports ================================================================

    // derive this class from class CompoundMenu
    return CompoundMenu.extend({ constructor: ContextMenu });

});
