/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/popup/compoundmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/basemenu',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/component',
    'io.ox/office/baseframework/view/viewobjectmixin',
    'gettext!io.ox/office/baseframework/main'
], function (Utils, Forms, BaseMenu, Button, Component, ViewObjectMixin, gt) {

    'use strict';

    var // CSS class name for separator lines
        SEPARATOR_LINE_CLASS = 'separator-line',

        // CSS class name for header labels
        HEADER_LABEL_CLASS = 'header-label';

    // class CompoundMenu =====================================================

    /**
     * A pop-up menu element that contains multiple independent control groups
     * bound to different controller items. Internally, a self-contained
     * instance of a view component (class Component) is used to store the
     * control groups.
     *
     * Triggers the following events:
     * - 'menu:userclose':
     *      After the 'Close' button shown in the menu header has been clicked.
     *
     * @constructor
     *
     * @extends BaseMenu
     * @extends ViewObjectMixin
     *
     * @param {BaseView} docView
     *  The document view instance containing this menu.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class BaseMenu. Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.autoHideGroups=false]
     *      If set to true, all disabled control groups contained in this menu
     *      will be hidden automatically.
     *  @param {Function} [initOptions.lazyInitHandler]
     *      A callback function that will be invoked exactly once before this
     *      compound menu will be shown the first time.
     *  @param {String} [initOptions.actionKey]
     *      If specified, the menu will contain a footer section with an action
     *      button, and a cancel button. The action button will be bound to the
     *      controller item with the key specified in this option. The enabled
     *      state of the button element will be bound to the state of the
     *      controller item.
     *  @param {String} [initOptions.actionLabel]
     *      A different text label for the action button in the footer section.
     *      If omitted, the button will contain the label 'OK'. This option has
     *      no effect, if the option 'actionKey' has been omitted.
     *  @param {Any} [initOptions.actionValue]
     *      A value, object, or function that will be used as fixed value for
     *      the action button when it has been clicked. Must not be null. A
     *      function will be evaluated every time the button has been
     *      activated. If another control group (instance of class Group) has
     *      been passed, the current value of that group will be used and
     *      forwarded as button value instead. This option has no effect, if
     *      the option 'actionKey' has been omitted.
     *  @param {Function} [options.enableActionHandler]
     *      A predicate callback function that can be used to influence the
     *      enabled state of the action button manually. MUST return a boolean
     *      value stating whether the control group is enabled.
     */
    function CompoundMenu(docView, initOptions) {

        var // self reference
            self = this,

            // lazy initialization callback function
            lazyInitHandler = Utils.getFunctionOption(initOptions, 'lazyInitHandler'),

            // callback predicate returning whether the OK button in the footer is currently enabled
            enableActionHandler = Utils.getFunctionOption(initOptions, 'enableActionHandler'),

            // whether to hide all disabled control groups automatically
            autoHideGroups = Utils.getBooleanOption(initOptions, 'autoHideGroups', false),

            // the view component in the drop-down menu for dynamic contents
            innerComponent = null,

            // the root node of the menu footer
            footerNode = null,

            // the OK button shown in the footer
            actionButton = null,

            // the Cancel button shown in the footer
            cancelButton = null,

            // saved scroll position of inner component during 2-step layouting
            scrollTop = 0;

        // base constructors --------------------------------------------------

        BaseMenu.call(this, Utils.extendOptions(initOptions, {
            suppressScrollBars: true,
            prepareLayoutHandler: prepareLayoutHandler,
            updateLayoutHandler: updateLayoutHandler
        }));
        ViewObjectMixin.call(this, docView);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed group into the menu component.
         */
        function groupInserter(group, options) {

            var // the root node of the passed group
                groupNode = group.getNode(),
                // the target node for the passed group
                targetNode = innerComponent.getNode(),
                // the last child in the view component
                lastNode = targetNode.children().last(),
                // insert given group before this specific element
                insertBefore = Utils.getOption(options, 'insertBefore', null),
                // save the separator, if existing, before the element given in "insertBefore"
                saveSeparator = Utils.getBooleanOption(options, 'saveSeparator', false);

            if (Utils.getBooleanOption(options, 'inline', false)) {
                if (lastNode.is('.inline-container')) {
                    targetNode = lastNode;
                } else {
                    targetNode = $('<div class="inline-container">').appendTo(targetNode);
                    if (lastNode.is('.group')) { targetNode.append(lastNode); }
                }
            }

            // insert the group node
            //  - prepend
            if (Utils.getBooleanOption(options, 'prepend', false)) {
                targetNode.prepend(groupNode);

            //  - insert before specific element (save separator?)
            } else if (insertBefore instanceof $) {
                // save separator
                if (saveSeparator && insertBefore.prev().is('.' + SEPARATOR_LINE_CLASS)) { insertBefore = insertBefore.prev(); }
                // insert
                insertBefore.before(groupNode);

            //  - append (normally)
            } else {
                targetNode.append(groupNode);
            }
        }

        /**
         * Prepares updating the position and size of this menu. Removes all
         * custom scroll settings from the inner view component.
         */
        function prepareLayoutHandler() {

            // DOM node of the inner component
            var innerCompNode = innerComponent ? innerComponent.getNode() : null;
            if (!innerCompNode) { return; }

            // bug 35832: save scroll position before removing overflow-y attribute
            scrollTop = innerCompNode.scrollTop();
            innerCompNode.css({ height: '', overflowY: '' });

            // separate all nodes into sections
            var allSectionData = [], currData = null;

            function newSectionData(separator) {
                allSectionData.push(currData = { separator: separator, header: null, groups: [] });
            }

            innerCompNode.children().each(function () {
                var childNode = $(this);
                if (childNode.is('.' + SEPARATOR_LINE_CLASS)) {
                    newSectionData(childNode);
                } else if (childNode.is('.' + HEADER_LABEL_CLASS)) {
                    if (!currData || (currData.groups.length > 0)) { newSectionData(null); }
                    currData.header = childNode;
                } else if (childNode.is('.group')) {
                    if (!currData) { newSectionData(null); }
                    currData.groups.push(childNode);
                }
            });

            // update visibility of the separator lines and header labels according to visible groups
            allSectionData.forEach(function (sectionData, index) {

                // detect sections with visible groups
                sectionData.visible = sectionData.groups.some(Forms.isVisibleNode);

                // separator line needs a visible group before and one after
                if (sectionData.separator) {
                    Forms.showNodes(sectionData.separator, (index > 0) && sectionData.visible && allSectionData[index - 1].visible);
                }

                // header label needs a visible group after
                if (sectionData.header) {
                    Forms.showNodes(sectionData.header, sectionData.visible);
                }
            });
        }

        /**
         * Makes the inner component scrollable, if there is not enough space.
         */
        function updateLayoutHandler(contentSize, availableSize) {

            var // DOM node of the inner component
                innerCompNode = innerComponent ? innerComponent.getNode() : null,
                // all child nodes (set to inline-block to be able to measure their required width)
                childNodes = self.getContentNode().children().css('display', 'inline-block'),
                // the maximum required width of all children
                childWidth = _.max(_.map(childNodes.get(), function (node) { return Utils.getCeilNodeSize(node).width; })),
                // height reduction for the inner component
                heightDiff = contentSize.height - availableSize.height;

            // if the menu is larger than the available height, make the inner view component
            // scrollable, and add space for vertical scroll bar to required width of the children
            if (innerCompNode && (heightDiff > 0)) {
                var innerCompSize = Utils.getCeilNodeSize(innerCompNode);
                innerCompNode.css({ height: innerCompSize.height - heightDiff, overflowY: 'scroll' });
                // bug 35832: restore scroll position after setting overflow-y attribute
                innerCompNode.scrollTop(scrollTop);
                childWidth = Math.max(childWidth, innerCompSize.width + Utils.SCROLLBAR_WIDTH);
            }

            // remove the explicit inline-block style from all children
            childNodes.css('display', '');

            // restrict the content node to the available width
            return { width: Math.min(childWidth, availableSize.width) };
        }

        // public methods -----------------------------------------------------

        /**
         * Appends the passed control group to the pop-up menu and binds it to
         * the specified controller item.
         *
         * @param {String|Null} key
         *  The key of a controller item associated to the control group. The
         *  enabled state and the current value of the group will be bound to
         *  the state and value of the controller item. If the group triggers a
         *  'group:change' event, the controller item will be executed. The key
         *  can be set to the value null to insert an unbound group into this
         *  pop-up menu.
         *
         * @param {Group} group
         *  The control group object to be inserted into the pop-up menu.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  Component.addGroup(), and the following additional options:
         *  @param {Boolean} [options.inline=false]
         *      Inserts the group node into the same menu row as the preceding
         *      inline group. If this group is the first inline group, it will
         *      be inserted into a new menu row.
         *  @param {Boolean} [options.sticky=false]
         *      If set to true, this pop-up menu will not be closed after the
         *      inserted group has been activated.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.addGroup = function (key, group, options) {

            // set the root node of the embedded view component as focus target node for sticky groups
            if (Utils.getBooleanOption(options, 'sticky', false)) {
                options = _.extend({ focusTarget: this.getContentNode() }, options);
                delete options.sticky;
            }

            // do not close this menu, if a sub menu from the component is opened
            if (_.isFunction(group.getMenuNode)) {
                this.registerFocusableNodes(group.getMenuNode());
            }

            // insert the passed group
            innerComponent.addGroup(key, group, options);
            return this;
        };

        /**
         * Removes the specified group from this pop-up menu, if it has been
         * registered before with the method CompoundMenu.addGroup(). The group
         * will be unregistered and removed, but it will NOT be destroyed.
         * Caller MUST take ownership of the passed group.
         *
         * @param {Group} group
         *  The control group object to be removed.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.removeGroup = function (group) {
            innerComponent.removeGroup(group);
            return this;
        };

        /**
         * Appends a separator line to the pop-up menu.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.addSeparator = function () {
            innerComponent.getNode().append('<div class="' + SEPARATOR_LINE_CLASS + '">');
            return this;
        };

        /**
         * Appends a header label to the pop-up menu. If the menu is not empty
         * anymore, automatically adds a separator line above the header label.
         *
         * @param {String} label
         *  The label text.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.addSectionLabel = function (label, options) {

            var // whether to add a separator line
                separator = Utils.getBooleanOption(options, 'separator', true);

            if (separator && innerComponent.getNode()[0].hasChildNodes()) {
                this.addSeparator();
            }

            innerComponent.getNode().append('<div class="' + HEADER_LABEL_CLASS + '" aria-hidden="true">' + Utils.escapeHTML(label) + '</div>');
            return this;
        };

        /**
         * Visits all control groups registered at this pop-up menu.
         *
         * @param {Function} iterator
         *  The iterator called for each control group. Receives the reference
         *  to the control group. If the iterator returns the Utils.BREAK
         *  object, the iteration process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateGroups = function (iterator, context) {
            return innerComponent ? innerComponent.iterateGroups(iterator, context) : undefined;
        };

        /**
         * Returns whether the view component contained in this pop-up menu
         * contains visible groups.
         *
         * @returns {Boolean}
         *  Whether this pop-up menu contains visible groups.
         */
        this.hasVisibleGroups = function () {
            return _.isObject(innerComponent) && innerComponent.hasVisibleGroups();
        };

        /**
         * Updates the visibility, enabled state, and value of all groups in
         * this pop-up menu.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.updateAllGroups = function () {
            innerComponent.updateAllGroups();
            return this;
        };

        /**
         * Removes and destroys all control groups from this pop-up menu, and
         * completely clears its root node.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.destroyAllGroups = function () {
            innerComponent.destroyAllGroups();
            return this;
        };

        /**
         * Releases ownership of the current view component used as container
         * for the form controls shown in this pop-up menu, and detaches its
         * root node from the content node of this menu.
         *
         * @returns {Component|Null}
         *  The released view component. Caller MUST take ownership!
         */
        this.releaseComponent = function () {

            if (!_.isObject(innerComponent)) { return null; }

            // stop inserting new groups into the component
            innerComponent.unregisterGroupInserter(groupInserter);

            // stop listening to any component events
            this.stopListeningTo(innerComponent);

            // remove component root node from this menu
            innerComponent.getNode().detach();

            // save reference to the component as method result
            var oldComponent = innerComponent;
            innerComponent = null;
            return oldComponent;
        };

        /**
         * Replaces the current view component with another view component. The
         * old view component will be destroyed.
         *
         * @param {Object} newComponent
         *  The new view component to be inserted. This menu instance takes
         *  ownership of the passed view component!
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.replaceComponent = function (newComponent) {

            // if it's already there, get out here
            if (innerComponent === newComponent) { return this; }

            // release (unregister) the current view component
            var oldComponent = this.releaseComponent();

            // destroy old component (removes itself from the DOM)
            if (_.isObject(oldComponent)) {
                oldComponent.destroy();
            }

            // register the new view component
            innerComponent = newComponent;
            if (footerNode) {
                innerComponent.getNode().insertBefore(footerNode);
                // keep OK button in the footer section up-to-date
                this.listenTo(innerComponent, 'component:change', this.updateFooter.bind(this));
            } else {
                this.appendContentNodes(innerComponent.getNode());
            }

            // insert new groups into the component
            innerComponent.registerGroupInserter(groupInserter);

            // resize the pop-up node when the contents of the view component have been changed
            this.listenTo(innerComponent, 'component:layout', function () { self.refreshImmediately(); });

            return this;
        };

        /**
         * Updates the OK/Cancel buttons contained in the footer, if existing.
         *
         * @returns {CompoundMenu}
         *  A reference to this instance.
         */
        this.updateFooter = function () {

            // enable/disable the OK button according to the callback function
            if (actionButton) {
                var enabled = !enableActionHandler || (enableActionHandler.call(self) === true);
                Forms.enableBSButton(actionButton, enabled);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // do not allow to clear the menu manually
        delete this.clearContents;

        // call lazy initialization handler before showing this menu
        if (_.isFunction(lazyInitHandler)) {
            this.one('popup:beforeshow', function () {
                lazyInitHandler.call(self);
                innerComponent.updateAllGroups();
            });
        }

        // update OK/Cancel buttons after showing the menu
        this.on('popup:show popup:idle', function () {
            self.updateFooter();
            if (cancelButton) {
                Forms.enableBSButton(cancelButton, true);
            }
        });

        // create the header and footer sections
        (function () {

            // the controller key for the OK button in the footer section
            var actionKey = Utils.getStringOption(initOptions, 'actionKey', '');
            if (actionKey.length === 0) { return; }

            function actionHandler() {
                var actionValue = Utils.getOption(initOptions, 'actionValue');
                if (_.isFunction(actionValue)) { actionValue = actionValue.call(self); }
                docView.executeControllerItem(actionKey, actionValue);
            }

            // close the menu when hitting the cancel button
            function cancelHandler() {
                self.hide();
                docView.grabFocus();
            }

            // the action and cancel button controls
            actionButton = $.button({ label: Utils.getStringOption(initOptions, 'actionLabel', gt('OK')), click: actionHandler }).addClass('btn-primary');
            cancelButton = $.button({ label: gt('Cancel'), click: cancelHandler }).addClass('btn-default');

            // create the footer section node with OK and Cancel button
            footerNode = $('<div class="footer-section">').append(cancelButton, actionButton);
            self.appendContentNodes(footerNode);
        }());

        // create and insert the inner view component into the drop-down menu
        this.replaceComponent(new Component(docView, { autoHideGroups: autoHideGroups }));

        // hide pop-up menu when application is not active
        this.listenTo(docView.getApp().getWindow(), 'hide', function () { self.hide(); });

        // additional cursor key shortcuts for focus navigation
        Forms.enableCursorFocusNavigation(this.getNode(), { homeEnd: true });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            if (_.isObject(innerComponent)) { innerComponent.destroy(); }
            self = docView = initOptions = lazyInitHandler = innerComponent = null;
            footerNode = actionButton = cancelButton = null;
        });

    } // class CompoundMenu

    // exports ================================================================

    // derive this class from class BaseMenu
    return BaseMenu.extend({ constructor: CompoundMenu });

});
