/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/baseframework/utils/apptooltipmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/tooltip'
], function (Utils, Forms, ToolTip) {

    'use strict';

    var MENUSELECTOR = '>.io-ox-office-main.popup-menu ';

    // mix-in class AppTooltipMixin ===========================================

    function AppTooltipMixin() {

        var // self reference
            self = this,

            //
            popup = new ToolTip({
                anchorAlign: 'center',
                anchorBorder: 'bottom top',
                classes: 'bootstraptooltip'
            });

        // private methods ----------------------------------------------------

        function mouseEnter(evt) {
            var holder = $(evt.target);
            if (!holder.hasClass(Forms.APP_TOOLTIP_CLASS)) {
                holder = $(evt.currentTarget);
                //TODO: info text could be a combination of currentTarget & target
            }
            if (!holder.hasClass(Forms.APP_TOOLTIP_CLASS)) {
                return;
            }

            popup.setAnchor(holder);
            popup.setText(holder.attr('data-original-title'));
            popup.show();

            var node = popup.getNode();
            var arrow = node.find('.arrow');

            var holderOffset = holder.offset();
            var popupOffset = node.offset();

            arrow.css('left', Math.round(holderOffset.left + (holder.width() / 2) - popupOffset.left));
            arrow.toggleClass('on-top', holderOffset.top > popupOffset.top);
        }

        function mouseLeave() {
            popup.hide();
        }

        function visibleStateHandler(visible) {
            mouseLeave();

            var rootNode = self.getRootNode();
            var body = $('body');

            var onOff = visible ? 'on' : 'off';

            rootNode[onOff]('mouseenter', Forms.APP_TOOLTIP_SELECTOR, mouseEnter);
            rootNode[onOff]('mouseleave mouseup', Forms.APP_TOOLTIP_SELECTOR, mouseLeave);

            body[onOff]('mouseenter', MENUSELECTOR + Forms.APP_TOOLTIP_SELECTOR, mouseEnter);
            body[onOff]('mouseleave mouseup', MENUSELECTOR + Forms.APP_TOOLTIP_SELECTOR, mouseLeave);
        }

        // public methods -----------------------------------------------------

        /**
         * shows the tooltip of the assigned menu directly after show
         * and disables the tooltip after hide
         *
         * @param {BasePopup} menu
         *
         * @param {String} tooltip
         *
         * @param {Object} [options]
         *  Options to control some special tool tip behaviors.
         *  The following options are supported:
         *  @param {String} [options.anchorBorders='bottom top']
         *      By default, the anchorBorders are 'bottom top'. In some cases we
         *      need the opposite way. Here you can change it as desired.
         */
        this.enableTooltipOnShow = function (menu, tooltip, options) {
            var anchorBorders = Utils.getStringOption(options, 'anchorBorders', 'bottom top');

            popup.setAnchorBorders(anchorBorders.split(' '));

            menu.one('popup:show', function () {
                var node = menu.getNode();
                Forms.setToolTip(node, { tooltip: tooltip, forceTouch: true });

                mouseEnter({ currentTarget: node[0] });

                menu.one('popup:hide', function () {
                    Forms.disableToolTip(node);
                });
            });
        };

        // initialization -----------------------------------------------------

        this.waitForImport(function () {
            var node = popup.getNode();
            var arrow = $('<div class="arrow">');
            node.append(arrow);
            this.registerVisibleStateHandler(visibleStateHandler);
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            visibleStateHandler(false);
            popup.destroy();

            self = popup = null;
        });

    } // class AppTooltipMixin

    // exports ================================================================

    return AppTooltipMixin;

});
