/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* global blankshield */

define('io.ox/office/baseframework/app/baseapplication', [
    'io.ox/core/notifications',
    'io.ox/core/tk/sessionrestore',
    'io.ox/mail/api',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/view/baselabels',
    'io.ox/office/baseframework/utils/errorcode',
    'io.ox/office/baseframework/utils/errormessages',
    'io.ox/office/baseframework/utils/apptooltipmixin',
    'io.ox/office/baseframework/app/applicationlauncher',
    'io.ox/office/baseframework/app/appobjectmixin',
    'settings!io.ox/office',
    'gettext!io.ox/office/baseframework/main'
], function (Notifications, SessionRestore, MailApi, Utils, IO, DriveUtils, BaseObject, TimerMixin, BaseLabels, ErrorCode, ErrorMessages, AppTooltipMixin, ApplicationLauncher, AppObjectMixin, Settings, gt) {

    'use strict';

    // map with the identifiers of all running instances of BaseApplication
    var runningAppMap = {};

    // private static functions ===============================================

    /**
     * Returns the close timeout hash fragment from the application URL.
     * This value is limited to 600 and float values will be rounded up.
     *
     * @returns {Number|Null}
     *  The delay time after the application will be closed automatically, in
     *  milliseconds.
     */
    function getAutoCloseDelay() {
        var appCloseTimeout = _.url.hash('closetimer');
        if (!$.isNumeric(appCloseTimeout)) { return null; }
        return Math.min(600, Math.ceil(appCloseTimeout)) * 1000;
    }

    // class BaseApplication ==================================================

    /**
     * A mix-in class that defines common public methods for an application
     * that is based on a document file.
     *
     * @constructor
     *
     * @extends ox.ui.App
     * @extends BaseObject
     * @extends TimerMixin
     * @extends AppObjectMixin
     *
     * @param {Function} ModelClass
     *  The constructor function of the document model class. MUST derive from
     *  the class Model. Receives a reference to this application instance.
     *  MUST NOT use the methods BaseApplication.getModel(),
     *  BaseApplication.getView(), or BaseApplication.getController() during
     *  construction. For further initialization depending on valid
     *  model/view/controller instances, the constructor can use the method
     *  BaseApplication.onInit().
     *
     * @param {Function} ViewClass
     *  The constructor function of the view class. MUST derive from the class
     *  View. Receives a reference to this application instance. MUST NOT use
     *  the methods BaseApplication.getModel(), BaseApplication.getView(), or
     *  BaseApplication.getController() during construction. For further
     *  initialization depending on valid model/view/controller instances, the
     *  constructor can use the method BaseApplication.onInit().
     *
     * @param {Function} ControllerClass
     *  The constructor function of the controller class. MUST derive from the
     *  class Controller. Receives a reference to this application instance.
     *  MUST NOT use the methods BaseApplication.getModel(),
     *  BaseApplication.getView(), or BaseApplication.getController() during
     *  construction. For further initialization depending on valid
     *  model/view/controller instances, the constructor can use the method
     *  BaseApplication.onInit().
     *
     * @param {Function} importHandler
     *  A function that will be called to import the document described in the
     *  file descriptor of this application. Will be called once after
     *  launching the application. Will be called in the context of this
     *  application. Must return a promise that will be resolved or rejected
     *  after the document has been loaded.
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method) that
     *  determine the actions to perform during application launch. The
     *  supported options are dependent on the actual application type. The
     *  following standard launch options are supported:
     *  @param {String} launchOptions.action
     *      Controls how to connect the application to a document file.
     *  @param {Object} [launchOptions.file]
     *      The descriptor of the file to be imported. May be missing for
     *      specific actions, e.g. when creating a new file (see constructor
     *      option 'initOptions.initFileHandler' for more details).
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Function} [initOptions.initFileHandler]
     *      A function that will be called before importing the document, used
     *      to initialize the file descriptor. Will be called once after
     *      launching the application. Will be called in the context of this
     *      application. Must return a promise that will be resolved with the
     *      new file descriptor, or rejected on any error.
     *  @param {Function} [initOptions.flushHandler]
     *      A function that will be called before the document will be accessed
     *      at its source location, e.g. before downloading it, printing it, or
     *      sending it as e-mail. Will be called in the context of this
     *      application. Receives a string identifier as first parameter,
     *      specifying the cause of the flush request:
     *      - 'download': The document will be prepared for downloading or
     *          printing.
     *      - 'email': The document will be prepared to be sent as e-mail.
     *      May return a promise. In this case, the application will wait until
     *      this promise has been resolved or rejected. In the latter case, all
     *      subsequent actions will be skipped.
     *  @param {Funtion} [initOptions.messageDataHandler]
     *      A function that will be called before the message data is used
     *      to display a message. This enables different applications to
     *      extend the message data in their needed way. Currently this is used
     *      to set actions for specific messages.
     */
    function BaseApplication(ModelClass, ViewClass, ControllerClass, importHandler, launchOptions, initOptions) {

        // self reference
        var self = this;

        // the application window
        var appWindow = this.getWindow();

        // root DOM node of the entire application
        var winRootNode = appWindow.nodes.outer;

        // the document model instance
        var docModel = null;

        // application view: contains panes, tool bars, etc.
        var docView = null;

        // the controller instance as single connection point between model and view
        var docController = null;

        // base context object, to let this instance act like a BaseObject
        var baseContext = new BaseObject();

        // file descriptor of the document edited by this application
        var file = Utils.getObjectOption(launchOptions, 'file', null);

        // callback to flush the document before accessing its source
        var initFileHandler = Utils.getFunctionOption(initOptions, 'initFileHandler');

        // callback to flush the document before accessing its source
        var flushHandler = Utils.getFunctionOption(initOptions, 'flushHandler', _.noop);

        // all registered before-quit handlers
        var beforeQuitHandlers = [];

        // all registered quit handlers
        var quitHandlers = [];

        // a deferred object representing the initialization state (pending: early application construction, resolved: regular application runtime)
        var initDef = $.Deferred();

        // a deferred object representing the import state (pending: before import, resolved/rejected: import already started)
        var importStartDef = $.Deferred();

        // a deferred object representing the import state (pending: importing, resolved/rejected: import finished with state)
        var importFinishDef = $.Deferred();

        // time stamp when the application launcher has started
        var launchStartTime = 0;

        // collector for timer log
        var timerLogCollector = {};

        // Registry/Index/Map for document converter functionality.
        var converterRegistry = {};

        // application auto close timeout in seconds
        var autoCloseDelay = getAutoCloseDelay();

        // a deferred object of the current call to Application.quit()
        var currentQuitDef = null;

        // identifier for a reason while quitting this application (passed to quit handlers)
        var quitReason = null;

        // error state of the application
        var errorState = new ErrorCode();

        // info state of the application (the last info state)
        var infoState = null;

        // private methods ----------------------------------------------------

        /**
         * Updates the application title according to the current file name. If
         * the application does not contain a file descriptor, shows the
         * localized word 'Unnamed' as title.
         */
        function updateTitle() {

            // the title for the application launcher and browser window
            var title = self.getShortFileName() || gt('unnamed');

            // set title for application launcher and browser tab/window
            self.setTitle(title);
            appWindow.setTitle(title);

            // add data from file descriptor to application root node, for Selenium testing
            if (file) {
                if (file.folder_id) { self.setRootAttribute('data-file-folder', file.folder_id); }
                if (file.id) { self.setRootAttribute('data-file-id', file.id); }
                if (file.filename) { self.setRootAttribute('data-file-name', file.filename); }
                if (file.version) { self.setRootAttribute('data-file-version', file.version); }
                if (file.source) { self.setRootAttribute('data-file-source', file.source); }
                if (file.attachment) { self.setRootAttribute('data-file-attachment', file.attachment); }
                if (file['com.openexchange.realtime.resourceID']) { self.setRootAttribute('data-file-resourceID', file['com.openexchange.realtime.resourceID']); }
            }
        }

        /**
         * Updates the browser URL according to the current file descriptor.
         */
        function updateUrl() {
            // only for files from OX Drive, or from OX Documents Portal
            if (DriveUtils.isDriveFile(file) && self.isActive()) {
                self.setState({ folder: file.folder_id, id: file.id });
            }
        }

        /**
         * Calls all handler functions contained in the passed array, and
         * returns a promise that accumulates the result of all handlers.
         * Passes all additional function parameters to the passed callback
         * handlers.
         */
        function callHandlers(handlers) {

            var // arguments to be passed to the handlers
                args = _.toArray(arguments).slice(1),
                // execute all handlers and store their results in an array
                results = _.map(handlers, function (handler) { return handler.apply(self, args); });

            // accumulate all results into a single promise
            return $.when.apply($, results);
        }

        /**
         * Handler for the 'unload' window events triggered when reloading or
         * closing the entire page. Executes the quit handlers, but not the
         * before-quit handlers.
         */
        function unloadHandler() {
            self.executeQuitHandlers('unload');
        }

        /**
         * Downloads the document file currently opened by this application.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.convert]
         *      If specified, the document will be converted to the requested
         *      file format (for example, the value 'pdf' will convert the
         *      document to PDF). If omitted, the document will be downloaded
         *      in its native file format.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the document file has been
         *  downloaded successfully, or rejected otherwise.
         */
        function download(options) {

            // creates and returns a rejected promise, containing the specified error cause
            function reject(cause) {
                return $.Deferred().reject({ cause: cause });
            }

            // nothing to do, if no file descriptor is available (internal error, server error, etc.)
            if (!self.hasFileDescriptor()) { return reject('missingfile'); }

            // document import must have been finished successfully
            if (!self.isImportSucceeded()) { return reject('importfailed'); }

            var // the promise that will be resolved with the download URL
                downloadPromise = null,
                // the pop-up window used to download the file
                popupWindow = null,
                // the target file format
                fileFormat = Utils.getStringOption(options, 'convert', 'native'),
                // whether to open a converted copy of the document in a pop-up window
                convert = fileFormat !== 'native';

            // open pop-up window early to prevent browser pop-up blockers (bug 28251)
            if (convert) {
                popupWindow = blankshield.open('about:blank', '_blank');
                window.focus();

                // browser plug-ins may still prevent opening pop-up windows
                if (!popupWindow) { return reject('popupblocker'); }
            }

            // block application window while waiting for the file URL
            docView.enterBusy({ immediate: true });

            // first, call the flush handler passed to the application constructor
            downloadPromise = $.when(flushHandler.call(self, 'download'));

            // check availability of document converter, if file will be converted to another format
            if (convert) {

                // check for document converter
                downloadPromise = downloadPromise.then(function () {
                    return self.requireServerFeatures('documentconverter');
                });

                // convert rejected promise with an appropriate error code
                downloadPromise = downloadPromise.then(null, function () {
                    return reject('docconverter');
                });
            }

            // propagate changed file to the files API (bug 30246: not for mail/task attachments!)
            downloadPromise = downloadPromise.then(function () {
                if (DriveUtils.isDriveFile(file)) {
                    return DriveUtils.propagateChangeFile(file);
                }
            });

            // download the file in the pop-up window opened before
            downloadPromise = downloadPromise.then(function () {

                // additional parameters inserted into the URL
                var params = {
                    action: 'getdocument',
                    documentformat: fileFormat,
                    mimetype: file.file_mimetype ? encodeURIComponent(file.file_mimetype) : '',
                    nocache: _.uniqueId() // needed to trick the browser cache (is not evaluated by the backend)
                };

                // the resulting download URL (bug 36734: use current version)
                var downloadURL = self.getServerModuleUrl(IO.CONVERTER_MODULE_NAME, params, { currentVersion: true });

                // download file into the pop-up window, show/focus it
                if (popupWindow) {
                    popupWindow.location = downloadURL;
                    popupWindow.focus();
                    return;
                }

                // download the resulting file
                return require(['io.ox/core/download']).then(function (Download) {
                    return Download.url(downloadURL);
                });
            });

            // hide the pop-up window, if URL is not available
            if (popupWindow) {
                downloadPromise.fail(function () {
                    popupWindow.close();
                    docView.grabFocus();
                });
            }

            // leave busy mode after download
            downloadPromise.always(function () {
                docView.leaveBusy();
            });

            return downloadPromise;
        }

        /**
         * Core implementation for closing this application.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the application has been
         *  closed completely.
         */
        function quitApplication() {

            // call all before-quit handlers (rejecting one will resume application)
            var beforeQuitPromise = callHandlers(beforeQuitHandlers, quitReason);

            // close application: run quit handlers, do further cleanup
            beforeQuitPromise.done(function () {

                // abort all running server requests, before invoking the quit handlers
                self.trigger('docs:beforequit');

                // Bug 32989: Create a local copy of the quit handler array, and clear the
                // original array, before executing the quit handlers. This allows a quit
                // handler to register other special quit handlers that will be executed e.g.
                // when calling app.executeQuitHandlers() from logout while the quit handlers are
                // still running.
                var quitPromise = callHandlers(quitHandlers.splice(0), quitReason);

                // do not care about the result of the quit handlers
                quitPromise.always(function () {

                    // remove unload listener of the browser window
                    $(window).off('unload', unloadHandler);

                    // destroy MVC instances
                    docController.destroy();
                    docView.destroy();
                    docModel.destroy();

                    // invoke all own destructor callbacks
                    baseContext.destroy();

                    // bug 32948: 'currentQuitDef' must not be cleared to keep the application in 'quitting' state, needed in
                    // case another app.quit() will be called, e.g. from 'Reload document' after internal error while quitting
                    self = appWindow = docModel = docView = docController = baseContext = null;
                    importHandler = flushHandler = beforeQuitHandlers = quitHandlers = null;

                    // remove application from global cache for type detection
                    delete runningAppMap[this.cid];

                    // always clean custom URL hash params on quit
                    _.url.hash('standalone', null);
                    _.url.hash('closetimer', null);

                    // always resolve (really close the application), regardless
                    // of the result of the quit handlers
                    currentQuitDef.resolve();
                });
            });

            // resume application: reject quit promise to indicate that the application remains alive
            beforeQuitPromise.fail(function () {
                currentQuitDef.reject();
            });

            return currentQuitDef.promise();
        }

        // handle mail composition block --------------------------------------

        function handleMailCompositionFailure(message) {
            if (!_.isString(message)) {
                var
                    cause = self.isImportSucceeded() ? 'unknown' : 'importfailed';

                switch (cause) {
                    case 'importfailed':
                        message = gt('The document was not loaded successfully. Send as mail is not possible.');
                        break;
                    default:
                        message = gt('The server does not answer the request. Send as mail is not possible.');
                }
            }
            docView.yell({ type: 'error', headline: BaseLabels.SEND_AS_MAIL_LABEL, message: (message || null) });
        }

        /**
         * Creates a new mail message with the current document attached as file.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the mail application has been
         *  launched successfully, or rejected otherwise.
         */
        function handleMailCompositionForFileAttachment() {

            function reply(mail, attachment) {
                return ox.registry.call('mail-compose', 'replyall', mail).then(function (compose) {
                    if (!compose) {
                        Utils.warn('compose element is not present, when mail is retriggered for the same file');
                        return;
                    }
                    // Bug 40376 could not send replay mail with attachment vecause of missing group
                    // and because of missing source (fixed in mailactions)
                    compose.app.model.attachFiles([_.extend(attachment, { group: 'file' })]);
                });
            }
            var attach = _.clone(file);

            // delete version number, that preview of the attached file is "current version"
            // Bug 45775
            delete attach.version;

            if (attach.source && attach.source === 'mail') {

                // bug 32760: special handling, if file is an attachment, e.g. of a mail or task
                // TODO: propably deprecated
                MailApi.get({ id: attach.id, folder_id: attach.folder_id }).done(function (mail) {
                    return reply(mail, mail.attachments[1]);
                });

            } else if (_.isObject(attach.origin) && attach.origin.source === 'mail') {

                // 34300: mail attachments copied to Drive
                MailApi.get({ id: attach.origin.id, folder_id: attach.origin.folder_id }).done(function (mail) {
                    return reply(mail, attach);
                });
            } else {
                return ox.registry.call('mail-compose', 'compose', { infostore_ids: [attach] });
            }
        }

        /**
         *
         */
        function handleMailCompositionForPDFAttachment() {
            // open mail compose
            var
                promisedAttachmentData = (self.canConvertTo('pdf-mail-attachment') && self.convertTo('pdf-mail-attachment')) || $.when({
                    id:         '',
                    filename:   ''
                });

            promisedAttachmentData.done(function (data) {
              //Utils.info('+++ handleMailCompositionForPDFAttachment :: DONE - data : ', data);
                var
                    folderId  = DriveUtils.getStandardTrashFolderId(),
                    fileId    = String(data.id),
                    fileName  = String(data.filename),

                    attachment  = {

                        folder_id:  folderId,
                        id:         fileId,
                        filename:   fileName,

                        is_delete_after_mail_compose: true
                    };

                return ox.registry.call('mail-compose', 'compose', { infostore_ids: [attachment] });
            });

            promisedAttachmentData.fail(function (msg) {
              //Utils.info('+++ handleMailCompositionForPDFAttachment :: FAIL - msg : ', msg);
                var
                    message = Utils.getStringOption(msg, 'text', null);
                if (message) {
                    handleMailCompositionFailure(message);
                }
            });

            return promisedAttachmentData;
        }

        /**
         * Creates a new mail message with this current document's content
         * being the content of the new mail's body.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the mail application has been
         *  launched successfully, or rejected otherwise.
         */
        function handleMailCompositionForInlineHTML() {
            // open mail compose
            var
                promisedDocument = (self.canConvertTo('inline-html-document') && self.convertTo('inline-html-document')) || $.when({
                    title:   '',
                    content: ''
                }),
                messageFormat = 'alternative'; //  messageFormat = MailSettings.get('messageFormat'); // HOTFIX due to module removal

            return promisedDocument.then(function (document) {
                var
                    options = {
                        subject: document.title,
                        attachments: {
                            html: [{ content: document.content }]
                        }
                        //  gets assigned explicitly only in case a user has
                        //  a 'plain text' mail message format preference
                        //
                        //  editorMode: 'html',
                        //  preferredEditorMode: 'html'
                    };
                if (messageFormat === 'text') {
                    options.editorMode = options.preferredEditorMode = 'html';
                }
                return ox.registry.call('mail-compose', 'compose', options);

            }).fail(function (msg) {
                var
                    message = Utils.getStringOption(msg, 'text', null);
                if (message) {
                    handleMailCompositionFailure(message);
                }
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns a globally unique identifier for this application.
         *
         * @returns {String}
         *  A globally unique identifier for this application that includes the
         *  session ID.
         */
        this.getGlobalUid = function () {
            return ox.session + ':' + this.get('uniqueID');
        };

        /**
         * Returns the document type supported by this application. The
         * document type equals the directory name of the application.
         *
         * @returns {String}
         *  The document type of this application.
         */
        this.getDocumentType = function () {
            return ApplicationLauncher.getDocumentType(this.getName());
        };

        /**
         * Returns the document model instance of this application.
         *
         * @returns {DocumentModel}
         *  The document model instance of this application.
         */
        this.getModel = function () {
            return docModel;
        };

        /**
         * Returns the view instance of this application.
         *
         * @returns {View}
         *  The view instance of this application.
         */
        this.getView = function () {
            return docView;
        };

        /**
         * Returns the controller instance of this application.
         *
         * @returns {Controller}
         *  The controller instance of this application.
         */
        this.getController = function () {
            return docController;
        };

        /**
         * Returns the specific error state of this application.
         *
         * @returns {ErrorCode}
         */
        this.getErrorState = function () {
            return errorState;
        };

        /**
         * Sets the error state of this application.
         *
         * @param {ErrorCode} errorCode
         *  The new error state to be set for this application.
         */
        this.setErrorState = function (errorCode) {
            errorState = errorCode;
        };

        /**
         * Sets the info state of this application.
         *
         * @returns {InfoState}
         *  The info state of this application.
         */
        this.getInfoState = function () {
            return infoState;
        };

        /**
         * Sets the info state of this application.
         *
         * @param {InfoState} newInfoState
         *  The new info state to be set for this application.
         */
        this.setInfoState = function (newInfoState) {
            infoState = newInfoState;
        };

        /**
         * Invokes the passed callback function as soon as this application has
         * finished its early construction phase. During early construction,
         * the model, view, and controller instances may not be available yet.
         * If this method is used after early construction, the passed callback
         * function will be invoked immediately.
         *
         * @attention
         *  Should be used in all class initialization code relying on existing
         *  model, view, and controller instances.
         *
         * @param {Function} callback
         *  The callback function invoked after the application has finished
         *  its early initialization.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.onInit = function (callback, context) {
            initDef.done(_.bind(callback, context));
            return this;
        };

        /**
         * Returns the global user settings for all applications of the same
         * type.
         *
         * @param {String} key
         *  The unique key of the user setting.
         *
         * @param {Any} defValue
         *  The default value in case the user setting does not exist yet.
         *
         * @returns {Any}
         *  The value of the global user setting.
         */
        this.getUserSettingsValue = function (key, defValue) {
            return Settings.get(this.getDocumentType() + '/' + key, defValue);
        };

        /**
         * Changes a global user setting for all applications of the same type.
         *
         * @param {String} key
         *  The unique key of the user setting.
         *
         * @param {Any} value
         *  The new value of the user setting.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.setUserSettingsValue = function (key, value) {
            Settings.set(this.getDocumentType() + '/' + key, value).save();
            return this;
        };

        /**
         * Returns whether this application has been launched in stand-alone
         * mode.
         *
         * @returns {Boolean}
         *  Whether this application has been launched in stand-alone mode.
         */
        this.isStandalone = function () {
            return ApplicationLauncher.getStandaloneMode();
        };

        /**
         * Returns the absolute start time of the launcher.
         *
         * @returns {Number}
         *  The launcher start time.
         */
        this.getLaunchStartTime = function () {
            return launchStartTime;
        };

        /**
         * Returns the collector for all timers.
         *
         * @returns {Object}
         *  The collector for timer logs.
         */
        this.getTimerLogCollector = function () {
            return timerLogCollector;
        };

        /**
         * Returns a promise that is pending as long as the document import has
         * not been started, and that will be resolved before importing the
         * document.
         *
         * @returns {jQuery.Promise}
         *  A promise that is pending as long as the document import has not
         *  been started.
         */
        this.getImportStartPromise = function () {
            return importStartDef.promise();
        };

        /**
         * Returns the import promise that is pending as long as the document
         * will be imported, and that will be resolved after successful import,
         * or rejected after import has failed.
         *
         * @returns {jQuery.Promise}
         *  A promise that is pending as long as the document import is not
         *  finished.
         */
        this.getImportFinishPromise = function () {
            return importFinishDef.promise();
        };

        /**
         * Returns whether the application contains pending document changes
         * not yet sent to the server. Intended to be overwritten on demand by
         * sub classes.
         *
         * @attention
         *  Do not rename this method. The OX core uses it to decide whether to
         *  show a warning before the browser refreshes or closes the page.
         *
         * @returns {Boolean}
         *  Whether the application contains pending document changes.
         */
        this.hasUnsavedChanges = function () {
            return false;
        };

        /**
         * Checks if application is processing before-quit or quit handlers.
         *
         * @returns {Boolean}
         *  Whether the application is currently processing any before-quit or
         *  quit handlers.
         */
        this.isInQuit = function () {
            return currentQuitDef !== null;
        };

        // application window object ------------------------------------------

        /**
         * Returns whether this application is currently active in OX AppSuite.
         *
         * @returns {Boolean}
         *  Whether this application is currently active in OX AppSuite.
         */
        this.isActive = function () {
            return !!appWindow.state.visible;
        };

        /**
         * Returns the root DOM node of the entire application window, in
         * difference to the method ox.ui.App.getWindowNode() which returns the
         * body node of the application window (a descendant element of the
         * window root node).
         *
         * @returns {jQuery}
         *  The root DOM node of the application window, as jQuery object.
         */
        this.getRootNode = function () {
            return winRootNode;
        };

        /**
         * Returns the unique identifier of the root DOM node of this
         * application.
         *
         * @returns {String}
         *  The unique window DOM identifier.
         */
        this.getWindowId = function () {
            return winRootNode.attr('id');
        };

        /**
         * Changes the value of a DOM element attribute of the root node of
         * this application.
         *
         * @param {String} name
         *  The name of the attribute to be changed.
         *
         * @param {String|gNumber|Boolean} value
         *  The attribute value.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.setRootAttribute = function (name, value) {
            winRootNode.attr(name, value);
            return this;
        };

        // file descriptor ----------------------------------------------------

        /**
         * Returns whether this application contains a valid file descriptor.
         *
         * @returns {Boolean}
         *  Whether this application contains a valid file descriptor.
         */
        this.hasFileDescriptor = function () {
            return _.isObject(file);
        };

        /**
         * Returns a clone of the file descriptor of the document edited by
         * this application.
         *
         * @returns {Object}
         *  A clone of the current file descriptor.
         */
        this.getFileDescriptor = function () {
            return _.clone(file);
        };

        /**
         * Returns whether this application instance works on the same document
         * file as represented by the passed file descriptor.
         *
         * @param {Object} fileDesc
         *  An arbitrary file descriptor, as accepted by the constructor of
         *  this class.
         *
         * @returns {Boolean}
         *  Whether this application instance works on the same document file
         *  as represented by the passed file descriptor.
         */
        this.isEqualFileDescriptor = function (fileDesc) {
            return !!file && Utils.hasEqualProperties(fileDesc, file, ['source', 'folder_id', 'id', 'attached']);
        };

        /**
         * Updates the current file descriptor of the document edited by this
         * application.
         *
         * @param {Object} fileOptions
         *  All file descriptor properties to be updated.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.updateFileDescriptor = function (fileOptions) {
            if (_.isObject(file)) {
                _.extend(file, fileOptions);
                DriveUtils.propagateChangeFile(file);
                updateTitle();
                updateUrl();
            }
            return this;
        };

        /**
         * Returns an object with attributes describing the file currently
         * opened by this application.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.encodeUrl=false]
         *      If set to true, special characters not allowed in URLs will be
         *      encoded.
         *  @param {Boolean} [options.currentVersion=false]
         *      If set to true, the version stored in the file descriptor will
         *      NOT be inserted into the result (thus, the server will always
         *      access the current version of the document).
         *
         * @returns {Object|Null}
         *  An object with file attributes, if existing; otherwise null.
         */
        this.getFileParameters = function (options) {

            var // function to encode a string to be URI conforming if specified
                encodeString = Utils.getBooleanOption(options, 'encodeUrl', false) ? encodeURIComponent : _.identity,
                // the resulting file parameters
                parameters = null;

            if (file) {
                parameters = {};

                // add the parameters to the result object, if they exist in the file descriptor
                _.each(['id', 'folder_id', 'filename', 'version', 'source', 'attached', 'module', 'com.openexchange.realtime.resourceID'], function (name) {
                    if (_.isString(file[name])) {
                        parameters[name] = encodeString(file[name]);
                    } else if (_.isNumber(file[name])) {
                        parameters[name] = file[name];
                    }
                });

                // remove the version identifier, if specified
                if (Utils.getBooleanOption(options, 'currentVersion', false)) {
                    delete parameters.version;
                }
            }

            return parameters;
        };

        /**
         * Returns the full file name of the current file, with file extension.
         *
         * @returns {String|Null}
         *  The file name of the current file descriptor; or null, if no file
         *  descriptor exists.
         */
        this.getFullFileName = function () {
            return Utils.getStringOption(file, 'com.openexchange.file.sanitizedFilename', null) || Utils.getStringOption(file, 'filename', null);
        };

        /**
         * Returns the short file name of the current file (without file
         * extension).
         *
         * @returns {String|Null}
         *  The short file name of the current file descriptor; or null, if no
         *  file descriptor exists.
         */
        this.getShortFileName = function () {
            var fileName = this.getFullFileName();
            return _.isString(fileName) ? Utils.getFileBaseName(fileName) : null;
        };

        /**
         * Returns the file extension of the current file (without leading
         * period).
         *
         * @returns {String|Null}
         *  The file extension of the current file descriptor; or null, if no
         *  file descriptor exists.
         */
        this.getFileExtension = function () {
            var fileName = this.getFullFileName();
            return _.isString(fileName) ? Utils.getFileExtension(fileName) : null;
        };

        // server requests ----------------------------------------------------

        /**
         * Extends the message data used to display a message with application
         * specific options.
         *
         * @param {Object} messageData
         *  The message data retrieved from ErrorMessages/InfoMessages.
         */
        this.extendMessageData = function (messageData) {
            var messageDataHandler = Utils.getFunctionOption(initOptions, 'messageDataHandler', null);
            if (_.isFunction(messageDataHandler)) {
                messageDataHandler.call(self, messageData);
            }
        };

        /**
         * Creates and returns the URL of a server request.
         *
         * @param {String} module
         *  The name of the server module.
         *
         * @param {Object} [params]
         *  Additional parameters inserted into the URL.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.currentVersion=false]
         *      If set to true, the version stored in the file descriptor will
         *      NOT be inserted into the generated URL (thus, the server will
         *      always access the current version of the document).
         *
         * @returns {String|Null}
         *  The URL of the server request; or null, if the application is not
         *  connected to a document file, or the current session is invalid.
         */
        this.getServerModuleUrl = function (module, params, options) {

            // return nothing if no file is present
            if (!ox.session || !this.hasFileDescriptor()) { return null; }

            // the parameters for the file currently loaded
            var fileParams = this.getFileParameters(Utils.extendOptions(options, { encodeUrl: true }));

            // add default parameters (session and UID), and file parameters
            params = _.extend({ session: ox.session, uid: this.get('uniqueID') }, fileParams, params);

            // build and return the resulting URL
            return ox.apiRoot + '/' + module + '?' + _.map(params, function (value, name) { return name + '=' + value; }).join('&');
        };

        /**
         * Creates an <img> element, sets the passed URL, and returns a promise
         * waiting that the image has been loaded. If the browser does not
         * trigger the appropriate events after the specified timeout, the
         * promise will be rejected automatically. If the application has been
         * shut down before the image has been loaded, the pending promise will
         * be rejected automatically.
         *
         * @param {String} url
         *  The image URL.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.timeout=10000]
         *      The time before the promise will be rejected without response
         *      from the image node.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the image element (as jQuery
         *  object), when the 'load' event has been received from the image
         *  node; or rejected, if the 'error' event has been received from the
         *  image, or after the specified timeout delay without response.
         */
        this.createImageNode = function (url, options) {

            var // the resulting deferred object
                def = $.Deferred(),
                // the image node
                imgNode = $('<img>', { src: url }),
                // the duration of the failure timeout
                timeout = Utils.getIntegerOption(options, 'timeout', 10000, 0),
                // the resulting promise
                promise = this.createAbortablePromise(def, _.noop, timeout),
                // the event handlers for the image node
                handlers = {
                    load: function () { def.resolve(imgNode); },
                    error: function () { def.reject(); }
                };

            // wait that the image is loaded
            imgNode.one(handlers);
            return promise.always(function () { imgNode.off(handlers); });
        };

        /**
         * Safely destroys the passed image nodes. On platforms with restricted
         * memory (especially iPad and iPhone), allocating too many images may
         * result in an immediate browser crash. Simply removing image nodes
         * from the DOM does not free the image resource data allocated by the
         * browser. The suggested workaround is to replace the 'src' attribute
         * of the image node with a local small image which will result in
         * releasing the original image data by the internal resource manager.
         * To be really sure that the garbage collector does not destroy the
         * DOM node instance before the resource manager releases the image
         * data, a reference to the node will be kept in memory for additional
         * 60 seconds.
         *
         * @param {HTMLElement|jQuery} nodes
         *  The image elements, or any other container elements with descendant
         *  image nodes, either as single DOM node, or as jQuery collection
         *  with one or multiple elements.
         */
        this.destroyImageNodes = function (nodes) {

            // filter passed image nodes, find descendant image nodes
            var imgNodes = $(nodes).filter('img').add($(nodes).find('img'));
            if (imgNodes.length === 0) { return; }

            // move image into temporary container, release original image data
            Utils.insertHiddenNodes(imgNodes);
            $(imgNodes).off().attr('src', 'data:image/gif;base64,R0lGODdhAgACAIAAAAAAAP///ywAAAAAAgACAAACAoRRADs=');
            _.delay(function () { imgNodes.remove(); imgNodes = null; }, 60000);
        };

        // application setup --------------------------------------------------

        /**
         * Registers a quit handler function that will be executed before the
         * application will be closed. The callback handler function may veto
         * the quit request to keep the application alive.
         *
         * @param {Function} beforeQuitHandler
         *  A function that will be called before the application will be
         *  closed. Will be called in the context of this application instance.
         *  Receives the reason for quitting that has been passed to the method
         *  BaseApplication.quit() as first parameter. May return a promise
         *  which must be resolved or rejected by the quit handler function. If
         *  the promise will be rejected, the application remains alive.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.registerBeforeQuitHandler = function (beforeQuitHandler) {
            beforeQuitHandlers.push(beforeQuitHandler);
            return this;
        };

        /**
         * Registers a quit handler function that will be executed before the
         * application will be destroyed. This may happen if the application
         * has been closed normally, if the user logs out from the entire
         * AppSuite, or before the browser window or browser tab will be closed
         * or refreshed. If the quit handlers return promises. Closing the
         * application, and logging out will be deferred until the promises
         * have been resolved or rejected.
         *
         * @param {Function} quitHandler
         *  A function that will be called before the application will be
         *  closed. Receives the reason for quitting that has been passed to
         *  the method BaseApplication.quit() as first parameter. Will be
         *  called in the context of this application instance. May return a
         *  promise which must be resolved or rejected by the quit handler
         *  function.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.registerQuitHandler = function (quitHandler) {
            quitHandlers.push(quitHandler);
            return this;
        };

        /**
         * Registers a callback handler that will be invoked every time the
         * visibility state of this application changes. The passed callback
         * handler will be directly invoked once when running this method.
         *
         * @param {Function} stateHandler
         *  The callback handler that will be called after the visibility state
         *  of this application has been changed. Receives the following
         *  parameters:
         *  (1) {Boolean} visible
         *      The current visibility state of this application.
         *  The callback function will be called in the context of this
         *  application.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.registerVisibleStateHandler = function (stateHandler) {
            function invokeStateHandler() { return stateHandler.call(self, self.isActive()); }
            docView.listenTo(appWindow, 'show hide', invokeStateHandler);
            invokeStateHandler();
            return this;
        };

        /**
         * Executes all registered quit handlers and returns a promise that
         * will be resolved or rejected according to the results of the
         * handlers.
         *
         * @internal
         *  Not for external use. Only used from the implementation of the
         *  'io.ox/core/logout' extension point and the window's 'unload' event
         *  handler.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved/rejected when all quit handlers
         *  have been executed.
         */
        this.executeQuitHandlers = function (reason) {
            // Bug 28044: unbind the 'unload' event handler immediately, otherwise
            // logging out may trigger quit handlers twice (from logout and unload)
            $(window).off('unload', unloadHandler);
            // Bug 32989: Create a local copy of the quit handler array, and clear
            // the original array, before executing the quit handlers.
            return quitHandlers ? callHandlers(quitHandlers.splice(0), reason) : $.when(reason);
        };

        /**
         * registers/collects special handler functions (callbacks) that do handle the
         * translation/conversion of the currently worked document into another form.
         *
         * @param {String} type
         *  The string based type that is associated with its `converter` function.
         *
         * @param {Function} converter
         *  The special `converter` function that is associated with its `type`.
         *
         * @returns {Function|Undefined}
         *  Returns, if valid, the provided `converter` or returns the `Undefined` value.
         */
        this.registerConverter = function (type, converter) {
            var handler;
            if (_.isString(type) && type && _.isFunction(converter)) {

                converterRegistry[type] = handler = converter;
            }
            return handler;
        };

        // application runtime ------------------------------------------------

        /**
         * Downloads the document file currently opened by this application.
         * This method should be used by the UI as the method will provide
         * error messages to the user in case of an error.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the document file has been
         *  downloaded successfully, or rejected otherwise.
         */
        this.download = function () {

            // promise for the converted and downloaded document
            var downloadPromise = download();

            // show error message on failure
            downloadPromise.fail(function (response) {

                var // the alert type
                    type = 'error',
                    // the message text
                    message = null;

                switch (Utils.getStringOption(response, 'cause')) {
                    case 'importfailed':
                        message = gt('The document was not loaded successfully. Download is not possible.');
                        break;
                    case 'popupblocker':
                        type = 'warning';
                        message = gt('The browser has blocked a pop-up window. Please enable pop-ups for download.');
                        break;
                    default:
                        message = gt('The server does not answer the request.');
                }

                docView.yell({ type: type, headline: gt('Download Error'), message: message });
            });

            return downloadPromise;
        };

        /**
         * Prints the document file currently opened by this application.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the document file has been
         *  printed successfully, or rejected otherwise.
         */
        this.print = function () {

            // promise for the converted and downloaded document
            var downloadPromise = download({ convert: 'pdf' });

            // show error message on failure
            downloadPromise.fail(function (response) {

                var // the alert type
                    type = 'error',
                    // the message text
                    message = null;

                switch (Utils.getStringOption(response, 'cause')) {
                    case 'docconverter':
                        message = gt('The document converter is currently not available.');
                        break;
                    case 'importfailed':
                        message = gt('The document was not loaded successfully. Printing is not possible.');
                        break;
                    case 'popupblocker':
                        type = 'warning';
                        message = gt('The browser has blocked a pop-up window. Please enable pop-ups for printing.');
                        break;
                    default:
                        message = gt('The server does not answer the request.');
                }

                docView.yell({ type: type, headline: gt('Printing Error'), message: message });
            });

            return downloadPromise;
        };

        /**
         * Default guard method that decides whether an application is able
         * of sending its document as mail or not.
         *
         * Other applications that do feature additional send-mail functionality
         * have to provide theirs own implemantation of `canSendMail`, overwriting
         * the base application's one.
         *
         * @returns {Boolean}
         * Sending a document as file attachment is every application's `sendMail`
         * default behavior. This method never makes use of `canSendMail`.
         * Thus `canSendMail` in its base implementation always return `false`
         * and has to be overwritten by other applications accordingly if they
         * are going making use of it.
         */
        this.canSendMail = function () {
            return false;
        };

        /**
         * Creates a new mail message ...
         *
         * ... either with this current document's content
         * being the content of the new mail's body ...
         *
         * ... or with the current document attached as file.
         *
         * @param {Object} config
         *  The `config` object that is a key value store for properties that
         *  are intended to steer the type of attachment and its content type.
         *
         * @param {String} [config.attachMode='inline-html']
         *  In case `attachMode` equals 'inline-html' this current document's
         *  entire content gets converted to stringified HTML markup and than
         *  attached as inline content of the new mail's body.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the mail application has been
         *  launched successfully, or rejected otherwise.
         */
        this.sendMail = function (config) {

            var attachMode            = (config && config.attachMode) || '',
                handleMailComposition = (attachMode === 'inline-html') ? handleMailCompositionForInlineHTML : ((attachMode === 'pdf-attachment') ? handleMailCompositionForPDFAttachment : handleMailCompositionForFileAttachment),

                // the Promise with the result of the flush callback handler
                flushPromise = (this.isImportSucceeded() && this.hasFileDescriptor()) ? $.when(flushHandler.call(this, 'email')) : $.Deferred().reject();

            /*
            flushPromise = flushPromise.then(function () {
                return DriveUtils.propagateChangeFile(self.getFileDescriptor());
            });
            */

            return flushPromise.then(handleMailComposition, handleMailCompositionFailure);
        };

        /**
         * Closes this application instance. First, calls the registered
         * before-quit handlers. If none of the handlers vetoes (e.g. due to
         * unsaved changes, or after asking the user), calls the registered
         * quit handlers, and destroys the MVC instances and this application.
         *
         * @attention
         *  After calling this method, this application instance may be
         *  destroyed. An external caller MUST NOT do anything else with this
         *  instance.
         *
         * @param {String} [reason='user']
         *  The reason for quitting the application. The reason will be passed
         *  to all registered before-quit handlers, and all registered quit
         *  handlers. The default reason 'user' is intended for quitting the
         *  application due to a user interaction, e.g. pressing a Quit button.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the application has actually
         *  been closed (no before-quit handler has vetoed quitting).
         */
        this.quit = (function (coreQuitMethod) {
            return function (reason) {

                // remember current quit() method (needs to be restored after resuming,
                // derived methods may have added their own quit() method)
                var origQuitMethod = this.quit;

                // create the result deferred (rejecting means resume application)
                currentQuitDef = $.Deferred();

                // rejected: resume to running state of application
                currentQuitDef.fail(function () {
                    self.quit = origQuitMethod;
                    currentQuitDef = null;
                });

                // override 'quit()' method to prevent repeated/recursive calls while in quit mode
                this.quit = function () { return currentQuitDef.promise(); };

                // store the reason in an internal variable (it is not possible
                // to pass any data to the core quit() method)
                quitReason = _.isString(reason) ? reason : 'user';

                // ensure that initialization is complete before calling destructors
                initDef.always(function () {

                    // Call core quit() method deferred to prevent fast destruction of any application
                    // child instances while calling own quit(), the core method will invoke the
                    // callback function registered via setQuit() that actually calls the registered
                    // quit handlers, and destroys the application (unless quit has been canceled).
                    _.defer(coreQuitMethod);
                });

                return currentQuitDef.promise();
            };
        }(_.bind(this.quit, this)));

        /**
         * helper that indicates whether an application is capable
         * of translating/converting the currently worked document
         * into another form or not.
         *
         * @param {String} type
         *  The type that is associated with its converter function.
         *
         * @returns {Boolean}
         */
        this.canConvertTo = function (type) {
            return (type in converterRegistry);
        };

        /**
         * helper that either returns the converter function that is
         * associated with the passed type or returns the `Null` value.
         *
         * @param {String} type
         *  The type that is ought to be associated with a converter function.
         *
         * @returns {Function|Null}
         */
        this.getConverter = function (type) {
            return (this.canConvertTo(type) ? converterRegistry[type] : null);
        };

        /**
         * generic method that directly triggers either the converter
         * associated with the passed type or that fails with throwing
         * a `TypeError`.
         *
         * @param {String} type
         *  The type that is ought to be associated with a converter function.
         *
         * @returns {any}
         *  The return value of the called converter function.
         *
         * @attention
         *  A converter function's direct return value might be a `Promise`
         *  or a `Deferred` too. In this case the converter functionality
         *  is encapsulated by its *Deferrable* and the result of the
         *  converting process gets passed along the resolve chain.
         *
         * @throws {TypeError}
         *  Does throw a `TypeError` exception in case `convertTo`
         *  had been called without checking its existence before.
         */
        this.convertTo = function (type) {
          // won't cover all possible future edge cases.
          //return (this.getConverter(type)());

            return this.getConverter(type).call(this);
        };

        // initialization -----------------------------------------------------

        // register application in global cache for type detection
        runningAppMap[this.cid] = this;

        // provide the API provided by a real BaseObject instance
        this.createAbortablePromise = baseContext.createAbortablePromise.bind(baseContext);
        this.registerDestructor = baseContext.registerDestructor.bind(baseContext);
        this.waitForSuccess = baseContext.waitForSuccess.bind(baseContext);
        this.waitForFailure = baseContext.waitForFailure.bind(baseContext);
        this.waitForAny = baseContext.waitForAny.bind(baseContext);
        this.waitForEvent = baseContext.waitForEvent.bind(baseContext);

        // add mix-in classes (need a complete application instance)
        TimerMixin.call(this);
        AppObjectMixin.call(this, this);
        AppTooltipMixin.call(this);

        // set quit callback invoked from the core method quit(), resolve/reject 'currentQuitDef'
        // according to the result of the registered before-quit handlers
        this.setQuit(quitApplication);

        // prevent usage of these methods in derived classes
        delete this.setLauncher;
        delete this.setQuit;

        // construct MVC instances, and initialize this application instance
        (function () {

            // duration of a single step between two log messages
            var lapTime = _.now();
            // launch source
            var actionSource = Utils.getStringOption(launchOptions, 'actionSource', null);
            // the promise waiting for a file descriptor from Files API
            var filePromise = null;

            function logLaunchDuration(message, key) {
                var time = _.now();
                if (key) { timerLogCollector[key] = time; }
                Utils.info('BaseApplication.initialize(): ' + message + ' finished in ' + (time - lapTime) + 'ms (total ' + (time - launchStartTime) + 'ms)');
                lapTime = time;
            }

            // initializes the user interface of the application
            function initializeGui() {

                // the resulting promise for entire GUI initialization
                var initGuiPromise = null;

                // hide the view, call the view initialization handler
                initGuiPromise = docView.hideBeforeImport().always(function () {
                    logLaunchDuration('GUI preparation', 'guiPrepared');
                });

                // call the file initialization handler
                if (_.isFunction(initFileHandler)) {
                    initGuiPromise = initGuiPromise.then(function () {
                        return initFileHandler.call(self).done(function (fileDesc) {
                            if (_.isObject(fileDesc)) {
                                if (file) {
                                    DriveUtils.propagateChangeFile(fileDesc);
                                } else {
                                    DriveUtils.propagateNewFile(fileDesc);
                                }
                                file = fileDesc;
                                updateTitle();
                                updateUrl();
                            }
                        }).always(function () {
                            logLaunchDuration('file descriptor initialization', 'fileInitialized');
                        });
                    });
                }

                // resolve/reject the before-import promise
                initGuiPromise.done(function () { importStartDef.resolve(); });
                initGuiPromise.fail(function () { importStartDef.reject(); });

                // call the import handler
                initGuiPromise = initGuiPromise.then(function () {
                    return importHandler.call(self).always(function () {
                        logLaunchDuration('document import', 'documentImported');
                    });
                });

                // makes the view visible after import
                function showAfterImport(resolved, result) {

                    // nothing to do, if application is shutting down
                    if (!self || self.isInQuit()) { return result; }

                    // the Deferred object used to forward the original state and result
                    var def = $.Deferred();

                    // make the view visible
                    docView.showAfterImport(!resolved).always(function () {
                        updateTitle();
                        logLaunchDuration('GUI initialization', 'guiInitialized');
                        // forward state and result of the originating promise
                        if (resolved) { def.resolve(result); } else { def.reject(result); }
                    });

                    return def.promise();
                }

                // finally, make the view visible (also if anything before has failed)
                initGuiPromise = initGuiPromise.then(_.partial(showAfterImport, true), _.partial(showAfterImport, false));

                // resolve the global import Deferred object on success, unless the
                // application is already closing (import canceled)
                initGuiPromise.done(function (result) {

                    // nothing to do, if application is shutting down
                    if (!self || self.isInQuit()) { return result; }

                    // set app auto-close timeout, if specified
                    if (_.isNumber(autoCloseDelay)) {
                        self.executeDelayed(function () { self.quit('autoclose'); }, autoCloseDelay);
                    }
                    importFinishDef.resolve(result);
                });

                // handle import failures, unless the  application is already closing (import canceled)
                initGuiPromise.fail(function (result) {

                    // do not show any messages nor trigger events, if application is shutting down
                    if (!self || self.isInQuit()) { return result; }

                    if (Utils.getBooleanOption(result, 'errorForErrorCode', false)) {
                        var error = new ErrorCode(result);
                        Utils.error('BaseApplication.initialize(): Importing document "' + self.getFullFileName() + '" failed. ' + error.getErrorText());
                        self.setErrorState(error);
                        importFinishDef.reject(error);
                    } else {
                        var cause = Utils.getStringOption(result, 'cause', 'unknown');
                        Utils.error('BaseApplication.initialize(): Importing document "' + self.getFullFileName() + '" failed. Error code: "' + cause + '".');
                        importFinishDef.reject(result);
                    }
                });

                // log import duration
                initGuiPromise.always(function () {
                    logLaunchDuration('postprocessing', 'launchEnd');
                });

                // need to update cache of FolderApi, which is done indirectly in DriveUtils
                initGuiPromise.then(function () {
                    return DriveUtils.getWriteState(file.folder_id);
                });

                // add marker attribute to application root node, e.g. for Selenium
                importFinishDef.always(function () {
                    self.setRootAttribute('data-doc-loaded', true);
                });
            }

            Utils.info('BaseApplication.initialize(): starting construction');
            launchStartTime = lapTime;

            // resolve file descriptor from URL options (before starting to construct other components)
            if (actionSource === 'url') {
                if (launchOptions.template) {
                    filePromise = DriveUtils.getFile(launchOptions.templateFile).done(function (fileDescriptor) {
                        launchOptions.templateFile = fileDescriptor;
                    });
                } else {
                    filePromise = DriveUtils.getFile(launchOptions.file).done(function (fileDescriptor) {
                        file = fileDescriptor;
                    });
                }
            } else {
                filePromise = $.when();
            }

            // Start construction after file descriptor has been resolved or rejected.
            // Must also be done in case of error, to initialize the application window
            // which is required by the following call to app.quit().
            filePromise.always(function () {

                // wait for unload events and execute quit handlers
                $(window).on('unload', unloadHandler);

                // create the MVC instances in a timeout (first, all application constructors have to complete)
                self.executeDelayed(function () {
                    docModel = new ModelClass(self);
                    docView = new ViewClass(self, docModel);
                    docController = new ControllerClass(self, docModel, docView);

                    // run all initialization callbacks registered in the MVC constructors
                    initDef.resolve();
                });
            });

            // early exit if file descriptor cannot be resolved
            filePromise.fail(function (response) {
                var showMessageForCode = Utils.getBooleanOption(response, 'errorForErrorCode', false);
                var messageText = showMessageForCode ? ErrorMessages.getMessageData(new ErrorCode(response)).message : Utils.getStringOption(response, 'error', 'unknown');
                Notifications.yell('error', messageText);
                self.quit('error');
            });

            // wait for initialization before showing the application window in order to get
            // the 'open' event of the window at all, it must be shown (also without file).
            initDef.done(function () {

                // whether the application is being launched in visible state (global
                // application launcher has shown the application window already)
                var isActive = self.isActive();
                // the value of the 'spellcheck' attribute of the <body> element
                var spellState = null;

                // disables the global browser spellchecker while a Documents application is active
                function disableGlobalSpelling() {
                    spellState = $(document.body).attr('spellcheck');
                    $(document.body).attr('spellcheck', false);
                }

                // restores the state of the global browser spellchecker
                function restoreGlobalSpelling() {
                    if (spellState) {
                        $(document.body).attr('spellcheck', spellState);
                    } else {
                        $(document.body).removeAttr('spellcheck');
                    }
                }

                // enables/disables hidden restore mode (restore application in hidden state after browser reload)
                function enableHiddenRestoreMode(enable) {
                    if (file) {
                        var module = self.getName() + '/main';
                        var key = file.id + '~' + module;
                        SessionRestore.state(key, enable ? { action: 'load', file: file, module: module, hidden: true } : null);
                    }
                }

                // disable global spell checking while this application is active
                appWindow.on('show', disableGlobalSpelling);
                appWindow.on('beforehide', restoreGlobalSpelling);
                if (isActive) { disableGlobalSpelling(); }

                // initialize hidden restore mode according to current visibility
                enableHiddenRestoreMode(!isActive);

                // update hidden restore mode according to visibility of this application
                self.registerVisibleStateHandler(function (visible) {
                    enableHiddenRestoreMode(!visible);
                });

                // remove hidden restore settings when closing this application regularly
                self.registerQuitHandler(function (reason) {
                    if (reason !== 'unload') { enableHiddenRestoreMode(false); }
                });

                // bug 29711: update URL when applications becomes active
                appWindow.on('show', updateUrl);
                if (isActive) { updateUrl(); }

                // set initial application title (document file name) early
                updateTitle();

                // wait for all other initialization handlers registered at 'initDef'
                self.executeDelayed(function () {

                    // log the time needed to construct the application window and MVC instances
                    logLaunchDuration('application construction', 'appConstructed');

                    // do not show the application, if it is being restored in hidden mode (after browser reload)
                    if (isActive) {
                        initializeGui();
                    } else {
                        appWindow.one('show', initializeGui);
                    }
                }, undefined, 'BaseApplication: GUI preparation');
            });
        }.call(this));

    } // class BaseApplication

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed object is an instance of BaseApplication. As
     * that class will be mixed into an instance of ox.ui.App, the built-in
     * operator 'instanceof' cannot be used for this test.
     *
     * @param {ox.ui.App} [app]
     *  Any application instance, or null/undefined.
     *
     * @returns {Boolean}
     *  Whether the passed value is an instance of the class BaseApplication.
     */
    BaseApplication.isInstance = function (app) {
        return !!app && (app.cid in runningAppMap);
    };

    // static initialization ==================================================

    // run quit handlers of all OX Documents applications immediately to unregister them
    // from server, but not if there are applications left with unsaved changes
    ox.on('beforeunload', function (unsavedChanges) {
        if (!unsavedChanges) {
            _.invoke(runningAppMap, 'executeQuitHandlers', 'unload');
        }
    });

    // exports ================================================================

    return _.makeExtendable(BaseApplication);

});
