/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/object/baseobject'
], function (BaseObject) {

    'use strict';

    // class BaseObject =======================================================

    describe('Toolkit class BaseObject', function () {

        it('should exist', function () {
            expect(BaseObject).to.be.a('function');
        });

        // protected methods --------------------------------------------------

        describe('method "registerDestructor"', function () {
            var obj = new BaseObject(), spy = sinon.spy();
            it('should exist', function () {
                expect(obj).to.respondTo('registerDestructor');
            });
            it('should register a destructor callback', function () {
                obj.registerDestructor(spy);
                expect(spy.called).to.equal(false);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getUid"', function () {
            var obj1 = new BaseObject(), obj2 = new BaseObject();
            it('should exist', function () {
                expect(obj1).to.respondTo('getUid');
            });
            it('should return a UID', function () {
                expect(obj1.getUid()).to.be.a('string');
            });
            it('should return the same UID for the same object', function () {
                expect(obj1.getUid()).to.equal(obj1.getUid());
            });
            it('should return different UIDs for different objects', function () {
                expect(obj1.getUid()).to.not.equal(obj2.getUid());
            });
        });

        describe('method "listenTo"', function () {
            var obj = new BaseObject();
            it('should exist', function () {
                expect(obj).to.respondTo('listenTo');
            });
            it('should call the event handler', function () {
                var src = $({}), spy = sinon.spy();
                obj.listenTo(src, 'test', spy);
                expect(spy.called).to.equal(false);
                src.trigger('wrong');
                expect(spy.called).to.equal(false);
                src.trigger('test');
                expect(spy.called).to.equal(true);
            });
            it('should call the event handler for multiple event types', function () {
                var src = $({}), spy = sinon.spy();
                obj.listenTo(src, 'test1 test2', spy);
                expect(spy.called).to.equal(false);
                src.trigger('wrong');
                expect(spy.called).to.equal(false);
                src.trigger('test1');
                expect(spy.calledOnce).to.equal(true);
                src.trigger('test2');
                expect(spy.calledTwice).to.equal(true);
            });
            it('should call the event handlers of an event map', function () {
                var src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, { test1: spy1, 'test1 test2': spy2 });
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('wrong');
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                src.trigger('test2');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledTwice).to.equal(true);
            });
        });

        describe('method "stopListeningTo"', function () {
            it('should exist', function () {
                var obj = new BaseObject();
                expect(obj).to.respondTo('stopListeningTo');
            });
            it('should unregister all event listeners', function () {
                var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2', spy1).listenTo(src, 'test1 test2', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src);
                src.trigger('test1').trigger('test2');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
            });
            it('should unregister a specific event type', function () {
                var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, 'test2 test3');
                src.trigger('test1').trigger('test2').trigger('test3');
                expect(spy1.calledTwice).to.equal(true);
                expect(spy2.calledTwice).to.equal(true);
            });
            it('should unregister a specific event listener', function () {
                var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2', spy1).listenTo(src, 'test1 test2', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, spy2);
                src.trigger('test1').trigger('test2');
                expect(spy1.calledThrice).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
            });
            it('should unregister a specific event type and listener', function () {
                var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, 'test2 test3', spy2);
                src.trigger('test1').trigger('test2').trigger('test3');
                expect(spy1.callCount).to.equal(4);
                expect(spy2.calledTwice).to.equal(true);
            });
            it('should unregister listeners from an event map', function () {
                var obj = new BaseObject(), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, { test1: spy1, 'test2 test3': spy2 });
                src.trigger('test1').trigger('test2').trigger('test3');
                expect(spy1.calledThrice).to.equal(true);
                expect(spy2.calledTwice).to.equal(true);
            });
        });

        describe('method "waitForSuccess"', function () {
            var obj = new BaseObject();
            it('should exist', function () {
                expect(obj).to.respondTo('waitForSuccess');
            });
            it('should wait for promise\'s success', function () {
                var def1 = $.Deferred(), spy1 = sinon.spy(),
                    def2 = $.Deferred(), spy2 = sinon.spy();
                obj.waitForSuccess(def1, spy1);
                obj.waitForSuccess(def2, spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                def1.resolve();
                def2.reject();
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(false);
            });
        });

        describe('method "waitForFailure"', function () {
            var obj = new BaseObject();
            it('should exist', function () {
                expect(obj).to.respondTo('waitForFailure');
            });
            it('should wait for promise\'s failure', function () {
                var def1 = $.Deferred(), spy1 = sinon.spy(),
                    def2 = $.Deferred(), spy2 = sinon.spy();
                obj.waitForFailure(def1, spy1);
                obj.waitForFailure(def2, spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                def1.resolve();
                def2.reject();
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(true);
            });
        });

        describe('method "waitForAny"', function () {
            var obj = new BaseObject();
            it('should exist', function () {
                expect(obj).to.respondTo('waitForAny');
            });
            it('should wait for promise\'s success or failure', function () {
                var def1 = $.Deferred(), spy1 = sinon.spy(),
                    def2 = $.Deferred(), spy2 = sinon.spy();
                obj.waitForAny(def1, spy1);
                obj.waitForAny(def2, spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                def1.resolve();
                def2.reject();
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(true);
            });
        });

        describe('method "destroy"', function () {

            var obj = new BaseObject(),
                src = $({}),
                src2 = $({}),
                def = $.Deferred(),
                spy1 = sinon.spy(),
                spy2 = sinon.spy(),
                spy3 = sinon.spy();

            src2.off = false;

            obj.registerDestructor(spy1);
            obj.registerDestructor(spy2);
            obj.listenTo(src, 'test', spy3);
            obj.listenTo(src2, 'test2', spy3);
            obj.waitForSuccess(def, spy3);

            it('should exist', function () {
                expect(obj).to.respondTo('destroy');
            });
            it('should call the registered destructors in reversed order', function () {
                expect(obj.destroyed).to.be.undefined;
                obj.destroy();
                expect(obj.destroyed).to.equal(true);
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(true);
                expect(spy2.calledBefore(spy1)).to.equal(true);
            });
            it('should delete all public methods', function () {
                expect(obj).not.to.respondTo('registerDestructor');
                expect(obj).not.to.respondTo('getUid');
                expect(obj).not.to.respondTo('listenTo');
                expect(obj).not.to.respondTo('stopListeningTo');
                expect(obj).not.to.respondTo('waitForSuccess');
                expect(obj).not.to.respondTo('waitForFailure');
                expect(obj).not.to.respondTo('waitForAny');
                expect(obj).not.to.respondTo('destroy');
            });
            it('should release event handlers', function () {
                src.trigger('test');
                expect(spy3.called).to.equal(false);
            });
            it('should release promise handlers', function () {
                def.resolve();
                expect(spy3.called).to.equal(false);
            });
        });
    });

    // ========================================================================
});
