/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/rangearray'
], function (Address, Interval, Range, AddressArray, IntervalArray, RangeArray) {

    'use strict';

    // class RangeArray =======================================================

    describe('Spreadsheet class RangeArray', function () {

        it('should exist', function () {
            expect(RangeArray).to.be.a('function');
        });
        it('should be a subclass of Array', function () {
            expect(new RangeArray()).to.be.an['instanceof'](Array);
        });

        // private helpers ----------------------------------------------------

        var ic = Interval.parseAsCols,
            ir = Interval.parseAsRows,
            a = Address.parse,
            r = Range.parse;

        function ica(str) { return new IntervalArray(str.split(/\s+/).map(ic)); }
        function ira(str) { return new IntervalArray(str.split(/\s+/).map(ir)); }
        function aa(str) { return new AddressArray(str.split(/\s+/).map(a)); }
        function ra(str) { return new RangeArray(str.split(/\s+/).map(r)); }

        function unorderedRangesMatcher(expected) {
            expected = _.invoke(expected, 'key').sort().join();
            return function (result) {
                result = _.invoke(result, 'key').sort().join();
                return expected === result;
            };
        }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a range array', function () {
                var ra = new RangeArray();
                expect(ra).to.be.an['instanceof'](RangeArray);
                expect(ra).to.be.empty;
            });
            var r1 = r('A2:C4'), r2 = r('B3:D5');
            it('should insert single ranges', function () {
                var ra1 = new RangeArray(r1);
                expect(ra1).to.have.length(1);
                expect(ra1[0]).to.equal(r1);
                var ra2 = new RangeArray(r1, r2, r1);
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
            });
            it('should insert plain arrays of ranges', function () {
                var ra1 = new RangeArray([r1]);
                expect(ra1).to.have.length(1);
                expect(ra1[0]).to.equal(r1);
                var ra2 = new RangeArray([r1, r2], [], [r1]);
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
            });
            it('should copy-construct range arrays', function () {
                var ra0 = new RangeArray(r1, r2),
                    ra1 = new RangeArray(ra0);
                expect(ra1).to.have.length(2);
                expect(ra1[0]).to.equal(r1);
                expect(ra1[1]).to.equal(r2);
                var ra2 = new RangeArray(ra0, new RangeArray(), ra1);
                expect(ra2).to.have.length(4);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
                expect(ra2[3]).to.equal(r2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "createFromIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('createFromIntervals');
            });
            var ia1 = ica('B:C C:D'), ia2 = ira('3:4 4:5');
            it('should create range addresses from column and row intervals', function () {
                expect(RangeArray.createFromIntervals(ia1, ia2)).to.deep.equal(ra('B3:C4 C3:D4 B4:C5 C4:D5'));
            });
            it('should accept single intervals', function () {
                var i1 = ic('B:D'), i2 = ir('3:5');
                expect(RangeArray.createFromIntervals(i1, ia2)).to.deep.equal(ra('B3:D4 B4:D5'));
                expect(RangeArray.createFromIntervals(ia1, i2)).to.deep.equal(ra('B3:C5 C3:D5'));
                expect(RangeArray.createFromIntervals(i1, i2)).to.deep.equal(ra('B3:D5'));
            });
            it('should accept empty arrays', function () {
                var ia0 = new IntervalArray();
                expect(RangeArray.createFromIntervals(ia1, ia0)).to.have.length(0);
                expect(RangeArray.createFromIntervals(ia0, ia2)).to.have.length(0);
                expect(RangeArray.createFromIntervals(ia0, ia0)).to.have.length(0);
            });
        });

        describe('method "createFromColIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('createFromColIntervals');
            });
            var ia1 = ica('B:C C:D');
            it('should create range addresses from column intervals', function () {
                expect(RangeArray.createFromColIntervals(ia1, 2, 3)).to.deep.equal(ra('B3:C4 C3:D4'));
                expect(RangeArray.createFromColIntervals(ia1, 2)).to.deep.equal(ra('B3:C3 C3:D3'));
            });
        });

        describe('method "createFromRowIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('createFromRowIntervals');
            });
            var ia1 = ira('3:4 4:5');
            it('should create range addresses from row intervals', function () {
                expect(RangeArray.createFromRowIntervals(ia1, 1, 2)).to.deep.equal(ra('B3:C4 B4:C5'));
                expect(RangeArray.createFromRowIntervals(ia1, 1)).to.deep.equal(ra('B3:B4 B4:B5'));
            });
        });

        describe('method "mergeAddresses"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('mergeAddresses');
            });
            it('should merge cell addresses to range', function () {
                expect(RangeArray.mergeAddresses(aa('A1 A3 A4 A2 A1'))).to.deep.equal(ra('A1:A4'));
                expect(RangeArray.mergeAddresses(aa('A1 B2 B1 A2'))).to.deep.equal(ra('A1:B2'));
            });
            it('should accept an empty array', function () {
                expect(RangeArray.mergeAddresses(new AddressArray())).to.be.empty;
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('clone');
            });
            it('should return a shallow clone', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.clone();
                expect(ra2).to.be.an['instanceof'](RangeArray);
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
                expect(ra2).to.deep.equal(ra1);
            });
            it('should return a deep clone', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.clone(true);
                expect(ra2).to.be.an['instanceof'](RangeArray);
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.not.equal(ra1[0]);
                expect(ra2[1]).to.not.equal(ra1[1]);
                expect(ra2).to.deep.equal(ra1);
            });
        });

        describe('method "cells"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('cells');
            });
            it('should count cells in ranges', function () {
                expect(ra('A1:C3 B2:D4').cells()).to.equal(18);
                expect(new RangeArray().cells()).to.equal(0);
            });
        });

        describe('method "containsCol"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsCol');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return false for columns outside the ranges', function () {
                expect(ra1.containsCol(0)).to.equal(false);
                expect(ra1.containsCol(5)).to.equal(false);
                expect(ra1.containsCol(8)).to.equal(false);
                expect(ra1.containsCol(9)).to.equal(false);
                expect(ra1.containsCol(10)).to.equal(false);
                expect(ra1.containsCol(11)).to.equal(false);
                expect(ra1.containsCol(12)).to.equal(false);
            });
            it('should return true for columns inside the ranges', function () {
                expect(ra1.containsCol(1)).to.equal(true);
                expect(ra1.containsCol(2)).to.equal(true);
                expect(ra1.containsCol(3)).to.equal(true);
                expect(ra1.containsCol(4)).to.equal(true);
                expect(ra1.containsCol(6)).to.equal(true);
                expect(ra1.containsCol(7)).to.equal(true);
            });
        });

        describe('method "containsRow"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsRow');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return false for rows outside the ranges', function () {
                expect(ra1.containsRow(0)).to.equal(false);
                expect(ra1.containsRow(1)).to.equal(false);
                expect(ra1.containsRow(2)).to.equal(false);
                expect(ra1.containsRow(7)).to.equal(false);
                expect(ra1.containsRow(10)).to.equal(false);
                expect(ra1.containsRow(11)).to.equal(false);
                expect(ra1.containsRow(12)).to.equal(false);
            });
            it('should return true for rows inside the ranges', function () {
                expect(ra1.containsRow(3)).to.equal(true);
                expect(ra1.containsRow(4)).to.equal(true);
                expect(ra1.containsRow(5)).to.equal(true);
                expect(ra1.containsRow(6)).to.equal(true);
                expect(ra1.containsRow(8)).to.equal(true);
                expect(ra1.containsRow(9)).to.equal(true);
            });
        });

        describe('method "containsIndex"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsIndex');
            });
            var ra1 = ra('C2:C2 E4:E4');
            it('should check column indexes', function () {
                expect(ra1.containsIndex(1, true)).to.equal(false);
                expect(ra1.containsIndex(2, true)).to.equal(true);
                expect(ra1.containsIndex(3, true)).to.equal(false);
                expect(ra1.containsIndex(4, true)).to.equal(true);
                expect(ra1.containsIndex(5, true)).to.equal(false);
            });
            it('should check row indexes', function () {
                expect(ra1.containsIndex(0, false)).to.equal(false);
                expect(ra1.containsIndex(1, false)).to.equal(true);
                expect(ra1.containsIndex(2, false)).to.equal(false);
                expect(ra1.containsIndex(3, false)).to.equal(true);
                expect(ra1.containsIndex(4, false)).to.equal(false);
            });
        });

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsAddress');
            });
            var ra1 = ra('B4:C5 C5:D6');
            it('should return false for cells outside the ranges', function () {
                'A3 B3 C3 D3 E3 A4 D4 E4 A5 E5 A6 B6 E6 A7 B7 C7 D7 E7'.split(' ').forEach(function (str) {
                    expect(ra1.containsAddress(a(str))).to.equal(false);
                });
            });
            it('should return true for cells inside the ranges', function () {
                expect(ra1.containsAddress(a('B4'))).to.equal(true);
                expect(ra1.containsAddress(a('C4'))).to.equal(true);
                expect(ra1.containsAddress(a('B5'))).to.equal(true);
                expect(ra1.containsAddress(a('C5'))).to.equal(true);
                expect(ra1.containsAddress(a('D5'))).to.equal(true);
                expect(ra1.containsAddress(a('C6'))).to.equal(true);
                expect(ra1.containsAddress(a('D6'))).to.equal(true);
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('contains');
            });
            var ra1 = ra('A1:B2 B2:C3 C3:D4');
            it('should return false for ranges outside the ranges', function () {
                expect(ra1.contains(ra('C1:D1 B2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 C1:D1 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C1:D1'))).to.equal(false);
            });
            it('should return false for overlapping ranges', function () {
                expect(ra1.contains(ra('A1:B3 B2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:C2 B2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 A2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B1:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:D3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C4 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 B3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C2:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C3:E4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C3:D5'))).to.equal(false);
            });
            it('should return true for contained ranges', function () {
                expect(ra1.contains(ra('A1:B1 A1:A2 A1:B2'))).to.equal(true);
                expect(ra1.contains(ra('A1:B1 B2:C2 C3:D3'))).to.equal(true);
                expect(ra1.contains(ra1)).to.equal(true);
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra0.contains(ra1)).to.equal(false);
                expect(ra1.contains(ra0)).to.equal(true);
                expect(ra0.contains(ra0)).to.equal(true);
            });
            it('should accept single range address', function () {
                expect(ra1.contains(r('C1:D1'))).to.equal(false);
                expect(ra1.contains(r('A1:B3'))).to.equal(false);
                expect(ra1.contains(r('B2:C3'))).to.equal(true);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('overlaps');
            });
            var ra1 = ra('A1:B2 B2:C3 C3:D4');
            it('should return false for distinct ranges', function () {
                expect(ra1.overlaps(ra('C1:D1 A3:A4 D1:D2 A4:B4'))).to.equal(false);
            });
            it('should return true for overlapping ranges', function () {
                expect(ra1.overlaps(ra('C1:D1 A3:B4 D1:D2'))).to.equal(true);
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra0.overlaps(ra1)).to.equal(false);
                expect(ra1.overlaps(ra0)).to.equal(false);
                expect(ra0.overlaps(ra0)).to.equal(false);
            });
            it('should accept single range address', function () {
                expect(ra1.overlaps(r('C1:D1'))).to.equal(false);
                expect(ra1.overlaps(r('A3:B4'))).to.equal(true);
            });
        });

        describe('method "overlapsSelf"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('overlapsSelf');
            });
            it('should return false for distinct ranges', function () {
                expect(ra('B3:C4 D3:E4 B5:C6 D5:E6').overlapsSelf()).to.equal(false);
            });
            it('should return true for overlapping ranges', function () {
                expect(ra('B3:C4 D3:E4 B5:D6 D5:E6 B3:C4').overlapsSelf()).to.equal(true);
                expect(ra('B3:C4 D3:E4 B5:D6 D5:E6 C4:D5').overlapsSelf()).to.equal(true);
            });
            it('should return false for an empty array', function () {
                expect(new RangeArray().overlapsSelf()).to.equal(false);
            });
        });

        describe('method "findByCol"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByCol');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return null for columns outside the ranges', function () {
                expect(ra1.findByCol(0)).to.equal(null);
                expect(ra1.findByCol(5)).to.equal(null);
                expect(ra1.findByCol(8)).to.equal(null);
                expect(ra1.findByCol(9)).to.equal(null);
                expect(ra1.findByCol(10)).to.equal(null);
                expect(ra1.findByCol(11)).to.equal(null);
                expect(ra1.findByCol(12)).to.equal(null);
            });
            it('should return first existing range', function () {
                expect(ra1.findByCol(1)).to.equal(ra1[0]);
                expect(ra1.findByCol(2)).to.equal(ra1[0]);
                expect(ra1.findByCol(3)).to.equal(ra1[0]);
                expect(ra1.findByCol(4)).to.equal(ra1[2]);
                expect(ra1.findByCol(6)).to.equal(ra1[1]);
                expect(ra1.findByCol(7)).to.equal(ra1[1]);
            });
        });

        describe('method "findByRow"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByRow');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return null for rows outside the ranges', function () {
                expect(ra1.findByRow(0)).to.equal(null);
                expect(ra1.findByRow(1)).to.equal(null);
                expect(ra1.findByRow(2)).to.equal(null);
                expect(ra1.findByRow(7)).to.equal(null);
                expect(ra1.findByRow(10)).to.equal(null);
                expect(ra1.findByRow(11)).to.equal(null);
                expect(ra1.findByRow(12)).to.equal(null);
            });
            it('should return first existing range', function () {
                expect(ra1.findByRow(3)).to.equal(ra1[0]);
                expect(ra1.findByRow(4)).to.equal(ra1[0]);
                expect(ra1.findByRow(5)).to.equal(ra1[0]);
                expect(ra1.findByRow(6)).to.equal(ra1[2]);
                expect(ra1.findByRow(8)).to.equal(ra1[1]);
                expect(ra1.findByRow(9)).to.equal(ra1[1]);
            });
        });

        describe('method "findByIndex"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByIndex');
            });
            var ra1 = ra('C2:C2 E4:E4');
            it('should check column indexes', function () {
                expect(ra1.findByIndex(1, true)).to.equal(null);
                expect(ra1.findByIndex(2, true)).to.equal(ra1[0]);
                expect(ra1.findByIndex(3, true)).to.equal(null);
                expect(ra1.findByIndex(4, true)).to.equal(ra1[1]);
                expect(ra1.findByIndex(5, true)).to.equal(null);
            });
            it('should check row indexes', function () {
                expect(ra1.findByIndex(0, false)).to.equal(null);
                expect(ra1.findByIndex(1, false)).to.equal(ra1[0]);
                expect(ra1.findByIndex(2, false)).to.equal(null);
                expect(ra1.findByIndex(3, false)).to.equal(ra1[1]);
                expect(ra1.findByIndex(4, false)).to.equal(null);
            });
        });

        describe('method "findByAddress"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByAddress');
            });
            var ra1 = ra('B4:C5 C5:D6');
            it('should return null for cells outside the ranges', function () {
                'A3 B3 C3 D3 E3 A4 D4 E4 A5 E5 A6 B6 E6 A7 B7 C7 D7 E7'.split(' ').forEach(function (str) {
                    expect(ra1.findByAddress(a(str))).to.equal(null);
                });
            });
            it('should return first existing range', function () {
                expect(ra1.findByAddress(a('B4'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('C4'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('B5'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('C5'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('D5'))).to.equal(ra1[1]);
                expect(ra1.findByAddress(a('C6'))).to.equal(ra1[1]);
                expect(ra1.findByAddress(a('D6'))).to.equal(ra1[1]);
            });
        });

        describe('method "colIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('colIntervals');
            });
            it('should return the column intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').colIntervals()).to.deep.equal(ica('B:C D:E C:D'));
                expect(new RangeArray().colIntervals()).to.be.empty;
            });
        });

        describe('method "rowIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('rowIntervals');
            });
            it('should return the row intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').rowIntervals()).to.deep.equal(ira('3:4 5:6 4:5'));
                expect(new RangeArray().rowIntervals()).to.be.empty;
            });
        });

        describe('method "intervals"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('intervals');
            });
            it('should return the column intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').intervals(true)).to.deep.equal(ica('B:C D:E C:D'));
            });
            it('should return the row intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').intervals(false)).to.deep.equal(ira('3:4 5:6 4:5'));
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('boundary');
            });
            it('should return the bounding range', function () {
                expect(ra('B3:C4 C4:D5').boundary()).to.deep.equal(r('B3:D5'));
                expect(ra('C4:D5 B3:C4 A2:B3').boundary()).to.deep.equal(r('A2:D5'));
            });
            it('should return a clone of a single range', function () {
                var ra1 = ra('B3:C4'), r1 = ra1.boundary();
                expect(r1).to.deep.equal(ra1[0]);
                expect(r1).to.not.equal(ra1[0]);
            });
            it('should return null for an empty array', function () {
                expect(new RangeArray().boundary()).to.equal(null);
            });
        });

        describe('method "unify"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('unify');
            });
            it('should return a shallow copy', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.unify();
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
            });
            it('should remove all duplicates', function () {
                var ra1 = ra('A2:C4 B3:D5 A2:C4 B3:C4 B3:D5 B3:C4'), ra2 = ra1.unify();
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
                expect(ra2[2]).to.equal(ra1[3]);
            });
            it('should accept empty array', function () {
                expect(new RangeArray().unify()).to.be.empty;
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('merge');
            });
            it('should not merge distinct ranges', function () {
                expect(ra('C6:D7 B3:C4 E4:F5').merge()).to.satisfy(unorderedRangesMatcher(ra('B3:C4 E4:F5 C6:D7')));
            });
            it('should merge adjacent ranges', function () {
                expect(ra('B5:C6 B3:C4').merge()).to.deep.equal(ra('B3:C6'));
                expect(ra('D3:E4 B3:C4').merge()).to.deep.equal(ra('B3:E4'));
                expect(ra('D3:E4 B5:C6 B3:C4').merge()).to.satisfy(unorderedRangesMatcher(ra('B3:E4 B5:C6')));
                expect(ra('D3:E4 B5:C6 B3:C4 D5:E6').merge()).to.deep.equal(ra('B3:E6'));
                expect(ra('B3:B5 B6:D6 E4:E6 C3:E3 C4:D5').merge()).to.deep.equal(ra('B3:E6'));
            });
            it('should merge overlapping ranges', function () {
                expect(ra('B4:C6 B3:C5').merge()).to.deep.equal(ra('B3:C6'));
                expect(ra('C3:E4 B3:D4').merge()).to.deep.equal(ra('B3:E4'));
                expect(ra('C3:E5 E5:G7').merge()).to.satisfy(unorderedRangesMatcher(ra('C3:E4 C5:G5 E6:G7')));
                expect(ra('C3:E4 B4:C6 B3:D4').merge()).to.satisfy(unorderedRangesMatcher(ra('B3:E4 B5:C6')));
                expect(ra('C3:E4 B5:D6 B3:D4 D4:E6').merge()).to.deep.equal(ra('B3:E6'));
                expect(ra('B3:B6 B6:E6 E3:E6 B3:E3 C3:C5 C4:E4 D4:D6 B5:D5').merge()).to.deep.equal(ra('B3:E6'));
            });
            it('should split ranges to prevent covering a cell multiple times', function () {
                expect(ra('C3:C5 B3:D3').merge()).to.satisfy(unorderedRangesMatcher(ra('B3:D3 C4:C5')));
                expect(ra('C3:C5 B4:D4').merge()).to.satisfy(unorderedRangesMatcher(ra('C3:C3 B4:D4 C5:C5')));
                expect(ra('C3:C5 B5:D5').merge()).to.satisfy(unorderedRangesMatcher(ra('C3:C4 B5:D5')));
            });
            it('should accept empty array', function () {
                var ra1 = new RangeArray(), ra2 = ra1.merge();
                expect(ra2).to.be.empty;
                expect(ra2).to.not.equal(ra1);
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('intersect');
            });
            var ra1 = ra('B3:D5 D5:F7');
            it('should return empty array for distinct ranges', function () {
                expect(ra1.intersect(ra('E3:F4 B6:C7'))).to.be.empty;
            });
            it('should return intersection ranges', function () {
                expect(ra1.intersect(ra1)).to.deep.equal(ra('B3:D5 D5:D5 D5:D5 D5:F7'));
                expect(ra1.intersect(ra('B3:D5'))).to.deep.equal(ra('B3:D5 D5:D5'));
                expect(ra1.intersect(ra('D5:E6'))).to.deep.equal(ra('D5:D5 D5:E6'));
                expect(ra1.intersect(ra('A1:C9'))).to.deep.equal(ra('B3:C5'));
                expect(ra1.intersect(ra('A5:I6'))).to.deep.equal(ra('B5:D5 D5:F6'));
                expect(ra1.intersect(ra('D1:D9 A5:I5'))).to.deep.equal(ra('D3:D5 B5:D5 D5:D7 D5:F5'));
            });
            it('should accept empty arrays', function () {
                var ra0 = new RangeArray();
                expect(ra0.intersect(ra1)).to.be.empty;
                expect(ra1.intersect(ra0)).to.be.empty;
                expect(ra0.intersect(ra0)).to.be.empty;
            });
        });

        describe('method "difference"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('difference');
            });
            var ra1 = ra('B3:D5 D5:F7');
            it('should return remaining ranges', function () {
                expect(ra1.difference(ra('E3:F4 B6:C7'))).to.deep.equal(ra1);
                expect(ra1.difference(ra('E3:F4 B6:C7'))).to.not.equal(ra1);
                expect(ra1.difference(ra('D3:F5 B5:D7'))).to.deep.equal(ra('B3:C4 E6:F7'));
                expect(ra1.difference(ra('C4:E6'))).to.deep.equal(ra('B3:D3 B4:B5 F5:F6 D7:F7'));
                expect(ra1.difference(ra('C4:C4 E6:E6'))).to.deep.equal(ra('B3:D3 B4:B4 D4:D4 B5:D5 D5:F5 D6:D6 F6:F6 D7:F7'));
                expect(ra1.difference(ra('C4:C4 E6:E6 D5:D5'))).to.deep.equal(ra('B3:D3 B4:B4 D4:D4 B5:C5 E5:F5 D6:D6 F6:F6 D7:F7'));
                expect(ra1.difference(ra('B3:F7'))).to.be.empty;
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra1.difference(ra0)).to.deep.equal(ra1);
                expect(ra1.difference(ra0)).to.not.equal(ra1);
                expect(ra0.difference(ra1)).to.be.empty;
                expect(ra0.difference(ra0)).to.be.empty;
            });
            it('should accept single range object instead of array', function () {
                expect(ra('B3:D5 D5:F7').difference(r('C4:E6'))).to.deep.equal(ra('B3:D3 B4:B5 F5:F6 D7:F7'));
            });
        });

        describe('method "shortenTo"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('shortenTo');
            });
            var ra1 = ra('A1:C3 C1:E3 E1:G3');
            it('should shorten ranges', function () {
                expect(ra1.shortenTo(-1)).to.be.empty;
                expect(ra1.shortenTo(0)).to.be.empty;
                expect(ra1.shortenTo(1)).to.deep.equal(ra('A1:A1'));
                expect(ra1.shortenTo(2)).to.deep.equal(ra('A1:B1'));
                expect(ra1.shortenTo(3)).to.deep.equal(ra('A1:C1'));
                expect(ra1.shortenTo(4)).to.deep.equal(ra('A1:C1 A2:A2'));
                expect(ra1.shortenTo(5)).to.deep.equal(ra('A1:C1 A2:B2'));
                expect(ra1.shortenTo(6)).to.deep.equal(ra('A1:C2'));
                expect(ra1.shortenTo(7)).to.deep.equal(ra('A1:C2 A3:A3'));
                expect(ra1.shortenTo(8)).to.deep.equal(ra('A1:C2 A3:B3'));
                expect(ra1.shortenTo(9)).to.deep.equal(ra('A1:C3'));
                expect(ra1.shortenTo(10)).to.deep.equal(ra('A1:C3 C1:C1'));
                expect(ra1.shortenTo(11)).to.deep.equal(ra('A1:C3 C1:D1'));
                expect(ra1.shortenTo(12)).to.deep.equal(ra('A1:C3 C1:E1'));
                expect(ra1.shortenTo(13)).to.deep.equal(ra('A1:C3 C1:E1 C2:C2'));
                expect(ra1.shortenTo(14)).to.deep.equal(ra('A1:C3 C1:E1 C2:D2'));
                expect(ra1.shortenTo(15)).to.deep.equal(ra('A1:C3 C1:E2'));
                expect(ra1.shortenTo(16)).to.deep.equal(ra('A1:C3 C1:E2 C3:C3'));
                expect(ra1.shortenTo(17)).to.deep.equal(ra('A1:C3 C1:E2 C3:D3'));
                expect(ra1.shortenTo(18)).to.deep.equal(ra('A1:C3 C1:E3'));
                expect(ra1.shortenTo(19)).to.deep.equal(ra('A1:C3 C1:E3 E1:E1'));
                expect(ra1.shortenTo(20)).to.deep.equal(ra('A1:C3 C1:E3 E1:F1'));
                expect(ra1.shortenTo(21)).to.deep.equal(ra('A1:C3 C1:E3 E1:G1'));
                expect(ra1.shortenTo(22)).to.deep.equal(ra('A1:C3 C1:E3 E1:G1 E2:E2'));
                expect(ra1.shortenTo(23)).to.deep.equal(ra('A1:C3 C1:E3 E1:G1 E2:F2'));
                expect(ra1.shortenTo(24)).to.deep.equal(ra('A1:C3 C1:E3 E1:G2'));
                expect(ra1.shortenTo(25)).to.deep.equal(ra('A1:C3 C1:E3 E1:G2 E3:E3'));
                expect(ra1.shortenTo(26)).to.deep.equal(ra('A1:C3 C1:E3 E1:G2 E3:F3'));
            });
            it('should not shorten ranges', function () {
                expect(ra1.shortenTo(27)).to.deep.equal(ra1);
                expect(ra1.shortenTo(27)).to.not.equal(ra1);
                expect(ra1.shortenTo(28)).to.deep.equal(ra1);
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra0.shortenTo(0)).to.be.empty;
                expect(ra0.shortenTo(9)).to.be.empty;
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('toString');
            });
            var ra1 = ra('A2:C4 B3:D5');
            it('should stringify the ranges', function () {
                expect(ra1.toString()).to.equal('A2:C4,B3:D5');
            });
            it('should use the separator string', function () {
                expect(ra1.toString(' + ')).to.equal('A2:C4 + B3:D5');
            });
            it('should stringify implicitly', function () {
                expect('<' + ra1 + '>').to.equal('<A2:C4,B3:D5>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('toJSON');
            });
            var ra1 = ra('A2:C4 B3:D5'), ra2 = [{ start: [0, 1], end: [2, 3] }, { start: [1, 2], end: [3, 4] }];
            it('should convert to JSON data', function () {
                expect(ra1.toJSON()).to.deep.equal(ra2);
            });
            it('should stringify to JSON', function () {
                expect(JSON.parse(JSON.stringify(ra1))).to.deep.equal(ra2);
            });
        });
    });

    // ========================================================================
});
