/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/range3d',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/range3darray'
], function (Range, Range3D, RangeArray, Range3DArray) {

    'use strict';

    // class Range3DArray =====================================================

    describe('Spreadsheet class Range3DArray', function () {

        it('should exist', function () {
            expect(Range3DArray).to.be.a('function');
        });
        it('should be a subclass of Array', function () {
            expect(new Range3DArray()).to.be.an['instanceof'](Array);
        });

        // private helpers ----------------------------------------------------

        var r = Range.parse;
        function ra(str) { return new RangeArray(str.split(/\s+/).map(r)); }

        function r3d(str) {
            var matches = /^(\d+):(\d+)!(.*)$/.exec(str);
            return Range3D.createFromRange(r(matches[3]), parseInt(matches[1], 10), parseInt(matches[2], 10));
        }
        function r3da(str) { return new Range3DArray(str.split(/\s+/).map(r3d)); }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a range array', function () {
                var ra = new Range3DArray();
                expect(ra).to.be.an['instanceof'](Range3DArray);
                expect(ra).to.be.empty;
            });
            var r1 = r3d('1:2!A2:C4'), r2 = r3d('6:8!B3:D5');
            it('should insert single ranges', function () {
                var ra1 = new Range3DArray(r1);
                expect(ra1).to.have.length(1);
                expect(ra1[0]).to.equal(r1);
                var ra2 = new Range3DArray(r1, r2, r1);
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
            });
            it('should insert plain arrays of ranges', function () {
                var ra1 = new Range3DArray([r1]);
                expect(ra1).to.have.length(1);
                expect(ra1[0]).to.equal(r1);
                var ra2 = new Range3DArray([r1, r2], [], [r1]);
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
            });
            it('should copy-construct range arrays', function () {
                var ra0 = new Range3DArray(r1, r2),
                    ra1 = new Range3DArray(ra0);
                expect(ra1).to.have.length(2);
                expect(ra1[0]).to.equal(r1);
                expect(ra1[1]).to.equal(r2);
                var ra2 = new Range3DArray(ra0, new Range3DArray(), ra1);
                expect(ra2).to.have.length(4);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
                expect(ra2[3]).to.equal(r2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "createFromRanges"', function () {
            it('should exist', function () {
                expect(Range3DArray).itself.to.respondTo('createFromRanges');
            });
            var ra1 = ra('A2:C4 B3:D5'), ra2 = r3da('6:8!A2:C4 6:8!B3:D5');
            it('should return a range array', function () {
                var ra3 = Range3DArray.createFromRanges(ra1, 6, 8);
                expect(ra3).to.be.an['instanceof'](Range3DArray);
                expect(ra3).to.have.length(2);
                expect(ra3[0]).to.be.an['instanceof'](Range3D);
                expect(ra3[1]).to.be.an['instanceof'](Range3D);
            });
            it('should return ranges with adjusted sheet indexes', function () {
                expect(Range3DArray.createFromRanges(ra1, 6, 8)).to.deep.equal(ra2);
                expect(Range3DArray.createFromRanges(ra1, 8, 6)).to.deep.equal(ra2);
            });
            it('should accept single sheet index', function () {
                var ra3 = Range3DArray.createFromRanges(ra1, 6);
                expect(ra3[0]).to.have.property('sheet1', 6);
                expect(ra3[0]).to.have.property('sheet2', 6);
                expect(ra3[1]).to.have.property('sheet1', 6);
                expect(ra3[1]).to.have.property('sheet2', 6);
            });
            it('should clone addresses', function () {
                var ra3 = Range3DArray.createFromRanges(ra1, 6, 8);
                expect(ra3[0].start).to.deep.equal(ra1[0].start);
                expect(ra3[0].start).to.not.equal(ra1[0].start);
                expect(ra3[1].end).to.deep.equal(ra1[1].end);
                expect(ra3[1].end).to.not.equal(ra1[1].end);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('clone');
            });
            var ra1 = r3da('1:1!A2:C4 6:8!B3:D5');
            it('should return a shallow clone', function () {
                var ra2 = ra1.clone();
                expect(ra2).to.be.an['instanceof'](Range3DArray);
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
                expect(ra2).to.deep.equal(ra1);
            });
            it('should return a deep clone', function () {
                var ra2 = ra1.clone(true);
                expect(ra2).to.be.an['instanceof'](Range3DArray);
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.not.equal(ra1[0]);
                expect(ra2[1]).to.not.equal(ra1[1]);
                expect(ra2).to.deep.equal(ra1);
            });
        });

        describe('method "cells"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('cells');
            });
            it('should count cells in ranges', function () {
                expect(r3da('1:1!A1:C3 6:8!B2:D4').cells()).to.equal(36);
                expect(new Range3DArray().cells()).to.equal(0);
            });
        });

        describe('method "singleSheet"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('singleSheet');
            });
            it('should return true if all ranges refer to the same single sheet', function () {
                expect(r3da('1:1!A1:C3 1:1!B2:D4').singleSheet()).to.equal(true);
            });
            it('should return false if ranges refer to different single sheets', function () {
                expect(r3da('1:1!A1:C3 2:2!B2:D4').singleSheet()).to.equal(false);
            });
            it('should return false if ranges refer to multiple sheets', function () {
                expect(r3da('1:2!A1:C3 1:2!B2:D4').singleSheet()).to.equal(false);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('overlaps');
            });
            var ra1 = r3da('1:2!A1:B2 2:3!B2:C3 3:4!C3:D4');
            it('should return false for distinct ranges', function () {
                expect(ra1.overlaps(r3da('0:0!A1:D4 3:3!A1:A1'))).to.equal(false);
            });
            it('should return true for overlapping ranges', function () {
                expect(ra1.overlaps(r3da('0:0!A1:D4 3:3!A1:B2'))).to.equal(true);
            });
            it('should accept empty array', function () {
                var ra0 = new Range3DArray();
                expect(ra0.overlaps(ra1)).to.equal(false);
                expect(ra1.overlaps(ra0)).to.equal(false);
                expect(ra0.overlaps(ra0)).to.equal(false);
            });
            it('should accept single range address', function () {
                expect(ra1.overlaps(r3d('0:0!A1:D4'))).to.equal(false);
                expect(ra1.overlaps(r3d('3:3!A1:B2'))).to.equal(true);
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('boundary');
            });
            it('should return the bounding range', function () {
                expect(r3da('2:4!B3:C4 1:3!C4:D5').boundary()).to.deep.equal(r3d('1:4!B3:D5'));
                expect(r3da('6:8!C4:D5 0:0!B3:C4 7:7!A2:B3').boundary()).to.deep.equal(r3d('0:8!A2:D5'));
            });
            it('should return a clone of a single range', function () {
                var ra1 = r3da('0:1!B3:C4'), r1 = ra1.boundary();
                expect(r1).to.deep.equal(ra1[0]);
                expect(r1).to.not.equal(ra1[0]);
            });
            it('should return null for an empty array', function () {
                expect(new Range3DArray().boundary()).to.equal(null);
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('intersect');
            });
            var ra1 = r3da('4:6!B3:D5 6:8!D5:F7');
            it('should return empty array for distinct ranges', function () {
                expect(ra1.intersect(r3da('4:8!E3:F4 4:8!B6:C7'))).to.be.empty;
            });
            it('should return intersection ranges', function () {
                expect(ra1.intersect(ra1)).to.deep.equal(r3da('4:6!B3:D5 6:6!D5:D5 6:6!D5:D5 6:8!D5:F7'));
                expect(ra1.intersect(r3da('4:6!B3:D5'))).to.deep.equal(r3da('4:6!B3:D5 6:6!D5:D5'));
                expect(ra1.intersect(r3da('6:7!D5:E6'))).to.deep.equal(r3da('6:6!D5:D5 6:7!D5:E6'));
                expect(ra1.intersect(r3da('5:7!A1:C9'))).to.deep.equal(r3da('5:6!B3:C5'));
                expect(ra1.intersect(r3da('5:7!A5:I6'))).to.deep.equal(r3da('5:6!B5:D5 6:7!D5:F6'));
                expect(ra1.intersect(r3da('3:7!D1:D9 5:9!A5:I5'))).to.deep.equal(r3da('4:6!D3:D5 5:6!B5:D5 6:7!D5:D7 6:8!D5:F5'));
            });
            it('should accept empty arrays', function () {
                var ra0 = new Range3DArray();
                expect(ra0.intersect(ra1)).to.be.empty;
                expect(ra1.intersect(ra0)).to.be.empty;
                expect(ra0.intersect(ra0)).to.be.empty;
            });
            it('should accept single range address', function () {
                expect(ra1.intersect(r3d('4:6!B3:D5'))).to.deep.equal(r3da('4:6!B3:D5 6:6!D5:D5'));
            });
        });

        describe('method "shortenTo"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('shortenTo');
            });
            var ra1 = r3da('0:0!A1:A1 1:3!B2:D4 4:4!E5:E5');
            it('should shorten ranges', function () {
                expect(ra1.shortenTo(-1)).to.be.empty;
                expect(ra1.shortenTo(0)).to.be.empty;
                expect(ra1.shortenTo(1)).to.deep.equal(r3da('0:0!A1:A1'));
                expect(ra1.shortenTo(2)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:B2'));
                expect(ra1.shortenTo(3)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:C2'));
                expect(ra1.shortenTo(4)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D2'));
                expect(ra1.shortenTo(5)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D2 1:1!B3:B3'));
                expect(ra1.shortenTo(6)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D2 1:1!B3:C3'));
                expect(ra1.shortenTo(7)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D3'));
                expect(ra1.shortenTo(8)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D3 1:1!B4:B4'));
                expect(ra1.shortenTo(9)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D3 1:1!B4:C4'));
                expect(ra1.shortenTo(10)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4'));
                expect(ra1.shortenTo(11)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:B2'));
                expect(ra1.shortenTo(12)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:C2'));
                expect(ra1.shortenTo(13)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:D2'));
                expect(ra1.shortenTo(14)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:D2 2:2!B3:B3'));
                expect(ra1.shortenTo(15)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:D2 2:2!B3:C3'));
                expect(ra1.shortenTo(16)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:D3'));
                expect(ra1.shortenTo(17)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:D3 2:2!B4:B4'));
                expect(ra1.shortenTo(18)).to.deep.equal(r3da('0:0!A1:A1 1:1!B2:D4 2:2!B2:D3 2:2!B4:C4'));
                expect(ra1.shortenTo(19)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4'));
                expect(ra1.shortenTo(20)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:B2'));
                expect(ra1.shortenTo(21)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:C2'));
                expect(ra1.shortenTo(22)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:D2'));
                expect(ra1.shortenTo(23)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:D2 3:3!B3:B3'));
                expect(ra1.shortenTo(24)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:D2 3:3!B3:C3'));
                expect(ra1.shortenTo(25)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:D3'));
                expect(ra1.shortenTo(26)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:D3 3:3!B4:B4'));
                expect(ra1.shortenTo(27)).to.deep.equal(r3da('0:0!A1:A1 1:2!B2:D4 3:3!B2:D3 3:3!B4:C4'));
                expect(ra1.shortenTo(28)).to.deep.equal(r3da('0:0!A1:A1 1:3!B2:D4'));
            });
            it('should not shorten ranges', function () {
                expect(ra1.shortenTo(29)).to.deep.equal(ra1);
                expect(ra1.shortenTo(29)).to.not.equal(ra1);
                expect(ra1.shortenTo(30)).to.deep.equal(ra1);
            });
            it('should accept empty array', function () {
                var ra0 = new Range3DArray();
                expect(ra0.shortenTo(0)).to.be.empty;
                expect(ra0.shortenTo(9)).to.be.empty;
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('toString');
            });
            var ra1 = r3da('1:1!A2:C4 6:8!B3:D5');
            it('should stringify the ranges', function () {
                expect(ra1.toString()).to.equal('1:1!A2:C4,6:8!B3:D5');
            });
            it('should use the separator string', function () {
                expect(ra1.toString(' + ')).to.equal('1:1!A2:C4 + 6:8!B3:D5');
            });
            it('should stringify implicitly', function () {
                expect('<' + ra1 + '>').to.equal('<1:1!A2:C4,6:8!B3:D5>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Range3DArray).to.respondTo('toJSON');
            });
            var ra1 = r3da('1:1!A2:C4 6:8!B3:D5'), ra2 = [{ sheet1: 1, sheet2: 1, start: [0, 1], end: [2, 3] }, { sheet1: 6, sheet2: 8, start: [1, 2], end: [3, 4] }];
            it('should convert to JSON data', function () {
                expect(ra1.toJSON()).to.deep.equal(ra2);
            });
            it('should stringify to JSON', function () {
                expect(JSON.parse(JSON.stringify(ra1))).to.deep.equal(ra2);
            });
        });
    });

    // ========================================================================
});
