/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/addressarray'
], function (Address, AddressArray) {

    'use strict';

    // class AddressArray =====================================================

    describe('Spreadsheet class AddressArray', function () {

        it('should exist', function () {
            expect(AddressArray).to.be.a('function');
        });
        it('should be a subclass of Array', function () {
            expect(new AddressArray()).to.be.an['instanceof'](Array);
        });

        // private helpers ----------------------------------------------------

        var a = Address.parse;
        function aa(str) { return new AddressArray(str.split(/\s+/).map(a)); }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create an address array', function () {
                var aa1 = new AddressArray();
                expect(aa1).to.be.an['instanceof'](AddressArray);
                expect(aa1).to.be.empty;
            });
            var a1 = a('A2'), a2 = a('C4');
            it('should insert single addresses', function () {
                var aa1 = new AddressArray(a1);
                expect(aa1).to.have.length(1);
                expect(aa1[0]).to.equal(a1);
                var aa2 = new AddressArray(a1, a2, a1);
                expect(aa2).to.have.length(3);
                expect(aa2[0]).to.equal(a1);
                expect(aa2[1]).to.equal(a2);
                expect(aa2[2]).to.equal(a1);
            });
            it('should insert plain arrays of addresses', function () {
                var aa1 = new AddressArray([a1]);
                expect(aa1).to.have.length(1);
                expect(aa1[0]).to.equal(a1);
                var aa2 = new AddressArray([a1, a2], [], [a1]);
                expect(aa2).to.have.length(3);
                expect(aa2[0]).to.equal(a1);
                expect(aa2[1]).to.equal(a2);
                expect(aa2[2]).to.equal(a1);
            });
            it('should copy-construct address arrays', function () {
                var aa0 = new AddressArray(a1, a2),
                    aa1 = new AddressArray(aa0);
                expect(aa1).to.have.length(2);
                expect(aa1[0]).to.equal(a1);
                expect(aa1[1]).to.equal(a2);
                var aa2 = new AddressArray(aa0, new AddressArray(), aa1);
                expect(aa2).to.have.length(4);
                expect(aa2[0]).to.equal(a1);
                expect(aa2[1]).to.equal(a2);
                expect(aa2[2]).to.equal(a1);
                expect(aa2[3]).to.equal(a2);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(AddressArray).to.respondTo('clone');
            });
            it('should return a shallow clone', function () {
                var aa1 = aa('A2 C4'), aa2 = aa1.clone();
                expect(aa2).to.be.an['instanceof'](AddressArray);
                expect(aa2).to.not.equal(aa1);
                expect(aa2[0]).to.equal(aa1[0]);
                expect(aa2[1]).to.equal(aa1[1]);
                expect(aa2).to.deep.equal(aa1);
            });
            it('should return a deep clone', function () {
                var aa1 = aa('A2 C4'), aa2 = aa1.clone(true);
                expect(aa2).to.be.an['instanceof'](AddressArray);
                expect(aa2).to.not.equal(aa1);
                expect(aa2[0]).to.not.equal(aa1[0]);
                expect(aa2[1]).to.not.equal(aa1[1]);
                expect(aa2).to.deep.equal(aa1);
            });
        });

        describe('method "unify"', function () {
            it('should exist', function () {
                expect(AddressArray).to.respondTo('unify');
            });
            it('should return a shallow copy', function () {
                var aa1 = aa('A2 C4'), aa2 = aa1.unify();
                expect(aa2).to.not.equal(aa1);
                expect(aa2[0]).to.equal(aa1[0]);
                expect(aa2[1]).to.equal(aa1[1]);
            });
            it('should remove all duplicates', function () {
                var aa1 = aa('A2 C4 A2 A4 C4 A4'), aa2 = aa1.unify();
                expect(aa2).to.have.length(3);
                expect(aa2[0]).to.equal(aa1[0]);
                expect(aa2[1]).to.equal(aa1[1]);
                expect(aa2[2]).to.equal(aa1[3]);
            });
            it('should accept empty array', function () {
                expect(new AddressArray().unify()).to.be.empty;
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(AddressArray).to.respondTo('toString');
            });
            var aa1 = aa('A2 C4');
            it('should stringify the addresses', function () {
                expect(aa1.toString()).to.equal('A2,C4');
            });
            it('should use the separator string', function () {
                expect(aa1.toString(' + ')).to.equal('A2 + C4');
            });
            it('should stringify implicitly', function () {
                expect('<' + aa1 + '>').to.equal('<A2,C4>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(AddressArray).to.respondTo('toJSON');
            });
            var aa1 = aa('A2 C4'), aa2 = [[0, 1], [2, 3]];
            it('should convert to JSON data', function () {
                expect(aa1.toJSON()).to.deep.equal(aa2);
            });
            it('should stringify to JSON', function () {
                expect(JSON.parse(JSON.stringify(aa1))).to.deep.equal(aa2);
            });
        });
    });

    // ========================================================================
});
