/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/model',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/cellref',
    'io.ox/office/spreadsheet/model/formula/sheetref',
    'io.ox/office/spreadsheet/model/formula/tokens'
], function (SheetUtils, SpreadsheetModel, Matrix, CellRef, SheetRef, Tokens) {

    'use strict';

    // global helpers =========================================================

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode,
        Interval = SheetUtils.Interval,
        Address = SheetUtils.Address,
        Range = SheetUtils.Range;

    // create an address or a range instance on-the-fly from A1 notation
    var a = Address.parse, r = Range.parse;

    // create a column/row interval instance on-the-fly
    function i(first, last) { return new Interval(first, last); }

    // create a CellRef instance on-the-fly from $1.$1 notation (column/row offsets)
    function cell(text) {
        var m = /^(\$)?(-?\d+)\.(\$)?(-?\d+)$/i.exec(text);
        return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
    }

    // create a SheetRef instance on-the-fly
    function sheet(sheet, abs) {
        return new SheetRef(sheet, !_.isBoolean(abs) || abs);
    }

    // the document model for testing
    var docModel = null;

    // grammar configuration for native and UI formulas
    var opConfig = null, uiConfig = null;

    // the operations to be applied by the document model
    var OPERATIONS = [
        { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
        { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
        { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
        { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
        { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
        { name: 'insertName', exprName: 'name', formula: 'Sheet1!$A$1' }, // global name
        { name: 'insertName', sheet: 1, exprName: 'name', formula: 'Sheet1!$A$1' } // sheet-local name
    ];

    // initialize formula resource data, and get grammar configurations from document
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml', OPERATIONS).done(function (app) {

            docModel = app.getModel();

            // get the formula grammar configurations
            docModel.onInitFormulaResource(function () {
                opConfig = docModel.getGrammarConfig('op');
                uiConfig = docModel.getGrammarConfig('ui');
                done();
            });
        });
    });

    // static class Tokens ====================================================

    describe('Spreadsheet module Tokens', function () {
        it('should exist', function () {
            expect(Tokens).to.be.an('object');
        });
    });

    // class FixedToken =======================================================

    describe('Spreadsheet class FixedToken', function () {
        it('should exist', function () {
            expect(Tokens.FixedToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.FixedToken('type', 'text');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('type');
            });
        });

        describe('method "isType"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('isType');
            });
            it('should test the token type', function () {
                expect(token.isType('type')).to.equal(true);
                expect(token.isType('other')).to.equal(false);
            });
        });

        describe('method "matchesType"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('matchesType');
            });
            it('should match the token type', function () {
                expect(token.matchesType(/^t/)).to.equal(true);
                expect(token.matchesType(/^o/)).to.equal(false);
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal('text');
                expect(token.getText(uiConfig)).to.equal('text');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('type[text]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('getValue');
            });
            it('should return the value', function () {
                expect(token.getValue()).to.equal('text');
            });
        });

        describe('method "setValue"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('setValue');
            });
            it('should return whether the value changes', function () {
                var token1 = new Tokens.FixedToken('type', 'text');
                expect(token1.setValue('text')).to.equal(false);
                expect(token1.getValue()).to.equal('text');
                expect(token1.setValue('other')).to.equal(true);
                expect(token1.getValue()).to.equal('other');
                expect(token1.getText(opConfig)).to.equal('other');
                expect(token1.getText(uiConfig)).to.equal('other');
            });
        });

        describe('method "appendValue"', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.respondTo('appendValue');
            });
            it('should return whether the value changes', function () {
                var token1 = new Tokens.FixedToken('type', 'other');
                expect(token1.appendValue('')).to.equal(false);
                expect(token1.getValue()).to.equal('other');
                expect(token1.appendValue('text')).to.equal(true);
                expect(token1.getValue()).to.equal('othertext');
                expect(token1.getText(opConfig)).to.equal('othertext');
                expect(token1.getText(uiConfig)).to.equal('othertext');
            });
        });
    });

    // class OperatorToken ====================================================

    describe('Spreadsheet class OperatorToken', function () {
        it('should exist', function () {
            expect(Tokens.OperatorToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.OperatorToken('list');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.OperatorToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('op');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.OperatorToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal(',');
                expect(token.getText(uiConfig)).to.equal(';');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.OperatorToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('op[list]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(Tokens.OperatorToken).to.respondTo('getValue');
            });
            it('should return the value', function () {
                expect(token.getValue()).to.equal('list');
            });
        });
    });

    // class SeparatorToken ===================================================

    describe('Spreadsheet class SeparatorToken', function () {
        it('should exist', function () {
            expect(Tokens.SeparatorToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.SeparatorToken();

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.SeparatorToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('sep');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.SeparatorToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal(',');
                expect(token.getText(uiConfig)).to.equal(';');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.SeparatorToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('sep');
            });
        });
    });

    // class ParenthesisToken =================================================

    describe('Spreadsheet class ParenthesisToken', function () {
        it('should exist', function () {
            expect(Tokens.ParenthesisToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var openToken = new Tokens.ParenthesisToken(true),
            closeToken = new Tokens.ParenthesisToken(false);

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.ParenthesisToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(openToken.getType()).to.equal('open');
                expect(closeToken.getType()).to.equal('close');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.ParenthesisToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(openToken.getText(opConfig)).to.equal('(');
                expect(openToken.getText(uiConfig)).to.equal('(');
                expect(closeToken.getText(opConfig)).to.equal(')');
                expect(closeToken.getText(uiConfig)).to.equal(')');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.ParenthesisToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(openToken.toString()).to.equal('open');
                expect(closeToken.toString()).to.equal('close');
            });
        });
    });

    // class MatrixDelimiterToken =============================================

    describe('Spreadsheet class MatrixDelimiterToken', function () {
        it('should exist', function () {
            expect(Tokens.MatrixDelimiterToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var openToken = new Tokens.MatrixDelimiterToken('mat_open'),
            rowToken = new Tokens.MatrixDelimiterToken('mat_row'),
            colToken = new Tokens.MatrixDelimiterToken('mat_col'),
            closeToken = new Tokens.MatrixDelimiterToken('mat_close');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixDelimiterToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(openToken.getType()).to.equal('mat_open');
                expect(rowToken.getType()).to.equal('mat_row');
                expect(colToken.getType()).to.equal('mat_col');
                expect(closeToken.getType()).to.equal('mat_close');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixDelimiterToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(openToken.getText(opConfig)).to.equal('{');
                expect(openToken.getText(uiConfig)).to.equal('{');
                expect(rowToken.getText(opConfig)).to.equal(';');
                expect(rowToken.getText(uiConfig)).to.equal('|');
                expect(colToken.getText(opConfig)).to.equal(',');
                expect(colToken.getText(uiConfig)).to.equal(';');
                expect(closeToken.getText(opConfig)).to.equal('}');
                expect(closeToken.getText(uiConfig)).to.equal('}');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixDelimiterToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(openToken.toString()).to.equal('mat_open');
                expect(rowToken.toString()).to.equal('mat_row');
                expect(colToken.toString()).to.equal('mat_col');
                expect(closeToken.toString()).to.equal('mat_close');
            });
        });
    });

    // class LiteralToken =====================================================

    describe('Spreadsheet class LiteralToken', function () {
        it('should exist', function () {
            expect(Tokens.LiteralToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var numToken = new Tokens.LiteralToken(12.5),
            strToken = new Tokens.LiteralToken('a"b"c'),
            falseToken = new Tokens.LiteralToken(false),
            trueToken = new Tokens.LiteralToken(true),
            errToken = new Tokens.LiteralToken(ErrorCode.REF),
            nullToken = new Tokens.LiteralToken(null);

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.LiteralToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(numToken.getType()).to.equal('lit');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.LiteralToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(numToken.getText(opConfig)).to.equal('12.5');
                expect(numToken.getText(uiConfig)).to.equal('12,5');
                expect(strToken.getText(opConfig)).to.equal('"a""b""c"');
                expect(strToken.getText(uiConfig)).to.equal('"a""b""c"');
                expect(falseToken.getText(opConfig)).to.equal('FALSE');
                expect(falseToken.getText(uiConfig)).to.equal('FALSCH');
                expect(trueToken.getText(opConfig)).to.equal('TRUE');
                expect(trueToken.getText(uiConfig)).to.equal('WAHR');
                expect(errToken.getText(opConfig)).to.equal('#REF!');
                expect(errToken.getText(uiConfig)).to.equal('#BEZUG!');
                expect(nullToken.getText(opConfig)).to.equal('');
                expect(nullToken.getText(uiConfig)).to.equal('');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.LiteralToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(numToken.toString()).to.equal('lit[12.5]');
                expect(strToken.toString()).to.equal('lit["a""b""c"]');
                expect(falseToken.toString()).to.equal('lit[FALSE]');
                expect(trueToken.toString()).to.equal('lit[TRUE]');
                expect(errToken.toString()).to.equal('lit[#REF]');
                expect(nullToken.toString()).to.equal('lit[null]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(Tokens.LiteralToken).to.respondTo('getValue');
            });
            it('should return the value', function () {
                expect(numToken.getValue()).to.equal(12.5);
                expect(strToken.getValue()).to.equal('a"b"c');
                expect(falseToken.getValue()).to.equal(false);
                expect(trueToken.getValue()).to.equal(true);
                expect(errToken.getValue()).to.equal(ErrorCode.REF);
                expect(nullToken.getValue()).to.equal(null);
            });
        });

        describe('method "setValue"', function () {
            it('should exist', function () {
                expect(Tokens.LiteralToken).to.respondTo('setValue');
            });
            it('should return whether the value changes', function () {
                var token = new Tokens.LiteralToken(12.5);
                expect(token.setValue(12.5)).to.equal(false);
                expect(token.getValue()).to.equal(12.5);
                expect(token.getText(uiConfig)).to.equal('12,5');
                expect(token.setValue(1)).to.equal(true);
                expect(token.getValue()).to.equal(1);
                expect(token.getText(uiConfig)).to.equal('1');
                expect(token.setValue('a"b"c')).to.equal(true);
                expect(token.getValue()).to.equal('a"b"c');
                expect(token.getText(uiConfig)).to.equal('"a""b""c"');
                expect(token.setValue(true)).to.equal(true);
                expect(token.getValue()).to.equal(true);
                expect(token.getText(uiConfig)).to.equal('WAHR');
                expect(token.setValue(ErrorCode.REF)).to.equal(true);
                expect(token.getValue()).to.equal(ErrorCode.REF);
                expect(token.getText(uiConfig)).to.equal('#BEZUG!');
                expect(token.setValue(null)).to.equal(true);
                expect(token.getValue()).to.equal(null);
                expect(token.getText(uiConfig)).to.equal('');
            });
        });
    });

    // class MatrixToken ======================================================

    describe('Spreadsheet class MatrixToken', function () {
        it('should exist', function () {
            expect(Tokens.MatrixToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var matrix = new Matrix([[12.5, 'a"b"c'], [true, ErrorCode.REF]]),
            token = new Tokens.MatrixToken(matrix);

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('mat');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal('{12.5,"a""b""c";TRUE,#REF!}');
                expect(token.getText(uiConfig)).to.equal('{12,5;"a""b""c"|WAHR;#BEZUG!}');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('mat[{12.5;"a""b""c"|TRUE;#REF}]');
            });
        });

        describe('method "getMatrix"', function () {
            it('should exist', function () {
                expect(Tokens.MatrixToken).to.respondTo('getMatrix');
            });
            it('should return the matrix', function () {
                expect(token.getMatrix()).to.equal(matrix);
            });
        });
    });

    // class FunctionToken ====================================================

    describe('Spreadsheet class FunctionToken', function () {
        it('should exist', function () {
            expect(Tokens.FunctionToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.FunctionToken('SUM');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.FunctionToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('func');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.FunctionToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal('SUM');
                expect(token.getText(uiConfig)).to.equal('SUMME');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.FunctionToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('func[SUM]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(Tokens.FunctionToken).to.respondTo('getValue');
            });
            it('should return the value of the token', function () {
                expect(token.getValue()).to.equal('SUM');
            });
        });
    });

    // class ReferenceToken ===================================================

    describe('Spreadsheet class ReferenceToken', function () {
        it('should exist', function () {
            expect(Tokens.ReferenceToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var cellToken, rangeToken, errToken, sheet1Token, sheet2Token, sheetErrToken;
        before(function () {
            cellToken = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, null, null);
            rangeToken = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
            errToken = new Tokens.ReferenceToken(docModel, null, null, null, null);
            sheet1Token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), null);
            sheet2Token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), sheet(3));
            sheetErrToken = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(-1), null);
        });

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(cellToken.getType()).to.equal('ref');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(cellToken.getText(opConfig)).to.equal('B$3');
                expect(cellToken.getText(uiConfig)).to.equal('B$3');
                expect(rangeToken.getText(opConfig)).to.equal('B$3:$D5');
                expect(rangeToken.getText(uiConfig)).to.equal('B$3:$D5');
                expect(errToken.getText(opConfig)).to.equal('#REF!');
                expect(errToken.getText(uiConfig)).to.equal('#BEZUG!');
                expect(sheet1Token.getText(opConfig)).to.equal('Sheet2!B$3');
                expect(sheet1Token.getText(uiConfig)).to.equal('Sheet2!B$3');
                expect(sheet2Token.getText(opConfig)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(sheet2Token.getText(uiConfig)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(sheetErrToken.getText(opConfig)).to.equal('#REF!!B$3');
                expect(sheetErrToken.getText(uiConfig)).to.equal('#BEZUG!!B$3');
            });
            it('should handle simple/complex sheet names correctly', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Sheet1'), null);
                expect(token1.getText(opConfig)).to.equal('Sheet1!A1');
                expect(token1.getText(uiConfig)).to.equal('Sheet1!A1');
                var token2 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Sheet 2'), null);
                expect(token2.getText(opConfig)).to.equal('\'Sheet 2\'!A1');
                expect(token2.getText(uiConfig)).to.equal('\'Sheet 2\'!A1');
                var token3 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('1Sheet'), null);
                expect(token3.getText(opConfig)).to.equal('\'1Sheet\'!A1');
                expect(token3.getText(uiConfig)).to.equal('\'1Sheet\'!A1');
                var token4 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Sheet+1'), null);
                expect(token4.getText(opConfig)).to.equal('\'Sheet+1\'!A1');
                expect(token4.getText(uiConfig)).to.equal('\'Sheet+1\'!A1');
                var token5 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Sheet1'), sheet('Sheet2'));
                expect(token5.getText(opConfig)).to.equal('Sheet1:Sheet2!A1');
                expect(token5.getText(uiConfig)).to.equal('Sheet1:Sheet2!A1');
                var token6 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Sheet1'), sheet('Sheet 2'));
                expect(token6.getText(opConfig)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(token6.getText(uiConfig)).to.equal('\'Sheet1:Sheet 2\'!A1');
                var token7 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Sheet 1'), sheet('Sheet2'));
                expect(token7.getText(opConfig)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(token7.getText(uiConfig)).to.equal('\'Sheet 1:Sheet2\'!A1');
            });
            it('should treat booleans as complex sheet names', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('TRUE'));
                expect(token1.getText(opConfig)).to.equal('\'TRUE\'!A1');
                expect(token1.getText(uiConfig)).to.equal('TRUE!A1');
                var token2 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('WAHR'));
                expect(token2.getText(opConfig)).to.equal('WAHR!A1');
                expect(token2.getText(uiConfig)).to.equal('\'WAHR\'!A1');
            });
            it('should treat cell addresses as complex sheet names', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('A1'));
                expect(token1.getText(opConfig)).to.equal('\'A1\'!A1');
                expect(token1.getText(uiConfig)).to.equal('\'A1\'!A1');
                var token2 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('ZZZZ1'));
                expect(token2.getText(opConfig)).to.equal('ZZZZ1!A1');
                expect(token2.getText(uiConfig)).to.equal('ZZZZ1!A1');
                var token3 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('R1C1'));
                expect(token3.getText(opConfig)).to.equal('\'R1C1\'!A1');
                expect(token3.getText(uiConfig)).to.equal('\'R1C1\'!A1');
                var token4 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sheet('Z1S1'));
                expect(token4.getText(opConfig)).to.equal('Z1S1!A1');
                expect(token4.getText(uiConfig)).to.equal('\'Z1S1\'!A1');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(cellToken.toString()).to.equal('ref[B$3]');
                expect(rangeToken.toString()).to.equal('ref[B$3:$D5]');
                expect(errToken.toString()).to.equal('ref[#REF]');
                expect(sheet1Token.toString()).to.equal('ref[$1!B$3]');
                expect(sheet2Token.toString()).to.equal('ref[$1:$3!B$3]');
                expect(sheetErrToken.toString()).to.equal('ref[#REF!B$3]');
            });
        });

        describe('method "hasSheetRef"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('hasSheetRef');
            });
            it('should return the correct result', function () {
                expect(cellToken.hasSheetRef()).to.equal(false);
                expect(rangeToken.hasSheetRef()).to.equal(false);
                expect(errToken.hasSheetRef()).to.equal(false);
                expect(sheet1Token.hasSheetRef()).to.equal(true);
                expect(sheet2Token.hasSheetRef()).to.equal(true);
                expect(sheetErrToken.hasSheetRef()).to.equal(true);
            });
        });

        describe('method "hasSheetError"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('hasSheetError');
            });
            it('should return the correct result', function () {
                expect(cellToken.hasSheetError()).to.equal(false);
                expect(rangeToken.hasSheetError()).to.equal(false);
                expect(errToken.hasSheetError()).to.equal(false);
                expect(sheet1Token.hasSheetError()).to.equal(false);
                expect(sheet2Token.hasSheetError()).to.equal(false);
                expect(sheetErrToken.hasSheetError()).to.equal(true);
            });
        });

        describe('method "getRange3D"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('getRange3D');
            });
            it('should return the correct range address', function () {
                expect(cellToken.getRange3D({ refSheet: 0 })).to.stringifyTo('0:0!B3:B3');
                expect(cellToken.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:B3');
                expect(rangeToken.getRange3D({ refSheet: 0 })).to.stringifyTo('0:0!B3:D5');
                expect(rangeToken.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:D5');
                expect(errToken.getRange3D({ refSheet: 0 })).to.equal(null);
                expect(errToken.getRange3D({ refSheet: 1 })).to.equal(null);
                expect(sheet1Token.getRange3D({ refSheet: 0 })).to.stringifyTo('1:1!B3:B3');
                expect(sheet1Token.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:B3');
                expect(sheet2Token.getRange3D()).to.stringifyTo('1:3!B3:B3');
                expect(sheet2Token.getRange3D({ refSheet: 0 })).to.stringifyTo('1:3!B3:B3');
                expect(sheet2Token.getRange3D({ refSheet: 1 })).to.stringifyTo('1:3!B3:B3');
                expect(sheetErrToken.getRange3D({ refSheet: 0 })).to.equal(null);
                expect(sheetErrToken.getRange3D({ refSheet: 1 })).to.equal(null);
            });
            it('should return the correct range address without reference sheet', function () {
                expect(cellToken.getRange3D()).to.equal(null);
                expect(rangeToken.getRange3D()).to.equal(null);
                expect(errToken.getRange3D()).to.equal(null);
                expect(sheet1Token.getRange3D()).to.stringifyTo('1:1!B3:B3');
                expect(sheetErrToken.getRange3D()).to.equal(null);
            });
            it('should relocate cell range without wrapping', function () {
                expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.stringifyTo('0:0!F3:F3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2') })).to.stringifyTo('0:0!A3:A3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('C2') })).to.equal(null);
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9') })).to.stringifyTo('0:0!F3:F3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2') })).to.equal(null);
                expect(rangeToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.stringifyTo('0:0!D3:F10');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B5') })).to.stringifyTo('0:0!A1:D3');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('C5') })).to.equal(null);
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B6') })).to.equal(null);
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9') })).to.stringifyTo('0:0!D3:F12');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2') })).to.equal(null);
                expect(errToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.equal(null);
            });
            it('should relocate cell range with wrapping', function () {
                expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.stringifyTo('0:0!F3:F3');
                expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('XFD1'), wrapReferences: true })).to.stringifyTo('0:0!A3:A3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!A3:A3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('C2'), wrapReferences: true })).to.stringifyTo('0:0!XFD3:XFD3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9'), wrapReferences: true })).to.stringifyTo('0:0!F3:F3');
                expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!XFB3:XFB3');
                expect(rangeToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.stringifyTo('0:0!D3:F10');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B5'), wrapReferences: true })).to.stringifyTo('0:0!A1:D3');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('C5'), wrapReferences: true })).to.stringifyTo('0:0!D1:XFD3');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B6'), wrapReferences: true })).to.stringifyTo('0:0!A3:D1048576');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9'), wrapReferences: true })).to.stringifyTo('0:0!D3:F12');
                expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!D3:XFB1048574');
                expect(errToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.equal(null);
            });
        });

        describe('method "setRange"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('setRange');
            });
            it('should change range address', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, null, null);
                expect(token1).to.stringifyTo('ref[B$3]');
                expect(token1.setRange(r('B3:B3'))).to.equal(false);
                expect(token1).to.stringifyTo('ref[B$3]');
                expect(token1.setRange(r('D5:D5'))).to.equal(true);
                expect(token1).to.stringifyTo('ref[D$5]');
                expect(token1.setRange(r('B3:D5'))).to.equal(true);
                expect(token1).to.stringifyTo('ref[B$3:D$5]');
                var token2 = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
                expect(token2.setRange(r('B3:D5'))).to.equal(false);
                expect(token2).to.stringifyTo('ref[B$3:$D5]');
                expect(token2.setRange(r('C4:D5'))).to.equal(true);
                expect(token2).to.stringifyTo('ref[C$4:$D5]');
                expect(token2.setRange(r('C4:E6'))).to.equal(true);
                expect(token2).to.stringifyTo('ref[C$4:$E6]');
                expect(token2.setRange(r('C4:C4'))).to.equal(true);
                expect(token2).to.stringifyTo('ref[C$4:$C4]');
            });
            it('should convert reference error to valid reference', function () {
                var token = new Tokens.ReferenceToken(docModel, null, null, null, null);
                expect(token).to.stringifyTo('ref[#REF]');
                expect(token.setRange(r('B3:D5'))).to.equal(true);
                expect(token).to.stringifyTo('ref[$B$3:$D$5]');
            });
        });

        describe('method "transformRange"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('transformRange');
            });
            it('should transform single cell address', function () {
                var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, null, null);
                expect(token).to.stringifyTo('ref[B$3]');
                expect(token.transformRange(1, i(0, 0), true, true, 0)).to.equal(false);
                expect(token).to.stringifyTo('ref[B$3]');
                expect(token.transformRange(1, i(0, 0), true, true, null)).to.equal(false);
                expect(token).to.stringifyTo('ref[B$3]');
                expect(token.transformRange(1, i(0, 0), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$3]');
                expect(token.transformRange(1, i(2, 3), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$3]');
                expect(token.transformRange(1, i(5, 5), true, true, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[E$3]');
                expect(token.transformRange(1, i(0, 0), true, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$4]');
                expect(token.transformRange(1, i(3, 4), true, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$6]');
                expect(token.transformRange(1, i(6, 6), true, false, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[E$6]');
                expect(token.transformRange(1, i(5, 6), false, true, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[E$6]');
                expect(token.transformRange(1, i(1, 2), false, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$6]');
                expect(token.transformRange(1, i(6, 7), false, false, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[C$6]');
                expect(token.transformRange(1, i(2, 3), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$4]');
                expect(token.transformRange(1, i(3, 4), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[#REF]');
            });
            it('should transform range address', function () {
                var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
                expect(token).to.stringifyTo('ref[B$3:$D5]');
                expect(token.transformRange(1, i(0, 0), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$3:$E5]');
                expect(token.transformRange(1, i(2, 3), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$3:$G5]');
                expect(token.transformRange(1, i(5, 6), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$3:$I5]');
                expect(token.transformRange(1, i(8, 8), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$3:$J5]');
                expect(token.transformRange(1, i(10, 11), true, true, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[E$3:$J5]');
                expect(token.transformRange(1, i(0, 0), true, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$4:$J6]');
                expect(token.transformRange(1, i(3, 4), true, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$6:$J8]');
                expect(token.transformRange(1, i(6, 7), true, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$6:$J10]');
                expect(token.transformRange(1, i(9, 9), true, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$6:$J11]');
                expect(token.transformRange(1, i(11, 12), true, false, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[E$6:$J11]');
                expect(token.transformRange(1, i(10, 11), false, true, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[E$6:$J11]');
                expect(token.transformRange(1, i(9, 10), false, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$6:$I11]');
                expect(token.transformRange(1, i(5, 6), false, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[E$6:$G11]');
                expect(token.transformRange(1, i(3, 4), false, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[D$6:$E11]');
                expect(token.transformRange(1, i(2, 2), false, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$6:$D11]');
                expect(token.transformRange(1, i(11, 12), false, false, 1)).to.equal(false);
                expect(token).to.stringifyTo('ref[C$6:$D11]');
                expect(token.transformRange(1, i(10, 11), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$6:$D10]');
                expect(token.transformRange(1, i(6, 7), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$6:$D8]');
                expect(token.transformRange(1, i(4, 5), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$5:$D6]');
                expect(token.transformRange(1, i(3, 3), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[C$4:$D5]');
                expect(token.transformRange(1, i(2, 5), false, false, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[#REF]');
            });
            it('should transform references with sheet correctly', function () {
                var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), null);
                expect(token.transformRange(0, i(0, 0), true, true, 0)).to.equal(false);
                expect(token.transformRange(0, i(0, 0), true, true, 1)).to.equal(false);
                expect(token.transformRange(0, i(0, 0), true, true, null)).to.equal(false);
                expect(token).to.stringifyTo('ref[$1!B$3]');
                expect(token.transformRange(1, i(0, 0), true, true, 0)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!C$3]');
                expect(token.transformRange(1, i(0, 0), true, true, 1)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!D$3]');
                expect(token.transformRange(1, i(0, 0), true, true, null)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!E$3]');
            });
            it('should not transform references with sheet range', function () {
                var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), sheet(3));
                expect(token.transformRange(0, i(0, 0), true, true, 0)).to.equal(false);
                expect(token.transformRange(0, i(0, 0), true, true, 1)).to.equal(false);
                expect(token.transformRange(0, i(0, 0), true, true, null)).to.equal(false);
                expect(token.transformRange(1, i(0, 0), true, true, 0)).to.equal(false);
                expect(token.transformRange(1, i(0, 0), true, true, 1)).to.equal(false);
                expect(token.transformRange(1, i(0, 0), true, true, null)).to.equal(false);
                expect(token.transformRange(2, i(0, 0), true, true, 0)).to.equal(false);
                expect(token.transformRange(2, i(0, 0), true, true, 1)).to.equal(false);
                expect(token.transformRange(2, i(0, 0), true, true, null)).to.equal(false);
                expect(token).to.stringifyTo('ref[$1:$3!B$3]');
            });
            it('should not transform error references', function () {
                expect(errToken.transformRange(0, i(0, 0), true, true, 0)).to.equal(false);
                expect(sheetErrToken.transformRange(0, i(0, 0), true, true, 0)).to.equal(false);
            });
        });

        describe('method "relocateRange"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('relocateRange');
            });
            it('should relocate cell range without wrapping', function () {
                var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), sheet(1), null);
                expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                expect(token.relocateRange(a('B2'), a('B2'), false)).to.equal(false);
                expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                expect(token.relocateRange(a('A2'), a('B4'), false)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!C$3:$D7]');
                expect(token.relocateRange(a('B4'), a('A2'), false)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                expect(token.relocateRange(a('C1'), a('A1'), false)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!#REF]');
            });
            it('should relocate cell range with wrapping', function () {
                var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), sheet(1), null);
                expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                expect(token.relocateRange(a('B2'), a('B2'), true)).to.equal(false);
                expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                expect(token.relocateRange(a('A2'), a('B4'), true)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!C$3:$D7]');
                expect(token.relocateRange(a('B4'), a('A2'), true)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                expect(token.relocateRange(a('C1'), a('A1'), true)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!XFD$3:$D5]');
                expect(token.relocateRange(a('A6'), a('A1'), true)).to.equal(true);
                expect(token).to.stringifyTo('ref[$1!XFD$3:$D1048576]');
            });
        });

        describe('method "transformSheet"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('transformSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(cellToken.transformSheet(0, null)).to.equal(false);
                expect(cellToken).to.stringifyTo('ref[B$3]');
                expect(rangeToken.transformSheet(0, null)).to.equal(false);
                expect(rangeToken).to.stringifyTo('ref[B$3:$D5]');
            });
            it('should transform sheet index', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), null);
                expect(token1.transformSheet(1, null)).to.equal(true);
                expect(token1).to.stringifyTo('ref[$2!B$3]');
                expect(token1.transformSheet(3, null)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$2!B$3]');
                expect(token1.transformSheet(null, 1)).to.equal(true);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                expect(token1.transformSheet(null, 2)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                var token2 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), sheet(3));
                expect(token2.transformSheet(1, null)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                expect(token2.transformSheet(4, null)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$2:$5!B$3]');
                expect(token2.transformSheet(6, null)).to.equal(false);
                expect(token2).to.stringifyTo('ref[$2:$5!B$3]');
                expect(token2.transformSheet(null, 4)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                expect(token2.transformSheet(null, 1)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                expect(token2.transformSheet(null, 4)).to.equal(false);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(sheetErrToken.transformSheet(0, null)).to.equal(false);
                expect(sheetErrToken).to.stringifyTo('ref[#REF!B$3]');
            });
        });

        describe('method "relocateSheet"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('relocateSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(cellToken.relocateSheet(1, 2)).to.equal(false);
                expect(cellToken).to.stringifyTo('ref[B$3]');
                expect(rangeToken.relocateSheet(1, 2)).to.equal(false);
                expect(rangeToken).to.stringifyTo('ref[B$3:$D5]');
            });
            it('should relocate sheet index', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), null);
                expect(token1.relocateSheet(0, 1)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                expect(token1.relocateSheet(1, 2)).to.equal(true);
                expect(token1).to.stringifyTo('ref[$2!B$3]');
                expect(token1.relocateSheet(3, 2)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$2!B$3]');
                expect(token1.relocateSheet(2, 1)).to.equal(true);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                var token2 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), sheet(3));
                expect(token2.relocateSheet(0, 1)).to.equal(false);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                expect(token2.relocateSheet(1, 2)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$2:$3!B$3]');
                expect(token2.relocateSheet(3, 4)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                expect(token2.relocateSheet(3, 4)).to.equal(false);
                expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                expect(token2.relocateSheet(4, 0)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$2:$0!B$3]');
                expect(token2.getRange3D()).to.stringifyTo('0:2!B3:B3');
            });
            it('should do nothing for token with sheet error', function () {
                expect(sheetErrToken.relocateSheet(1, 2)).to.equal(false);
                expect(sheetErrToken).to.stringifyTo('ref[#REF!B$3]');
            });
        });

        describe('method "renameSheet"', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.respondTo('renameSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(cellToken.renameSheet(1)).to.equal(false);
                expect(cellToken).to.stringifyTo('ref[B$3]');
                expect(rangeToken.renameSheet(1)).to.equal(false);
                expect(rangeToken).to.stringifyTo('ref[B$3:$D5]');
            });
            it('should not modify the token but return true if affected', function () {
                var token1 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), null);
                expect(token1.renameSheet(0)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                expect(token1.renameSheet(1)).to.equal(true);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                expect(token1.renameSheet(2)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                expect(token1.renameSheet(3)).to.equal(false);
                expect(token1).to.stringifyTo('ref[$1!B$3]');
                var token2 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sheet(1), sheet(3));
                expect(token2.renameSheet(0)).to.equal(false);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                expect(token2.renameSheet(1)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                expect(token2.renameSheet(2)).to.equal(false);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                expect(token2.renameSheet(3)).to.equal(true);
                expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(sheetErrToken.renameSheet(1)).to.equal(false);
                expect(sheetErrToken).to.stringifyTo('ref[#REF!B$3]');
            });
        });
    });

    // class NameToken ========================================================

    describe('Spreadsheet class NameToken', function () {
        it('should exist', function () {
            expect(Tokens.NameToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var nameToken, sheetToken, errorToken;
        before(function () {
            nameToken = new Tokens.NameToken(docModel, 'name', null);
            sheetToken = new Tokens.NameToken(docModel, 'Name', sheet(1));
            errorToken = new Tokens.NameToken(docModel, 'NAME', sheet(-1));
        });

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(nameToken.getType()).to.equal('name');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(nameToken.getText(opConfig)).to.equal('name');
                expect(nameToken.getText(uiConfig)).to.equal('name');
                expect(sheetToken.getText(opConfig)).to.equal('Sheet2!Name');
                expect(sheetToken.getText(uiConfig)).to.equal('Sheet2!Name');
                expect(errorToken.getText(opConfig)).to.equal('#REF!!NAME');
                expect(errorToken.getText(uiConfig)).to.equal('#BEZUG!!NAME');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(nameToken.toString()).to.equal('name[name]');
                expect(sheetToken.toString()).to.equal('name[$1!Name]');
                expect(errorToken.toString()).to.equal('name[#REF!NAME]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('getValue');
            });
            it('should return the value of the token', function () {
                expect(nameToken.getValue()).to.equal('name');
                expect(sheetToken.getValue()).to.equal('Name');
                expect(errorToken.getValue()).to.equal('NAME');
            });
        });

        describe('method "hasSheetRef"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('hasSheetRef');
            });
            it('should return the correct result', function () {
                expect(nameToken.hasSheetRef()).to.equal(false);
                expect(sheetToken.hasSheetRef()).to.equal(true);
                expect(errorToken.hasSheetRef()).to.equal(true);
            });
        });

        describe('method "hasSheetError"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('hasSheetError');
            });
            it('should return the correct result', function () {
                expect(nameToken.hasSheetError()).to.equal(false);
                expect(sheetToken.hasSheetError()).to.equal(false);
                expect(errorToken.hasSheetError()).to.equal(true);
            });
        });

        describe('method "resolveNameModel"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('resolveNameModel');
            });
            it('should return globally defined name', function () {
                var globalName = docModel.getNameCollection().getNameModel('name');
                expect(nameToken.resolveNameModel(null)).to.be.an('object').and.to.equal(globalName);
                expect(nameToken.resolveNameModel(0)).to.be.an('object').and.to.equal(globalName);
            });
            it('should return sheet-local defined name', function () {
                var localName = docModel.getSheetModel(1).getNameCollection().getNameModel('name');
                expect(nameToken.resolveNameModel(1)).to.be.an('object').and.to.equal(localName);
                expect(sheetToken.resolveNameModel(null)).to.be.an('object').and.to.equal(localName);
                expect(sheetToken.resolveNameModel(0)).to.be.an('object').and.to.equal(localName);
                expect(sheetToken.resolveNameModel(1)).to.be.an('object').and.to.equal(localName);
            });
            it('should return null for unknown names', function () {
                expect(new Tokens.NameToken(docModel, 'invalid', null).resolveNameModel(0)).to.equal(null);
                expect(new Tokens.NameToken(docModel, 'invalid', sheet(1)).resolveNameModel(0)).to.equal(null);
                expect(new Tokens.NameToken(docModel, 'name', sheet(0)).resolveNameModel(null)).to.equal(null);
                expect(new Tokens.NameToken(docModel, 'name', sheet(0)).resolveNameModel(0)).to.equal(null);
                expect(new Tokens.NameToken(docModel, 'name', sheet(0)).resolveNameModel(1)).to.equal(null);
            });
            it('should return null for token with sheet error', function () {
                expect(errorToken.resolveNameModel(null)).to.equal(null);
                expect(errorToken.resolveNameModel(0)).to.equal(null);
            });
        });

        describe('method "transformSheet"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('transformSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(nameToken.transformSheet(0, null)).to.equal(false);
                expect(nameToken).to.stringifyTo('name[name]');
            });
            it('should transform sheet index', function () {
                var token = new Tokens.NameToken(docModel, 'name', sheet(1));
                expect(token).to.stringifyTo('name[$1!name]');
                expect(token.transformSheet(1, null)).to.equal(true);
                expect(token).to.stringifyTo('name[$2!name]');
                expect(token.transformSheet(3, null)).to.equal(false);
                expect(token).to.stringifyTo('name[$2!name]');
                expect(token.transformSheet(null, 1)).to.equal(true);
                expect(token).to.stringifyTo('name[$1!name]');
                expect(token.transformSheet(null, 2)).to.equal(false);
                expect(token).to.stringifyTo('name[$1!name]');
                expect(token.transformSheet(null, 1)).to.equal(true);
                expect(token).to.stringifyTo('name[#REF!name]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(errorToken.transformSheet(0, null)).to.equal(false);
                expect(errorToken).to.stringifyTo('name[#REF!NAME]');
            });
        });

        describe('method "relocateSheet"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('relocateSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(nameToken.relocateSheet(1, 2)).to.equal(false);
                expect(nameToken).to.stringifyTo('name[name]');
            });
            it('should relocate sheet index', function () {
                var token = new Tokens.NameToken(docModel, 'name', sheet(1));
                expect(token.relocateSheet(0, 1)).to.equal(false);
                expect(token).to.stringifyTo('name[$1!name]');
                expect(token.relocateSheet(1, 2)).to.equal(true);
                expect(token).to.stringifyTo('name[$2!name]');
                expect(token.relocateSheet(3, 2)).to.equal(false);
                expect(token).to.stringifyTo('name[$2!name]');
                expect(token.relocateSheet(2, 0)).to.equal(true);
                expect(token).to.stringifyTo('name[$0!name]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(errorToken.relocateSheet(1, 2)).to.equal(false);
                expect(errorToken).to.stringifyTo('name[#REF!NAME]');
            });
        });

        describe('method "renameSheet"', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.respondTo('renameSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(nameToken.renameSheet(1)).to.equal(false);
                expect(nameToken).to.stringifyTo('name[name]');
            });
            it('should not modify the token but return true if affected', function () {
                var token = new Tokens.NameToken(docModel, 'name', sheet(1));
                expect(token.renameSheet(0)).to.equal(false);
                expect(token).to.stringifyTo('name[$1!name]');
                expect(token.renameSheet(1)).to.equal(true);
                expect(token).to.stringifyTo('name[$1!name]');
                expect(token.renameSheet(2)).to.equal(false);
                expect(token).to.stringifyTo('name[$1!name]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(errorToken.renameSheet(1)).to.equal(false);
                expect(errorToken).to.stringifyTo('name[#REF!NAME]');
            });
        });
    });

    // class MacroToken =======================================================

    describe('Spreadsheet class MacroToken', function () {
        it('should exist', function () {
            expect(Tokens.MacroToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var macroToken, sheetToken, errorToken;
        before(function () {
            macroToken = new Tokens.MacroToken(docModel, 'Sum', null);
            sheetToken = new Tokens.MacroToken(docModel, 'Macro', sheet(3));
            errorToken = new Tokens.MacroToken(docModel, 'Macro', sheet(-1));
        });

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(macroToken.getType()).to.equal('macro');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(macroToken.getText(opConfig)).to.equal('Sum');
                expect(macroToken.getText(uiConfig)).to.equal('Sum');
                expect(sheetToken.getText(opConfig)).to.equal('\'Sheet 4\'!Macro');
                expect(sheetToken.getText(uiConfig)).to.equal('\'Sheet 4\'!Macro');
                expect(errorToken.getText(opConfig)).to.equal('#REF!!Macro');
                expect(errorToken.getText(uiConfig)).to.equal('#BEZUG!!Macro');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(macroToken.toString()).to.equal('macro[Sum]');
                expect(sheetToken.toString()).to.equal('macro[$3!Macro]');
                expect(errorToken.toString()).to.equal('macro[#REF!Macro]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('getValue');
            });
            it('should return the value of the token', function () {
                expect(macroToken.getValue()).to.equal('Sum');
                expect(sheetToken.getValue()).to.equal('Macro');
                expect(errorToken.getValue()).to.equal('Macro');
            });
        });

        describe('method "hasSheetRef"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('hasSheetRef');
            });
            it('should return the correct result', function () {
                expect(macroToken.hasSheetRef()).to.equal(false);
                expect(sheetToken.hasSheetRef()).to.equal(true);
                expect(errorToken.hasSheetRef()).to.equal(true);
            });
        });

        describe('method "hasSheetError"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('hasSheetError');
            });
            it('should return the correct result', function () {
                expect(macroToken.hasSheetError()).to.equal(false);
                expect(sheetToken.hasSheetError()).to.equal(false);
                expect(errorToken.hasSheetError()).to.equal(true);
            });
        });

        describe('method "transformSheet"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('transformSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(macroToken.transformSheet(0, null)).to.equal(false);
                expect(macroToken).to.stringifyTo('macro[Sum]');
            });
            it('should transform sheet index', function () {
                var token = new Tokens.MacroToken(docModel, 'Sum', sheet(1));
                expect(token).to.stringifyTo('macro[$1!Sum]');
                expect(token.transformSheet(1, null)).to.equal(true);
                expect(token).to.stringifyTo('macro[$2!Sum]');
                expect(token.transformSheet(3, null)).to.equal(false);
                expect(token).to.stringifyTo('macro[$2!Sum]');
                expect(token.transformSheet(null, 1)).to.equal(true);
                expect(token).to.stringifyTo('macro[$1!Sum]');
                expect(token.transformSheet(null, 2)).to.equal(false);
                expect(token).to.stringifyTo('macro[$1!Sum]');
                expect(token.transformSheet(null, 1)).to.equal(true);
                expect(token).to.stringifyTo('macro[#REF!Sum]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(errorToken.transformSheet(0, null)).to.equal(false);
                expect(errorToken).to.stringifyTo('macro[#REF!Macro]');
            });
        });

        describe('method "relocateSheet"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('relocateSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(macroToken.relocateSheet(1, 2)).to.equal(false);
                expect(macroToken).to.stringifyTo('macro[Sum]');
            });
            it('should relocate sheet index', function () {
                var token = new Tokens.MacroToken(docModel, 'Sum', sheet(1));
                expect(token.relocateSheet(0, 1)).to.equal(false);
                expect(token).to.stringifyTo('macro[$1!Sum]');
                expect(token.relocateSheet(1, 2)).to.equal(true);
                expect(token).to.stringifyTo('macro[$2!Sum]');
                expect(token.relocateSheet(3, 2)).to.equal(false);
                expect(token).to.stringifyTo('macro[$2!Sum]');
                expect(token.relocateSheet(2, 0)).to.equal(true);
                expect(token).to.stringifyTo('macro[$0!Sum]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(errorToken.relocateSheet(1, 2)).to.equal(false);
                expect(errorToken).to.stringifyTo('macro[#REF!Macro]');
            });
        });

        describe('method "renameSheet"', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.respondTo('renameSheet');
            });
            it('should do nothing for tokens without sheet', function () {
                expect(macroToken.renameSheet(1)).to.equal(false);
                expect(macroToken).to.stringifyTo('macro[Sum]');
            });
            it('should not modify the token but return true if affected', function () {
                var token = new Tokens.MacroToken(docModel, 'Macro', sheet(1));
                expect(token.renameSheet(0)).to.equal(false);
                expect(token).to.stringifyTo('macro[$1!Macro]');
                expect(token.renameSheet(1)).to.equal(true);
                expect(token).to.stringifyTo('macro[$1!Macro]');
                expect(token.renameSheet(2)).to.equal(false);
                expect(token).to.stringifyTo('macro[$1!Macro]');
            });
            it('should do nothing for token with sheet error', function () {
                expect(errorToken.renameSheet(1)).to.equal(false);
                expect(errorToken).to.stringifyTo('macro[#REF!Macro]');
            });
        });
    });

    // ========================================================================
});
