/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/formula/tokens',
    'io.ox/office/spreadsheet/model/formula/tokenizer'
], function (Tokens, Tokenizer) {

    'use strict';

    // class Tokenizer ========================================================

    describe('Spreadsheet class Tokenizer', function () {

        it('should exist', function () {
            expect(Tokenizer).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the models of an OOXML and ODF document for testing
        var ooxDocModel = null;
        var odfDocModel = null;

        // the formula tokenizer for OOXML and ODF files
        var ooxTokenizer = null;
        var odfTokenizer = null;

        // the operations to be applied by the OOXML document model
        var OOX_OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
            { name: 'insertName', exprName: 'name', formula: 'Sheet1!$A$1' }, // global name
            { name: 'insertName', sheet: 1, exprName: 'name', formula: 'Sheet1!$A$1' } // sheet-local name
        ];

        // the operations to be applied by the ODF document model
        var ODF_OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 1024, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
            { name: 'insertName', exprName: 'name', formula: 'Sheet1!$A$1' } // global name
        ];

        // initialize formula resource data, and get grammar configurations from document
        before(function (done) {
            ox.test.spreadsheet.createApp('ooxml', OOX_OPERATIONS).done(function (ooxApp) {
                ooxDocModel = ooxApp.getModel();
                ooxDocModel.onInitFormulaResource(function () {
                    ox.test.spreadsheet.createApp('odf', ODF_OPERATIONS).done(function (odfApp) {
                        odfDocModel = odfApp.getModel();
                        odfDocModel.onInitFormulaResource(function () { done(); });
                    });
                });
            });
        });

        function expectResult(result, expected) {
            expect(result).to.be.an('array');
            expect(_.pluck(result, 'token').join(' ')).to.equal(expected);
        }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a formula tokenizer', function () {
                ooxTokenizer = new Tokenizer(ooxDocModel);
                expect(ooxTokenizer).to.be.an('object');
                odfTokenizer = new Tokenizer(odfDocModel);
                expect(odfTokenizer).to.be.an('object');
            });
        });

        // public methods -----------------------------------------------------

        describe('method "parseFormula"', function () {
            it('should exist', function () {
                expect(ooxTokenizer).to.respondTo('parseFormula');
            });
            it('should return an array of token descriptors', function () {
                var result = ooxTokenizer.parseFormula('op', '42+012.50 *-.01');
                expect(result).to.be.an('array').and.to.have.length(6);
                expect(result[0]).to.have.a.property('token').that.is.an['instanceof'](Tokens.LiteralToken);
                expect(result[0].token.getValue()).to.equal(42);
                expect(result[0]).to.have.a.property('index', 0);
                expect(result[0]).to.have.a.property('text', '42');
                expect(result[0]).to.have.a.property('start', 0);
                expect(result[0]).to.have.a.property('end', 2);
                expect(result[1]).to.have.a.property('token').that.is.an['instanceof'](Tokens.OperatorToken);
                expect(result[1].token.getValue()).to.equal('add');
                expect(result[1]).to.have.a.property('index', 1);
                expect(result[1]).to.have.a.property('text', '+');
                expect(result[1]).to.have.a.property('start', 2);
                expect(result[1]).to.have.a.property('end', 3);
                expect(result[2]).to.have.a.property('token').that.is.an['instanceof'](Tokens.LiteralToken);
                expect(result[2].token.getValue()).to.equal(12.5);
                expect(result[2]).to.have.a.property('index', 2);
                expect(result[2]).to.have.a.property('text', '012.50');
                expect(result[2]).to.have.a.property('start', 3);
                expect(result[2]).to.have.a.property('end', 9);
                expect(result[3]).to.have.a.property('token').that.is.an['instanceof'](Tokens.FixedToken);
                expect(result[3].token.getType()).to.equal('ws');
                expect(result[3].token.getValue()).to.equal(' ');
                expect(result[3]).to.have.a.property('index', 3);
                expect(result[3]).to.have.a.property('text', ' ');
                expect(result[3]).to.have.a.property('start', 9);
                expect(result[3]).to.have.a.property('end', 10);
                expect(result[4]).to.have.a.property('token').that.is.an['instanceof'](Tokens.OperatorToken);
                expect(result[4].token.getValue()).to.equal('mul');
                expect(result[4]).to.have.a.property('index', 4);
                expect(result[4]).to.have.a.property('text', '*');
                expect(result[4]).to.have.a.property('start', 10);
                expect(result[4]).to.have.a.property('end', 11);
                expect(result[5]).to.have.a.property('token').that.is.an['instanceof'](Tokens.LiteralToken);
                expect(result[5].token.getValue()).to.equal(-0.01);
                expect(result[5]).to.have.a.property('index', 5);
                expect(result[5]).to.have.a.property('text', '-.01');
                expect(result[5]).to.have.a.property('start', 11);
                expect(result[5]).to.have.a.property('end', 15);
                expectResult(result, 'lit[42] op[add] lit[12.5] ws[ ] op[mul] lit[-0.01]');
            });

            it('should parse number literals and numeric operators', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(ooxTokenizer.parseFormula('op', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(ooxTokenizer.parseFormula('op', '.010*010.'), 'lit[0.01] op[mul] lit[10]');
                expectResult(ooxTokenizer.parseFormula('op', '077.880/0.0'), 'lit[77.88] op[div] lit[0]');
                expectResult(ooxTokenizer.parseFormula('op', '1e30^001.100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(ooxTokenizer.parseFormula('op', '.01e+30+10.E-30-10.01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(ooxTokenizer.parseFormula('ui', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(ooxTokenizer.parseFormula('ui', ',010*010,'), 'lit[0.01] op[mul] lit[10]');
                expectResult(ooxTokenizer.parseFormula('ui', '077,880/0,0'), 'lit[77.88] op[div] lit[0]');
                expectResult(ooxTokenizer.parseFormula('ui', '1e30^001,100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(ooxTokenizer.parseFormula('ui', ',01e+30+10,E-30-10,01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(odfTokenizer.parseFormula('op', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(odfTokenizer.parseFormula('op', '.010*010.'), 'lit[0.01] op[mul] lit[10]');
                expectResult(odfTokenizer.parseFormula('op', '077.880/0.0'), 'lit[77.88] op[div] lit[0]');
                expectResult(odfTokenizer.parseFormula('op', '1e30^001.100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(odfTokenizer.parseFormula('op', '.01e+30+10.E-30-10.01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(odfTokenizer.parseFormula('ui', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(odfTokenizer.parseFormula('ui', ',010*010,'), 'lit[0.01] op[mul] lit[10]');
                expectResult(odfTokenizer.parseFormula('ui', '077,880/0,0'), 'lit[77.88] op[div] lit[0]');
                expectResult(odfTokenizer.parseFormula('ui', '1e30^001,100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(odfTokenizer.parseFormula('ui', ',01e+30+10,E-30-10,01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
            });

            it('should parse boolean literals and comparison operators', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'true<false>True<>False'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxTokenizer.parseFormula('op', 'tRUE<=fALSE>=TRUE=FALSE'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'wahr<falsch>Wahr<>Falsch'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxTokenizer.parseFormula('ui', 'wAHR<=fALSCH>=WAHR=FALSCH'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
                // native ODF
                expectResult(ooxTokenizer.parseFormula('op', 'true<false>True<>False'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxTokenizer.parseFormula('op', 'tRUE<=fALSE>=TRUE=FALSE'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
                // localized ODF
                expectResult(ooxTokenizer.parseFormula('ui', 'wahr<falsch>Wahr<>Falsch'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxTokenizer.parseFormula('ui', 'wAHR<=fALSCH>=WAHR=FALSCH'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
            });

            it('should parse string literals and string operators', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
            });

            it('should parse error code literals', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '#NULL!#DIV/0!#ref!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(ooxTokenizer.parseFormula('op', '#VALUE!#NUM!#num!#N/A'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '#NULL!#DIV/0!#bezug!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(ooxTokenizer.parseFormula('ui', '#WERT!#ZAHL!#zahl!#NV'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '#NULL!#DIV/0!#ref!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(odfTokenizer.parseFormula('op', '#VALUE!#NUM!#num!#N/A'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '#NULL!#DIV/0!#bezug!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(odfTokenizer.parseFormula('ui', '#WERT!#ZAHL!#zahl!#NV'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
            });

            it('should parse matrix literals', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '{1,0042;+1,-0042}'), 'mat_open lit[1] mat_col lit[42] mat_row lit[1] mat_col lit[-42] mat_close');
                expectResult(ooxTokenizer.parseFormula('op', '{.010,-010.;077.880,+0.0}'), 'mat_open lit[0.01] mat_col lit[-10] mat_row lit[77.88] mat_col lit[0] mat_close');
                expectResult(ooxTokenizer.parseFormula('op', '{1e30,+001.100E-0030;-0e0,-.01e+30}'), 'mat_open lit[1e+30] mat_col lit[1.1e-30] mat_row lit[0] mat_col lit[-1e+28] mat_close');
                expectResult(ooxTokenizer.parseFormula('op', '{true,fAlSe;"abc","D""E";","";""|",""}'), 'mat_open lit[TRUE] mat_col lit[FALSE] mat_row lit["abc"] mat_col lit["D""E"] mat_row lit[","";""|"] mat_col lit[""] mat_close');
                expectResult(ooxTokenizer.parseFormula('op', '{#VALUE!,#num!}'), 'mat_open lit[#VALUE] mat_col lit[#NUM] mat_close');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '{1;0042|+1;-0042}'), 'mat_open lit[1] mat_col lit[42] mat_row lit[1] mat_col lit[-42] mat_close');
                expectResult(ooxTokenizer.parseFormula('ui', '{,010;-010,|077,880;+0,0}'), 'mat_open lit[0.01] mat_col lit[-10] mat_row lit[77.88] mat_col lit[0] mat_close');
                expectResult(ooxTokenizer.parseFormula('ui', '{1e30;+001,100E-0030|-0e0;-,01e+30}'), 'mat_open lit[1e+30] mat_col lit[1.1e-30] mat_row lit[0] mat_col lit[-1e+28] mat_close');
                expectResult(ooxTokenizer.parseFormula('ui', '{wahr;fAlScH|"abc";"D""E"|","";""|";""}'), 'mat_open lit[TRUE] mat_col lit[FALSE] mat_row lit["abc"] mat_col lit["D""E"] mat_row lit[","";""|"] mat_col lit[""] mat_close');
                expectResult(ooxTokenizer.parseFormula('ui', '{#WERT!;#zahl!}'), 'mat_open lit[#VALUE] mat_col lit[#NUM] mat_close');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '{1;0042|+1;-0042}'), 'mat_open lit[1] mat_col lit[42] mat_row lit[1] mat_col lit[-42] mat_close');
                expectResult(odfTokenizer.parseFormula('op', '{.010;-010.|077.880;+0.0}'), 'mat_open lit[0.01] mat_col lit[-10] mat_row lit[77.88] mat_col lit[0] mat_close');
                expectResult(odfTokenizer.parseFormula('op', '{1e30;+001.100E-0030|-0e0;-.01e+30}'), 'mat_open lit[1e+30] mat_col lit[1.1e-30] mat_row lit[0] mat_col lit[-1e+28] mat_close');
                expectResult(odfTokenizer.parseFormula('op', '{true;fAlSe|"abc";"D""E"|","";""|";""}'), 'mat_open lit[TRUE] mat_col lit[FALSE] mat_row lit["abc"] mat_col lit["D""E"] mat_row lit[","";""|"] mat_col lit[""] mat_close');
                expectResult(odfTokenizer.parseFormula('op', '{#VALUE!;#num!}'), 'mat_open lit[#VALUE] mat_col lit[#NUM] mat_close');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '{1;0042|+1;-0042}'), 'mat_open lit[1] mat_col lit[42] mat_row lit[1] mat_col lit[-42] mat_close');
                expectResult(odfTokenizer.parseFormula('ui', '{,010;-010,|077,880;+0,0}'), 'mat_open lit[0.01] mat_col lit[-10] mat_row lit[77.88] mat_col lit[0] mat_close');
                expectResult(odfTokenizer.parseFormula('ui', '{1e30;+001,100E-0030|-0e0;-,01e+30}'), 'mat_open lit[1e+30] mat_col lit[1.1e-30] mat_row lit[0] mat_col lit[-1e+28] mat_close');
                expectResult(odfTokenizer.parseFormula('ui', '{wahr;fAlScH|"abc";"D""E"|","";""|";""}'), 'mat_open lit[TRUE] mat_col lit[FALSE] mat_row lit["abc"] mat_col lit["D""E"] mat_row lit[","";""|"] mat_col lit[""] mat_close');
                expectResult(odfTokenizer.parseFormula('ui', '{#WERT!;#zahl!}'), 'mat_open lit[#VALUE] mat_col lit[#NUM] mat_close');
            });

            it('should parse local cell references and separators', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'A1,xfd001048576'), 'ref[A1] sep ref[XFD1048576]');
                expectResult(ooxTokenizer.parseFormula('op', 'B3,B$3 $B3\n  $B$3'), 'ref[B3] sep ref[B$3] op[isect] ref[$B3] ws[\n] op[isect] ws[ ] ref[$B$3]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'A1;xfd001048576'), 'ref[A1] sep ref[XFD1048576]');
                expectResult(ooxTokenizer.parseFormula('ui', 'B3;B$3;$B3 $B$3'), 'ref[B3] sep ref[B$3] sep ref[$B3] op[isect] ref[$B$3]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '[.A1],[.amj001048576]'), 'ref[A1] sep ref[AMJ1048576]');
                expectResult(odfTokenizer.parseFormula('op', '[.B3],[.B$3]~[.$B3]![.$B$3]'), 'ref[B3] sep ref[B$3] op[list] ref[$B3] op[isect] ref[$B$3]');
                expectResult(odfTokenizer.parseFormula('op', '[.B#REF!],[.#REF!$3],[.$B3:.$#REF!$3]'), 'ref[#REF] sep ref[#REF] sep ref[#REF]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'A1;amj001048576'), 'ref[A1] sep ref[AMJ1048576]');
                expectResult(odfTokenizer.parseFormula('ui', 'B3;B$3;$B3 $B$3'), 'ref[B3] sep ref[B$3] sep ref[$B3] op[isect] ref[$B$3]');
            });

            it('should parse local range references and range operators', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'B3:D7:B7:D3:D3:B7:D7:B3'), 'ref[B3:D7] op[range] ref[B7:D3] op[range] ref[D3:B7] op[range] ref[D7:B3]');
                expectResult(ooxTokenizer.parseFormula('op', 'B$3:$D7:d7:$b$3:$B$3:$D$7'), 'ref[B$3:$D7] op[range] ref[D7:$B$3] op[range] ref[$B$3:$D$7]');
                expectResult(ooxTokenizer.parseFormula('op', 'B3:B3:$B3:B3'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(ooxTokenizer.parseFormula('op', 'B3:B3:B3,B3'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'B3:D7:B7:D3:D3:B7:D7:B3'), 'ref[B3:D7] op[range] ref[B7:D3] op[range] ref[D3:B7] op[range] ref[D7:B3]');
                expectResult(ooxTokenizer.parseFormula('ui', 'B$3:$D7:d7:$b$3:$B$3:$D$7'), 'ref[B$3:$D7] op[range] ref[D7:$B$3] op[range] ref[$B$3:$D$7]');
                expectResult(ooxTokenizer.parseFormula('ui', 'B3:B3:$B3:B3'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(ooxTokenizer.parseFormula('ui', 'B3:B3:B3;B3'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '[.B3:.D7]:[.B7:.D3]:[.D3:.B7]:[.D7:.B3]'), 'ref[B3:D7] op[range] ref[B7:D3] op[range] ref[D3:B7] op[range] ref[D7:B3]');
                expectResult(odfTokenizer.parseFormula('op', '[.B$3:.$D7]:[.d7:.$b$3]:[.$B$3:.$D$7]'), 'ref[B$3:$D7] op[range] ref[D7:$B$3] op[range] ref[$B$3:$D$7]');
                expectResult(odfTokenizer.parseFormula('op', '[.B3:.B3]:[.$B3:.B3]'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(odfTokenizer.parseFormula('op', '[.B3:.B3]:[.B3],[.B3]'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'B3:D7:B7:D3:D3:B7:D7:B3'), 'ref[B3:D7] op[range] ref[B7:D3] op[range] ref[D3:B7] op[range] ref[D7:B3]');
                expectResult(odfTokenizer.parseFormula('ui', 'B$3:$D7:d7:$b$3:$B$3:$D$7'), 'ref[B$3:$D7] op[range] ref[D7:$B$3] op[range] ref[$B$3:$D$7]');
                expectResult(odfTokenizer.parseFormula('ui', 'B3:B3:$B3:B3'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(odfTokenizer.parseFormula('ui', 'B3:B3:B3;B3'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
            });

            it('should parse local column/row references', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'A:xfd:A:$B:$A:B:$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[A$1:$B$1048576] op[range] ref[$A$1:B$1048576] op[range] ref[$B$1:$A$1048576]');
                expectResult(ooxTokenizer.parseFormula('op', '1:001048576:1:$2:$1:2:$002:$001'), 'ref[$A1:$XFD1048576] op[range] ref[$A1:$XFD$2] op[range] ref[$A$1:$XFD2] op[range] ref[$A$2:$XFD$1]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'A:xfd:A:$B:$A:B:$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[A$1:$B$1048576] op[range] ref[$A$1:B$1048576] op[range] ref[$B$1:$A$1048576]');
                expectResult(ooxTokenizer.parseFormula('ui', '1:001048576:1:$2:$1:2:$002:$001'), 'ref[$A1:$XFD1048576] op[range] ref[$A1:$XFD$2] op[range] ref[$A$1:$XFD2] op[range] ref[$A$2:$XFD$1]');
                // native ODF
                // ... not available
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'A:amj:A:$B:$A:B:$b:$a'), 'ref[A$1:AMJ$1048576] op[range] ref[A$1:$B$1048576] op[range] ref[$A$1:B$1048576] op[range] ref[$B$1:$A$1048576]');
                expectResult(odfTokenizer.parseFormula('ui', '1:001048576:1:$2:$1:2:$002:$001'), 'ref[$A1:$AMJ1048576] op[range] ref[$A1:$AMJ$2] op[range] ref[$A$1:$AMJ2] op[range] ref[$A$2:$AMJ$1]');
            });

            it('should parse cell and range references with sheets', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'A1:Sheet1!A1:\'Sheet 4\'!xfd001048576'), 'ref[A1] op[range] ref[$0!A1] op[range] ref[$3!XFD1048576]');
                expectResult(ooxTokenizer.parseFormula('op', 'A1:Sheet1:Sheet3!A$1:$C2:\'Sheet 4:Sheet1\'!$a$1:$c$2'), 'ref[A1] op[range] ref[$0:$2!A$1:$C2] op[range] ref[$3:$0!$A$1:$C$2]');
                expectResult(ooxTokenizer.parseFormula('op', 'A1:A1!A1:Sheet1:A1!A1:A1:Sheet1:A1!A1'), 'ref[A1] op[range] ref[$\'A1\'!A1] op[range] ref[$0:$\'A1\'!A1:A1] op[range] ref[$0:$\'A1\'!A1]');
                expectResult(ooxTokenizer.parseFormula('op', 'TRUE!A1:WAHR!A1'), 'ref[$\'TRUE\'!A1] op[range] ref[$\'WAHR\'!A1]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'A1:Sheet1!A1:\'Sheet 4\'!xfd001048576'), 'ref[A1] op[range] ref[$0!A1] op[range] ref[$3!XFD1048576]');
                expectResult(ooxTokenizer.parseFormula('ui', 'A1:Sheet1:Sheet3!A$1:$C2:\'Sheet 4:Sheet1\'!$a$1:$c$2'), 'ref[A1] op[range] ref[$0:$2!A$1:$C2] op[range] ref[$3:$0!$A$1:$C$2]');
                expectResult(ooxTokenizer.parseFormula('ui', 'A1:A1!A1:Sheet1:A1!A1:A1:Sheet1:A1!A1'), 'ref[A1] op[range] ref[$\'A1\'!A1] op[range] ref[$0:$\'A1\'!A1:A1] op[range] ref[$0:$\'A1\'!A1]');
                expectResult(ooxTokenizer.parseFormula('ui', 'TRUE!A1:WAHR!A1'), 'ref[$\'TRUE\'!A1] op[range] ref[$\'WAHR\'!A1]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '[.A1]:[Sheet1.A1]:[$\'Sheet 4\'.amj001048576]'), 'ref[A1] op[range] ref[0!A1] op[range] ref[$3!AMJ1048576]');
                expectResult(odfTokenizer.parseFormula('op', '[.A1]:[$Sheet1.A$1:Sheet3.$C2]:[\'Sheet 4\'.$a$1:$Sheet1.$c$2]'), 'ref[A1] op[range] ref[$0:2!A$1:$C2] op[range] ref[3:$0!$A$1:$C$2]');
                expectResult(odfTokenizer.parseFormula('op', '[.A1]:[A1.A1]:[Sheet1.A1:A1.A1]:[Sheet1.A1:A1.A1]'), 'ref[A1] op[range] ref[\'A1\'!A1] op[range] ref[0:\'A1\'!A1:A1] op[range] ref[0:\'A1\'!A1:A1]');
                expectResult(odfTokenizer.parseFormula('op', '[TRUE.A1]:[WAHR.A1]'), 'ref[\'TRUE\'!A1] op[range] ref[\'WAHR\'!A1]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'A1:Sheet1!A1:$\'Sheet 4\'!amj001048576'), 'ref[A1] op[range] ref[0!A1] op[range] ref[$3!AMJ1048576]');
                expectResult(odfTokenizer.parseFormula('ui', 'A1:$Sheet1:Sheet3!A$1:$C2:\'Sheet 4\':$Sheet1!$a$1:$c$2'), 'ref[A1] op[range] ref[$0:2!A$1:$C2] op[range] ref[3:$0!$A$1:$C$2]');
                expectResult(odfTokenizer.parseFormula('ui', 'A1:A1!A1:Sheet1:A1!A1:A1:Sheet1:A1!A1'), 'ref[A1] op[range] ref[\'A1\'!A1] op[range] ref[0:\'A1\'!A1:A1] op[range] ref[0:\'A1\'!A1]');
                expectResult(odfTokenizer.parseFormula('ui', 'TRUE!A1:WAHR!A1'), 'ref[\'TRUE\'!A1] op[range] ref[\'WAHR\'!A1]');
            });

            it('should parse column/row references with sheets', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'A:XFD:Sheet1!A:$B:\'Sheet 4:Sheet1\'!$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[$0!A$1:$B$1048576] op[range] ref[$3:$0!$B$1:$A$1048576]');
                expectResult(ooxTokenizer.parseFormula('op', '1:001048576+1:$2+$1:2+$002:$001'), 'ref[$A1:$XFD1048576] op[add] ref[$A1:$XFD$2] op[add] ref[$A$1:$XFD2] op[add] ref[$A$2:$XFD$1]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'A:XFD:Sheet1!A:$B:\'Sheet 4:Sheet1\'!$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[$0!A$1:$B$1048576] op[range] ref[$3:$0!$B$1:$A$1048576]');
                expectResult(ooxTokenizer.parseFormula('ui', '1:001048576+1:$2+$1:2+$002:$001'), 'ref[$A1:$XFD1048576] op[add] ref[$A1:$XFD$2] op[add] ref[$A$1:$XFD2] op[add] ref[$A$2:$XFD$1]');
                // native ODF
                // ... not available
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'A:AMJ:Sheet1!A:$B:$\'Sheet 4\':Sheet1!$b:$a'), 'ref[A$1:AMJ$1048576] op[range] ref[0!A$1:$B$1048576] op[range] ref[$3:0!$B$1:$A$1048576]');
                expectResult(odfTokenizer.parseFormula('ui', '1:001048576+1:$2+$1:2+$002:$001'), 'ref[$A1:$AMJ1048576] op[add] ref[$A1:$AMJ$2] op[add] ref[$A$1:$AMJ2] op[add] ref[$A$2:$AMJ$1]');
            });

            it('should not accept oversized cell addresses', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'A1048757+XFE1+Sheet1!A1:XFE1'), 'name[A1048757] op[add] name[XFE1] op[add] ref[$0!A1] op[range] name[XFE1]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'A1048757+XFE1+Sheet1!A1:XFE1'), 'name[A1048757] op[add] name[XFE1] op[add] ref[$0!A1] op[range] name[XFE1]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '[.A1048757]+[.AMK1]+[Sheet1.A1:.XFE1]'), 'ref[#REF] op[add] ref[#REF] op[add] ref[0!#REF]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'A1048757+AMK1+Sheet1!A1:AMK1'), 'name[A1048757] op[add] name[AMK1] op[add] ref[0!A1] op[range] name[AMK1]');
            });

            it('should parse function names', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'true()+wahr()+my_func()'), 'func[TRUE] open close op[add] macro[wahr] open close op[add] macro[my_func] open close');
                expectResult(ooxTokenizer.parseFormula('op', 'Sheet1!true()+Sheet1!wahr()'), 'macro[$0!true] open close op[add] macro[$0!wahr] open close');
                expectResult(ooxTokenizer.parseFormula('op', 'Sheet1:Sheet2!my_func()+Sheet1!A1()'), 'name[Sheet1] op[range] macro[$1!my_func] open close op[add] macro[$0!A1] open close');
                expectResult(ooxTokenizer.parseFormula('op', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'func[FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] macro[FORMULA] open close');
                expectResult(ooxTokenizer.parseFormula('op', 'FORMELTEXT()+FORMEL()'), 'macro[FORMELTEXT] open close op[add] macro[FORMEL] open close');
                expectResult(ooxTokenizer.parseFormula('op', 'SUM (A1)'), 'name[SUM] op[isect] open ref[A1] close');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'true()+wahr()+my_func()'), 'macro[true] open close op[add] func[TRUE] open close op[add] macro[my_func] open close');
                expectResult(ooxTokenizer.parseFormula('ui', 'Sheet1!true()+Sheet1!wahr()'), 'macro[$0!true] open close op[add] macro[$0!wahr] open close');
                expectResult(ooxTokenizer.parseFormula('ui', 'Sheet1:Sheet2!my_func()+Sheet1!A1()'), 'name[Sheet1] op[range] macro[$1!my_func] open close op[add] macro[$0!A1] open close');
                expectResult(ooxTokenizer.parseFormula('ui', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'macro[_xlfn.FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] macro[FORMULA] open close');
                expectResult(ooxTokenizer.parseFormula('ui', 'FORMELTEXT()+FORMEL()'), 'func[FORMULATEXT] open close op[add] macro[FORMEL] open close');
                expectResult(ooxTokenizer.parseFormula('ui', 'SUMME (A1)'), 'name[SUMME] op[isect] open ref[A1] close');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', 'true()+wahr()+my_func()'), 'func[TRUE] open close op[add] macro[wahr] open close op[add] macro[my_func] open close');
                expectResult(odfTokenizer.parseFormula('op', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'macro[_xlfn.FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] func[FORMULATEXT] open close');
                expectResult(odfTokenizer.parseFormula('op', 'FORMELTEXT()+FORMEL()'), 'macro[FORMELTEXT] open close op[add] macro[FORMEL] open close');
                expectResult(odfTokenizer.parseFormula('op', 'SUM ([.A1])'), 'func[SUM] open ref[A1] close');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'true()+wahr()+my_func()'), 'macro[true] open close op[add] func[TRUE] open close op[add] macro[my_func] open close');
                expectResult(odfTokenizer.parseFormula('ui', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'macro[_xlfn.FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] macro[FORMULA] open close');
                expectResult(odfTokenizer.parseFormula('ui', 'FORMELTEXT()+FORMEL()'), 'macro[FORMELTEXT] open close op[add] func[FORMULATEXT] open close');
                expectResult(odfTokenizer.parseFormula('ui', 'SUMME (A1)'), 'name[SUMME] op[isect] open ref[A1] close');
            });

            it('should parse defined names', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                expectResult(ooxTokenizer.parseFormula('op', 'Sheet1!name1+\'Sheet 4\'!name2+Sheet1:Sheet2!name3'), 'name[$0!name1] op[add] name[$3!name2] op[add] name[Sheet1] op[range] name[$1!name3]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                expectResult(ooxTokenizer.parseFormula('ui', 'Sheet1!name1+\'Sheet 4\'!name2+Sheet1:Sheet2!name3'), 'name[$0!name1] op[add] name[$3!name2] op[add] name[Sheet1] op[range] name[$1!name3]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                expectResult(odfTokenizer.parseFormula('ui', 'Sheet1!name1'), 'bad[Sheet1!name1]');
            });

            it('should recognize negative numbers after operators', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '-1-1--1-(-1)-1,-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '-1-1--1-(-1)-1;-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '-1-1--1-(-1)-1,-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '-1-1--1-(-1)-1;-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
            });

            it('should fail for invalid matrix literals', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '{1,{2}}'), 'mat_open lit[1] mat_col bad[{2}}]');
                expectResult(ooxTokenizer.parseFormula('op', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(ooxTokenizer.parseFormula('op', '{1*1}'), 'mat_open lit[1] bad[*1}]');
                expectResult(ooxTokenizer.parseFormula('op', '{1,A1}'), 'mat_open lit[1] mat_col bad[A1}]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '{1;{2}}'), 'mat_open lit[1] mat_col bad[{2}}]');
                expectResult(ooxTokenizer.parseFormula('ui', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(ooxTokenizer.parseFormula('ui', '{1*1}'), 'mat_open lit[1] bad[*1}]');
                expectResult(ooxTokenizer.parseFormula('ui', '{1;A1}'), 'mat_open lit[1] mat_col bad[A1}]');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '{1;{2}}'), 'mat_open lit[1] mat_col bad[{2}}]');
                expectResult(odfTokenizer.parseFormula('op', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(odfTokenizer.parseFormula('op', '{1*1}'), 'mat_open lit[1] bad[*1}]');
                expectResult(odfTokenizer.parseFormula('op', '{1;A1}'), 'mat_open lit[1] mat_col bad[A1}]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '{1;{2}}'), 'mat_open lit[1] mat_col bad[{2}}]');
                expectResult(odfTokenizer.parseFormula('ui', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(odfTokenizer.parseFormula('ui', '{1*1}'), 'mat_open lit[1] bad[*1}]');
                expectResult(odfTokenizer.parseFormula('ui', '{1;A1}'), 'mat_open lit[1] mat_col bad[A1}]');
            });

            it('should fail for tokens of other grammars', function () {
                // native OOXML
                expectResult(ooxTokenizer.parseFormula('op', '1.2+3,4'), 'lit[1.2] op[add] lit[3] sep lit[4]');
                expectResult(ooxTokenizer.parseFormula('op', 'True=Wahr=False=Falsch'), 'lit[TRUE] op[eq] name[Wahr] op[eq] lit[FALSE] op[eq] name[Falsch]');
                expectResult(ooxTokenizer.parseFormula('op', '#Ref!<>#Bezug!+1'), 'lit[#REF] op[ne] bad[#Bezug!+1]');
                expectResult(ooxTokenizer.parseFormula('op', '{1,2;3,4}+{1;2|3;4}'), 'mat_open lit[1] mat_col lit[2] mat_row lit[3] mat_col lit[4] mat_close op[add] mat_open lit[1] mat_row lit[2] bad[|3;4}]');
                expectResult(ooxTokenizer.parseFormula('op', 'SUMME(1,2;3)'), 'macro[SUMME] open lit[1] sep lit[2] bad[;3)]');
                // localized OOXML
                expectResult(ooxTokenizer.parseFormula('ui', '1.2+3,4'), 'lit[1] bad[.2+3,4]');
                expectResult(ooxTokenizer.parseFormula('ui', 'True=Wahr=False=Falsch'), 'name[True] op[eq] lit[TRUE] op[eq] name[False] op[eq] lit[FALSE]');
                expectResult(ooxTokenizer.parseFormula('ui', '{1,2;3,4}+{1;2|3;4}'), 'mat_open lit[1.2] mat_col lit[3.4] mat_close op[add] mat_open lit[1] mat_col lit[2] mat_row lit[3] mat_col lit[4] mat_close');
                expectResult(ooxTokenizer.parseFormula('ui', 'SUMME(1,2;3)'), 'func[SUM] open lit[1.2] sep lit[3] close');
                // native ODF
                expectResult(odfTokenizer.parseFormula('op', '1.2+3,4'), 'lit[1.2] op[add] lit[3] sep lit[4]');
                expectResult(odfTokenizer.parseFormula('op', 'True=Wahr=False=Falsch'), 'lit[TRUE] op[eq] name[Wahr] op[eq] lit[FALSE] op[eq] name[Falsch]');
                expectResult(odfTokenizer.parseFormula('op', '#Ref!<>#Bezug!+1'), 'lit[#REF] op[ne] bad[#Bezug!+1]');
                expectResult(odfTokenizer.parseFormula('op', '{1,2;3,4}+{1;2|3;4}'), 'mat_open lit[1] bad[,2;3,4}+{1;2|3;4}]');
                expectResult(odfTokenizer.parseFormula('op', 'SUMME(1,2;3)'), 'macro[SUMME] open lit[1] sep lit[2] bad[;3)]');
                // localized ODF
                expectResult(odfTokenizer.parseFormula('ui', '1.2+3,4'), 'lit[1] bad[.2+3,4]');
                expectResult(odfTokenizer.parseFormula('ui', 'True=Wahr=False=Falsch'), 'name[True] op[eq] lit[TRUE] op[eq] name[False] op[eq] lit[FALSE]');
                expectResult(odfTokenizer.parseFormula('ui', '{1,2;3,4}+{1;2|3;4}'), 'mat_open lit[1.2] mat_col lit[3.4] mat_close op[add] mat_open lit[1] mat_col lit[2] mat_row lit[3] mat_col lit[4] mat_close');
                expectResult(odfTokenizer.parseFormula('ui', 'SUMME(1,2;3)'), 'func[SUM] open lit[1.2] sep lit[3] close');
            });
        });
    });

    // ========================================================================
});
