/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/impl/textfuncs'
], function (ErrorCode, FormulaContext, TextFuncs) {

    'use strict';

    // initialize the formula context
    var context = null;
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml').done(function (app) {
            context = new FormulaContext(app.getModel());
            done();
        });
    });

    // module TextFuncs =======================================================

    describe('Spreadsheet module TextFuncs', function () {

        it('should exist', function () {
            expect(TextFuncs).to.be.an('object');
        });

        // function implementations -------------------------------------------

        describe('function "FIXED"', function () {
            var FIXED = TextFuncs.FIXED;
            it('should exist', function () {
                expect(FIXED).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(FIXED).itself.to.respondTo('resolve');
            });
            it('should return the fixed formated string of assigned number', function () {
                expect(FIXED.resolve.call(context, 1234.567, 1)).to.equal('1.234,6');

                expect(FIXED.resolve.call(context, 1234.567, 1)).to.equal('1.234,6');
                expect(FIXED.resolve.call(context, 1234.567, -1)).to.equal('1.230');
                expect(FIXED.resolve.call(context, 1234.567, -1, true)).to.equal('1230');
                expect(FIXED.resolve.call(context, -1234.567, -1, true)).to.equal('-1230');

                expect(FIXED.resolve.call(context, 44.332, 2, true)).to.equal('44,33');
                expect(FIXED.resolve.call(context, 4444.332)).to.equal('4.444,33');
                expect(FIXED.resolve.call(context, 44.332)).to.equal('44,33');

                expect(FIXED.resolve.call(context, 1234567.332, 2, true)).to.equal('1234567,33');
                expect(FIXED.resolve.call(context, 1234567.332)).to.equal('1.234.567,33');

                expect(FIXED.resolve.call(context, 12345, -1, true)).to.equal('12350');
                expect(FIXED.resolve.call(context, 888, -1)).to.equal('890');

                expect(FIXED.resolve.call(context, 0.5, 2)).to.equal('0,50');

                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, 5)).to.equal('1.111.111.111,11111');
                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, 1)).to.equal('1.111.111.111,1');
                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, 0)).to.equal('1.111.111.111');
                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, -1)).to.equal('1.111.111.110');
                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, -5)).to.equal('1.111.100.000');

                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, -9)).to.equal('1.000.000.000');
                expect(FIXED.resolve.call(context, 1111111111 + 1 / 9, -10)).to.equal('0');

                expect(FIXED.resolve.call(context, 1, -1000)).to.equal('0');
                expect(FIXED.resolve.call(context, 10, -16)).to.equal('0');

                expect(FIXED.resolve.bind(context, 10, 128)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('function "ROT13"', function () {
            var ROT13 = TextFuncs.ROT13;
            it('should exist', function () {
                expect(ROT13).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(ROT13).itself.to.respondTo('resolve');
            });
            it('should return the rot13 decoding of assigned strings', function () {
                expect(ROT13.resolve.call(context, 'hallo')).to.equal('unyyb');
                expect(ROT13.resolve.call(context, '123')).to.equal('123');
            });
        });
    });

    // ========================================================================
});
