/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/impl/mathfuncs'
], function (ErrorCode, Matrix, FormulaContext, MathFuncs) {

    'use strict';

    // initialize the formula context
    var context = null;
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml').done(function (app) {
            context = new FormulaContext(app.getModel());
            done();
        });
    });

    // module MathFuncs =======================================================

    describe('Spreadsheet module MathFuncs', function () {

        it('should exist', function () {
            expect(MathFuncs).to.be.an('object');
        });

        // function implementations -------------------------------------------

        describe('function "SERIESSUM"', function () {
            var SERIESSUM = MathFuncs.SERIESSUM;
            it('should exist', function () {
                expect(SERIESSUM).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(SERIESSUM).itself.to.respondTo('resolve');
            });
            it('should return the sum of a power series', function () {
                expect(SERIESSUM.resolve.call(context, 1, 2, 3, new Matrix([[0]]))).to.equal(0);
                expect(SERIESSUM.resolve.call(context, 0.785398163, 0, 2, new Matrix([[1, -0.5, 0.041666667, -0.001388889]]))).to.almostEqual(0.7071032152046538);

                expect(SERIESSUM.resolve.call(context, 0.785398163, 0, 2, new Matrix([[1, 1], [-0.5, -0.5], [0.041666667, 0.041666667], [-0.001388889, -0.001388889]]))).to.almostEqual(0.8094800884021413);
                expect(SERIESSUM.resolve.call(context, 0.785398163, 0, 2, new Matrix([[1, 1], [-0.5, -0.5]]))).to.almostEqual(0.9547220449219412);
                expect(SERIESSUM.resolve.call(context, 0.785398163, 0, 2, new Matrix([[0.041666667, 0.041666667], [-0.001388889, -0.001388889]]))).to.almostEqual(0.0563382828657192);
            });
        });

        describe('function "SUM"', function () {
            var SUM = MathFuncs.SUM;
            it('should exist', function () {
                expect(SUM).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(SUM).itself.to.respondTo('resolve');
            });
            it('should return the sum of its arguments', function () {
                expect(SUM.resolve.call(context, 1, 2, 3)).to.equal(6);
                expect(SUM.resolve.call(context, 1, true, 3)).to.equal(5);
                expect(SUM.resolve.call(context, 1, '2', 3)).to.equal(6);
                expect(SUM.resolve.call(context, 1, null, 3)).to.equal(4);
            });
            it('should return the sum of a matrix', function () {
                expect(SUM.resolve.call(context, new Matrix([[1, 2], [3, 4]]))).to.equal(10);
                expect(SUM.resolve.call(context, new Matrix([[1, true], [3, 4]]))).to.equal(8);
                expect(SUM.resolve.call(context, new Matrix([[1, '2'], [3, 4]]))).to.equal(8);
                expect(SUM.resolve.call(context, new Matrix([[1, 'abc'], [3, 4]]))).to.equal(8);
                expect(SUM.resolve.call(context, new Matrix([[1, ''], [3, 4]]))).to.equal(8);
            });
            it('should throw #VALUE! error for invalid arguments', function () {
                expect(SUM.resolve.bind(context, 1, 'abc', 3)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(SUM.resolve.bind(context, 1, '', 3)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('function "SUMPRODUCT"', function () {
            var SUMPRODUCT = MathFuncs.SUMPRODUCT;
            it('should exist', function () {
                expect(SUMPRODUCT).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(SUMPRODUCT).itself.to.respondTo('resolve');
            });
            it('should return the sum product of its arguments', function () {
                expect(SUMPRODUCT.resolve.call(context, new Matrix([[3, 4], [8, 6], [1, 9]]))).to.equal(31);
                expect(SUMPRODUCT.resolve.call(context, new Matrix([[3, 4], [8, 6], [1, 9]]), new Matrix([[2, 7], [6, 7], [5, 3]]))).to.equal(156);

                expect(SUMPRODUCT.resolve.bind(context, new Matrix([[3, 4, 1], [8, 6, 1], [1, 9, 1]]), new Matrix([[2, 7], [6, 7], [5, 3]]))).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('function "CEILING.MATH"', function () {
            var CEILING_MATH = MathFuncs['CEILING.MATH'];
            it('should exist', function () {
                expect(CEILING_MATH).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(CEILING_MATH).itself.to.respondTo('resolve');
            });
            it('should return the ceiling of its arguments', function () {
                expect(CEILING_MATH.resolve.call(context, 6.7)).to.equal(7);
                expect(CEILING_MATH.resolve.call(context, 24.3, 5)).to.equal(25);
                expect(CEILING_MATH.resolve.call(context, 22, 5)).to.equal(25);
                expect(CEILING_MATH.resolve.call(context, -8.1, 2)).to.equal(-8);
            });

            it('should return the ceiling of its arguments with assigned mode', function () {
                expect(CEILING_MATH.resolve.call(context, -5.4, 1, -1)).to.equal(-6);
                expect(CEILING_MATH.resolve.call(context, -5.4, 1, 1)).to.equal(-6);
                expect(CEILING_MATH.resolve.call(context, -5.4, 1, 0)).to.equal(-5);
                expect(CEILING_MATH.resolve.call(context, -5.4, 1)).to.equal(-5);

                expect(CEILING_MATH.resolve.call(context, 5.4, 1, -1)).to.equal(6);
                expect(CEILING_MATH.resolve.call(context, 5.4, 1, 1)).to.equal(6);
                expect(CEILING_MATH.resolve.call(context, 5.4, 1, 0)).to.equal(6);
                expect(CEILING_MATH.resolve.call(context, 5.4, 1)).to.equal(6);

                expect(CEILING_MATH.resolve.call(context, -5.4, 4, -1)).to.equal(-8);
                expect(CEILING_MATH.resolve.call(context, -5.4, 4, 1)).to.equal(-8);
                expect(CEILING_MATH.resolve.call(context, -5.4, 4, 0)).to.equal(-4);
                expect(CEILING_MATH.resolve.call(context, -5.4, 4)).to.equal(-4);
            });
        });

        describe('function "FLOOR.MATH"', function () {
            var FLOOR_MATH = MathFuncs['FLOOR.MATH'];
            it('should exist', function () {
                expect(FLOOR_MATH).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(FLOOR_MATH).itself.to.respondTo('resolve');
            });
            it('should return the ceiling of its arguments', function () {
                expect(FLOOR_MATH.resolve.call(context, 6.7)).to.equal(6);
                expect(FLOOR_MATH.resolve.call(context, 24.3, 5)).to.equal(20);
                expect(FLOOR_MATH.resolve.call(context, 22, 5)).to.equal(20);
                expect(FLOOR_MATH.resolve.call(context, -8.1, 2)).to.equal(-10);
            });

            it('should return the ceiling of its arguments with assigned mode', function () {
                expect(FLOOR_MATH.resolve.call(context, -5.6, 1, -1)).to.equal(-5);
                expect(FLOOR_MATH.resolve.call(context, -5.6, 1, 1)).to.equal(-5);
                expect(FLOOR_MATH.resolve.call(context, -5.6, 1, 0)).to.equal(-6);
                expect(FLOOR_MATH.resolve.call(context, -5.6, 1)).to.equal(-6);

                expect(FLOOR_MATH.resolve.call(context, 5.6, 1, -1)).to.equal(5);
                expect(FLOOR_MATH.resolve.call(context, 5.6, 1, 1)).to.equal(5);
                expect(FLOOR_MATH.resolve.call(context, 5.6, 1, 0)).to.equal(5);
                expect(FLOOR_MATH.resolve.call(context, 5.6, 1)).to.equal(5);

                expect(FLOOR_MATH.resolve.call(context, -5.6, 4, -1)).to.equal(-4);
                expect(FLOOR_MATH.resolve.call(context, -5.6, 4, 1)).to.equal(-4);
                expect(FLOOR_MATH.resolve.call(context, -5.6, 4, 0)).to.equal(-8);
                expect(FLOOR_MATH.resolve.call(context, -5.6, 4)).to.equal(-8);
            });
        });
    });

    // ========================================================================
});
