/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/impl/conversionfuncs'
], function (Utils, ErrorCode, FormulaContext, ConversionFuncs) {

    'use strict';

    // initialize the formula context
    var context = null;
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml').done(function (app) {
            context = new FormulaContext(app.getModel());
            done();
        });
    });

    // module ConversionFuncs =================================================

    describe('Spreadsheet module ConversionFuncs', function () {

        it('should exist', function () {
            expect(ConversionFuncs).to.be.an('object');
        });

        // function implementations -------------------------------------------

        describe('function "BIN2DEC"', function () {
            var BIN2DEC = ConversionFuncs.BIN2DEC;
            it('should exist', function () {
                expect(BIN2DEC).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(BIN2DEC).itself.to.respondTo('resolve');
            });
            it('should return the decimal of its hex arguments', function () {
                expect(BIN2DEC.resolve.call(context, '1100100')).to.equal(100);
                expect(BIN2DEC.resolve.call(context, '1111111111')).to.equal(-1);

                expect(BIN2DEC.resolve.bind(context, '2')).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(BIN2DEC.resolve.bind(context, '121')).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(BIN2DEC.resolve.bind(context, '11111111111')).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        // DECIMAL
        describe('function "DECIMAL"', function () {
            var DECIMAL = ConversionFuncs.DECIMAL;
            it('should exist', function () {
                expect(DECIMAL).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(DECIMAL).itself.to.respondTo('resolve');
            });
            it('should return the decimal of its hex arguments', function () {
                expect(DECIMAL.resolve.call(context, 1, 36)).to.equal(1);
                expect(DECIMAL.resolve.call(context, 'FF', 16)).to.equal(255);
                expect(DECIMAL.resolve.call(context, '111', 2)).to.equal(7);
                expect(DECIMAL.resolve.call(context, 'zap', 36)).to.equal(45745);
                expect(DECIMAL.resolve.call(context,  Utils.repeatString('f', 198), 36)).to.almostEqual(6.02446334927868e+307);

                expect(DECIMAL.resolve.bind(context, 1, 37)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(DECIMAL.resolve.bind(context, -1, 36)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);

                expect(DECIMAL.resolve.bind(context,  Utils.repeatString('f', 199), 36)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);

                expect(DECIMAL.resolve.bind(context, '123h', 8)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(DECIMAL.resolve.bind(context, 'ff', 15)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('function "HEX2DEC"', function () {
            var HEX2DEC = ConversionFuncs.HEX2DEC;
            it('should exist', function () {
                expect(HEX2DEC).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(HEX2DEC).itself.to.respondTo('resolve');
            });
            it('should return the decimal of its hex arguments', function () {
                expect(HEX2DEC.resolve.call(context, 'A5')).to.equal(165);
                expect(HEX2DEC.resolve.call(context, 'FFFFFFFF5B')).to.equal(-165);
                expect(HEX2DEC.resolve.call(context, '3DA408B9')).to.equal(1034160313);

                expect(HEX2DEC.resolve.bind(context, 'G')).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('function "ROMAN"', function () {
            var ROMAN = ConversionFuncs.ROMAN;
            it('should exist', function () {
                expect(ROMAN).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(ROMAN).itself.to.respondTo('resolve');
            });
            it('should return the roman of its arguments', function () {
                expect(ROMAN.resolve.call(context, 2)).to.equal('II');
                expect(ROMAN.resolve.call(context, 1)).to.equal('I');

                expect(ROMAN.resolve.call(context, 499)).to.equal('CDXCIX');
                expect(ROMAN.resolve.call(context, 499, 0)).to.equal('CDXCIX');
                expect(ROMAN.resolve.call(context, 499, 1)).to.equal('LDVLIV');
                expect(ROMAN.resolve.call(context, 499, 2)).to.equal('XDIX');
                expect(ROMAN.resolve.call(context, 499, 3)).to.equal('VDIV');
                expect(ROMAN.resolve.call(context, 499, 4)).to.equal('ID');

                expect(ROMAN.resolve.call(context, 0)).to.equal('');

                expect(ROMAN.resolve.bind(context, -1)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(ROMAN.resolve.bind(context, 49999)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });

        });

    });

    // ========================================================================
});
