/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/portal/portalutils'
], function (PortalUtils) {

    'use strict';

    // static class PortalUtils ===============================================

    describe('Portal module PortalUtils', function () {

        it('should exist', function () {
            expect(PortalUtils).to.be.an('object');
        });

        it('should have a icon-create function', function () {
            expect(PortalUtils).to.respondTo('createDocumentIcon');
        });

        // it('should return the correct module-name', function () {
        //     expect(PortalUtils.getModuleNameForFile('bla.docx')).to.equal('io.ox/office/text');
        // });

        it('createDocumentIcon function should return the correct app-icon', function () {
            expect(PortalUtils.createDocumentIcon('nevermind')[0].outerHTML).to.equal('<i class="document-icon fa fa-file"></i>');
          //expect(PortalUtils.createDocumentIcon('bla.docx')[0].outerHTML).to.equal('<i class="document-icon fa fa-file-text text"></i>');
            expect(PortalUtils.createDocumentIcon('bla.xlsx')[0].outerHTML).to.equal('<i class="document-icon fa fa-table spreadsheet"></i>');
        });

        it('formatDate should return the formatted date', function () {
            var TIME_FORMAT     = new RegExp('^([0-1]?[0-9]|2[0-3]):[0-5][0-9] [APap][mM]$'),  // z.B. => 3:00 AM
                currentDate     = new Date(),
                currentYear     = currentDate.getFullYear(),
                currentMonth    = ('0' + (currentDate.getMonth() + 1)).slice(-2),
                currentDay      = ('0' + currentDate.getDate()).slice(-2),

                date1 = new Date(currentYear + '-03-25T12:00:00'),
                date2 = new Date(currentYear + '-' + currentMonth + '-' + currentDay + 'T02:00:01');

            expect(PortalUtils.formatDate(43200)).to.equal('01/01/1970');
            expect(PortalUtils.formatDate('string')).to.equal('unknown');
            expect(PortalUtils.formatDate(date1.getTime())).to.equal('25.Mar');
            expect(PortalUtils.formatDate(date2.getTime())).to.match(TIME_FORMAT);
        });

        it('should return the correct application base name', function () {
            expect(PortalUtils.getAppBaseName('nevermind')).to.equal(null);
          //expect(PortalUtils.getAppBaseName('bla.docx')).to.equal('text');
            expect(PortalUtils.getAppBaseName('bla.xlsx')).to.equal('spreadsheet');
        });

        it('should return the filename without extension', function () {
            expect(PortalUtils.removeFileExtension('nevermind')).to.equal('nevermind');
            expect(PortalUtils.removeFileExtension('bla.docx')).to.equal('bla');
            expect(PortalUtils.removeFileExtension('bla.xlsx')).to.equal('bla');
        });

        it('should return whether the filename end with the given suffix', function () {
            expect(PortalUtils.stringEndsWith('nevermind', 'docx')).to.equal(false);
            expect(PortalUtils.stringEndsWith('bla.docx', 'docx')).to.equal(true);
            expect(PortalUtils.stringEndsWith('bla.xlsx', 'docx')).to.equal(false);
        });

        it('should return the enabled applications', function () {
            expect(PortalUtils.getEnabledApps()).to.deep.equal(['text', 'spreadsheet', 'presentation']);
        });

        it('should return the title of the active app in lowercase', function () {
            expect(PortalUtils.getActiveApp({ attributes: { title: 'Text' } })).to.equal('text');
            expect(PortalUtils.getActiveApp({ attributes: { title: 'Text' } })).not.to.equal('Text');
        });

        it('moveFocus function should exist', function () {
            expect(PortalUtils).to.respondTo('moveFocus');
        });

        it('should set the focus and not return anything', function () {
            var firstNode = $('<div>'),
                secondNode = $('<div>'),
                thirdNode = $('<div>'),

                fourthNode = $('<div tabindex="1">'),
                fifthNode = $('<div tabindex="2">'),
                sixthNode = $('<div tabindex="3">');

            $('<div>').append(firstNode, secondNode, thirdNode);
            $('<div>').append(fourthNode, fifthNode, sixthNode);

            expect(PortalUtils.moveFocus()).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode)).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode, 'left')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode, 'down')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode, 'up')).to.be.equal(undefined);

            expect(PortalUtils.moveFocus(fifthNode)).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(fifthNode, 'left')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(fifthNode, 'down')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(fifthNode, 'up')).to.be.equal(undefined);
        });

        it('keydownHandler should exist', function () {
            expect(PortalUtils).to.respondTo('keydownHandler');
        });

        it('should handle keydown events', function () {
            var spy = sinon.spy(),
                ele = $('<a>').on('click', spy),

                e1      = new $.Event('click', { target: ele }),
                e2      = new $.Event('click', { target: ele, data: { grid: true } }),
                e3a     = new $.Event('click', { target: ele, keyCode: 13 }),
                e3b     = new $.Event('click', { target: ele, keyCode: 37 }),
                e3c     = new $.Event('click', { target: ele, keyCode: 38 }),
                e3c2    = new $.Event('click', { target: ele, keyCode: 38, data: { grid: true } }),
                e3d     = new $.Event('click', { target: ele, keyCode: 39 }),
                e3e     = new $.Event('click', { target: ele, keyCode: 40 }),
                e3e2    = new $.Event('click', { target: ele, keyCode: 40, data: { grid: true } });

            expect(PortalUtils.keydownHandler()).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e1)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e2)).to.be.equal(undefined);

            expect(PortalUtils.keydownHandler(e3a)).to.be.equal(undefined);
            expect(spy.calledOnce).to.equal(true);

            expect(PortalUtils.keydownHandler(e3b)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3c)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3c2)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3d)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3e)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3e2)).to.be.equal(undefined);
        });

    });

    // ========================================================================
});
