/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var _ = require('underscore');
    var Utils = require('../utils/gruntutils')(grunt);
    var minimatch = require('minimatch');
    var Esprima = require('esprima');
    var Syntax = Esprima.Syntax;

    // task configuration =====================================================

    grunt.config('reqlint', {

        options: {
            base: 'apps',
            externals: 'io.ox/{backbone,contacts,core,files,mail,realtime,metrics}/**/*',
            libs: 'io.ox/office/drawinglayer/lib/**/*',
            plugins: {
                less: {
                    ext: 'less'
                },
                json: {
                    ext: 'json'
                },
                settings: {
                    match: 'io.ox/{core,files,office}'
                },
                gettext: {
                    match: function (module, dep) {

                        var moduleDirs = module.split('/');
                        var depDirs = dep.split('/');

                        if (moduleDirs.slice(0, 3).join('/') !== depDirs.slice(0, 3).join('/')) {
                            return 'Path must match path of project "' + moduleDirs[2] + '"';
                        }

                        if (depDirs.length <= 3) {
                            return 'Path must point into project "' + moduleDirs[2] + '"';
                        }
                    }
                }
            }
        },

        apps: {
            src: 'apps/**/*.js',
            options: {
                defines: ['define', 'define.async'],
                anonymous: false,
                packages: {
                    toolkit: {
                        src: 'io.ox/office/{tk,settings}/**/*'
                    },
                    baseframework: {
                        src: 'io.ox/office/baseframework/**/*',
                        dependsOn: 'toolkit'
                    },
                    portal: {
                        src: 'io.ox/office/portal/**/*',
                        dependsOn: 'editframework'
                    },
                    editframework: {
                        src: 'io.ox/office/{editframework,drawinglayer}/**/*',
                        dependsOn: 'baseframework'
                    },
                    textframework: {
                        src: 'io.ox/office/textframework/**/*',
                        dependsOn: 'editframework'
                    },
                    text: {
                        src: 'io.ox/office/text/**/*',
                        dependsOn: 'textframework'
                    },
                    presentation: {
                        src: 'io.ox/office/presentation/**/*',
                        dependsOn: 'textframework'
                    },
                    spreadsheet: {
                        src: 'io.ox/office/spreadsheet/**/*',
                        dependsOn: 'editframework'
                    }
                }
            }
        },

        spec: {
            src: 'spec/**/*_spec.js',
            options: {
                anonymous: true
            }
        }
    });

    // helper functions =======================================================

    /**
     * Returns whether the passed module path matches the glob pattern.
     *
     * @param {String} module
     *  The name of a module to be matched against the pattern.
     *
     * @param {String|Array<String>|Null} pattern
     *  The glob pattern to be used to match the passed module name. Can be a
     *  single pattern as string, an array of patterns, or null which matches
     *  everything.
     */
    function isMatchingModule(module, pattern) {
        function match(ptn) { return minimatch(module, ptn); }
        return _.isString(pattern) ? match(pattern) : _.isArray(pattern) ? pattern.some(match) : true;
    }

    // class SourceModule =====================================================

    /**
     * Representation of a source file defining a RequireJS module.
     *
     * @constructor
     *
     * @property {String} path
     *  The path to the source file, as passed to the constructor.
     *
     * @property {Object} options
     *  All task configuration settings for the source file, as passed to the
     *  constructor.
     *
     * @property {String} source
     *  The original source code.
     *
     * @property {Array<String>} lines
     *  The source code, split into single source lines.
     *
     * @property {Object} syntax
     *  The full syntax tree of the source file, as JSON object.
     *
     * @property {String|Null} name
     *  The shortened module name (no extension, no base path), as defined in
     *  the source file. The value null represents anonymous modules.
     *
     * @property {Array<Object>|Null} imports
     *  The imported modules, as array of AST elements (JSON objects); or null,
     *  if no modules will be imported.
     *
     * @property {Object|Null} contents
     *  The contents of this module, as AST element (JSON object).
     *
     * @property {Number} errors
     *  The number of errors already found in the source file.
     */
    function SourceModule(path, options) {

        this.path = path;
        this.options = options;

        this.source = grunt.file.read(path);
        this.lines = this.source.split(/\n/);
        this.syntax = Esprima.parse(this.source, { loc: true });

        this.name = null;
        this.imports = null;
        this.contents = null;

        this.errors = 0;

    } // class SourceModule

    // public methods ---------------------------------------------------------

    /**
     * Returns the file path to the passed module by prepending the base path
     * set in the task configuration. A file extension will be appended, if
     * specified.
     */
    SourceModule.prototype.getFilePath = function (module, ext) {
        function posixPath(path) { return path.replace(/\\/g, '/').replace(/\/$/g, ''); }
        var base = posixPath(this.options.base);
        return (base ? (base + '/') : '') + posixPath(module) + (ext ? ('.' + ext) : '');
    };

    /**
     * Writes the specified source lines to the logger.
     *
     * @param {Number} start
     *  Zero-based index of the first line to be logged.
     *
     * @param {Number} count
     *  Number of lines to be logged.
     */
    SourceModule.prototype.dumpLines = function (start, count) {
        this.lines.slice(start, start + count).forEach(function (l, i) {
            grunt.log.writeln(('     ' + (start + i + 1)).substr(-6) + ' |' + l);
        });
    };

    /**
     * Writes an error message with code excerpt to the logger.
     *
     * @param {String} message
     *  The message text to be logged.
     *
     * @param {Object} [position]
     *  A source code position, with the properties 'line' and 'column', as
     *  provided by the Esprima parser.
     *
     * @returns {Boolean}
     *  The constant value false.
     */
    SourceModule.prototype.dumpError = function (message, position) {
        grunt.log.writeln(message.bold + ' at ' + this.path.green + ' :');
        if (position) {
            this.dumpLines(position.line - 3, 3);
            grunt.log.writeln(new Array(position.column + 9).join('-') + '^');
            this.dumpLines(position.line, 2);
        }
        this.errors += 1;
        return false;
    };

    /**
     * Returns whether the passed AST element is a string literal.
     *
     * @param {Object} element
     *  The AST element to be analyzed.
     *
     * @returns {Boolean}
     *  Whether the passed AST element is a string literal.
     */
    SourceModule.prototype.isStringLiteral = function (element) {
        return (element.type === Syntax.Literal) && _.isString(element.value);
    };

    /**
     * Returns whether the passed AST element is an identifier with a specific
     * name.
     *
     * @param {Object} element
     *  The AST element to be analyzed.
     *
     * @param {String} identifier
     *  The expected name of the identifier.
     *
     * @returns {Boolean}
     *  Whether the passed AST element is the expected identifier.
     */
    SourceModule.prototype.isIdentifier = function (element, identifier) {
        return (element.type === Syntax.Identifier) && (element.name === identifier);
    };

    /**
     * Returns whether the passed AST element is a chained member expression of
     * arbitrary length, and with specific property names.
     *
     * @param {Object} element
     *  The AST element to be analyzed.
     *
     * @param {Object} element
     *  The AST element to be analyzed.
     *
     * @param {Array<String>} identifiers
     *  An array of expected identifier names. The array must contain at least
     *  two identifier names. Example: The parameter ['a', 'b', 'c', 'd'] will
     *  match the chained member expression 'a.b.c.d'.
     *
     * @returns {Boolean}
     *  Whether the passed AST element is the expected member expression.
     */
    SourceModule.prototype.isMemberExpression = function (element, identifiers) {
        return (element.type === Syntax.MemberExpression) &&
            !element.computed &&
            ((identifiers.length === 2) ? this.isIdentifier(element.object, identifiers[0]) : this.isMemberExpression(element.object, identifiers.slice(0, -1))) &&
            this.isIdentifier(element.property, identifiers[identifiers.length - 1]);
    };

    /**
     * Returns whether the passed AST element is the call of a local or global
     * function, or an object method from a property chain with arbitrary
     * length.
     *
     * @param {Object} element
     *  The AST element to be analyzed.
     *
     * @param {String|Array<String>} identifiers
     *  A single identifier name (as string or array with one element) for
     *  simple function calls, or an array of at least two identifier names for
     *  an object method call. Example: The parameter value ['a','b','c','d']
     *  will match the method call 'a.b.c.d()'.
     *
     * @returns {Boolean}
     *  Whether the passed AST element is the expected method call.
     */
    SourceModule.prototype.isCallExpression = function (element, identifiers) {
        return (element.type === Syntax.CallExpression) &&
            (_.isString(identifiers) ? this.isIdentifier(element.callee, identifiers) :
            (identifiers.length === 1) ? this.isIdentifier(element.callee, identifiers[0]) :
            this.isMemberExpression(element.callee, identifiers));
    };

    /**
     * Performs initial checks for file validity. If this method returns false,
     * no further checks shall run on the source file. On success, the
     * properties 'name' and 'args' of this instance have been initialized.
     *
     * @returns {Boolean}
     *  Whether the source file defines a RequireJS module with a valid module
     *  name.
     */
    SourceModule.prototype.checkModuleDefinition = function () {

        // basic check for valid JS programs
        if (!_.isObject(this.syntax) || (this.syntax.type !== Syntax.Program)) {
            return this.dumpError('Invalid JavaScript');
        }

        // check existence of contents
        var body = this.syntax.body;
        if (!_.isArray(body) || (body.length === 0)) {
            return this.dumpError('Source file is empty');
        }

        // the entire module must consist of a single expression statement
        if (this.syntax.body.length > 1) {
            return this.dumpError('Module must consist of a single expression statement', body[1].loc.start);
        }

        // first expression statement must be a call expression
        if ((body[0].type !== Syntax.ExpressionStatement) || (body[0].expression.type !== Syntax.CallExpression)) {
            return this.dumpError('Module must consist of a single call expression', body[0].loc.start);
        }

        // check for supported define() calls
        var expression = body[0].expression;
        if (!this.options.defines.some(function (define) { return this.isCallExpression(expression, define); }, this)) {
            return this.dumpError('Invalid module definition', expression.callee.loc.start);
        }

        // reject empty define() calls
        var args = expression.arguments.slice();
        if (args.length === 0) {
            return this.dumpError('Missing module definition', expression.callee.loc.end);
        }

        // check module name (according to anonymous mode)
        var nameArg = args[0];
        if (this.isStringLiteral(nameArg)) {

            // no module name allowed, if anonymous mode is required
            if (this.options.anonymous === true) {
                return this.dumpError('Expecting anonymous module', nameArg.loc.start);
            }

            // module name must expand to file path
            if (this.path !== this.getFilePath(nameArg.value, 'js')) {
                return this.dumpError('Module name must match file name', nameArg.loc.start);
            }

            this.name = nameArg.value;
        }

        // do not accept anonymous module when module names are required
        if (!this.name && (this.options.anonymous === false)) {
            return this.dumpError('Missing module name', nameArg.loc.start);
        }

        // extract the array of imported modules
        var importsArg = args[this.name ? 1 : 0];
        if (importsArg && (importsArg.type === Syntax.ArrayExpression)) {
            this.imports = importsArg.elements;
        }

        // extract the module contents
        var contArg = args[(this.name ? 1 : 0) + (this.imports ? 1 : 0)];
        if (!contArg || ((contArg.type !== Syntax.FunctionExpression) && (contArg.type !== Syntax.ObjectExpression) && (contArg.type !== Syntax.ArrayExpression))) {
            return this.dumpError('Missing module contents', contArg ? contArg.loc.start : args[args.length - 1].end);
        }
        this.contents = contArg;

        return true;
    };

    /**
     * Checks the passed AST element which is expected to be the name of an
     * imported module.
     *
     * @param {Object} element
     *  An AST element that is expected to be a literal string.
     *
     * @returns {Boolean}
     *  Whether the passed AST element is a valid external module name.
     */
    SourceModule.prototype.checkModuleImport = function (element) {

        // array elements must be non-empty string literals
        if (!this.isStringLiteral(element) || (element.value.length === 0)) {
            return this.dumpError('Non-empty string literal expected', element.loc.start);
        }

        // exact position in string for error messages
        var pos = { line: element.loc.start.line, column: element.loc.start.column + 1 };

        // name of an imported module (with optional leading plug-in name)
        var exclamPos = element.value.indexOf('!');
        var pluginName = (exclamPos >= 0) ? element.value.substr(0, exclamPos) : null;
        var importModule = element.value.substr(exclamPos + 1);

        // build file configuration according to path and RequireJS plug-in
        var fileSettings = _.isString(pluginName) ? this.options.plugins[pluginName] : { ext: 'js' };
        if (!_.isObject(fileSettings)) {
            return this.dumpError('Unknown RequireJS plug-in (please adjust option "plugins" in task configuration)', pos);
        }

        // adjust position to start of module name (after plug-in name)
        if (_.isString(pluginName)) {
            pos.column += pluginName.length + 1;
        }

        // handle plug-ins that do not refer to actual files
        if (fileSettings.match) {
            var result = false;
            if (_.isBoolean(fileSettings.match)) {
                result = fileSettings.match;
            } else if (_.isString(fileSettings.match) || _.isArray(fileSettings.match)) {
                result = isMatchingModule(importModule, fileSettings.match);
            } else if (_.isFunction(fileSettings.match)) {
                result = fileSettings.match(this.name, importModule);
            } else {
                grunt.warn('Unrecognized plug-in matcher. Please check option "plugins" in task configuration.');
            }
            if (!_.isUndefined(result) && (result !== true)) {
                return this.dumpError(result || 'Plug-in argument does not match allowed values', pos);
            }

            // nothing more to test
            return true;
        }

        // no recursive imports of this module
        if (!pluginName && (this.name === importModule)) {
            return this.dumpError('Recursive import of this module', pos);
        }

        // ignore files from external projects
        if (isMatchingModule(importModule, this.options.externals)) {
            return true;
        }

        // check existence, but not for third-party libraries
        if (!isMatchingModule(importModule, this.options.libs)) {
            // build the full path of the imported file
            var importPath = this.getFilePath(importModule, fileSettings.ext);
            if (!grunt.file.exists(importPath)) {
                return this.dumpError('File does not exist', pos);
            }
        }

        // check package dependencies (not for anonymous modules)
        if (_.isString(this.name)) {
            _.every(this.options.packages, function (settings) {
                var error = isMatchingModule(this.name, settings.src) && !isMatchingModule(importModule, settings.patterns);
                return error ? this.dumpError('Package dependency violation', pos) : true;
            }, this);
        }

        return true;
    };

    // task implementation ====================================================

    grunt.registerMultiTask('reqlint', 'Checks the require.js moule definitions and imports.', function () {

        // read task options for current target
        var options = this.options({
            base: '',
            defines: 'define',
            anonymous: null,
            externals: null,
            libs: null,
            plugins: null,
            packages: null
        });

        // prepare supported define() calls
        options.defines = Utils.makeArray(options.defines).map(function (define) {
            return define.split('.');
        });

        // prepare package settings
        _.each(options.packages, function preparePackage(settings) {

            // do not process prepared packages (preparation is done recursively, see below)
            if (settings.patterns) { return; }

            // create new property 'patterns' with package-internal source patterns
            settings.patterns = Utils.makeArray(settings.src).slice();

            // add the import paths of parent packages to the 'patterns' property
            Utils.makeArray(settings.dependsOn).forEach(function (parentName) {
                var parentSettings = options.packages[parentName];
                preparePackage(parentSettings);
                settings.patterns = settings.patterns.concat(parentSettings.patterns);
            });
        });

        // number of errors found so far in all files
        var errors = 0;

        // process all source files
        this.filesSrc.forEach(function (path) {

            // read and parse the source file
            var sourceModule = new SourceModule(path, options);

            // use try/finally construct to allow early returns while updating global error count
            try {

                // check module definition (do not run further checks on errors)
                if (!sourceModule.checkModuleDefinition()) { return; }

                // check global module imports
                if (_.isArray(sourceModule.imports)) {
                    sourceModule.imports.forEach(sourceModule.checkModuleImport, sourceModule);
                }

                // search for require() calls in source code (the initial reg-exp check speeds up build time significantly)
                if ((sourceModule.contents.type === Syntax.FunctionExpression) && /require\(/.test(sourceModule.source)) {
                    (function checkContentElement(element) {

                        // check calls to the require() function
                        if (sourceModule.isCallExpression(element, 'require')) {
                            var args = element.arguments;
                            if ((args.length >= 1) && (args[0].type === Syntax.ArrayExpression)) {
                                args[0].elements.forEach(function (arrayElement) {
                                    // silently skip everything that is not a simple string literal
                                    if (sourceModule.isStringLiteral(arrayElement)) {
                                        sourceModule.checkModuleImport(arrayElement);
                                    }
                                });
                            }
                        }

                        // crawl into the AST element (also for a require() function call,
                        // because its parameters may be callback functions)
                        _.each(element, function (value, key) {

                            // skip default properties of all ASt elements
                            if ((key === 'type') || (key === 'loc')) { return; }

                            // recursively check object/array properties
                            if (_.isArray(value)) {
                                value.forEach(checkContentElement);
                            } else if (_.isObject(value)) {
                                checkContentElement(value);
                            }
                        });
                    }(sourceModule.contents.body));
                }

            } finally {
                errors += sourceModule.errors;
            }
        });

        // show resulting task message
        var resolveTask = this.async();
        if (errors > 0) {
            grunt.log.error(errors + ' module definition ' + grunt.util.pluralize(errors, 'error/errors') + ' found.');
            resolveTask(false);
        } else {
            var files = this.filesSrc.length;
            grunt.log.ok(files + ' ' + grunt.util.pluralize(files, 'file/files') + ' without module definition errors.');
            resolveTask(true);
        }
    });
};
