/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/domconsole', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/forms'
], function (Config, Forms) {

    'use strict';

    // static class DOMConsole ================================================

    var DOMConsole = {};

    // static methods ---------------------------------------------------------

    /**
     * Forwards console output into a fixed DOM node. Especially useful for
     * mobile devices and a better stability (some browsers, e.g. Chrome, have
     * problems with very large console logs).
     */
    DOMConsole.enable = Config.DEBUG ? _.once(function () {

        var ICONS = { info: 'fa-info-circle', warn: 'fa-warning', error: 'fa-times-circle' },
            consoleNode = $('<div>', { id: 'io-ox-office-console' }).addClass('noI18n collapsed').hide().appendTo('body'),
            outputNode = $('<div class="output">').appendTo(consoleNode),
            buttonsNode = $('<tr>').appendTo($('<table class="buttons">').appendTo(consoleNode)),
            states = {};

        function scrollDown() {
            if (states.scroll) {
                outputNode.scrollTop(outputNode[0].scrollHeight);
            }
        }

        function writeToDOMConsole(type, args) {
            var rowNode = $('<p class="' + type + '">').appendTo(outputNode).append(
                '<span class="time">' + moment().format('HH:mm:ss.SSS') + '</span>',
                '<i class="fa ' + (ICONS[type] || 'fa-none') + '">'
            );
            args.forEach(function (msg) {
                if (_.isNull(msg) || _.isUndefined(msg)) {
                    rowNode.append('<span style="font-style:italic">' + msg + '</span>');
                } else {
                    var tooltip = _.isObject(msg) ? Forms.createLoggingTooltipMarkup(msg) : null;
                    var span = '<span' + (tooltip ? (' title="' + tooltip + '"') : '') + '>';
                    rowNode.append($(span).text(msg).toggleClass('hover', !!tooltip));
                }
            });
            scrollDown();
        }

        function selectAll() {
            var docRange = window.document.createRange();
            docRange.setStart(consoleNode[0], 0);
            docRange.setEnd(consoleNode[0], 1);
            window.getSelection().removeAllRanges();
            window.getSelection().addRange(docRange);
        }

        function handleButtonClick(button, callback) {
            button.on({
                click: function () { callback(); return false; },
                dblclick: false
            });
        }

        function createButton(label, tooltip, callback) {
            var button = $('<td>').attr('title', tooltip).text(label).appendTo(buttonsNode);
            handleButtonClick(button, callback);
            return button;
        }

        function updateToggleButton(button, stateName, state, callback) {
            states[stateName] = state;
            button.find('>i').attr('class', 'fa ' + (state ? 'fa-check-square-o' : 'fa-square-o'));
            callback(state);
        }

        function createToggleButton(label, tooltip, stateName, initState, callback) {
            var button = createButton(label, tooltip, function () {
                updateToggleButton(button, stateName, !states[stateName], callback);
            });
            button.prepend('<i>');
            updateToggleButton(button, stateName, initState, callback);
            return button;
        }

        // create and prepare buttons
        createToggleButton('Show Time', 'Show timestamps for all messages', 'time', true, function (state) { outputNode.toggleClass('show-time', state); });
        createToggleButton('Auto Errors', 'Automatically show the console on uncaught JS exceptions', 'error', true, _.noop);
        createToggleButton('Auto Scroll', 'Automatically scroll down after new messages', 'scroll', true, _.noop);
        createButton('Select All', 'Select all messages', selectAll);
        createButton('Clear', 'Remove all messages', function () { outputNode.empty(); });
        createButton('Hide', 'Collapse this console window', function () { consoleNode.addClass('collapsed'); });

        // show console when clicking on the collapsed icon
        consoleNode.append('<i class="fa fa-bug">');
        handleButtonClick(consoleNode, function () {
            consoleNode.removeClass('collapsed');
            scrollDown();
        });

        // replace implementations of browser console methods
        ['log', 'info', 'warn', 'error'].forEach(function (methodName) {
            window.console[methodName] = _.wrap(window.console[methodName], function (origMethod) {
                var args = _.toArray(arguments).slice(1);
                writeToDOMConsole(methodName, args);
                return _.isFunction(origMethod) ? origMethod.apply(this, args) : undefined;
            });
        });

        // global exception handler to forward unhandled exceptions to the DOM console
        if (_.isFunction(window.onerror)) {
            window.console.warn('Logger.enableDOMConsole(): global error handler exists already');
        } else {
            window.onerror = function () {
                if (states.error) { consoleNode.removeClass('collapsed'); }
                writeToDOMConsole('error', _.toArray(arguments));
                return false;
            };
        }

        // load CSS formatting on demand (not globally before using the DOM console)
        require(['less!io.ox/office/tk/utils/domconsole']).done(function () {
            consoleNode.show();
        });

    }) : _.noop;

    // exports ================================================================

    return DOMConsole;

});
